/*
**	machine.c 980120 Mikael Forselius <mikaelf@comenius.se>
**
**	980309 - ShowMachProfile() added
**
**	This file is subject to the terms and conditions of the GNU General Public
**	License.  See the file COPYING in the main directory of this archive
**	for more details.
**
**	Return machine name
*/

#include <Gestalt.h>
#include <LowMem.h>
#include <TextUtils.h>

#include "penguin_prototypes.h"
#include "penguin_utils.h"
#include "machine.h"

#define MAC_STR_RESID		128

typedef struct gestalt_tbl {
	OSType	selector;
	long	testBit;
	char	*name;
	char	*note;
} gestalt_tbl;

static char			*gestaltbit_yn(OSType selector, long whichBit);
static char			*strGestaltYes = "Yes";
static char			*strGestaltNo = "No";
static char			*strGestaltUnknown = "(not detected)";
static char			strHex[16] = {
	'0', '1', '2', '3', '4', '5', '6', '7', '8',
	'9', 'A', 'B', 'C', 'D', 'E', 'F'
};

static char			*gestalt_cpus[] = {
	"No CPU",
	"68000",
	"68010",
	"68020",
	"68030",
	"68040",
};

static char			*gestalt_fpus[] = {
	"No FPU",
	"68881",
	"68882",
	"68040 built-in"
};

static char			*gestalt_mmus[] = {
	"No MMU",
	"AMU",
	"68851",
	"68030 built-in",
	"68040 built-in",
	"Emulated"
};

static char			*gestalt_adbs[] = {
	"MacII style",
	"Egret 1 (IIsi style)",
	"Egret 2 (IIsi style)",
	"Cuda",
	"Unknown",
	"Unknown",
	"Unknown",
	"Unknown"
};

static gestalt_tbl	gestalt_info_tbl[] = {
	{ gestaltHardwareAttr,		gestaltHasVIA1,			"VIA1",				""						},
	{ gestaltHardwareAttr,		gestaltHasVIA2,			"VIA2",				""						},
	{ gestaltHardwareAttr,		gestaltHasRBV,			"RBV",				""						},
	{ gestaltHardwareAttr,		gestaltHasOSS,			"OSS",				""						},
	{ gestaltHardwareAttr,		gestaltHasASC,			"ASC",				""						},
	{ gestaltHardwareAttr,		gestaltHasSCC,			"SCC",				""						},
	{ gestaltHardwareAttr,		gestaltHasSCSI,			"SCSI_5380",		""						},
	{ gestaltHardwareAttr,		gestaltHasSCSIDMA,		"SCSI_5380 DMA",	""						},
	{ gestaltHardwareAttr,		gestaltHasSCSI961,		"SCSI_5396:1",		"(internal bus)"		},
	{ gestaltHardwareAttr,		gestaltHasSCSI962,		"SCSI_5396:2",		"(external bus)"		},
	{ gestaltHardwareAttr,		gestaltHasSWIMIOP,		"SWIM IOP",			""						},
	{ gestaltHardwareAttr,		gestaltHasSCCIOP,		"SCC IOP",			""						},
	{ gestaltHardwareAttr,		gestaltHasSonic,		"Sonic",			"(built-in ethernet)"	},
	{ gestaltHardwareAttr,		gestaltHasSoftPowerOff,	"SoftPowerOff",		""						},
	{ -1,						-1,						"",					""						},
};

/*
 *	ShowMachProfile()
 *
 *	Dump general machine info
 */
void
ShowMachProfile(void)
{
	gestalt_tbl		*gt;
	long			r, l;
	unsigned long	ul;
	unsigned short	hwFlags;
	char			*model;
	Str255			strModel;

	cprintf("########### Machine Profile ##########\n");

	// Mac
	Gestalt(gestaltMachineType, &r);
	model = GetModelName(r, strModel);
	cprintf("  %-16s : %d (%s)\n", "Mac", r, (model ? model : "Unknown"));

	// Sys
	Gestalt(gestaltSystemVersion, &r);
	cprintf("  %-16s : %d.%d.%d\n", "Sys", (int)((r>>8) & 255), (int)((r>>4) & 15), (int)(r & 15));

	// Mem
	Gestalt(gestaltPhysicalRAMSize, &r);
	cprintf("  %-16s : %d MB\n", "RAM", r / (1024L * 1024L));

	// CPU
	Gestalt(gestaltProcessorType, &r);
	cprintf("  %-16s : %s\n", "CPU", gestalt_cpus[r]);

	// FPU
	/* Check if SoftwareFPU is present */
	if (Gestalt('FPUE', &r) == noErr) {
		cprintf("  %-16s : %s\n", "FPU", gestalt_fpus[0]);
	} else {
		Gestalt(gestaltFPUType, &r);
		cprintf("  %-16s : %s\n", "FPU", gestalt_fpus[r]);
	}
	
	// MMU
	Gestalt(gestaltMMUType, &r);
	cprintf("  %-16s : %s\n", "MMU", gestalt_mmus[r]);

	// ADB, i.e. clock chip
	ul = *(unsigned long *)0xDD4;	// UnivROMBits
	if ( ul != -1 ) {
		ul &= 0x07000000;
		ul >>= 24L;
		if (ul != 0)
			cprintf("  %-16s : v%ld - %s\n", "ADB", ul, gestalt_adbs[ul]);
		else {
			if ( *(unsigned long *)0xD18 != -1)	// PmgrBase
				cprintf("  %-16s : v%ld - %s (guessing)\n", "ADB", 0, "PowerManager");
			else
				cprintf("  %-16s : v%ld - %s\n", "ADB", 0, gestalt_adbs[0]);
		}
	} else {
		cprintf("  %-16s : %s (guessing)\n", "ADB", gestalt_adbs[0]);
	}

	// ATA/IDE
	hwFlags = LMGetHWCfgFlags();
	cprintf("  %-16s : %-3s %s\n", "ATA/IDE", (hwFlags & 0x0080) ? strGestaltYes : strGestaltNo, "(ATA hardware)");

	// NuBus
	if (Gestalt(gestaltNuBusConnectors, &r) == noErr) {
		model = (char *)strModel;
		for(l = 9; l < 15; l++) {
			if ( r & (1 << l) ) {
				*model++ = '$';
				*model++ = strHex[l];
				*model++ = ',';
			}
		}
		if (model > (char *)strModel) {
			*--model = 0;
			cprintf("  %-16s : %-3s (slots %s)\n", "NuBus", strGestaltYes, strModel);
		} else {
			cprintf("  %-16s : %-3s %s\n", "NuBus", strGestaltNo, "(no slots present)");
		}
	} else {
		cprintf("  %-16s : %-3s %s\n", "NuBus", strGestaltNo, strGestaltUnknown);
	}

	// Bits
	for(gt = gestalt_info_tbl; gt->selector != -1; gt++)
		cprintf("  %-16s : %-3s %s\n", gt->name, gestaltbit_yn(gt->selector, gt->testBit), gt->note);

	cprintf("######################################\n");
}

/*
 *	GetModelName()
 *
 *	Return model name
 */
char *
GetModelName(long gestaltID, Str255 strMac)
{
	char		*macModel;

	GetIndString(strMac, MAC_STR_RESID, gestaltID);
	strMac[strMac[0]+1] = 0;
	if (strMac[0] != 0)
		macModel = (char *)&strMac[1];
	else
		macModel = NULL;

	return macModel;
}

/*
 *	IsVIA2Mac
 *
 *	Returns 1 if this machine has VIA2
 */
int
IsVIA2Mac(void)
{
	long	r;

	Gestalt(gestaltHardwareAttr, &r);
	return (r & (1 << gestaltHasVIA2));
}

/*
 *	IsRBVMac
 *
 *	Returns 1 if this machine has RBV
 */
int
IsRBVMac(void)
{
	long	r;

	Gestalt(gestaltHardwareAttr, &r);
	return (r & (1 << gestaltHasRBV));
}

/*
 *	IsOSSMac
 *
 *	Returns 1 if this machine has OSS
 */
int
IsOSSMac(void)
{
	long	r;

	Gestalt(gestaltHardwareAttr, &r);
	return (r & (1 << gestaltHasOSS));
}

/*
 *	MacHasHardware
 *
 *	Returns true if Mac has the requested hardware
 */
int
MacHasHardware(unsigned long gestaltBit)
{
	long	r;

	Gestalt(gestaltHardwareAttr, &r);
	return ( (r & (1 << gestaltBit)) != 0);
}


static char
*gestaltbit_yn(OSType selector, long whichBit)
{
	long	r;
	char	*result;

	result = strGestaltNo;

	if (Gestalt(selector, &r) == noErr) {
		if (r & (1L << whichBit))
			result = strGestaltYes;
	} else
		result = strGestaltUnknown;

	return result;
}

