// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/class_loader/Resolve.cpp,v 1.4 2001/11/12 10:24:49 gwu2 Exp $
//


//
// exceptions that can be thrown during class resolution:
//
//	(0) LinkageError exceptions during loading and linking
//	(1) ExceptionInInitializerError: class initializer completes
//		abruptly by throwing an exception
//	(2) IllegalAccessError: current class or interface does not have
//		permission to access the class or interface being resolved.
//
//	class resolution can occur indirectly via resolution of other
//	constant pool entries (Fieldref, Methodref, InterfaceMethodref),
//	or directly by the following byte codes:
//
//	(0) anewarray		(pg 162)
//	(1) checkcast		(pg 174)
//	(2) instanceof		(pg 256)
//	(3) multianewarray	(pg 316) 
//		- also throws the linking exception, IllegalAccessError, if 
//		the current class does have permission to access the *base* 
//		class of the resolved array class.
//	(4) new				(pg 318) 
//		- also throws the linking exception, InstantiationError, if 
//		the resolved class is an abstract class or an interface.
//
//	resolution of constants occurs directly by the following byte codes:
//	(0) ldc		(pg 291)
//	(1) ldc_w	(pg 292)
//	(2) ldc2_w	(pg 294)
//
//	of these ldc byte codes, only the ldc and ldc_w can cause exceptions
//	and only when they refer to CONSTANT_String entries.  The only exception
//	possible seems to be the VirtualMachineError exception that happens
//	when the VM runs out of internal resources.
//
//	exceptions that can be thrown during field and method resolution:
//
//	(0) any of the exceptions for resolving classes
//	(1) NoSuchFieldError: referenced field does not exist in specified 
//		class or interface.
//	(2) IllegalAccessError: the current class does not have permission
//		to access the referenced field.
//
//	During resolution of methods that are declared as native, if the code
//	for the native method cannot be found, then the VM throws an 
//	UnsatisfiedLinkError.  Note, that this causes a problem because the
//	code for native methods are usually loaded in by the static initializer
//	code of a class.  Therefore, this condition can only be checked at
//	run-time, when the native method is first called.
//
//	In addition, the byte codes that refer to the constant pool entries
//  can throw the following exceptions:
//
//	(0)	IncompatibleClassChangeError: thrown by getfield and putfield 
//		if the field reference is resolved to a static field.  
//	(1) IncompatibleClassChangeError: thrown by getstatic and putstatic
//		if the field reference is resolved to a non-static field.
//	(2) IncompatibleClassChangeError: thrown by invokespecial, 
//		invokeinterface and invokevirtual if the method reference is 
//		resolved to a static method.
//	(3) AbstractMethodError: thrown by invokespecial, invokeinterface
//		and invokevirtual if the method reference is resolved to an 
//		abstract method.
//	(4) IncompatibleClassChangeError: thrown by invokestatic if the
//		method reference is resolved to a non-static method.
//
//	Invokeinterface throws an IncompatibleClassChangeError if no method
//	matching the resolved name and description can be found in the class
//	of the object that is being invoked, or if the method being invoked is
//	a class (static) method.
//
#include "platform.h"
#include <iostream.h>

#include "Class.h"
#include "environment.h"
#include "jit_intf.h"
#include "compile.h"
#include "nogc.h"


#ifdef _DEBUG

void print_linking_error_message(Global_Env *env,
                                 Class *clss,
                                 unsigned cp_index,
                                 Loader_Exception exception)
{
	Const_Pool *cp = clss->const_pool;

    char *p_entry = "";

    const char *cp_type;
    switch(cp_tag(cp, cp_index)) {
    case CONSTANT_Class:
        cp_type = "class";
        p_entry = cp[cp_index].string->bytes;
        break;
    case CONSTANT_Fieldref:
        cp_type = "field";
        break;
    case CONSTANT_Methodref:
        cp_type = "method";
        break;
    case CONSTANT_InterfaceMethodref:
        cp_type = "interface method";
        break;
    case CONSTANT_String:
    case CONSTANT_NameAndType:
    default:
        cp_type = "????";
        break;
    }

    const char *error_name;
    switch(exception) {
    case LD_NoClassDefFoundError:
        error_name = "NoClassDefFoundError";
        break;
    default:
        error_name = "????";
        break;
    }

    orp_cout << "Linking error: " << error_name << " (" << (int)exception
             << ") while resolving "
             << cp_type << " at constant pool index " << cp_index
             << " in class " << clss->name->bytes << endl;

    orp_cout << p_entry << endl;

} //print_linking_error_message

#endif



Loader_Exception _resolve_class(Global_Env *env,
                                Class *clss,
                                unsigned cp_index)
{

    Const_Pool *cp = clss->const_pool;
    if(!cp_is_resolved(cp, cp_index)) {
        String *classname = cp[cp_index].string;

        //
        // load the class in
        //
        Loader_Exception exception;
        Class *other_clss;

        if(clss->class_loader) {
#if 0 //def _DEBUG
            orp_cout << "++++++ " << clss->name->bytes
                << " will use a non-system loader for "
                << classname->bytes << endl;
#endif
            other_clss = load_class_by_loader(env,
                                              classname,
                                              clss->class_loader,
                                              &exception);
        } else {
            other_clss = load_class_by_system_loader(env, classname, &exception);
        }
        if(other_clss == NULL)
        {
#ifdef _DEBUG
            printf("Error ---> can not find the following:  %s\n", classname->bytes);
#endif
            return exception;
        }
		//
		// link the loaded class
		//
		class_prepare(other_clss);

		//
		// check access control
		//
		if (class_is_public(other_clss) || other_clss == clss) {
			cp_resolve_to_class(cp,cp_index,other_clss);
			return LD_NoException;
		}

		//
		// package access; check to see if other_clss and clss are in 
		// the same package
		//
		if (other_clss->package == clss->package) {
			cp_resolve_to_class(cp,cp_index,other_clss);
			return LD_NoException;
		}

		//
		// IllegalAccessError
		//
		return LD_IllegalAccessError;
	} 
	return LD_NoException;
} //_resolve_class



Loader_Exception _resolve_class_new(Global_Env *env,
									Class *clss,
									unsigned cp_index)
{
	Loader_Exception exception = _resolve_class(env,clss,cp_index);
	if (exception != LD_NoException)
		return exception;
	Class *new_clss = clss->const_pool[cp_index].clss;
	if (class_is_abstract(new_clss)) {
		//
		// cannot new an instance of an abstract class
		// Note, that we can also do this check when we do the new
		// operation at run time.
		//
		return LD_InstantiationError;
	}
	return LD_NoException;
} //_resolve_class_new



Boolean check_member_access(Class_Member *member, Class *other_clss)
{
	Class *clss = member->get_class();
	//
	// check access permissions
	//
	if (member->is_public() || other_clss == clss) {
		// no problemo
		return 1;
	} else if (member->is_private()) {
		//
		// IllegalAccessError
		//
		return 0;
	} else if (member->is_protected()) {
		//
		// check if superclass
		//
		Class *c;
		for (c = clss->super_class; c != NULL; c = c->super_class) {
			if (c == other_clss)
				break;
		}
		if (c == NULL) {
			//
			// IllegalAccessError
			//
			return 0;
		}
	} 
	return 1;
}



Loader_Exception _resolve_field(Global_Env *env,
								Class *clss,
							    unsigned cp_index)
{

	Const_Pool *cp = clss->const_pool;
	Field *field;
	Class *other_clss;
	if (!cp_is_resolved(cp,cp_index)) {
		//
		// constant pool entry hasn't been resolved yet
		//
		unsigned other_index = cp[cp_index].cp_tag.class_index;
		Loader_Exception exception = _resolve_class(env,clss,other_index);
		//
		// check error condition from resolve class
		//
		if (exception != LD_NoException)
			return exception;

		other_clss = cp[other_index].clss;

		//::To implement the Step3 in JVM specification 5.4.3.2
		//Added a loop to lookup in superclasses
		do{
			//
			// lookup field in class
			//
			Signature *sig = cp[cp[cp_index].cp_tag.name_and_type_index].signature;
			field = class_lookup_field(other_clss,sig);
		}while(!field && (other_clss=other_clss->super_class)) ;

		if (field == NULL) {
			//
			// NoSuchFieldError
			//
			return LD_NoSuchFieldError;
		}
		//
		// check access permissions
		//
		if (check_member_access(field,other_clss) == 0) {
			//
			// IllegalAccessError
			//
			return LD_IllegalAccessError;
		}
		cp_resolve_to_field(cp,cp_index,field);
	} 
	return LD_NoException;
} //_resolve_field



Loader_Exception _resolve_static_field(Global_Env *env,
									   Class *clss,
									   unsigned cp_index)
{

	Loader_Exception exception = _resolve_field(env,clss,cp_index);

	if (exception != LD_NoException)
		return exception;

	Field *field = clss->const_pool[cp_index].field;
	if (!field->is_static()) {
		//
		// IncompatibleClassChangeError
		//
		return LD_IncompatibleClassChangeError;
	}
	return LD_NoException;
} //_resolve_static_field



Loader_Exception _resolve_nonstatic_field(Global_Env *env,
										  Class *clss,
										  unsigned cp_index)
{

	Loader_Exception exception = _resolve_field(env,clss,cp_index);

	if (exception != LD_NoException)
		return exception;

	Field *field = clss->const_pool[cp_index].field;
	if (field->is_static()) {
		//
		// IncompatibleClassChangeError
		//
		return LD_IncompatibleClassChangeError;
	}
	return LD_NoException;
} //_resolve_nonstatic_field



Loader_Exception _resolve_method(Global_Env *env,
								 Class *clss,
								 unsigned cp_index)
{
	Const_Pool *cp = clss->const_pool;
	Method *method;
	Class *other_clss;
	if (!cp_is_resolved(cp,cp_index)) {
		//
		// constant pool entry hasn't been resolved yet
		//
		unsigned other_index = cp[cp_index].cp_tag.class_index;
		Loader_Exception exception = _resolve_class(env,clss,other_index);
		//
		// check error condition from resolve class
		//
		if (exception != LD_NoException)
			return exception;

		other_clss = cp[other_index].clss;

        // CONSTANT_Methodref must refer to a class, not an interface, and
        // CONSTANT_InterfaceMethodref must refer to an  interface (orp spec 4.4.2)

        if (cp_is_methodref(cp, cp_index)   &&
            class_is_interface(other_clss)     )
            return LD_NoSuchMethodError;

        if (cp_is_interfacemethodref(cp, cp_index)  &&
            !class_is_interface(other_clss)             )
            return LD_NoSuchMethodError;

        //
		// lookup method in class
		//
		Signature *sig = cp[cp[cp_index].cp_tag.name_and_type_index].signature;
		method = class_lookup_method_recursive(other_clss,sig);
		if (method == NULL) {
			//
			// NoSuchMethodError
			//
			return LD_NoSuchMethodError;
		}
		//
		// check access permissions
		//
		if (check_member_access(method,other_clss) == 0) {
			//
			// IllegalAccessError
			//
			return LD_IllegalAccessError;
		}
		cp_resolve_to_method(cp,cp_index,method);
	}	
	return LD_NoException;
} //_resolve_method



Loader_Exception _resolve_static_method(Global_Env *env,
										Class *clss,
										unsigned cp_index)
{

	Loader_Exception exception = _resolve_method(env,clss,cp_index);

	if (exception != LD_NoException)
		return exception;

	Method *method = clss->const_pool[cp_index].method;
	if (!method->is_static()) {
		//
		// IncompatibleClassChangeError
		//
		return LD_IncompatibleClassChangeError;
	}
	if (method->is_abstract()) {
		//
		// AbstractMethodError
		//
		return LD_AbstractMethodError;
	}
	return LD_NoException;
} //_resolve_static_method



Loader_Exception _resolve_nonstatic_method(Global_Env *env,
										   Class *clss,
										   unsigned cp_index)
{

	Loader_Exception exception = _resolve_method(env,clss,cp_index);

	if (exception != LD_NoException)
		return exception;

	Method *method = clss->const_pool[cp_index].method;
	if (method->is_static()) {
		//
		// IncompatibleClassChangeError
		//
		return LD_IncompatibleClassChangeError;
	}
#if 0 // JMS
	if (method->is_abstract()) {
		//
		// AbstractMethodError
		//
		return LD_AbstractMethodError;
	}
#endif // 0
	return LD_NoException;
} //_resolve_nonstatic_method



Loader_Exception _resolve_interface_method(Global_Env *env,
										   Class *clss,
										   unsigned cp_index)
{

	Loader_Exception exception = _resolve_method(env,clss,cp_index);

	if (exception != LD_NoException)
		return exception;

	Method *method = clss->const_pool[cp_index].method;
	if (method->is_static()) {
		//
		// IncompatibleClassChangeError
		//
		return LD_IncompatibleClassChangeError;
	}
	return LD_NoException;
} //_resolve_interface_method



//
// resolve constant pool reference to a non-static field
// used for getfield and putfield
//
Field_Handle resolve_nonstatic_field(Compile_Handle h,
                                     Class_Handle c,
                                     unsigned index,
                                     Loader_Exception *exc)
{
	Loader_Exception exception =
        _resolve_nonstatic_field(((Compilation_Handle *)h)->env,
                                 (Class *)c, index);
	if (exception != LD_NoException) {
        if(exc) {
		    *exc = exception;
        }
#if _DEBUG
        print_linking_error_message(((Compilation_Handle *)h)->env,
                                    (Class *)c,
                                    index,
                                    exception);
#endif
		return NULL;
	}
	return ((Class *)c)->const_pool[index].field;
} //resolve_nonstatic_field



//
// resolve constant pool reference to a static field
// used for getstatic and putstatic
//
Field_Handle resolve_static_field(Compile_Handle h,
                                  Class_Handle c,
                                  unsigned index,
                                  Loader_Exception *exc)
{
	Loader_Exception exception =
        _resolve_static_field(((Compilation_Handle *)h)->env, (Class *)c, index);
	if (exception != LD_NoException) {
        if(exc) {
		    *exc = exception;
        }
#if _DEBUG
        print_linking_error_message(((Compilation_Handle *)h)->env,
                                    (Class *)c,
                                    index,
                                    exception);
#endif
		return NULL;
	}
	return ((Class *)c)->const_pool[index].field;
} //resolve_static_field



//
// resolve constant pool reference to a virtual method
// used for invokevirtual
//
Method_Handle resolve_nonstatic_method(Compile_Handle h,
                                       Class_Handle c,
                                       unsigned index,
                                       Loader_Exception *exc)
{
//::    orp_cout
//::        << "Please use the new API: resolve_virtual_method and resolve_special_method"
//::        << endl;
	Loader_Exception exception =
        _resolve_nonstatic_method(((Compilation_Handle *)h)->env,
                                  (Class *)c,
                                  index);
	if (exception != LD_NoException) {
        if(exc) {
		    *exc = exception;
        }
#if _DEBUG
        print_linking_error_message(((Compilation_Handle *)h)->env,
                                    (Class *)c,
                                    index,
                                    exception);
#endif
		return NULL;
	}
	return ((Class *)c)->const_pool[index].method;
} //resolve_nonstatic_method



//
// resolve constant pool reference to a virtual method
// used for invokevirtual
//
Method_Handle resolve_virtual_method(Compile_Handle h,
                                     Class_Handle c,
                                     unsigned index,
                                     Loader_Exception *exc)
{
	Loader_Exception exception =
        _resolve_nonstatic_method(((Compilation_Handle *)h)->env,
                                  (Class *)c,
                                  index);
	if (exception != LD_NoException) {
        if(exc) {
		    *exc = exception;
        }
#if _DEBUG
        print_linking_error_message(((Compilation_Handle *)h)->env,
                                    (Class *)c,
                                    index,
                                    exception);
#endif
		return NULL;
	}
	return ((Class *)c)->const_pool[index].method;
} //resolve_virtual_method



//
// resolve constant pool reference to a method
// used for invokespecial
//
Method_Handle resolve_special_method_env(Global_Env *env,
                                         Class_Handle c,
                                         unsigned index,
                                         Loader_Exception *exc)
{
    Class *curr_clss = (Class *)c;
	Loader_Exception exception =
        _resolve_nonstatic_method(env, curr_clss, index);
	if (exception != LD_NoException) {
        if(exc) {
		    *exc = exception;
        }
#if _DEBUG
        print_linking_error_message(env, curr_clss, index, exception);
                _resolve_nonstatic_method(env, curr_clss, index);

#endif
		return NULL;
	}
    Method *std_rules_meth = curr_clss->const_pool[index].method;
    Class *std_rules_meth_class = std_rules_meth->get_class();
    if(std_rules_meth->get_name() != env->Init_String &&
       (!std_rules_meth->is_private()) &&
       class_is_super(curr_clss)) {
        for(Class *clss = curr_clss->super_class; clss; clss = clss->super_class) {
            if(clss == std_rules_meth_class) {
                Signature *sig = std_rules_meth->get_signature();
                Class *super = curr_clss->super_class;
                if(!super)
                    super = curr_clss;
                Method *special_rules_meth = 
                    class_lookup_method_recursive(super, sig->name->bytes, sig->descriptor->bytes);
                return special_rules_meth;
            }
        }
    }
	return std_rules_meth;
} //resolve_special_method_env



#if 0
Method_Handle lookup_special_method_if_resolved(Global_Env *env,
                                                Class *clss,
                                                unsigned index)
{
    if(cp_is_resolved(clss->const_pool, index)) {
        return resolve_special_method_env(env, clss, index, 0);
    } else {
        return 0;
    }
} //lookup_special_method_if_resolved
#endif


//
// resolve constant pool reference to a method
// used for invokespecial
//
Method_Handle resolve_special_method(Compile_Handle h,
                                     Class_Handle c,
                                     unsigned index,
                                     Loader_Exception *exc)
{
    return resolve_special_method_env(((Compilation_Handle *)h)->env, c, index, exc);
} //resolve_special_method



//
// resolve constant pool reference to a static method
// used for invokestatic
//
Method_Handle resolve_static_method(Compile_Handle h,
                                    Class_Handle c,
                                    unsigned index,
                                    Loader_Exception *exc) 
{
	Loader_Exception exception =
        _resolve_static_method(((Compilation_Handle *)h)->env, (Class *)c, index);
	if (exception != LD_NoException) {
        if(exc) {
		    *exc = exception;
        }
#if _DEBUG
        print_linking_error_message(((Compilation_Handle *)h)->env,
                                    (Class *)c,
                                    index,
                                    exception);
#endif
		return NULL;
	}
	return ((Class *)c)->const_pool[index].method;
} //resolve_static_method



//
// resolve constant pool reference to a method
// used for invokeinterface
//
Method_Handle resolve_interface_method(Compile_Handle h,
                                       Class_Handle c,
                                       unsigned index,
                                       Loader_Exception *exc) 
{
	Loader_Exception exception = 
        _resolve_interface_method(((Compilation_Handle *)h)->env,
                                  (Class *)c, index);
	if (exception != LD_NoException) {
        if(exc) {
		    *exc = exception;
        }
#if _DEBUG
        print_linking_error_message(((Compilation_Handle *)h)->env,
                                    (Class *)c,
                                    index,
                                    exception);
#endif
		return NULL;
	}
	return ((Class *)c)->const_pool[index].method;
} //resolve_interface_method



//
// resolve constant pool reference to a class
// used for
//		(1) new	
//				- InstantiationError exception if resolved class is abstract
//		(2) anewarray
//		(3) checkcast
//		(4) instanceof
//		(5) multianewarray
//
// resolve_class_new is used for resolving references to class entries by the
// the new byte code.
//
Class_Handle resolve_class_new(Compile_Handle h,
                               Class_Handle c,
                               unsigned index,
                               Loader_Exception *exc) 
{
	Loader_Exception exception = 
        _resolve_class_new(((Compilation_Handle *)h)->env, (Class *)c, index);
	if (exception != LD_NoException) {
        if(exc) {
		    *exc = exception;
        }
#if _DEBUG
        print_linking_error_message(((Compilation_Handle *)h)->env,
                                    (Class *)c,
                                    index,
                                    exception);
#endif
		return NULL;
	}
	return ((Class *)c)->const_pool[index].clss;
} //resolve_class_new



//
// resolve_class is used by all the other byte codes that reference classes.
//
Class_Handle resolve_class(Compile_Handle h,
                           Class_Handle c,
                           unsigned index,
                           Loader_Exception *exc) 
{
	Loader_Exception exception =
        _resolve_class(h ? ((Compilation_Handle *)h)->env : 0,
                      (Class *)c, index);
	if (exception != LD_NoException) {
        if(exc) {
		    *exc = exception;
        }
#if _DEBUG
        print_linking_error_message(((Compilation_Handle *)h)->env,
                                    (Class *)c,
                                    index,
                                    exception);
#endif
		return NULL;
	}
	return ((Class *)c)->const_pool[index].clss;
} //resolve_class



Class *resolve_class_array_of_class(Global_Env *env,
                                    Class *cc,
                                    Loader_Exception *exc)
{
    char *array_name = (char *)gc_malloc_fixed_data_C(cc->name->len + 5);
    if(cc->name->bytes[0] == '[') {
        sprintf(array_name, "[%s", cc->name->bytes);
    } else {
        sprintf(array_name, "[L%s;", cc->name->bytes);
    }
    String *arr_str = env->string_pool.lookup(array_name);
    gc_free_fixed_data_C(array_name);
    // Class *arr_clss = class_load_verify_prepare(env, arr_str, exc);
	Class *arr_clss = class_load_verify_prepare_by_loader(env, arr_str, cc->class_loader, exc);
    return arr_clss;
} //resolve_class_array_of_class



//
// Given a class handle c construct a class handle of the type
// representing array of c.
//
Class_Handle resolve_class_array_of_class(Compile_Handle h,
                                          Class_Handle c,
                                          Loader_Exception *exc) 
{
    assert(h);
    Global_Env *env = ((Compilation_Handle *)h)->env;

    Class *arr_clss = resolve_class_array_of_class(env, (Class *)c, exc);
    assert(arr_clss);

    return arr_clss;
} //resolve_class_array_of_class



//
// resolve_constant is used by the ldc byte code
//
static Java_Type _class_get_cp_const_type(Class *clss, unsigned cp_index)
{
#ifdef _TRACE
    orp_cout << "class_get_cp_const_type " << *clss << "."
        << cp_index << endl;
#endif
	Const_Pool *cp = clss->const_pool;
    //assert(cp_is_cnst(cp, cp_index));
    switch(cp_tag(cp, cp_index)) {
    case CONSTANT_String:
        return JAVA_TYPE_STRING;
    case CONSTANT_Integer:
            return JAVA_TYPE_INT;
    case CONSTANT_Float:
            return JAVA_TYPE_FLOAT;
    case CONSTANT_Long:
            return JAVA_TYPE_LONG;
    case CONSTANT_Double:
            return JAVA_TYPE_DOUBLE;
    
    default:
        //assert(0);
        ;
    }
    return JAVA_TYPE_INVALID;
} //class_get_cp_const_type



Java_Type 
resolve_constant(Compile_Handle h, Class_Handle c, unsigned index, void *val) {
    //return JAVA_TYPE_VOID;
    return _class_get_cp_const_type((Class *)c, index);
}



Loader_Exception resolve_const_pool(Class *clss, Global_Env& env) {
	Loader_Exception exception = LD_NoException;
	Const_Pool *cp = clss->const_pool;
	// It's possible that cp is null when defining class on the fly
	if(!cp)return LD_NoException;
	unsigned cp_size = clss->cp_size;
	unsigned char *cp_tags = cp[0].tags;
	for (unsigned i = 1; i < cp_size; i++) {

		if (!cp_is_resolved(cp,i)) switch (cp_tags[i]) {
			case CONSTANT_Class:
				if ((exception = _resolve_class(&env,clss,i)) != LD_NoException)
					return exception;
				break;
			case CONSTANT_Fieldref:
				if ((exception = _resolve_field(&env,clss,i)) != LD_NoException)
					return exception;
				break;
			case CONSTANT_Methodref:
				if ((exception = _resolve_method(&env,clss,i)) != LD_NoException)
					return exception;
				break;
			case CONSTANT_InterfaceMethodref:
				if ((exception = _resolve_method(&env,clss,i)) != LD_NoException)
					return exception;
				break;
			case CONSTANT_NameAndType:
			case CONSTANT_Utf8:
				break;
			case CONSTANT_String:
			case CONSTANT_Integer:
			case CONSTANT_Float:
				break;
			case CONSTANT_Long:
			case CONSTANT_Double:
				i++;
				break;
		}
	}
	return LD_NoException;
} //resolve_const_pool

