// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/base_natives/gnu_classpath/java_io_RandomAccessFile.cpp,v 1.3 2001/08/07 06:53:02 gwu2 Exp $
//


#include "platform.h"


#include <sys/types.h>
#include <sys/stat.h>

#ifdef ORP_NT
#include <io.h>

#elif defined (ORP_POSIX)
#include <unistd.h>
#endif

#include "native_utils.h"
#include "jni_direct.h"
#include "jni_utils.h"
#include "object_layout.h"

#include <fcntl.h>

#ifdef ORP_NT
#include <direct.h>
#endif

#include <errno.h>


#include "java_io_File.h"
#include "java_io_FileInputStream.h"
#include "java_io_FileOutputStream.h"
#include "java_io_RandomAccessFile.h"
#include "gnu_classpath_jni_utils.h"


/*
 * Class:     RandomAccessFile
 * Method:    closeInternal
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_java_io_RandomAccessFile_closeInternal
  (JNIEnv *jenv, jobject jobj, jint file_d)
{
    Java_java_io_FileInputStream_closeInternal(jenv, jobj, file_d);
}


/*
 * Class:     RandomAccessFile
 * Method:    getFilePointerInternal
 * Signature: (I)J
 */
JNIEXPORT jlong JNICALL Java_java_io_RandomAccessFile_getFilePointerInternal
  (JNIEnv *jenv, jobject jobj, jint file_d)
{
#ifdef ORP_NT
    int64 pos = _lseek(file_d, 0L, SEEK_CUR);  // get current position
#elif defined (ORP_POSIX) 
    int64 pos = lseek(file_d, 0L, SEEK_CUR);  // get current position
#endif
	if (pos == -1) {
		// Throw IOException since read failed somehow.. use strerror(errno)
		throw_exception_from_jni(jenv, "java/io/IOException", 0);
		return 0;
	}
    return (jlong)pos;
}


/*
 * Class:     RandomAccessFile
 * Method:    lengthInternal
 * Signature: (I)J
 */
JNIEXPORT jlong JNICALL Java_java_io_RandomAccessFile_lengthInternal
  (JNIEnv *jenv, jobject jobj, jint file_d)
{
	int result;
	jlong ret;
 
	// Get data associated with file   
#ifdef ORP_NT
	struct _stat buf;
	result = _fstat( file_d, &buf );
#else
	struct stat buf;
	result = fstat( file_d, &buf );
#endif

	// Check if statistics are valid
	if (result != 0) {
		perror( "Problem getting information from call to stat in File.attr()" ); 
		assert(0);
	}
	else {
		ret = buf.st_size;
	}
    return (jlong)ret;
}


/*
 * Class:     RandomAccessFile
 * Method:    open
 * Signature: (Ljava/lang/String;Z)I
 */
JNIEXPORT jint JNICALL Java_java_io_RandomAccessFile_open
  (JNIEnv *jenv, jobject jobj, jstring jpath, jboolean read_only)
{
	jboolean is_copy;
	const char *path = jenv->GetStringUTFChars(jpath, &is_copy);
	assert(path); 
	
	//Should add O_CREAT flag in case the file doesn't exist
	int fmode = O_CREAT;
	//int fmode = 0;

    if (read_only)
    {
        fmode |= O_RDONLY;
    }
    else  {
        fmode |= O_RDWR;
    }

#ifdef ORP_NT
	// Rope in _O_BINARY so that Ctrl-Z will not be interpreted as an EOF indicator, 
	// and translations are suppressed. Is it okay to have this ALWAYS ??? BUG?????
	fmode |= _O_BINARY;
#endif

	// Actually open the file; use default permissions
#ifdef ORP_NT
	int file_descr = _open(path, fmode, 0755);
#elif (ORP_POSIX)
	int file_descr = open(path, fmode, 0755);
#endif

	if (file_descr == -1) {
		char *msg = (char *) malloc(300);
#if 0 
		// Need to get strerror() working...23 compile errors!	
		sprintf(msg, "%s: %s", path, strerror (errno));
#else
		strcpy(msg, "Open failed???");
#endif

		throw_exception_from_jni(jenv, "java/io/FileNotFoundException", msg);
		free(msg);
	}
	return file_descr;
}


/*
 * Class:     RandomAccessFile
 * Method:    readInternal
 * Signature: (I[BII)I
 */
JNIEXPORT jint JNICALL Java_java_io_RandomAccessFile_readInternal
  (JNIEnv *jenv, jobject jobj, jint fd, jbyteArray barr, jint offset, jint count)
{
    int ret = Java_java_io_FileInputStream_readInternal(jenv, jobj, fd, barr, offset, count);
    
    // The problem here is that Java_java_io_RandomAccessFile_readInternal expects
    // return value of "0" to flag EOF -- whereas Java_java_io_FileInputStream_readInternal
    // returns -1 when EOF is reached and zero bytes were read.

    if (ret == -1) 
        return 0;
    return ret;
}


/*
 * Class:     RandomAccessFile
 * Method:    seekInternal
 * Signature: (IJ)V
 */


JNIEXPORT void JNICALL Java_java_io_RandomAccessFile_seekInternal
  (JNIEnv *jenv, jobject jobj, jint file_d, jlong offset)
{
#ifdef ORP_NT
    _int64 pos = _lseek(file_d, 0L, SEEK_SET);  // at beginning of file
#elif defined (ORP_POSIX)
    int64 pos = lseek(file_d, 0L, SEEK_SET);  // at beginning of file
#endif

	if (pos == -1) {
		// Throw IOException since read failed somehow.. use strerror(errno)
		throw_exception_from_jni(jenv, "java/io/IOException", 0);
	}
#ifdef ORP_NT
    pos = _lseek(file_d, (long)offset, SEEK_CUR);  // to offset
#elif defined (ORP_POSIX)
    pos = lseek(file_d, (long)offset, SEEK_CUR);  // to offset
#endif
	if (pos == -1) {
		// Throw IOException since read failed somehow.. use strerror(errno)
		throw_exception_from_jni(jenv, "java/io/IOException", 0);
	}
}


/*
 * Class:     RandomAccessFile
 * Method:    setLengthInternal
 * Signature: (IJ)V
 */
JNIEXPORT void JNICALL Java_java_io_RandomAccessFile_setLengthInternal
  (JNIEnv *jenv, jobject jobj, jint file_d, jlong offset)
{
    assert(0);
}


/*
 * Class:     RandomAccessFile
 * Method:    skipInternal
 * Signature: (II)I
 */
JNIEXPORT jint JNICALL Java_java_io_RandomAccessFile_skipInternal
  (JNIEnv *jenv, jobject jobj, jint file_d, jint skip_count)
{
    return (jint)Java_java_io_FileInputStream_skipInternal(jenv, jobj, file_d, skip_count);
}


/*
 * Class:     RandomAccessFile
 * Method:    writeInternal
 * Signature: (I[BII)V
 */
JNIEXPORT void JNICALL Java_java_io_RandomAccessFile_writeInternal
  (JNIEnv *jenv, jobject jobj, jint fd, jbyteArray barr, jint offset, jint len)
{
    Java_java_io_FileOutputStream_writeInternal(jenv, jobj, fd, barr, offset, len);
}
