// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/opt_peephole.cpp,v 1.5 2001/09/04 07:40:57 xhshi Exp $
//



#include "defines.h"
#include "flow_graph.h"

extern bool O3_is_PPro;

static bool is_same_mem(Operand *mem1, Operand *mem2)
{
    if (mem1->kind != mem2->kind)
        return false;
    if (mem1->kind == Operand::Static)
        return ((Static_Operand *)mem1)->addr == ((Static_Operand *)mem2)->addr;
    if (mem1->kind != Operand::Array && mem1->kind != Operand::Field)
        return false;
    Operand *b1 = mem1->base();
    Operand *b2 = mem2->base();
    if (b1->bv_position() != b2->bv_position())
        return false;

	//::If the type of mem1 is field, we should consider other conditions.
    switch(mem1->kind){
    case Operand::Array:{
        Operand *i1 = mem1->index();
        Operand *i2 = mem2->index();
        assert(i1 && i2);
        if (i1->bv_position() != i2->bv_position())
            return false;
        break;}
    case Operand::Field:{
        if( ((Field_Operand*)mem1)->offset() != 
            ((Field_Operand*)mem2)->offset() )
            return false;
        break;}
	default:
		assert(0) ;
		return false ;
    }

    return true;
}

// Look for "eax=[]; eax=eax+imm; []=eax", last use of eax.
// Change to "[]=[]+imm".
static bool is_increment(Inst *&inst, Inst *last)
{
    Inst *load = inst;
    Inst *add = load->next();
    Inst *store= add->next();
    if (add == last || store == last)
        return false;
    if (!load->is_assignment())
        return false;
    if (!store->is_assignment())
        return false;
    if (!add->is_add() && !add->is_sub())  // XXX- lots of other conditions would do.
        return false;
    if (add->type() != JIT_TYPE_INT)
        return false;
    Operand *eax = load->dst();
    if (!eax->is_reg())
        return false;
    unsigned bvp = eax->bv_position();
    if (add->dst()->bv_position() != bvp)
        return false;
    if ((!add->src(0)->is_reg() || add->src(0)->bv_position() != bvp) &&
        (!add->src(1)->is_reg() || add->src(1)->bv_position() != bvp))
        return false;
    if (!store->src(0)->is_reg())
        return false;
    if (store->src(0)->bv_position() != bvp)
        return false;
    Operand *mem = load->src(0);
    if (!mem->is_mem())
        return false;
    if (!is_same_mem(store->dst(), mem))
        return false;
    if (!(store->live_ranges_ended & 4))
        return false;
    Operand *imm = add->src(1);
    unsigned imm_bvp = -1;
    if (imm->is_reg())
        imm_bvp = imm->bv_position();
    bool commuted = false;
    if (imm_bvp == eax->bv_position())
    {
        imm = add->src(0);
        commuted = true;
    }
    if (imm_bvp == eax->bv_position())
        return false;
    if (imm->kind != Operand::Immediate) // XXX- could also be physical register
        return false;
    add->set_dst(mem);
    add->replace_src((commuted ? 1 : 0), mem);
    unsigned lre = (store->live_ranges_ended & 3);
    add->live_ranges_ended = lre | (lre << 2);
    // XXX- adjust bits 4 and 5 if physical register
    load->unlink();
    store->unlink();
    inst = add;
    return true;
}

static bool is_movzx(Inst *&inst, Inst *last)
{
    Inst *movzx = inst;
    if (!movzx->is_widen())
        return false;
    if (!((Widen_Inst *)movzx)->is_signed())
        return false;
    if (!((Widen_Inst *)movzx)->is_half())
        return false;
    Inst *shl = movzx->next();
    if (shl == last || !shl->is_bitwise())
        return false;
    if (((Bitwise_Inst *)shl)->kind != Bitwise_Inst::shl)
        return false;
    Inst *shr = shl->next();
    if (shr == last || !shr->is_bitwise())
        return false;
    if (((Bitwise_Inst *)shr)->kind != Bitwise_Inst::shr)
        return false;
    Operand *dst = movzx->dst();
    if (!dst->is_reg())
        return false;
    unsigned bvp = dst->bv_position();
    Operand *src = shl->src(0);
    if (!src->is_reg() || src->bv_position() != bvp)
        return false;
    src = shr->src(0);
    if (!src->is_reg() || src->bv_position() != bvp)
        return false;
    src = shl->src(1);
    if (src->kind != Operand::Immediate || ((Imm_Operand *)src)->imm() != 16)
        return false;
    src = shr->src(1);
    if (src->kind != Operand::Immediate || ((Imm_Operand *)src)->imm() != 16)
        return false;
    ((Widen_Inst *)movzx)->set_signed(false);
    shl->unlink();
    shr->unlink();
    return true;
}

static bool is_reg_cascade(Inst *&inst, Inst *last)
{
    Inst *mov1 = inst;
    if (!mov1->is_assignment() && !mov1->is_widen())
        return false;
    Inst *mov2 = mov1->next();
    if (mov2 == last || !mov2->is_assignment())
        return false;
    if (!(mov2->live_ranges_ended & 4))
        return false;
    Operand *reg2 = mov1->dst();
    if (!reg2->is_reg())
        return false;
    if (IS_FP_DBL_TYPE(reg2->type))
        return false;
    Operand *reg1 = mov1->src(0);
    Operand *reg2a = mov2->src(0);
    if (!reg2a->is_reg())
        return false;
    unsigned bvp2 = reg2->bv_position();
    if (reg2a->bv_position() != bvp2)
        return false;
    Operand *reg3 = mov2->dst();
    if (!reg3->is_reg())
        return false;
    unsigned bvp1 = (reg1->is_reg() ? reg1->bv_position() : n_reg);
    unsigned bvp3 = reg3->bv_position();
    if (bvp1 >= n_reg && bvp3 >= n_reg)
        return false;
#if 0
    if (mov1->is_widen() && !((Widen_Inst *)mov1)->is_half() &&
        (bvp3 >= n_reg || !(ALL_X86_BYTE_REGS & (1u << bvp3))))
        return false;
#endif // 0
    if (mov1->is_widen() && reg3->bv_position() >= n_reg)
        return false;
    mov1->set_dst(reg3);
    mov2->unlink();
    return true;
}

static bool is_movzx_1(Inst *&inst, Inst *last)
{
    Inst *movsx = inst;
    if (!movsx->is_widen())
        return false;
    Inst *i_and = movsx->next();
    if (i_and == last || !i_and->is_bitwise())
        return false;
    Bitwise_Inst *band = (Bitwise_Inst *) i_and;
    if (band->kind != Bitwise_Inst::k_and)
        return false;
    Operand *dst = movsx->dst();
    if (!dst->is_reg())
        return false;
    unsigned bvp = dst->bv_position();
    Operand *src = i_and->src(0);
    if (!src->is_reg() || src->bv_position() != bvp)
        return false;
    src = i_and->dst();
    if (!src->is_reg() || src->bv_position() != bvp)
        return false;
    src = i_and->src(1);
    if (src->kind != Operand::Immediate)
        return false;
    unsigned mask = (((Widen_Inst *)movsx)->is_half() ? 0xffff : 0xff);
    if (mask != ((Imm_Operand *)src)->imm())
        return false;
    ((Widen_Inst *)movsx)->set_signed(false);
    i_and->unlink();
    return true;
}

// Look for "and eax, 0xff; movsx eax, al".
// Remove the redundant "and" instruction.
static bool is_and_widen(Inst *&inst, Inst *last)
{
    Inst *i_and = inst;
    if (!i_and->is_bitwise())
        return false;
    Bitwise_Inst *band = (Bitwise_Inst *) i_and;
    if (band->kind != Bitwise_Inst::k_and)
        return false;
    Inst *movsx = i_and->next();
    if (movsx == last || !movsx->is_widen())
        return false;
    Operand *reg = movsx->src(0);
    if (!reg->is_reg())
        return false;
    unsigned bvp = reg->bv_position();
    Operand *dst = i_and->dst();
    if (!dst->is_reg() || dst->bv_position() != bvp)
        return false;
    Operand *immopnd = i_and->src(1);
    if (immopnd->kind != Operand::Immediate)
        return false;
    unsigned immval = ((Imm_Operand *)immopnd)->imm();
    unsigned mask = (((Widen_Inst *)movsx)->is_half() ? 0xffff : 0xff);
    if ((immval & mask) != mask)
        return false;
    i_and->unlink();
    inst = movsx->prev();
    if (inst == last)
        inst = movsx;
    return true;
}

static void peephole(Cfg_Node *node, Closure *c)
{
    Inst *last = node->IR_instruction_list();
    Inst *nxt,*inst;
    for (inst=last->next(); inst!=last; inst=nxt)
    {
        nxt = inst->next();
        if (inst->is_same_reg_copying() && !inst->is_fp_pop() && inst->can_eliminate())
        {
            inst->unlink();
            continue;
        }
        if (
            is_increment(inst, last) ||
            is_movzx(inst, last) ||
            is_reg_cascade(inst, last) ||
            is_movzx_1(inst, last) ||
            is_and_widen(inst, last) ||
            false)
        {
            nxt = inst;
            continue;
        }
    }
}

// Look for if/then/else, with same dst in each block.
static void find_cmov_1(Cfg_Node *node, Flow_Graph *fg, bool &result)
{
    Inst *head = node->IR_instruction_list();
    Inst *last = head->prev();
    // non-empty BB
    if (head == last)
        return;
    // ends in a branch
    if (!last->is_branch())
        return;
    // next-to-last is a compare
    Inst *cmp = last->prev();
    if (cmp == head)
        return;
    if (!cmp->is_compare())
        return;
    // 2 out-edges
    if (node->out_edge_size() != 2)
        return;
    Cfg_Node *node1 = node->out_edges(0);
    Cfg_Node *node2 = node->out_edges(1);
    // targets have single in-edge
    if (node1->in_edge_size() != 1)
        return;
    if (node2->in_edge_size() != 1)
        return;
    // targets have single out-edge
    if (node1->out_edge_size() != 1)
        return;
    if (node2->out_edge_size() != 1)
        return;
    // targets merge to same point
    if (node1->out_edges(0) != node2->out_edges(0))
        return;
    Inst *head1 = node1->IR_instruction_list();
    Inst *head2 = node2->IR_instruction_list();
    Inst *assn1 = head1->next();
    Inst *assn2 = head2->next();
    // Single instruction in both BBs
    if (head1 == assn1 || head2 == assn2 || assn1->next() != head1 || assn2->next() != head2)
        return;
    // Assignment statements
    if (!assn1->is_assignment() || !assn2->is_assignment())
        return;
    // Same destination
    if (assn1->dst() != assn2->dst())
        return;
    // Don't do FP for now.
    if (IS_FP_DBL_TYPE(assn1->dst()->type))
        return;
    // dst must be physical register
    if (assn1->dst()->bv_position() >= n_reg)
        return;
    // cmov can't take an immediate operand.
    if (assn1->src(0)->kind == Operand::Immediate || assn2->src(0)->kind == Operand::Immediate)
        return;
    Cfg_Node *merge = node1->out_edges(0);
    // node1 is fallthrough, node2 is branch target
    assn1->unlink();
    assn2->unlink();
    ((Assign_Inst *)assn2)->set_cmov((unsigned char) ((Branch_Inst *)last)->kind(),
        ((Branch_Inst *)last)->is_signed);
    last->unlink();
    assn2->insert_after(cmp); // very last instruction
    assn1->insert_after(cmp); // next-to-last instruction
    node1->delete_edge(merge);
    node2->delete_edge(merge);
    node->delete_edge(node2);
    node->delete_edge(node1);
    node->add_edge(fg->mem_manager, merge);
    node1->linearization_node()->unlink();
    node2->linearization_node()->unlink();
    result = true;

#if 0
    cout << class_get_name(fg->c_handle()) << "." << method_get_name(fg->m_handle()) << endl;
    cout << "  cmov opportunity in node " << node->label << endl;
#endif // 0
}

static unsigned char commute(Branch_Inst::Kind kind)
{
    Branch_Inst::Kind result;
    switch (kind)
    {
    case Branch_Inst::blt:
        result = Branch_Inst::bge; break;
    case Branch_Inst::bgt:
        result = Branch_Inst::ble; break;
    case Branch_Inst::ble:
        result = Branch_Inst::bgt; break;
    case Branch_Inst::bge:
        result = Branch_Inst::blt; break;
    case Branch_Inst::beq:
        result = Branch_Inst::bne; break;
    case Branch_Inst::bne:
        result = Branch_Inst::beq; break;  
    default:
        assert(0);
    }
    return (unsigned char) result;
}

// Look for if/then, with single assignment.
static void find_cmov_2(Cfg_Node *node, Flow_Graph *fg, bool &result)
{
    Inst *head = node->IR_instruction_list();
    Inst *last = head->prev();
    // non-empty BB
    if (head == last)
        return;
    // ends in a branch
    if (!last->is_branch())
        return;
    // next-to-last is a compare
    Inst *cmp = last->prev();
    if (cmp == head)
        return;
    if (!cmp->is_compare())
        return;
    // 2 out-edges
    if (node->out_edge_size() != 2)
        return;
    Cfg_Node *node1 = node->out_edges(0); // fallthrough
    Cfg_Node *node2 = node->out_edges(1); // target/merge
    // fallthrough has single in-edge
    if (node1->in_edge_size() != 1)
        return;
    // fallthrough has single out-edge
    if (node1->out_edge_size() != 1)
        return;
    // target is merge fallthrough
    if (node1->out_edges(0) != node2)
        return;
    Inst *head1 = node1->IR_instruction_list();
    Inst *assn1 = head1->next();
    // Single instruction in fallthrough
    if (head1 == assn1 || assn1->next() != head1)
        return;
    // Assignment statement
    if (!assn1->is_assignment())
        return;
    // Don't do FP for now.
    if (IS_FP_DBL_TYPE(assn1->dst()->type))
        return;
    // dst must be physical register
    if (!assn1->dst()->is_reg())
        return;
    if (assn1->dst()->bv_position() >= n_reg)
        return;
    // cmov can't take an immediate operand.
    if (assn1->src(0)->kind == Operand::Immediate)
        return;
#if 1
    Cfg_Node *merge = node2;
    assn1->unlink();
    ((Assign_Inst *)assn1)->set_cmov(commute(((Branch_Inst *)last)->kind()),
        ((Branch_Inst *)last)->is_signed);
    last->unlink();
    assn1->insert_after(cmp); // last instruction
    node1->delete_edge(merge);
    node->delete_edge(node1);
    node1->linearization_node()->unlink();
    result = true;
#endif // 0

#if 0
    cout << class_get_name(fg->c_handle()) << "." << method_get_name(fg->m_handle()) << endl;
    cout << "  cmov opportunity in node " << node->label << endl;
#endif // 0
}

static void find_cmov(Cfg_Node *node, Closure *c)
{
    Flow_Graph *fg = (Flow_Graph *)c;
    bool result = false;
#if 0
    // I'm not sure this is a good idea in practice, because the cmov is so expensive.
    find_cmov_1(node, fg, result);
    if (result)
        return;
#endif // 0
    find_cmov_2(node, fg, result);
    if (result)
        return;
}

void peephole_opt(Flow_Graph *fg)
{
    fg->apply(peephole, NULL);
    if (O3_is_PPro)
    {
		//
		// I have to comment the find_cmov
		// Because CMOVcc will raise exceptions when the offset is illegal on PIII Xeon
		// sxh 2001.7.2
		// 
        //fg->apply(find_cmov, (Closure *)fg);

        //fg->remove_empty_blocks(); // shouldn't be necessary...
    }
}
