// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o3_jit/build_ir_routines.cpp,v 1.3 2001/09/05 04:58:25 xhshi Exp $
//




#define ORP_DOES_AASTORE  // see also ir.cpp

#include "defines.h"
#include <iostream.h>
#include "jit_intf.h"
#include "Mem_Manager.h"
#include "ir.h"
#include "stack.h"
#include "expression.h"
#include "local_cse.h"
#include "build_ir_routines.h"
#include "flow_graph.h"
#include "mtable.h"
#include "handles.h"
#include "jit_common.h"

#ifdef JIT_SAPPHIRE
#include "gc_for_orp.h"
#endif

Spill_Operand *create_spill_opnd(Mem_Manager& mem, unsigned spill_id, O3_Jit_Type ty) {
    Spill_Operand *sp = new (mem) Spill_Operand(spill_id,ty);
    if (IS_64BIT(ty))
        sp->set_hi_opnd(new (mem) Spill_Operand(spill_id+1,ty));
    return sp;
}

//
// create a constant operand
// A long constant is further divided into lo and hi
//
Const_Operand *create_const_opnd(Mem_Manager& mem, Value *val, O3_Jit_Type ty) {
    Const_Operand *con = new (mem) Const_Operand(val,ty);
    if (ty == JIT_TYPE_LONG) {
        con->set_lo(new (mem) Imm_Operand(val->l.lo, JIT_TYPE_INT));
        con->set_hi(new (mem) Imm_Operand(val->l.hi, JIT_TYPE_INT));
    }
    else if (ty == JIT_TYPE_DOUBLE)
    {
        con->set_hi_opnd(new (mem) Const_Operand(val,ty));
    }
    return con;
}

//
// create a static memory operand
// hi_opnd is used when long inst is expanded
//
Static_Operand *create_static_opnd(Mem_Manager& mem, void *addr, O3_Jit_Type ty, Field_Handle fh){
    Static_Operand *stc = new (mem) Static_Operand(addr,ty, fh);
    if (IS_64BIT(ty)) 
        stc->set_hi_opnd(new (mem) Static_Operand((void*)((unsigned)addr+sizeof(int)),ty, fh));
    return stc;
}

//
// create a ret operand
// hi_opnd is used when long inst is expanded
//
Ret_Operand *create_ret_opnd(Mem_Manager& mem, unsigned reg_id, O3_Jit_Type ty) {
    Ret_Operand *ret = new (mem) Ret_Operand(reg_id,ty);
    if (ty == JIT_TYPE_LONG) {
        Ret_Operand *ret_hi = new (mem) Ret_Operand(reg_id+1,ty);
        ret->set_hi_opnd(ret_hi);
        ret_hi->assign_call_convention_reg();
    }
    ret->assign_call_convention_reg();
    return ret;
}

//
// create a virtual reg operand
// hi_opnd is used when long inst is expanded
//
Virtual_Reg *create_vreg_opnd(Mem_Manager& mem, unsigned reg_id, O3_Jit_Type ty) {
    Virtual_Reg *vreg = new (mem) Virtual_Reg(reg_id,ty);
    if (IS_64BIT(ty))
        vreg->set_hi_opnd(new (mem) Virtual_Reg(reg_id+1,ty));
    return vreg;
}

//
// create an arg operand
//
Arg_Operand *create_arg_opnd(Mem_Manager& mem, unsigned ith_arg, 
                             unsigned reg_id, O3_Jit_Type ty) {
    Arg_Operand *arg = new (mem) Arg_Operand(ith_arg,reg_id,ty);
    //
    // hi_opnd is used later when we expand long 
    //
    if (IS_64BIT(ty)) {
        Arg_Operand *arg_hi = new (mem) Arg_Operand(ith_arg+1,reg_id+1,ty);
        arg->set_hi_opnd(arg_hi);
    }
    return arg;
}

//
// create a new array operand
// base and index must be registers because of tuple representation
//
Array_Operand *create_array_operand(Mem_Manager& mem, Inst *base, Inst *index, 
                                    Inst *bnd, unsigned shift, unsigned offset,
                                    O3_Jit_Type ty, bool optimize) {
    //
    // make sure that base & indx are register operand
    //
	Reg_Operand *base_dst = (Reg_Operand*)base->dst();
	Reg_Operand *indx_dst = (Reg_Operand*)index->dst();
    assert(base_dst && base_dst->kind == Operand::GCTrack &&
		   indx_dst && indx_dst->kind == Operand::GCTrack);
    //
	// create array operand
    //
#if 1
    if (optimize && indx_dst->is_temp_reg()) {
        Inst *i = ((Temp_Reg*)indx_dst)->inst();
        // 
        // check if i is "add t, 1"
        //
        int imm = 0;
        if (i != NULL && i->src(0)->is_reg() && (i->is_add() || i->is_sub())) {
            int sign = 0;
            if (i->is_add() && ((Add_Inst*)i)->kind == Add_Inst::add)
                sign = 1;
            else if (i->is_sub() && ((Sub_Inst*)i)->kind == Sub_Inst::sub)
                sign = -1;
            if (i->src(1)->kind == Operand::Immediate)
                imm = ((Imm_Operand*)i->src(1))->imm();
            else if (i->src(1)->is_temp_reg() && ((Temp_Reg*)i->src(1))->inst()->is_imm_assignment())
                imm = ((Imm_Operand*)((Temp_Reg*)i->src(1))->inst()->src(0))->imm();
            else sign = 0;
            //
            // produce [b + i*4 + (8+sign*imm*4)]
            //
            if (sign != 0) {
                indx_dst = (Reg_Operand*)i->src(0);
                offset += (imm*sign)<<shift;
            }
        }
    }
#endif
	Array_Operand * arry = new (mem) Array_Operand(base_dst,indx_dst,bnd->dst(),shift,offset,ty);
    if (IS_64BIT(ty))
        arry->set_hi_opnd(new (mem) Array_Operand(base_dst,indx_dst,bnd->dst(),shift,offset+4,ty));
    return arry;
}

//
// create a new field operand
// base  must be a register because of tuple representation
//
Field_Operand *create_field_operand(Mem_Manager& mem, Inst *base, 
                                    unsigned offset, O3_Jit_Type ty,
                                    FIELD_UID fid) {
    //
    // make sure that base is register operand
    //
	Reg_Operand *base_dst = (Reg_Operand*)base->dst();
	assert(base_dst && base_dst->is_reg());
    //
    // create field operand	
    // we assume that the fields may throw null pointer exception
    //
	Field_Operand *fld = new (mem) Field_Operand(base_dst,offset,ty,fid,true);
    if (IS_64BIT(ty))
        fld->set_hi_opnd(new (mem) Field_Operand(base_dst,offset+4,ty,fid,true));
    return fld;
}

//
// create an assignment expression and then generate an instruction
//
void create_assign_inst(Mem_Manager& mem, Expressions& exprs, Inst *inst_head,
                        Operand *dst, Exp *dst_exp, Inst *src, O3_Jit_Type type) {
    // If this is an astore of type ReturnAddress, "type" will actually be
    // JIT_TYPE_CLASS, and needs to be changed.
#if 0
    if (type == JIT_TYPE_CLASS && src->type() == JIT_TYPE_RETADDR)
        type = JIT_TYPE_RETADDR;
#endif
    Exp *asgn = exprs.lookup_inst_exp(Exp::Assign,dst_exp,src->exp,type);
    new (mem) Assign_Inst(dst,src->dst(),asgn,inst_head);
    //
    // assignment kills local cse
    //
    dst_exp->asgn_kill_local_cse(exprs.live_lcse());
    //
    // local copy propagation
    //
    exprs.gen_local_cse(dst_exp, src, true);
}

void array_load(Mem_Manager& mem, Expressions& exprs,
                Stack& stack, Inst *inst_head, char ty) {
	Inst *indx = stack.pop();
	Inst *base = stack.pop();
    stack.push(array_load(mem,exprs,inst_head,base,indx,ty));
}

#ifdef JIT_SAPPHIRE
Inst *array_load(Mem_Manager& mem, Expressions& exprs,
                Inst *inst_head, Inst *base, Inst *indx, char ty) {
    //
    // bound checking code
    //
    Inst *len = exprs.lookup_inst(Exp::Length,base,NULL,JIT_TYPE_INT,inst_head);
    Inst *bnd = gen_array_cmp_inst(mem,exprs,inst_head,indx,len,JIT_TYPE_INT,true);

	O3_Jit_Type type = (O3_Jit_Type)ty;
	//:: Change the number 8 -- array_offset , to a VM interface get_array_offset()
	unsigned array_offset = get_array_offset((Java_Type)ty) ;
	unsigned shift ;
	switch (type) {
	case 'B':	// boolean or 8-bit signed
		shift = 0 ; break ;
	case 'C':	// 16-bit unsigned char
	case 'S':	// signed 16-bit short
		shift = 1 ; break ;
	case 'F':	// single FP
	case 'I':	// 32-bit int
	case 'L':	// object reference
		shift = 2 ; break ;
	case 'D':	// double FP
	case 'J':	// 64-bit long
		shift = 3 ; break ;
	default:
	    assert(0); break ;
	}

	//
	//Read barrier (t55~t58 duplicate in Exp::Widen?)
	//  t55 = .I shift
	//	t56 = .I 8
	//	t57 = shl.I indx, t55
	//	t58 = add.I t57, t56
	//	read_barrier base, t58
	//
	Inst *s   = exprs.lookup_imm(shift,JIT_TYPE_INT,inst_head);
	Inst *i8  = exprs.lookup_imm(/*8*/array_offset,JIT_TYPE_INT,inst_head);
	Inst *shf = exprs.lookup_inst(Exp::Shl,indx,s,JIT_TYPE_INT,inst_head);
	Inst *off = exprs.lookup_inst(Exp::Add,shf,i8,JIT_TYPE_INT,inst_head);
	Exp *rb_exp = exprs.lookup_inst_exp(Exp::ReadBarrier, base->exp, off->exp, JIT_TYPE_VOID);
	new(mem) Readbarrier_Inst(base->dst(), off->dst(), rb_exp, type, inst_head);

	if(shift <=1 ){ //'B','C','S'
        Array_Exp *arry_exp = exprs.lookup_array_exp(base->exp,indx->exp,bnd->exp,shift,/*8*/array_offset,type);
        Inst_Exp* widen_exp = exprs.lookup_inst_exp(Exp::Widen,arry_exp,NULL,type);
        Array_Operand *array_opnd = create_array_operand(mem,base,indx,bnd,shift,/*8*/array_offset,type,true);
        Inst tmp;
        tmp.set_dst(array_opnd);
        return exprs.gen_inst_tuple(inst_head,widen_exp,Exp::Widen,&tmp,NULL);
	}else{
		return exprs.lookup_array(base,indx,bnd,shift,/*8*/array_offset,type,inst_head);
	}
}

#else
Inst *array_load(Mem_Manager& mem, Expressions& exprs,
                Inst *inst_head, Inst *base, Inst *indx, char ty) {
    //
    // bound checking code
    //
    Inst *len = exprs.lookup_inst(Exp::Length,base,NULL,JIT_TYPE_INT,inst_head);
    Inst *bnd = gen_array_cmp_inst(mem,exprs,inst_head,indx,len,JIT_TYPE_INT,true);

	O3_Jit_Type type = (O3_Jit_Type)ty;
	//:: Change the number 8 -- array_offset , to a VM interface get_array_offset()
	unsigned array_offset = get_array_offset((Java_Type)ty) ;
	switch (type) {
	case 'B':	// boolean or 8-bit signed
	case 'C':	// 16-bit unsigned char
	case 'S':	// signed 16-bit short
        {
        unsigned shift =  (type == 'B') ? 0 : 1;
#if 0
		Inst *ld_array = exprs.lookup_array(base,indx,bnd,shift,/*8*/array_offset,type,inst_head);
        return exprs.lookup_inst(Exp::Widen,ld_array,NULL,type,inst_head);
#else
        Array_Exp *arry_exp = exprs.lookup_array_exp(base->exp,indx->exp,bnd->exp,shift,/*8*/array_offset,type);
        Inst_Exp* widen_exp = exprs.lookup_inst_exp(Exp::Widen,arry_exp,NULL,type);
        Array_Operand *array_opnd = create_array_operand(mem,base,indx,bnd,shift,/*8*/array_offset,type,true);
        Inst tmp;
        tmp.set_dst(array_opnd);
        return exprs.gen_inst_tuple(inst_head,widen_exp,Exp::Widen,&tmp,NULL);
#endif
        }
		break;
	case 'F':	// single FP
	case 'I':	// 32-bit int
	case 'L':	// object reference
		return exprs.lookup_array(base,indx,bnd,2,/*8*/array_offset,type,inst_head);
		break;
	case 'D':	// double FP
	case 'J':	// 64-bit long
		return exprs.lookup_array(base,indx,bnd,3,/*8*/array_offset,type,inst_head);
        break;
	}
    assert(0);
    return NULL;
}
#endif

void array_store(Mem_Manager& mem, Expressions& exprs,
                 Stack& stack, Inst *inst_head, char ty) {
	Inst *src  = stack.pop();
	Inst *indx = stack.pop();
	Inst *base = stack.pop();

    bool gen_aastore_call =
#ifdef ORP_DOES_AASTORE
        (O3_Jit_Type)ty == JIT_TYPE_CLASS && !src->exp->is_imm_exp();
#else
        false;
#endif // ORP_DOES_AASTORE

    //
    // bound checking code
    //
    Inst *len = exprs.lookup_inst(Exp::Length,base,NULL,JIT_TYPE_INT,inst_head);
    Inst *bnd = gen_array_cmp_inst(mem,exprs,inst_head,indx,len,JIT_TYPE_INT,!gen_aastore_call);

    unsigned shift;
	//:: Change the number 8 -- array_offset , to a VM interface get_array_offset()
	unsigned array_offset = get_array_offset((Java_Type)ty) ;
    switch (ty) {
	case 'B':	// boolean or 8-bit signed
		// mov8 [ebx+ecx],eax
		// To do! src must be byte addressable reg
        shift = 0; break;
	case 'C':	// 16-bit unsigned char
	case 'S':	// signed 16-bit short
        shift = 1; break;
	case 'F':	// single FP
	case 'I':	// 32-bit int
	case 'L':  	// object reference
        shift = 2; break;
    case 'D':   // double FP
	case 'J':   // long 
        shift = 3; break;
	default:
		assert(0);
	}
#ifndef JIT_SAPPHIRE
	Exp *array;
    Array_Operand *array_opnd;
    //
    // create array expression and operand
    //
	O3_Jit_Type type = (O3_Jit_Type) ty;
    array = exprs.lookup_array_exp(base->exp,indx->exp,bnd->exp,shift,/*8*/array_offset,type);
    array_opnd = create_array_operand(mem,base,indx,bnd,shift,/*8*/array_offset,type,!gen_aastore_call);
    //
    // generate array assignment
    //
    create_assign_inst(mem,exprs,inst_head,array_opnd,array,src,type);
    if (gen_aastore_call) {
        bnd->mark_dead();  // helper functions makes the bounds-check implicitly available
        assert(inst_head->prev()->is_assignment());
        ((Assign_Inst*)inst_head->prev())->set_aastore_call();
    }
#else
	Exp *array;
	Array_Operand *array_opnd;
	//
	// create array expression and operand
	//
	O3_Jit_Type type = (O3_Jit_Type) ty;
	array = exprs.lookup_array_exp(base->exp,indx->exp,bnd->exp,shift,/*8*/array_offset,type);
	array_opnd = create_array_operand(mem,base,indx,bnd,shift,/*8*/array_offset,type,!gen_aastore_call);
	//
	// generate array assignment
	//
	create_assign_inst(mem,exprs,inst_head,array_opnd,array,src,type);
    if (gen_aastore_call) {
		bnd->mark_dead();  // helper functions makes the bounds-check implicitly available
		assert(inst_head->prev()->is_assignment());
		((Assign_Inst*)inst_head->prev())->set_aastore_call();
    } else {

		// FIX this!  use lea later
		Inst *s   = exprs.lookup_imm(shift,JIT_TYPE_INT,inst_head);
		Inst *i8  = exprs.lookup_imm(/*8*/array_offset,JIT_TYPE_INT,inst_head);
		Inst *shf = exprs.lookup_inst(Exp::Shl,indx,s,JIT_TYPE_INT,inst_head);
		Inst *off = exprs.lookup_inst(Exp::Add,shf,i8,JIT_TYPE_INT,inst_head);
		Exp *wb_exp = exprs.lookup_inst_exp(Exp::WriteBarrier, base->exp, 
											src->exp, JIT_TYPE_VOID);
		new(mem) Writebarrier_Inst(base->dst(), off->dst(), src->dst(), 
			                       wb_exp, type, inst_head);
	}
#endif
}

//
// reset pseudo assignments (t = reg_id)
//
void disable_pseudo_assignment(Stack& stack, unsigned reg_id) {
    unsigned i;
    for (i = 0; i < stack.top(); i++) {
        Inst *elem = stack.ith_elem(i);
        if (elem->is_assignment() && elem->src(0)->is_reg() && 
            ((Reg_Operand*)elem->src(0))->id == reg_id)
            ((Assign_Inst*)elem)->disable_pseudo_asgn();
    }
}

//
// generate an assignement inst "vreg = src"
//
void gen_assign(Mem_Manager& mem,  Expressions& exprs, 
                Stack& stack,      Inst *inst_head,
				unsigned  var_no,  Inst   *src, 
				O3_Jit_Type ty) {
    if (src->type() == JIT_TYPE_RETADDR)
        ty = JIT_TYPE_RETADDR;
	unsigned reg_id = exprs.reg_map.virtual_reg_id(var_no,ty);
    Operand_Exp *vreg = exprs.lookup_reg_exp(reg_id,ty,1); // lookup virtual reg

    create_assign_inst(mem,exprs,inst_head,vreg->opnd,vreg,src,ty);
#if 1
    disable_pseudo_assignment(stack,reg_id);
#endif
}

//
// generate t = src0 op src1
//
void gen_alu(Mem_Manager& mem_manager, Expressions& exprs, 
               Stack& stack, Inst *inst_head, Exp::Kind op, char ty) {
    O3_Jit_Type type = (O3_Jit_Type)ty;
    Inst *src1 = stack.pop();
    Inst *src0 = stack.pop();
    stack.push(exprs.lookup_inst(op,src0,src1,type,inst_head));
}

//
// generate t = op src
//
void gen_alu1(Mem_Manager& mem_manager, Expressions& exprs, 
               Stack& stack, Inst *inst_head, Exp::Kind op) {
    Inst *src = stack.pop();
    stack.push(exprs.lookup_inst(op,src,NULL,src->exp->type,inst_head));
}

//
// return src
// if src is NULL, then the method have a void return type
//
void gen_return(Mem_Manager& mem,Expressions& exprs,Stack& stack,
                Inst *inst_head, Inst *src){
    if (src != NULL)  { // not a void return
        Java_Type jt = (Java_Type)src->type();
        CONVERT_TO_INT_JAVA(jt);
        O3_Jit_Type src_ty = (O3_Jit_Type) jt;
        Operand_Exp *ret_val = exprs.lookup_ret_exp(src_ty);
        create_assign_inst(mem,exprs,inst_head,ret_val->opnd,ret_val,src,src_ty);
    }
}

//
// duplicate the top of the stack
//
void gen_dup(Mem_Manager& mem, Expressions& exprs, Stack& stack, Inst *inst_head) {
#if 0
    Inst *src = stack.pop();
    Exp  *exp = src->exp;
	Inst *i = new (mem) Assign_Inst(exprs.create_new_temp_reg(exp->type),src->dst(),exp,inst_head);
    stack.push(i);
    stack.push(src);
#else // 0
    Reg_Operand *treg = (Reg_Operand *) stack.ith_elem(0)->dst();
    assert(stack.ith_elem(0)->dst()->is_reg());
    stack.push(exprs.gen_opnd_tuple(inst_head, exprs.lookup_temp_reg_exp(treg)));
    //stack.push(exprs.lookup_stack(0, stack.ith_elem(0)->type(), inst_head));
#endif // 0
}

//
// duplicate the top of the stack and put three down
//
void gen_dup_x1(Mem_Manager& mem, Expressions& exprs, Stack& stack, Inst *inst_head) {
    Inst *word1 = stack.pop();
    Inst *word2 = stack.pop();

    Exp  *exp = word1->exp;
	Inst *i = new (mem) Assign_Inst(exprs.create_new_temp_reg(exp->type),word1->dst(),exp,inst_head);
    stack.push(i);
    stack.push(word2);
    stack.push(word1);
}

//
// duplicate the top of the stack and put three down
//
void gen_dup_x2(Mem_Manager& mem, Expressions& exprs, Stack& stack, Inst *inst_head) {
    Inst *word1 = stack.pop();
    Inst *word2 = stack.pop();
    Inst *word3 = (word2->is_64bit())? NULL : stack.pop();

    Exp  *exp = word1->exp;
	Inst *i = new (mem) Assign_Inst(exprs.create_new_temp_reg(exp->type),word1->dst(),exp,inst_head);
    stack.push(i);
    if (word3 != NULL) stack.push(word3);
    stack.push(word2);
    stack.push(word1);
}

//
// duplicate word1 and word2, then push duplicate word2 and word1
//
static void duplicate_2(Mem_Manager& mem,Expressions& exprs, Stack& stack, Inst* inst_head,
                        Inst *word1, Inst *word2) {
    Exp *exp;
    Temp_Reg *new_tmp;
    if (word2 != NULL) { // word2 != NULL means that word1 is a 32-bit value
        exp = word2->exp;
        new_tmp = exprs.create_new_temp_reg(exp->type);
        stack.push(new (mem) Assign_Inst(new_tmp,word2->dst(),exp,inst_head));
    }
    exp = word1->exp;
    new_tmp = exprs.create_new_temp_reg(exp->type);
    stack.push(new (mem) Assign_Inst(new_tmp,word1->dst(),exp,inst_head));
}

// 
// duplicate the top 2 elements (two 32-bit)
//
void gen_dup2(Mem_Manager& mem,Expressions& exprs, Stack& stack, Inst* inst_head) {
    //
    // check if top element is a 64 bit value
    //
    Inst *word1 = stack.pop();
    Inst *word2 = (word1->is_64bit())? NULL : stack.pop();

    duplicate_2(mem,exprs,stack,inst_head,word1,word2);

    if (word2 != NULL) stack.push(word2);
    stack.push(word1);
}

// 
// duplicate the top 2 elements (two 32-bit) and push 3 down
//
void gen_dup2_x1(Mem_Manager& mem,Expressions& exprs, Stack& stack, Inst* inst_head) {
    //
    // check if top element is a 64 bit value
    //
    Inst *word1 = stack.pop();
    Inst *word2 = (word1->is_64bit())? NULL : stack.pop();
    Inst *word3 = stack.pop(); assert(!word3->is_64bit());

    duplicate_2(mem,exprs,stack,inst_head,word1,word2);

    stack.push(word3);
    if (word2 != NULL) stack.push(word2);
    stack.push(word1);
}

// 
// duplicate the top 2 elements (two 32-bit) and push 4 down
//
void gen_dup2_x2(Mem_Manager& mem,Expressions& exprs, Stack& stack, Inst* inst_head) {
    //
    // check if top element is a 64 bit value
    //
    Inst *word1 = stack.pop();
    Inst *word2 = (word1->is_64bit())? NULL : stack.pop();
    Inst *word3 = stack.pop();
    Inst *word4 = (word3->is_64bit())? NULL : stack.pop();

    duplicate_2(mem,exprs,stack,inst_head,word1,word2);

    if (word4 != NULL) stack.push(word4);
    stack.push(word3);
    if (word2 != NULL) stack.push(word2);
    stack.push(word1);
}

//
// generate x = x + inc
//
void gen_iinc(Mem_Manager& mem, Expressions& exprs, Stack& stack, Inst *inst_head,
              unsigned var_no, int inc) {
#if 1
    O3_Jit_Type ty = JIT_TYPE_INT;
    unsigned reg_id = exprs.reg_map.virtual_reg_id(var_no,ty);

    Operand_Exp *var = exprs.lookup_reg_exp(reg_id,ty,1); // lookup virtual reg
    //
    // x = 0
    // x = x + 1  ==>  x = 1
    //
    if (var->inst() != NULL && var->inst()->is_imm_assignment())
    {
        int c = ((Imm_Operand*)var->inst()->src(0))->imm() + inc;
        Inst *imm = exprs.lookup_imm(c, ty, inst_head);
        gen_assign(mem, exprs, stack, inst_head, var_no, imm, ty);
    }
    else
    {
        Operand_Exp *imm = exprs.lookup_imm_exp(inc,JIT_TYPE_INT);
        // generate x = x + inc
        Exp *plus = exprs.lookup_inst_exp(Exp::Add,var,imm,JIT_TYPE_INT);
        Inst *i = exprs.local_cse(plus);
        if (i == NULL) {
            Exp *asgn = exprs.lookup_inst_exp(Exp::Assign,var,plus,JIT_TYPE_INT);
            i = new (mem) Add_Inst(Add_Inst::add,var->opnd,imm->opnd,asgn,inst_head);
            i->set_dst(var->opnd);
            ((Exp *)var)->asgn_kill_local_cse(exprs.live_lcse());
            asgn->compute_kill(mem, exprs.kill_id_map.get_curr_kill_id());
            disable_pseudo_assignment(stack,reg_id);
        } else {
            gen_assign(mem, exprs, stack, inst_head, var_no, i, JIT_TYPE_INT);
        }
    }
#else
    // generate exp for virtual reg
    Inst *ld_var = exprs.lookup_vreg(var_no,JIT_TYPE_INT,inst_head); 
    // generate inc
    Inst *ld_inc = exprs.lookup_imm(inc,inst_head);
    Inst *add    = exprs.lookup_inst(Exp::Add,ld_var,ld_inc,JIT_TYPE_INT,inst_head);
    // generate x = x + inc
    gen_assign(mem, exprs, stack, inst_head, var_no, add, JIT_TYPE_INT);
#endif
}

static Java_Type get_constant_value(Class_Handle ch, unsigned index, Value *val)
{
    Java_Type t = class_get_const_type(ch, index);

    if (t == JAVA_TYPE_STRING) {
        return JAVA_TYPE_STRING;
    } else {
        const uint32 *a = (const uint32 *)class_get_const_addr(ch, index);
        uint32 *dst = (uint32 *)val;
        *dst = *a;
        if (t == JAVA_TYPE_DOUBLE || t == JAVA_TYPE_LONG)
            *++dst = *++a;
        return t;
    }
}

//
// load constant from constant pool
//
void gen_const(Mem_Manager& mem, Expressions& exprs, Stack& stack, Inst *inst_head,
			   Compile_Handle comp_handle,Class_Handle class_handle, unsigned index) {
	Value	val;
    Java_Type type = (Java_Type) get_constant_value(class_handle, index, &val);
	switch (type) {
    case JAVA_TYPE_STRING:
    case JAVA_TYPE_CLASS: 
    {
        Inst *pool_idx = exprs.lookup_imm(index,JIT_TYPE_INT,inst_head);
        Inst *handle   = exprs.lookup_imm((unsigned)class_handle,JIT_TYPE_ADDR,inst_head);
        stack.push(exprs.lookup_inst(Exp::String,pool_idx,handle,JIT_TYPE_CLASS,inst_head));
		break;
    }
	case JAVA_TYPE_INT:
		stack.push(exprs.lookup_imm(val.i,JIT_TYPE_INT,inst_head));
		break;
	case JAVA_TYPE_FLOAT:
        stack.push(exprs.lookup_const(&val,JIT_TYPE_FLOAT,inst_head));
        break;
	case JAVA_TYPE_DOUBLE:
        stack.push(exprs.lookup_const(&val,JIT_TYPE_DOUBLE,inst_head));
        break;
	case JAVA_TYPE_LONG:
		stack.push(exprs.lookup_const(&val,JIT_TYPE_LONG,inst_head));
		break;
    default:
        assert(0);
        break;
	}
}

Inst *gen_test_inst(Mem_Manager& mem, Expressions& exprs, Inst *inst_head, Inst *src) {
    Exp *test = exprs.lookup_inst_exp(Exp::Test,src->exp,NULL,/*JIT_TYPE_INT*/src->type());
    Inst *i = new (mem) Compare_Inst(Compare_Inst::test,false,src->dst(),test,false,inst_head);
    unsigned id = exprs.reg_map.get_tmp_reg_id(0);
    Status_Flags *status = new (mem) Status_Flags(id,JIT_TYPE_INT/*src->type()*/);
    i->set_dst(status);
    return i;
}

static Inst *new_cmp_inst(Mem_Manager& mem,Expressions& exprs,Inst *inst_head, 
                          Exp *cmp, Inst *src0, Inst *src1, O3_Jit_Type ty, bool is_cmpg,
                          bool cse_for_this) {
    //
    // create compare inst and set local_cse
    //
    Inst *i = new (mem) Compare_Inst(Compare_Inst::cmp,is_cmpg,src0->dst(),src1->dst(),
                                     cmp, cse_for_this, inst_head);
    Status_Flags *status = new (mem) Status_Flags(exprs.reg_map.get_tmp_reg_id(0),JIT_TYPE_INT/*ty*/);
    i->set_dst(status);
    //
    // if we want this cmp to be a cse for other subsequent cmps (array
    // bound checking) then we generate a local_cse record for the cmp.
    //
    if (ty == JIT_TYPE_INT && exprs.live_lcse() != NULL) {
        exprs.live_lcse()->kill_lcse_contain(cmp);
        exprs.gen_local_cse(cmp,i);
    }
    return i;
}
//
// generate bounds checking  
//
Inst *gen_array_cmp_inst(Mem_Manager& mem, Expressions& exprs, Inst *inst_head, 
                         Inst *src0, Inst *src1, O3_Jit_Type ty,
                         bool cse_for_this) {
    Exp *cmp = exprs.lookup_array_cmp_exp(src0->exp,src1->exp,ty);
    if (cse_for_this && exprs.is_local_cse(cmp) != 0)
        return cmp->inst();  // local cse found
    return new_cmp_inst(mem, exprs, inst_head, cmp, src0, src1, ty, false, cse_for_this);
}
//
// generate compare 
//
Inst *gen_cmp_inst(Mem_Manager& mem, Expressions& exprs, Inst *inst_head, 
                   Inst *src0, Inst *src1, O3_Jit_Type ty, bool is_cmpg) {
    //
    // we don't want compare inst to be cse for compare-branch because we 
    // need to worry about which instructions set status flags.
    //
    Exp *cmp = exprs.lookup_inst_exp(Exp::Compare,src0->exp,src1->exp,ty);
    return new_cmp_inst(mem, exprs, inst_head, cmp, src0, src1, ty, is_cmpg, false);
}

extern bool O3_unsafe_fcmp;
void gen_cmp_branch(Mem_Manager& mem, Expressions& exprs, Stack& stack, 
             Inst *inst_head, Exp::Kind op, O3_Jit_Type ty, bool is_cmpg,
             char *sig_out,  // stack type signature of current BB
             int  &sig_out_size) {
    Inst *src1 = stack.pop();
    Inst *src0 = stack.pop();
    //
    // spill whatever left on the mimic stack
    //
    spill_left_on_stack(mem,exprs,stack,inst_head,sig_out,sig_out_size);
    Inst *src = gen_cmp_inst(mem,exprs,inst_head,src0,src1,ty, is_cmpg);
    // By default, use a signed comparison.
    bool signed_branch = true;
    // Don't use a signed branch for floating point comparisons, because the floating
    // point status flags are different from the integer status flags.
    if (IS_FP_DBL_TYPE(ty))
        signed_branch = false;
    // For a long comparison, we first generate a single compare and a single branch.
    // Later, during expansion, we expand only the compare instruction, into a compare
    // of the high operands and two signed branches.  The original branch doesn't get
    // expanded, and must be unsigned because it compares the low operands.
    if (ty == JIT_TYPE_LONG)
        signed_branch = false;
    gen_branch(mem,exprs,inst_head,signed_branch,op,src);
    if (ty == JIT_TYPE_LONG || (!O3_unsafe_fcmp && IS_FP_DBL_TYPE(ty)))
        ((Branch_Inst *)inst_head->prev())->dont_commute(); // XXX- hack; should pass ty to gen_branch()
}

#ifdef JIT_SAPPHIRE
//
// forward declaration
//
static void push_arg(Mem_Manager& mem, Expressions& exprs, Inst *inst_head,
                     unsigned ith_arg, Inst *src, O3_Jit_Type ty,
                     Inst **argarray, unsigned &argarraysize);

void gen_acmp_branch(Mem_Manager& mem, 
                     Expressions& exprs, 
                     Stack& stack, 
                     Inst *inst_head, 
                     Exp::Kind op, 
                     char *sig_out,  // stack type signature of current BB
                     int  &sig_out_size) {
    // only for acmpeq and acmpne
    assert(op == Exp::Beq || op == Exp::Bne);

    Inst *src1 = stack.pop();
    Inst *src0 = stack.pop();
    //
    // if one of the sources is NULL, the code generated using gen_cmp_branch() 
    // is still valid for sapphire
    //
    if (src1->is_imm_assignment() || src0->is_imm_assignment())
    {
        stack.push(src0);
        stack.push(src1);
        gen_cmp_branch(mem,exprs,stack,inst_head,op,JIT_TYPE_CLASS,false,
                       sig_out,sig_out_size);
        return;
    }
    //
    // spill whatever left on the mimic stack
    //
    spill_left_on_stack(mem,exprs,stack,inst_head,sig_out,sig_out_size);
    //
    // create an array for holding arguments (2 args)
    //
    Inst **argarray = (Inst**)mem.alloc(2 * sizeof(*argarray));
    //
    // place src0 and src1 in argarray[]
    //
    unsigned argarraysize = 0;
    push_arg(mem,exprs,inst_head,0,src0,JIT_TYPE_CLASS,argarray,argarraysize);
    push_arg(mem,exprs,inst_head,1,src1,JIT_TYPE_CLASS,argarray,argarraysize);
    //
    // create acmp_call instruction
    //
	Inst_Exp *call_exp = exprs.lookup_inst_exp(Exp::Call,src0->exp,src1->exp,JIT_TYPE_INT);
    Call_Inst *cll = new(mem) Call_Inst(Call_Inst::acmp_call, call_exp, NULL, false, inst_head);
	//
    // caller-pop; adjust esp by adding n_srcs*4
	//
    Operand_Exp *esp = exprs.lookup_reg_exp(esp_reg,JIT_TYPE_INT,0);
    Operand_Exp *imm = exprs.lookup_imm_exp(argarraysize*4, JIT_TYPE_INT);
    Inst_Exp *add = exprs.lookup_inst_exp(Exp::Add,esp,imm,JIT_TYPE_INT);
    Inst *i = new (mem) Add_Inst(Add_Inst::add,esp->opnd,imm->opnd,add,inst_head);
    i->set_dst(esp->opnd);
    i->dont_eliminate();
    //
    // create ret instruction
    //
    Inst *ret = exprs.lookup_ret(JIT_TYPE_INT,inst_head);
    //
    // set args, and ret for the call
    //
    cll->set_args(argarray, 2, ret);
    //
    // generate "ret == 1" (acmp_eq) or "ret == 0" (acmp_ne)
    // we push ret and eq (1 or 0) onto the stack so as to 
    // use gen_cmp_branch to generate the code
    //
    Inst *eq = exprs.lookup_imm((op == Exp::Beq) ? 1 : 0, JIT_TYPE_INT, inst_head);
    stack.push(ret);
    stack.push(eq);
    gen_cmp_branch(mem,exprs,stack,inst_head,Exp::Beq,JIT_TYPE_INT,false,
                   sig_out,sig_out_size);
}
#endif

//
// operands left on the mimic stack are spilled at the end of BB
//
void spill_left_on_stack(Mem_Manager& mem, Expressions& exprs, Stack& stack, 
                      Inst *inst_head, char *stack_sig, int& stack_sig_size) {
    stack_sig_size = stack.top();  // long/double are counted as one
    //
    // generate Stk = ...
    //
    Operand_Exp *stk;
    O3_Jit_Type ty;
    int i, depth;
    for (i = 0, depth = 0; i < stack_sig_size; i++) {
        //
        // don't pop because we want them to live across extended BB
        //
        Inst *src = stack.ith_elem(stack_sig_size - i - 1);
        ty = src->type();
        stk = exprs.lookup_reg_exp(exprs.reg_map.stack_reg_id(depth,ty),ty,1);
        create_assign_inst(mem,exprs,inst_head,stk->opnd,stk,src,ty);

        stack_sig[i] = (char)ty; // record stack type signature
        depth += (src->is_64bit())? 2 : 1;
    }
}
//
// generate branch against zero 
//
void gen_branch(Mem_Manager& mem, Expressions& exprs, Inst *inst_head, bool is_signed,
                Exp::Kind op, Inst *src) {
    //
    // lookup branch and create instruction
    //
    Inst_Exp *br = exprs.lookup_inst_exp(op,src->exp,NULL,JIT_TYPE_VOID);
    Branch_Inst::Kind bop = (Branch_Inst::Kind)(Branch_Inst::beq + (op - Exp::Beq));
    new (mem) Branch_Inst(bop,/*true*/is_signed,src->dst(),br,inst_head); 
}

//
// generate get field ( T = [b + off] )
//
void gen_getfield(Mem_Manager &mem, Expressions& exprs, Stack& stack, Inst *inst_head,
                  Compile_Handle comp_handle,Class_Handle class_handle, unsigned index){
    //
    // will be replaced with our VM2JIT interface
    //
    Loader_Exception exc;
	Field_Handle fh = resolve_nonstatic_field(comp_handle,class_handle,index,&exc);
	unsigned off = field_get_offset(fh);
	Java_Type field_type = field_get_type(fh);
	
	Inst *base = stack.pop();
#ifdef JIT_SAPPHIRE
	Operand_Exp* imm_exp = (Operand_Exp*)exprs.lookup_imm_exp(off, JIT_TYPE_INT) ;
	Exp *rb_exp = exprs.lookup_inst_exp(Exp::ReadBarrier, base->exp, 
		                                imm_exp, JIT_TYPE_VOID);
	new(mem) Readbarrier_Inst(base->dst(), imm_exp->opnd, rb_exp, (O3_Jit_Type)field_type, inst_head);
#endif

    CONVERT_TO_INT_JAVA(field_type);
	stack.push(exprs.lookup_field(base,off,(O3_Jit_Type)field_type,inst_head,(FIELD_UID)fh));
}

//
// generate put field ( [b + off] = ... )
//
#ifndef JIT_SAPPHIRE
void gen_putfield(Mem_Manager &mem, Expressions& exprs, Stack& stack, 
           Inst *inst_head,Compile_Handle comp_handle,Class_Handle class_handle,
           unsigned index, bool gc_requires_write_barriers){
    //
    // will be replaced with our VM2JIT interface
    //
    Loader_Exception exc;
	Field_Handle fh = resolve_nonstatic_field(comp_handle,class_handle,index,&exc);
	unsigned off = field_get_offset(fh);
	O3_Jit_Type field_ty = (O3_Jit_Type) field_get_type(fh);
	
	Inst *src  = stack.pop();
	Inst *base = stack.pop();
    //
	// lookup [b+off] expression
    //
	Field_Exp *exp = exprs.lookup_field_exp(base->exp,off,field_ty,(FIELD_UID)fh);
	Field_Operand *field = create_field_operand(mem,base,off,field_ty,(FIELD_UID)fh);
    //
	// generate assignment
    //
    create_assign_inst(mem,exprs,inst_head,field,exp,src,field_ty);

#ifndef DISABLE_GC_WRITE_BARRIERS
    if (gc_requires_write_barriers)
    {
        assert(src->type() != JIT_TYPE_ARRAY);
        if (src->type() == JIT_TYPE_CLASS)
        {
            Exp *write_barrier_exp =
                exprs.lookup_inst_exp(Exp::WriteBarrier, exp, NULL, JIT_TYPE_CLASS);
            new(mem) Writebarrier_Inst(field, src->dst(), write_barrier_exp, inst_head);
        }
    }
#endif // DISABLE_GC_WRITE_BARRIERS
}
#else 
void gen_putfield(Mem_Manager &mem, Expressions& exprs, Stack& stack, 
           Inst *inst_head,Compile_Handle comp_handle,Class_Handle class_handle,
           unsigned index, bool gc_requires_write_barriers){
    //
    // will be replaced with our VM2JIT interface
    //
    Loader_Exception exc;
	Field_Handle fh = resolve_nonstatic_field(comp_handle,class_handle,index,&exc);
	unsigned off = field_get_offset(fh);
	O3_Jit_Type field_ty = (O3_Jit_Type) field_get_type(fh);
	
	Inst *src  = stack.pop();
	Inst *base = stack.pop();
    Operand *imm  = ((Operand_Exp*)exprs.lookup_imm_exp(off, JIT_TYPE_INT))->opnd;
	Exp *wb_exp = exprs.lookup_inst_exp(Exp::WriteBarrier, base->exp, 
		                                src->exp, JIT_TYPE_VOID);
	new(mem) Writebarrier_Inst(base->dst(), imm, src->dst(), 
		                       wb_exp, field_ty, inst_head);
    //
	// lookup [b+off] expression
    //
	Field_Exp *exp = exprs.lookup_field_exp(base->exp,off,field_ty,(FIELD_UID)fh);
	Field_Operand *field = create_field_operand(mem,base,off,field_ty,(FIELD_UID)fh);
    //
	// generate assignment
    //
    create_assign_inst(mem,exprs,inst_head,field,exp,src,field_ty);
}
#endif

void gen_getstatic(Mem_Manager &mem, Expressions& exprs, Stack& stack, Inst *inst_head,
                  Compile_Handle comp_handle,Class_Handle class_handle, unsigned index){
    //
    // will be replaced with our VM2JIT interface
    //
    Loader_Exception exc;
	Field_Handle fh = resolve_static_field(comp_handle,class_handle,index,&exc);
	Java_Type field_type = field_get_type(fh);

    Class_Handle field_class = field_get_class(fh);
    if (field_class != class_handle && !class_is_initialized(field_class)) {
        Inst *fc_inst = exprs.lookup_imm((unsigned)field_class, JIT_TYPE_ADDR, inst_head);
        Exp *expr = exprs.lookup_inst_exp(Exp::ClassInit, fc_inst->exp, NULL, JIT_TYPE_VOID);
        new(mem) Classinit_Inst(fc_inst->dst(), expr, inst_head);
    }

    void *addr = field_get_addr(fh);
    CONVERT_TO_INT_JAVA(field_type);
    O3_Jit_Type ft = (O3_Jit_Type) field_type;
	
#ifdef JIT_SAPPHIRE

	if (ft == JIT_TYPE_CLASS) {
		Inst *base = exprs.lookup_imm((unsigned)addr,JIT_TYPE_ADDR,inst_head);
		Exp *rb_exp = exprs.lookup_inst_exp(Exp::ReadBarrier, base->exp, NULL,JIT_TYPE_VOID);
		Inst* tmp = new(mem) Readbarrier_Inst(base->dst(), rb_exp, inst_head);
		/* version 1
		Inst *base = exprs.lookup_imm((unsigned)addr,JIT_TYPE_ADDR,inst_head);
		Operand_Exp *result_exp = exprs.lookup_static_exp(addr,ft,addr, fh);
		Exp *wb_exp = exprs.lookup_inst_exp(Exp::ReadBarrier, base->exp, 
											result_exp, JIT_TYPE_VOID);
		Inst* tmp = new(mem) Readbarrier_Inst(base->dst(), result_exp->opnd, wb_exp, inst_head);
		Inst* result = ((Exp*)result_exp)->build_inst(exprs,inst_head) ;
		stack.push(result);
		*/
		/* version 2
		//Temp_Reg* tmp = exprs.create_new_temp_reg(ft) ; 
		//Operand_Exp* r_exp = exprs.lookup_temp_reg_exp(tmp) ;//(eax_reg,JIT_TYPE_ADDR,0) ;//tmp->inst() ;//lookup_exp(exprs,inst_head) ;
		//Inst* r_inst = exprs.lookup_inst_exp(Exp::Assign,r_exp,NULL,ft) ;

		Inst *base = exprs.lookup_imm((unsigned)addr,JIT_TYPE_ADDR,inst_head);
		Inst* result = exprs.lookup_static(addr,ft,inst_head, fh) ; //lookup_field(base,0,JIT_TYPE_ADDR,inst_head,(FIELD_UID)fh) ; //
		//Operand_Exp *result_exp = exprs.lookup_static_exp(addr,ft,addr, fh);
		Exp *wb_exp = exprs.lookup_inst_exp(Exp::ReadBarrier, base->exp, 
											result->exp, JIT_TYPE_VOID);
		//Inst* tmp = new(mem) Readbarrier_Inst(base->dst(), result_exp->opnd, wb_exp, inst_head);
		//Inst* result = ((Exp*)result_exp)->build_inst(exprs,inst_head) ;

	    //new(exprs.mem) Deref_Inst(base->dst(), base->exp, inst_head);
	    //create_assign_inst(mem,exprs,inst_head,r_exp->opnd,r_exp,base,(O3_Jit_Type)field_type);
		new(mem) Readbarrier_Inst(base->dst(), result->dst(), wb_exp, inst_head); 
		stack.push(result);
		*/
	}
#endif

	stack.push(exprs.lookup_static(addr,ft,inst_head, fh));

}

void gen_putstatic(Mem_Manager &mem, Expressions& exprs, Stack& stack, 
           Inst *inst_head,Compile_Handle comp_handle,Class_Handle class_handle, unsigned index){
    //
    // will be replaced with our VM2JIT interface
    //
    Loader_Exception exc;
	Field_Handle fh = resolve_static_field(comp_handle,class_handle,index,&exc);
	Java_Type field_type = field_get_type(fh);

    Class_Handle field_class = field_get_class(fh);
    if (field_class != class_handle && !class_is_initialized(field_class)) {
        Inst *fc_inst = exprs.lookup_imm((unsigned)field_class, JIT_TYPE_ADDR, inst_head);
        Exp *expr = exprs.lookup_inst_exp(Exp::ClassInit, fc_inst->exp, NULL, JIT_TYPE_VOID);
        new(mem) Classinit_Inst(fc_inst->dst(), expr, inst_head);
    }

    void *addr = field_get_addr(fh);
	
	Inst *src  = stack.pop();
	Operand_Exp *exp = exprs.lookup_static_exp(addr,(O3_Jit_Type)field_type,addr, fh);
    //
	// generate assignment
    //
#ifndef JIT_SAPPHIRE
    create_assign_inst(mem,exprs,inst_head,exp->opnd,exp,src,(O3_Jit_Type)field_type);
#else
	if (((O3_Jit_Type)field_type) == JIT_TYPE_CLASS) {
		Inst *base = exprs.lookup_imm((unsigned)addr,JIT_TYPE_ADDR,inst_head);
		Exp *wb_exp = exprs.lookup_inst_exp(Exp::WriteBarrier, base->exp, 
											src->exp, JIT_TYPE_VOID);
		new(mem) Writebarrier_Inst(base->dst(), src->dst(), wb_exp, inst_head);
	} 
	create_assign_inst(mem,exprs,inst_head,exp->opnd,exp,src,(O3_Jit_Type)field_type);
#endif
}

//
// convert src to the value of the type conv_to
//
void gen_conversion(Mem_Manager& mem, Expressions& exprs, Stack& stack,
                    Inst *inst_head, O3_Jit_Type conv_to, O3_Jit_Type conv_fm) {
    Inst *src = stack.pop();
    assert(src->exp->type == conv_fm || conv_fm == JIT_TYPE_INT);
    Inst_Exp *conv = exprs.lookup_inst_exp(Exp::Convt,src->exp,NULL,conv_to);
    stack.push(exprs.gen_inst_tuple(inst_head,conv,Exp::Convt,src,NULL));
}

//
// return from subroutine
//
void gen_ret(Mem_Manager& mem, Expressions& exprs, Inst *inst_head, unsigned var_no) {
    Inst *ret_addr = exprs.lookup_vreg(var_no,JIT_TYPE_RETADDR,inst_head);
    Inst_Exp *jmp = exprs.lookup_inst_exp(Exp::Jump,ret_addr->exp,NULL,JIT_TYPE_VOID);
    new (mem) Jump_Inst(Jump_Inst::jump_s,ret_addr->dst(),jmp,inst_head);
}

//
// generate switch instruction
//
void gen_switch(Mem_Manager& mem, Expressions& exprs, Inst *inst_head, 
                Exp::Kind op, Inst *src, Cfg_Extra_Info *switch_info) {
    assert(switch_info);
    Inst_Exp *sw = exprs.lookup_inst_exp(op,src->exp,NULL,JIT_TYPE_VOID);
    Switch_Inst::Kind swop = 
    (op == Exp::Tableswitch)? Switch_Inst::table : Switch_Inst::lookup;
    exprs.inc_data_space(switch_info->data_space_needed());
    new (mem) Switch_Inst(swop,src->dst(),sw,inst_head,switch_info);
}

//
// Try to generate a native (or whatever) instruction.
//
static bool gen_intrinsic(Mem_Manager& mem, Expressions& exprs, Stack& stack,
                          Inst *inst_head, Method_Handle handle)
{
    //return false;  // disable
    // A native method doesn't have any bytecodes.
    if (method_get_class(handle) != cached_class_handles[ch_java_lang_Math])
        return false;
    Exp::Kind kind;
    if (method_get_byte_code_size(handle) > 0)
    {
        if (handle != cached_method_handles[mh_java_lang_Math_abs__1] &&
            handle != cached_method_handles[mh_java_lang_Math_abs__2])
            return false;
        kind = Exp::Intr_Abs;
    }
    else
    {
        //unsigned size = n_words_of_method_arg_type(handle);
        if (handle == cached_method_handles[mh_java_lang_Math_sin])
            kind = Exp::Intr_Sin;
        else if (handle == cached_method_handles[mh_java_lang_Math_cos])
            kind = Exp::Intr_Cos;
        else if (handle == cached_method_handles[mh_java_lang_Math_sqrt])
            kind = Exp::Intr_Sqrt;
        else if (handle == cached_method_handles[mh_java_lang_Math_rint])
            kind = Exp::Intr_Rndint;
        else
            return false;
    }
    O3_Jit_Type type = (O3_Jit_Type) method_get_return_type(handle);
    Inst *src0;
    switch (kind)
    {
    case Exp::Intr_Sin:
    case Exp::Intr_Cos:
    case Exp::Intr_Sqrt:
    case Exp::Intr_Rndint:
    case Exp::Intr_Abs:
        src0 = stack.pop();
        stack.push(exprs.lookup_inst(kind, src0, NULL, type, inst_head));
        break;
    default:
        return false;
        break;
    }

    return true;
}

//
// push one argument ( arg# = ...)
//
static void push_arg(Mem_Manager& mem, Expressions& exprs, Inst *inst_head,
                     unsigned ith_arg, Inst *src, O3_Jit_Type ty,
                     Inst **argarray, unsigned &argarraysize) {
    Operand_Exp *arg_exp = exprs.lookup_arg_exp(ith_arg,ty);
    create_assign_inst(mem,exprs,inst_head,arg_exp->opnd,arg_exp,src,ty);
    Inst *inst = inst_head->prev();
    argarray[argarraysize] = inst;
    argarraysize ++;
}

static unsigned n_words_of_method_arg_type_without_double(Method_Handle mh)
{
    unsigned size = 0;
    Arg_List_Iterator iter = method_get_argument_list(mh);
    Java_Type typ;
    while((typ = curr_arg(iter)) != JAVA_TYPE_END)
    {
        size += num_words_of_type(typ);
        if (typ == JAVA_TYPE_DOUBLE)
            size --;
        iter = advance_arg_iterator(iter);
    }
    return size;
}

//
// generate instructions that push arguments
//
static void gen_push_args(Mem_Manager& mem, Expressions& exprs, Stack& stack, 
                          Inst *inst_head, Method_Handle mh, 
                          unsigned size, unsigned ith_arg,
                          Inst **argarray, unsigned &argarraysize) {
    Arg_List_Iterator iter = method_get_argument_list(mh);
    Java_Type type;
    while((type = curr_arg(iter)) != JAVA_TYPE_END) {
        //if (type == JAVA_TYPE_ARRAY) type = JAVA_TYPE_CLASS;
        // push arguments
        Inst *src = stack.ith_word(size-ith_arg);
        push_arg(mem,exprs,inst_head,ith_arg,src,(O3_Jit_Type)type,
            argarray,argarraysize);
//        assert(src->exp->type == (Java_Type)type); // types must match
        ith_arg += num_words_of_type(type);
        iter = advance_arg_iterator(iter);
    }
	stack.pop(size);  // pop all arguments
}

//
// instructions for invoking virtual method calls
//
void gen_invokevirtual(Mem_Manager& mem, Expressions& exprs, Stack& stack,
					   Inst *inst_head, Compile_Handle comp_handle,
					   Class_Handle c_handle, unsigned index) {
    Loader_Exception exc;
	Method_Handle handle = resolve_virtual_method(comp_handle,c_handle,index,&exc);
	unsigned size = n_words_of_method_arg_type(handle) + 1;

#ifdef STAT_INDIRECT_CALL
//	Exp *imm = exprs.lookup_imm_exp(0, JIT_TYPE_INT); // imm exp is purely used for creating push
//	Call_Inst *cll = new (mem) Call_Inst(Call_Inst::stat_indirect_call,imm,NULL,false,inst_head);
	/*
	int ins_size = 2 ;
	Inst *ins_arg1 = exprs.lookup_imm(0xf1927f,JIT_TYPE_ADDR,inst_head);
	Inst **ins_argarray = (Inst**)mem.alloc(2 * sizeof(*ins_argarray)) ;
	unsigned ins_argarraysize = 0 ;
    push_arg(mem,exprs,inst_head,0,ins_arg1,JIT_TYPE_VOID,ins_argarray,ins_argarraysize); 
	Inst_Exp *ins_call_exp = exprs.lookup_inst_exp(Exp::StatIndirectCall,ins_arg1->exp,NULL,JIT_TYPE_VOID);
	Call_Inst *cll = new (mem) Call_Inst(Call_Inst::stat_indirect_call,ins_call_exp,NULL,false,inst_head);
    exprs.live_lcse()->kill_lcse_alias_call();
    cll->set_args(ins_argarray,ins_argarraysize,NULL);
	cll->set_gc_unsafe() ;
	*/
	/*
	Inst *this_ptr2 = stack.ith_word(size);
	Inst *ld_vtab2  = exprs.lookup_inst(Exp::Vtable,this_ptr2,NULL,JIT_TYPE_ADDR,inst_head);
	Inst *addr2 = exprs.lookup_field(ld_vtab2,method_get_offset(handle),JIT_TYPE_ADDR,
                                    inst_head,(FIELD_UID)handle);

	//Inst *base = exprs.lookup_imm((unsigned)method_get_offset(handle),JIT_TYPE_ADDR,inst_head);
	Exp *ib_exp = exprs.lookup_inst_exp(Exp::StatIndirectCall, addr2->exp, NULL,JIT_TYPE_VOID);
	Inst* stat_inst = new(mem) StatIndirectCall_Inst(addr2->dst(), ib_exp, inst_head);
	*/
#endif

    Inst **argarray = (Inst**)mem.alloc(size * sizeof(*argarray));
    unsigned argarraysize = 0;

    //
    // push this pointer
    //
	Inst *this_ptr = stack.ith_word(size);
    push_arg(mem,exprs,inst_head,0,this_ptr,JIT_TYPE_CLASS,
        argarray,argarraysize); // push this pointer
    //
    // push the rest arguments
    //
    gen_push_args(mem,exprs,stack,inst_head,handle,size,1,
                  argarray,argarraysize);
	//
	// access virtual func table
	//
	Inst *ld_vtab  = exprs.lookup_inst(Exp::Vtable,this_ptr,NULL,JIT_TYPE_ADDR,inst_head);
	Inst *addr = exprs.lookup_field(ld_vtab,method_get_offset(handle),JIT_TYPE_ADDR,
                                    inst_head,(FIELD_UID)handle);
    //
    // generate call
    //
    Value val;
    val.addr = (void*)handle; // keep handle around for inlining
    Operand_Exp *mh_addr = exprs.lookup_const_exp(&val,JIT_TYPE_ADDR);
	Java_Type ret_ty = method_get_return_type(handle);
	Inst_Exp *call_exp = exprs.lookup_inst_exp(Exp::Call,addr->exp,mh_addr,(O3_Jit_Type)ret_ty);
    Call_Inst *call = new (mem) Call_Inst(Call_Inst::virtual_call,addr->dst(),
                                          call_exp,handle,true,inst_head);

#ifdef STAT_INDIRECT_CALL
	//
	// Set the stat_inst of call to the StatIndirectCall_Inst above.
	// Used when inlining
	// 
//	call->stat_inst = stat_inst ;
#endif

    //
    // local cse that are aliased across call are removed from the list
    //
    exprs.live_lcse()->kill_lcse_alias_call();
    //
    // generate return value
    //
    Inst *ret = NULL;
    CONVERT_TO_INT_JAVA(ret_ty);
    if (ret_ty != JAVA_TYPE_VOID)
    {
        ret = exprs.lookup_ret((O3_Jit_Type)ret_ty,inst_head);
        stack.push(ret);
    }
    size = n_words_of_method_arg_type_without_double(handle) + 1;
    call->set_args(argarray,argarraysize,size,ret);
    assert(argarraysize <= size);
}

void gen_invokespecial(Mem_Manager& mem, Expressions& exprs, Stack& stack,
					   Inst *inst_head, Compile_Handle comp_handle, Class_Handle c_handle, unsigned index) {
    Loader_Exception exc;
	Method_Handle handle = resolve_special_method(comp_handle,c_handle,index,&exc);
	unsigned size = n_words_of_method_arg_type(handle) + 1;

    Inst **argarray = (Inst**)mem.alloc(size * sizeof(*argarray));
    Operand **opndargarray = (Operand**)mem.alloc(size * sizeof(*opndargarray));
    unsigned argarraysize = 0;

    //
    // push this pointer
    //
	Inst *this_ptr = stack.ith_word(size);
    push_arg(mem,exprs,inst_head,0,this_ptr,JIT_TYPE_CLASS,
        argarray,argarraysize); // push this pointer
    //
    // push the rest arguments
    //
    gen_push_args(mem,exprs,stack,inst_head,handle,size,1,
                  argarray,argarraysize);
	//
	// generate call to method
	//
    Value val;
    val.addr = (void*)handle;
    Inst *mh_addr = exprs.lookup_const(&val,JIT_TYPE_ADDR,inst_head);
	Java_Type ret_ty = method_get_return_type(handle);
	Inst_Exp *call_exp = exprs.lookup_inst_exp(Exp::Call,mh_addr->exp,NULL,(O3_Jit_Type)ret_ty);
    Call_Inst *call = new (mem) Call_Inst(Call_Inst::special_call,mh_addr->dst(),
                                          call_exp,handle,true,inst_head);
    //
    // local cse that are aliased across call are removed from the list
    //
    exprs.live_lcse()->kill_lcse_alias_call();
    //
    // generate return value
    //
    Inst *ret = NULL;
    CONVERT_TO_INT_JAVA(ret_ty);
    if (ret_ty != JAVA_TYPE_VOID)
    {
        ret = exprs.lookup_ret((O3_Jit_Type)ret_ty,inst_head);
        stack.push(ret);
    }
    size = n_words_of_method_arg_type_without_double(handle) + 1;
    call->set_args(argarray,argarraysize,size,ret);
    assert(argarraysize <= size);
}

void gen_invokestatic(Mem_Manager& mem, Expressions& exprs, Stack& stack,
                      Inst *inst_head, Compile_Handle comp_handle, Class_Handle c_handle, unsigned index)
{
    Loader_Exception exc;
	Method_Handle handle = resolve_static_method(comp_handle,c_handle,index,&exc);
    unsigned size = n_words_of_method_arg_type(handle);

    if (gen_intrinsic(mem, exprs, stack, inst_head, handle))
        return;

    Inst **argarray = (Inst**)mem.alloc(size * sizeof(*argarray));
    Operand **opndargarray = (Operand**)mem.alloc(size * sizeof(*opndargarray));
    unsigned argarraysize = 0;

    //
    // push the rest arguments
    //
    gen_push_args(mem,exprs,stack,inst_head,handle,size,0,
                  argarray,argarraysize);
    //
    // generate call to method
    //
    Value val;
    val.addr = (void*)handle;
    Inst *mh_addr = exprs.lookup_const(&val,JIT_TYPE_ADDR,inst_head);
    Java_Type ret_ty = method_get_return_type(handle);
    Inst_Exp *call_exp = exprs.lookup_inst_exp(Exp::Call,mh_addr->exp,NULL,(O3_Jit_Type)ret_ty);
    Call_Inst *call = new (mem) Call_Inst(Call_Inst::static_call,mh_addr->dst(),
                                          call_exp,handle,true,inst_head);
    //
    // local cse that are aliased across call are removed from the list
    //
    exprs.live_lcse()->kill_lcse_alias_call();
    //
    // generate return value
    //
    Inst *ret = NULL;
    CONVERT_TO_INT_JAVA(ret_ty);
    if (ret_ty != JAVA_TYPE_VOID)
    {
        ret = exprs.lookup_ret((O3_Jit_Type)ret_ty,inst_head);
        stack.push(ret);
    }
    size = n_words_of_method_arg_type_without_double(handle);
    call->set_args(argarray,argarraysize,size,ret);
    assert(argarraysize <= size);
}

void gen_invokeinterface(Mem_Manager& mem, Expressions& exprs, Stack& stack,
                         Inst *inst_head, Compile_Handle comp_handle,Class_Handle c_handle,
                         unsigned index, unsigned n_args)
{
    // Implementation:
    //  Resolve the interface method, getting a method handle.
    //  Get the method offset into the eventual vtable.
    //  Generate a resolve_interface runtime call, passing class_handle and "this".
    //  The resolve_interface call returns a vtable pointer in eax.
    //  Push all the arguments.
    //  Emit a call to [eax+offset].

    Loader_Exception exc;
	Method_Handle handle = resolve_interface_method(comp_handle,c_handle,index,&exc);
	unsigned size = n_words_of_method_arg_type(handle) + 1;
    unsigned offset = method_get_offset(handle);

    Inst **argarray = (Inst**)mem.alloc(size * sizeof(*argarray));
    unsigned argarraysize = 0;

    //Value val;
    //val.addr = (void*)c_handle;
	Inst *this_ptr = stack.ith_word(size);
    //Inst *ch_addr = exprs.lookup_const(&val,JIT_TYPE_ADDR,inst_head);
    Class_Handle iid = method_get_class(handle);
    Inst *ch_addr = exprs.lookup_imm((unsigned)iid,JIT_TYPE_ADDR,inst_head);
    Inst *getvtable = exprs.lookup_inst(Exp::InterfaceVtable, this_ptr, ch_addr,
        JIT_TYPE_ADDR, inst_head);
    push_arg(mem,exprs,inst_head,0,this_ptr,JIT_TYPE_CLASS,
        argarray,argarraysize); // push this pointer
    gen_push_args(mem,exprs,stack,inst_head,handle,size,1,
        argarray,argarraysize);

    //
    // generate call
    //
    Inst *call_addr = exprs.lookup_field(getvtable,offset,JIT_TYPE_ADDR,
                                    inst_head,(FIELD_UID)handle);
    Value val;
    val.addr = (void*)handle; // keep handle around
    Operand_Exp *mh_addr = exprs.lookup_const_exp(&val,JIT_TYPE_ADDR);
	Java_Type ret_ty = method_get_return_type(handle);
	Inst_Exp *call_exp = exprs.lookup_inst_exp(Exp::Call,call_addr->exp,mh_addr,(O3_Jit_Type)ret_ty);
    Call_Inst *call = new (mem) Call_Inst(Call_Inst::interface_call,call_addr->dst(),
                                          call_exp,handle,true,inst_head);
    //
    // local cse that are aliased across call are removed from the list
    //
    exprs.live_lcse()->kill_lcse_alias_call();
    //
    // generate return value
    //
    Inst *ret = NULL;
    CONVERT_TO_INT_JAVA(ret_ty);
    if (ret_ty != JAVA_TYPE_VOID)
    {
        ret = exprs.lookup_ret((O3_Jit_Type)ret_ty,inst_head);
        stack.push(ret);
    }
    size = n_words_of_method_arg_type_without_double(handle) + 1;
    call->set_args(argarray,argarraysize,size,ret);
    assert(argarraysize <= size);
}

static void gen_call_new(Mem_Manager& mem, Expressions& exprs, Stack& stack,
				  Inst *inst_head, Compile_Handle comp_handle, Class_Handle c_handle, 
                  unsigned index, Exp::Kind op, unsigned ith_arg,
                  Inst **argarray, unsigned& argarraysize, unsigned size) {
    if (op == Exp::Newarray)
    {
        // push index
        push_arg(mem,exprs,inst_head,ith_arg++,exprs.lookup_imm(index,JIT_TYPE_INT,inst_head),
            JIT_TYPE_INT, argarray,argarraysize);
    }
    else
    {
        // lookup resolved class
        Loader_Exception ld_exc;
        Class_Handle arr_elem_class_handle;
        switch (op)
        {
        case Exp::New:
            c_handle = resolve_class_new(comp_handle, c_handle, index, &ld_exc);
            break;
        case Exp::Anewarray:
            arr_elem_class_handle = resolve_class(comp_handle, c_handle, index, &ld_exc);
            c_handle = resolve_class_array_of_class(comp_handle, arr_elem_class_handle, &ld_exc);
            break;
        case Exp::Multinew:
            c_handle = resolve_class(comp_handle, c_handle, index, &ld_exc);
            assert(c_handle);
            break;
        }
    }

    // push class handle
    if (op != Exp::Newarray)
    {
        Inst *ch_addr = exprs.lookup_imm((unsigned)c_handle,JIT_TYPE_ADDR,inst_head);
        push_arg(mem,exprs,inst_head,ith_arg++,ch_addr,JIT_TYPE_ADDR,
            argarray,argarraysize);
    }

    //
    // generate call new/newarray/anewarray/multianewarray
    //
    Inst_Exp *call_exp = exprs.lookup_inst_exp(op,NULL,NULL,JIT_TYPE_ADDR);
    Call_Inst::Kind callkind = 
        (Call_Inst::Kind)(Call_Inst::new_call + (op - Exp::New));
    Call_Inst *call =
        new (mem) Call_Inst(callkind,call_exp,c_handle,false,inst_head);
    //
    // local cse that are aliased across call are removed from the list
    //
    exprs.live_lcse()->kill_lcse_alias_call();

    if (op == Exp::Multinew) // caller-pop; adjust esp by adding argarraysize*4
    {
        Operand_Exp *esp = exprs.lookup_reg_exp(esp_reg,JIT_TYPE_INT,0);
        Operand_Exp *imm = exprs.lookup_imm_exp(argarraysize*4, JIT_TYPE_INT);
        Inst_Exp *add = exprs.lookup_inst_exp(Exp::Add,esp,imm,JIT_TYPE_INT);
        Inst *i = new (mem) Add_Inst(Add_Inst::add,esp->opnd,imm->opnd,add,inst_head);
        i->set_dst(esp->opnd);
        i->dont_eliminate();
    }
	//
	// push the return value
	//
    O3_Jit_Type ret_ty = JIT_TYPE_CLASS;
    Inst *ret;
    if (op == Exp::Newarray && argarray[0]->src(0)->is_temp_reg()) {
        //
        // for constant bound elimination
        //
        Operand_Exp *ret_exp = exprs.lookup_ret_exp(ret_ty);
        Temp_Reg *treg = exprs.create_new_temp_reg(ret_ty);
        Operand_Exp *tmp_exp = exprs.lookup_temp_reg_exp(treg);
        ret = new (mem) Assign_Inst(treg, ret_exp->opnd, tmp_exp, inst_head);
        //
        // create bounds checking code so that we can eliminate subsequent 
        // bounds checking code.
        //
        Inst *len = exprs.lookup_inst(Exp::Length,ret,NULL,JIT_TYPE_INT,inst_head);
        Inst *n_elem = ((Temp_Reg*)argarray[0]->src(0))->inst();
        Inst *bnd = gen_array_cmp_inst(mem,exprs,inst_head,n_elem,len,JIT_TYPE_INT,false);
        bnd->mark_dead();
    } else {
        ret = exprs.lookup_ret(ret_ty,inst_head);
    }
    call->set_args(argarray,argarraysize,size,ret);
    stack.push(ret);
    assert(argarraysize <= size);
}

void gen_new(Mem_Manager& mem, Expressions& exprs, Stack& stack,
				  Inst *inst_head, Compile_Handle comp_handle,
                  Class_Handle c_handle, unsigned index) {
    unsigned size = 1;
    Inst **argarray = (Inst**)mem.alloc(size * sizeof(*argarray));
    unsigned argarraysize = 0;
    gen_call_new(mem,exprs,stack,inst_head,comp_handle,
                 c_handle,index,Exp::New,0, // arg0
                 argarray, argarraysize, size);
}

//
// generate MultiANewArray(class,index,n_dim,c1,c1,...);
//
void gen_multianewarray(Mem_Manager& mem, Expressions& exprs, Stack& stack,
					    Inst *inst_head, Compile_Handle comp_handle, Class_Handle c_handle,
                        unsigned n_dim, unsigned index) {
    Inst **argarray = (Inst**)mem.alloc((n_dim+2) * sizeof(*argarray));
    unsigned argarraysize = 0;
    //
    // push length of each dimension (from the rightmost dimension first)
    //
    unsigned ith_arg;
	for (ith_arg = 0; ith_arg < n_dim; ith_arg++) {
        Inst *arg = stack.ith_elem(n_dim - 1 - ith_arg);
        push_arg(mem,exprs,inst_head,ith_arg,arg,arg->type(),
            argarray,argarraysize);
        assert(arg->type() == JIT_TYPE_INT);
	}
    stack.pop(ith_arg); // pop the arguments
   
    // push n_dim
    push_arg(mem,exprs,inst_head,ith_arg++,exprs.lookup_imm(n_dim,JIT_TYPE_INT,inst_head),
        JIT_TYPE_INT, argarray,argarraysize);
    //
    // generate call multianewarray
    //
    gen_call_new(mem, exprs, stack, inst_head, comp_handle, c_handle, index,
        Exp::Multinew, ith_arg, argarray, argarraysize, n_dim + 2);
}

//
// generate newarray/anewarray(class,index,n_elem);
//
void gen_new_array(Mem_Manager& mem, Expressions& exprs, Stack& stack,
				   Inst *inst_head, Compile_Handle comp_handle, Class_Handle c_handle,
                   unsigned index, Inst *n_elem, Exp::Kind op) {
    unsigned sz_array = 2;
    Inst **argarray = (Inst**)mem.alloc(sz_array * sizeof(*argarray));
    unsigned argarraysize = 0;
    // push n_elem (arg0)
    push_arg(mem,exprs,inst_head,0,n_elem,JIT_TYPE_INT,
        argarray,argarraysize);
    //
    // new object
    //
    gen_call_new(mem,exprs,stack,inst_head,comp_handle, c_handle,index,op,1, // from arg1
                 argarray, argarraysize, sz_array); 
}

void gen_checkcast(Mem_Manager& mem, Expressions& exprs, Stack& stack,
                   Inst *inst_head, Compile_Handle comp_handle,
                   Class_Handle c_handle, unsigned index) {
    Loader_Exception exc;
    Class_Handle ch = resolve_class(comp_handle, c_handle,index,&exc);
    Exp *to = exprs.lookup_imm_exp((unsigned)ch,JIT_TYPE_ADDR);
    Inst *src = stack.pop();
    //
    // check if instanceof is available at this point.  
    // If so, we don't need to checkcast
    //
    Exp *instance = exprs.lookup_inst_exp(Exp::Instanceof,to,src->exp,JIT_TYPE_INT);
    if (exprs.is_local_cse(instance))
        stack.push(src);
    else {
        Inst *cast_to = exprs.lookup_imm((unsigned)ch,JIT_TYPE_ADDR,inst_head);
        stack.push(exprs.lookup_inst(Exp::Checkcast,cast_to,src,
                                     JIT_TYPE_CLASS,inst_head));
    }
}

void gen_instanceof(Mem_Manager& mem, Expressions& exprs, Stack& stack,
                    Inst *inst_head, Compile_Handle comp_handle,
                    Class_Handle c_handle, unsigned index) {
    Loader_Exception exc;
    Class_Handle ch = resolve_class(comp_handle, c_handle,index,&exc);
    Inst *of = exprs.lookup_imm((unsigned)ch,JIT_TYPE_ADDR,inst_head);
    Inst *src = stack.pop();
    stack.push(exprs.lookup_inst(Exp::Instanceof,of,src,JIT_TYPE_INT,inst_head));
}

//
// generate monitor enter/exit
//
void gen_monitor(Mem_Manager& mem, Expressions& exprs, Stack& stack,
                 Inst *inst_head, Exp::Kind op) {
    assert(op == Exp::Monitorenter || op == Exp::Monitorexit);
    Inst **argarray = (Inst**)mem.alloc(1 * sizeof(*argarray));
    unsigned argarraysize = 0;
    Inst *obj = stack.pop();
    push_arg(mem,exprs,inst_head,0,obj,JIT_TYPE_CLASS,argarray,argarraysize);
    Inst_Exp *monitor = exprs.lookup_inst_exp(op,NULL,NULL,JIT_TYPE_VOID);
    Call_Inst *call;
    if (op == Exp::Monitorenter)
        call = new (mem) Call_Inst(Call_Inst::monenter_call,monitor,NULL,false,inst_head);
    else
        call = new (mem) Call_Inst(Call_Inst::monexit_call,monitor,NULL,false,inst_head);
    call->set_args(argarray,argarraysize,NULL);
}

//
// generate throw exception
//
void gen_athrow(Mem_Manager& mem, Expressions& exprs, Stack& stack, 
                Inst *inst_head) {
    Inst **argarray = (Inst**)mem.alloc(1 * sizeof(*argarray));
    unsigned argarraysize = 0;
    Inst *obj = stack.pop();
    push_arg(mem,exprs,inst_head,0,obj,JIT_TYPE_CLASS,
        argarray,argarraysize);
    Inst_Exp *athrow = exprs.lookup_inst_exp(Exp::Athrow,obj->exp,NULL,JIT_TYPE_VOID);
    Call_Inst *call =
        new (mem) Call_Inst(Call_Inst::athrow_call,/*obj->dst(),*/athrow,NULL,false,inst_head);
    call->set_args(argarray,argarraysize,NULL);
}

Inst *create_assign_inst(Expressions& exprs, Inst *inst_head,
                                Operand_Exp *dst, Operand_Exp *src, 
                                O3_Jit_Type type) {
    Exp *asgn = exprs.lookup_inst_exp(Exp::Assign,dst,src,type);
    return new (exprs.mem) Assign_Inst(dst->opnd,src->opnd,asgn,inst_head);
}

//
// generate   a0 = this_ptr/class_handle
//            call monitor_enter/monitor_exit
//
static void gen_monitor_enter_exit(Expressions& exprs,
                                   Class_Handle  c_handle,
                                   Inst *inst,
                                   Exp::Kind cop,
                                   bool is_static) {
    Operand_Exp *a0;
    Inst **argarray = (Inst**)exprs.mem.alloc(1 * sizeof(*argarray));
    Call_Inst *cll;
    if (is_static) { // static method
        a0 = exprs.lookup_arg_exp(0,JIT_TYPE_ADDR);
        Operand_Exp *ch = exprs.lookup_imm_exp((unsigned)c_handle, JIT_TYPE_ADDR);
        argarray[0] = create_assign_inst(exprs,inst,a0,ch,JIT_TYPE_CLASS);
	} else {
        a0 = exprs.lookup_arg_exp(0,JIT_TYPE_CLASS);
	    unsigned reg_id = exprs.reg_map.virtual_reg_id(0,JIT_TYPE_CLASS);
        Operand_Exp *v0 = exprs.lookup_reg_exp(reg_id,JIT_TYPE_CLASS,1); // lookup virtual reg
        argarray[0] = create_assign_inst(exprs,inst,a0,v0,JIT_TYPE_CLASS);
	}
    Inst_Exp *mon_call = exprs.lookup_inst_exp(cop,NULL,NULL,JIT_TYPE_VOID);
    Call_Inst::Kind c = (Call_Inst::Kind)(Call_Inst::monenter_call +(cop-Exp::Monitorenter));
    cll = new (exprs.mem) Call_Inst(c,mon_call,NULL,false,inst);
    cll->set_args(argarray,1,NULL);
}

//
// generate   a0 = this_ptr/class_handle
//            call monitor_enter
//
void gen_synch_method_enter(Expressions& exprs,
                            Class_Handle  c_handle,
							Method_Handle m_handle,
                            Inst *inst) {
    if (method_is_static(m_handle)) // static method
        gen_monitor_enter_exit(exprs,c_handle,inst,Exp::Monitorenter_static,true);
	else
        gen_monitor_enter_exit(exprs,c_handle,inst,Exp::Monitorenter,false);
}

//
// generate    t = Ret.I
//             push this_ptr/class_handle
//             call monitor_exit
//             Ret.I = t
// Save return value so that the return value is not destroyed by montor_exit
//
void gen_synch_method_exit(Expressions& exprs,
                           Class_Handle  c_handle,
						   Method_Handle m_handle,
                           Inst *inst) {
    O3_Jit_Type ret_type = (O3_Jit_Type)method_get_return_type(m_handle);
    CONVERT_TO_INT_JIT(ret_type);
    Inst *t = NULL;
    // insert   t = Ret.I
    if (ret_type != JIT_TYPE_VOID)
        t = exprs.lookup_ret(ret_type,inst);

    //
    // insert   a0 = this_ptr/class_handle
    //          call monitor_exit
    //
	if (method_is_static(m_handle)) // static method
        gen_monitor_enter_exit(exprs,c_handle,inst,Exp::Monitorexit_static,true);
	else
        gen_monitor_enter_exit(exprs,c_handle,inst,Exp::Monitorexit,false);
    //
    // insert    Ret.I = t
    //
    if (ret_type != JIT_TYPE_VOID) {
        Operand_Exp *ret_exp = exprs.lookup_ret_exp(ret_type);
        Exp *asgn = exprs.lookup_inst_exp(Exp::Assign,ret_exp,t->exp,ret_type);
        new (exprs.mem) Assign_Inst(ret_exp->opnd,t->dst(),asgn,inst);
    }
}

static void formal_to_actual(Mem_Manager& mem, Expressions& exprs, 
                             Inst *inst_head, unsigned reg_id,unsigned ith_arg, 
                             O3_Jit_Type ty) {
    Operand_Exp *var = exprs.lookup_reg_exp(reg_id,ty,1);
    Inst *arg = exprs.lookup_arg(ith_arg,ty,inst_head);
    create_assign_inst(mem,exprs,inst_head,var->opnd,var,arg,ty);
}
//
// explicitly generate argument passing code (formals to actuals)
//       v0 = arg0
//       v1 = arg1
//          ...
// 
void build_incoming_arg_assignment(Class_Handle c_handle, Method_Handle m_handle,
    Mem_Manager& mem, Expressions& exprs, Inst *inst_head) {

    Inst *last = inst_head->prev();
    unsigned ith = 0, reg_id;
    // move this pointer
    if (!method_is_static(m_handle)) {
        reg_id = exprs.reg_map.virtual_reg_id(ith,JIT_TYPE_CLASS); // var0's reg mapping
        formal_to_actual(mem,exprs,inst_head,reg_id,ith,JIT_TYPE_CLASS);
        ith++;
    }

    Arg_List_Iterator iter = method_get_argument_list(m_handle);
    Java_Type type;
    while((type = curr_arg(iter)) != JAVA_TYPE_END) {
        //if (type == JAVA_TYPE_ARRAY) type = JAVA_TYPE_CLASS;
        CONVERT_TO_INT_JAVA(type);
        reg_id = exprs.reg_map.virtual_reg_id(ith,(O3_Jit_Type)type);
        formal_to_actual(mem,exprs,inst_head,reg_id,ith,(O3_Jit_Type)type);
        ith += num_words_of_type(type);
        iter = advance_arg_iterator(iter);
    }
}
