/*
 *  OpenDuke
 *  Copyright (C) 1999  Rusty Wagner
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */


//---------------------------------------------------------------------------
#ifndef PLATFORM_UNIX
#include <vcl.h>
#pragma hdrstop

#include <d3d.h>
#else
#include <stdarg.h>
#include "linux_inc.h"
#endif
#include <stdio.h>
#include "console.h"
#include "Render.h"
#include "anm.h"

#include <iostream>

using namespace std;
#define CS_UP 0
#define CS_HALFDOWN 1
#define CS_DOWN 2

int consoleDelta,consoleY,consoleMaxY;
int consoleWidth,consoleHeight;
int cursorX,cursorY,outputX,outputY,commandStartY;
int consoleUpdateTime;
Texture* consoleTexture;
void* consoleFont;
Poly consolePoly,cursorPoly;
char **consoleText=NULL;
int scrnX,scrnY;
int commandReady;
char command[1024];
int consoleState;
#ifndef PLATFORM_UNIX
int graphicalConsole=1;
#endif
HANDLE consoleScreenBuf;

// Redraw line numbers startY to endY in the text-based console
#ifndef PLATFORM_UNIX
// all consoles in Linux will be graphical
void RedrawNonGraphicalConsole(int startY,int endY)
{
    // Check the range of the line numbers and fix
    // if necessary
    if (startY<0) startY=0;
    if (startY>=consoleHeight) return;
    if (endY<0) return;
    if (endY>=consoleHeight) endY=consoleHeight-1;
    COORD coord;
    // Draw each line
    for (int y=startY;y<=endY;y++)
    {
        DWORD x;
        coord.X=0; coord.Y=y;
        WriteConsoleOutputCharacter(consoleScreenBuf,
            consoleText[y],strlen(consoleText[y]),coord,&x);
        if (strlen(consoleText[y])<80)
        {
            coord.X=strlen(consoleText[y]); coord.Y=y;
            FillConsoleOutputCharacter(consoleScreenBuf,
                ' ',80-strlen(consoleText[y]),coord,&x);
        }
    }
    // Reset cursor position
    coord.X=cursorX; coord.Y=cursorY;
    SetConsoleCursorPosition(consoleScreenBuf,coord);
}
#endif

void CreateConsole(GroupFile *group,int _scrnX,int _scrnY,Art *art)
{
    if (group)
    {
        // Create a font for the console
// FIXME consoleFont stuff
#ifndef PLATFORM_UNIX
        consoleFont=render->AddFont(CreateFont(16,0,0,0,FW_NORMAL,0,0,0,DEFAULT_CHARSET,
            OUT_DEFAULT_PRECIS,CLIP_DEFAULT_PRECIS,DEFAULT_QUALITY,FF_MODERN,
            "Courier New"));
        render->UseFont(consoleFont);
#endif
        // Generate the polygons for the graphical console and
        // the cursor
        consolePoly.AddVert(-10,0,10,0.5/512,199.5/256,0,0,RGBA_MAKE(108,108,108,255));
        consolePoly.AddVert(10,0,10,319.5/512,199.5/256,0,0,RGBA_MAKE(108,108,108,255));
        consolePoly.AddVert(10,10,10,319.5/512,0.5/256,0,0,RGBA_MAKE(108,108,108,255));
        consolePoly.AddVert(-10,10,10,0.5/512,0.5/256,0,0,RGBA_MAKE(108,108,108,255));
        ANMFile *consoleBackAnim=new ANMFile(group,"RADLOGO.ANM",art);
        consolePoly.texture=consoleBackAnim->GetTexture();
        consolePoly.flags=PF_TWOSIDED|PF_PRIORITY;
        cursorPoly.AddVert(0,0,5,0,0,0,0,RGBA_MAKE(255,255,255,255));
        cursorPoly.AddVert(0,0,5,0,0,0,0,RGBA_MAKE(255,255,255,255));
        cursorPoly.AddVert(0,0,5,0,0,0,0,RGBA_MAKE(255,255,255,255));
        cursorPoly.AddVert(0,0,5,0,0,0,0,RGBA_MAKE(255,255,255,255));
        cursorPoly.texture=NULL;
        cursorPoly.flags=PF_TWOSIDED|PF_PRIORITY;
    }
    // Set variables to their inital values
    scrnX=_scrnX; scrnY=_scrnY;
    consoleWidth=(scrnX/8)-4;
    consoleHeight=(scrnY/12)-2;
    consoleY=0;
    consoleY=-scrnY;
    consoleMaxY=0;
    consoleState=CS_UP;
    cursorX=1; cursorY=consoleHeight-1;
    outputX=0; outputY=consoleHeight-2;
    commandStartY=cursorY;
    consoleDelta=0;
    consoleUpdateTime=GetTickCount()+2;
#ifndef PLATFORM_UNIX
    if (!graphicalConsole)
    {
        // If the console is a text-based one, the
        // size is always 80x25 characters
        consoleWidth=80; consoleHeight=25;
        cursorX=1; cursorY=consoleHeight-1;
        outputX=0; outputY=consoleHeight-2;
        commandStartY=cursorY;
        if (!consoleText)
        {
            // Allocate and initialize a text array
            consoleText=new char*[consoleHeight];
            for (int i=0;i<consoleHeight;i++)
            {
                consoleText[i]=new char[consoleWidth+1];
                memset(consoleText[i],' ',consoleWidth);
                consoleText[i][consoleWidth]=0;
            }
            // Put in the command prompt
            consoleText[consoleHeight-1][0]='>';
        }
        commandReady=0;
        // Draw the console text
        RedrawNonGraphicalConsole(0,consoleHeight-1);
        return;
    }
#endif
    // Allocate and initialize the text array for the
    // graphical console
    consoleText=new char*[consoleHeight];
    for (int i=0;i<consoleHeight;i++)
    {
        consoleText[i]=new char[consoleWidth+1];
        memset(consoleText[i],' ',consoleWidth);
        consoleText[i][consoleWidth]=0;
    }
    // Put in the command prompt
    consoleText[consoleHeight-1][0]='>';
    commandReady=0;
}

void ResizeConsole(int _scrnX,int _scrnY)
{
    // Recreate font for the console
// FIXME consoleFont stuff
#ifndef PLATFORM_UNIX
    consoleFont=render->AddFont(CreateFont(16,0,0,0,FW_NORMAL,0,0,0,DEFAULT_CHARSET,
        OUT_DEFAULT_PRECIS,CLIP_DEFAULT_PRECIS,DEFAULT_QUALITY,FF_MODERN,
        "Courier New"));
    render->UseFont(consoleFont);
    // Return if the console is text-based (size is constant)
    if (!graphicalConsole) return;
#endif
    // Return if the console hasn't been created yet
    if (consoleText==NULL) return;
    // Save the old size and text
    int oldWidth=consoleWidth;
    int oldHeight=consoleHeight;
    char** oldText=consoleText;
    // Change the variables to the new size
    scrnX=_scrnX; scrnY=_scrnY;
    consoleWidth=(scrnX/8)-4;
    consoleHeight=(scrnY/12)-2;
    // Reposition the console according to the new
    // screen size
    switch(consoleState)
    {
        case CS_UP:
            consoleY=-scrnY;
            consoleMaxY=(consoleMaxY==0)?0:-(scrnY>>1);
            break;
        case CS_HALFDOWN:
            consoleY=-(scrnY>>1);
            consoleMaxY=-(scrnY>>1);
            break;
        case CS_DOWN:
            consoleY=0;
            consoleMaxY=0;
            break;
    }
    // Initialize input variables for the new size
    cursorX=1; cursorY=consoleHeight-1;
    outputX=0; outputY=consoleHeight-2;
    commandStartY=cursorY;
    consoleDelta=0;
    consoleUpdateTime=GetTickCount()+2;
    // Create and initialize the new text array
    consoleText=new char*[consoleHeight];
    for (int i=0;i<consoleHeight;i++)
    {
        consoleText[i]=new char[consoleWidth+1];
        memset(consoleText[i],' ',consoleWidth);
        consoleText[i][consoleWidth]=0;
    }
    // Put all the text from the old text array into
    // the new one
    for (int y=(consoleHeight-oldHeight),i=1;y<(consoleHeight-1);y++,i++)
    {
        if (y<0) continue;
        if (consoleWidth>oldWidth)
            memcpy(consoleText[y],oldText[i],oldWidth);
        else
            memcpy(consoleText[y],oldText[i],consoleWidth);
    }
    // Delete the old console text array
    for (int y=0;y<oldHeight;y++)
        delete[] oldText[y];
    delete[] oldText;
    // Put in the command prompt
    consoleText[consoleHeight-1][0]='>';
    commandReady=0;
}

void ConsoleGameReady()
{
    // The game now has something to render, so hide
    // the console
#ifndef PLATFORM_UNIX
    if (!graphicalConsole) return;
#endif
    consoleMaxY=-(scrnY>>1);
    consoleDelta=-1;
    consoleState=CS_UP;
}

#ifndef PLATFORM_UNIX
BOOL CALLBACK Enum(HWND hWnd,LPARAM)
{
    // When the text-based console window is found,
    // bring it to the top
    char str[256];
    GetWindowText(hWnd,str,256);
    if (!strcmp(str,"Duke Nukem 3D Console"))
    {
        SetWindowPos(hWnd,HWND_TOP,0,0,0,0,SWP_NOMOVE|
            SWP_NOSIZE);
    }
    return 1;
}
#endif

void ToggleConsole()
{
#ifndef PLATFORM_UNIX
    if (!graphicalConsole)
    {
        // Find the text-based console window and
        // activate it if the console is text-based
        EnumWindows((WNDENUMPROC)Enum,0);
        return;
    }
#endif
    // Return if the console isn't initialized yet
    if (consoleText==NULL) return;
    if (consoleDelta==-1)
    {
        // If the console is on the way up, make
        // it go down
        consoleDelta=1;
        if (consoleMaxY==0)
            consoleState=CS_DOWN;
        else
            consoleState=CS_HALFDOWN;
    }
    else if (consoleDelta==1)
    {
        // If the console is on the way down, make
        // it go up
        consoleDelta=-1;
        consoleState=CS_UP;
    }
    else
    {
        if (consoleY<consoleMaxY)
        {
            // If the console is up, then make it go down
            consoleDelta=1;
            if (consoleMaxY==0)
                consoleState=CS_DOWN;
            else
                consoleState=CS_HALFDOWN;
        }
        else
        {
            // Else the console is down, make it go up
            consoleDelta=-1;
            consoleState=CS_UP;
        }
    }
}

void ToggleConsoleImmediate()
{
#ifndef PLATFORM_UNIX
    if (!graphicalConsole)
    {
        // Find the text-based console window and
        // activate it if the console is text-based
        EnumWindows((WNDENUMPROC)Enum,0);
        return;
    }
#endif
    // Return if the console isn't initialized yet
    if (consoleText==NULL) return;
    if (consoleDelta==-1)
    {
        // If the console is on the way up, make
        // it go down
        consoleDelta=1;
        consoleY=consoleMaxY;
        if (consoleMaxY==0)
            consoleState=CS_DOWN;
        else
            consoleState=CS_HALFDOWN;
    }
    else if (consoleDelta==1)
    {
        // If the console is on the way down, make
        // it go up
        consoleDelta=-1;
        consoleY=-(scrnY>>1);
        consoleState=CS_UP;
    }
    else
    {
        if (consoleY<consoleMaxY)
        {
            // If the console is up, then make it go down
            consoleDelta=1;
            consoleY=consoleMaxY;
            if (consoleMaxY==0)
                consoleState=CS_DOWN;
            else
                consoleState=CS_HALFDOWN;
        }
        else
        {
            // Else the console is down, make it go up
            consoleDelta=-1;
            consoleY=-scrnY;
            consoleState=CS_UP;
        }
    }
}

void ConsoleOutput(char *text)
{
    // Return if the console isn't initialized yet
#ifdef PLATFORM_UNIX
	// I wanna see what's going on - DDOI
	cout << text;
#endif
    if (consoleText==NULL) return;
    // Write out each character
    for (int i=0;text[i];i++)
    {
        if (text[i]=='\n')
        {
            // Handle newline character
            outputX=0;
            // Shift text up
            for (int y=0;y<outputY;y++)
                memcpy(consoleText[y],consoleText[y+1],consoleWidth);
            memset(consoleText[outputY],' ',consoleWidth);
            // Redraw text-based console if enabled
#ifndef PLATFORM_UNIX
            if (!graphicalConsole)
                RedrawNonGraphicalConsole(0,consoleHeight-1);
#endif
        }
        else
        {
            if (outputX>=consoleWidth)
            {
                // If text must wrap around, go to next line
                outputX=0;
                // Shift text up
                for (int y=0;y<outputY;y++)
                    memcpy(consoleText[y],consoleText[y+1],consoleWidth);
                memset(consoleText[outputY],' ',consoleWidth);
                // Redraw text-based console if enabled
#ifndef PLATFORM_UNIX
                if (!graphicalConsole)
                    RedrawNonGraphicalConsole(0,consoleHeight-1);
#endif
            }
            // Write out character
            consoleText[outputY][outputX]=text[i];
            outputX++;
#ifndef PLATFORM_UNIX
            // Redraw text-based console if enabled
            if (text[i+1]==0)
            {
                if (!graphicalConsole)
                    RedrawNonGraphicalConsole(outputY,outputY);
            }
#endif
        }
    }
}

void cprintf(char *str, ...)
{
    va_list argptr;
    char buf[256];

    if (str)
    {
        // Convert to string and pass to ConsoleOutput function
        va_start(argptr,str);
        vsprintf(buf,str,argptr);
        va_end(argptr);
        ConsoleOutput(buf);
    }
}

#ifdef PLATFORM_UNIX
int ConsoleTypeChar(WORD Key)
#else
int ConsoleTypeChar(char Key)
#endif
{
#ifndef PLATFORM_UNIX
    if (graphicalConsole)
#endif
    {
        // Return if console is not initialized yet
        if (consoleText==NULL) return 0;
        // Return if console is hidden
        if (consoleY<=(-scrnY)) return 0;
    }
    if ((Key>=' ')&&(Key!='`'))
    {
        // Key is a valid character for the console,
        // put it in the array
        consoleText[cursorY][cursorX]=Key;
        cursorX++;
#ifndef PLATFORM_UNIX
        // Redraw text-based console if enabled
        if (!graphicalConsole)
            RedrawNonGraphicalConsole(cursorY,cursorY);
#endif
        if (cursorX>=consoleWidth)
        {
            // If text must wrap around, go to next line
            cursorX=0;
            cursorY++;
            if (cursorY>=consoleHeight)
            {
                // If new line requires text to scroll
                // up, do so
                for (int y=0;y<(consoleHeight-1);y++)
                    memcpy(consoleText[y],consoleText[y+1],consoleWidth);
                memset(consoleText[consoleHeight-1],' ',consoleWidth);
                cursorY=consoleHeight-1;
                commandStartY--;
                outputY--;
#ifndef PLATFORM_UNIX
                // Redraw text-based console if enabled
                if (!graphicalConsole)
                    RedrawNonGraphicalConsole(0,consoleHeight-1);
#endif
            }
        }
        return 1;
    }
    else if (Key==13)
    {
        // Handle enter key
        commandReady=1;
        int i=0;
        // Process the text array and convert the inputted
        // text to a contiguous string
        for (int y=commandStartY;y<=cursorY;y++)
        {
            if (y!=cursorY)
            {
                for (int x=(y==commandStartY)?1:0;x<consoleWidth;x++)
                    command[i++]=consoleText[y][x];
            }
            else
            {
                for (int x=(y==commandStartY)?1:0;x<cursorX;x++)
                    command[i++]=consoleText[y][x];
            }
        }
        command[i]=0;
        // Go to a new line
        cursorX=0;
        cursorY++;
        if (cursorY>=consoleHeight)
        {
            // If new line requires text to scroll
            // up, do so
            for (int y=0;y<(consoleHeight-1);y++)
                memcpy(consoleText[y],consoleText[y+1],consoleWidth);
            memset(consoleText[consoleHeight-1],' ',consoleWidth);
            cursorY=consoleHeight-1;
        }
        // Put in a new command prompt
        consoleText[cursorY][cursorX]='>';
        cursorX++;
        commandStartY=cursorY;
        outputX=consoleWidth;
        outputY=cursorY-1;
#ifndef PLATFORM_UNIX
        // Redraw text-based console if enabled
        if (!graphicalConsole)
            RedrawNonGraphicalConsole(0,consoleHeight-1);
#endif
        return 1;
    }
    else if (Key==8)
    {
        // Handle backspace key
        if ((cursorY!=commandStartY)||(cursorX>1))
        {
            if (cursorX==0)
            {
                // Go to previous line if needed
                cursorX=consoleWidth;
                cursorY--;
            }
            // Delete the character
            cursorX--;
            consoleText[cursorY][cursorX]=' ';
#ifndef PLATFORM_UNIX
            // Redraw text-based console if enabled
            if (!graphicalConsole)
                RedrawNonGraphicalConsole(cursorY,cursorY+1);
#endif
        }
        return 1;
    }
    return 0;
}

void RenderConsole()
{
#ifndef PLATFORM_UNIX
    // Return if console is text based
    if (!graphicalConsole) return;
#endif
    // Return if console is not intialized yet
    if (consoleText==NULL) return;
    if (consoleY!=(-scrnY))
    {
        // Render console if it is not hidden
        render->UseOriginView();
        render->BeginScene();
        // Change y coordinates of the console polygon
        // to match the current position of the console
        consolePoly.v[0].y=-20.0*((float)(consoleY+scrnY)/(float)scrnY-0.5)*((float)scrnY/(float)scrnX);
        consolePoly.v[1].y=-20.0*((float)(consoleY+scrnY)/(float)scrnY-0.5)*((float)scrnY/(float)scrnX);
        consolePoly.v[2].y=-20.0*((float)consoleY/(float)scrnY-0.5)*((float)scrnY/(float)scrnX);
        consolePoly.v[3].y=-20.0*((float)consoleY/(float)scrnY-0.5)*((float)scrnY/(float)scrnX);
        // Render the polygon
        render->RenderPoly(consolePoly);
        render->EndScene();
        render->BeginScene();
        // Draw the text in the console
        render->StartText();
        render->UseFont(consoleFont);
        for (int y=0;y<consoleHeight;y++)
        {
            render->RenderText(16,(scrnY+((y-consoleHeight-1)*12))+consoleY,consoleText[y],
                RGBA_MAKE(255,255,255,255));
        }
        render->EndText();
        render->EndScene();
        render->BeginScene();
        // Set the coordinates for the cursor polygon
        cursorPoly.v[0].x=10.0*((float)(cursorX*8+16)/(float)scrnX-0.5);
        cursorPoly.v[1].x=10.0*((float)(cursorX*8+23)/(float)scrnX-0.5);
        cursorPoly.v[2].x=10.0*((float)(cursorX*8+23)/(float)scrnX-0.5);
        cursorPoly.v[3].x=10.0*((float)(cursorX*8+16)/(float)scrnX-0.5);
        cursorPoly.v[0].y=-10.0*((float)(consoleY+cursorY*12+26)/(float)scrnY-0.5)*((float)scrnY/(float)scrnX);
        cursorPoly.v[1].y=-10.0*((float)(consoleY+cursorY*12+26)/(float)scrnY-0.5)*((float)scrnY/(float)scrnX);
        cursorPoly.v[2].y=-10.0*((float)(consoleY+cursorY*12+15)/(float)scrnY-0.5)*((float)scrnY/(float)scrnX);
        cursorPoly.v[3].y=-10.0*((float)(consoleY+cursorY*12+15)/(float)scrnY-0.5)*((float)scrnY/(float)scrnX);
        // Render the cursor polygon
        render->RenderPoly(cursorPoly);
        render->EndScene();
        render->UseNormalView();
    }
}

void UpdateConsolePos()
{
#ifndef PLATFORM_UNIX
    // Return if the console is text based
    if (!graphicalConsole) return;
#endif
    // Return if the console is not initialized yet
    if (consoleText==NULL) return;
    // Move the console 1 pixel for each update
    // period elapsed
    while (((int)GetTickCount()-consoleUpdateTime)>0)
    {
        consoleY+=consoleDelta;
        consoleUpdateTime+=2;
    }
    // Clip the console position to the maximum
    // and minimum values
    if ((consoleY<(-scrnY))&&(consoleDelta==-1))
    {
        consoleY=-scrnY;
        consoleDelta=0;
    }
    else if ((consoleY>consoleMaxY)&&(consoleDelta==1))
    {
        consoleY=consoleMaxY;
        consoleDelta=0;
    }
}

void ForceConsoleRedraw()
{
#ifndef PLATFORM_UNIX
    if (!graphicalConsole) return;
#endif
    if (consoleText==NULL) return;
    render->BeginRender();
    RenderConsole();
    render->EndRender();
}

int IsCommandReady()
{
    return commandReady;
}

char* CommandString()
{
    commandReady=0;
    return command;
}

#ifndef PLATFORM_UNIX
void CheckNonGraphicalConsoleInput(HWND hWnd)
{
    INPUT_RECORD ir[32];
    DWORD n;
    // Check to see if any keys were pressed
    PeekConsoleInput(GetStdHandle(STD_INPUT_HANDLE),ir,32,&n);
    if (n)
    {
        // Read the keys pressed into an array
        ReadConsoleInput(GetStdHandle(STD_INPUT_HANDLE),ir,32,&n);
        for (int i=0;i<n;i++)
        {
            // Check to see if it was a key down event
            if ((ir[i].EventType==KEY_EVENT)&&(ir[i].Event.KeyEvent.bKeyDown))
            {
                // If it was the ~ key, deactivate the console
                if (ir[i].Event.KeyEvent.uChar.AsciiChar=='`')
                    SetWindowPos(hWnd,HWND_TOP,0,0,0,0,SWP_NOMOVE|SWP_NOSIZE);
                // Send the key to the console key handler
                ConsoleTypeChar(ir[i].Event.KeyEvent.uChar.AsciiChar);
            }
        }
    }
}
#endif

int IsConsoleDown()
{
    if (consoleState==CS_UP)
        return 0;
    return 1;
}

//---------------------------------------------------------------------------
#pragma package(smart_init)
