/* 
 *   Creation Date: <1999/11/27 22:14:57 samuel>
 *   Time-stamp: <2001/05/21 22:20:20 samuel>
 *   
 *	<osi_blk.c>
 *	
 *	MacOS block driver
 *   
 *   Copyright (C) 1999, 2000, 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

/* #define VERBOSE  */

#include <pthread.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/uio.h>
#include <sys/ioctl.h>

#include "booter.h"
#include "driver_mgr.h"
#include "os_interface.h"
#include "memory.h"
#include "verbose.h"
#include "llseek.h"
#include "res_manager.h"
#include "osi_driver.h"
#include "session.h"
#include "disk.h"

#define LINUX_COMPILE
#include "blk_shared.h"

SET_VERBOSE_NAME("osi_blk");

struct drive;

static int	osi_blk_init( void );
static void     osi_blk_cleanup( void );

static int	osip_blk_read( int sel, int *params );
static int	osip_blk_write( int sel, int *params );
static int	osip_blk_drive_info( int sel, int *params );

static void	add_drive( bdev_desc_t *bdev );

#if 0
static int	save_blk_state( void );
static void	check_session_state( void );
static int	get_mdb_checksum( struct drive *drv, ulong *checksum );
#endif

driver_interface_t osi_blk_driver = {
    "osi_blk", osi_blk_init, osi_blk_cleanup
};

/***************** PCI constants ********************/

#define VENDOR		0x1000
#define DEVICE_ID	0x0003
#define CLASS_CODE	0x0100

static pci_dev_info_t pci_config = { 
	VENDOR, DEVICE_ID, 0x02, CLASS_CODE
};

static struct osi_driver *osi_driver = NULL;


/****************** Variables ***********************/

typedef struct drive {
	long		id;
	bdev_desc_t	*bdev;

	int		locked;
	ulong		blkoffs;
	ulong		nblks;
	
//	ulong		name_checksum;
//	ulong		mdb_checksum;

	struct drive 	*next;
} drive_t;

static drive_t 		*s_first_drive;
static int 		s_next_id = 1;


/************************************************************************/
/*	Init /cleanup							*/
/************************************************************************/

int 
osi_blk_init( void )
{
	bdev_desc_t *bdev;
#if 1
	extern void setup_drives( void );
	bdev_setup_drives();
#endif
	/* is osi_enet disabled? */
	if( get_bool_res("disable_osi_blk")==1 ) {
		LOG("*** OSI BLOCK DRIVER DISABLED **\n");
		return 0;
	}
	if( !(osi_driver = register_osi_driver( "blk", "MOLBlockDriver", &pci_config )) ){
		printm("----> Failed to register the block driver!\n");
		return 0;
	}
	os_interface_add_proc( OSI_BLK_READ, osip_blk_read );
	os_interface_add_proc( OSI_BLK_WRITE, osip_blk_write );
	os_interface_add_proc( OSI_BLK_DRIVE_INFO, osip_blk_drive_info );

	while( (bdev=bdev_get_volume(BDEV_TYPE_NOT_BOOT)) )
		add_drive(bdev);
	while( (bdev=bdev_get_volume(BDEV_TYPE_BOOT)) )
		add_drive(bdev);
	
#if 0	
	session_save_proc( save_blk_state, NULL, kDynamicChunk );
	if( loading_session() )
		check_session_state();
#endif
	return 1;
}

void
osi_blk_cleanup( void )
{
	drive_t *dr, *next;

	if( !osi_driver )
		return;

	for( dr=s_first_drive; dr; dr=next ){
		bdev_close_volume( dr->bdev );
		next = dr->next;
		free( dr );
	}
	s_first_drive = NULL;

	free_osi_driver( osi_driver );
	osi_driver = NULL;
}

static void
add_drive( bdev_desc_t *bdev )
{
	drive_t *d = malloc( sizeof(drive_t) );
	CLEAR( *d );

	d->bdev = bdev;
	d->id = s_next_id++;
	d->blkoffs = bdev->soffs >> 9;
	d->nblks = bdev->size >> 9;
	d->locked = !(bdev->flags & BF_ENABLE_WRITE);

	d->next = s_first_drive;
	s_first_drive = d;
}

#if 0
static int
save_blk_state( void )
{
	drive_t *drv; 
	int i;
	
	for( i=0, drv=s_first_drive; drv; drv=drv->next, i++ ) {
		if( get_mdb_checksum( drv, &drv->mdb_checksum ) ) {
			printm("Could not calculate MDB checksum\n");
			return 1;
		}
		if( write_session_data( "oblk", i, (char*)drv, sizeof(drive_t) ))
			return 1;
	}
	return 0;
}

static void
check_session_state( void )
{
	drive_t *d, drv, drv2;
	int i, err;
	
	for( d=s_first_drive, i=0; d; d=d->next, i++) {
		drv = *d;
		if( (err = read_session_data( "oblk", i, (char*)&drv2, sizeof(drive_t))) == 1 ) {
			session_failure("Number of block devices has changed\n");
			break;
		}
		if( err < 0 )
			session_failure("Could not read block driver state\n");

		/* Compare drv2 and drv */
		drv2.next = drv.next = NULL;
		drv2.name = drv.name = NULL;
		drv2.fd = drv.fd = 0;
		err = 0;
		if( get_mdb_checksum( d, &drv.mdb_checksum ) )
			session_failure("Failed to obtain MDB checksum\n");
		while( memcmp(&drv2, &drv, sizeof(drive_t)) ){
			if( drv2.locked && !drv.locked ) {
				d->locked = drv.locked = drv2.locked;
				err = 1;
				continue;
			}
			if( drv2.name_checksum == drv.name_checksum 
			    && drv2.mdb_checksum != drv.mdb_checksum ) 
			{
				printm("*** Device '%s' has been modified since the session was saved!\n", d->name );
				session_failure("Can't restore a session if one of the volumes has been modified.\n");
			}
			session_failure("Block device state in saved session differ!\n");
		}
		if( err == 1 )
			printm("Forcing read-only for block device (session was read-only)\n");
	}
	if( get_session_data_size("oblk", i) >=0 )
		session_failure("Number of block devices has changed\n");
}


/* Calculate a checksum for the HFS(+) MDB (master directory block). 
 * In particular, the modification date is included in the checksum.
 */
static int 
get_hfs_checksum( drive_t *drv, ulong *checksum )
{
	hfs_mdb_t	mdb;
	hfs_plus_mdb_t	mdb_plus;
	ulong		val=0;

	blk_lseek( drv->fd, drv->blkoffs+2, SEEK_SET );
	read( drv->fd, &mdb, sizeof(mdb) );
	if( hfs_get_ushort(mdb.drSigWord) != HFS_SIGNATURE )
		return -1;

	// printm("HFS volume detected\n");

	if( hfs_get_ushort(mdb.drEmbedSigWord) == HFS_PLUS_SIGNATURE ) {
		int sblock = hfs_get_ushort(mdb.drAlBlSt);
		sblock += (hfs_get_uint(mdb.drAlBlkSiz) / 512) * (hfs_get_uint(mdb.drEmbedExtent) >> 16);

		blk_lseek( drv->fd, drv->blkoffs + sblock + 2, SEEK_SET );
		read( drv->fd, &mdb_plus, sizeof(mdb_plus) );

		if( mdb_plus.signature != HFS_PLUS_SIGNATURE ) {
			printm("HFS_PLUS_SIGNATURE expected\n");
			return -1;
		}
		val += calc_checksum( (char*)&mdb_plus, sizeof(mdb_plus) );

		// printm("HFS+ volume detected\n");
	}
	val += calc_checksum( (char*)&mdb, sizeof(mdb_plus) );
	*checksum = val;

	// printm("HFS-MDB checksum %08lX\n", *checksum );
	return 0;
}

static int
get_mdb_checksum( drive_t *drv, ulong *checksum )
{
	char buf[2048];

	if( !get_hfs_checksum( drv, checksum ) )
		return 0;

	if( !drv->locked || !(drv->flags & bf_force) ) {
		printm("Save session does not support r/w volumes which are not HFS(+)\n");
		return 1;
	}

	/* fallback - read the first four sectors */
	blk_lseek( drv->fd, drv->blkoffs, SEEK_SET );
	read( drv->fd, &buf, sizeof(buf) );

	*checksum = calc_checksum( buf, sizeof(buf) );
	return 0;
}
#endif


/************************************************************************/
/*	OS interface							*/
/************************************************************************/

/* int index, ulong mphys_retinfo_ptr -- retcode
 *   retcode:   -1 error, 0 = OK, 1 -- index out of range 
 */ 
static int
osip_blk_drive_info( int sel, int *params )
{
	int ind = params[0];
	DriveInfoOSI *info;
	drive_t *dr;
	
	if( mphys_to_lvptr( params[1], (char**)&info ) ) {
		printm("osip_blk_read: Not in RAM access\n");
		return -1;
	}

	for( dr = s_first_drive; dr ; dr=dr->next ) {
		if( ind-- != 0 )
			continue;
		break;
	}

	if( dr ) {
		info->linux_id = dr->id;
		info->num_blocks = dr->nblks;
		info->locked = dr->locked;
		return 0;
	}
	return 1;
}


#define ID_TO_DRV( x ) ({ 						\
	drive_t *dr;  						   	\
	for( dr = s_first_drive; dr && dr->id != (x); dr=dr->next )	\
		;  							\
	if( !dr ) { 							\
		printm("osi_blk: No such ID!\n"); 			\
		return -1; 						\
	}								\
	dr; 								\
})


/* ulong mphys, ulong pos, ulong size, long id, int last */
static int
osip_blk_read( int sel, int *params )
{
	drive_t 	*dr; 
	char 		*ptr;
	ulong		size = params[2] << 9;
	bdev_desc_t	*bdev;

	dr = ID_TO_DRV( params[3] );
	bdev = dr->bdev;

	VPRINT("osip_blk_read mphys: pos %08x, size 0x%x\n",
		params[1]+dr->blkoffs, size ); 
	
	if( mphys_to_lvptr( params[0], &ptr ) ) {
		printm("osip_blk_read: Not in RAM access\n");
		return -1;
	}

	if( blk_lseek( bdev->fd, params[1]+dr->blkoffs, 0 ) < 0 ) {
		LOG("blk_lseek error\n");
		return -1;
	}
	if( params[1]+params[2] > dr->nblks ){
		printm("osi_blk: Read outside partition\n");
		return -1;
	}
	if( read( bdev->fd, ptr, size ) != size ) {
		perrorm("blk read: ");
		return -1;
	}
	return 0;
}

/* long mphys, ulong pos, ulong size, long id, int last */
static int
osip_blk_write( int sel, int *params )
{
	ulong		size = params[2] << 9;
	bdev_desc_t	*bdev;
	drive_t		*dr;
	char		*ptr;

	dr = ID_TO_DRV( params[3] );
	bdev = dr->bdev;

	VPRINT("osip_blk_write mphys: %08x, pos %08x, size 0x%x\n",
	       params[0], params[1]+dr->blkoffs, size );

	if( dr->locked )
		return -1;
	if( params[1]+params[2] > dr->nblks ){
		printm("osi_blk: Write outside partition!\n");
		return -1;
	}
	if( blk_lseek( bdev->fd, params[1]+dr->blkoffs, 0 ) < 0 ){
		LOG("blk_lseek error\n");
		return -1;
	}
	if( mphys_to_lvptr( params[0], &ptr ) ) {
		printm("osip_blk_write: Not in RAM access\n");
		return -1;
	}
	if( write( bdev->fd, ptr, size ) != size ) {
		LOG("Write error");
		return -1;
	}
	return 0;
}
