/* 
 *   Creation Date: <1999/11/03 01:50:43 samuel>
 *   Time-stamp: <2000/12/31 04:32:42 samuel>
 *   
 *	<rtas.c>
 *	
 *	Run Time Abstraction Services (RTAS)
 *   
 *   Copyright (C) 1999, 2000 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"

#include "rtas.h"
#include "promif.h"
#include "debugger.h"
#include "memory.h"
#include "os_interface.h"
#include "nvram.h"
#include "io.h"
#include "../drivers/include/pci.h"

/* ofglue.S exports */
extern char of_rtas_start[], of_rtas_end[];

static int osip_of_rtas( int sel, int *params );

static int rtas_size = 0;

typedef int rtas_proc( ulong args[], ulong ret[] );

typedef struct rtas_args {
	ulong		token;
        int 		nargs;
        int 		nret;
        ulong 		args[10];			/* MAX NUM ARGS! */
} rtas_args_t;

typedef struct rtas_cmd
{
	ulong		token;
	char 		*service;
	rtas_proc	*proc;
	int		nargs;
	int		nret;
	int		translate_args;
} rtas_cmd_t;

#define A0		0x1
#define A1		0x2
#define A2		0x4
#define A3		0x8

static rtas_proc	rc_restart_rtas, rc_nvram_fetch, rc_nvram_store, rc_nop;
static rtas_proc        rc_write_pcicfg, rc_read_pcicfg;

static rtas_cmd_t rtas_table[] = {
  {0xABCDEF01, "restart-rtas",  	rc_restart_rtas, 0,1, 0 },  /* -- status */
  {0xABCDEF02, "nvram-fetch", 		rc_nvram_fetch, 3,2, A1 }, /* offs buf length -- status actlen */
  {0xABCDEF03, "nvram-store", 		rc_nvram_store, 3,2, A1 }, /* offs buf length -- status actlen */
  {0xABCDEF04, "get-time-of-day",  	rc_nop,		0,8, 0 },   /* -- status y m d hour min sec ns */
  {0xABCDEF05, "set-time-of-day",  	rc_nop,		7,1, 0 },   /* y m d hour min sec ns -- status */
  {0xABCDEF06, "set-time-for-power-on", rc_nop,		7,1, 0 },   /* y m d hour min sec ns -- status */
  {0xABCDEF07, "event-scan", 	 	rc_nop,		4,1, 0 },   /* eventmask critical buffer len -- status */
  {0xABCDEF08, "check-exception",  	rc_nop,		6,1, 0 },   /* vector_offs info evmask critical buf len -- status */
  {0xABCDEF09, "read-pci-config",  	rc_read_pcicfg,	2,2, 0 },   /* config_addr size -- status value */
  {0xABCDEF0A, "write-pci-config", 	rc_write_pcicfg,3,1, 0 },   /* config_addr size value -- status */
  /* 0xABCDEF0B display-character 1,1, ( char -- status ) */
  {0xABCDEF0C, "set-indicator",   	rc_nop,		3,1, 0 },   /* token index state -- status */
  /* 0xABCDEF0D get-sensor-state 2,2 ( token index -- status state ) */
  /* set-power-level 2,2 */
  /* get-power-level 1,2 */
  /* assume-power-management 0,1 ( -- success ) */
  /* relinquish-power-management 0,1 ( -- success ) */
  {0xABCDEF12, "power-off", 		rc_nop,		2,1, 0 },   /* power_on_mask_hi power_on_mask_lo -- status */
  /* suspend 3,2 */
  /* hibernate 3,1 */
  {0xABCDEF15, "system-reboot", 	rc_nop,		0,1, 0 },   /* -- status */
  /* SMP: */
  /* cache-control 2,1 */
  /* freeze-time-base */
  /* thaw-time-base */
  /* stop-self */
  /* start-cpu */
  /* ? */
  {0xABCDEF1E, "get-time-for-power-on", rc_nop,	1,1, 0 },
  {0,0,0,0,0,0 }
};

static char *s_rtas_name;
static int  s_rtas_nargs, s_rtas_nret;

/************************************************************************/
/*	FUNCTIONS							*/
/************************************************************************/

void 
rtas_init( void )
{
	ulong size=2;
	mol_device_node_t *dn;
	
	while( size < (ulong)of_rtas_end - (ulong)of_rtas_start )
		size *= 2;

	dn = prom_find_dev_by_path( "/rtas" );
	if( !dn ){
		printm("----> The node /rtas is missing!\n" );
		return;
	}
	prom_add_property( dn, "rtas-size", (char*)&size, sizeof(ulong) );
	rtas_size = size;

	os_interface_add_proc( OSI_OF_RTAS, osip_of_rtas );

	/* XXX: make sure the correct tokens are in the device tree! */
}

void
rtas_cleanup( void )
{
}

ulong
rtas_instantiate( ulong mphys )
{
	if( verify_mrange( mphys, rtas_size ) != 0 ) {
		printm("----> Bad RTAS range!\n");
		return -1;
	}
	memcpy( mphys + ram.lvbase, of_rtas_start, rtas_size );

	printm("RTAS instantiated at %08lX\n", mphys );

	/* proc */
	return mphys;
}

/* Rtas argument buffer: function-token, num_iargs, num_oargs, first iarg ... first oarg */
int
osip_of_rtas( int sel, int *params )
{
	int 		i,j;
	rtas_args_t	*pb;
	ulong		largs[10];

/*	stop_emulation(); */

	if( mphys_to_lvptr( params[0], (char**)&pb ) != 0 ) {
		printm("RTAS parameter block not in RAM!\n");
		return 0;
	}
/*	printm("RTAS call, token: %08lX\n", pb->token ); */

	for(i=0; rtas_table[i].token; i++ ) {
		if( rtas_table[i].token != pb->token )
			continue;
		if( rtas_table[i].nargs != -1 && rtas_table[i].nargs != pb->nargs )
			continue;
		if( rtas_table[i].nret != -1 && rtas_table[i].nret != pb->nret )
			continue;
		break;
	}
	if( !rtas_table[i].token ){
		printm(">>>> Missing RTAS function %08lX (%d/%d)<<<<\n", 
		       pb->token, pb->nargs, pb->nret );
		return -1;
	}
	for(j=0; j<rtas_table[i].nargs; j++ ){
		largs[j] = pb->args[j];
		if( rtas_table[i].translate_args & (1<<j) && pb->args[j] ) {
			if( mphys_to_lvptr( largs[j], (char**)&largs[j] ) < 0 ) {
				printm("RTAS pointer argument not in RAM\n");
				return -1;
			}
		}
	}

	s_rtas_name = rtas_table[i].service;
	s_rtas_nargs = pb->nargs;
	s_rtas_nret = pb->nret;
	return (*rtas_table[i].proc)( largs, &pb->args[pb->nargs] );
}


/************************************************************************/
/*	RTAS CMDS							*/
/************************************************************************/

static int 
rc_nop( ulong args[], ulong ret[] ) 
{
	int i;
	printm(">>> Unimplemented RTAS '%s' (%d/%d)\n", s_rtas_name, s_rtas_nargs, s_rtas_nret );
	for( i=0; i<s_rtas_nargs; i++ )
		printm("  %08lX ", args[i] );
	printm("\n");
	
	return 0;	/* -1 might be better... */
}


/* -- status */
static int 
rc_restart_rtas( ulong args[], ulong ret[] ) 
{
	printm("******** restart-rtas ********\n");
	return 0;
}

/* offs buf length -- status actlen */
static int 
rc_nvram_fetch( ulong args[], ulong ret[] ) 
{
/*	printm("RTAS nvram-fetch %04lx %08lX %ld\n", args[0], args[1], args[2] );  */

	ret[1] = nvram_read( args[0], (char*)args[1], args[2] );
	ret[0] = 0;
	if( ret[1] != args[2] ){
		printm("---> nvram_read parameter error\n");
		ret[0] = -3;	/* parameter error */
	}
	/* printm("RTAS nvram-fetch %08lX [len=%d] %02X\n", args[0], args[2], 
		 *(char*)args[1] ); */
	return 0;
}

/* offs buf length -- status actlen */
static int 
rc_nvram_store( ulong args[], ulong ret[] ) 
{
/*	printm("nvram-store %04lx %08lX %ld\n", args[0], args[1], args[2] ); */

	ret[1] = nvram_write( args[0], (char*)args[1], args[2] );
	ret[0] = 0;
	if( ret[1] != args[2] ){
		printm("---> nvram_store parameter error\n");
		ret[0] = -3;	/* parameter error */
	}
	return 0;
}

/* config_addr size -- status value */
static int 
rc_read_pcicfg( ulong args[], ulong ret[] ) 
{
	ulong v;	
	/* 0,bus,devfn,reg */	
	v = read_pci_config( (args[0]>>16)&0xff, (args[0]>>8)&0xff, args[0]&0xff, args[1] );
	if( args[1] == 2 )
		v = ld_le32(&v) >> 16;
	if( args[1] == 4 )
		v = ld_le32(&v);
	ret[1] = v;
	
	/* printm("RTAS: read_pci_config (%ld:%02lX) + 0x%02lx [%ld]  :  %08lX\n", 
		   (args[0]>>16)&0xff, (args[0]>>8)&0xff, args[0]&0xff, args[1], ret[1]  );*/
	ret[0] = 0;
	return 0;
}

/* config_addr size value -- status */
static int
rc_write_pcicfg( ulong args[], ulong ret[] ) 
{
	/* printm("RTAS: write_pci_config (%ld:%02lX) + 0x%02lx %08lX [%ld]\n", 
		   (args[0]>>16)&0xff, (args[0]>>8)&0xff, args[0]&0xff, args[2], args[1] ); */
	if( args[1] == 2 )
		args[2] = ld_le32( &args[2] ) >> 16;
	if( args[1] == 4 )
		args[2] = ld_le32( &args[2] );

	write_pci_config( (args[0]>>16)&0xff, (args[0]>>8)&0xff, args[0]&0xff, args[2], args[1] );
	ret[0] = 0;
	return 0;
}

