package com.ibm.bsf;

import java.lang.reflect.*;
import java.util.*;
import java.io.*;
import java.beans.*;

import com.ibm.bsf.util.*;

/**
 * This class is the entry point to the bean scripting framework. An
 * application wishing to integrate scripting to a Java app would 
 * place an instance of a BSFManager in their code and use its services
 * to register the beans they want to make available for scripting,
 * load scripting engines, and run scripts. 
 * <p>
 * BSFManager serves as the registry of available scripting engines
 * as well. Loading and unloading of scripting engines is
 * supported as well.
 * 
 * @author   Sanjiva Weerawarana
 * @author   Matthew J. Duftler
 * @author   Sam Ruby
 */
public class BSFManager {
  // table of registered scripting engines
  protected static Hashtable registeredEngines = new Hashtable ();

  // mapping of file extensions to languages
  protected static Hashtable extn2Lang = new Hashtable ();

  // table of scripting engine instances created by this manager. 
  // only one instance of a given language engine is created by a single
  // manager instance.
  protected Hashtable loadedEngines = new Hashtable ();

  // table of registered beans for use by scripting engines.
  protected ObjectRegistry objectRegistry = new ObjectRegistry ();

  // prop change support containing loaded engines to inform when any
  // of my interesting properties change
  protected PropertyChangeSupport pcs;

  // the class loader to use if a class loader is needed. Default is
  // he who loaded me (which may be null in which case its Class.forName).
  protected ClassLoader classLoader = getClass().getClassLoader();

  // temporary directory to use to dump temporary files into. Note that
  // if class files are dropped here then unless this dir is in the 
  // classpath or unless the classloader knows to look here, the classes
  // will not be found.
  protected String tempDir = ".";

  // classpath used by those that need a classpath
  protected String classPath;

  // debug stream
  protected PrintStream debugStream = System.err;
  
  // stores BSFDeclaredBeans representing objects introduced by a client of
  // BSFManager
  protected Vector declaredBeans = new Vector ();

  // debug mode flag
  boolean debug = false;

  //////////////////////////////////////////////////////////////////////////

  /**
   * Register a scripting engine in the static registry of the 
   * BSFManager.
   *
   * @param lang string identifying language
   * @param engineClassName fully qualified name of the class interfacing
   *        the language to BSF.
   * @param extensions array of file extensions that should be mapped to
   *        this language type. may be null.
   */
  public static void registerScriptingEngine (String lang,
					      String engineClassName,
					      String[] extensions) {
    registeredEngines.put (lang, engineClassName);
    if (extensions != null) {
      for (int i = 0; i < extensions.length; i++) {
	extn2Lang.put (extensions[i], lang);
      }
    }
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Determine whether a language is registered.
   *
   * @param lang string identifying a language
   *
   * @return true iff it is
   */
  public static boolean isLanguageRegistered (String lang) {
    return (registeredEngines.get (lang) != null);
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Determine the language of a script file by looking at the file
   * extension. 
   *
   * @param filename the name of the file
   *
   * @return the scripting language the file is in if the file extension
   *         is known to me (must have been registered via 
   *         registerScriptingEngine).
   *
   * @exception BSFException if file's extension is unknown.
   */
  public static String getLangFromFilename (String fileName) 
                                                     throws BSFException {
    int dotIndex = fileName.lastIndexOf (".");

    if (dotIndex != -1) {
      String extn = fileName.substring (dotIndex + 1);
      String lang = (String) extn2Lang.get (extn);

      if (lang != null) {
        return lang;
      }
    }
    throw new BSFException (BSFException.REASON_OTHER_ERROR,
			    "file extension missing or unknown: " +
			    "unable to determine language for '" +
			     fileName + "'");
  }

  //////////////////////////////////////////////////////////////////////////

  public BSFManager () {
    pcs = new PropertyChangeSupport (this);
  }

  //////////////////////////////////////////////////////////////////////////
  //
  // BSF Manager Properties
  //
  //////////////////////////////////////////////////////////////////////////

  /**
   * Set the object registry used by this manager. By default a new
   * one is created when the manager is new'ed and this overwrites 
   * that one.
   *
   * @param objectRegistry the registry to use
   */
  public void setObjectRegistry (ObjectRegistry objectRegistry) {
    this.objectRegistry = objectRegistry;
  }
  
  /**
   * Return the current object registry of the manager.
   *
   * @return the current registry.
   */
  public ObjectRegistry getObjectRegistry () {
    return objectRegistry;
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Send debug output to this print stream.  Default is System.err.
   *
   * @param debugStream stream to direct debug output to.
   */
  public void setDebugStream (PrintStream debugStream) {
    pcs.firePropertyChange ("debugStream", this.debugStream, debugStream);
    this.debugStream = debugStream;
  }

  /**
   * Get debug stream
   */
  public PrintStream getDebugStream () {
    return debugStream;
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Turn on off debugging output to System.err. Default is off (false).
   *
   * @param debug value to set debug flag to
   */
  public void setDebug (boolean debug) {
    pcs.firePropertyChange ("debug", new Boolean (this.debug), 
			    new Boolean (debug));
    this.debug = debug;
  }

  /**
   * Get debug status
   */
  public boolean getDebug () {
    return debug;
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Set the classpath for those that need to use it. Default is the value
   * of the java.class.path property.
   *
   * @param classPath the classpath to use
   */
  public void setClassPath (String classPath) {
    pcs.firePropertyChange ("classPath", this.classPath, classPath);
    this.classPath = classPath;
  }

  /**
   * Get classPath
   */
  public String getClassPath () {
    if (classPath == null) {
      try {
	classPath = System.getProperty ("java.class.path");
      } catch (Throwable t) {
	// prolly a security exception .. so no can do
      }
    }
    return classPath;
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Set the class loader for those that need to use it. Default is he
   * who loaded me or null (i.e., its Class.forName).
   *
   * @param classLoader the class loader to use.
   */
  public void setClassLoader (ClassLoader classLoader) {
    pcs.firePropertyChange ("classLoader", this.classLoader, classLoader);
    this.classLoader = classLoader;
  }

  /**
   * Get classLoader
   */
  public ClassLoader getClassLoader () {
    return classLoader;
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Temporary directory to put stuff into (for those who need to). Note
   * that unless this directory is in the classpath or unless the classloader
   * knows to look in here, any classes here will not be found! BSFManager
   * provides a service method to load a class which uses either the 
   * classLoader provided by the class loader property or, if that fails,
   * a class loader which knows to load from the tempdir to try to load
   * the class. Default value of tempDir is "." (current working dir). 
   *
   * @param tempDir the temporary directory
   */
  public void setTempDir (String tempDir) {
    pcs.firePropertyChange ("tempDir", this.tempDir, tempDir);
    this.tempDir = tempDir;
  }

  /**
   * Get tempDir
   */
  public String getTempDir () {
    return tempDir;
  }


  //////////////////////////////////////////////////////////////////////////
  //
  // Bean scripting framework services
  //
  //////////////////////////////////////////////////////////////////////////

  /**
   * Load a scripting engine based on the lang string identifying it.
   *
   * @param lang string identifying language
   * @exception BSFException if the language is unknown (i.e., if it
   *            has not been registered) with a reason of 
   *            REASON_UNKNOWN_LANGUAGE. If the language is known but
   *            if the interface can't be created for some reason, then
   *            the reason is set to REASON_OTHER_ERROR and the actual
   *            exception is passed on as well.
   */
  public BSFEngine loadScriptingEngine (String lang) throws BSFException {
    // if its already loaded return that
    BSFEngine eng = (BSFEngine) loadedEngines.get (lang);
    if (eng != null) {
      return eng;
    }

    // is it a registered language?
    String engineClassName = (String) registeredEngines.get (lang);
    if (engineClassName == null) {
      throw new BSFException (BSFException.REASON_UNKNOWN_LANGUAGE,
			      "unsupported language: " + lang);
    }
    
    // create the engine and initialize it. if anything goes wrong
    // except.
    try {
      Class engineClass = (classLoader == null) ? 
	Class.forName (engineClassName) :
	classLoader.loadClass (engineClassName);
      eng = (BSFEngine) engineClass.newInstance ();
      eng.initialize (this, lang, declaredBeans);
      loadedEngines.put (lang, eng);
      pcs.addPropertyChangeListener (eng);
      return eng;
    } catch (BSFException e) {
      throw e; ///Already a bsf exception just rethrow it.
    } catch (Throwable t) {
      throw new BSFException (BSFException.REASON_OTHER_ERROR,
			      "unable to load language: " + lang,
			      t);
    }
  }

  //////////////////////////////////////////////////////////////////////////

  /** 
   * Registering a bean allows a scripting engine or the application to 
   * access that bean by name and to manipulate it.
   *
   * @param beanName name to register under
   * @param bean     the bean to register
   */
  public void registerBean (String beanName, Object bean) {
    objectRegistry.register (beanName, bean);
  }

  //////////////////////////////////////////////////////////////////////////

  /** 
   * Unregister a previously registered bean. Silent if name is not found.
   *
   * @param beanName name of bean to unregister
   */
  public void unregisterBean (String beanName) {
    objectRegistry.unregister (beanName);
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * return a handle to a bean registered in the bean registry by the
   * application or a scripting engine. Returns null if bean is not found. 
   *
   * @param beanName name of bean to look up
   *
   * @return the bean if its found or null
   */
  public Object lookupBean (String beanName) {
    try {
      return objectRegistry.lookup (beanName);
    } catch (IllegalArgumentException e) {
      return null;
    }
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Declare a bean. The difference between declaring and registering 
   * is that engines are spsed to make declared beans "pre-available"
   * in the scripts as far as possible. That is, if a script author
   * needs a registered bean, he needs to look it up in some way. However
   * if he needs a declared bean, the language has the responsibility to
   * make those beans avaialable "automatically."
   * <p>
   * When a bean is declared it is automatically registered as well
   * so that any declared bean can be gotton to by looking it up as well.
   * <p>
   * If any of the languages that are already running in this manager
   * says they don't like this (by throwing an exception) then this
   * method will simply quit with that exception. That is, any engines
   * that come after than in the engine enumeration will not even be
   * told about this new bean. 
   * <p>
   * So, in general its best to declare beans before the manager has
   * been asked to load any engines because then the user can be informed
   * when an engine rejects it. Also, its much more likely that an engine
   * can declare a bean at start time than it can at any time.
   *
   * @param beanName name to declare bean as
   * @param bean     the bean that's being declared
   * @param type     the type to represent the bean as
   *
   * @exception BSFException if any of the languages that are already
   *            running decides to throw an exception when asked to
   *            declare this bean.
   */ 
  public void declareBean (String beanName, Object bean, Class type)
                                                          throws BSFException {
    registerBean (beanName, bean);

    BSFDeclaredBean tempBean = new BSFDeclaredBean (beanName, bean, type);
    declaredBeans.addElement (tempBean);

    Enumeration enginesEnum = loadedEngines.elements ();
    BSFEngine   engine;
    while (enginesEnum.hasMoreElements ()) {
      engine = (BSFEngine) enginesEnum.nextElement ();
      engine.declareBean (tempBean);
    }
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Undeclare a previously declared bean. This removes the bean from
   * the list of declared beans in the manager as well as asks every
   * running engine to undeclared the bean. As with above, if any
   * of the engines except when asked to undeclare, this method does
   * not catch that exception. Quietly returns if the bean is unknown.
   *
   * @param beanName name of bean to undeclare
   *
   * @exception BSFException if any of the languages that are already
   *            running decides to throw an exception when asked to
   *            undeclare this bean.
   */
  public void undeclareBean (String beanName) throws BSFException {
    unregisterBean (beanName);

    BSFDeclaredBean tempBean  = null;
    for (int i = 0; i < declaredBeans.size (); i++) {
      tempBean = (BSFDeclaredBean) declaredBeans.elementAt (i);
      if (tempBean.name.equals (beanName)) {
	break;
      }
    }

    if (tempBean != null) {
      declaredBeans.removeElement (tempBean);

      Enumeration enginesEnum = loadedEngines.elements ();
      while (enginesEnum.hasMoreElements ()) {
        BSFEngine engine = (BSFEngine) enginesEnum.nextElement ();
        engine.undeclareBean (tempBean);
      }
    }
  }


  //////////////////////////////////////////////////////////////////////////
  //
  // Convenience functions for exec'ing and eval'ing scripts directly
  // without loading and dealing with engines etc..
  //
  //////////////////////////////////////////////////////////////////////////

  /**
   * Execute the given script of the given language.
   *
   * @param lang     language identifier
   * @param source   (context info) the source of this expression
            (e.g., filename)
   * @param lineNo   (context info) the line number in source for expr
   * @param columnNo (context info) the column number in source for expr
   * @param script   the script to execute
   *
   * @exception BSFException if anything goes wrong while running the script
   */
  public void exec (String lang, String source, int lineNo, int columnNo,
		    Object script) throws BSFException {
    BSFEngine e = loadScriptingEngine (lang);
    e.exec (source, lineNo, columnNo, script);
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Evaluate the given expression of the given language and return the
   * resulting value.
   *
   * @param lang language identifier
   * @param source (context info) the source of this expression
            (e.g., filename)
   * @param lineNo (context info) the line number in source for expr
   * @param columnNo (context info) the column number in source for expr
   * @param expr the expression to evaluate
   *
   * @exception BSFException if anything goes wrong while running the script
   */
  public Object eval (String lang, String source, int lineNo,
		      int columnNo, Object expr) throws BSFException {
    BSFEngine e = loadScriptingEngine (lang);
    return e.eval (source, lineNo, columnNo, expr);
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Apply the given anonymous function of the given language to the given
   * parameters and return the resulting value.
   *
   * @param lang language identifier
   * @param source (context info) the source of this expression
            (e.g., filename)
   * @param lineNo (context info) the line number in source for expr
   * @param columnNo (context info) the column number in source for expr
   * @param funcBody the multi-line, value returning script to evaluate
   * @param paramNames the names of the parameters above assumes
   * @param arguments values of the above parameters
   *
   * @exception BSFException if anything goes wrong while running the script
   */
  public Object apply (String lang, String source, int lineNo,
		       int columnNo, Object funcBody, Vector paramNames,
		       Vector arguments) throws BSFException {
    BSFEngine e = loadScriptingEngine (lang);
    return e.apply (source, lineNo, columnNo, funcBody, paramNames, arguments);
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Compile the given expression of the given language into the given 
   * <tt>CodeBuffer</tt>.
   *
   * @param lang     language identifier
   * @param source   (context info) the source of this expression
            (e.g., filename)
   * @param lineNo   (context info) the line number in source for expr
   * @param columnNo (context info) the column number in source for expr
   * @param expr     the expression to compile
   * @param cb       code buffer to compile into
   *
   * @exception BSFException if any error while compiling the expression
   */
  public void compileExpr (String lang, String source, int lineNo,
			   int columnNo, Object expr, CodeBuffer cb)
                                                    throws BSFException {
    BSFEngine e = loadScriptingEngine (lang);
    e.compileExpr (source, lineNo, columnNo, expr, cb);
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Compile the given script of the given language into the given 
   * <tt>CodeBuffer</tt>.
   *
   * @param lang     language identifier
   * @param source   (context info) the source of this script
            (e.g., filename)
   * @param lineNo   (context info) the line number in source for script
   * @param columnNo (context info) the column number in source for script
   * @param script   the script to compile
   * @param cb       code buffer to compile into
   *
   * @exception BSFException if any error while compiling the script
   */
  public void compileScript (String lang, String source, int lineNo,
           int columnNo, Object script, CodeBuffer cb)
                                                    throws BSFException {
    BSFEngine e = loadScriptingEngine (lang);
    e.compileScript (source, lineNo, columnNo, script, cb);
  }

  //////////////////////////////////////////////////////////////////////////

  /**
   * Compile the application of the given anonymous function of the given
   * language to the given parameters into the given <tt>CodeBuffer</tt>.
   *
   * @param lang language identifier
   * @param source (context info) the source of this expression
            (e.g., filename)
   * @param lineNo (context info) the line number in source for expr
   * @param columnNo (context info) the column number in source for expr
   * @param funcBody the multi-line, value returning script to evaluate
   * @param paramNames the names of the parameters above assumes
   * @param arguments values of the above parameters
   * @param cb       code buffer to compile into
   *
   * @exception BSFException if anything goes wrong while running the script
   */
  public void compileApply (String lang, String source, int lineNo,
			    int columnNo, Object funcBody, Vector paramNames,
			    Vector arguments, CodeBuffer cb)
       throws BSFException {
    BSFEngine e = loadScriptingEngine (lang);
    e.compileApply (source, lineNo, columnNo, funcBody, paramNames,
		    arguments, cb);
  }


  //////////////////////////////////////////////////////////////////////////
  //
  // pre-register engines that BSF supports off the shelf
  //
  //////////////////////////////////////////////////////////////////////////

  static {
    try {
      ResourceBundle rb =
	  ResourceBundle.getBundle("com.ibm.bsf.Languages");

      Enumeration keys = rb.getKeys();
      while (keys.hasMoreElements()) {
	  String key = (String)keys.nextElement();
	  String value = rb.getString(key);

          StringTokenizer tokens = new StringTokenizer(value, ",");
	  String className = (String)tokens.nextToken();

	  // get the extensions for this language
	  String exts = (String)tokens.nextToken();
	  StringTokenizer st = new StringTokenizer(exts, "|");
	  String[] extensions = new String[st.countTokens()];
	  for (int i = 0; st.hasMoreTokens(); i++) {
	    extensions[i] = ((String)st.nextToken()).trim ();
	  }
/*
	  URL codebases[] = null;
	  if (tokens.hasMoreTokens()) {
	      codebases = new URL[tokens.countTokens() - 1];
	      int i = 0;
	      while (tokens.hasMoreTokens()) {
		  String urlString = (String)tokens.nextToken();
		  try {
		      codebases[i++] = new URL(urlString);
		  } catch (MalformedURLException mue) {
		      System.err.println("Malformed URL: " + urlString);
		  }
	      }
	  }
*/

	  registerScriptingEngine (key, className, extensions);
      }

    } catch (NoSuchElementException nsee) {
      nsee.printStackTrace ();
      System.err.println("Syntax error in Languages resource bundle");
    } catch (MissingResourceException mre) {
      mre.printStackTrace ();
      System.err.println("Initialization error: " + mre.toString());
    }
  }

  /**
   * Gracefully terminate all engines
   */
  public void terminate (){
    Enumeration enginesEnum = loadedEngines.elements ();
    BSFEngine   engine;
    while (enginesEnum.hasMoreElements ()) {
      engine = (BSFEngine) enginesEnum.nextElement ();
      engine.terminate ();
    }

    loadedEngines = new Hashtable ();
  }
}
