/*********************************************************************
 *
 *	Copyright (C) 1999 Nathan Fiedler
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * PROJECT:     Utilities
 * MODULE:      ResourcePool Test
 * FILE:        TestResourcePool.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      nf      10/24/99         Initial version
 *
 * DESCRIPTION:
 *      Implements test-wrapper for the ResourcePool class.
 *
 ********************************************************************/

package com.bluemarsh.util.test;

import com.bluemarsh.util.IntHashtable;
import com.bluemarsh.util.ResourcePool;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

/**
 * Test-wrapper for the ResourcePool class.
 *
 * @author  Nathan Fiedler
 * @version 1.0  1/9/99
 */
public class TestResourcePool {
    /** Table of usage amounts. Tracks the number of times
     * a given resource was pulled from the pool. */
    protected IntHashtable usageTable = new IntHashtable();

    /**
     * Main method, tests the ResourcePool class.
     *
     * @param  args  command-line string arguments
     */
    public static void main(String[] args) {
        Runtime rt = Runtime.getRuntime();
        int max = Integer.MIN_VALUE;
        int min = Integer.MAX_VALUE;
        int total = 0;
        int loopCount = 10;
        for (int i = 0; i < loopCount; i++) {
            System.out.println("Free Memory: " + rt.freeMemory());
            TestResourcePool tester = new TestResourcePool();
            int c = tester.runTest();
            if (c > max) {
                max = c;
            }
            if (c < min) {
                min = c;
            }
            total += c;
            rt.gc();
        }
        System.out.println("Maximum: " + max);
        System.out.println("Minimum: " + min);
        System.out.println("Total  : " + total);
        System.out.println("Average: " + total / loopCount);
    } // main

    /**
     * Performs the test.
     *
     * @return  Number of distinct resources allocated during test.
     */
    protected int runTest() {
        ResourcePool pool = new ResourcePool(String.class);
        int loopCount = 100;
        List threads = new ArrayList(loopCount);

        // Start a bunch of helpers.
        for (int i = 0; i < loopCount; i++) {
            Helper h = new Helper(pool,
                                  (int) (Math.random() * 10 + 1),
                                  (int) (Math.random() * 300 + 10),
                                  (int) (Math.random() * 100 + 50));
            Thread th = new Thread(h);
            th.start();
            threads.add(th);
        }

        // Wait for them to all die.
        System.out.println("Waiting for threads to die...");
        while (threads.size() > 0) {
            for (int i = threads.size() - 1; i >= 0; i--) {
                Thread th = (Thread) threads.get(i);
                if (!th.isAlive()) {
                    threads.remove(i);
                }
            }
            try {
                Thread.sleep(100);
            } catch (InterruptedException ie) {
            }
        }

        // Dump the contents of the usage table.
        int size = usageTable.size();
        System.out.println(size + " distinct resources.");
        Enumeration enum = usageTable.elements();
        while (enum.hasMoreElements()) {
            Integer i = (Integer) enum.nextElement();
            System.out.print(i.intValue() + ", ");
        }
        System.out.println();
        return size;
    } // runTest

    /**
     * Helper class. Runs a test.
     */
    protected class Helper implements Runnable {
        /** Resource pool object. */
        protected ResourcePool resourcePool;
        /** Number of times to run test loop. */
        protected int loopCount;
        /** Time test loop spends holding resource (ms). */
        protected int holdTime;
        /** Time test loop spends sleeping (ms). */
        protected int idleTime;

        /**
         * Constructs a Helper.
         */
        public Helper(ResourcePool pool, int count, int hold, int idle) {
            this.resourcePool = pool;
            this.loopCount = count;
            this.holdTime = hold;
            this.idleTime = idle;
        } // Helper

        /**
         * Run a single test loop.
         */
        public void run() {
            for (int i = 0; i < loopCount; i++) {
                // Get a resource from the pool.
                Object o = null;
                try {
                    o = resourcePool.getResource();
                } catch (InstantiationException inste) {
                    break;
                }

                // Using the resource, get its usage count.
                int hash = System.identityHashCode(o);
                Integer integer = (Integer) usageTable.get(hash);
                int count = 0;
                if (integer != null) {
                    count = integer.intValue();
                }

                // Update the resource's usage count.
                usageTable.put(hash, new Integer(count + 1));
                try {
                    // Hold the object for a while.
                    Thread.sleep(holdTime);
                    // Return the resource to the pool.
                    resourcePool.returnResource(o);
                    // Sleep for a while.
                    Thread.sleep(idleTime);
                } catch (InterruptedException ie) {
                }
            }
        } // run

        /**
         * Returns a string representation of this helper.
         *
         * @return  String.
         */
        public String toString() {
            StringBuffer buff = new StringBuffer("Helper=[");
            buff.append(", loopCount = " + loopCount);
            buff.append(", holdTime = " + holdTime);
            buff.append(", idleTime = " + idleTime);
            buff.append("]");
            return buff.toString();
        } // toString
    } // Helper
} // TestResourcePool
