/*****************************************************************************
 *                                                                           *
 *  This file is part of the BeanShell Java Scripting distribution.          *
 *  Documentation and updates may be found at http://www.beanshell.org/      *
 *                                                                           *
 *  Sun Public License Notice:                                               *
 *                                                                           *
 *  The contents of this file are subject to the Sun Public License Version  *
 *  1.0 (the "License"); you may not use this file except in compliance with *
 *  the License. A copy of the License is available at http://www.sun.com    * 
 *                                                                           *
 *  The Original Code is BeanShell. The Initial Developer of the Original    *
 *  Code is Pat Niemeyer. Portions created by Pat Niemeyer are Copyright     *
 *  (C) 2000.  All Rights Reserved.                                          *
 *                                                                           *
 *  GNU Public License Notice:                                               *
 *                                                                           *
 *  Alternatively, the contents of this file may be used under the terms of  *
 *  the GNU Lesser General Public License (the "LGPL"), in which case the    *
 *  provisions of LGPL are applicable instead of those above. If you wish to *
 *  allow use of your version of this file only under the  terms of the LGPL *
 *  and not to allow others to use your version of this file under the SPL,  *
 *  indicate your decision by deleting the provisions above and replace      *
 *  them with the notice and other provisions required by the LGPL.  If you  *
 *  do not delete the provisions above, a recipient may use your version of  *
 *  this file under either the SPL or the LGPL.                              *
 *                                                                           *
 *  Patrick Niemeyer (pat@pat.net)                                           *
 *  Author of Learning Java, O'Reilly & Associates                           *
 *  http://www.pat.net/~pat/                                                 *
 *                                                                           *
 *****************************************************************************/

package bsh;

import java.lang.reflect.*;
import java.io.*;
import java.util.Vector;

/**
    All of the reflection API code lies here.  It is in the form
	of static utilities.  See the design note about object wrappers 
	in LHS.java for lamentations regarding this.

	Note: More work to do in here to fix up the extended signature matching.
	need to work in a search along with findMostSpecificSignature...
*/
class Reflect {

    /*
		Invoke method on object, may be static, dynamic, or This
		Note: we could probably remove This handling and prevent it
		from coming here...
	*/
    public static Object invokeObjectMethod(
		Interpreter interpreter, Object object, String methodName, 
		Object[] args) 
		throws ReflectError, InvocationTargetException, EvalError 
	{
        Interpreter.debug("invoke Method " + methodName + " on object " 
			+ object + " with args (");

		if ( object instanceof This )
			return ((This)object).invokeMethod( methodName, args, interpreter );
        else
			return invokeMethod( object.getClass(), object, methodName, args );
    }

    /** 
		Invoke static method
	*/
    public static Object invokeStaticMethod(
		Class clas, String methodName, Object [] args)
        throws ReflectError, InvocationTargetException
    {
        Interpreter.debug("invoke static Method");
        return invokeMethod(clas, null, methodName, args);
    }

    public static Object getIndex(Object array, int index)
        throws ReflectError
    {
        try {
            Object val = Array.get(array, index);
            return wrapPrimitive(val, array.getClass().getComponentType());
        }
        catch(Exception e) {
            throw new ReflectError("Array access:" + e);
        }
    }

    public static void setIndex(Object array, int index, Object val)
        throws ReflectError
    {
        try {
            val = unwrapPrimitive(val);
            Array.set(array, index, val);
        }
        catch(Exception e) {
            throw new ReflectError("Array access:" + e);
        }
    }

    public static Object getStaticField(Class clas, String fieldName)
        throws ReflectError
    {
        return getFieldValue(clas, null, fieldName);
    }

    public static Object getObjectField(Object object, String fieldName)
        throws ReflectError
    {
		if ( object instanceof This )
			return ((This)object).namespace.getVariable( fieldName );
		else {
			try {
				return getFieldValue(object.getClass(), object, fieldName);
			} catch ( ReflectError e ) {
				// no field, try property acces

				if ( hasObjectPropertyGetter( object.getClass(), fieldName ) )
					return getObjectProperty( object, fieldName );
				else
					throw e;
			}
		}
    }

    static LHS getLHSStaticField(Class clas, String fieldName)
        throws ReflectError
    {
        Field f = getField(clas, fieldName);
        return new LHS(f);
    }

	/**
		Get an LHS reference to an object field.

		This method also deals with the field style property access.
		In the field does not exist we check for a property setter.
	*/
    static LHS getLHSObjectField(Object object, String fieldName)
        throws ReflectError
    {
		if ( object instanceof This )
			return new LHS(((This)object).namespace, fieldName );

		try {
			Field f = getField(object.getClass(), fieldName);
			return new LHS(object, f);
		} catch ( ReflectError e ) {
			// not a field, try property access

			if ( hasObjectPropertySetter( object.getClass(), fieldName ) )
				return new LHS( object, fieldName );
			else
				throw e;
		}
    }

    private static Object getFieldValue(
		Class clas, Object object, String fieldName) throws ReflectError
    {
        try {
            Field f = getField(clas, fieldName);
// experiment
//f.setAccessible(true);
            if(f == null)
                throw new ReflectError("internal error 234423");

            Object value = f.get(object);
            Class returnType = f.getType();
            return wrapPrimitive(value, returnType);

        }
        catch(NullPointerException e) {
            throw new ReflectError(
				"???" + fieldName + " is not a static field.");
        }
        catch(IllegalAccessException e) {
            throw new ReflectError("Can't access field: " + fieldName);
        }
    }

    private static Field getField(Class clas, String fieldName)
        throws ReflectError
    {
        try
        {
			// need to fix this for accessibility
			// this one only finds public 
            return clas.getField(fieldName);
			// this one doesn't find interfaces, etc.
            //return clas.getDeclaredField(fieldName);
        }
        catch(NoSuchFieldException e)
        {
            throw new ReflectError("No such field: " + fieldName );
        }
    }

    /*
        The full blown invoke method.
        Everybody should come here.

		Note: Method invocation could probably be speeded up if we eliminated
		the throwing of exceptions in the search for the proper method.
		We could probably cache our knowledge of method structure as well.
    */
    private static Object invokeMethod(
		Class clas, Object object, String name, Object[] args)
        throws ReflectError, InvocationTargetException
    {
        if(args == null)
            args = new Object[] { };

        // simple sanity check for voids
        // maybe this should have been caught further up?
        for(int i=0; i<args.length; i++)
            if(args[i] == Primitive.VOID)
                throw new ReflectError("Attempt to pass void argument " +
                    "(position " + i + ") to method: " + name);

        Class returnType = null;
        Object returnValue = null;

        Class[] types = getTypes(args);
        unwrapPrimitives(args);

        /*  This is structured poorly...  */
        try
        {
            try
            {
				Method m = findAccessibleMethod(clas, name, types);
                returnValue =  m.invoke(object, args);
                if(returnValue == null)
                    returnValue = Primitive.NULL;
                returnType = m.getReturnType();
            }
            catch(ReflectError e)
            {
                Interpreter.debug("Exact method " + StringUtil.methodString(name, types) +
                    " not found in '" + clas.getName() + "'");
            }

            if ( returnValue == null ) {
				if ( types.length == 0 )
					throw new ReflectError("No args method " + 
						StringUtil.methodString(name, types) + " not found in class'" + 
						clas.getName() + "'");

				// try to find an assignable method
				Method[] methods = clas.getMethods();
				Method m = findMostSpecificMethod(name, types, methods);

				if(m == null)
					m = findExtendedMethod(name, args, methods);

				if (m == null )
					throw new ReflectError("Method " + 
						StringUtil.methodString(name, types) + 
						" not found in class'" + clas.getName() + "'");

				// have the method
				m = findAccessibleMethod(
					clas, m.getName(), m.getParameterTypes());

				returnValue = m.invoke(object, args);
				returnType = m.getReturnType();
            }
        } catch(IllegalAccessException e) {
            throw new ReflectError( 
				"Cannot access method " + StringUtil.methodString(name, types) +
                " in '" + clas.getName() + "' :" + e);
        }

        return wrapPrimitive(returnValue, returnType);
    }

	/*
		Locate a version of the method that is accessible via a public 
		interface or through a public superclass.

		This solves the problem that arises when a package private class
		or private inner class implements a public interface or derives from
		a public type.
	*/
	static Method findAccessibleMethod( 
		Class clas, String name, Class [] types ) throws ReflectError 
	{
		Method meth = null;
		Vector classQ = new Vector();

		classQ.addElement( clas );
		while ( classQ.size() > 0 ) {
			Class c = (Class)classQ.firstElement();
			classQ.removeElementAt(0);

			// Is this it?
			if ( Modifier.isPublic( c.getModifiers() ) ) {
				try {
					meth = c.getDeclaredMethod( name, types );
					if ( /*meth != null &&*/
						Modifier.isPublic( meth.getModifiers() ) )
						return meth; // Yes, it is.
				} catch ( Exception e ) { 
					// ignore and move on
				}
			}
			// No, it is not.
			
			// Is this a class?
			if ( !c.isInterface() ) {
				Class superclass = c.getSuperclass();
				if ( superclass != null )
					classQ.addElement((Object)superclass);
			}

			// search all of its interfaces breadth first
			Class [] intfs = c.getInterfaces();
			for( int i=0; i< intfs.length; i++ )
				classQ.addElement((Object)intfs[i]);
		}
		
		throw new ReflectError( 
			"Can't find publically accessible version of method: "+
			StringUtil.methodString(name, types) +
			" in interfaces or class hierarchy of class "+clas.getName() );
	}

    private static Object wrapPrimitive(
		Object value, Class returnType) throws ReflectError
    {
        if(value == null)
            return Primitive.NULL;

        if(returnType == Void.TYPE)
            return Primitive.VOID;

        else
            if(returnType.isPrimitive())
            {
                if(value instanceof Number)
                    return new Primitive((Number)value);
                if(value instanceof Boolean)
                    return new Primitive((Boolean)value);
                if(value instanceof Character)
                    return new Primitive((Character)value);

                throw new ReflectError("Something bad happened");
            }
            else
                return value;
    }

    public static Class[] getTypes( Object[] args)
    {
        if(args == null)
            return new Class[0];

        Class[] types = new Class[args.length];

        for(int i=0; i<args.length; i++)
        {
            if(args[i] instanceof Primitive)
                types[i] = ((Primitive)args[i]).getType();
            else
                types[i] = args[i].getClass();
        }

        return types;
    }

    /*
        Replace Primitive wrappers with their java.lang wrapper values

        These barf if one of the args is void...  maybe these should throw
        an exception on void arg to force the rest of the code to clean up.
        There are places where we don't check right now... (constructors, index)
    */
    private static void unwrapPrimitives(Object[] args)
    {
        for(int i=0; i<args.length; i++)
            args[i] = unwrapPrimitive(args[i]);
    }

    private static Object unwrapPrimitive(Object arg)
    {
        if(arg instanceof Primitive)
            return((Primitive)arg).getValue();
        else
            return arg;
    }

    static Object constructObject(String clas, Object[] args)
        throws ReflectError, InvocationTargetException
    {
		Class c = BshClassManager.classForName( clas );
		if ( c == null )
			throw new ReflectError("Class not found: "+clas); 

		return constructObject( c, args );
	}

	/**
		Primary object constructor
	*/
    static Object constructObject(Class clas, Object[] args)
        throws ReflectError, InvocationTargetException
    {
        // simple sanity check for arguments
        for(int i=0; i<args.length; i++)
            if(args[i] == Primitive.VOID)
                throw new ReflectError("Attempt to pass void argument " +
                    "(position " + i + ") to constructor for: " + clas);

		if ( clas.isInterface() )
			throw new ReflectError(
				"Can't create instance of an interface: "+clas);

        Object obj = null;
        Class[] types = getTypes(args);
        unwrapPrimitives(args);
        Constructor con = null;

		/* 
			Find an appropriate constructor
			use declared here to see package and private as well
			(there are no inherited constructors to worry about) 
		*/
		Constructor[] constructors = clas.getDeclaredConstructors();
		con = findMostSpecificConstructor(types, constructors);

		if ( con == null )
			if ( types.length == 0 )
				throw new ReflectError(
					"Can't find default constructor for: "+clas);
			else
				con = findExtendedConstructor(args, constructors);

		if(con == null)
			throw new ReflectError("Can't find constructor: " 
				+ clas );

        try {
            obj = con.newInstance(args);
        } catch(InstantiationException e) {
            throw new ReflectError("the class is abstract ");
        } catch(IllegalAccessException e) {
            throw new ReflectError(
				"we don't have permission to create an instance");
        } catch(IllegalArgumentException e) {
            throw new ReflectError("the number of arguments was wrong");
        } 
		if (obj == null)
            throw new ReflectError("couldn't construct the object");

        return obj;
    }

    /*
        Implement JLS 15.11.2 for method resolution
    */
    static Method findMostSpecificMethod(
		String name, Class[] idealMatch, Method[] methods)
    {

		// Pull out the method signatures whos name matches
		Vector sigs = new Vector();
		Vector meths = new Vector();
		for(int i=0; i<methods.length; i++)
			if ( methods[i].getName().equals( name ) ) {
				meths.addElement( methods[i] );
				sigs.addElement( methods[i].getParameterTypes() );
			}

		Class [][] candidates = new Class [ sigs.size() ][];
		sigs.copyInto( candidates );

		int match = findMostSpecificSignature( idealMatch, candidates );
		if ( match == -1 )
			return null;
		else
			return (Method)meths.elementAt( match );
    }

	/**
		This uses the NameSpace.checkAssignableFrom() method to determine
		compatability of args.  This allows special (non standard Java) bsh 
		widening operations...
	*/
    static Method findExtendedMethod(
		String name, Object[] args, Method[] methods)
    {
        Method bestMatch = null;
        Object[] tempArgs = new Object[args.length];

        for(int i = 0; i < methods.length; i++) {
            Method currentMethod = methods[i];
            if ( name.equals( currentMethod.getName() )) {
                Class[] parameters = currentMethod.getParameterTypes();
		
				if ( parameters.length != args.length )
					continue;
                try {
                    for(int j = 0; j < parameters.length; j++)
                        tempArgs[j] = NameSpace.checkAssignableFrom( 
							args[j], parameters[j]);

                    // if you get here, all the arguments were assignable
                    System.arraycopy(tempArgs, 0, args, 0, args.length);
                    return currentMethod;
                } catch(EvalError e) {
                    // do nothing (exception breaks you out of the for loop).
                }
            }
        }

        return null;
    }

    /*
        This method should exactly parallel findMostSpecificMethod()
    */
    static Constructor findMostSpecificConstructor(Class[] idealMatch,
        Constructor[] constructors)
    {

		Class [][] candidates = new Class [ constructors.length ] [];
		for(int i=0; i< candidates.length; i++ )
			candidates[i] = constructors[i].getParameterTypes();

		int match = findMostSpecificSignature( idealMatch, candidates );
		if ( match == -1 )
			return null;
		else
			return constructors[ match ];
    }


	/**
		This uses the NameSpace.checkAssignableFrom() method to determine
		compatability of args.  This allows special (non standard Java) bsh 
		widening operations...
	*/
    static Constructor findExtendedConstructor(
		Object[] args, Constructor[] constructors )
    {
        Constructor bestMatch = null;
        Object[] tempArgs = new Object[args.length];

        for(int i = 0; i < constructors.length; i++)
        {
            Constructor currentConstructor = constructors[i];
            Class[] parameters = currentConstructor.getParameterTypes();
			if ( parameters.length != args.length )
				continue;
            try {
                for(int j = 0; j < parameters.length; j++)
                    tempArgs[j] = 
						NameSpace.checkAssignableFrom(args[j], parameters[j]);

                // if you get here, all the arguments were assignable
                System.arraycopy(tempArgs, 0, args, 0, args.length);
                return currentConstructor;
            }
            catch(EvalError e)
            {
                // do nothing (exception breaks you out of the for loop).
            }
        }

        return null;
    }



	/**
        Implement JLS 15.11.2
		Return the index of the most specific arguments match or -1 if no	
		match is found.
	*/
	static int findMostSpecificSignature(
		Class [] idealMatch, Class [][] candidates )
	{
		Class [] bestMatch = null;
		int bestMatchIndex = -1;

		for (int i=0; i < candidates.length; i++) {
			Class[] targetMatch = candidates[i];

            /*
                If idealMatch fits targetMatch and this is the first match 
				or targetMatch is more specific than the best match, make it 
				the new best match.
            */
			if ( isAssignable(idealMatch, targetMatch ) &&
				((bestMatch == null) ||
					isAssignable( targetMatch, bestMatch )))
			{
				bestMatch = targetMatch;
				bestMatchIndex = i;
			}
		}

		if ( bestMatch != null ) {
			Interpreter.debug("best match: " + StringUtil.methodString("args",bestMatch));
			return bestMatchIndex;
		}
		else {
			Interpreter.debug("no match found");
			return -1;
		}
	}

	/**
		Determine if the 'from' signature is assignable to the 'to' signature
		'from' arg types, 'to' candidate types
		null value in 'to' type parameter indicates loose type.

		null value in either arg is considered empty array
	*/
    static boolean isAssignable(Class[] from, Class[] to)
    {
		if ( from == null )
			from = new Class[0];
		if ( to == null )
			to = new Class[0];

        if (from.length != to.length)
            return false;

        for(int i=0; i<from.length; i++)
        {
			// Null type indicates loose type.  Match anything.
			if ( to[i] == null )
				continue;

            // Let null arg type match any reference type
            if (from[i] == null) {

                if (!(to[i].isPrimitive()))
                    continue;
                else
                    return false;
            }

            if(!isAssignableFrom(to[i], from[i]))
                return false;
        }

        return true;
    }

    /**
		This base method is meant to address a deficiency of 
		Class.isAssignableFrom() which does not take primitive widening 
		conversions into account.

		Note that the checkAssignable() method in NameSpace is the primary
		bsh method for checking assignability.  It adds extended bsh
		conversions, etc.
	*/
    static boolean isAssignableFrom(Class lhs, Class rhs)
    {
        if(lhs.isPrimitive() && rhs.isPrimitive())
        {
            if(lhs == rhs)
                return true;

            // handle primitive widening conversions - JLS 5.1.2
            if((rhs == Byte.TYPE) && (lhs == Short.TYPE || lhs == Integer.TYPE ||
                lhs == Long.TYPE || lhs == Float.TYPE || lhs == Double.TYPE))
                    return true;

            if((rhs == Short.TYPE) && (lhs == Integer.TYPE || lhs == Long.TYPE ||
                lhs == Float.TYPE || lhs == Double.TYPE))
                    return true;

            if((rhs == Character.TYPE) && (lhs == Integer.TYPE || lhs == Long.TYPE ||
                lhs == Float.TYPE || lhs == Double.TYPE))
                    return true;

            if((rhs == Integer.TYPE) && (lhs == Long.TYPE || lhs == Float.TYPE ||
                lhs == Double.TYPE))
                    return true;

            if((rhs == Long.TYPE) && (lhs == Float.TYPE || lhs == Double.TYPE))
                return true;

            if((rhs == Float.TYPE) && (lhs == Double.TYPE))
                return true;
        }
        else
            if(lhs.isAssignableFrom(rhs))
                return true;

        return false;
    }

	private static String accessorName( String getorset, String propName ) {
        return getorset 
			+ String.valueOf(Character.toUpperCase(propName.charAt(0))) 
			+ propName.substring(1);
	}

    public static boolean hasObjectPropertyGetter( 
		Class clas, String propName ) 
	{
		String getterName = accessorName("get", propName );
		try {
			clas.getMethod( getterName, new Class [0] );
			return true;
		} catch ( NoSuchMethodException e ) {
			return false;
		}
	}

    public static boolean hasObjectPropertySetter( 
		Class clas, String propName ) 
	{
		String setterName = accessorName("set", propName );
		Class [] sig = new Class [] { clas };
		Method [] methods = clas.getMethods();

		// we don't know the right hand side of the assignment yet.
		// has at least one setter of the right name?
		for(int i=0; i<methods.length; i++)
			if ( methods[i].getName().equals( setterName ) )
				return true;
		return false;
	}

    public static Object getObjectProperty(
		Object obj, String propName)
        throws ReflectError
    {
        String accessorName = accessorName( "get", propName );
        Object[] args = new Object[] { };

        Interpreter.debug("property access: ");
        try {
            // null interpreter, accessor doesn't need to know
			try {
            return invokeObjectMethod(null, obj, accessorName, args);
			} catch ( EvalError e ) {
				// what does this mean?
				throw new ReflectError("getter: "+e);
			}
        }
        catch(InvocationTargetException e)
        {
            throw new ReflectError(
			"Property accessor threw exception:" + e );
        }
    }

    public static void setObjectProperty(
		Object obj, String propName, Object value)
        throws ReflectError, EvalError
    {
        String accessorName = accessorName( "set", propName );
        Object[] args = new Object[] { value };

        Interpreter.debug("property access: ");
        try {
            // null interpreter, accessor doesn't need to know
            invokeObjectMethod(null, obj, accessorName, args);
        }
        catch(InvocationTargetException e)
        {
            throw new EvalError("Property accessor threw exception!");
        }
    }

    /** 
		This method is meant to convert a JVM-array class name to the correct
    	'fully-qualified name' for the array class - JLS 6.7
	*/
    public static String normalizeClassName(Class type)
    {
        if(!type.isArray())
            return type.getName();

        StringBuffer className = new StringBuffer();
        try
        {
            className.append(getArrayBaseType(type).getName());
            for(int i = 0; i < getArrayDimensions(type); i++)
                className.append("[]");
        }
        catch(Exception e) { }

        return className.toString();
    }

	/**[
		returns the dimensionality of the Class
		returns 0 if the Class is not an array class
	*/
    public static int getArrayDimensions(Class arrayClass)
    {
        if(!arrayClass.isArray())
            return 0;

        return arrayClass.getName().lastIndexOf('[') + 1;
    }

    /**

		Returns the base type of an array Class.
    	throws ReflectError if the Class is not an array class.
	*/
    public static Class getArrayBaseType(Class arrayClass) throws ReflectError
    {
        if(!arrayClass.isArray())
            throw new ReflectError("The class is not an array.");

		return arrayClass.getComponentType();

    }

}

