//
// File:        IORSource.java
// Package:     gov.llnl.babel.backend.ior
// Revision:    @(#) $Id: IORSource.java 4462 2005-03-23 19:29:24Z leek2 $
// Description: generate IOR implementation source to a pretty writer stream
//
// Copyright (c) 2000-2004, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.ior;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.c.C;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.rmi.RMIIORSource;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Assertion;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.MethodCall;
import gov.llnl.babel.symbols.Package;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import gov.llnl.babel.BabelConfiguration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.Comparator;

/**
 * Class <code>IORSource</code> generates an IOR implementation source file
 * to a language writer output stream.  The constructor takes a language
 * writer stream and method <code>generateCode</code> generates intermediate
 * object representation for the specified symbol to the output stream.  The
 * language writer output stream is not closed by this object.
 */
public class IORSource {
  private static int s_longestBuiltin;
  private static final String s_castBuiltin   = IOR.getBuiltinName(IOR.CAST);
  private static final String s_deleteBuiltin = IOR.getBuiltinName(IOR.DELETE);
  private static final String s_execBuiltin   = IOR.getBuiltinName(IOR.EXEC);

  private static final String s_self          = Utilities.s_self;
  private static final String s_exception_var = Utilities.s_exception;
  private static final String s_superBuiltin  = "_super";


  /**
   * Some assertion checking defaults...at least until we provide another
   * mechanism.
   */
  private final static int ASSERT_DECLARE = 1;
  private final static int ASSERT_RAISE   = 2;

  /**
   * Store the SymbolID for sidl.BaseClass, if the extendable being
   * printed is not abstract (i.e., is a concrete class).
   */
  SymbolID d_baseClass = null;

  /**
   * Store the SymbolID for sidl.ClassInfo, if the extendable being
   * printed is not abstract (i.e., is a concrete class).
   */
  SymbolID d_classInfo = null;

  /**
   * Store the SymbolID for sidl.ClassInfoI, if the extendable being
   * printed is not abstract (i.e., is a concrete class).
   */
  SymbolID d_classInfoI = null;

  static {
    s_longestBuiltin = 0;
    for(int j = 0; j < IOR.CLASS_BUILTIN_METHODS; ++j) {
      String mname = IOR.getBuiltinName(j, true);
      if (mname.length() > s_longestBuiltin) {
        s_longestBuiltin = mname.length();
      }
    }
  }

  private LanguageWriterForC d_writer;

  /**
   * This is a convenience utility function that writes the symbol
   * source information into the provided language writer output
   * stream.  The output stream is not closed on exit.  A code
   * generation exception is thrown if an error is detected.
   */
  public static void generateCode(Symbol symbol, LanguageWriterForC writer)
    throws CodeGenerationException 
  {
    IORSource source = new IORSource(writer);
    source.generateCode(symbol);
  }

  /**
   * Create a <code>IORSource</code> object that will write symbol information
   * to the provided output writer stream.
   */
  public IORSource(LanguageWriterForC writer) {
    d_writer = writer;
  }

  /**
   * Write IOR source information for the provided symbol to the language
   * writer output stream provided in the constructor.  This method does
   * not close the language writer output stream and may be called for more
   * than one symbol (although the generated source may not be valid input
   * for the C compiler).  A code generation exception is generated if an
   * error is detected.  No code is generated for enumerated and package
   * symbols.
   */
  public void generateCode(Symbol symbol) throws CodeGenerationException {
    if (symbol != null) {
      switch (symbol.getSymbolType()) {
      case Symbol.PACKAGE:
      case Symbol.INTERFACE:
        break;
      case Symbol.CLASS:
        generateSource((Class) symbol);
        break;
      }
    }
  }

  /**
   * Lookup the SymbolIDs for sidl.BaseClass, sidl.ClassInfo and
   * sidl.ClassInfoI.
   */
  private void lookupSymbolIDs() {
    SymbolTable table = SymbolTable.getInstance();
    d_baseClass = table.
      lookupSymbol(BabelConfiguration.getBaseClass()).getSymbolID();
    d_classInfo = table.
      lookupSymbol(BabelConfiguration.getClassInfo()).getSymbolID();
    d_classInfoI = table.
      lookupSymbol(BabelConfiguration.getClassInfoI()).getSymbolID();
  }

  /**
   * Generate the IOR source for a SIDL class or interface.  The source
   * file begins with a banner and include files followed by a declaration
   * of static methods and (for a class) external methods expected in the
   * skeleton file.  For classes, the source file then defines a number
   * of functions (cast, delete, initialize EPVs, new, init, and fini).
   */
  private void generateSource(Class cls) throws CodeGenerationException {
    /*
     * Generate the file banner and include files.
     */
    SymbolID id = cls.getSymbolID();
    SymbolID baseClass = null;
    SymbolID classInfo = null;
    SymbolID classInfoI = null;
    String source = IOR.getSourceFile(id);
    String header = IOR.getHeaderFile(id);

    d_writer.writeBanner(cls, source, CodeConstants.C_IS_NOT_IMPL,
                         CodeConstants.C_DESC_IOR_PREFIX + id.getFullName());

    d_writer.printlnUnformatted("#include <stdlib.h>");
    d_writer.printlnUnformatted("#include <stddef.h>");
    d_writer.printlnUnformatted("#include <string.h>");
    if (IOR.supportAssertions(cls)) {
      d_writer.printlnUnformatted("#include <stdio.h>");
      d_writer.printlnUnformatted("#include <sys/time.h>");
      d_writer.printlnUnformatted("#include <time.h>");
      d_writer.printlnUnformatted("#include \"sidlAsserts.h\"");
      d_writer.printlnUnformatted("#include \"sidlAssertUtils.h\"");
      if (IOR.ASSERTION_LEVEL != BabelConfiguration.FULL_CHECKING_STATS) {
        d_writer.print("//");
      }
      d_writer.println("#define " + IOR.S_FULL_STATS_MACRO);
      d_writer.println("#define " + IOR.S_TEXT_STATS_MACRO);
      d_writer.println();
    }
    d_writer.generateInclude(header, false);
    //if (!cls.isAbstract()) {
      lookupSymbolIDs();
      d_writer.generateInclude(C.getImplHeaderFile(d_baseClass), true);
      d_writer.generateInclude(C.getHeaderFile(d_baseClass), true);
      d_writer.generateInclude(C.getHeaderFile(d_classInfo), true);
      d_writer.generateInclude(C.getHeaderFile(d_classInfoI), true);
      //}
    d_writer.println();

    d_writer.printlnUnformatted("#ifndef " + C.NULL);
    d_writer.printlnUnformatted("#define " + C.NULL + " 0");
    d_writer.printlnUnformatted("#endif");
    d_writer.println();

    String my_mutex=IOR.getSymbolName(id) + "__mutex";
    String lock_name=IOR.getLockStaticGlobalsMacroName();
    String unlock_name=IOR.getUnlockStaticGlobalsMacroName();
    String have_lock= IOR.getHaveLockStaticGlobalsMacroName();
    d_writer.printlnUnformatted("#include \"sidl_thread.h\"");
    d_writer.printlnUnformatted("#ifdef HAVE_PTHREAD");
    d_writer.printlnUnformatted("static struct sidl_recursive_mutex_t " 
      + my_mutex + "= SIDL_RECURSIVE_MUTEX_INITIALIZER;");
    d_writer.printlnUnformatted("#define " + lock_name 
      + " sidl_recursive_mutex_lock( &" + my_mutex + " )" );
    d_writer.printlnUnformatted("#define " + unlock_name 
      + " sidl_recursive_mutex_unlock( &" + my_mutex + " )" );
    d_writer.printlnUnformatted("/* #define " + have_lock 
      + " (sidl_recursive_mutex_trylock( &" + my_mutex + " )==EDEADLOCK) */");
    d_writer.printlnUnformatted("#else");
    d_writer.printlnUnformatted("#define " + lock_name);
    d_writer.printlnUnformatted("#define " + unlock_name);
    d_writer.printlnUnformatted("/* #define " + have_lock + " (1) */");
    d_writer.printlnUnformatted("#endif");
    d_writer.println();
				
    /*
     * Generate internal static variables and external references to be
     * supplied by the skeleton file.
     */
    generateStaticVariables(cls);
    generateExternalReferences(cls);

    /*
     * Generate a number of local functions (cast, delete, initialization
     * of EPVs, new, init, and fini).  These functions are only needed
     * for classes.
     */
    boolean doStatic   = cls.hasStaticMethod(true);
    boolean doAsserts  = IOR.supportAssertions(cls);
    boolean doIceptors = IOR.supportInterceptors(cls);

    RMIIORSource.generateCode(cls, d_writer);
    generateEnsureLoad(cls);
    generateCastFunction(cls);
    if (doAsserts) {
      if (doStatic) {
        generateChecksFunction(cls, true);
        generateDumpStatsFunction(cls, true);
      }
      generateChecksFunction(cls, false);
      generateDumpStatsFunction(cls, false);
      generateAllChecks(cls);
    }
    if (doIceptors) {
      if (doStatic) {
        generateInterceptorFunction(cls, true);
      }
      generateInterceptorFunction(cls, false);
      generateAllInterceptors(cls);
    }
    generateDeleteFunction(cls);
    generateMainExec(cls);
    generateInitEPV(cls);
    generateInitSEPV(cls);
    generateStaticFunction(cls, false);
    generateSuperFunction(cls);
    if (doAsserts || doIceptors) {
      generateStaticFunction(cls, true);
    }
    generateInitClassInfo(cls);
    generateInitMetadata(cls);
    generateNewFunction(cls);
    generateInitFunction(cls);
    generateFiniFunction(cls);
    generateVersionFunction(cls);
    generateExternalFunc(cls);
  }

  /**
   * Generate a single line comment.  This is called out as a separate
   * method to make the code formatting below a little prettier.
   */
  private void comment(String s) {
    d_writer.writeComment(s, false);
  }

  /**
   * Generate the static variables used to store the EPVs and also the
   * initialization flags for the EPVs.  For interfaces, we only need
   * to generate a remote EPV structure.  Classes require EPVs for their
   * static methods (if present), standard methods, remote methods, and
   * old, new, and remote versions for all parent classes and interfaces.
   */
  private void generateStaticVariables(Class cls) {
    comment("Static variables to hold version of IOR");
    d_writer.println("static const int32_t s_IOR_MAJOR_VERSION = " 
      + IOR.MAJOR_VERSION + ";");
    d_writer.println("static const int32_t s_IOR_MINOR_VERSION = " 
      + IOR.MINOR_VERSION + ";");
    d_writer.println();

    if (!cls.isAbstract()) {
      comment("Static variable to hold shared ClassInfo interface.");
      d_writer.println("static " + C.getObjectName(d_classInfo) 
        + " s_classInfo = " + C.NULL + ";");
      d_writer.println("static int s_classInfo_init = 1;");
      d_writer.println();
    }

    comment("Static variable to make sure _load called no more than once");
    d_writer.println("static int s_load_called = 0;");

    comment("Static variables for managing EPV initialization.");

    /*
     * Output the initialization flags for the EPV structures.
     */
    boolean has_static = cls.hasStaticMethod(true);
    boolean doAsserts  = IOR.supportAssertions(cls);
    boolean doIceptors = IOR.supportInterceptors(cls);

    d_writer.println("static int s_method_initialized = 0;");
    if (has_static) {
      d_writer.println("static int s_static_initialized = 0;");
    }
    d_writer.println();

    /*
     * Output the EPV, remote EPV, and static EPV for this object for
     * each supported type of static EPV.
     */
    IOR.generateStaticEPVVariables(d_writer, cls, has_static, false, IOR.SET_PUBLIC);
    if (doAsserts) {
      IOR.generateStaticEPVVariables(d_writer, cls,  has_static, false,
                                     IOR.SET_ASSERTIONS);
    }
    if (doIceptors) {
      IOR.generateStaticEPVVariables(d_writer, cls,  has_static, false,
                                     IOR.SET_INTERCEPTORS);
    }

    /*
     * Collect all the parents of the class in a set
     * and output EPV structures for the parents.
     */
      Set   parents        = Utilities.getAllParents(cls);
      Set   new_interfaces = Utilities.getUniqueInterfaceIDs(cls);

      if (!parents.isEmpty()) {
        List sorted = Utilities.sort(parents);
        for (Iterator i = sorted.iterator(); i.hasNext(); ) {
          SymbolID p_id     = (SymbolID) i.next();
          String   p_name   = IOR.getSymbolName(p_id);
          String   p_lower  = p_name.toLowerCase();
          String   p_epv    = "static " + IOR.getEPVName(p_id);
          boolean  is_old   = !new_interfaces.contains(p_id);
          String   p_epvStr = (is_old ? p_epv + "  " : p_epv + " ");

          d_writer.print(p_epvStr);
          d_writer.println(IOR.getStaticEPVVariable(p_id, IOR.EPV_NEW, 
                                                    IOR.SET_PUBLIC) + ";");

          if (is_old) {
            d_writer.print(p_epv + "* ");
            d_writer.println(IOR.getStaticEPVVariable(p_id, IOR.EPV_OLD, 
                                                      IOR.SET_PUBLIC) + ";");
          }
          d_writer.println();
        }
      }

      if (has_static && (doAsserts || doIceptors)) {
        comment("Static variables for assertion and/or interceptor controls.");
        String asCtrls = "static " + IOR.getControlsStruct(cls.getSymbolID()); 
        int asWidth = asCtrls.length() + 1;
        d_writer.printAligned(asCtrls, asWidth);
        d_writer.println(IOR.S_CONTROLS + ";");
        d_writer.println();
      }
      if (doAsserts) {
        comment("Static file for assertion statistics.");
        d_writer.println("static FILE* " + IOR.S_DUMP_FILE + " = " + C.NULL 
          + ";");
        d_writer.println();
      }
    }

  /**
   * Generate external references for skeleton routines that define the
   * functions in the EPVs.  A class will have a method EPV.  If there are
   * static functions, then there must also be a static EPV.
   */
  private void generateExternalReferences(Class cls) {
    comment("Declare EPV routines defined in the skeleton file.");

    SymbolID id = cls.getSymbolID();
    d_writer.openCxxExtern();
    d_writer.println("extern void " + IOR.getSetEPVName(id) + "(");
    d_writer.tab();
    d_writer.println(IOR.getEPVName(id) + "* epv);");
    d_writer.backTab();

    if (cls.hasStaticMethod(true)) {
      d_writer.println("extern void " + IOR.getSetSEPVName(id) + "(");
      d_writer.tab();
      d_writer.println(IOR.getSEPVName(id) + "* sepv);");
      d_writer.backTab();
    }
    d_writer.println("extern void " + IOR.getCallLoadName(id) + "(void);");
    d_writer.closeCxxExtern();

    d_writer.println();
  }

  /**
   * Generate the cast function for a class.  This will return null if
   * the cast is invalid and a pointer to the object otherwise.  The logic
   * generates tests for the current class and then recursively queries the
   * parent classes.
   */
  private void generateCastFunction(Class cls) {
    comment("CAST: dynamic type casting support.");

    /*
     * Define the method signature and begin the method implementation.
     */
    SymbolID id = cls.getSymbolID();

    d_writer.println("static void* " + getIORMethodName(id, s_castBuiltin) 
      + '(');
    d_writer.tab();
    d_writer.println(IOR.getObjectName(id) + "* " + s_self + ",");
    d_writer.println("const char* name)");
    d_writer.backTab();
    d_writer.println("{");
    d_writer.tab();

    /*
     * Output parent pointers to simplify access to parent classes.
     * Recursively generate the casting checks for this class and
     * all parent classes.
     */
    d_writer.println("void* cast = " + C.NULL + ";");
    d_writer.println();

    generateParentSelf(cls, 0, 0);
    d_writer.println();

    generateCastLogic(cls, 0);
    d_writer.println();

    /*
     * Return the cast pointer and close the function body.
     */
    d_writer.println("return cast;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Recursively generate the cast logic for a class and its interfaces.
   * This routine first generates the check against the class type and
   * then against all newly defined interfaces in the class.  Finally,
   * it recurses on the class parent (if it exists).
   */
  private void generateCastLogic(Class cls, int level) {
    String self = "s" + String.valueOf(level);

    /*
     * Check the cast against the class type.
     */
    if (level > 0) {
      d_writer.print("} else ");
    }
    d_writer.println("if (!strcmp(name, \"" + cls.getFullName() + "\")) {");
    d_writer.tab();
    d_writer.println("cast = (void*) " + self + ";");
    d_writer.backTab();

    /*
     * Check the cast against all newly defined interfaces in this class.
     */
    List interfaces = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
    for (Iterator i = interfaces.iterator(); i.hasNext(); ) {
      SymbolID id    = (SymbolID) i.next();
      String   fn    = id.getFullName();
      String   lower = IOR.getSymbolName(id).toLowerCase();;
      d_writer.println("} else if (!strcmp(name, \"" + fn + "\")) {");
      d_writer.tab();
      d_writer.println("cast = (void*) &" + self + "->d_" + lower + ";");
      d_writer.backTab();
    }

    /*
     * If there is a parent class, then recursively call the cast function
     * on the parent class.  Otherwise, close the if test block.
     */
    Class parent = cls.getParentClass();
    if (parent != null) {
      generateCastLogic(parent, level + 1);
    } else {
      d_writer.println("}");
    }
  }

  /**
   * Returns the controls variable base for the specified version.
   */
  private String getBaseControls(boolean doStatic, String self) {
    return (doStatic ? IOR.S_CONTROLS : self + "->" + IOR.D_CONTROLS);
  }

  /**
   * Returns the method controls variable base for the specified base
   * and index variable name.
   */
  private String getBaseMethodControls(String base, String indexVar) {
    return base + IOR.D_METHOD_CONTROLS + "[" + indexVar + "]";
  }

  /**
   * Returns the method controls variable base for the specified version
   * and index variable name.
   */
  private String getBaseMethodControls(boolean doStatic, String self,
                                       String indexVar) 
  {
    return getBaseMethodControls(getBaseControls(doStatic, self), indexVar);
  }

  /**
   * Generate the countdown macro call.
   */
  private void generateStatisticsUpdate(String ctrlBase, String methBase, 
                                        String aOkay, String mOkay, 
                                        String methTime, String totalTime)
  {
    d_writer.printlnUnformatted("#ifdef " + IOR.S_FULL_STATS_MACRO);
    if (mOkay.equals("0")) {
      d_writer.println(methBase + IOR.D_METHOD_EXCEPT + " += 1;");
    } else if (mOkay.equals("mOkay")) {
      d_writer.println(methBase + IOR.D_TRIES + " += 1;");
      d_writer.println("SIDL_INCR_IF_TRUE(!" + mOkay + "," + methBase 
        + IOR.D_METHOD_EXCEPT + ")");
      d_writer.println("SIDL_INCR_IF_TRUE(" + aOkay + "," + methBase 
        + IOR.D_SUCCESSES + ")");
    }
    d_writer.printlnUnformatted("#endif /* " + IOR.S_FULL_STATS_MACRO + " */");
    d_writer.println("SIDL_SET_COUNTDOWN(" + ctrlBase + IOR.D_CHECKS + ",");
    d_writer.tab();
    d_writer.println(ctrlBase + IOR.D_RATE + ", " + methBase + IOR.D_COUNTDOWN 
      + ", " + aOkay + ", " + mOkay + ", ");
    d_writer.println("" + methTime + ", " + totalTime + ")");
    d_writer.backTab();
  }

  /**
   * Returns the name of the built-in method, prepending "ior_" and the
   * name of the symbol.
   */
  private String getIORMethodName(SymbolID id, String name) {
    return "ior_" + IOR.getSymbolName(id) + '_' + name;
  }

  /**
   * Returns the name of the specified version of the function to set the 
   * assertion checking level.
   */
  private String getSetChecksMethodName(SymbolID id, boolean doStatic) {
    return getIORMethodName(id, IOR.getBuiltinName(IOR.CHECKS, doStatic));
  }

  /**
   * Generate the specified function to set the assertion checking level.  
   */
  private void generateChecksFunction(Extendable ext, boolean doStatic) {
    String desc = doStatic ? "static " : "";
    comment("CHECKS: set the " + desc + "assertion checking level.");

    SymbolID id        = ext.getSymbolID();
    String   name      = IOR.getSymbolName(id);
    boolean  doAsserts = IOR.supportAssertions(ext);
    boolean  doBoth    = doAsserts && IOR.supportInterceptors(ext);

    d_writer.println("static void " + getSetChecksMethodName(id, doStatic) 
      + '(');
    d_writer.tab();
    if (!doStatic) {
      d_writer.println(IOR.getObjectName(id) + "* " + s_self + ",");
    }
    d_writer.println("int32_t level,");
    d_writer.println("double  rate,");
    d_writer.println("int32_t resetCounters)");
    d_writer.backTab();

    d_writer.println("{");
    d_writer.tab();

    if (doAsserts) {
      String base  = getBaseControls(doStatic, s_self) + ".";
      String mBase = "mc->";

      d_writer.println(IOR.getMethodControlsStruct(id) + " *mc;"); 
      if (!doStatic) {
        d_writer.println("int checkAsserts = CHECK_ASSERTIONS & level;");
      }
      d_writer.println(base + IOR.D_CHECKS    + "  = level;");
      d_writer.println(base + IOR.D_RATE + " = rate;");
      d_writer.println();
      d_writer.println("if (resetCounters) {");
      d_writer.tab();
      d_writer.println("int i;");
      d_writer.println("for (i =" + IOR.getMethodIndex(id, "MIN") + ";");
      d_writer.println("     i<=" + IOR.getMethodIndex(id, "MAX") + "; i++)"
        + " {");
      d_writer.tab();
      d_writer.println("mc = &" + getBaseMethodControls(base, "i") + ";");
      d_writer.printlnUnformatted("#ifdef " + IOR.S_FULL_STATS_MACRO);
      d_writer.println(mBase + IOR.D_CALLS + "             = 0;");
      d_writer.println(mBase + IOR.D_METHOD_EXCEPT + " = 0;");
      d_writer.println(mBase + IOR.D_TRIES + "             = 0;");
      d_writer.println(mBase + IOR.D_SUCCESSES + "         = 0;");
      d_writer.println(mBase + IOR.D_COUNTDOWN + "         = 0;");
      d_writer.printlnUnformatted("#else /* !"+ IOR.S_FULL_STATS_MACRO + " */");
      d_writer.println(mBase + IOR.D_COUNTDOWN + "         = 0;");
      d_writer.printlnUnformatted("#endif /* "+ IOR.S_FULL_STATS_MACRO + " */");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.backTab();
      d_writer.println("}");
      if (!doStatic) {
        d_writer.println();
        d_writer.println("/* Ensure the EPVs are set properly. */");
      }
    }

    if (!doStatic) {
      String publicepv = s_self + "->" + IOR.getEPVVar(IOR.PUBLIC_EPV);
      if (doBoth) {
        String baseepv = s_self + "->" + IOR.getEPVVar(IOR.BASE_EPV);
        String use     = getBaseControls(false, s_self) + "." 
                         + IOR.D_INTERCEPTORS;
        d_writer.println("if (checkAsserts && (" + use + ")) {");
        d_writer.tab();
        d_writer.println(publicepv + "  = &" 
          + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_ASSERTIONS) 
          + ";");
        d_writer.println(baseepv + " = &" 
          + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_INTERCEPTORS) 
          + ";");
        d_writer.backTab();
        d_writer.println("} else if (checkAsserts) {");
        d_writer.tab();
        d_writer.println(publicepv + "  = &" 
          + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_ASSERTIONS) 
          + ";");
        d_writer.println(baseepv + " = &" 
          + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_PUBLIC) + ";");
        d_writer.backTab();
        d_writer.println("} else if (" + use + ") {");
        d_writer.tab();
        d_writer.println(publicepv + "  = &" 
          + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_INTERCEPTORS) 
          + ";");
        d_writer.println(baseepv + " = &" 
          + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_PUBLIC) + ";");
        d_writer.backTab();
        d_writer.println("} else {");
        d_writer.tab();
        d_writer.println(publicepv + "  = &" 
          + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_PUBLIC) + ";");
        d_writer.println(baseepv + " = &" 
          + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_PUBLIC) + ";");
        d_writer.backTab();
        d_writer.println("}");
      } else if (doAsserts) {
        d_writer.println("if (checkAsserts) {");
        d_writer.tab();
        d_writer.println(publicepv + " = &" 
          + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_ASSERTIONS) 
          + ";");
        d_writer.backTab();
        d_writer.println("} else {");
        d_writer.tab();
        d_writer.println(publicepv + " = &" 
          + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_PUBLIC) + ";");
        d_writer.backTab();
        d_writer.println("}");
      } else {
        comment("Nothing to do since assertion support not needed.");
      }
    }

    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Returns the name of the assertion statistics dump method.
   */
  private String getDumpStatsMethodName(SymbolID id, boolean doStatic) {
    return getIORMethodName(id, IOR.getBuiltinName(IOR.DUMP_STATS, doStatic));
  }

  /**
   * Generate the specified version of the dump assertion statistics and 
   * control data function.
   */
  private void generateDumpStatsFunction(Extendable ext, boolean doStatic) {
    String desc = doStatic ? "static " : "";
    comment("DUMP: Dump the " + desc + "assertion statistics.");

    SymbolID id         = ext.getSymbolID();
    String   name       = IOR.getSymbolName(id);
    boolean  doAsserts  = IOR.supportAssertions(ext);
    boolean  doIceptors = IOR.supportInterceptors(ext);

    d_writer.println("static void " + getDumpStatsMethodName(id, doStatic) 
      + '(');
    d_writer.tab();
    if (!doStatic) {
      d_writer.println(IOR.getObjectName(id) + "* " + s_self + ",");
    }
    d_writer.println("const char* filename)");
    d_writer.backTab();
    d_writer.println("{");
    d_writer.tab();

    if (doAsserts) {
      String base  = getBaseControls(doStatic, s_self) + ".";
      String mBase = "mc->";

      d_writer.println(IOR.getMethodControlsStruct(id) + " *mc;"); 
      d_writer.println("time_t currTime = time(" + C.NULL + ");");
      d_writer.println("int    level    = " + base + IOR.D_CHECKS + ";");
      d_writer.printlnUnformatted("#ifdef " + IOR.S_TEXT_STATS_MACRO);
      d_writer.println("char*  desc     = sidl_getCheckDescription(level);");
      d_writer.printlnUnformatted("#else /* !" + IOR.S_TEXT_STATS_MACRO 
        + " */");
      d_writer.println("char*  desc     = sidl_getCheckFrequencyDesc(level);");
      d_writer.printlnUnformatted("#endif /* " + IOR.S_TEXT_STATS_MACRO 
        + " */");
      d_writer.println("int    i;");
      d_writer.println();
      d_writer.println("if (" + IOR.S_DUMP_FILE + " == " + C.NULL + ") {");
      d_writer.tab();
      d_writer.println("if ((" + IOR.S_DUMP_FILE 
        + "=fopen(filename,\"w\")) == " + C.NULL + ") {");
      d_writer.tab();
      d_writer.println("printf(\"Cannot open file %s to dump the " + desc 
        + "assertion statistics.\\n\", filename);");
      d_writer.println("return;");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println();

      d_writer.printlnUnformatted("#ifdef " + IOR.S_TEXT_STATS_MACRO);
      d_writer.println("fprintf(" + IOR.S_DUMP_FILE + ", \"" 
        + desc.toUpperCase() + "ASSERTION CONTROL AND STATISTICS AT %s\\n\","
        + " ctime(&currTime));");
      d_writer.println("fprintf(" + IOR.S_DUMP_FILE 
        + ", \"Checking Level      = %d (%s)\\n\", level, desc);");
      d_writer.println("fprintf(" + IOR.S_DUMP_FILE 
        + ", \"Checking Threshold  = %f\\n\", " + base + IOR.D_RATE 
        + ");");
      if (doIceptors) {
        d_writer.println("fprintf(" + IOR.S_DUMP_FILE
          + ", \"Interceptor Support = %s\\n\", (" + base 
          + IOR.D_INTERCEPTORS + " ? \"on\" : \"off\"));");
      }
      d_writer.printlnUnformatted("#ifdef " + IOR.S_FULL_STATS_MACRO);
      d_writer.print("fprintf(" + IOR.S_DUMP_FILE 
        + ", \"\\n%26s  \\t%s (%s (%s), %s)  \\t%s (%s, %s)\\n\",");
      d_writer.println("\"Method          \", \"Calls\", \"Chkd\", "
        + "\"OK\", \"MExcs\", \"Cd\", \"Meth\", \"Total\");"); 
      d_writer.printlnUnformatted("#else /* !"+ IOR.S_FULL_STATS_MACRO + " */");
      d_writer.print("fprintf(" + IOR.S_DUMP_FILE + ", \"\\n%26s  \\t%s\\n\",");
      d_writer.println("\"Method          \", \"Cd\");"); 
      d_writer.printlnUnformatted("#endif /* " + IOR.S_FULL_STATS_MACRO+ " */");
      d_writer.printlnUnformatted("#endif /* " + IOR.S_TEXT_STATS_MACRO 
        + " */");
      d_writer.println("for (i =" + IOR.getMethodIndex(id, "MIN") + ";");
      d_writer.println("     i<=" + IOR.getMethodIndex(id, "MAX") + "; i++)"
        + " {");
      d_writer.tab();
      d_writer.println("mc = &" + getBaseMethodControls(base, "i") + ";");
      d_writer.print("if (  (");
      if (!doStatic) {
        d_writer.print("!");
      }
      d_writer.println(IOR.getMethodDescDataName(id) + "[i].isStatic) ");
      d_writer.printlnUnformatted("#ifdef " + IOR.S_FULL_STATS_MACRO);
      d_writer.println("   && (" + mBase + IOR.D_CALLS + " > 0) ) {");
      d_writer.tab();
      d_writer.printlnUnformatted("#ifdef " + IOR.S_TEXT_STATS_MACRO);
      d_writer.println("fprintf(" + IOR.S_DUMP_FILE 
        + ", \"%26s  \\t%5d (%d (%d), %d)  \\t\\t%2d\\n\",");
      d_writer.printlnUnformatted("#else /* !" + IOR.S_TEXT_STATS_MACRO +" */");
      d_writer.println("fprintf(" + IOR.S_DUMP_FILE 
        + ", \"%s\\t%f\\t%s\\t%d\\t%d\\t%d\\t%d\\t%d\\n\",");
      d_writer.println("        desc,");
      d_writer.println("        " + base + IOR.D_RATE + ",");
      d_writer.printlnUnformatted("#endif /* " + IOR.S_TEXT_STATS_MACRO +" */");
      d_writer.printlnUnformatted("#else /* !"+ IOR.S_FULL_STATS_MACRO + " */");
      d_writer.backTab();
      d_writer.println("   && (" + mBase + IOR.D_COUNTDOWN + " > 0) ) {");
      d_writer.tab();
      d_writer.printlnUnformatted("#ifdef " + IOR.S_TEXT_STATS_MACRO);
      d_writer.println("fprintf(" + IOR.S_DUMP_FILE + ", \"%26s  \\t%5d\\n\",");
      d_writer.printlnUnformatted("#else /* !" + IOR.S_TEXT_STATS_MACRO +" */");
      d_writer.println("fprintf(" + IOR.S_DUMP_FILE 
        + ", \"%s\\t%f\\t%s\\t%d\\n\",");
      d_writer.println("        desc,");
      d_writer.println("        " + base + IOR.D_RATE + ",");
      d_writer.printlnUnformatted("#endif /* " + IOR.S_TEXT_STATS_MACRO +" */");
      d_writer.printlnUnformatted("#endif /* "+ IOR.S_FULL_STATS_MACRO + " */");
      d_writer.println("        " + IOR.getMethodDescDataName(id) 
        + "[i].name,");
      d_writer.printlnUnformatted("#ifdef " + IOR.S_FULL_STATS_MACRO);
      d_writer.println("        " + mBase + IOR.D_CALLS + ",");
      d_writer.println("        " + mBase + IOR.D_TRIES + ",");
      d_writer.println("        " + mBase + IOR.D_SUCCESSES + ",");
      d_writer.println("        " + mBase + IOR.D_METHOD_EXCEPT + ",");
      d_writer.println("        " + mBase + IOR.D_COUNTDOWN + ");");
      d_writer.printlnUnformatted("#else /* !"+ IOR.S_FULL_STATS_MACRO + " */");
      d_writer.println("        " + mBase + IOR.D_COUNTDOWN + ");");
      d_writer.printlnUnformatted("#endif /* "+ IOR.S_FULL_STATS_MACRO + " */");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.printlnUnformatted("#ifdef " + IOR.S_TEXT_STATS_MACRO);
      d_writer.println("fprintf(" + IOR.S_DUMP_FILE + ", \"\\n\\n\");");
      d_writer.printlnUnformatted("#endif /* " + IOR.S_TEXT_STATS_MACRO 
        + " */");
      d_writer.println("fflush(" + IOR.S_DUMP_FILE + ");");
      d_writer.println("free(desc);");
      d_writer.println("return;");
    } else {
      comment("Nothing to do since assertion support not needed.");
    }

    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Generate any assertion checks associated with each method of the class.
   * Optimally, this will only be called if the class has assertions that need
   * to be checked.
   */
  private void generateAllChecks(Class cls) throws CodeGenerationException {
    ArrayList invariants = cls.getAllInvariants();

    List     normal = (List) cls.getMethods(true);
    Iterator i      = normal.iterator();
    Method   meth   = null;
    while (i.hasNext()) {
      meth = (Method)i.next();
      generateMethodChecks(cls, meth, invariants);
    }
  }

  /**
   * Returns the name of the assertion checking method for the specified
   * method.
   */
  private String getCheckMethodName(Extendable ext, Method meth) {
    return "check_" + IOR.getSymbolName(ext.getSymbolID()) + '_' 
           + meth.getLongMethodName();
  }

  /**
   * Generate call to the base method.
   */
  private void generateBaseCall(String name, String suffix, String var,
                                List args, boolean isStatic, boolean doThrows, 
                                Type returnType)
    throws CodeGenerationException
  {
     boolean isOrig = suffix.equals("");
     if (  (isOrig) && (returnType != null) 
        && (returnType.getType() != Type.VOID) ) 
     {
       d_writer.print(C.FUNCTION_RESULT + " = ");
     }
     //d_writer.print("(*" + var + "->");
     d_writer.print("(" + var + "->");
     d_writer.println(IOR.getVectorEntry(name + suffix) + ")(");
     d_writer.tab();
     d_writer.tab();
     C.generateArguments(d_writer, s_self, args, isStatic, doThrows, 
       ((suffix.equals(IOR.GENERIC_POST_SUFFIX)) ? returnType : null), false,
       false, false);
     d_writer.println(");");
     d_writer.backTab();
     d_writer.backTab();
  }

  /**
   * Generate the checks associated with the invariants, preconditions, and
   * postconditions, if any, for the given method.  Assumes being called only
   * if assertion check generation is enabled.
   *
   * @param cls         The class associated with the method.
   * @param m           Method to be checked.
   * @param invariants  List of invariants, if any.
   */ 
  private void generateMethodChecks(Class cls, Method m, ArrayList invariants)
    throws CodeGenerationException
  {
    SymbolID  id              = cls.getSymbolID();
    String    name            = m.getLongMethodName();
    ArrayList preconditions   = m.getPreconditions();
    ArrayList postconditions  = m.getPostconditions();
    Assertion as              = null;
    int       numInv          = invariants.size();
    int       numPre          = preconditions.size();
    int       numPure         = (m.hasPureAssertion()) ? 1 : 0;
    int       numPost         = postconditions.size() - numPure;
    int       total           = (2 * numInv) + numPre + numPost;
    comment("Check relevant assertions, if any, before and after the method "
      + "call.");
    String methname = getCheckMethodName(cls, m);
  
    Type    type      = m.getReturnType();
    boolean hasReturn = false;
    if ( (type != null) && (type.getType() != Type.VOID) ) {
       hasReturn = true;
    }
    d_writer.println("static " + getReturnString(type) + " " + methname + '(');
    d_writer.tab();
    List    args      = m.getArgumentList();
    boolean hasThrows = !m.getThrows().isEmpty();
    boolean isStatic  = m.isStatic();
    C.generateArguments(d_writer, IOR.getObjectName(id) + "* " + s_self, 
      args, isStatic, hasThrows, null, true, true, false);
    d_writer.println(")");
    d_writer.backTab();
    d_writer.println("{");
    d_writer.tab();

    if (hasReturn) {
       d_writer.println(getReturnString(type) + " " + C.FUNCTION_RESULT + ";");
    }

    /*
     * WARNING/ToDo...Must figure out how to ensure the exception type is 
     * declared should the class never throw an exception AND the base 
     * exception type ever stop being the base interface!
     */
    if ( (!hasThrows) && (m.hasUnreservedMethodAssertion(false)) ) {
       d_writer.println(IOR.getExceptionFundamentalType() + "* " 
         + s_exception_var + ";");
    }
    String  var;
    boolean addBlank   = false;
    boolean doIceptors = IOR.supportInterceptors(cls);
    if (isStatic) {
       var = "sepv";
       if (doIceptors) {
         d_writer.println("int _type = " + getBaseControls(isStatic, "") + "."
           + IOR.D_INTERCEPTORS + " ? "
           + IOR.getStaticTypeOption(id, IOR.SET_INTERCEPTORS) + " : "
           + IOR.getStaticTypeOption(id, IOR.SET_PUBLIC) + ";");
         d_writer.println(IOR.getSEPVName(id) + "* " + var + " = " 
           + IOR.getLocalStaticsName(id) + "(_type);");
       } else {
         d_writer.println(IOR.getSEPVName(id) + "* " + var + " = " 
           + IOR.getLocalStaticsName(id) + "("
           + IOR.getStaticTypeOption(id, IOR.SET_PUBLIC) + ");");
       }
       addBlank = true;
    } else {
       if (doIceptors) {
         var = s_self + "->" + IOR.getEPVVar(IOR.BASE_EPV);
       } else {
         var = "epv";
         d_writer.println(IOR.getEPVName(id) + "* " + var + " = &" 
           + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_PUBLIC) + ";");
         addBlank = true;
       }
    }
    if (addBlank) {
      d_writer.println();
    }
    if (total > 0) {
      String ctrlBase = getBaseControls(isStatic, s_self) + ".";
      String methIndex = IOR.getMethodIndex(id, m);
      String bvar      = (isStatic || doIceptors) ? var 
                            : s_self + "->" + IOR.getEPVVar(IOR.PUBLIC_EPV);
      String excvar    = hasThrows ? s_exception_var : "";
      String methBase = "mc->";

      d_writer.println("int aOkay    = 1;");
      d_writer.println("int mOkay    = 1;");
      d_writer.println("int doTiming = 0;");
      d_writer.println("struct timezone tz;");
      /*
       * ToDo...Need to determine the number of each macro return type
       * actually needed.  (This would be the maximum number associated
       * with any given assertion!)
       */
      int totalMacros = 0;
      int maxMacros = m.getMaxArrayIterMacros(
          MethodCall.MACRO_RETURN_TYPE[MethodCall.MACRO_RETURNS_BOOLEAN_IND]);
      if (maxMacros > 0) {
        d_writer.println("int " + MethodCall.ARRAY_BOOLEAN_RESULT_VAR 
          + "[" + maxMacros + "];");
        totalMacros += maxMacros;
      } 
      maxMacros = m.getMaxArrayIterMacros(
         MethodCall.MACRO_RETURN_TYPE[MethodCall.MACRO_RETURNS_DOUBLE_IND]);
      if (maxMacros > 0) {
        d_writer.println("double  " + MethodCall.ARRAY_DOUBLE_RESULT_VAR 
          + "[" + maxMacros + "];");
        totalMacros += maxMacros;
      } else {
        d_writer.println("double  " + MethodCall.ARRAY_DOUBLE_RESULT_VAR + ";");
      }
      maxMacros = m.getMaxArrayIterMacros(
         MethodCall.MACRO_RETURN_TYPE[MethodCall.MACRO_RETURNS_INTEGER_IND]);
      if (maxMacros > 0) {
        d_writer.println("int32_t " + MethodCall.ARRAY_INTEGER_RESULT_VAR 
          + "[" + maxMacros + "];");
        totalMacros += maxMacros;
      }
      if (totalMacros > 0) {
        d_writer.println("int32_t " + MethodCall.ARRAY_COUNT_VAR + ", "
          + MethodCall.ARRAY_ITER_VAR + ", " + MethodCall.ARRAY_SIZE_VAR + ";");
      }
      boolean doAssertBegin = ((numInv > 0) || (numPre > 0));
      boolean doAssertEnd   = ((numInv > 0) || (numPost > 0));
      if (doAssertBegin && doAssertEnd) {
        d_writer.println("struct timeval  aBegin, aEnd;");
      } else if (doAssertBegin) {
        d_writer.println("struct timeval  aBegin;");
      } else if (doAssertEnd) {
        d_writer.println("struct timeval  aEnd;");
      } 
      d_writer.println("struct timeval  mBegin, mEnd;");
      d_writer.println("char*           errMsg;");
      if (numInv > 0) {
        generateAssertionViolation(Assertion.INVARIANT, ASSERT_DECLARE, null, 
          excvar);
      }
      if (numPre > 0) {
        generateAssertionViolation(Assertion.REQUIRE, ASSERT_DECLARE, null, 
          excvar);
      }
      if (numPost > 0) {
        generateAssertionViolation(Assertion.ENSURE, ASSERT_DECLARE, null, 
          excvar);
      }

      d_writer.println();
      d_writer.println(IOR.getMethodControlsStruct(id) + " *mc = "); 
      d_writer.tab();
      d_writer.println("&" + getBaseMethodControls(ctrlBase, methIndex) 
        + ";");
      d_writer.backTab();

      d_writer.println("doTiming = (" + ctrlBase + IOR.D_CHECKS 
        + " & CHECK_TIMING) && (" + methBase + IOR.D_COUNTDOWN 
        + " >= 0);");
      if (hasThrows) {
        d_writer.println("(*" + s_exception_var + ") = NULL;");
      }
      d_writer.printlnUnformatted("#ifdef " + IOR.S_FULL_STATS_MACRO);
      d_writer.println(methBase + IOR.D_CALLS + " += 1;");
      d_writer.printlnUnformatted("#endif /* " + IOR.S_FULL_STATS_MACRO +" */");
      d_writer.println();

      d_writer.println("if (" + methBase + IOR.D_COUNTDOWN + " > 0) {");
      d_writer.tab();
      //d_writer.println("/* Fast path since not time to check yet */");
      d_writer.println(methBase + IOR.D_COUNTDOWN + " -= 1;");
      generateBaseCall(name, "", var, args, isStatic, hasThrows, type);
      if (hasThrows) {
        d_writer.println("if ((*" + s_exception_var + ") != NULL) {");
        d_writer.tab();
        generateStatisticsUpdate(ctrlBase, methBase, "1", "0", "0", "0");
        d_writer.backTab();
        d_writer.println("}");
      }
      d_writer.backTab();
      d_writer.println("} else {");
      d_writer.tab();
      //d_writer.println("/* Slower path since it is time to perform the "
      //  + "checks */");
      if (doAssertBegin) {
        d_writer.println("if (doTiming) { gettimeofday(&aBegin,&tz); }");
      }
      if (numPre > 0) {
        d_writer.println("if (" + ctrlBase + IOR.D_CHECKS 
          + " & CHECK_PRECONDITIONS) {");
        d_writer.tab();
        generateAssertionChecks(preconditions, bvar, isStatic, excvar);
        d_writer.backTab();
        d_writer.println("}");
      }
      if (numInv > 0) {
        d_writer.println("if (" + ctrlBase + IOR.D_CHECKS 
          + " & CHECK_INVARIANTS) {");
        d_writer.tab();
        generateAssertionChecks(invariants, bvar, isStatic, excvar);
        d_writer.backTab();
        d_writer.println("}");
      }
      d_writer.println("if (doTiming) { gettimeofday(&mBegin,&tz); }");

      if (hasThrows) {
        d_writer.println("if (aOkay) {");
        d_writer.tab();
      }

      generateBaseCall(name, "", var, args, isStatic, hasThrows, type);
      if (hasThrows) {
        d_writer.println("if ((*" + s_exception_var + ") != NULL) {");
        d_writer.tab();
        d_writer.println("mOkay = 0;");
        d_writer.backTab();
        d_writer.println("}");
        d_writer.backTab();
        d_writer.println("}");
      }

      if (doAssertEnd) {
        d_writer.println("if (doTiming) { gettimeofday(&mEnd,&tz); }");
      } else {
        d_writer.println("if (doTiming) {");
        d_writer.tab();
        d_writer.println("gettimeofday(&mEnd,&tz);");
      }

      /*
       * ToDo...Really need to properly handle inherited postconditions
       * (i.e., used to weaken the contract NOT strengthen)!
       * 
       * One idea (but make sure it works first):  
       *
       * <proceed with local postconditions as do now>
       * if (!aOkay) {
       *   aOkay = 1;
       *   <proceed with inherited ones the same as do now with local>
       * }
       * 
       */
      if (numPost > 0) {
        d_writer.println("if (" + ctrlBase + IOR.D_CHECKS 
          + " & CHECK_POSTCONDITIONS) {");
        d_writer.tab();
        generateAssertionChecks(postconditions, bvar, isStatic, excvar);
        d_writer.backTab();
        d_writer.println("}");
      }
      if (numInv > 0) {
        d_writer.println("if (" + ctrlBase + IOR.D_CHECKS 
          + " & CHECK_INVARIANTS) {");
        d_writer.tab();
        generateAssertionChecks(invariants, bvar, isStatic, excvar);
        d_writer.backTab();
        d_writer.println("}");
      }
      if (doAssertEnd) {
        d_writer.println("if (doTiming) {");
        d_writer.tab();
        d_writer.println("gettimeofday(&aEnd,&tz);");
        d_writer.println();
      }
      String assertTime;
      String methodTime = "SIDL_DIFF_MICROSECONDS(mEnd,mBegin)";
      String timeMacroStart = "SIDL_DIFF_MICROSECONDS(";
      if (doAssertBegin && doAssertEnd) {
        assertTime = timeMacroStart + "aEnd,aBegin)";
      } else if (doAssertBegin) {
        assertTime = timeMacroStart + "mEnd,aBegin)";
      } else if (doAssertEnd) {
        assertTime = timeMacroStart + "aEnd,mBegin)";
      } else {
        assertTime = timeMacroStart + "mEnd,mBegin)";
      } 
      generateStatisticsUpdate(ctrlBase, methBase, "aOkay", "mOkay", methodTime,
        assertTime);
      d_writer.backTab();
      d_writer.println("} else {");
      d_writer.tab();
      generateStatisticsUpdate(ctrlBase, methBase, "aOkay", "mOkay", "0", "0");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.backTab();
      d_writer.println("}");
    } else {
      generateBaseCall(name, "", var, args, isStatic, hasThrows, type);
    }
    if (hasReturn) {
      d_writer.println();
      d_writer.println("return " + C.FUNCTION_RESULT + ";");
    }
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Generate a return string for the specified SIDL type.  Most
   * of the SIDL return strings are listed in the static structures defined
   * at the start of this class.  Symbol types and array types require
   * special processing.
   */
  private static String getReturnString(Type type)
    throws CodeGenerationException
  {
    return IOR.getReturnString(type, true, false);
  }

  /**
   * Returns the next set of starting indices after those in the 
   * specified assertion.
   */
  private int[] getNextStartInd(Assertion as, int[] startInd) {
    int[]  nextStartInd = new int[MethodCall.MAX_VALID_MACRO_RETURNS];
    for (int i=0; i<MethodCall.MAX_VALID_MACRO_RETURNS; i++) {
       nextStartInd[i] = startInd[i]
         + as.getNumArrayIterMacrosByType(MethodCall.MACRO_RETURN_TYPE[i]);
    }
    return nextStartInd;
  }

  /**
   * Generate the assertion checks associated with the list of assertions.
   */
  private void generateAssertionChecks(ArrayList assertions, String var, 
                                       boolean doStatic, String excvar)
  {
     if (!assertions.isEmpty()) {
        boolean  first    = true;
        Iterator i        = assertions.iterator();
        int[]    startInd = new int[MethodCall.MAX_VALID_MACRO_RETURNS];
        for (int j=0; j<MethodCall.MAX_VALID_MACRO_RETURNS; j++) {
           startInd[j] = 0;
        }
        while (i.hasNext()) {
           Assertion as = (Assertion)i.next();
           String expr = as.cExpression(var, startInd);
           if ( (expr != null) && (!expr.equals("")) ) {
             generateAssertion(as, var, startInd, doStatic, expr, excvar, 
               first);
             first = false;
           }
           startInd = getNextStartInd(as, startInd);
        }
     }
  }

  /**
   * Generate the assertion check.
   */
  private void generateAssertion(Assertion as, String var, int[] startInd, 
                                 boolean doStatic, String expr, String excvar, 
                                 boolean first)
  {
     ArrayList mlist         = as.getArrayIterMacros(var, startInd);
     int       numIterMacros = (mlist != null) ? mlist.size() : 0;
     boolean   extraIndent   = false;
     if (!first) {
       if (numIterMacros <= 0) {
         d_writer.disableLineBreak();
         d_writer.println("else if (!" + expr + ") {");
         d_writer.enableLineBreak();
       } else {
         d_writer.println("if ( aOkay ) {");
         extraIndent = true;
         d_writer.tab();
         generateArrayIterMacros(mlist);
         d_writer.disableLineBreak();
         d_writer.println("if (!" + expr + ") {");
         d_writer.enableLineBreak();
       }
     } else {
       if (numIterMacros > 0) {
         generateArrayIterMacros(mlist);
       }
       d_writer.disableLineBreak();
       d_writer.println("if (!" + expr + ") {");
       d_writer.enableLineBreak();
     }
     d_writer.tab();
     d_writer.println("aOkay  = 0;");
     d_writer.disableLineBreak();
     d_writer.println("errMsg = \"" + as.errorMessage() + "\";");
     generateCheckErrorCall(as, var, doStatic);
     d_writer.enableLineBreak();
     if (!excvar.equals("")) {
       d_writer.println("if ((*" + excvar + ") == NULL) {");
       d_writer.tab();
       generateAssertionViolation(as.getType(), ASSERT_RAISE, as, excvar);
       d_writer.backTab();
       d_writer.println("}");
     }
     if (extraIndent) {
       d_writer.backTab();
       d_writer.println("}");
     }
     d_writer.backTab();
     d_writer.println("}");
  }

  /**
   * Generate the array iteration macro calls.  
   */
  private void generateArrayIterMacros(ArrayList list) {
     Iterator i   = list.iterator();
     while (i.hasNext()) {
       String mac   = (String)i.next();
       d_writer.disableLineBreak();
       d_writer.println(mac);
       d_writer.enableLineBreak();
     }
  }

  /**
   * Generate the appropriate assertion violation code based on genType
   * as follows:
   *
   *   ASSERT_DECLARE = Generate the assertion violation variable declaration
   *   ASSERT_RAISE   = Generate the assertion violation instantiation
   *
   * Note that as and excVar are only needed when ASSERT_RAISE is specified.
   */
  private void generateAssertionViolation(int assertType, int genType, 
                                          Assertion as, String excvar) 
  {
    String prefix;
    String errStr;
    String errType;
    switch (assertType) {
      case Assertion.REQUIRE: 
      case Assertion.REQUIRE_ELSE: 
        errStr  = "pre_err";
        errType = IOR.getPreconditionExceptType();
        prefix  = IOR.PRECONDITION_CALL_PREFIX;
        break;
      case Assertion.ENSURE: 
      case Assertion.ENSURE_THEN: 
        errStr  = "post_err";
        errType = IOR.getPostconditionExceptType();
        prefix  = IOR.POSTCONDITION_CALL_PREFIX;
        break;
      case Assertion.INVARIANT: 
      default:
        errStr  = "inv_err";
        errType = IOR.getInvariantExceptType();
        prefix  = IOR.INVARIANT_CALL_PREFIX;
    }
    if (genType == ASSERT_DECLARE) {
      d_writer.println(errType + errStr + ";");
    } else {
      d_writer.println(errStr + " = " + prefix + "__create();");
      d_writer.println(prefix + "_setNote(" + errStr + ", errMsg);"); 
      d_writer.println("(*" + excvar + ") = " + IOR.FUND_EXCEPTION_CALL_PREFIX 
        + "__cast(" + errStr + ");");
    }
  }

  /**
   * Return the specified built-in error check function pointer name.
   */
  private String getBuiltinErrorCheckFunctionPtr(boolean doStatic) {
    return IOR.getVectorEntry(IOR.getBuiltinName(IOR.CHECK_ERROR, doStatic));
  }

  /**
   * Generate call to the specified assertion error method implemented by the 
   * user.
   */
  private void generateCheckErrorCall(Assertion as, String var,
                                      boolean doStatic)
  {
     //d_writer.print("((*" + var + ")->");
     d_writer.print("(" + var + "->");
     d_writer.print(getBuiltinErrorCheckFunctionPtr(doStatic) + ")(");
     if (!doStatic) {
       d_writer.print(s_self + ", ");
     } 
     d_writer.println("errMsg);");
  }

  /**
   * Return the specified built-in error check function name.
   */
  private String getBuiltinInterceptorName(SymbolID id, boolean doStatic) {
    return getIORMethodName(id, IOR.getBuiltinName(IOR.INTERCEPTORS, doStatic));
  }

  /**
   * Generate the specified function to set interceptors activation.  
   */
  private void generateInterceptorFunction(Class cls, boolean doStatic) {
    String desc = doStatic ? "static " : "";
    comment("INTERCEPTORS: set " + desc + "interceptor activation.");

    SymbolID id   = cls.getSymbolID();
    d_writer.println("static void " + getBuiltinInterceptorName(id, doStatic)
      + '(');
    d_writer.tab();
    if (!doStatic) {
      d_writer.println(IOR.getObjectName(id) + "* " + s_self + ",");
    }
    d_writer.println("int on)");
    d_writer.backTab();
    d_writer.println("{");
    d_writer.tab();

    if (IOR.supportInterceptors(cls)) {
      String base = getBaseControls(doStatic, s_self) + ".";
      String use  = base + IOR.D_INTERCEPTORS;
      d_writer.println(use + " = on;");
      if (!doStatic) {
        d_writer.println();
        d_writer.println("/* Ensure the EPVs are set properly. */");
        if (IOR.supportAssertions(cls)) {
          String checks = base + IOR.D_CHECKS;
          String rate   = base + IOR.D_RATE;
          d_writer.print(getSetChecksMethodName(id, doStatic) + "(");
          d_writer.print(s_self + ", ");
          d_writer.println(checks + ", " + rate + ", 0);");
        } else {
          String publicepv = s_self + "->" + IOR.getEPVVar(IOR.PUBLIC_EPV);
          d_writer.println("if (" + use + ") {");
          d_writer.tab();
          d_writer.println(publicepv + "  = &" 
            + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_INTERCEPTORS) 
            + ";");
          d_writer.backTab();
          d_writer.println("} else {");
          d_writer.tab();
          d_writer.println(publicepv + "  = &" 
            + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_PUBLIC) + ";");
          d_writer.backTab();
          d_writer.println("}");
          d_writer.println("");
        }
      }
    } else {
      comment("Nothing to do since interceptor support not needed.");
    }

    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Returns the name of the interceptor method for the specified method.
   */
  private String getInterceptorMethodName(Extendable ext, Method meth) {
    return "iceptor_" + IOR.getSymbolName(ext.getSymbolID()) + '_' 
           + meth.getLongMethodName();
  }

  /**
   * Generate all interceptor functions.  These are the functions that
   * invoke the internal pre- and post-method calls before invoking the
   * actual call.  It is assumed these methods are only generated when
   * interceptor generation is enabled.
   */
  private void generateAllInterceptors(Class cls) throws CodeGenerationException
  {
    ArrayList invariants = cls.getAllInvariants();

    List     normal = (List) cls.getMethods(true);
    Iterator i      = normal.iterator();
    Method   meth   = null;
    while (i.hasNext()) {
      meth = (Method)i.next();
      generateInterceptor(cls, meth);
    }
  }

  /**
   * Generate the interceptor method associated with the specified method.
   */
  private void generateInterceptor(Class cls, Method meth) 
    throws CodeGenerationException
  {
    SymbolID  id        = cls.getSymbolID();
    String    name      = meth.getLongMethodName();
    String    methname  = getInterceptorMethodName(cls, meth);
    Type      type      = meth.getReturnType();
    String    retStr    = getReturnString(type);
    String    var;
    boolean   hasReturn = false;
    if ( (type != null) && (type.getType() != Type.VOID) ) {
       hasReturn = true;
    }
    comment("Sandwich the execution of the method between the pre- and post- "
      + "calls.");
    d_writer.println("static " + retStr + " " + methname + "(");
    d_writer.tab();
    List args = meth.getArgumentList();
    boolean hasThrows = !meth.getThrows().isEmpty();
    boolean isStatic  = meth.isStatic();
    C.generateArguments(d_writer, IOR.getObjectName(id) + "* " + s_self, args,
                        isStatic, hasThrows, null, true, true, false);
    d_writer.println(")");
    d_writer.backTab();

    d_writer.println("{");
    d_writer.tab();

    boolean addBlank = false;
    if (hasReturn) {
       d_writer.println(retStr + " " + C.FUNCTION_RESULT + ";");
       addBlank = true;
    }
    if (isStatic) {
       var = "sepv";
       d_writer.println(IOR.getSEPVName(id) + "* " + var + " = " 
         + IOR.getLocalStaticsName(id) + "(" 
         + IOR.getStaticTypeOption(id, IOR.SET_PUBLIC) + ");");
       addBlank = true;
    } else {
       if (IOR.supportAssertions(cls)) {
         var = s_self + "->" + IOR.getEPVVar(IOR.BASE_EPV);
       } else {
         var = "epv";
         d_writer.println(IOR.getEPVName(id) + "* " + var + " = &" 
           + IOR.getStaticEPVVariable(id, IOR.EPV_NEW, IOR.SET_PUBLIC) + ";");
         addBlank = true;
       }
    }
    if (addBlank) {
       d_writer.println();
    }
    generateBaseCall(name, IOR.GENERIC_PRE_SUFFIX, var, args, isStatic, false, 
                     null);
    generateBaseCall(name, "", var, args, isStatic, hasThrows, type);
    generateBaseCall(name, IOR.GENERIC_POST_SUFFIX, var, args, isStatic, false,
                     type);

    if (hasReturn) {
      d_writer.println();
      d_writer.println("return " + C.FUNCTION_RESULT + ";");
    }
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Call the destructor for the object and then deallocate the associated
   * storage using <code>free</code>.
   */
  private void generateDeleteFunction(Class cls) {
    comment("DELETE: call destructor and free object memory.");

    SymbolID id   = cls.getSymbolID();
    String   name = IOR.getSymbolName(id);

    d_writer.println("static void ior_" + name + '_' + s_deleteBuiltin + '(');
    d_writer.tab();
    d_writer.println(IOR.getObjectName(id) + "* self)");
    d_writer.backTab();

    d_writer.println("{");
    d_writer.tab();

    d_writer.println(IOR.getFiniName(id) + "(self);");
    d_writer.println("memset((void*)self, 0, sizeof(" + IOR.getObjectName(id) 
      + "));");
    d_writer.println("free((void*) self);");

    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * @param cls
   */
  private void generateMainExec(Class cls) {
    SymbolID id = cls.getSymbolID();
    String my_symbolName = IOR.getSymbolName(id);
    d_writer.println("struct " + my_symbolName + "__method {");
    d_writer.tab();
    d_writer.println("const char *d_name;");
    d_writer.println("void (*d_func)(struct " + my_symbolName + "__object*,");
    d_writer.tab();
    d_writer.println("struct sidl_io_Deserializer__object *,");
    d_writer.println("struct sidl_io_Serializer__object *);");
    d_writer.backTab();
    d_writer.backTab();
    d_writer.println("};");
    d_writer.println();
    d_writer.println("static void");
    d_writer.println("ior_" + my_symbolName + "__exec(");
    d_writer.println("    struct " + my_symbolName + "__object* self,");
    d_writer.println("    const char* methodName,");
    d_writer.println("    struct sidl_io_Deserializer__object* inArgs,");
    d_writer.println("    struct sidl_io_Serializer__object* outArgs ) { ");
    d_writer.tab();
    ArrayList methods = new ArrayList(cls.getMethods(true));
    Collections.sort(methods, new IOR.CompareMethods());
    d_writer.println("static const struct " + my_symbolName 
      + "__method  s_methods[] = {");
    d_writer.tab();
    for(Iterator i = methods.iterator(); i.hasNext();){
      Method m = (Method)i.next();
      d_writer.println("{ \"" + m.getLongMethodName() + "\", " + my_symbolName 
        + '_' + m.getLongMethodName() + "__exec }" + (i.hasNext() ? "," : ""));
    }
    d_writer.backTab();
    d_writer.println("};");
    d_writer.println("int i, cmp, l = 0;");
    d_writer.println("int u = sizeof(s_methods)/sizeof(struct " + my_symbolName
      + "__method);");
    d_writer.println("if (methodName) {");
    d_writer.tab();
    d_writer.writeCommentLine("Use binary search to locate method");
    d_writer.println("while (l < u) {");
    d_writer.tab();
    d_writer.println("i = (l + u) >> 1;");
    d_writer.println("if (!(cmp=strcmp(methodName, s_methods[i].d_name))) {");
    d_writer.tab();
    d_writer.println("(s_methods[i].d_func)(self, inArgs, outArgs);");
    d_writer.println("return;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println("else if (cmp < 0) u = i;");
    d_writer.println("else l = i + 1;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.writeCommentLine("TODO: add code for method not found");
    d_writer.backTab();
    d_writer.println("}");
  }

  /**
   * Call the destructor for the object and then deallocate the associated
   * storage using <code>free</code>.
   */
  private void generateSuperFunction(Class cls) {
    comment("SUPER: return's parent's non-overrided EPV");

    SymbolID id   = cls.getSymbolID();
    String   name = IOR.getSymbolName(id);
    Class  parent = cls.getParentClass();
    if (parent != null ) {
        SymbolID pid = parent.getSymbolID();

        d_writer.println("static " + IOR.getEPVName(pid) + "* " 
          + IOR.getSymbolName(id) + '_' + s_superBuiltin + "(void) {");
        d_writer.tab();
        d_writer.println( "return " 
          + IOR.getStaticEPVVariable(pid, IOR.EPV_OLD, IOR.SET_PUBLIC) + ";");
        d_writer.backTab();
        d_writer.println("}");
        d_writer.println();
    }
  }

  /**
   * Generate the function that initializes the method entry point vector
   * for this class.  This class performs three functions.  First, it saves
   * the EPVs from the parent classes and interfaces (assuming there is a
   * parent).  Second, it fills the EPV for this class using information from
   * the parent (if there is a parent, and NULL otherwise).  It then calls
   * the skeleton method initialization function for the EPV.  Third, this
   * function resets interface EPV pointers and parent EPV pointers to use
   * the correct function pointers from the class EPV structure.
   */
  private void generateInitEPV(Class cls) throws CodeGenerationException {
    comment("EPV: create method entry point vector (EPV) structure.");

    /*
     * Declare the method signature and open the implementation body.
     */
    SymbolID id     = cls.getSymbolID();
    String   name   = IOR.getSymbolName(id);
    String   object = IOR.getObjectName(id);

    d_writer.println("static void " + name + "__init_epv(");
    d_writer.tab();
    d_writer.println(object + "* " + s_self + ")");
    d_writer.backTab();

    d_writer.println("{");
    comment( "assert( " + IOR.getHaveLockStaticGlobalsMacroName() + " );" );
    d_writer.tab();

    /*
     * Output pointers to the parent classes to simplify access to
     * their data structures.
     */
    if (cls.getParentClass() != null) {
      generateParentSelf(cls, 0, 0);
      d_writer.println();
    }

    /*
     * Output entry point vectors aliases for each parent class and
     * interface as well as a special one for the current object.
     */
    List parents = Utilities.sort(Utilities.getAllParents(cls));
    aliasEPVs(cls, parents, false);

    /*
     * Save all parent interface and class EPVs in static pointers.
     */
    Class parent = cls.getParentClass();
    if (parent != null) {
      saveEPVs(parent, 1);
    }

    /*
     * Generate a list of the nonstatic methods in the class and get
     * the width of every method name.
     */
    List methods = (List) cls.getNonstaticMethods(true);
    int  mwidth  = Math.max(Utilities.getWidth(methods), s_longestBuiltin) 
                     + IOR.getVectorEntry("").length()
                     + IOR.GENERIC_POST_SUFFIX.length();

    /*
     * Output builtin methods.
     */
    for (int j = 0; j < IOR.CLASS_BUILTIN_METHODS; j++) {
      if (  IOR.isBuiltinBasic(j)
         || ( IOR.isBuiltinAssert(j) && IOR.supportAssertions(cls) )
         || ( (j == IOR.INTERCEPTORS) && IOR.supportInterceptors(cls) ) ) {
        String mname = IOR.getBuiltinName(j);
        d_writer.print("epv->");
        d_writer.printAligned(IOR.getVectorEntry(mname), mwidth);
        if (  (j == IOR.CONSTRUCTOR) || (j == IOR.DESTRUCTOR)
           || (j == IOR.CHECK_ERROR) )
        {
            d_writer.println(" = " + C.NULL + ";");
        } else {
            d_writer.println(" = " + getIORMethodName(id, mname) + ';');
        }
      }
    }

    /*
     * Output the class methods.
     */
    generateNonstaticMethods(cls, object, methods, "epv", parent, mwidth);

    /*
     * Iterate through all parent EPVs and set each of the function pointers
     * to use the corresponding function in the parent EPV structure.
     */
    copyEPVs(parents);
    d_writer.println("s_method_initialized = 1;");
    d_writer.println("ior_" + name + "__ensure_load_called();");

    /*
     * Close the function definition.
     */
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Generate the non-static methods for the specified class.
   */
  private void generateNonstaticMethods(Class cls, String object, List methods,
                                        String var, Class parent, int mwidth)
    throws CodeGenerationException
  {
    SymbolID id = cls.getSymbolID();

    /*
     * Iterate through all of the nonstatic methods.  Assign them to
     * NULL if the parent class does not have the method and to the parent
     * function pointer if the parent has the method.
     */
    for (Iterator i = methods.iterator(); i.hasNext(); ) {
      Method method = (Method) i.next();
      String mname  = method.getLongMethodName();
      boolean parentHas = (  (parent != null) 
                          && (parent.hasMethodByLongName(mname, true)) );
      generateNonstaticMethod(cls, method, object, var, mname, parentHas, 
                              mwidth);
    }
    d_writer.println();

    /*
     * Call the user initialization function to set up the EPV structure.
     */
    d_writer.println(IOR.getSetEPVName(id) + "(" + var + ");");

    /*
     * Now, if necessary, clone the static that has been set up and 
     * overlay with local methods for assertions and/or interceptors.
     */
    String xEPV;
    d_writer.println();
    boolean addBlank = false;
    if (IOR.supportAssertions(cls)) {
      xEPV = "aepv";
      d_writer.println("memcpy((void*)" + xEPV + ", " + var + ", sizeof(" 
        + IOR.getEPVName(id) + "));");
      generateEPVMethodAssignments(cls, IOR.SET_ASSERTIONS, xEPV, false);
      addBlank = true;
    }
    if (IOR.supportInterceptors(cls)) {
      if (addBlank) {
        d_writer.println();
        addBlank = false;
      }
      xEPV = "iepv";
      d_writer.println("memcpy((void*)" + xEPV + ", " + var + ", sizeof(" 
        + IOR.getEPVName(id) + "));");
      generateEPVMethodAssignments(cls, IOR.SET_INTERCEPTORS, xEPV, false);
      addBlank = true;
    }
    if (addBlank) {
      d_writer.println();
    }
  }

  /**
   * Generate the non-static method epv entry(ies) for the method.
   */
  private void generateNonstaticMethod(Extendable ext, Method m, String object,
                                       String var, String name, 
                                       boolean parentHas, int width)
    throws CodeGenerationException
  {
    String ename = IOR.getVectorEntry(name);
    if (IOR.supportInterceptors(ext)) {
       d_writer.print(var + "->");
       d_writer.printAligned(ename + IOR.GENERIC_PRE_SUFFIX, width);
       d_writer.println(" = " + C.NULL + ";");
       generateNonstaticEPV(m, object, var, ename, parentHas, width);
       d_writer.print(var + "->");
       d_writer.printAligned(ename + IOR.GENERIC_POST_SUFFIX, width);
       d_writer.println(" = " + C.NULL + ";");
    } else {
       generateNonstaticEPV(m, object, var, ename, parentHas, width);
    }
  }

  /**
   * Generate the non-static method epv entry for the method.
   */
  private void generateNonstaticEPV(Method m, String object, String var,
                                    String ename, boolean parentHas, int width) 
    throws CodeGenerationException
  {
    d_writer.print(var + "->");
    d_writer.printAligned(ename, width);
    if (parentHas) {
      d_writer.print(" = ");
      d_writer.print(IOR.getCast(m, object + "*"));
      d_writer.println(" s1->" + IOR.getEPVVar(IOR.PUBLIC_EPV) + "->" + ename 
        + ";");
    } else {
      d_writer.println(" = " + C.NULL + ";");
    }
  }

  /**
   * Generate the function that initializes the static entry point
   * vector (assuming the class contains static methods).  If the
   * parent class contains static methods, then copy the pointers
   * from the parent class, since static methods are "inherited" from
   * the parent (in a funny kinda way).  Call the skeleton-supplied
   * function to initialize the remaining static methods.
   */
  private void generateInitSEPV(Class cls) throws CodeGenerationException {
    if (cls.hasStaticMethod(true)) {
      comment("SEPV: create the static entry point vector (SEPV).");

      /*
       * Define the method signature and begin the method body.
       */
      SymbolID id    = cls.getSymbolID();
      String   name  = IOR.getSymbolName(id);
      String   lower = name.toLowerCase();
      int      width = name.length();

      d_writer.println("static void " + name + "__init_sepv(void)");
      d_writer.println("{");
      d_writer.tab();
      comment( "assert( " + IOR.getHaveLockStaticGlobalsMacroName() + " );" );

      /*
       * Get information about the parent (if it exists and has static
       * methods).
       */
      Class    parent = cls.getParentClass();
      SymbolID pid    = null;
      boolean  includeParent = (parent != null) && parent.hasStaticMethod(true);
      boolean  doAsserts     = IOR.supportAssertions(cls);
      boolean  doIceptors    = IOR.supportInterceptors(cls);

      if (includeParent) {
        pid = parent.getSymbolID();
        int w = IOR.getSymbolName(pid).length();
        if (w > width) {
          width = w;
        }
      }
      width += "struct __sepv*".length() + 1;

      /*
       * Create pointers to the local static EPV structure and one
       * to the parent SEPV structure (if the parent exists and has
       * static members).
       */
      d_writer.printAligned(IOR.getSEPVName(id) + "*", width);
      d_writer.println(" s = &" 
        + IOR.getStaticEPVVariable(id, IOR.EPV_STATIC, IOR.SET_PUBLIC) + ";");
      if (doAsserts) {
        d_writer.printAligned(IOR.getSEPVName(id) + "*", width);
        d_writer.println("as = &" 
          + IOR.getStaticEPVVariable(id, IOR.EPV_STATIC, IOR.SET_ASSERTIONS) 
          + ";");
      }
      if (doIceptors) {
        d_writer.printAligned(IOR.getSEPVName(id) + "*", width);
        d_writer.println("is = &" 
          + IOR.getStaticEPVVariable(id, IOR.EPV_STATIC, IOR.SET_INTERCEPTORS) 
          + ";");
      }
      if (includeParent) {
        d_writer.printAligned(IOR.getSEPVName(pid) + "*", width);
        d_writer.println(" p = " + IOR.getStaticsName(pid) + "();");
      }
      d_writer.println();

      String var = "s";
      /*
       * Output builtin methods.
       */
      for(int j = 0; j < IOR.CLASS_BUILTIN_METHODS; j++) {
        if (  IOR.hasStaticBuiltin(j) 
           && (  ( IOR.isBuiltinAssert(j) && doAsserts )
              || ( (j == IOR.INTERCEPTORS) && doIceptors) ) ) {
          String mname = IOR.getBuiltinName(j, true);
          d_writer.print(var + "->");
          d_writer.printAligned(IOR.getVectorEntry(mname), width);
          if (  (IOR.CHECKS == j) || (IOR.INTERCEPTORS == j) 
             || (IOR.DUMP_STATS == j)  ) {
            d_writer.println(" = " + getIORMethodName(id, mname) + ';');
          } else {
            d_writer.println(" = " + C.NULL + ";");
          }
        }
      }

      /*
       * Output the static methods.
       */
      generateStaticMethods(cls, var, parent);

      /*
       * Initialize the remainder of the static functions by calling
       * the skeleton initialization routine.
       */
      d_writer.println(IOR.getSetSEPVName(id) + "(" + var + ");");
      d_writer.println();

      /*
       * If necessary, initialize the static assertion and/or interceptor
       * functions as well.
       */
      String  sEPV;
      boolean addBlank = false;
      if (doIceptors) {
        sEPV = "is";
        d_writer.println("memcpy((void*)" + sEPV + ", " + var + ", sizeof(" 
          + IOR.getSEPVName(id) + "));");
        generateEPVMethodAssignments(cls, IOR.SET_INTERCEPTORS, sEPV, true);
        d_writer.println();
        if (doAsserts) {
          d_writer.println(getBaseControls(true, "")  + "." + IOR.D_INTERCEPTORS
            + " = " + IOR.DEFAULT_OPTION_INTERCEPTORS + ";");
        } else {
          d_writer.println(getBuiltinInterceptorName(id, true) + "(" 
            + IOR.DEFAULT_OPTION_INTERCEPTORS + ");");
        }
        addBlank = true;
      }
      if (doAsserts) {
        if (addBlank) {
          d_writer.println();
        }
        sEPV = "as";
        d_writer.println("memcpy((void*)" + sEPV + ", " + var + ", sizeof(" 
          + IOR.getSEPVName(id) + "));");
        generateEPVMethodAssignments(cls, IOR.SET_ASSERTIONS, sEPV, true);
        d_writer.println();
        d_writer.println(getSetChecksMethodName(id, true) 
          + "(" + IOR.DEFAULT_CHECK_LEVEL + ", " + IOR.DEFAULT_THRESHOLD + ", 0);");
        addBlank = true;
      }
      if (addBlank) {
        d_writer.println();
      }
      d_writer.println("s_static_initialized = 1;");
      d_writer.println("ior_" + name + "__ensure_load_called();");

      /*
       * Close the initialization guard and end the function body.
       */
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println();
    }
  }
    
  /**
   */
  private void generateEnsureLoad(Class cls ) { 
    SymbolID id    = cls.getSymbolID();
    String   name  = IOR.getSymbolName(id);
    String   lower = name.toLowerCase();
    
    d_writer.println("static void ior_" + name 
      + "__ensure_load_called(void) {");
    d_writer.tab();
    comment( "assert( " + IOR.getHaveLockStaticGlobalsMacroName() + " );" );
    d_writer.println("if (! s_load_called ) {");
    d_writer.tab();
    d_writer.println(name + "__call_load();");
    d_writer.println("s_load_called=1;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.backTab();
    d_writer.println("}");
  }

  /**
   * Generate the static methods for the specified class.
   */
  private void generateStaticMethods(Class cls, String var, Class parent)
    throws CodeGenerationException
  {
    SymbolID id      = cls.getSymbolID();
    List     methods = (List) cls.getStaticMethods(true);
    int      mwidth  = Utilities.getWidth(methods)
                     + IOR.getVectorEntry("").length()
                     + IOR.GENERIC_SUFFIX_MAXLEN + var.length() + 2;
    /*
     * Iterate through all of the static methods.  Assign them to NULL
     * if the parent class does not have the method and to the parent
     * function pointer if the parent has the method.
     */
    for (Iterator i = methods.iterator(); i.hasNext(); ) {
      Method method = (Method) i.next();
      String mname  = method.getLongMethodName();
      boolean parentHas = (  (parent != null) 
                          && (parent.hasMethodByLongName(mname, true)) );
      String ename  = IOR.getVectorEntry(mname);
      if (IOR.supportInterceptors(cls)) {
         d_writer.print(var + "->");
         d_writer.printAligned(ename + IOR.GENERIC_PRE_SUFFIX, mwidth);
         d_writer.println(" = " + C.NULL + ";");
         generateStaticEPV(var, ename, parentHas, mwidth);
         d_writer.print(var + "->");
         d_writer.printAligned(ename + IOR.GENERIC_POST_SUFFIX, mwidth);
         d_writer.println(" = " + C.NULL + ";");
      } else {
         generateStaticEPV(var, ename, parentHas, mwidth);
      }
    }
    d_writer.println();
  }

  /**
   * Generate the static method epv entry for the method.
   */
  private void generateStaticEPV(String var, String ename, boolean parentHas,
                                 int mwidth) 
    throws CodeGenerationException
  {
    d_writer.print(var + "->");
    d_writer.printAligned(ename, mwidth);
    if (parentHas) {
      d_writer.println(" = p->" + ename + ";");
    } else {
      d_writer.println(" = " + C.NULL + ";");
    }
  }

  /**
   * Generate the appropriate assertion or interceptor methods assignments.
   */
  void generateEPVMethodAssignments(Extendable ext, int setType, String ptrVar,
                                    boolean doStatics) 
  {
    List     methods = (doStatics ? (List)ext.getStaticMethods(true)
                                  : (List)ext.getNonstaticMethods(true));
    Iterator i       = methods.iterator();
    Method   m       = null;
    String   mName, tName;
    int      mwidth  = Math.max(Utilities.getWidth(methods), s_longestBuiltin)
                       + IOR.getVectorEntry("").length()
                       + ptrVar.length() + 2;
    while (i.hasNext()) {
      m     = (Method)i.next();
      mName = m.getLongMethodName();
      d_writer.printAligned(ptrVar + "->" 
        + IOR.getVectorEntry(mName), mwidth);
      tName = (setType == IOR.SET_ASSERTIONS) ? getCheckMethodName(ext, m)
                                        : getInterceptorMethodName(ext, m);
      d_writer.println(" = " + tName + ";");
    }
  }

  /**
   * Generate a function that will return a pointer to the static entry
   * point vector.  This function is called by clients to get access to
   * static methods in the class.  If this class has no static methods,
   * then do not generate an implementation.
   */
  private void generateStaticFunction(Class cls, boolean doLocal) {
    boolean  doAsserts  = IOR.supportAssertions(cls);
    boolean  doIceptors = IOR.supportInterceptors(cls);

    if (  cls.hasStaticMethod(true)
       && ((!doLocal) || (doAsserts || doIceptors)) )
    {
      String desc = doLocal ? "specified " : "";
      comment("STATIC: return pointer to " + desc + "static EPV structure.");

      SymbolID id         = cls.getSymbolID();
      String   name       = IOR.getSymbolName(id);
      String   lower      = name.toLowerCase();

      d_writer.println(IOR.getSEPVName(id) + "*");
      if (doLocal) {
        d_writer.println(IOR.getLocalStaticsName(id) + "(int type)");
      } else {
        d_writer.println(IOR.getStaticsName(id) + "(void)");
      }
      d_writer.println("{");
      d_writer.tab();

      d_writer.println( IOR.getLockStaticGlobalsMacroName() + ";" );

      if (doAsserts || doIceptors) {
        d_writer.println(IOR.getSEPVName(id) + "* sepv;");
      }
      d_writer.println("if (!s_static_initialized) {");
      d_writer.tab();
      d_writer.println(name + "__init_sepv();");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println( IOR.getUnlockStaticGlobalsMacroName() + ";" );

      boolean cont = false;
      if (doAsserts) {
        cont = true;
        if (doLocal) {
          d_writer.println("if (type == " 
            + IOR.getStaticTypeOption(id, IOR.SET_ASSERTIONS) + ") {");
        } else {
          d_writer.println("if (" + getBaseControls(true, "") + "." 
            + IOR.D_CHECKS + " & CHECK_ASSERTIONS) {");
        } 
        d_writer.tab();
        d_writer.println("sepv = &"
          + IOR.getStaticEPVVariable(id, IOR.EPV_STATIC, IOR.SET_ASSERTIONS) 
          + ";");
        d_writer.backTab();
        d_writer.print("}");
      }
      if (doIceptors) {
        if (cont) {
          d_writer.print(" else ");
        }
        if (doLocal) {
          d_writer.println("if (type == " 
            + IOR.getStaticTypeOption(id, IOR.SET_INTERCEPTORS) + ") {");
        } else {
          d_writer.println("if (" + getBaseControls(true, "") + "."
            + IOR.D_INTERCEPTORS + ") {");
        } 
        d_writer.tab();
        d_writer.println("sepv = &"
          + IOR.getStaticEPVVariable(id, IOR.EPV_STATIC, IOR.SET_INTERCEPTORS) 
          + ";");
        d_writer.backTab();
        d_writer.print("}");
      }
      if (doAsserts || doIceptors) {
        d_writer.println(" else {");
        d_writer.tab();
        d_writer.println("sepv = &"
          + IOR.getStaticEPVVariable(id, IOR.EPV_STATIC, IOR.SET_PUBLIC) + ";");
        d_writer.backTab();
        d_writer.println("}");
        d_writer.println("return sepv;");
      } else {
        d_writer.println("return &"
          + IOR.getStaticEPVVariable(id, IOR.EPV_STATIC, IOR.SET_PUBLIC) + ";");
      }
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println();
    }
  }

  private void generateInitClassInfo(Class cls) {
    if (!cls.isAbstract()) {
      if (d_classInfoI.equals(cls.getSymbolID())){
        d_writer.println("static void initMetadata("
          + IOR.getObjectName(cls.getSymbolID()) + "*);");
        d_writer.println();
      }
      comment("initClassInfo: create a ClassInfo interface if necessary.");
      d_writer.println("static void");
      d_writer.println("initClassInfo(" + C.getObjectName(d_classInfo) 
        + " *info)");
      d_writer.println("{");
      d_writer.tab();
      if (! d_classInfoI.equals(cls.getSymbolID())){
	d_writer.println( IOR.getLockStaticGlobalsMacroName() + ";" );
      } else { 
        d_writer.writeCommentLine("ClassInfo for ClassInfoI is a special case");
	d_writer.writeCommentLine( IOR.getUnlockStaticGlobalsMacroName() 
          + ";" );
      }
      d_writer.println("if (s_classInfo_init) {");
      d_writer.tab();
      d_writer.println(C.getObjectName(d_classInfoI) + " impl;");
      d_writer.println("s_classInfo_init = 0;");
      d_writer.println("impl = " 
        + C.getFullMethodName(d_classInfoI, "_create") + "();");
      d_writer.println("s_classInfo = " 
        + C.getFullMethodName(d_classInfo, "_cast") + "(impl);");
      d_writer.println("if (impl) {");
      d_writer.tab();
      d_writer.println(C.getFullMethodName(d_classInfoI, "setName") 
        + "(impl, \"" + cls.getFullName() + "\");");
      d_writer.println(C.getFullMethodName(d_classInfoI, "setIORVersion") 
        + "(impl, s_IOR_MAJOR_VERSION, s_IOR_MINOR_VERSION);");
      if (d_classInfoI.equals(cls.getSymbolID())) {
        d_writer.println("initMetadata(impl);");
      }
      d_writer.backTab();
      d_writer.println("}");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println("if (s_classInfo) {");
      d_writer.tab();
      d_writer.println("if (*info) {");
      d_writer.tab();
      d_writer.println(C.getFullMethodName(d_classInfo, "deleteRef") 
        + "(*info);");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println("*info = s_classInfo;");
      d_writer.println(C.getFullMethodName(d_classInfo, "addRef") 
        + "(*info);");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.backTab();
      if (! d_classInfoI.equals(cls.getSymbolID())){
	  d_writer.println( IOR.getUnlockStaticGlobalsMacroName() + ";" );
      } else { 
	  d_writer.writeCommentLine("ClassInfo for ClassInfoI is a special "
            + "case");
	  d_writer.writeCommentLine(IOR.getUnlockStaticGlobalsMacroName() 
            + ";" );
      }
      d_writer.println("}");
      d_writer.println();
    }
  }

  private void generateInitMetadata(Class cls) {
    if (!cls.isAbstract()) {
      SymbolID id     = cls.getSymbolID();
      String   object = IOR.getObjectName(id);
      comment("initMetadata: store IOR version & class in sidl.BaseClass's "
        + "data");
      d_writer.println("static void");
      d_writer.println("initMetadata(" + object + "* " + s_self + ")");
      d_writer.println("{");
      d_writer.tab();
      d_writer.println("if (" + s_self + ") {");
      d_writer.tab();
      d_writer.println(C.getDataName(d_baseClass) + " *data = " 
        + C.getDataGetName(d_baseClass) + "(" 
        + C.getFullMethodName(d_baseClass, "_cast") + "(" + s_self + "));");
      d_writer.println("if (data) {");
      d_writer.tab();
      d_writer.println("data->d_IOR_major_version = s_IOR_MAJOR_VERSION;");
      d_writer.println("data->d_IOR_minor_version = s_IOR_MINOR_VERSION;");
      d_writer.println("initClassInfo(&(data->d_classinfo));");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println();
    }
  }

  /**
   * Allocate the memory for a new instance of the class and then call
   * the constructor (initializer) on the object.  This method is not
   * output for abstract classes.
   */
  private void generateNewFunction(Class cls) {
    if (!cls.isAbstract()) {
      comment("NEW: allocate object and initialize it.");

      SymbolID id     = cls.getSymbolID();
      String   object = IOR.getObjectName(id);

      d_writer.println(object + "*");
      d_writer.println(IOR.getNewName(id) + "(void)");
      d_writer.println("{");
      d_writer.tab();

      d_writer.println(object + "* " + s_self + " =");
      d_writer.tab();
      d_writer.println("(" + object + "*) malloc(");
      d_writer.tab();
      d_writer.println("sizeof(" + object + "));");
      d_writer.backTab();
      d_writer.backTab();

      d_writer.println(IOR.getInitName(id) + "(" + s_self + ");");
      d_writer.println("initMetadata(" + s_self + ");");
      d_writer.println("return " + s_self + ";");

      d_writer.backTab();
      d_writer.println("}");
      d_writer.println();
    }
  }

  /**
   * Generate the initialization function that acts as the constructor
   * for a class object.  The logic is as follows.  First, if there is
   * a parent for this class, call its constructor.  Then, make sure that
   * the EPVs have been initialized.  If there are parents, make sure that
   * all parent EPVs are updated to point to the corrected EPVs.  Then
   * initialize the new interfaces in this object.  Finally, call the
   * user-defined constructor.
   */
  private void generateInitFunction(Class cls) {
    comment("INIT: initialize a new instance of the class object.");

    /*
     * Declare the method signature and open the implementation body.
     */
    SymbolID id = cls.getSymbolID();
    d_writer.println("void " + IOR.getInitName(id) + "(");
    d_writer.tab();
    d_writer.println(IOR.getObjectName(id) + "* " + s_self + ")");
    d_writer.backTab();

    d_writer.println("{");
    d_writer.tab();

    /*
     * Output parent pointers to simplify access to parent classes.
     */
    generateParentSelf(cls, 0, 0);
    d_writer.println();

    /*
     * If there is a parent class, then call its constructor.
     */
    Class parent = cls.getParentClass();
    if (parent != null) {
      String p = IOR.getSymbolName(parent.getSymbolID());
      d_writer.println(IOR.getInitName(parent.getSymbolID()) + "(s1);");
      d_writer.println();
    }

    /*
     * Ensure that the EPV has been initialized.
     */
    d_writer.println( IOR.getLockStaticGlobalsMacroName() + ";" );
    d_writer.println("if (!s_method_initialized) {");
    d_writer.tab();
    d_writer.println(IOR.getSymbolName(id) + "__init_epv(s0);");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println( IOR.getUnlockStaticGlobalsMacroName() + ";" );
    d_writer.println();

    /*
     * Modify the EPVs in parent classes and their interfaces.
     */
    fixEPVs(cls, 0, true);

    /*
     * Iterate through the interfaces defined in this class and allocate
     * the symbol identifier and set the self pointer.
     */
    List interfaces = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
    for (Iterator i = interfaces.iterator(); i.hasNext(); ) {
      SymbolID iid = (SymbolID) i.next();
      String name = IOR.getSymbolName(iid).toLowerCase();
      d_writer.println("s0->d_" + name + ".d_object = " + s_self + ";");
      d_writer.println();
    }

    /*
     * Finally, set the state and data pointer for this class and allocate the
     * symbol identifier.
     */
    d_writer.println("s0->" + IOR.D_DATA + " = " + C.NULL + ";");
    d_writer.println();
    if (IOR.supportInterceptors(cls)) {
      if (IOR.supportAssertions(cls)) {
        d_writer.println("s0->" + IOR.D_CONTROLS + "." + IOR.D_INTERCEPTORS 
          + " = " + IOR.DEFAULT_OPTION_INTERCEPTORS + ";");
      } else {
        d_writer.println(getBuiltinInterceptorName(id, false) + "(s0, " 
          + IOR.DEFAULT_OPTION_INTERCEPTORS + ");");
      }
    }
    if (IOR.supportAssertions(cls)) {
       d_writer.println(getSetChecksMethodName(id, false) 
         + "(s0, " + IOR.DEFAULT_CHECK_LEVEL + ", " + IOR.DEFAULT_THRESHOLD 
         + ", 0);");
    }
    d_writer.println();

    /*
     * Call the user constructor now that the object has been constructed.
     */
    d_writer.println(methodCall(cls, s_self,
                                IOR.getBuiltinName(IOR.CONSTRUCTOR), ""));

    /*
     * Close the function definition.
     */
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Generate the finalization function that acts as the destructor
   * for a class object.  The logic is as follows.  First, call the
   * user-defined destructor for the object.  Deallocate all identifiers
   * for this object.  If there are parents, reset the EPVs to their
   * previous values and call the parent destructor.
   */
  private void generateFiniFunction(Class cls) {
    comment("FINI: deallocate a class instance (destructor).");

    /*
     * Declare the method signature and open the implementation block.
     */
    SymbolID id = cls.getSymbolID();
    d_writer.println("void " + IOR.getFiniName(id) + "(");
    d_writer.tab();
    d_writer.println(IOR.getObjectName(id) + "* " + s_self + ")");
    d_writer.backTab();

    d_writer.println("{");
    d_writer.tab();

    /*
     * Output parent pointers to simplify access to parent classes.
     */
    generateParentSelf(cls, 0, 0);
    d_writer.println();

    /*
     * Call the user-defined destructor for this class.
     */
    d_writer.println(methodCall(cls, "s0", IOR.getBuiltinName(IOR.DESTRUCTOR), 
                                ""));

    /*
     * If there is a parent class, then reset all parent pointers and
     * call the parent destructor.
     */
    Class parent = cls.getParentClass();
    if (parent != null) {
      d_writer.println();
      fixEPVs(parent, 1, false);
      String p = IOR.getSymbolName(parent.getSymbolID());
      d_writer.println(IOR.getFiniName(parent.getSymbolID()) + "(s1);");
    }

    /*
     * Close the function definition.
     */
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Generate the version function.
   */
  private void generateVersionFunction(Class cls) {
    comment("VERSION: Return the version of the IOR used to generate this "
      + "IOR.");

    /*
     * Declare the method signature and open the implementation block.
     */
    SymbolID id = cls.getSymbolID();
    d_writer.println("void");
    d_writer.println(IOR.getVersionName(id) + "(int32_t *major, int32_t *minor"
      + ")");
    d_writer.println("{");
    d_writer.tab();
    d_writer.println("*major = s_IOR_MAJOR_VERSION;");
    d_writer.println("*minor = s_IOR_MINOR_VERSION;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Recursively output self pointers to the SIDL objects for this class
   * and its parents.  The self pointers are of the form sN, where N is
   * an integer represented by the level argument.  If the width is zero,
   * then the width of all parents is generated automatically.
   */
  private void generateParentSelf(Class cls, int level, int width) {
    if (cls != null) {

      /*
       * Calculate the width of this class and all parents for pretty
       * output.  Ooh, very pretty.
       */
      if (width == 0) {
        Class parent = cls;
        while (parent != null) {
          int w = IOR.getObjectName(parent.getSymbolID()).length();
          if (w > width) {
            width = w;
          }
          parent = parent.getParentClass();
        }
      }

      /*
       * Now use the width information to print out symbols.
       */
      SymbolID id = cls.getSymbolID();
      d_writer.printAligned(IOR.getObjectName(id) + "*", width+1);
      d_writer.print(" s" + String.valueOf(level) + " = ");
      if (level == 0) {
        d_writer.println(s_self + ";");
      } else {
        d_writer.print("&s" + String.valueOf(level-1) + "->d_");
        d_writer.println(IOR.getSymbolName(id).toLowerCase() + ";");
      }
      generateParentSelf(cls.getParentClass(), level + 1, width);
    }
  }

  /*
   * Recursively modify the EPVs in parent classes and set up interface
   * pointers.  Nothing is done if the class argument is null.  The flag
   * is_new determines whether the EPVs are being set to a newly defined
   * EPV or to a previously saved EPV.
   */
  private void fixEPVs(Class cls, int level, boolean is_new) {
    if (cls != null) {
      fixEPVs(cls.getParentClass(), level+1, is_new);

      /*
       * Update the EPVs for all of the new interfaces in this
       * particular class.
       */
      String self   = "s" + String.valueOf(level);
      int    epvType = is_new ? IOR.EPV_NEW : IOR.EPV_OLD;
      String prefix  = is_new ? "&" : "";
      List   ifce   = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
      String epv    = IOR.getEPVVar(IOR.PUBLIC_EPV);
      int    width  = Utilities.getWidth(ifce) + epv.length() + 3;

      int    e = 1;
      String name;
      for (Iterator i = ifce.iterator(); i.hasNext(); ) {
        SymbolID id = (SymbolID) i.next();
        name        = IOR.getSymbolName(id).toLowerCase();
        d_writer.print(self + "->");
        d_writer.printAligned("d_" + name + "." + epv, width);
        d_writer.print(" = ");
        d_writer.println(prefix
          + IOR.getStaticEPVVariable(id, epvType, IOR.SET_PUBLIC) + ";");
      }

      /*
       * Modify the class entry point vector.
       */
      String rhs = prefix + IOR.getStaticEPVVariable(cls.getSymbolID(), epvType, 
                                                 IOR.SET_PUBLIC) + ";";
      d_writer.print(self + "->");
      d_writer.printAligned(epv, width);
      d_writer.println(" = " + rhs);
      if (IOR.supportBaseEPVAttr(cls)) {
        d_writer.print(self + "->");
        d_writer.printAligned(IOR.getEPVVar(IOR.BASE_EPV), width);
        d_writer.println(" = " + rhs);
      }
      d_writer.println();
    }
  }

  /*
   * Recursively save the class and interface EPVs in this class and all
   * parent classes.  Nothing is done if the class argument is null.
   */
  private void saveEPVs(Class cls, int level) {
    if (cls != null) {
      saveEPVs(cls.getParentClass(), level+1);

      /*
       * Save the EPVs for all of the new interfaces in this class.
       */
      String self  = "s" + String.valueOf(level);
      List   ifce  = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
      int    width = Utilities.getWidth(ifce);

      for (Iterator i = ifce.iterator(); i.hasNext(); ) {
        SymbolID id = (SymbolID) i.next();
        d_writer.print(IOR.getStaticEPVVariable(id, IOR.EPV_OLD, 
                                                IOR.SET_PUBLIC));
        String name = IOR.getSymbolName(id).toLowerCase();
        d_writer.printSpaces(width - name.length());
        d_writer.println(" = " + self + "->d_" + name + "." 
          + IOR.getEPVVar(IOR.PUBLIC_EPV) + ";");
      }

      /*
       * Save the class entry point vector.
       */
      d_writer.print(IOR.getStaticEPVVariable(cls.getSymbolID(), IOR.EPV_OLD, 
                                          IOR.SET_PUBLIC));
      String lower = IOR.getSymbolName(cls.getSymbolID()).toLowerCase();
      d_writer.printSpaces(width - lower.length());
      d_writer.println(" = " + self + "->" + IOR.getEPVVar(IOR.PUBLIC_EPV) 
        + ";");
      d_writer.println();
    }
  }

  /*
   * Recursively modify the remote EPVs in parent classes and set up
   * the interfaces.
   */
  private void remoteEPVs(Class cls, int level) {
    if (cls != null) {
      remoteEPVs(cls.getParentClass(), level+1);

      /*
       * Update the EPVs for all of the new interfaces in this
       * particular class.
       */
      String self = "s" + String.valueOf(level);
      List   ifce = Utilities.sort(Utilities.getUniqueInterfaceIDs(cls));
      for (Iterator i = ifce.iterator(); i.hasNext(); ) {
        SymbolID id   = (SymbolID) i.next();
        String   name = IOR.getSymbolName(id).toLowerCase();
        d_writer.println(self + "->d_" + name + "." 
          + IOR.getEPVVar(IOR.PUBLIC_EPV) + "    = &" 
          + IOR.getStaticEPVVariable(id, IOR.EPV_REMOTE, IOR.SET_PUBLIC) + ";");

        d_writer.println(self + "->d_" + name + ".d_object = " + C.NULL 
          + "; /* FIXME */");
        d_writer.println();
      }

      /*
       * Modify the class entry point vector.
       */
      d_writer.println(self + "->d_data = " + C.NULL + "; /* FIXME */");
      d_writer.println(self + "->" + IOR.getEPVVar(IOR.PUBLIC_EPV) + "  = &" 
        + IOR.getStaticEPVVariable(cls.getSymbolID(), IOR.EPV_REMOTE, 
                                   IOR.SET_PUBLIC) + ";");
      d_writer.println();
    }
  }

  /**
   * Generate the entry point vector alias for each parent class and each
   * interface as well as a special one for the current object.
   */
  private void aliasEPVs(Class cls, Collection parents, boolean remote) {
    /*
     * Get the width of the symbols for pretty printing.
     */
    SymbolID id      = cls.getSymbolID();
    String   name    = IOR.getSymbolName(id);
    String   epv     = IOR.getEPVName(id);
    int      epvType = remote ? IOR.EPV_REMOTE : IOR.EPV_NEW;

    int width = epv.length() + 2;
    int w     = Utilities.getWidth(parents) + "struct __epv*".length();
    if (w > width) {
      width = w;
    }

    /*
     * Output the EPV pointer for this class and its class and interface
     * parents.
     */
    d_writer.printAligned(epv + "*", width);
    d_writer.println(" epv  = &" 
      + IOR.getStaticEPVVariable(id, epvType, IOR.SET_PUBLIC) + ";");
    if (IOR.supportAssertions(cls)) {
      d_writer.printAligned(epv + "*", width);
      d_writer.println(" aepv = &" 
        + IOR.getStaticEPVVariable(id, epvType, IOR.SET_ASSERTIONS) + ";");
    }
    if (IOR.supportInterceptors(cls)) {
      d_writer.printAligned(epv + "*", width);
      d_writer.println(" iepv = &" 
        + IOR.getStaticEPVVariable(id, epvType, IOR.SET_INTERCEPTORS) + ";");
    }

    int e = 0;
    for (Iterator i = parents.iterator(); i.hasNext(); ) {
      SymbolID sid   = (SymbolID) i.next();
      String   lower = IOR.getSymbolName(sid).toLowerCase();
      d_writer.printAligned(IOR.getEPVName(sid) + "*", width);
      d_writer.printAligned(" e" + String.valueOf(e++), 4);
      d_writer.println("  = &" 
        + IOR.getStaticEPVVariable(sid, epvType, IOR.SET_PUBLIC) + ";");
    }
    d_writer.println();
  }

  /*
   * Copy EPV function pointers from the most derived EPV data structure
   * into all parent class and interface EPVs.
   */
  private void copyEPVs(Collection parents) throws CodeGenerationException {
    int e = 0;
    for (Iterator i = parents.iterator(); i.hasNext(); ) {
      /*
       * Extract information about the parent extendable object.
       * Generate a list of the nonstatic methods and calculate
       * the width of every method name.
       */
      SymbolID   id   = (SymbolID) i.next();
      Extendable ext  = (Extendable) Utilities.lookupSymbol(id);
      String     name = IOR.getSymbolName(id);

      List methods = (List) ext.getNonstaticMethods(true);
      int  mwidth  = Math.max(Utilities.getWidth(methods), s_longestBuiltin) 
                     + IOR.getVectorEntry("").length();

      /*
       * Calculate the "self" pointer for the cast function.
       */
      String self = null;
      if (ext.isInterface()) {
        self = "void*";
      } else {
        self = IOR.getObjectName(id) + "*";
      }

      /*
       * Generate the assignments for the cast and delete functions.
       */
      String estring = "e" + String.valueOf(e) + "->";
      String vecEntry = IOR.getVectorEntry(s_castBuiltin);

      d_writer.print(estring);
      d_writer.printAligned(vecEntry, mwidth);
      d_writer.print(" = (void* (*)(" + self);
      d_writer.print(",const char*)) epv->");
      d_writer.print(vecEntry);
      d_writer.println(";");

      vecEntry = IOR.getVectorEntry(s_deleteBuiltin);
      d_writer.print(estring);
      d_writer.printAligned(vecEntry, mwidth);
      d_writer.println(" = (void (*)(" + self + ")) epv->" + vecEntry + ";");
      /**
       * Generate the assignment for exec function
       */
      vecEntry = IOR.getVectorEntry(s_execBuiltin);
      Method m_exec = C.getExecMethod();
      String ename  = m_exec.getLongMethodName();
      d_writer.print(estring);
      d_writer.printAligned(IOR.getVectorEntry(ename), mwidth);
      d_writer.print(" = " + IOR.getCast(m_exec, self));
      d_writer.println(" epv->" + IOR.getVectorEntry(ename) + ";");

      //      d_writer.print(estring);
      //d_writer.printAligned(vecEntry, mwidth);
      //d_writer.print(" = (void (*)(" + self);
      //d_writer.print(");
      //d_writer.println(")) epv->" + vecEntry + ";");

      /*
       * Iterate over all methods in the EPV and set the method pointer.
       */
      for (Iterator j = methods.iterator(); j.hasNext(); ) {
        Method method = (Method) j.next();
        String mname  = method.getLongMethodName();
        d_writer.print(estring);
        d_writer.printAligned(IOR.getVectorEntry(mname), mwidth);
        d_writer.print(" = " + IOR.getCast(method, self));
        d_writer.println(" epv->" + IOR.getVectorEntry(mname) + ";");
      }

      d_writer.println();
      e++;
    }
  }

  private static String methodCall(Extendable ext, String var, String method,
                                   String args)
  {
    String result = "(*(" + var + "->" + IOR.getEPVVar(IOR.PUBLIC_EPV) + "->" 
                    + IOR.getVectorEntry(method) + "))(" + var;
    if (ext.isInterface()) {
      result += "->d_object";
    }
    result += args;
    result += ");";
    return result;
  }

  private void generateReplaceValue(Symbol symbol) {
    d_writer.println("if (result) {");
    d_writer.tab();
    if (symbol instanceof Extendable) {
      d_writer.println("if (value) {");
      d_writer.tab();
      d_writer.println(methodCall((Extendable)symbol, "value",
                                  "addRef", ""));
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println("if (*result) {");
      d_writer.tab();
      d_writer.println(methodCall((Extendable)symbol, "(*result)",
                                  "deleteRef", ""));
      d_writer.backTab();
      d_writer.println("}");
    }
    d_writer.println("*result = value;");
    d_writer.backTab();
    d_writer.println("}");
  }

  public static void generateExternalSignature(LanguageWriterForC lw,
                                               Symbol sym, String terminator)
  {
    final SymbolID id = sym.getSymbolID();
    lw.beginBlockComment(false);
    lw.println("This function returns a pointer to a static structure of");
    lw.println("pointers to function entry points.  Its purpose is to provide");
    lw.println("one-stop shopping for loading DLLs.");
    lw.endBlockComment(false);
    lw.println("const " + IOR.getExternalName(id) + "*");
    lw.println(IOR.getExternalFunc(id) + "(void)" + terminator);
  }

  private void generateExternalFunc(Symbol sym) {
    SymbolID id  = sym.getSymbolID();
    Class    cls = null;
    if (sym instanceof Class) {
      cls = (Class)sym;
/*
      if (IOR.supportInterceptors(cls) || IOR.supportAssertions(cls) ) {
        d_writer.println("const " + IOR.getExternalName(id));
      } else {
*/
        d_writer.println("static const " + IOR.getExternalName(id));
/*
      }
*/
    }
    d_writer.println("s_externalEntryPoints = {");
    d_writer.tab();
    if (sym instanceof Class) {
      if (!cls.isAbstract()){
        d_writer.println(IOR.getNewName(id) + ",");
      }
      if (cls.hasStaticMethod(true)) {
        d_writer.println(IOR.getStaticsName(id) + ",");
      }
      if (cls.getParentClass() != null ) {
          d_writer.println(IOR.getSymbolName(id) + "__super");
      }
    }
    d_writer.backTab();
    d_writer.println("};");
    d_writer.println();
    generateExternalSignature(d_writer, sym, "");
    d_writer.println("{");
    d_writer.tab();
    d_writer.println("return &s_externalEntryPoints;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }
}
