/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.system;

import org.apache.avalon.framework.activity.*;
import org.apache.avalon.framework.component.*;
import org.apache.avalon.framework.configuration.*;
import org.apache.avalon.framework.context.*;
import org.apache.avalon.framework.logger.*;

import org.apache.avalon.excalibur.collections.FixedSizeBuffer;
import org.apache.avalon.excalibur.command.*;
import org.apache.avalon.excalibur.system.handler.*;
import org.apache.avalon.excalibur.logger.LoggerManager;
import org.apache.avalon.excalibur.event.Queue;
import org.apache.avalon.excalibur.pool.PoolManager;
import org.apache.avalon.excalibur.util.ComponentStateValidator;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import java.lang.reflect.Constructor;

/**
 * The Container is an interface used to mark the Containers in your system.  It
 * exposes a protected getComponentManager() method so that the Container's
 * Manager can expose that to the instantiating class.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version CVS $Revision: 1.4 $ $Date: 2002/01/30 15:44:06 $
 */
public abstract class AbstractContainer
   extends AbstractLogEnabled
   implements Contextualizable, Composable, Configurable, Initializable, Disposable
{
    private final ComponentStateValidator  m_validator = new ComponentStateValidator( this );

    private Context                m_context;
    private ComponentManager       m_manager;
    private ComponentManager       m_childManager;
    private LoggerManager          m_logManager;
    private PoolManager            m_poolManager;
    private Queue                  m_commandQueue;
    private ClassLoader            m_classLoader;
    private RoleManager            m_roleManager;
    private Configuration          m_configuration;
    private List                   m_components;
    private Map                    m_configs;

    /**
     * Wrap this so that ComponentStateValidator functions properly.
     */
    public void enableLogging( Logger logger )
    {
        m_validator.checkLogEnabled();
        super.enableLogging( logger );
    }

    /**
     * Pull the manager items from the context so we can use them to set up the
     * system.
     */
    public void contextualize( Context containerContext )
        throws ContextException
    {
        m_validator.checkContextualized();

        m_context = containerContext;

        m_logManager = (LoggerManager) m_context.get(Container.LOGGER_MANAGER);
        m_poolManager = (PoolManager) m_context.get(Container.POOL_MANAGER);

        try
        {
            m_classLoader = (ClassLoader) m_context.get(Container.CONTEXT_CLASSLOADER);
        }
        catch ( ContextException ce )
        {
            m_classLoader = Thread.currentThread().getContextClassLoader();
        }

        try
        {
            m_commandQueue = (Queue) m_context.get(Container.COMMAND_QUEUE);
        }
        catch ( ContextException ce )
        {
            m_commandQueue = null;
            getLogger().warn("No Container.COMMAND_QUEUE is given, all management will be performed synchronously");
        }

        try
        {
            m_roleManager = (RoleManager) m_context.get(Container.ROLE_MANAGER);
        }
        catch ( ContextException ce )
        {
            m_roleManager = new ExcaliburRoleManager();
        }
    }

    /**
     * Process the configuration and set up the components and their mappings.
     * At this point, all components are prepared and all mappings are made.
     * However, nothing is initialized.
     */
    public void configure( Configuration configElement )
        throws ConfigurationException
    {
        m_validator.checkConfigured();
        m_configuration = configElement;
        Map managerMap = new HashMap();
        m_childManager = new ContainerComponentManager( managerMap, m_manager );

        Configuration[] components = configElement.getChildren();

        for ( int i = 0; i < components.length; i++ )
        {
            final String name = components[ i ].getName();
            String role;
            Class klass;
            Class handlerKlass;
            Configuration config = null;

            if ( name.equals( "component" ) )
            {
                config = components[ i ];
                role = config.getAttribute( "role" );

                try
                {
                    klass = m_classLoader.loadClass( config.getAttribute( "class" ) );
                    handlerKlass = m_classLoader.loadClass( config.getAttribute( "handler" ) );
                }
                catch ( Exception e )
                {
                    if ( getLogger().isDebugEnabled() )
                    {
                        getLogger().debug( "Component class '" + config.getAttribute( "class" ) +
                                           "' is not valid.", e );
                    }
                    continue;
                }

                assignHandler( getHandler( handlerKlass, klass, config ), config, managerMap );
            }
            else
            {
                handleConfiguration( components[ i ], managerMap );
            }
        }
    }

    /**
     * Handles when a configuration name is used that is not "component", so it
     * makes it easier to handle ComponentSelector hierarchies.  It is meant to
     * either return a ComponentHandler or a ComponentSelector
     */
    protected void handleConfiguration( final Configuration configItem,
                                        final Map managerMap )
        throws ConfigurationException
    {
        DefaultConfiguration temp = new DefaultConfiguration( "component", "AbstractContainer-rewrite" );
        Class klass = m_roleManager.getClassForName( configItem.getName() );
        Class handlerKlass = m_roleManager.getHandlerClassForClass( klass );
        String role = m_roleManager.getRoleForClass( klass );

        temp.setAttribute( "role", role );
        temp.setAttribute( "class", klass.getName() );
        temp.setAttribute( "handler", handlerKlass.getName() );

        final String id = configItem.getAttribute( "id", null );
        if ( null != id )
        {
            temp.setAttribute( "id",  id );
        }

        Configuration[] children = configItem.getChildren();
        for ( int i = 0; i < children.length; i++ )
        {
            temp.addChild( children[ i ] );
        }

        temp.makeReadOnly();

        assignHandler( getHandler( klass, handlerKlass, temp ), temp, managerMap );
    }

    /**
     * Adds the ComponentHandler and Configuration to the system.
     */
    protected void assignHandler( ComponentHandler handler, Configuration config, Map managerMap )
        throws ConfigurationException
    {
        m_components.add( handler );
        m_configs.put( handler, config );
        String role = config.getAttribute( "role" );

        Object contents = managerMap.get( role );
        if ( null == contents )
        {
            managerMap.put( role, handler );
        }
        else
        {
            if ( contents instanceof ComponentHandler )
            {
                Map selectorMap = new HashMap( 3 );
                selectorMap.put( ( (Configuration) m_configs.get(contents) )
                                 .getAttribute( "id", "1" ), contents );
                selectorMap.put( config.getAttribute( "id", "2" ), contents );

                assignSelector( role, new ContainerComponentSelector( selectorMap ), managerMap );
            }
            else if ( contents instanceof ContainerComponentSelector )
            {
                ( (ContainerComponentSelector) contents )
                    .addComponentHandler( config.getAttribute( "id", null ),
                                          handler);
            }
        }
    }

    /**
     * Adds the ComponentHandler and Configuration to the system.
     */
    protected void assignSelector( String role, ComponentSelector selector, Map managerMap )
        throws ConfigurationException
    {
        managerMap.put( role, selector );
    }

    /**
     * Get a ComponentHandler with the standard <code>HANDLER_CONSTRUCTOR</code>
     * for the component class passed in.
     */
    protected ComponentHandler getHandler( Class handlerKlass,
                                           Class klass,
                                           Configuration configuration )
    {
        Constructor constructor;
        ComponentHandler handler = null;

        try
        {
            constructor = handlerKlass.getConstructor( ComponentHandler.HANDLER_CONSTRUCTOR );
            handler = (ComponentHandler) constructor.newInstance(new Object[] {
                klass, configuration, m_childManager, m_context
            });
        }
        catch ( Exception e )
        {
            if ( getLogger().isDebugEnabled() )
            {
                getLogger().debug( "Could not create the '" + handlerKlass.getName() +
                                   "' handler for the '" + klass.getName() +
                                   "' component.", e );
            }
        }

        return handler;
    }

    /**
     * Root ComponentManager.  The Container may choose to have it's ComponentManager
     * delegate to the root manager, or it may choose to be entirely self contained.
     */
    public void compose( ComponentManager manager )
        throws ComponentException
    {
        m_validator.checkComposed();
        m_manager = manager;
    }

    /**
     * Initializes all components so that the system is ready to be used.
     */
    public void initialize()
        throws Exception
    {
        m_validator.checkInitialized();

        Iterator i = m_components.iterator();
        FixedSizeBuffer buffer = new FixedSizeBuffer( m_components.size() );

        while ( i.hasNext() )
        {
            try
            {
                if ( null != m_commandQueue )
                {
                    m_commandQueue.enqueue(
                        new InitComponentHandlerCommand( (ComponentHandler) i.next(),
                                                         getLogger() )
                    );
                }
                else
                {
                    ( (ComponentHandler) i.next() ).initialize();
                }
            }
            catch ( Exception e )
            {
                getLogger().warn( "Could not initialize component", e );
                buffer.add( e );
            }
        }

        if ( buffer.size() > 0 )
        {
            StringBuffer message = new StringBuffer();

            while ( ! buffer.isEmpty() )
            {
                message.append( ( (Exception) buffer.remove() ).getMessage() );
            }

            throw new Exception( message.toString() );
        }
    }

    /**
     * Disposes of all components and frees resources that they consume.
     */
    public void dispose()
    {
        m_validator.checkDisposed();

        Iterator i = m_components.iterator();

        while ( i.hasNext() )
        {
            if ( null != m_commandQueue )
            {
                try
                {
                    m_commandQueue.enqueue(
                        new DisposeComponentHandlerCommand( (ComponentHandler) i.next(),
                                                            getLogger() )
                    );

                    i.remove();
                }
                catch ( Exception e )
                {
                    if ( getLogger().isWarnEnabled() )
                    {
                        getLogger().warn( "Could not dispose component", e );
                    }
                }
            }
            else
            {
                ( (ComponentHandler) i.next() ).dispose();
            }
        }
    }

    /**
     * This is the Default ComponentManager for the Container.  It provides
     * a very simple abstraction, and makes it easy for the Container to manage
     * the references.
     */
    protected final static class ContainerComponentManager
        implements ComponentManager
    {
        private final Map m_components;
        private final Map m_used;
        private final ComponentManager m_parent;

        /**
         * This constructor is for a ContainerComponentManager with no parent
         * ComponentManager
         */
        protected ContainerComponentManager( Map componentMap )
        {
            this( componentMap, null );
        }

        /**
         * This constructor is for a ContainerComponentManager with a parent
         * ComponentManager
         */
        protected ContainerComponentManager( Map componentMap, ComponentManager parent )
        {
            m_parent = null;
            m_components = componentMap;
            m_used = new HashMap( m_components.size() );
        }

        public Component lookup( String role )
            throws ComponentException
        {
            Object temp = m_components.get( role );

            if ( temp instanceof ComponentSelector )
            {
                return (ComponentSelector) temp;
            }

            if ( ! ( temp instanceof ComponentHandler ) )
            {
                throw new ComponentException( "Invalid entry in component manager: " + temp );
            }

            ComponentHandler handler = (ComponentHandler) temp;

            if ( null == handler )
            {
                if ( null != m_parent )
                {
                    return m_parent.lookup( role );
                }
                else
                {
                    throw new ComponentException( "The role does not exist in the ComponentManager" );
                }
            }

            final Component component;

            try
            {
                if ( ! handler.isInitialized() )
                {
                    handler.initialize();
                }

                component = handler.get();
            }
            catch ( Exception e )
            {
                throw new ComponentException( "Could not return a reference to the Component", e );
            }

            synchronized ( m_used )
            {
                m_used.put( component, handler );
            }

            return component;
        }

        public boolean hasComponent( String role )
        {
            final boolean hasComponent = m_components.containsKey( role );

            if ( ! hasComponent && null != m_parent )
            {
                return m_parent.hasComponent( role );
            }

            return hasComponent;
        }

        public void release( Component component )
        {
            final ComponentHandler handler;

            synchronized ( m_used )
            {
                handler = (ComponentHandler) m_used.remove( component );
            }

            if ( null == handler && null != m_parent )
            {
                m_parent.release( component );
                return;
            }

            handler.put( component );
        }
    }

    /**
     * This is the Default ComponentSelector for the Container.  It provides
     * a very simple abstraction, and makes it easy for the Container to manage
     * the references.
     */
    protected final static class ContainerComponentSelector
        implements ComponentSelector
    {
        private final Map m_components;
        private final Map m_used;

        protected ContainerComponentSelector( Map selectorMap )
        {
            m_components = selectorMap;
            m_used = new HashMap( m_components.size() );
        }

        public Component select( Object hint )
            throws ComponentException
        {
            ComponentHandler handler = (ComponentHandler) m_components.get( hint );

            if ( null == handler )
            {
                throw new ComponentException( "The role does not exist in the ComponentSelector" );
            }

            final Component component;

            try
            {
                if ( ! handler.isInitialized() )
                {
                    handler.initialize();
                }

                component = handler.get();
            }
            catch ( Exception e )
            {
                throw new ComponentException( "Could not return a reference to the Component", e );
            }

            synchronized ( m_used )
            {
                m_used.put( component, handler );
            }

            return component;
        }

        public boolean hasComponent( Object hint )
        {
            return m_components.containsKey( hint );
        }

        public void release( Component component )
        {
            final ComponentHandler handler;

            synchronized ( m_used )
            {
                handler = (ComponentHandler) m_used.remove( component );
            }

            handler.put( component );
        }

        protected void addComponentHandler( Object hint, ComponentHandler handler )
        {
            if ( null == hint )
            {
                hint = new Integer( m_components.size() ).toString();
            }

            m_components.put( hint, handler );
        }
    }

    /**
     * This is the command class to initialize a ComponentHandler
     */
    private final static class InitComponentHandlerCommand implements Command
    {
        private final ComponentHandler m_handler;
        private final Logger           m_logger;

        protected InitComponentHandlerCommand( ComponentHandler handler, Logger logger )
        {
            m_handler = handler;
            m_logger = logger;
        }

        public void execute()
            throws Exception
        {
            try
            {
                if ( ! m_handler.isInitialized() )
                {
                    m_handler.initialize();
                }
            }
            catch ( Exception e )
            {
                if ( m_logger.isErrorEnabled() )
                {
                    m_logger.error( "Could not initialize ComponentHandler", e );
                }

                throw e;
            }
        }
    }

    /**
     * This is the command class to dispose a ComponentHandler
     */
    private final static class DisposeComponentHandlerCommand implements Command
    {
        private final ComponentHandler m_handler;
        private final Logger           m_logger;

        protected DisposeComponentHandlerCommand( ComponentHandler handler, Logger logger )
        {
            m_handler = handler;
            m_logger = logger;
        }

        public void execute()
        {
            m_handler.dispose();
        }
    }
}

