/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.profile;

/**
 * The Profilable interface is to mark objects that can be sampled by a Profiler.
 * The interface provides a method to initialize the profiler, plus two methods
 * to provide an optimization cue for the object (when it is safe not to track
 * events).
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 */
public interface Profilable
{
    /**
     * Empty Profilable array for use in hierarchical Profilable systems.
     */
    Profilable[] EMPTY_PROFILABLE_ARRAY = new Profilable[] {};

    /**
     * Obtain a reference to all the ProfilePoints that the Profilable
     * object wishes to expose.  All sampling is done directly through
     * the ProfilePoints as opposed to the Profilable interface.
     */
    ProfilePoint[] getProfilePoints();

    /**
     * The Profiler will call this method when it begins taking samples.
     * This is an optimization cue to the Profilable class.  It does take
     * resources to hold ProfilePoints and update them.  A class may keep
     * a <code>boolean</code> to flag whether the ProfilePoints are to be
     * maintained.
     */
    void startProfiling();

    /**
     * The Profiler will call this method when it no longer is interested
     * in taking samples.  It is an optimization cue to the Profilable
     * class so that it can release resources and stop maintaining the
     * ProfilePoints.
     */
    void stopProfiling();

    /**
     * The Profiler will recursively call this method until it receives an
     * EMPTY_PROFILABLE_ARRAY.  This method should never return null.
     */
    Profilable[] getChildProfilables();
}

