/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.datasource;

import java.sql.Connection;
import java.sql.SQLException;

import org.apache.avalon.excalibur.datasource.DataSourceComponent;
import org.apache.avalon.excalibur.datasource.JdbcConnectionFactory;
import org.apache.avalon.excalibur.datasource.NoAvailableConnectionException;
import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.logger.AbstractLogEnabled;

/**
 * The ResourceLimiting implementation for DataSources in Avalon.  This uses the
 * normal <code>java.sql.Connection</code> object and
 * <code>java.sql.DriverManager</code>.  The Configuration is like this:
 *
 * <pre>
 *   &lt;jdbc&gt;
 *     &lt;pool-controller max="<i>10</i>" blocking="<i>true</i>"
 *       timeout="<i>-1</i>" trim-interval="<i>60000</i>"
 *       connection-class="<i>my.overrided.ConnectionClass</i>"&gt;
 *       &lt;keep-alive disable="false"&gt;select 1&lt;/keep-alive&gt;
 *     &lt;/pool-controller&gt;
 *     &lt;driver&gt;<i>com.database.jdbc.JdbcDriver</i>&lt;/driver&gt;
 *     &lt;dburl&gt;<i>jdbc:driver://host/mydb</i>&lt;/dburl&gt;
 *     &lt;user&gt;<i>username</i>&lt;/user&gt;
 *     &lt;password&gt;<i>password</i>&lt;/password&gt;
 *   &lt;/jdbc&gt;
 * </pre>
 *
 * With the following roles declaration:
 *
 * <pre>
 *   &lt;role name="org.apache.avalon.excalibur.datasource.DataSourceComponentSelector"
 *     shorthand="datasources"
 *     default-class="org.apache.avalon.excalibur.component.ExcaliburComponentSelector"&gt;
 *     &lt;hint shorthand="jdbc"
 *       class="org.apache.avalon.excalibur.datasource.ResourceLimitingJdbcDataSource"/&gt;
 *   &lt;/role&gt;
 * </pre>
 *
 * @author <a href="mailto:leif@silveregg.co.jp">Leif Mortenson</a>
 * @version CVS $Revision: 1.3 $ $Date: 2002/01/26 16:58:26 $
 * @since 4.1
 */
public class ResourceLimitingJdbcDataSource 
    extends AbstractLogEnabled 
    implements DataSourceComponent, Disposable
{
    private boolean m_configured;
    private boolean m_disposed;
    protected ResourceLimitingJdbcConnectionPool m_pool;
    
    /*---------------------------------------------------------------
     * Constructors
     *-------------------------------------------------------------*/
    public ResourceLimitingJdbcDataSource() {}

    /*---------------------------------------------------------------
     * DataSourceComponent Methods
     *-------------------------------------------------------------*/
    /**
     * Gets the Connection to the database
     *
     * @throws NoValidConnectionException when there is no valid Connection wrapper
     *         available in the classloader.
     *
     * @throws NoAvailableConnectionException when there are no more available
     *         Connections in the pool.
     */
    public Connection getConnection()
        throws SQLException
    {
        if ( !m_configured ) throw new IllegalStateException( "Not Configured" );
        if ( m_disposed ) throw new IllegalStateException( "Already Disposed" );
        
        Connection connection;
        try
        {
            connection = (Connection)m_pool.get();
        }
        catch ( SQLException e )
        {
            if (getLogger().isWarnEnabled())
            {
                getLogger().warn("Could not return Connection", e);
            }
            
            throw e;
        }
        catch ( Exception e )
        {
            if ( getLogger().isWarnEnabled() )
            {
                getLogger().warn( "Could not return Connection", e );
            }
            
            throw new NoAvailableConnectionException( e.getMessage() );
        }
        
        return connection;
    }
    
    /*---------------------------------------------------------------
     * DataSourceComponent (Configurable) Methods
     *-------------------------------------------------------------*/
    /**
     * Pass the <code>Configuration</code> to the <code>Configurable</code>
     * class. This method must always be called after the constructor
     * and before any other method.
     *
     * @param configuration the class configurations.
     */
    public void configure(Configuration configuration) throws ConfigurationException {
        if (m_configured) throw new IllegalStateException("Already Configured");
        
        final String  driver   = configuration.getChild( "driver"   ).getValue( "" );
        final String  dburl    = configuration.getChild( "dburl"    ).getValue( null );
        final String  user     = configuration.getChild( "user"     ).getValue( null );
        final String  passwd   = configuration.getChild( "password" ).getValue( null );
        
        final Configuration controller = configuration.getChild( "pool-controller" );
        String keepAlive = controller.getChild( "keep-alive" ).getValue( "SELECT 1" );
        final boolean disableKeepAlive = controller.getChild( "keep-alive" ).getAttributeAsBoolean( "disable", false );
        
        final int     max          = controller.getAttributeAsInteger( "max", 3 );
        final boolean blocking     = controller.getAttributeAsBoolean( "blocking", true );
        final long    timeout      = controller.getAttributeAsLong   ( "timeout", -1 );
        final long    trimInterval = controller.getAttributeAsLong   ( "trim-interval", 60000 );
        final boolean oradb        = controller.getAttributeAsBoolean( "oradb", false );
        
        final boolean autoCommit = configuration.getChild("auto-commit").getValueAsBoolean(true);
        // Get the JdbcConnection class.  The factory will resolve one if null.
        final String connectionClass = controller.getAttribute( "connection-class", null ); 

        final int l_max;

        // If driver is specified....
        if ( ! "".equals(driver) )
        {
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Loading new driver: " + driver);
            }

            try
            {
                Class.forName( driver, true, Thread.currentThread().getContextClassLoader() );
            }
            catch (ClassNotFoundException cnfe)
            {
                if (getLogger().isWarnEnabled())
                {
                    getLogger().warn( "Could not load driver: " + driver, cnfe );
                }
            }
        }

        // Validate the max pool size values.
        if( max < 1 )
        {
            if (getLogger().isWarnEnabled())
            {
                getLogger().warn( "Maximum number of connections specified must be at least 1." );
            }

            l_max = 1;
        }
        else
        {
            l_max = max;
        }

        // If the keepAlive disable attribute was set, then set the keepAlive query to null, disabling it.
        if (disableKeepAlive)
        {
            keepAlive = null;
        }
        
        // If the oradb attribute was set, then override the keepAlive query.
        // This will override any specified keepalive value even if disabled.
        //  (Deprecated, but keep this for backwards-compatability)
        if (oradb)
        {
            keepAlive = "SELECT 1 FROM DUAL";

            if (getLogger().isWarnEnabled())
            {
                getLogger().warn("The oradb attribute is deprecated, please use the" +
                                 "keep-alive element instead.");
            }
        }

        
        final JdbcConnectionFactory factory = new JdbcConnectionFactory
            ( dburl, user, passwd, autoCommit, keepAlive, connectionClass );

        factory.enableLogging( getLogger() );

        try
        {
            m_pool = new ResourceLimitingJdbcConnectionPool
                (factory, l_max, true, blocking, timeout, trimInterval, autoCommit );
            m_pool.enableLogging( getLogger() );
        }
        catch (Exception e)
        {
            if (getLogger().isDebugEnabled())
            {
                getLogger().debug("Error configuring ResourceLimitingJdbcDataSource", e);
            }

            throw new ConfigurationException("Error configuring ResourceLimitingJdbcDataSource", e);
        }
        
        m_configured = true;
    }
    
    /*---------------------------------------------------------------
     * Disposable Methods
     *-------------------------------------------------------------*/
    /**
     * The dispose operation is called at the end of a components lifecycle.
     * This method will be called after Startable.stop() method (if implemented
     * by component). Components use this method to release and destroy any
     * resources that the Component owns.
     */
    public void dispose() {
        m_disposed = true;
        m_pool.dispose();
        m_pool = null;
    }
}

