/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                   Author :  Paul Taylor                               */
/*                   Date   :  June 1995                                 */
/*-----------------------------------------------------------------------*/
/*                 Stream linked list class                              */
/*                                                                       */
/*=======================================================================*/

#include <stdlib.h>
#include <stdio.h>
#include <fstream.h>
#include "ling_class/EST_Stream.h"
#include "stream_io.h" // for HTK_UNITS_PER_SECOND macro

EST_Stream::EST_Stream(const EST_String &ps)
{
    init(ps);
}

EST_Stream::EST_Stream()
{
    default_vals();
}

EST_Stream::EST_Stream(const EST_Stream &s)
{
    default_vals();
    copy(s);
}

void EST_Stream::default_vals()
{
    p_stream_name = EST_String::Empty;
    p_name = EST_String::Empty;
    h = 0;
    t = 0;
    next_addr = 0;
}

void EST_Stream::copy(const EST_Stream &s)
{
    clear();
    EST_Stream_Item *ptr, item;
    p_stream_name = s.stream_name();
    p_name = s.name();

    for (ptr = s.head(); ptr != 0; ptr = next(ptr))
      append(*ptr);
}

EST_Stream &EST_Stream::operator=(const EST_Stream &s)
{
    copy(s);
    return *this;
}

void EST_Stream::init(const EST_String &st)
{
    default_vals();
    p_stream_name = st;
}

void EST_Stream::clear()
{
    EST_Stream_Item *ptr, *nptr;

    for (ptr = head(); ptr != 0; ptr = nptr)
    {
	nptr = next(ptr);
	delete ptr;
    }

    init(EST_String::Empty);
}

EST_Stream_Item *EST_Stream::item(int a)  const
{
    EST_Stream_Item *ptr;

    for (ptr = head(); ptr != 0; ptr = next(ptr))
	if (ptr->addr() == a)
	    return ptr;

    cerr << a << ": No such address in " << stream_name() << " stream\n";
    return 0;
}

EST_Stream_Item *EST_Stream::nth(int n)  const
{
    EST_Stream_Item *ptr;
    int i;

    for (i = 0, ptr = head(); ptr != 0; ptr = next(ptr), ++i)
    {
	if (i == n)
	    return ptr;
    }

    cerr << "Request item " << n << " out of range in EST_Stream\n";
    return 0;
}

int EST_Stream::length() const
{
    EST_Stream_Item *ptr;
    int n = 0;

    for (ptr = head(); ptr != 0; ptr = next(ptr))
	++n;
    return n;
}

// set name of stream and sets all stream_items to be this also
void EST_Stream::set_stream_name(const EST_String &s)
{
    p_stream_name = s;
    EST_Stream_Item *ptr;
    for (ptr = head(); ptr != 0; ptr = next(ptr))
	ptr->set_stream_name(s);
}

void EST_Stream::set_name(const EST_String &s)
{
    p_name = s;
}

bool EST_Stream::empty() const
{
    return (length() == 0) ? true : false;
}

EST_Stream_Item *EST_Stream::append_item(EST_Stream_Item *item)
{
    item->p = t;
    if (h == 0)
	h = item;
    else
	t->n = item;
    t = item;
    return item;
}

EST_Stream_Item *EST_Stream::prepend_item(EST_Stream_Item *item)
{
    item->n = h;
    if (h == 0)
	t = item;
    else
	h->p = item;
    h = item;
    return item;
}

EST_Stream_Item *EST_Stream::append()
{
    // Generates a new one and puts it on the end 
    EST_Stream_Item *tmp = new EST_Stream_Item;
    tmp->set_stream_name(p_stream_name);
    return append_item(tmp);
}

EST_Stream_Item *EST_Stream::append(const EST_Stream_Item &item)
{
    // Copies the given one and puts it at the end
    EST_Stream_Item *tmp = new EST_Stream_Item(item);
    return append_item(tmp);
}

EST_Stream_Item *EST_Stream::prepend(const EST_Stream_Item &item)
{
    EST_Stream_Item *tmp = new EST_Stream_Item(item);
    return prepend_item(tmp);
}

EST_Stream_Item *EST_Stream::prepend()
{
    EST_Stream_Item *tmp = new EST_Stream_Item;
    tmp->set_stream_name(p_stream_name);
    return prepend_item(tmp);
}

EST_Stream_Item *EST_Stream::insert_after(EST_Stream_Item *ptr, 
					  EST_Stream_Item &newitem)
{
    if (ptr == 0)
    {
	cerr << "EST_Stream: can't insert_after NULL" << endl;
	return 0;
    }
    EST_Stream_Item *tmp = new EST_Stream_Item(newitem);

    tmp->p = ptr;
    tmp->n = ptr->n;
    if (ptr->n == 0)
	t = tmp;
    else
	ptr->n->p = tmp;
    ptr->n = tmp;
    return tmp;
}

EST_Stream_Item *EST_Stream::insert_before(EST_Stream_Item *ptr, 
					   EST_Stream_Item &newitem)
{
    if (ptr == 0)
    {
	cerr << "EST_Stream: can't insert_before NULL" << endl;
	return 0;
    }
    EST_Stream_Item *tmp = new EST_Stream_Item(newitem);

    tmp->p = ptr->p;
    tmp->n = ptr;
    if (ptr->p == 0)
	h = tmp;
    else
	ptr->p->n = tmp;
    ptr->p = tmp;
    return tmp;
}

EST_Stream_Item *EST_Stream::remove(EST_Stream_Item *item)
{   // removes item from list, returning a pointer to the previous item
    EST_Stream_Item *prev_item, *next_item;
    
    prev_item = item->p;
    next_item = item->n;
    
    if (next_item == 0)
	t = prev_item;
    else
	next_item->p = prev_item;
    
    if (prev_item == 0)
	h = next_item;
    else
	prev_item->n = next_item;
    
    // Zero it so it doesn't appear to be in an EST_Stream
    item->p = 0;
    item->n = 0;
    item->set_stream_name("");
    delete item;
    return prev_item;
}

void EST_Stream::remove(int a)
{
    remove(item(a));
}

EST_read_status EST_Stream::load(const EST_String &filename)
{
    EST_Stream tmp;
    EST_read_status r_val;
    EST_String myname;

    tmp.set_stream_name(p_stream_name);
    tmp.f = f;
    if ((r_val = load_esps_label(filename, tmp))
	!= wrong_format)
	*this = tmp;

    return r_val;
}

// Many label formats do not have distinctive headers, therefore we
// have to explicitly say what the file type is.

EST_read_status EST_Stream::load_type(const EST_String &filename,const EST_String &type, int sample)
{
    EST_Stream tmp;
    EST_read_status r_val;

    tmp.set_stream_name(p_stream_name);
    if ((type == "htk") || (sample == 0))
	sample = 10000000;

    if ((type == "ascii") || (type == "timit") || (type == "htk"))
    {
	if ((r_val = load_sample_label(filename, tmp, sample)) != wrong_format)
	{	
	    *this = tmp;
	    return r_val;
	}
    }
    else if ((type == "words"))
    {
	if ((r_val = load_words_label(filename, tmp)) != wrong_format)
	{	
	    *this = tmp;
	    return r_val;
	}
    }
    else if ((type == "ogi"))
    {
	if ((r_val = load_ogi_label(filename, tmp)) != wrong_format)
	{	
	    *this = tmp;
	    return r_val;
	}
    }
    else if ((type == "esps"))
    {
    if ((r_val = load_esps_label(filename, tmp)) 
	!= wrong_format)
	{	
	    *this = tmp;
	    return r_val;
	}
    }
    else 
    {
	cerr << "Unknown label type: " << type << endl;
	return read_format_error;
    }

    return read_ok;
}

EST_String options_stream_filetypes(void)
{
    // valid stream file types 
    return "esps htk spn names";
}

EST_write_status EST_Stream::save(const EST_String &filename,
				  const EST_String &type, 
				  const EST_String &features) const
{
    if  (type == "")
	return save_label_esps(filename, *this, features);
    if  (type == "esps")
	return save_label_esps(filename, *this, features);
    if  (type == "htk")
	return save_label_htk(filename, *this);
    if  (type == "spn")
	return save_label_spn(filename, *this);
    if  (type == "names")
	return save_label_names(filename, *this, features);

    cerr << "Unknown label output file type " << type << endl;
    return write_error;
}

ostream& operator << (ostream &st, const EST_Stream &a)
{
    EST_Stream_Item *ptr;
    
//    st << a.stream_name() << endl;
    
    for (ptr = a.head(); ptr != 0; ptr = next(ptr))
	st << *ptr << endl;
    
    return st;
}


