 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Fri Apr  4 1997                              */
 /************************************************************************/
 /*                                                                      */
 /* Simple Hash classes.                                                 */
 /*                                                                      */
 /************************************************************************/


#include "EST_THash.h"

template<class K, class V>
EST_THash<K,V>::EST_THash(int size,  unsigned int (*hash_function)(const K &key, unsigned int size))
{
p_num_entries =0;

p_num_buckets = size;

p_buckets = new EST_Hash_Pair<K,V> *[size];
for(int i=0; i<size;i++)
  p_buckets[i] = NULL;

p_hash_function = hash_function;
}

template<class K, class V>
void EST_THash<K,V>::clear(void)
{
  for(unsigned int i=0; i<p_num_buckets;i++)
    {
      EST_Hash_Pair<K,V> *p, *n;
      for(p=p_buckets[i]; p != NULL; p=n)
	{
	  n = p->next;
	  delete p;
	}
      p_buckets[i]=NULL;
    }
  p_num_entries=0;
}

template<class K, class V>
EST_THash<K,V>::~EST_THash(void)
{
  if (p_buckets)
    {
    clear();
    delete[] p_buckets;
    }
}

template<class K, class V>
int EST_THash<K,V>::present(const K &key) const
{
  unsigned int b;
  if (p_hash_function)
    b = (*p_hash_function)(key, p_num_buckets);
  else
    b = DefaultHashFunction((void *)&key, sizeof(key), p_num_buckets);

  EST_Hash_Pair<K,V> *p;

  for(p=p_buckets[b]; p!=NULL; p=p->next)
    if (p->key == key)
      return TRUE;
  
return FALSE;
}

template<class K, class V>
V &EST_THash<K,V>::val(const K &key, int &found) const
{
  unsigned int b;
  if (p_hash_function)
    b = (*p_hash_function)(key, p_num_buckets);
  else
    b = DefaultHashFunction((void *)&key, sizeof(key), p_num_buckets);

  EST_Hash_Pair<K,V> *p;

  for(p=p_buckets[b]; p!=NULL; p=p->next)
    if (p->key == key)
      {
	found=1;
	return p->value;
      }
  
found=0;
return Dummy_Value;
}

template<class K, class V>
void EST_THash<K,V>::map(void (*func)(K&, V&))
{
  for(unsigned int i=0; i<p_num_buckets; i++)
    {
      EST_Hash_Pair<K,V> *p;

      for(p=p_buckets[i]; p!=NULL; p=p->next)
	(*func)(p->key, p->value);
    }
  
}

template<class K, class V>
int EST_THash<K,V>::add_item(const K &key, const V &value, int no_search)
{
  unsigned int b;
  if (p_hash_function)
    b = (*p_hash_function)(key, p_num_buckets);
  else
    b = DefaultHashFunction((void *)&key, sizeof(key), p_num_buckets);

  EST_Hash_Pair<K,V> *p;

  if (!no_search)
    for(p=p_buckets[b]; p!=NULL; p=p->next)
      if (p->key == key)
	{
	  p->value = value;
	  return FALSE;
	}
      
  p = new EST_Hash_Pair<K,V>;
  p->key = key;
  p->value = value;
  p->next = p_buckets[b];
  p_buckets[b] = p;
  return TRUE;
}

template<class K, class V>
int EST_THash<K,V>::remove_item(const K &rkey, int quiet)
{
  unsigned int b;
  if (p_hash_function)
    b = (*p_hash_function)(rkey, p_num_buckets);
  else
    b = DefaultHashFunction((void *)&rkey, sizeof(rkey), p_num_buckets);

  EST_Hash_Pair<K,V> **p;

  for (p = &(p_buckets[b]); *p!=NULL; p=&((*p)->next))
    if ( (*p)->key == rkey )
      {
	EST_Hash_Pair<K,V> *n = (*p)->next;
	delete *p;
	*p = n;
	return 0;
      }
      
  if (!quiet)
    cerr << "THash: no item labelled \"" << rkey << "\"" << endl;
  return -1;
}


template<class K, class V>
void EST_THash<K,V>::dump(ostream &stream, int all)
{
  for(unsigned int i=0; i<p_num_buckets; i++)
    if (all || p_buckets[i])
      {
	stream << i << ": ";
	EST_Hash_Pair<K,V> *p;
	for(p=p_buckets[i]; p!=NULL; p=p->next)
	  stream << "[" << p->key << "],(" << p->value << ") ";
	stream << "\n";
      }
}






