
/*
 * DREADERD/MAIN.C - diablo newsreader
 *
 *	This is the startup for the dreaderd daemon.
 *
 *	The diablo newsreader requires one or remote diablo news boxes to 
 *	obtain articles from, but keeps track of article assignments to groups
 *	and overview information locally.
 *
 *	Operationally, dreaderd runs as a standalone multi-forking, 
 *	multi-threaded daemon.  It forks X resolvers and Y readers.  Each
 *	reader can support N connections (NxY total connections) and maintains
 *	its own links to one or more news servers (diablo, INN, whatever). 
 *	Overview information is utilized and articles are accessed by 
 *	message-id.
 *
 *	The diablo newsreader is also able to accept header-only feeds from
 *	one or more diablo servers via 'mode headfeed'.  Processing of these
 *	feeds is not multi-threaded, only processing of newsreader functions is
 *	multithreaded.  The diablo newsreader does not operate with a dhistory
 *	file.  If taking a feed from several sources you must also run the
 *	diablo server on the same box and use that to feed the dreaderd system,
 *	though the diablo server need not store any articles itself.
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

#if NEED_TERMIOS
#include <sys/termios.h>
#endif

Prototype char *ReportedHostName;
Prototype char *XRefHost;
Prototype MemPool *SysBufPool;
Prototype int CacheMode;
Prototype void RTClearStatus(int index, int count);
Prototype void RTUpdateStatus(int slot, const char *ctl, ...);

Prototype int NumReaderForks;
Prototype char *RTStatus;
Prototype char *PathHost;

pid_t ForkPipe(int *pfd, void (*task)(int fd), const char *description);
void HandleDnsThread(ForkDesc *desc);
void ValidateTcpBufferSize(int *psize);
void SetMyHostName(void);

char *PathHost = NULL;
char *BindHost = NULL;
char *ReportedHostName = NULL;
char *XRefHost = NULL;
char *RTStatus;
int BindPort = 119;
int TxBufSize = 4096;
int RxBufSize = 2048;
int STxBufSize = 4096;
int SRxBufSize = 16384;
int MaxFeedForks = 10;
int MaxDnsForks = 5;
int MaxSDnsForks = 0;		/* future				*/
int MaxConnects;
int CacheMode = 1;		/* turned on				*/
int ConnectCount;
int FailCount;

int NumFeedForks;
int NumReaderForks;
int NumDnsForks;
int NumSDnsForks;
int NumPending;			/* pending connections	*/
int NumActive;			/* active connections	*/

MemPool *SysBufPool;

int
main(int ac, char **av)
{
    int i;
    int lfd = -1;

    LoadDiabloConfig();

    for (i = 1; i < ac; ++i) {
	char *ptr = av[i];
	if (*ptr != '-') {
	    fprintf(stderr, "Expected option: %s\n", ptr);
	    exit(1);
	}
	ptr += 2;
	switch(ptr[-1]) {
	case 'd':
	    DebugOpt = (*ptr) ? strtol(ptr, NULL, 0) : 1;
	    break;
	case 'p':
	    PathHost = (*ptr) ? ptr : av[++i];
	    if (strcmp(PathHost, "0") == 0)
		PathHost = "";
	    break;
	case 's':
	    ptr = (*ptr) ? ptr : av[++i];
	    SetStatusLine(ptr, strlen(ptr));
	    break;
	case 'x':
	    XRefHost = (*ptr) ? ptr : av[++i];
	    break;
	case 'T':
	    TxBufSize = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'R':
	    RxBufSize = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'S':
	    switch(*ptr) {
	    case 'T':
		++ptr;
		STxBufSize = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
		break;
	    case 0:
	    case '0':
	    case '1':
	    case '2':
	    case '3':
	    case '4':
	    case '5':
	    case '6':
	    case '7':
	    case '8':
	    case '9':
		STxBufSize = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    case 'R':
		++ptr;
		SRxBufSize = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
		break;
	    default:
		fatal("bad option: -S (must be R#, T#, or #)");
		break;
	    }
	    break;
	case 'F':
	    MaxFeedForks = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'D':
	    MaxDnsForks = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'M':
	    DiabloReaderForks = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'N':
	    DiabloReaderThreads = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'h':
	    ReportedHostName = (*ptr) ? ptr : av[++i];
	    break;
	case 'P':
	    if (*ptr == 0)
		ptr = av[++i];
	    if (strrchr(ptr, ':')) {
		BindPort = strtol(strrchr(ptr, ':') + 1, NULL, 0);
		BindHost = strdup(ptr);
		*strrchr(BindHost, ':') = 0;
	    } else {
		BindPort = strtol(ptr, NULL, 0);
	    }
	    break;
	case 'r':
	    RTStatus = (*ptr) ? ptr : av[++i];
	    break;
	default:
	    fprintf(stderr, "unknown option: %s\n", ptr - 2);
	    exit(1);
	}
    }
    ValidateTcpBufferSize(&RxBufSize);
    ValidateTcpBufferSize(&TxBufSize);
    ValidateTcpBufferSize(&SRxBufSize);
    ValidateTcpBufferSize(&STxBufSize);

    MaxConnects = DiabloReaderThreads * DiabloReaderForks;

    /*
     * To stdout, not an error, no arguments == user requesting 
     * options list.  We can do this because the -p option is mandatory.
     */

    if (PathHost == NULL) {
	printf(
	    "dreaderd [-d#] -p pathhost [-s \"...\"] [-T txbufsize]"
	    " [-R rxbufsize] [-F maxfeedforks] [-M maxreaderforks]"
	    " [-N maxthreads/reader] [-h reportedhostname]"
	    " [-P [bindhost]:port]\n"
	);
	exit(0);
    }

    /*
     * Open realtime status file
     */

    if (RTStatus == NULL) {
	RTStatus = malloc(strlen(NewsHome) + 32);
	sprintf(RTStatus, "%s/log/dreaderd.status", NewsHome);
    }

    /*
     * Set host name
     */

    if (ReportedHostName == NULL)
	SetMyHostName();

    /*
     * Setup memory zone for localized data so the children do not
     * inherit it.
     */

    /*
     * bind to our socket prior to changing uid/gid
     */

    {
	struct sockaddr_in lsin;

	lfd = socket(AF_INET, SOCK_STREAM, 0);

	bzero(&lsin, sizeof(lsin));
	lsin.sin_addr.s_addr = INADDR_ANY;
	lsin.sin_port = htons(BindPort);

	if (BindHost != NULL) {
	    struct hostent *he;

	    if ((he = gethostbyname(BindHost)) != NULL) {
		lsin.sin_addr = *(struct in_addr *)he->h_addr;
	    } else {
		lsin.sin_addr.s_addr = inet_addr(BindHost);
		if (lsin.sin_addr.s_addr == INADDR_NONE) {
		    perror("gethostbyname(BindHost)");
		    exit(1);
		}
	    }
	}
	{
	    int on = 1;

	    setsockopt(lfd, SOL_SOCKET, SO_REUSEADDR, (void *)&on, sizeof(on));
	    setsockopt(lfd, SOL_SOCKET, SO_KEEPALIVE, (void *)&on, sizeof(on));
	}
	setsockopt(lfd, SOL_SOCKET, SO_SNDBUF, (void *)&TxBufSize, sizeof(int));
	setsockopt(lfd, SOL_SOCKET, SO_RCVBUF, (void *)&RxBufSize, sizeof(int));

	if (bind(lfd, (struct sockaddr *)&lsin, sizeof(lsin)) < 0) {
	    perror("bind");
	    exit(1);
	}
	listen(lfd, 256);
	AddThread("acceptor", lfd, -1, THREAD_LISTEN, -1);
    }

    /*
     * Fork if not in debug mode and detach from terminal
     */

    if (DebugOpt == 0) {
	pid_t pid;
	int fd;

	fflush(stdout);
	fflush(stderr);

	if ((pid = fork()) != 0) {
	    if (pid < 0) {
		perror("fork");
		exit(1);
	    }
	    exit(0);
	}
	/*
	 * child continues, close stdin, stdout, stderr, detach tty,
	 * detach process group.
	 */
	{
	    int fd = open("/dev/null", O_RDWR);
	    if (fd != 0)
		dup2(fd, 0);
	    if (fd != 1)
		dup2(fd, 1);
	    if (fd != 2)
		dup2(fd, 2);
	    if (fd > 2)
		close(fd);
	}
	if ((fd = open("/dev/tty", O_RDWR)) >= 0) {
	    ioctl(fd, TIOCNOTTY, 0);
	    close(fd);
	}
#if USE_SYSV_SETPGRP
        setpgrp();
#else
	setpgrp(0, 0);
#endif
    }

    /*
     * Ignore SIGPIPE, in case a write() fails we do not
     * want to segfault.
     */
    rsignal(SIGPIPE, SIG_IGN);

    /*
     * change my uid/gid
     */

    {
        struct passwd *pw = getpwnam("news");
        struct group *gr = getgrnam("news");
        gid_t gid;

        if (pw == NULL) {
            perror("getpwnam('news')");
            exit(1);
        }
        if (gr == NULL) {
            perror("getgrnam('news')");
            exit(1);
        }
        gid = gr->gr_gid;
        setgroups(1, &gid);
        setgid(gr->gr_gid);
        setuid(pw->pw_uid);
    }

    /*
     * RTStatus file initial open
     */

    RTStatusOpen(RTStatus, 0, 1);

    /*
     * open syslog
     */

    openlog("dreaderd", (DebugOpt ? LOG_PERROR : 0) | LOG_NDELAY | LOG_PID, LOG_NEWS);

    /*
     * accept()/pipe-processing loop, fork child processes as required
     * to maintain requested parameters (initial fork is done as part of
     * the nominal re-fork code when a child dies).
     */

    for (;;) {
	/*
	 * handle required forks and child process cleanup
	 *
	 * (A) reader processes
	 */

	while (NumReaderForks < DiabloReaderForks) {
	    pid_t pid;
	    int fd;

	    pid = ForkPipe(&fd, ReaderTask, "reader");
	    if (pid < 0)
		break;
	    AddThread("reader", fd, pid, THREAD_READER, -1);
	    ++NumReaderForks;
	}

	/*
	 * (B) standard DNS processes
	 */

	while (NumDnsForks < MaxDnsForks) {
	    pid_t pid;
	    int fd;

	    if ((pid = ForkPipe(&fd, DnsTask, "dns")) < 0)
		break;
	    AddThread("dns", fd, pid, THREAD_DNS, -1);
	    ++NumDnsForks;
	}

#ifdef NOTDEF
	/*
	 * (C) server hostname resolver	(future)
	 */

	while (NumSDnsForks < MaxSDnsForks) {
	    pid_t pid;
	    int fd;

	    if ((pid = ForkPipe(&fd, DnsTask, "sdns")) < 0)
		break;
	    AddThread("sdns", fd, pid, THREAD_SDNS, -1);
	    ++NumSDnsForks;
	}
#endif

	/*
	 * (D) dead processes
	 */

	{
	    pid_t pid;

	    while ((pid = wait3(NULL, WNOHANG, NULL)) > 0) {
		ForkDesc *desc;

		if ((desc = FindThread(-1, pid)) != NULL) {
		    if (desc->d_Type == THREAD_DNS) {
			--NumDnsForks;
			if (desc->d_FdPend >= 0) {
			    logit(LOG_INFO, "pending descriptor from %s on dead DNS pid closed",
				inet_ntoa(desc->d_SaveRSin.sin_addr)
			    );
			    close(desc->d_FdPend);
			    desc->d_FdPend = -1;
			    --NumPending;
			    --NumActive;
			}
		    } else if (desc->d_Type == THREAD_READER) {
			--NumReaderForks;
			NumActive -= desc->d_Count;
		    } else if (desc->d_Type == THREAD_SDNS) {
			--NumSDnsForks;
		    } else {
			fatal("main loop, unknown thread type %d pid %d\n", desc->d_Type, (int)pid);
		    }
		    DelThread(desc);
		} else {
		    fatal("main loop, wait3() pid %d, pid is not known to me!", (int)pid);
		}
	    }
	}

	/*
	 * (F) select core
	 *
	 * select on descriptors (listen socket and pipe sockets only),
	 * process results
	 */
	{
	    struct timeval tv = { 10, 0 };
	    fd_set rfds = SFds;
	    int i;

	    stprintf("Connect=%d Failed=%d Dns=%d/%d Act=%d/%d", 
		ConnectCount, FailCount,
		NumPending, MaxDnsForks, 
		NumActive, MaxConnects
	    );
	    RTStatusUpdate(0, "Connect=%d Failed=%d Dns=%d/%d Act=%d/%d", 
		ConnectCount, FailCount,
		NumPending, MaxDnsForks, 
		NumActive, MaxConnects
	    );

	    select(MaxFds, &rfds, NULL, NULL, &tv);

	    for (i = 0; i < MaxFds; ++i) {
		if (FD_ISSET(i, &rfds)) {
		    ForkDesc *desc;

		    if ((desc = FindThread(i, -1)) != NULL) {
			switch(desc->d_Type) {
			case THREAD_LISTEN:
			    /*
			     * accept new descriptor, depend on socket
			     * buffer for write to not block if descriptor
			     * beyond what we can handle.
			     */
			    {
				struct sockaddr_in rsin;
				int rslen = sizeof(rsin);

				int fd = accept(lfd, (struct sockaddr *)&rsin, &rslen);
				if (fd >= MAXFDS) {
				    char *s = "500 Too many open descriptors, closing\r\n";
				    logit(LOG_CRIT, "fd beyond MAXFDS: %d, closing", fd);
				    write(fd, s, strlen(s));
				    close(fd);
				    fd = -1;
				    ++ConnectCount;
				    ++FailCount;
				}
				if (fd >= 0) {
				    /*
				     * queue descriptor information to DNS
				     * resolver and put descriptor on hold.
				     */
				    ForkDesc *best;
				    struct sockaddr_in lsin;
				    int lslen = sizeof(lsin);

				    getsockname(fd, (struct sockaddr *)&lsin, &lslen);

				    ++ConnectCount;

				    best = FindLeastUsedThread(THREAD_DNS, 1);

				    if (best != NULL) {
					DnsReq dreq;

					++NumActive;
					++NumPending;
					++best->d_Count;

					bzero(&dreq, sizeof(dreq));
					dreq.dr_RSin = rsin;
					dreq.dr_LSin = lsin;

					best->d_FdPend = fd;
					best->d_SaveRSin = rsin;
					write(best->d_Fd, &dreq, sizeof(dreq));
					/* 
					 * ignore return code.  If DNS process died, it
					 * will be cleaned up later.
					 */
				    } else {
					char *s = "500 Software error resolving DNS\r\n";
					logit(LOG_CRIT, "software error resolving DNS");
					write(fd, s, strlen(s));
					close(fd);
					fd = -1;
					++FailCount;
				    }
				}
			    }
			    break;
			case THREAD_DNS:
			    /*
			     * read query result from DNS handler, then
			     * issue appropriate action to descriptor.
			     */
			    --NumPending;
			    HandleDnsThread(desc);
			    break;
			case THREAD_READER:
			    /*
			     * reader process returns one byte for each
			     * closed connection.  Update our ref counts.
			     */
			    {
				char dummy[16];
				int r;

				r = read(desc->d_Fd, dummy, sizeof(dummy));
				if (r > 0) {		/* read N bytes */
				    NumActive -= r;
				    desc->d_Count -= r;
				}
			    }
			    break;
			default:
			    fatal("Unknown descriptor type %d", desc->d_Type);
			    break;
			}
		    } else {
			fatal("select returned unknown descriptor %d", i);
		    }
		}
	    }
	}

	/*
	 * Disable listen socket if we have too many connections pending or
	 * we are full up, enable listen socket otherwise.  NOTE: NumPending
	 * is included in the NumActive number.
	 */
	if (NumPending < NumDnsForks && 
	    NumActive < NumReaderForks * DiabloReaderThreads
	) {
	    FD_SET(lfd, &SFds);
	} else {
	    FD_CLR(lfd, &SFds);
	}
    }
    return(0);
}

pid_t
ForkPipe(int *pfd, void (*task)(int fd), const char *description)
{
    int fds[2];
    pid_t pid;

    *pfd = -1;

    /*
     * NOTE: must use socketpair in order to be able to pass descriptors.
     */

    if (socketpair(PF_UNIX, SOCK_STREAM, IPPROTO_IP, fds) < 0) {
	logit(LOG_ERR, "socketpair() call failed");
	return((pid_t)-1);
    }
    if (fds[0] > MAXFDS || fds[1] > MAXFDS) {
	logit(LOG_ERR, "pipe desc > MAXFDS: %d/%d/%d", fds[0], fds[1], MAXFDS);
	close(fds[0]);
	close(fds[1]);
	return((pid_t)-1);
    }

    *pfd = fds[0];

    fflush(stdout);
    fflush(stderr);

    closelog();
    pid = fork();
    openlog(description, (DebugOpt ? LOG_PERROR : 0) | LOG_NDELAY | LOG_PID, LOG_NEWS);
    stprintf("%s startup", description);

    if (pid == 0) {
	ResetThreads();
	close(*pfd);
	task(fds[1]);
	_exit(1);
    } else if (pid < 0) {
	logit(LOG_ERR, "fork() failed: %s", strerror(errno));
	close(*pfd);
	close(fds[1]);
	*pfd = -1;
    } else if (pid > 0) {
	close(fds[1]);
	if (DebugOpt)
	    printf("forked %s child process %d\n", description, (int)pid);
    }
    return(pid);
}


void
HandleDnsThread(ForkDesc *desc)
{
    DnsRes dres;
    ForkDesc *reader;

    if (read(desc->d_Fd, &dres, sizeof(dres)) != sizeof(dres)) {
	/*
	 * let the child process cleanup cleanup any operations in
	 * progress.
	 */
	logit(LOG_CRIT, "read error on DNS subprocess, killing");
	kill(desc->d_Pid, 9);
	printf("killing pid %d\n", desc->d_Pid);
	return;
    }
    --desc->d_Count;

    if (dres.dr_Code == 0) {
	const char *s = "500 Unauthorized connection rejected\r\n";
	logit(
	    LOG_INFO, 
	    "connection from %s rejected: %s",
	    inet_ntoa(desc->d_SaveRSin.sin_addr), 
	    dres.dr_Host
	);
	write(desc->d_FdPend, s, strlen(s));
	close(desc->d_FdPend);
	desc->d_FdPend = -1;
	++FailCount;
	return;
    }

    /*
     * This shouldn't happen, but if it does lets be nice about it.
     */

    if ((reader=FindLeastUsedThread(THREAD_READER, DiabloReaderThreads)) == NULL) {
	const char *s = "500 Server maxed out, please try again later\r\n";
	logit(
	    LOG_INFO, 
	    "connection from %s (%s) rejected", 
	    dres.dr_Host,
	    inet_ntoa(desc->d_SaveRSin.sin_addr)
	);
	write(desc->d_FdPend, s, strlen(s));
	close(desc->d_FdPend);
	desc->d_FdPend = -1;
	++FailCount;
	return;
    }

    /*
     * pass the descriptor to an appropriate subprocess
     */

    {
	struct msghdr msg;
	struct iovec iov;
	struct {
#if FDPASS_USES_CMSG
	    struct cmsghdr cmsg;
#endif
	    int fd;
	} cmsg;

	bzero(&msg, sizeof(msg));
	bzero(&cmsg, sizeof(cmsg));

	iov.iov_base = (void *)&dres;
	iov.iov_len = sizeof(dres);

	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;
	msg.msg_control = (caddr_t)&cmsg;
	msg.msg_controllen = sizeof(cmsg);
#if FDPASS_USES_CMSG
	msg.msg_flags = 0;
	cmsg.cmsg.cmsg_len = sizeof(cmsg);
	cmsg.cmsg.cmsg_level = SOL_SOCKET;
	cmsg.cmsg.cmsg_type = SCM_RIGHTS;
#endif
	cmsg.fd = desc->d_FdPend;

	if (DebugOpt)
	    printf("Pass file descriptor fd=%d\n", (int)cmsg.fd);

	errno = 0;

	if (sendmsg(reader->d_Fd, &msg, 0) < 0) {
	    const char *s = "500 Server error, fd pass failed\r\n";
	    logit(
		LOG_INFO, 
		"connect: error passing file descriptor: %s", 
		strerror(errno)
	    );
	    if (DebugOpt)
		printf("sendmsg() failed: %s\n", strerror(errno));
	    write(desc->d_FdPend, s, strlen(s));
	    close(desc->d_FdPend);
	    desc->d_FdPend = -1;
	    ++FailCount;
	    return;
	}
	close(desc->d_FdPend);
	desc->d_FdPend = -1;
	++reader->d_Count;

	logit(
	    LOG_INFO,
	    "connect from %s%s%s (%s)", 
	    (dres.dr_User[0] ? dres.dr_User : ""),
	    (dres.dr_User[0] ? "@" : ""),
	    dres.dr_Host,
	    inet_ntoa(desc->d_SaveRSin.sin_addr)
	);
    }
}

void
ValidateTcpBufferSize(int *psize)
{
    if (*psize < 512)
	*psize = 512;
    if (*psize > 256*1024)
	*psize = 256*1024;
}

void
SetMyHostName(void)
{
    char buf[256];

    buf[sizeof(buf)-1] = 0;
    gethostname(buf, sizeof(buf) - 1);
    ReportedHostName = malloc(strlen(buf) + 1);
    strcpy(ReportedHostName, buf);
}

