/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB

   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.

   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.	The License grants you the right to
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* This file defines all time functions */

#ifdef __GNUC__
#pragma implementation				// gcc: Class implementation
#endif

#include "mysql_priv.h"
#include <m_ctype.h>
#include <time.h>

/*
** Todo: Move month and days to language files
*/

static String month_names[] = { "January", "February", "March", "April",
			       "May", "June", "July", "August",
			       "September", "October", "November", "December" };
static String day_names[] = { "Monday", "Tuesday", "Wednesday",
			     "Thursday", "Friday", "Saturday" ,"Sunday" };

/*
** Get a array of numbers from a string object.
** The string object may have separators or not. Anything is allowed as a
** separator.  To allow no separators, start from the end from the string.
** All numbers, except the first one, must be 2 characters (or 1 + separator).
** This functions only works for count > 1
*/

static bool get_interval_info(const char *str,uint length,uint count,
			      long *values)
{
  ulong value;
  if (length < (count-1)*2+1 || !isdigit(str[length-1]))
    return 1;
  bool sign=0;
  if (str[0] == '-')
  {
    sign=1;
    str++;
    length--;
  }
  length--;					// For a better offset
  while (--count)
  {
    if (length < 2)
      return 1;
    value= (uint) (str[length--] - '0');
    if (isdigit(str[length]))
    {
      value+=(uint) (str[length--] - '0')*10;
      if (length > 1 && !isdigit(str[length]))
	length--;
    }
    else
      length--;
    if (!isdigit(str[length]))
      return 1;					// Too short or wrong format
    values[count]=value;
  }

  /* Get first value, that may be big */
  const char *end=str+length;
  value=0;
  do
  {
    if (!isdigit(*str))
      return 1;
    value= value*10+ (uint) (*str - '0');
  } while (str++ != end);
  values[0]= sign ? -(long) value : (long) value;
  return 0;
}


longlong Item_func_period_add::val_int()
{
  ulong period=(ulong) args[0]->val_int();
  int months=(int) args[1]->val_int();

  if ((null_value=args[0]->null_value || args[1]->null_value) ||
      period == 0L)
    return 0; /* purecov: inspected */
  return (longlong)
    convert_month_to_period((uint) ((int) convert_period_to_month(period)+
				    months));
}


longlong Item_func_period_diff::val_int()
{
  ulong period1=(ulong) args[0]->val_int();
  ulong period2=(ulong) args[1]->val_int();

  if ((null_value=args[0]->null_value || args[1]->null_value))
    return 0; /* purecov: inspected */
  return (longlong) ((long) convert_period_to_month(period1)-
		     (long) convert_period_to_month(period2));
}



longlong Item_func_to_days::val_int()
{
  TIME ltime;
  String *str;
  null_value=0;
  if (!(str=args[0]->str(&value)))
  {
      null_value=1;				/* purecov: inspected */
      return 0;					/* purecov: inspected */
  }
  if (str_to_TIME(str->ptr(),str->length(),&ltime) == TIMESTAMP_NONE)
  {
    null_value=1;
    return(0);
  }
  return (longlong) calc_daynr(ltime.year,ltime.month,ltime.day);
}


longlong Item_func_dayofmonth::val_int()
{
  String *str=args[0]->str(&value);
  if ((null_value=args[0]->null_value))
    return 0;
  TIME l_time;
  if (str_to_TIME(str->ptr(),str->length(),&l_time) == TIMESTAMP_NONE)
  {
    null_value=1;
    return(0);
  }
  return (longlong) l_time.day;
}


longlong Item_func_dayofyear::val_int()
{
  String *str=args[0]->str(&value);
  if ((null_value=args[0]->null_value))
    return 0;
  TIME l_time;
  if (str_to_TIME(str->ptr(),str->length(),&l_time) == TIMESTAMP_NONE)
  {
    null_value=1;
    return(0);
  }
  return (longlong) calc_daynr(l_time.year,l_time.month,l_time.day) -
    calc_daynr(l_time.year,1,1) + 1;
}


// Returns the hour in time_exp in the range of 0 - 23

longlong Item_func_hour::val_int()
{
  String *str=args[0]->str(&value);
  ulong time_exp;
  if ((null_value=args[0]->null_value))
    return 0;
  time_exp= str_to_time(str->ptr(),str->length());
  return (longlong) (time_exp/10000);
}


// Returns the minute in time_exp in the range of 0 - 59

longlong Item_func_minute::val_int()
{
  String *str=args[0]->str(&value);
  ulong time_exp;
  if ((null_value=args[0]->null_value))
    return 0;
  time_exp= str_to_time(str->ptr(),str->length());
  return (longlong) ((time_exp % 10000) / 100);
}


longlong Item_func_month::val_int()
{
  String *str=args[0]->str(&value);
  if ((null_value=args[0]->null_value))
    return 0;
  TIME l_time;
  if (str_to_TIME(str->ptr(),str->length(),&l_time) == TIMESTAMP_NONE)
  {
    null_value=1;
    return(0);
  }
  return (longlong) l_time.month;
}

String* Item_func_monthname::str(String* str)
{
  uint month=(uint) val_int();
  if (null_value)
    return (String*) 0;
  return &month_names[month-1];
}


// Returns the quarter of the year

longlong Item_func_quarter::val_int()
{
  String *str=args[0]->str(&value);
  if ((null_value=args[0]->null_value))
    return 0;
  TIME l_time;
  if (str_to_TIME(str->ptr(),str->length(),&l_time) == TIMESTAMP_NONE)
  {
    null_value=1;
    return(0);
  }
  return (longlong) ((l_time.month+2)/3);
}


// Returns the second in time_exp in the range of 0 - 59

longlong Item_func_second::val_int()
{
  String *str=args[0]->str(&value);
  ulong time_exp;
  if ((null_value=args[0]->null_value))
    return 0;
  time_exp= str_to_time(str->ptr(),str->length());
  return (longlong) (time_exp % 100);
}


// Returns the week of year in the range of 1 - 53

longlong Item_func_week::val_int()
{
  String *str=args[0]->str(&value);
  long daynr,first_daynr;
  TIME l_time;
  uint year,month,day,weekday;

  if ((null_value=args[0]->null_value))
    return 0;
  if (str_to_TIME(str->ptr(),str->length(),&l_time) == TIMESTAMP_NONE)
  {
    null_value=1;
    return(0);
  }
  year=l_time.year;
  month=l_time.month;
  day=l_time.day;

  daynr=      calc_daynr(year,month,day);
  first_daynr=calc_daynr(year,1,1);

	  /* Caculate year and first daynr of year */
  if (month == 1 && (weekday=calc_weekday(first_daynr)) >= 4 &&
      day <= 7-weekday)
  {
    if (!year--)
      year=99;
    first_daynr=first_daynr-calc_days_in_year(year);
  }
  else if (month == 12 &&
	   (weekday=calc_weekday(first_daynr+calc_days_in_year(year))) <= 3 &&
	   day > 31-weekday)
  {
    first_daynr=first_daynr+calc_days_in_year(year);
    if (year++ == 99)
      year=0;
  }

	/* Calulate daynr of first day of week 1 */
  if ((weekday= calc_weekday(first_daynr)) >= 4)
    first_daynr+= (7-weekday);
  else
    first_daynr-=weekday;

  return (longlong) ((daynr-first_daynr)/7+1);
}


/* weekday() has a automatic to_days() on item */

longlong Item_func_weekday::val_int()
{
  ulong tmp_value=(ulong) args[0]->val_int();
  uint weekday;
  if ((null_value=(args[0]->null_value || !tmp_value)))
    return 0; /* purecov: inspected */

  weekday=calc_weekday(tmp_value);
  return (longlong) (odbc_type	? ((weekday+1) % 7) +1 : weekday);
}

String* Item_func_dayname::str(String* str)
{
  uint weekday=(uint) val_int();
  if (null_value)
    return (String*) 0;
  return &day_names[weekday];
}


longlong Item_func_year::val_int()
{
  String *str=args[0]->str(&value);
  if ((null_value=args[0]->null_value))
    return 0;
  TIME l_time;
  if (str_to_TIME(str->ptr(),str->length(),&l_time) == TIMESTAMP_NONE)
  {
    null_value=1;
    return(0);
  }
  return (longlong) l_time.year;
}


longlong Item_func_unix_timestamp::val_int()
{
  if (arg_count == 0)
    return (longlong) current_thd->query_start();
  if (args[0]->type() == FIELD_ITEM)
  {						// Optimize timestamp field
    Field *field=((Item_field*) args[0])->field;
    if (field->type() == FIELD_TYPE_TIMESTAMP)
      return ((Field_timestamp*) field)->get_time();
  }
  String *str=args[0]->str(&value);
  if ((null_value=args[0]->null_value))
  {
    return 0; /* purecov: inspected */
  }
  return (longlong) str_to_timestamp(str->ptr(),str->length());
}


longlong Item_func_time_to_sec::val_int()
{
  String *str=args[0]->str(&value);
  if ((null_value=args[0]->null_value))
    return 0; /* purecov: inspected */
  ulong tmp=str_to_time(str->ptr(),str->length());
  return (longlong) ((tmp/10000)*3600+((tmp/100)%100)*60+(tmp%100));
}


/*
** Get an easily manupulatable INTERVAL object from an interval.
** To make code easy, allow interval objects without separators.
*/


bool Item_interval::val_time(INTERVAL *t)
{
  String *res;
  long array[4],value;
  LINT_INIT(res);
  LINT_INIT(value);

  if ((int) int_type <= SECOND)
  {
    value=(long) args[0]->val_int();
    if ((null_value=args[0]->null_value))
      return 1;
  }
  else if (!(res=args[0]->str(&str_value)))
    return ((null_value=1));

  null_value=0;
  bzero(t,sizeof(*t));

  switch (int_type) {
  case YEAR:
    t->year=value;
    break;
  case MONTH:
    t->month=value;
    break;
  case DAY:
    t->day=value;
    break;
  case HOUR:
    t->hour=value;
    break;
  case MINUTE:
    t->minute=value;
    break;
  case SECOND:
    t->second=value;
    break;
  case YEAR_MONTH:				// Allow YEAR-MONTH YYYYYMM
    if (get_interval_info(res->ptr(),res->length(),2,array))
      return ((null_value=1));
    t->year=array[0];
    t->month=array[1];
    break;
  case DAY_HOUR:
    if (get_interval_info(res->ptr(),res->length(),2,array))
      return ((null_value=1));
    t->day=array[0];
    t->hour=array[1];
    break;
  case DAY_MINUTE:
    if (get_interval_info(res->ptr(),res->length(),3,array))
      return ((null_value=1));
    t->day=array[0];
    t->hour=array[1];
    t->minute=array[2];
    break;
  case DAY_SECOND:
    if (get_interval_info(res->ptr(),res->length(),4,array))
      return ((null_value=1));
    t->day=array[0];
    t->hour=array[1];
    t->minute=array[2];
    t->second=array[3];
    break;
  case HOUR_MINUTE:
    if (get_interval_info(res->ptr(),res->length(),2,array))
      return ((null_value=1));
    t->hour=array[0];
    t->minute=array[1];
    break;
  case HOUR_SECOND:
    if (get_interval_info(res->ptr(),res->length(),3,array))
      return ((null_value=1));
    t->hour=array[1];
    t->minute=array[2];
    t->second=array[3];
    break;
  case MINUTE_SECOND:
    if (get_interval_info(res->ptr(),res->length(),2,array))
      return ((null_value=1));
    t->minute=array[2];
    t->second=array[3];
    break;
  }
  return 0;
}

String *Item_date::str(String *str)
{
  ulong value=(ulong) val_int();
  if (null_value)
    return (String*) 0;
  if (!value)					// zero daynr
  {
    str->copy("0000-00-00");
    return str;
  }
  if (str->alloc(32))
    return &empty_string;			/* purecov: inspected */
  sprintf((char*) str->ptr(),"%04d-%02d-%02d",
	  (int) (value/10000L),(int) (value/100)%100,(int) (value%100));
  str->length(strlen(str->ptr()));
  return str;
}


longlong Item_func_from_days::val_int()
{
  longlong value=args[0]->val_int();
  if ((null_value=args[0]->null_value))
    return 0; /* purecov: inspected */

  uint year,month,day;
  get_date_from_daynr((long) value,&year,&month,&day);
  return (longlong) (year*10000L+month*100+day);
}


void Item_func_date_add_mm::fix_length_and_dec()
{
  /* Try to guess if arg[0] is a timestamp or a date */
  max_length= (args[0]->max_length <= 10) ? 10 : 19;
}


String *Item_func_date_add_mm::str(String *str)
{
  String *res= args[0]->str(str);
  long count= (long) args[1]->val_int();
  timestamp_type type;
  ulong period;
  TIME l_time;

  /* If some of the arguments was NULL, return NULL */
  if ((null_value=(args[0]->null_value || args[1]->null_value)))
    return 0;					// Return NULL pointer

  /* If not a TIMESTAMP or DATE, return a empty string */
  if ((type=str_to_TIME(res->ptr(),res->length(),&l_time)) == TIMESTAMP_NONE)
    return &empty_string;

  /* As res->ptr() isn't needed anymore, we can return result in str */

  str->alloc(19);				// Safety check
  period=l_time.year*12+l_time.month-1+count;
  if ((long) period < 0)
    bzero(&l_time,sizeof(l_time));
  else if (period >= 120000L)
  {
    l_time.year= 9999;
    l_time.month= 12;
    l_time.day= 31;
    l_time.hour=23;
    l_time.minute=l_time.sec=59;
  }
  else
  {
    l_time.year= (uint) (period/12);
    l_time.month= (uint) (period % 12L)+1;
  }
  /* Return result in DATE or TIMESTAMP format, according to the argument */
  if (type == TIMESTAMP_DATE)
    sprintf((char*) str->ptr(),"%04d-%02d-%02d",
	    l_time.year,l_time.month,l_time.day);
  else
    sprintf((char*) str->ptr(),"%04d-%02d-%02d %02d:%02d:%02d",
	    l_time.year,l_time.month,l_time.day,
	    l_time.hour,l_time.minute,l_time.sec);
  str->length(strlen(str->ptr()));
  return str;
}


void Item_func_curdate::fix_length_and_dec()
{
  struct tm tm_tmp,*start_time;
  time_t query_start=current_thd->query_start();
  decimals=0; max_length=10;
  localtime_r(&query_start,&tm_tmp);
  start_time=&tm_tmp;
  value=(longlong) ((ulong) ((uint) start_time->tm_year+1900)*10000L+
		    ((uint) start_time->tm_mon+1)*100+
		    (uint) start_time->tm_mday);
}


longlong Item_func_curdate::val_int()
{
  return value;
}

void Item_func_curtime::fix_length_and_dec()
{
  struct tm tm_tmp,*start_time;
  time_t query_start=current_thd->query_start();
  decimals=0; max_length=8;
  localtime_r(&query_start,&tm_tmp);
  start_time=&tm_tmp;
  value=(longlong) ((ulong) ((uint) start_time->tm_hour)*10000L+
		    (ulong) (((uint) start_time->tm_min)*100L+
			     (uint) start_time->tm_sec));
  sprintf(buff,"%02d:%02d:%02d",
	  (int) start_time->tm_hour,
	  (int) start_time->tm_min,
	  (int) start_time->tm_sec);
  str_value.set((const char*) buff,strlen(buff));
}


void Item_func_now::fix_length_and_dec()
{
  struct tm tm_tmp,*start_time;
  time_t query_start=current_thd->query_start();
  decimals=0; max_length=19;
  localtime_r(&query_start,&tm_tmp);
  start_time=&tm_tmp;
  value=((longlong) ((ulong) ((uint) start_time->tm_year+1900)*10000L+
		     (((uint) start_time->tm_mon+1)*100+
		      (uint) start_time->tm_mday))*(longlong) 1000000L+
	 (longlong) ((ulong) ((uint) start_time->tm_hour)*10000L+
		     (ulong) (((uint) start_time->tm_min)*100L+
			    (uint) start_time->tm_sec)));
  sprintf(buff,"%04d-%02d-%02d %02d:%02d:%02d",
	  ((int) (start_time->tm_year+1900)) % 10000,
	  (int) start_time->tm_mon+1,
	  (int) start_time->tm_mday,
	  (int) start_time->tm_hour,
	  (int) start_time->tm_min,
	  (int) start_time->tm_sec);
  str_value.set((const char*) buff,strlen(buff));
}


longlong Item_func_now::val_int()
{
  return value;
}


String *Item_func_now::str(String *str)
{
  return &str_value;
}


String *Item_func_sec_to_time::str(String *str)
{
  char buff[23];
  ulonglong time=(ulonglong) args[0]->val_int();
  if ((null_value=args[0]->null_value))
    return (String*) 0;
  uint sec= (uint) (time % 3600);
  sprintf(buff,"%02lu:%02u:%02u",(ulong) (time/3600),sec/60, sec % 60);
  str->copy(buff,strlen(buff));
  return str;
}


longlong Item_func_sec_to_time::val_int()
{
  ulonglong time=(ulonglong) args[0]->val_int();
  if ((null_value=args[0]->null_value))
    return 0;
  return (time / 3600)*10000+((time/60) % 60)*100+ (time % 60);
}


void Item_func_date_format::fix_length_and_dec()
{
  decimals=0;
  if (args[1]->type() == STRING_ITEM)
  {						// Optimize the normal case
    fixed_length=1;
    max_length=format_length(((Item_string*) args[1])->const_string());
  }
  else
  {
    fixed_length=0;
    max_length=args[1]->max_length*10;
    set_if_smaller(max_length,MAX_BLOB_WIDTH);
  }
  maybe_null=1;					// If wrong date
}


uint Item_func_date_format::format_length(const String *format)
{
  uint size=0;
  const char *ptr=format->ptr();
  const char *end=ptr+format->length();

  for (; ptr != end ; ptr++)
  {
    switch(*ptr) {
    case 'M': /* month, textual */
    case 'W': /* day (of the week), textual */
      size += 9;
      break;
    case 'D': /* day (of the month), numeric plus english suffix */
    case 'Y': /* year, numeric, 4 digits */
      size += 4;
      break;
    case 'a': /* locale's abbreviated weekday name (Sun..Sat) */
    case 'b': /* locale's abbreviated month name (Jan.Dec) */
    case 'j': /* day of year (001..366) */
      size += 3;
      break;
    case 'H': /* hour (00..23) */
    case 'y': /* year, numeric, 2 digits */
    case 'm': /* month, numeric */
    case 'd': /* day (of the month), numeric */
    case 'h': /* hour (01..12) */
    case 'I': /* --||-- */
    case 'i': /* minutes, numeric */
    case 'k': /* hour ( 0..23) */
    case 'l': /* hour ( 1..12) */
    case 'p': /* locale's AM or PM */
    case 'S': /* second (00..61) */
    case 's': /* seconds, numeric */
      size += 2;
      break;
    case 'r': /* time, 12-hour (hh:mm:ss [AP]M) */
      size += 11;
      break;
    case 'T': /* time, 24-hour (hh:mm:ss) */
      size += 8;
      break;
    case '%':		// Ignore one '%' this for now
      if (ptr+1 != end && ptr[1] == '%')
      {
	size++;
	ptr++;
      }
      break;
    case 'w': /* day (of the week), numeric */
    default:
      size++;
      break;
    }
  }
  return size;
}


String *Item_func_date_format::str(String *str)
{
  String *res,*format;
  TIME l_time;
  char intbuff[15];
  uint size,weekday;

  null_value=0;

  if (!(res=args[0]->str(str)))
  {
    null_value=1;
    return 0;
  }

  if(!date_or_time)
  {
    if (str_to_TIME(res->ptr(),res->length(),&l_time) == TIMESTAMP_NONE)
    {
      null_value=1;
      return 0;
    }
  }
  else
  {
    ulong time_exp;
    time_exp= str_to_time(res->ptr(),res->length());
    bzero(&l_time,sizeof(l_time));
    l_time.sec=    time_exp%100;
    l_time.minute= (time_exp/100)%100;
    l_time.hour=   (time_exp/10000);
  }

  if (!(format = args[1]->str(str)) || !format->length())
  {
    null_value=1;
    return 0;
  }

  if (fixed_length)
    size=max_length;
  else
    size=format_length(format);
  if (format == str)
    str=&str_value;				// Save result here
  if (str->alloc(size))
  {
    null_value=1;
    return 0;
  }
  str->length(0);

  /* Create the result string */
  const char *ptr=format->ptr();
  const char *end=ptr+format->length();
  for ( ; ptr != end ; ptr++)
  {
    switch(*ptr) {
    case 'M':
      if(!l_time.month)
      {
	null_value=1;
	return 0;
      }
      str->append(month_names[l_time.month-1]);
      break;
    case 'b':
      if(!l_time.month)
      {
	null_value=1;
	return 0;
      }
      str->append(month_names[l_time.month-1].ptr(),3);
      break;
    case 'W':
      if(date_or_time)
      {
	null_value=1;
	return 0;
      }
      weekday=calc_weekday(calc_daynr(l_time.year,l_time.month,l_time.day));
      str->append(day_names[weekday]);
      break;
    case 'a':
      if(date_or_time)
      {
	null_value=1;
	return 0;
      }
      weekday=calc_weekday(calc_daynr(l_time.year,l_time.month,l_time.day));
      str->append(day_names[weekday].ptr(),3);
      break;
    case 'D':
      if(date_or_time)
      {
	null_value=1;
	return 0;
      }
      sprintf(intbuff,"%d",l_time.day);
      str->append(intbuff);
      if (l_time.day >= 10 &&  l_time.day <= 19)
	str->append("th");
      else
      {
	switch (l_time.day %10)
	{
	case 1:
	  str->append("st");
	  break;
	case 2:
	  str->append("nd");
	  break;
	case 3:
	  str->append("rd");
	  break;
	default:
	  str->append("th");
	  break;
	}
      }
      break;
    case 'Y':
      sprintf(intbuff,"%04d",l_time.year);
      str->append(intbuff);
      break;
    case 'y':
      sprintf(intbuff,"%02d",l_time.year%100);
      str->append(intbuff);
      break;
    case 'm':
      sprintf(intbuff,"%02d",l_time.month);
      str->append(intbuff);
      break;
    case 'd':
      sprintf(intbuff,"%02d",l_time.day);
      str->append(intbuff);
      break;
    case 'H':
      sprintf(intbuff,"%02d",l_time.hour);
      str->append(intbuff);
      break;
    case 'h':
    case 'I':
      sprintf(intbuff,"%02d", (l_time.hour+11)%12+1);
      str->append(intbuff);
      break;
    case 'i':					/* minutes */
      sprintf(intbuff,"%02d",l_time.minute);
      str->append(intbuff);
      break;
    case 'j':
      if(date_or_time)
      {
	null_value=1;
	return 0;
      }
      sprintf(intbuff,"%03d",
	      (int) (calc_daynr(l_time.year,l_time.month,l_time.day) -
		     calc_daynr(l_time.year,1,1)) + 1);
      str->append(intbuff);
      break;
    case 'k':
      sprintf(intbuff,"%d",l_time.hour);
      str->append(intbuff);
      break;
    case 'l':
      sprintf(intbuff,"%d", (l_time.hour+11)%12+1);
      str->append(intbuff);
      break;
    case 'p':
      str->append(l_time.hour < 12 ? "AM" : "PM");
      break;
    case 'r':
      sprintf(intbuff,(l_time.hour < 12) ? "%02d:%02d:%02d AM" :
	      "%02d:%02d:%02d PM",(l_time.hour+11)%12+1,l_time.minute,
	      l_time.sec);
      str->append(intbuff);
      break;
    case 'S':
    case 's':
      sprintf(intbuff,"%02d",l_time.sec);
      str->append(intbuff);
      break;
    case 'T':
      sprintf(intbuff,"%02d:%02d:%02d",l_time.hour,l_time.minute,l_time.sec);
      str->append(intbuff);
      break;
    case 'w':
      weekday=calc_weekday(calc_daynr(l_time.year,l_time.month,l_time.day));
      sprintf(intbuff,"%01d",(weekday+1)%7);
      str->append(intbuff);
      break;
    case '%':				// '%' will be prefix in the future
      if (ptr != end && ptr[1] == '%')
      {
	str->append('%');
	ptr++;
      }
      break;
    default:
      str->append(*ptr);
      break;
    }
  }
  return str;
}


String *Item_func_from_unixtime::str(String *str)
{
  struct tm tm_tmp,*start_time;
  time_t tmp=(time_t) args[0]->val_int();
  if ((null_value=args[0]->null_value))
    return 0;
  localtime_r(&tmp,&tm_tmp);
  start_time=&tm_tmp;
  if (str->alloc(32))				// Some extra space for safety
    return str;					/* purecov: inspected */
  sprintf((char*) str->ptr(),"%04d-%02d-%02d %02d:%02d:%02d",
	  (int) ((start_time->tm_year+1900) % 10000),
	  (int) start_time->tm_mon+1,
	  (int) start_time->tm_mday,
	  (int) start_time->tm_hour,
	  (int) start_time->tm_min,
	  (int) start_time->tm_sec);
  str->length(strlen(str->ptr()));		// Can't trust sprintf
  return str;
}


longlong Item_func_from_unixtime::val_int()
{
  time_t tmp=(time_t) (ulong) args[0]->val_int();
  if ((null_value=args[0]->null_value))
    return 0;
  struct tm tm_tmp,*start_time;
  localtime_r(&tmp,&tm_tmp);
  start_time= &tm_tmp;
  return ((longlong) ((ulong) ((uint) start_time->tm_year+1900)*10000L+
		      (((uint) start_time->tm_mon+1)*100+
		       (uint) start_time->tm_mday))*LL(1000000)+
	  (longlong) ((ulong) ((uint) start_time->tm_hour)*10000L+
		      (ulong) (((uint) start_time->tm_min)*100L+
			       (uint) start_time->tm_sec)));
}


void Item_date_add_interval::fix_length_and_dec()
{
  /* Try to guess if arg[0] is a timestamp or a date */
  max_length= (args[0]->max_length <= 10) ? 10 : 19;
}


String *Item_date_add_interval::str(String *str)
{
  /* Here arg[1] is a Item_interval object */
  String *res= args[0]->str(str);
  timestamp_type type;
  ulong period;
  INTERVAL interval;
  TIME ltime;
  Item_interval *interval_item=(Item_interval*) args[1];

  if (interval_item->val_time(&interval))
  {
    null_value=1;
    return 0;
  }
  null_value=0;
  /* If not a TIMESTAMP or DATE, return a empty string */
  if ((type=str_to_TIME(res->ptr(),res->length(),&ltime)) == TIMESTAMP_NONE)
    return &empty_string;

  str->alloc(32);				// Some extra space for safety
  switch (interval_item->int_type) {
  case YEAR:
    ltime.year += interval.year;
    if ((int) ltime.year < 0)
      bzero(&ltime,sizeof(ltime));
    else if (ltime.year >= 1000L)
    {
      ltime.year= 9999;
      ltime.month= 12;
      ltime.day= 31;
      ltime.hour=23;
      ltime.minute=ltime.sec=59;
    }
    break;
  case YEAR_MONTH:
  case MONTH:
    period=ltime.year*12+interval.year+ltime.month-1+interval.month;
    if ((long) period < 0)
      bzero(&ltime,sizeof(ltime));
    else if (period >= 120000L)
    {
      ltime.year= 9999;
      ltime.month= 12;
      ltime.day= 31;
      ltime.hour=23;
      ltime.minute=ltime.sec=59;
    }
    else
    {
      ltime.year= (uint) (period/12);
      ltime.month= (uint) (period % 12L)+1;
    }
    break;
  default:
    return &empty_string;				// Don't crash yet.
  }
  /* Return result in DATE or TIMESTAMP format, according to the argument */
  if (type == TIMESTAMP_DATE)
    sprintf((char*) str->ptr(),"%04d-%02d-%02d",
	    ltime.year,ltime.month,ltime.day);
  else
    sprintf((char*) str->ptr(),"%04d-%02d-%02d %02d:%02d:%02d",
	    ltime.year,ltime.month,ltime.day,
	    ltime.hour,ltime.minute,ltime.sec);
  str->length(strlen(str->ptr()));
  return str;
}
