char *fnsv = "C-Kermit functions, 6.1.154, 8 February 1998";

/*  C K C F N S  --  System-independent Kermit protocol support functions.  */

/*  ...Part 1 (others moved to ckcfn2,3 to make this module smaller) */

/*
  Author: Frank da Cruz <fdc@columbia.edu>,
  Columbia University Academic Information Systems, New York City.

  Copyright (C) 1985, 1998, Trustees of Columbia University in the City of New
  York.  The C-Kermit software may not be, in whole or in part, licensed or
  sold for profit as a software product itself, nor may it be included in or
  distributed with commercial products or otherwise distributed by commercial
  concerns to their clients or customers without written permission of the
  Office of Kermit Development and Distribution, Columbia University.  This
  copyright notice must not be removed, altered, or obscured.
*/
/*
 System-dependent primitives defined in:

   ck?tio.c -- terminal (communications) i/o
   cx?fio.c -- file i/o, directory structure
*/
#include "ckcsym.h"			/* Needed for Stratus VOS */
#include "ckcasc.h"			/* ASCII symbols */
#include "ckcdeb.h"			/* Debug formats, typedefs, etc. */
#include "ckcker.h"			/* Symbol definitions for Kermit */
#include "ckcxla.h"			/* Character set symbols */

_PROTOTYP( static int bgetpkt, (int) );

#ifndef NOSPL
_PROTOTYP( int zzstring, (char *, char **, int *) );
#endif /* NOSPL */

#ifdef OS2ORUNIX
_PROTOTYP( long zfsize, (char *) );
#endif /* OS2ORUNIX */

#ifdef OS2
#include <io.h>
#endif /* OS2 */

#ifdef VMS
#include <errno.h>
#endif /* VMS */

/* Externals from ckcmai.c */
extern int spsiz, spmax, rpsiz, timint, srvtim, rtimo, npad, ebq, ebqflg,
 rpt, rptq, rptflg, capas, keep, fncact, pkttim, autopar, spsizr, xitsta;
extern int pktnum, bctr, bctu, bctl, fmask, clfils, sbufnum, protocol,
 size, osize, spktl, nfils, ckwarn, timef, spsizf, sndtyp, rcvtyp, success;
extern int parity, turn, network, what, whatru, fsecs, justone, slostart,
 ckdelay, displa, xflg, mypadn, remfile, moving, recursive, nettype;
extern long filcnt, ffc, flci, flco, tlci, tlco, tfc, fsize, sendstart, rs_len;
extern long filrej, oldcps, cps, peakcps, ccu, ccp, calibrate, filestatus;
extern int fblksiz, frecl, frecfm, forg, fcctrl, fdispla;
extern int spackets, rpackets, timeouts, retrans, crunched, wmax, wcur;
extern int hcflg, binary, fncnv, b_save, f_save, server, cxseen, czseen;
extern int nakstate, discard, rejection, local, xfermode;
extern int rq, rqf, sq, wslots, wslotn, wslotr, winlo, urpsiz, rln;
extern int fnspath, fnrpath, eofmethod;
extern int atcapr, atcapb, atcapu;
extern int lpcapr, lpcapb, lpcapu;
extern int swcapr, swcapb, swcapu;
extern int lscapr, lscapb, lscapu;
extern int rscapr, rscapb, rscapu;
extern int rptena, rptmin;
extern int sseqtbl[];
extern int numerrs;
extern int alphacase;
extern long rptn;
extern int maxtry;
extern int stdouf;
extern int sendmode;
extern int carrier, ttprty;

extern int g_binary, g_fncnv;

#ifdef GFTIMER
extern CKFLOAT fpfsecs;
#endif /* GFTIMER */

#ifdef OS2
extern struct zattr iattr;
#endif /* OS2 */

#ifdef DEBUG
extern int deblog;
#endif /* DEBUG */

#ifdef PIPESEND
extern int pipesend, usepipes;
#endif /* PIPESEND */

#ifndef NOCSETS
extern int tcharset, fcharset;
extern int ntcsets;
extern struct csinfo tcsinfo[], fcsinfo[];
#endif /* NOCSETS */

/* Per-file text/binary-mode recognition */

#ifdef PATTERNS
extern int patterns;
extern char *txtpatterns[];
extern char *binpatterns[];
#endif /* PATTERNS */

#ifdef STREAMING
extern int streamrq, streaming, streamed, streamok;
#endif /* STREAMING */
extern int reliable, clearrq, cleared, urclear;

extern int
  atenci, atenco, atdati, atdato, atleni, atleno, atblki, atblko,
  attypi, attypo, atsidi, atsido, atsysi, atsyso, atdisi, atdiso; 

extern int bigsbsiz, bigrbsiz;
extern char *versio;
extern char *filefile;
extern char whoareu[], * cksysid;

#ifndef NOSERVER
extern int ngetpath;
extern char * getpath[];
extern int fromgetpath;
#endif /* NOSERVER */

extern CHAR *srvcmd, * epktmsg;
extern CHAR padch, mypadc, eol, seol, feol, ctlq, myctlq, sstate, myrptq;
extern CHAR *data, padbuf[], stchr, mystch;
extern CHAR *srvptr;
extern CHAR *rdatap;
extern char *cmarg, *cmarg2, **cmlist, filnam[];
extern char fspec[], *sfspec, *rfspec;

#ifndef NOMSEND
extern struct filelist * filehead, * filenext;
extern int addlist;
#endif /* NOMSEND */

_PROTOTYP( CHAR *rpar, (void) );
_PROTOTYP( int lslook, (unsigned int b) );	/* Locking Shift Lookahead */
_PROTOTYP( int szeof, (CHAR *s) );
_PROTOTYP( VOID fnlist, (void) );
_PROTOTYP( static int nxtdir, (void) );
_PROTOTYP( static int nxtdel, (void) );

/* International character sets */

#ifndef NOCSETS
/* Pointers to translation functions */
#ifdef CK_ANSIC
extern CHAR (*xls[MAXTCSETS+1][MAXFCSETS+1])(CHAR); /* Character set */
extern CHAR (*xlr[MAXTCSETS+1][MAXFCSETS+1])(CHAR); /* translation functions */
#else
extern CHAR (*xls[MAXTCSETS+1][MAXFCSETS+1])();	/* Character set */
extern CHAR (*xlr[MAXTCSETS+1][MAXFCSETS+1])();	/* translation functions. */
#endif /* CK_ANSIC */
_PROTOTYP( CHAR (*rx), (CHAR) );	/* Input translation function */
_PROTOTYP( CHAR (*sx), (CHAR) );	/* Output translation function */
_PROTOTYP( CHAR ident, (CHAR) );	/* Identity translation function */
#endif /* NOCSETS */

/* Windowing things */

extern int rseqtbl[];			/* Rec'd-packet sequence # table */

/* (PWP) external def. of things used in buffered file input and output */

#ifdef DYNAMIC
extern char *zinbuffer, *zoutbuffer;
#else
extern char zinbuffer[], zoutbuffer[];
#endif /* DYNAMIC */
extern char *zinptr, *zoutptr;
extern int zincnt, zoutcnt, zobufsize;

extern long crcta[], crctb[];		/* CRC-16 generation tables */

/*
  Variables defined in this module but shared by other modules.
*/

long crc16 = 0L;			/* File CRC = \v(crc16) */
int docrc  = 0;				/* Accumulate CRC for \v(crc16) */
int xfrbel = 1;
char * ofperms = "";			/* Output file permissions */

#ifdef CALIBRATE
#define CAL_O 3
#define CAL_M 253

int cal_j = 0;

CHAR
cal_a[] = { 
 16, 45, 98,  3, 52, 41, 14,  7, 76,165,122, 11,104, 77,166, 15,
160, 93, 18, 19,112, 85, 54, 23,232,213, 90, 27, 12, 81,126, 31,
  4,205, 34, 35,144, 73,110, 39, 28,133,218, 43,156, 65,102, 47,
 84, 61, 50, 51,208,117, 86, 55,  8,245, 74, 59, 44,125,222, 63,
 80,  1,162, 67,116,105,206, 71,120,  9,250, 75, 88, 97,  6, 79,
100,221, 82, 83, 36, 89, 94, 87, 40, 21,106, 91,236,145,150, 95,
228, 33,130, 99,148,137,198,103,108,169, 42,107,184,129, 78,111,
  0, 49,114,115, 32,121,254,119,172, 57,138,123,152,177, 22,127,
240,193,  2,131,176,  5, 38,135,204,229, 10,139,200,161,174,143,
128, 17,146,147, 68,153, 30,151, 72,217,170,155, 24,209, 62,159,
 64,225,194,163,244,201, 70,167,216,197,234,171,188,109,230,175,
212,113,178,179,132,185,190,183,136,249,202,187, 92,241,118,191,
 48,237, 66,195, 96,233,142,199,248, 37, 58,203, 60, 13,134,207,
 20, 29,210,211,164,149,182,215,220, 25, 26,219,124,157,246,223,
180,141,226,227,192,101,238,231, 56, 69,154,235,252,173, 46,239,
224,253,242,243,196, 53,214,247,168,181,186,251,140,189,158,255
};
#endif /* CALIBRATE */

char * rf_err = "Error receiving file";	/* rcvfil() error message */

#ifdef CK_SPEED
short ctlp[256] = {		/* Control-Prefix table */
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* C0  */
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* G0  */
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1, /* DEL */
  1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1, /* C1  */
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0, /* G1  */
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,1  /* 255 */
};
#endif /* CK_SPEED */

int sndsrc;		/* Flag for where to get names of files to send: */
					/* -1: znext() function */
					/*  0: stdin */
					/* >0: list in cmlist or other list */
					/* -9: calibrate */

int  memstr;				/* Flag for input from memory string */
int  funcstr;				/* Flag for input from function */
int  bestlen = 0;
int  maxsend = 0;

int gnf_binary = 0;			/* Prevailing xfer mode for gnfile */

#ifdef pdp11
CHAR myinit[32];			/* Copy of my Send-Init data */
#else
CHAR myinit[100];			/* Copy of my Send-Init data */
#endif /* pdp11 */

/* Variables local to this module */

#ifdef TLOG
#ifndef XYZ_INTERNAL
static
#endif /* XYZ_INTERNAL */
char *fncnam[] = {
  "rename", "replace", "backup", "append", "discard", "ask", "update", ""
};
#endif /* TLOG */

static char *memptr;			/* Pointer for memory strings */

#ifdef VMS
extern int batch;
#else
extern int backgrd;
#endif /* VMS */

#ifdef CK_CTRLZ
static int lastchar = 0;
#endif /* CK_CTRLZ */

#ifdef CK_ANSI
static int (*funcptr)(void);		/* Pointer for function strings */
#else
static int (*funcptr)();
#endif /* CK_ANSI */

#ifdef pdp11
static char cmdstr[50];			/* System command string. */
#else
static char cmdstr[256];
#endif /* pdp11 */

static int drain;			/* For draining stacked-up ACKs. */

static int first;			/* Flag for first char from input */
static CHAR t;				/* Current character */
#ifdef COMMENT
static CHAR next;			/* Next character */
#endif /* COMMENT */

static int ebqsent = 0;			/* 8th-bit prefix bid that I sent */
static int lsstate = 0;			/* Locking shift state */
static int lsquote = 0;			/* Locking shift quote */

#ifdef datageneral
extern int quiet;
#endif /* datageneral */

/*  E N C S T R  --  Encode a string from memory. */

/*
  Call this instead of getpkt() if source is a string, rather than a file.
  Note: Character set translation is never done in this case.
*/

#ifdef COMMENT
#define ENCBUFL 200
#ifndef pdp11
CHAR encbuf[ENCBUFL];
#else
/* This is gross, but the pdp11 root segment is out of space */
/* Will allocate it in ckuusr.c. */
extern CHAR encbuf[];
#endif /* pdp11 */
#endif /* COMMENT */

/*
  Encode packet data from a string in memory rather than from a file.
  Returns 0 on success, -1 if the string could not be completely encoded
  into the currently negotiated data field length.
*/
int
encstr(s) CHAR* s; {
#ifdef COMMENT
    int m; char *p;
    CHAR *dsave;

    if (!s) s = (CHAR *)"";
    if ((m = (int)strlen((char *)s)) > ENCBUFL) {
	debug(F111,"encstr string too long for buffer",s,ENCBUFL);
	s[ENCBUFL] = '\0';
    }
    if (m > spsiz-bctl-3) {
	debug(F111,"encstr string too long for packet",s,spsiz-bctl-3);
	s[spsiz-bctl-3] = '\0';
    }
    m = memstr; p = memptr;		/* Save these. */

    memptr = (char *)s;			/* Point to the string. */
    memstr = 1;				/* Flag memory string as source. */
    first = 1;				/* Initialize character lookahead. */
    dsave = data;			/* Boy is this ugly... */
    data = encbuf + 7;			/* Why + 7?  See spack()... */
    *data = NUL;			/* In case s is empty */
    getpkt(spsiz,0);			/* Fill a packet from the string. */
    data = dsave;			/* (sorry...) */
    memstr = m;				/* Restore memory string flag */
    memptr = p;				/* and pointer */
    first = 1;				/* Put this back as we found it. */
    return(0);
#else
/*
  Recoded 30 Jul 94 to use the regular data buffer and the negotiated
  maximum packet size.  Previously we were limited to the length of encbuf[].
  Also, to return a failure code if the entire encoded string would not fit.
*/
    int m, rc, slen; char *p;
    if (!data) return(-1);		/* Watch out for null pointers. */
    if (!s) s = (CHAR *)"";		/* Ditto. */
    slen = strlen((char *)s);		/* Length of source string. */
    rc = 0;				/* Return code. */
    m = memstr; p = memptr;		/* Save these. */
    memptr = (char *)s;			/* Point to the string. */
    memstr = 1;				/* Flag memory string as source. */
    first = 1;				/* Initialize character lookahead. */
    *data = NUL;			/* In case s is empty */
    getpkt(spsiz,0);			/* Fill a packet from the string. */

    if (
#ifdef COMMENT
	*memptr
#else
	memptr < (char *)(s + slen)
#endif /* COMMENT */
	) {				/* This means we didn't encode */
	rc = -1;			/* the whole string. */
	debug(F101,"encstr string too big","",size);
    } else
      debug(F101,"encstr fits OK, size","",size);
    memstr = m;				/* Restore memory string flag */
    memptr = p;				/* and pointer */
    first = 1;				/* Put this back as we found it. */
    return(rc);
#endif /* COMMENT */
}

#ifdef COMMENT
/*
  We don't use this routine any more -- the code has been incorporated
  directly into getpkt() to reduce per-character function call overhead.
  Also, watch out: this routine hasn't been updated since it was commented
  out a long time ago.
*/
/* E N C O D E - Kermit packet encoding procedure */

VOID
encode(a) CHAR a; {			/* The current character */
    int a7;				/* Low order 7 bits of character */
    int b8;				/* 8th bit of character */
 
#ifndef NOCSETS
    if (!binary && sx) a = (*sx)(a);	/* Translate. */
#endif /* NOCSETS */

    if (rptflg)	{   	    		/* Repeat processing? */
        if (a == next && (first == 0)) { /* Got a run... */
	    if (++rpt < 94)		/* Below max, just count */
                return;
	    else if (rpt == 94) {	/* Reached max, must dump */
                data[size++] = rptq;
                data[size++] = tochar(rpt);
		rptn += rpt;		/* Count, for stats */
                rpt = 0;
	    }
        } else if (rpt == 1) {		/* Run broken, only 2? */
            rpt = 0;			/* Yes, reset repeat flag & count. */
	    encode(a);			/* Do the character twice. */
	    if (size <= maxsize) osize = size;
	    rpt = 0;
	    encode(a);
	    return;
	} else if (rpt > 1) {		/* More than two */
            data[size++] = rptq;	/* Insert the repeat prefix */
            data[size++] = tochar(++rpt); /* and count. */
	    rptn += rpt;
            rpt = 0;			/* Reset repeat counter. */
        }
    }
    a7 = a & 0177;			/* Isolate ASCII part */
    b8 = a & 0200;			/* and 8th (parity) bit. */

    if (ebqflg && b8) {			/* Do 8th bit prefix if necessary. */
        data[size++] = ebq;
        a = a7;
    }
    if ((a7 < SP) || (a7==DEL))	{	/* Do control prefix if necessary */
        data[size++] = myctlq;
	a = ctl(a);
    }
    if (a7 == myctlq)			/* Prefix the control prefix */
        data[size++] = myctlq;

    if ((rptflg) && (a7 == rptq))	/* If it's the repeat prefix, */
        data[size++] = myctlq;		/* quote it if doing repeat counts. */

    if ((ebqflg) && (a7 == ebq))	/* Prefix the 8th bit prefix */
        data[size++] = myctlq;		/* if doing 8th-bit prefixes */

    data[size++] = a;			/* Finally, insert the character */
    data[size] = '\0';			/* itself, and mark the end. */
}
#endif /* COMMENT */

/*  Output functions passed to 'decode':  */

int			       /*  Put character in server command buffer  */
#ifdef CK_ANSIC
putsrv(char c)
#else
putsrv(c) register char c;
#endif /* CK_ANSIC */
/* putsrv */ {
    *srvptr++ = c;
    *srvptr = '\0';		/* Make sure buffer is null-terminated */
    return(0);
}

int					/*  Output character to console.  */
#ifdef CK_ANSIC
puttrm(char c)
#else
puttrm(c) register char c;
#endif /* CK_ANSIC */
/* puttrm */ {
#ifndef NOSPL    
    extern char * qbufp;		/* If REMOTE QUERY active, */
    extern int query, qbufn;		/* also store response in */
    if (query && qbufn++ < 1024) {	/* query buffer. */
	*qbufp++ = c;
	*qbufp = NUL;
    } /* else */			/* else means don't display */
#endif /* NOSPL */
    conoc(c);
    return(0);
}

int					/*  Output char to file. */
#ifdef CK_ANSIC
putfil(char c)
#else
putfil(c) register char c;
#endif /* CK_ANSIC */
/* putfil */ {
    debug(F000,"putfil","",c);
    if (zchout(ZOFILE, (char) (c & fmask)) < 0) {
	czseen = 1;   			/* If write error... */
	debug(F101,"putfil zchout write error, setting czseen","",1);
	return(-1);
    }
    return(0);
}

/*
  The following function is a wrapper for putfil().  The only reason for its
  existence is to be passed as a function pointer to decode(), which treats
  putfil() itself specially -- bypassing it and using an internal macro
  instead to speed things up.  Use zputfil() instead of putfil() in cases where
  we do not want this to happen, e.g. when we need to send output to the file
  with a mixture of zchout() and zsout()/zsoutl() calls (as is the case with
  incoming short-form REMOTE command replies redirected to a file), which would
  otherwise result in data written to the file out of order.
*/
int
#ifdef CK_ANSIC
zputfil(char c)
#else
zputfil(c) register char c;
#endif /* CK_ANSIC */
/* zputfil */ {
    return(putfil(c));
}


/* D E C O D E  --  Kermit packet decoding procedure */

/*
 Call with string to be decoded and an output function.
 Returns 0 on success, -1 on failure (e.g. disk full).

 This is the "inner loop" when receiving files, and must be coded as 
 efficiently as possible.  Note some potential problems:  if a packet
 is badly formed, having a prefixed sequence ending prematurely, this
 function, as coded, could read past the end of the packet.  This has
 never happened, thus the additional (time-consuming) tests have not
 been added.
*/

static CHAR *xdbuf;	/* Global version of decode()'s buffer pointer */
                        /* for use by translation functions. */

/* Function for pushing a character onto decode()'s input stream. */

VOID
#ifdef CK_ANSIC
zdstuff(CHAR c)
#else
zdstuff(c) CHAR c;
#endif /* CK_ANSIC */
/* zdstuff */ {	
    xdbuf--;				/* Back up the pointer. */
    *xdbuf = c;				/* Stuff the character. */
}

#ifdef CKTUNING
/*
  Trimmed-down packet decoder for binary-mode no-parity transfers.
  decode() is the full version.
*/
int
#ifdef CK_ANSIC
bdecode(CHAR *buf, int (*fn)(char))
#else
bdecode(buf,fn) register CHAR *buf; register int (*fn)();
#endif /* CK_ANSIC */
/* bdecode */ {
    register unsigned int a, a7;	/* Various copies of current char */
    int ccpflg;				/* For Ctrl-unprefixing stats */
    int t;				/* Int version of character */
    int len;
    long z;				/* For CRC calculation */
    CHAR c;				/* Current character */

    if (!binary || parity || fn != putfil) /* JUST IN CASE */
      return(decode(buf,fn,1));
    debug(F100,"BDECODE","",0);

    xdbuf = buf;			/* Global copy of source pointer. */

    len = rln;				/* Number of bytes in data field */
    while (len > 0) {
        a = *xdbuf++ & 0xff;		/* Get next character */
	len--;
	rpt = 0;			/* Initialize repeat count. */
	if (a == rptq && rptflg) {	/* Got a repeat prefix? */
	    rpt = xunchar(*xdbuf++ & 0xFF); /* Yes, get the repeat count, */
	    rptn += rpt;
	    a = *xdbuf++ & 0xFF;	/* and get the prefixed character. */
	    len -= 2;
	}
	ccpflg = 0;			/* Control prefix flag. */
	if (a == ctlq) {		/* If control prefix, */
	    a  = *xdbuf++ & 0xFF;	/* get its operand */
	    len--;
	    a7 = a & 0x7F;		/* and its low 7 bits. */
	    if ((a7 >= 0100 && a7 <= 0137) || a7 == '?') { /* Controllify */
		a = ctl(a);		/* if in control range. */
		a7 = a & 0x7F;
		ccpflg = 1;		/* Note that we did this */
		ccp++;			/* Count for stats */
	    }
	} else a7 = a & 0x7f;		/* Not control quote */
	if (a7 < 32 || a7 == 127)	/* A bare control character? */
	  if (!ccpflg) ccu++;		/* Count it */
	if (!rpt) rpt = 1;
	for (; rpt > 0; rpt--) {	/* Output the char RPT times */
#ifdef CALIBRATE
	    if (calibrate) {
		ffc++;
		continue;
	    }
#endif /* CALIBRATE */
	    t = zmchout(a & fmask);	/* zmchout is a macro */
	    if (t < 0) {
		debug(F101,"bdecode write error - errno","",errno);
		return(-1);
	    }
	    ffc++;			/* Count the character */
	    if (docrc && !remfile) {	/* Update file CRC */
		c = a;			/* Force conversion to unsigned char */
		z = crc16 ^ (long)c;
		crc16 = (crc16 >> 8) ^
		  (crcta[(z & 0xF0) >> 4] ^ crctb[z & 0x0F]);
	    }
	}
#ifdef CK_CTRLZ
	lastchar = a;
#endif /* CK_CTRLZ */
    }
    return(0);
}
#endif /* CKTUNING */

/* decode() - full version */

int
#ifdef CK_ANSIC
decode(CHAR *buf, int (*fn)(char), int xlate)
#else
decode(buf,fn,xlate) register CHAR *buf; register int (*fn)(); int xlate;
#endif /* CK_ANSIC */
/* decode */ {
    register unsigned int a, a7, a8, b8; /* Various copies of current char */
    int t;				/* Int version of character */
    int ssflg;				/* Character was single-shifted */
    int ccpflg;				/* For Ctrl-unprefixing stats */
    int len;
    long z;
    CHAR c;
/*
  Catch the case in which we are asked to decode into a file that is not open,
  for example, if the user interrupted the transfer, but the other Kermit
  keeps sending.
*/
    if ((cxseen || czseen || discard) && (fn == putfil))
      return(0);

#ifdef COMMENT
#ifdef CKTUNING
    if (binary && !parity)
      return(bdecode(buf,fn));
#endif /* CKTUNING */
#endif /* COMMENT */
    debug(F100,"DECODE","",0);

    xdbuf = buf;			/* Make global copy of pointer. */
    rpt = 0;				/* Initialize repeat count. */

    len = rln;				/* Number of bytes in data field */
    while (len > 0) {			/* Loop for each byte */
        a = *xdbuf++ & 0xff;		/* Get next character */
	len--;
	if (a == rptq && rptflg) {	/* Got a repeat prefix? */
	    rpt = xunchar(*xdbuf++ & 0xFF); /* Yes, get the repeat count, */
	    rptn += rpt;
	    a = *xdbuf++ & 0xFF;	/* and get the prefixed character. */
	    len -= 2;
	}
	b8 = lsstate ? 0200 : 0;	/* 8th-bit value from SHIFT-STATE */
	if (ebqflg && a == ebq) {	/* Have 8th-bit prefix? */
	    b8 ^= 0200;			/* Yes, invert the 8th bit's value, */
	    ssflg = 1;			/* remember we did this, */
	    a = *xdbuf++ & 0xFF;	/* and get the prefixed character. */
	    len--;
	} else ssflg = 0;
	ccpflg = 0;
	if (a == ctlq) {		/* If control prefix, */
	    a  = *xdbuf++ & 0xFF;	/* get its operand */
	    len--;
	    a7 = a & 0x7F;		/* and its low 7 bits. */
	    if ((a7 >= 0100 && a7 <= 0137) || a7 == '?') { /* Controllify */
		a = ctl(a);		/* if in control range. */
		a7 = a & 0x7F;
		ccpflg = 1;		/* Note that we did this */
		ccp++;			/* Count for stats */
	    }
	} else a7 = a & 0x7f;		/* Not control quote */
	if (a7 < 32 || a7 == 127) {	/* Control character? */
	    if (!ccpflg) ccu++;		/* A bare one, count it */
	    if (lscapu) {		/* If doing locking shifts... */
		if (lsstate)		/* If SHIFTED */
		  a8 = (a & ~b8) & 0xFF; /* Invert meaning of 8th bit */
		else			/* otherwise */
		  a8 = a | b8;		/* OR in 8th bit */
		/* If we're not in a quoted sequence */
		if (!lsquote && (!lsstate || !ssflg)) {
		    if (a8 == DLE) {	/* Check for DLE quote */
			lsquote = 1;	/* prefixed by single shift! */
			continue;
		    } else if (a8 == SO) { /* Check for Shift-Out */
			lsstate = 1;	/* SHIFT-STATE = SHIFTED */
			continue;
		    } else if (a8 == SI) { /* or Shift-In */
			lsstate = 0;	/* SHIFT-STATE = UNSHIFTED */
			continue;
		    }
		} else lsquote = 0;
	    }
	}
	a |= b8;			/* OR in the 8th bit */
	if (rpt == 0) rpt = 1;		/* If no repeats, then one */
	if (!binary) {			/* If in text mode, */
	    if (feol && a == CR) continue; /* Convert CRLF to newline char */
    	    if (feol && a == LF) a = feol;

#ifndef NOCSETS				/* Character-set translation */
#ifdef KANJI				/* For Kanji transfers, */
	    if (tcharset != TC_JEUC)	/* postpone translation. */
#endif /* KANJI */
	      if (xlate && rx) a = (*rx)((CHAR) a); /* Translate charset */
#endif /* NOCSETS */
    	}
	if (fn == putfil) { /* (PWP) speedup via buffered output and a macro */
	    
	    for (; rpt > 0; rpt--) {	/* Output the char RPT times */
#ifdef CALIBRATE
		if (calibrate) {
		    ffc++;
		    continue;
		}
#endif /* CALIBRATE */
#ifndef NOCSETS
#ifdef KANJI
		if (!binary && tcharset == TC_JEUC &&
		    fcharset != FC_JEUC) { /* Translating from J-EUC */
		    if (ffc == 0L) xkanjf();
		    if (xkanji(a,fn) < 0)  /* to something else? */
		      return(-1);
		    else t = 1;
		} else
#endif /* KANJI */
#endif /* NOCSETS */
#ifdef OS2
		  if (xflg && !remfile) { /* Write to virtual screen */
		      char _a;
		      _a = a & fmask;
		      t = conoc(_a);
		      if (t < 1) t = -1;
		  } else
#endif /* OS2 */
		    t = zmchout(a & fmask); /* zmchout is a macro */
		if (t < 0) {
		    debug(F101,"decode write error - errno","",errno);
		    return(-1);
		}
		ffc++;			/* Count the character */
		if (docrc && !xflg && !remfile) { /* Update file CRC */
		    c = a;		/* Force conversion to unsigned char */
		    z = crc16 ^ (long)c;
		    crc16 = (crc16 >> 8) ^
		      (crcta[(z & 0xF0) >> 4] ^ crctb[z & 0x0F]);
		}
	    }
	} else {			/* Output to something else. */
	    a &= fmask;			/* Apply file mask */
	    for (; rpt > 0; rpt--) {	/* Output the char RPT times */
#ifdef CALIBRATE
		if (calibrate) {
		    ffc++;
		    continue;
		}
#endif /* CALIBRATE */
		if ((*fn)((char) a) < 0) return(-1); /* Send to output func. */
#ifdef COMMENT
/*
  This was causing the server to count extra bytes that were part of
  server command packets, like FINISH.
*/
		ffc++;
#endif /* COMMENT */
	    }
	}
#ifdef CK_CTRLZ
	lastchar = a;
#endif /* CK_CTRLZ */
    }
    return(0);
}

/*  G E T P K T -- Fill a packet data field  */

/*
 Gets characters from the current source -- file or memory string.
 Encodes the data into the packet, filling the packet optimally.
 Set first = 1 when calling for the first time on a given input stream
 (string or file).

 Call with:
 bufmax -- current send-packet size
 xlate  -- flag: 0 to skip character-set translation, 1 to translate

 Uses global variables:
 t     -- current character.
 first -- flag: 1 to start up, 0 for input in progress, -1 for EOF.
 next  -- next character (not used any more).
 data  -- pointer to the packet data buffer.
 size  -- number of characters in the data buffer.
 memstr - flag that input is coming from a memory string instead of a file.
 memptr - pointer to string in memory.
 (*sx)()  character set translation function

Returns the size as value of the function, and also sets global "size",
and fills (and null-terminates) the global data array.  Returns 0 upon eof.

Rewritten by Paul W. Placeway (PWP) of Ohio State University, March 1989.
Incorporates old getchx() and encode() inline to reduce function calls,
uses buffered input for much-improved efficiency, and clears up some
confusion with line termination (CRLF vs LF vs CR).

Rewritten again by Frank da Cruz to incorporate locking shift mechanism,
May 1991.
*/

/*
  Lookahead function to decide whether locking shift is worth it.  Looks at
  the next four input characters to see if all of their 8th bits match the
  argument.  Call with 0 or 0200.  Returns 1 if so, 0 if not.  If we don't
  happen to have at least 4 more characters waiting in the input buffer,
  returns 1.  Note that zinptr points two characters ahead of the current
  character because of repeat-count lookahead.
*/

#ifdef KANJI
int
kgetf(
#ifdef CK_ANSIC
      VOID
#endif /* CK_ANSIC */
      ) {
    if (funcstr)
      return((*funcptr)());
    else
      return(zminchar());
}

int
kgetm(
#ifdef CK_ANSIC
      VOID
#endif /* CK_ANSIC */
      ) {
    int x;
    if (x = *memptr++) return(x);
    else return(-1);
}
#endif /* KANJI */

int
lslook(b) unsigned int b; {		/* Locking Shift Lookahead */
    int i;
    if (zincnt < 3)			/* If not enough chars in buffer, */
      return(1);			/* force shift-state switch. */
    b &= 0200;				/* Force argument to proper form. */
    for (i = -1; i < 3; i++)		/* Look at next 5 characters to */
      if (((*(zinptr+i)) & 0200) != b)	/* see if all their 8th bits match.  */
	return(0);			/* They don't. */
    return(1);				/* They do. */
}

int
maxdata() {				/* Get maximum data length */
    int n, len;
    debug(F101,"maxdata spsiz 1","",spsiz);
    if (spsiz < 0)			/* How could this happen? */
      spsiz = DSPSIZ;
    debug(F101,"maxdata spsiz 2","",spsiz);
    n = spsiz - 5;			/* Space for Data and Checksum */
    if (n > 92 && n < 96) n = 92;	/* "Short" Long packets don't pay */
    if (n > 92 && lpcapu == 0)		/* If long packets needed, */
      n = 92;				/* make sure they've been negotiated */
    len = n - bctl;			/* Space for data */
    if (n > 92) len -= 3;		/* Long packet needs header chksum */
    debug(F101,"maxdata len 1","",len);
    if (len < 0) len = 10;
    debug(F101,"maxdata len 2","",len);
    return(len);
}

#ifdef CKTUNING
/*
  When CKTUNING is defined we use this special trimmed-down version of getpkt
  to speed up binary-mode no-parity transfers.  When CKTUNING is not defined,
  or for text-mode or parity transfers, we use the regular getpkt() function.
  Call just like getpkt() but test first for transfer mode and parity.  NOTE:
  This routine is only to be called when sending a real file -- not for
  filenames, server responses, etc, because it only reads from the input file.
  See getpkt() for more detailed commentary.
*/
static int
bgetpkt(bufmax) int bufmax; {
    register CHAR rt = t, rnext;
    register CHAR *dp, *odp, *p1, *p2;
    register int x = 0, a7;

    static CHAR leftover[9] = { '\0','\0','\0','\0','\0','\0','\0','\0','\0' };
    static int nleft = 0;

    CHAR xxrc, xxcq;			/* Pieces of prefixed sequence */

    long z;				/* A long worker (for CRC) */

    if (!binary || parity)		/* JUST IN CASE caller didn't test */
      return(getpkt(bufmax,!binary));
  
    if (!data) return(-1);

    bufmax = maxdata();			/* Get maximum data length */

#ifdef DEBUG
    if (deblog)
      debug(F101,"bgetpkt bufmax","",bufmax);
#endif /* DEBUG */

    if (first == 1) {			/* If first character of this file.. */
	ffc = 0L;			/* reset file character counter */
	first = 0;			/* Next character won't be first */
	*leftover = '\0';		/* Discard any interrupted leftovers */
	nleft = 0;

	/* Get first character of file into rt, watching out for null file */

#ifdef CALIBRATE
	if (calibrate) {
#ifdef NORANDOM
	    rt = 17;
#else
	    rt = cal_a[rand() & 0xff];
#endif /* NORANDOM */
	    first = 0;
	} else 
#endif /* CALIBRATE */
	if ((x = zminchar()) < 0) { /* End of file or input error */
	    first = -1;
	    size = 0;
	    if (x == -2) {		/* Error */
		debug(F100,"bgetpkt: input error","",0);
		cxseen = 1;		/* Interrupt the file transfer */
	    } else {
		debug(F100,"bgetpkt empty file","",0);
	    }
	    return(0);
	}
	ffc++;				/* Count a file character */
	rt = (CHAR) x;			/* Convert int to char */
	if (docrc && what == W_SEND) {	/* Accumulate file crc */
	    z = crc16 ^ (long)rt;
	    crc16 = (crc16 >> 8) ^
	      (crcta[(z & 0xF0) >> 4] ^ crctb[z & 0x0F]);
	}		
	rt &= fmask;			/* Apply SET FILE BYTESIZE mask */

    } else if (first == -1 && nleft == 0) { /* EOF from last time */

        return(size = 0);
    }
/*
  Here we handle characters that were encoded for the last packet but
  did not fit, and so were saved in the "leftover" array.
*/
    dp = data;				/* Point to packet data buffer */
    if (nleft) {
	for (p1 = leftover; nleft > 0; nleft--) /* Copy leftovers */
	  *dp++ = *p1++;
	*leftover = '\0';		/* Delete leftovers */
	nleft = 0;
    }
    if (first == -1)			/* Handle EOF */
      return(size = (dp - data));
  
/* Now fill up the rest of the packet. */

    rpt = 0;				/* Initialize character repeat count */

    while (first > -1) {		/* Until EOF... */
#ifdef CALIBRATE
	if (calibrate) {		/* We generate our own "file" */
	    if (ffc >= calibrate) {	/* EOF */
		first = -1;
		ffc--;
	    } else {			/* Generate next character */
		if (cal_j > CAL_M * ffc)
		  cal_j = cal_a[ffc & 0xff]; 
		x = (unsigned)cal_a[(cal_j & 0xff)];
		if (x == rt) x ^= 2;
	    }
	    ffc++;
	    cal_j += (unsigned int)(ffc + CAL_O);
	} else
#endif /* CALIBRATE */
	if ((x = zminchar()) < 0) {	/* Check for EOF */
	    first = -1;			/* Flag eof for next time. */
	    if (x == -2) cxseen = 1;	/* If error, cancel this file. */
	} else {
	    ffc++;			/* Count the character */
	    if (docrc && what == W_SEND) { /* Accumulate file crc */
		z = crc16 ^ (long)((CHAR)x & 0xff);
		crc16 = (crc16 >> 8) ^
		  (crcta[(z & 0xF0) >> 4] ^ crctb[z & 0x0F]);
	    } 
	}
	rnext = (CHAR) (x & fmask);	/* Apply file mask */
/*
  At this point, the character we just read is in rnext,
  and the character we are about to encode into the packet is in rt.
*/
	odp = dp;			/* Remember where we started. */
	xxrc = xxcq = NUL;		/* Clear these. */
/*
  Now encode the character according to the options that are in effect:
    ctlp[]: whether this control character needs prefixing.
    rptflg: repeat counts enabled.
    Other options don't apply in this routine.
*/
	if (rptflg && rt == rnext && first == 0) { /* Got a run... */
	    if (++rpt < 94) {		/* Below max, just count */
		continue;		/* go back and get another */
	    } else if (rpt == 94) {	/* Reached max, must dump */
		xxrc = (CHAR) tochar(rpt); /* Put the repeat count here */
		rptn += rpt;		/* Accumulate it for statistics */
		rpt = 0;		/* And reset it */
	    }
	} else if (rpt > 0) {		/* End of run */
	    xxrc = (CHAR)tochar(++rpt); /* The count */
	    rptn += rpt;		/* For stats */
	    rpt = 0;			/* Reset repeat count */
	}
	a7 = rt & 0177;			/* Get low 7 bits of character */
	if (
#ifdef CK_SPEED
	    ctlp[(rt & 0xff)]		/* Lop off any "sign" extension */
#else
	    (a7 < SP) || (a7 == DEL)
#endif /* CK_SPEED */
	    ) {				/* Do control prefixing if necessary */
	    xxcq = myctlq;		/* The prefix */
	    ccp++;			/* Count it */
	    rt = (CHAR) ctl(rt);	/* Uncontrollify the character */
	}
#ifdef CK_SPEED
	else if ((a7 < SP) || (a7 == DEL)) /* Count an unprefixed one */
	  ccu++;
#endif /* CK_SPEED */

	if (a7 == myctlq)		/* Always prefix the control prefix */
	  xxcq = myctlq;

	if ((rptflg) && (a7 == rptq))	/* If it's the repeat prefix, */
	  xxcq = myctlq;		/* prefix it if doing repeat counts */

/* Now construct the prefixed sequence */

	if (xxrc) {			/* Repeat count */
	    if (xxrc == (CHAR) '"' && !xxcq) { /* 2 in a row & not prefixed */
		*dp++ = rt;		/* So just do this */
	    } else {			/* More than two or prefixed */
		*dp++ = (CHAR) rptq; *dp++ = xxrc; /* Emit repeat sequence */
	    }
	}
	if (xxcq) { *dp++ = myctlq; }	/* Control prefix */
	*dp++ = rt;			/* Finally, the character itself */
	rt = rnext;			/* Next character is now current. */

/* Done encoding the character.  Now take care of packet buffer overflow. */

	size = dp - data;		/* How many bytes we put in buffer. */
	if (size >= bufmax) {		/* If too big, save some for next. */
	    *dp = '\0';			/* Mark the end. */
	    if (size > bufmax) {	/* if packet is overfull */
		/* Copy the part that doesn't fit into the leftover buffer, */
		/* taking care not to split a prefixed sequence. */
		int i;
		nleft = dp - odp;
		p1 = leftover;
		p2 = odp;
		for (i = 0; i < nleft; i++)
		  *p1++ = *p2++;
		size = odp - data;	/* Return truncated packet. */
		*odp = '\0';		/* Mark the new end */
	    }
	    t = rt;			/* Save for next time */
	    return(size);
	}
    }					/* Otherwise, keep filling. */
    size = dp - data;			/* End of file */
    *dp = '\0';				/* Mark the end of the data. */
    return(size);		     /* Return partially filled last packet. */
}
#endif /* CKTUNING */

/*  G E T P K T  --  Fill a packet data field from the indicated source.  */

/*
  Parameters:
    bufmax: Maximum length of entire packet.
    xlate:  Flag for whether to translate charsets when in text mode.
  Returns:  Number of characters written to packet data field, 0 or more,
            Or -1 on failure.

  This is the full version allowing for parity and text-mode conversions;
  i.e. it works in all cases.   Also see bgetpkt(), a special lean/mean/fast
  packet encoder that works only for binary-mode no-parity transfers.
*/
int
getpkt(bufmax,xlate) int bufmax, xlate; { /* Fill one packet buffer */
    register CHAR rt = t, rnext = NUL;	  /* Register shadows of the globals */
    register CHAR *dp, *odp, *odp2, *p1, *p2; /* pointers... */
    register int x;			/* Loop index. */
    register int a7;			/* Low 7 bits of character */

    static CHAR leftover[9] = { '\0','\0','\0','\0','\0','\0','\0','\0','\0' };
    static int nleft = 0;

    CHAR xxls, xxdl, xxrc, xxss, xxcq;	/* Pieces of prefixed sequence */

    long z;				/* A long worker (for CRC) */
/*
  Assume bufmax is the receiver's total receive-packet buffer length.
  Our whole packet has to fit into it, so we adjust the data field length.
  We also decide optimally whether it is better to use a short-format or
  long-format packet when we're near the borderline.
*/
    if (!data) return(-1);

    bufmax = maxdata();			/* Get maximum data length */

    if (first == 1) {			/* If first character of this file.. */
	if (!memstr && !funcstr)	/* and real file... */
	  ffc = 0L;			/* reset file character counter */
	first = 0;			/* Next character won't be first */
	*leftover = '\0';		/* Discard any interrupted leftovers */
	nleft = 0;

	/* Get first character of file into rt, watching out for null file */

#ifdef CALIBRATE
	if (calibrate && !memstr) {
#ifdef NORANDOM
	    x = rt = 53;
#else
	    x = rt = cal_a[rand() & 0xff];
#endif /* NORANDOM */
	    first = 0;
	    ffc++;
	} else 
#endif /* CALIBRATE */
#ifndef NOCSETS
#ifdef KANJI
	if (!binary && tcharset == TC_JEUC && xlate) {
	    x = zkanjf();
	    if ((x = zkanji(memstr ? kgetm : kgetf)) < 0) {
	        first = -1;
	        size = 0;
	        if (x == -2) {
	            debug(F100,"getpkt zkanji: input error","",0);
	            cxseen = 1;
	        } else debug(F100,"getpkt zkanji: empty string/file","",0);
	        return(0);
	    }
	    rt = x;
	    if (!memstr) {
		ffc++;
		if (docrc && what == W_SEND) { /* Accumulate file crc */
		    z = crc16 ^ (long)rt;
		    crc16 = (crc16 >> 8) ^
		      (crcta[(z & 0xF0) >> 4] ^ crctb[z & 0x0F]);
		}		
	    }
	} else {
#endif /* KANJI */
#endif /* not NOCSETS */
	if (memstr) {			/* Reading data from memory string */
	    if ((rt = *memptr++) == '\0') { /* end of string ==> EOF */
		first = -1;
	        size = 0;
		debug(F100,"getpkt: empty string","",0);
		return (0);
	    }

	} else if (funcstr) {		/* Reading data from a function */

	    if ((x = (*funcptr)()) < 0) { /* End of input  */
		first = -1;
	        size = 0;
		debug(F100,"getpkt: empty input function","",0); /* Empty */
		return(0);
	    }
	    ffc++;			/* Count a file character */
	    rt = (CHAR) x;		/* Convert int to char */
	    debug(F101,"getpkt funcstr","",rt);

	} else {			/* Reading data from a file */

	    if ((x = zminchar()) < 0) { /* End of file or input error */
		first = -1;
	        size = 0;
		if (x == -2) {		/* Error */
		    debug(F100,"getpkt: input error","",0);
		    cxseen = 1;		/* Interrupt the file transfer */
		} else {
		    debug(F100,"getpkt empty file","",0);
		}
		return(0);
	    }
#ifdef CK_CTRLZ
	    else if (x == 26 && eofmethod == XYEOF_Z && !binary) { /* Ctrl-Z */
		first == -1;
		size = 0;
		debug(F100,"getpkt: EOF on Ctrl-Z 1","",0);
		return(0);
	    }
#endif /* CK_CTRLZ */
	    ffc++;			/* Count a file character */
	    rt = (CHAR) x;		/* Convert int to char */
#ifdef DEBUG
	    if (deblog)
	      debug(F101,"getpkt 1st char","",rt);
#endif /* DEBUG */
	    if (docrc && what == W_SEND) {	/* Accumulate file crc */
		z = crc16 ^ (long)rt;
		crc16 = (crc16 >> 8) ^
		  (crcta[(z & 0xF0) >> 4] ^ crctb[z & 0x0F]);
	    }		
	}
#ifndef NOCSETS
#ifdef KANJI
	}
#endif /* KANJI */
#endif /* not NOCSETS */

	rt &= fmask;			/* Apply SET FILE BYTESIZE mask */
#ifdef DEBUG
	if (deblog) {
	    debug(F101,"getpkt fmask","",fmask);
	    debug(F101,"getpkt new rt","",rt);
	}
#endif /* DEBUG */
#ifndef NOCSETS
#ifdef DEBUG
	if (xlate && deblog) {
	    debug(F101,"getpkt about to call translate function","",rt);
	    debug(F101,"tcharset","",tcharset);
	    debug(F101,"fcharset","",fcharset);
	}
#endif /* DEBUG */
#ifdef KANJI
	if (tcharset != TC_JEUC)
#endif /* KANJI */
	  if (!binary && sx && xlate) {
	      rt = (*sx)(rt); /* Translate */
#ifdef DEBUG
	      if (deblog)
		debug(F101," translate function returns","",rt);
#endif /* DEBUG */
	  }
#endif /* not NOCSETS */

	/* PWP: handling of feol is done later (in the while loop)... */

    } else if ((first == -1) && (nleft == 0)) { /* EOF from last time */
#ifdef DEBUG
	if (deblog) {
	    debug(F101,"getpkt eof crc16","",crc16);
	    debug(F101,"getpkt eof ffc","",ffc);
	}
#endif /* DEBUG */
        return(size = 0);
    }
/*
  Here we handle characters that were encoded for the last packet but
  did not fit, and so were saved in the "leftover" array.
*/
    dp = data;				/* Point to packet data buffer */
    if (nleft) {
	for (p1 = leftover; nleft > 0; nleft--) /* Copy leftovers */
	  *dp++ = *p1++;
	*leftover = '\0';			/* Delete leftovers */
	nleft = 0;
    }
    if (first == -1)			/* Handle EOF */
      return(size = (dp - data));
  
/* Now fill up the rest of the packet. */

    rpt = 0;				/* Initialize character repeat count */

    while (first > -1) {		/* Until EOF... */
#ifdef CALIBRATE
	if (calibrate && !memstr) {	/* We generate our own "file" */
	    if (ffc >= calibrate) {	/* EOF */
		first = -1;
		ffc--;
	    } else {			/* Generate next character */
		if (cal_j > CAL_M * ffc)
		  cal_j = cal_a[ffc & 0xff]; 
		x = (unsigned)cal_a[(cal_j & 0xff)];
		if (x == rt) x ^= 2;
	    }
	    cal_j += (unsigned int)(ffc + CAL_O);
	    ffc++;
	} else
#endif /* CALIBRATE */
#ifdef KANJI
	if (!binary && xlate && tcharset == TC_JEUC) {
	    if ((x = zkanji(memstr ? kgetm : kgetf)) < 0) {
	        first = -1;
	        if (x == -2) cxseen = 1;
	    } else if (!memstr) ffc++;
	    rnext = (CHAR) (x & fmask);
	} else {
#endif /* KANJI */
	    if (memstr) {		/* Get next char from memory string */
		if ((x = *memptr++) == '\0') /* End of string means EOF */
		  first = -1;		/* Flag EOF for next time. */
		rnext = (CHAR) (x & fmask); /* Apply file mask */
	    } else if (funcstr) {	/* Get next char from function */
		if ((x = (*funcptr)()) < 0) /* End of string means EOF */
		  first = -1;		/* Flag EOF for next time. */
		rnext = (CHAR) (x & fmask); /* Apply file mask */
	    } else {
		/* Real file, get next character */
		if ((x = zminchar()) < 0) { /* Check for EOF */
		    first = -1;		/* Flag eof for next time. */
		    if (x == -2) cxseen = 1; /* If error, cancel this file. */
		}
#ifdef CK_CTRLZ
		/* Or signal EOF on Ctrl-Z if elected */
		else if (x == 26 && eofmethod == XYEOF_Z && !binary) {
		    debug(F100,"getpkt: EOF on Ctrl-Z 2","",0);
		    first = -1;
		}
#endif /* CK_CTRLZ */
		else {
		    ffc++;		/* Count the character */
		    if (docrc && what == W_SEND) { /* Accumulate file crc */
			z = crc16 ^ (long)((CHAR)x & 0xff);
			crc16 = (crc16 >> 8) ^
			  (crcta[(z & 0xF0) >> 4] ^ crctb[z & 0x0F]);
		    } 
		}
		rnext = (CHAR) (x & fmask);	/* Apply file mask */
	    }
#ifdef KANJI
	}
#endif /* KANJI */
/*
  At this point, the character we just read is in rnext,
  and the character we are about to encode into the packet is in rt.
*/
#ifndef NOCSETS
#ifdef KANJI
	if (tcharset != TC_JEUC)
#endif /* KANJI */
	    if (!binary && sx && xlate) {
		rnext = (*sx)(rnext); /* Translate */
#ifdef DEBUG
		if (deblog)
		  debug(F101,"getpkt xlated rnext to","",rnext);
#endif /* DEBUG */
	    }
#endif /* not NOCSETS */

	odp = dp;			/* Remember where we started. */
	xxls = xxdl = xxrc = xxss = xxcq = NUL;	/* Clear these. */

/*
  Now encode the character according to the options that are in effect:
    ctlp[]: whether this control character needs prefixing.
    binary: text or binary mode.
    rptflg: repeat counts enabled.
    ebqflg: 8th-bit prefixing enabled.
    lscapu: locking shifts enabled.
*/
	if (rptflg) {			/* Repeat processing is on? */
	    if (
		/*
		 * If the next char is really CRLF, then we cannot
		 * be doing a repeat (unless CR,CR,LF which becomes
		 * "~ <n-1> CR CR LF", which is OK but not most efficient).
		 * I just plain don't worry about this case.  The actual
		 * conversion from NL to CRLF is done after the rptflg if...
		 */
	    (!feol || binary || (feol && (rnext != feol))) &&
	    (rt == rnext) && (first == 0)) { /* Got a run... */
		if (++rpt < 94) {	/* Below max, just count */
		    continue;		/* go back and get another */
		} else if (rpt == 94) {	/* Reached max, must dump */
		    xxrc = (CHAR) tochar(rpt); /* Put the repeat count here */
		    rptn += rpt;	/* Accumulate it for statistics */
		    rpt = 0;		/* And reset it */
		}
	    } else if (rpt > 1) {	/* More than two */
		xxrc = (CHAR) tochar(++rpt); /* and count. */
		rptn += rpt;
		rpt = 0;		/* Reset repeat counter. */
	    }
	    /*
	      If (rpt == 1) we must encode exactly two characters.
	      This is done later, after the first character is encoded.
	    */
	}
	if (!binary && feol && (rt == feol)) { /* It's the newline character */
	    if (lscapu && lsstate) {	/* If SHIFT-STATE is SHIFTED */
		if (ebqflg) {		/* If single shifts enabled, */
		    *dp++ = (CHAR) ebq;	/* insert a single shift. */
		} else {		/* Otherwise must shift in. */
		    *dp++ = myctlq;	/* Insert shift-out code */
		    *dp++ = 'O';
		    lsstate = 0;	/* Change shift state */
		}
	    }
#ifdef CK_SPEED
	    if (ctlp[CR]) {
		*dp++ = myctlq;		/* Insert carriage return directly */
		*dp++ = 'M';
		ccp++;
	    } else {
		*dp++ = CR;		/* Perhaps literally */
		ccu++;
	    }
#else /* !CK_SPEED */
	    *dp++ = myctlq;		/* Insert carriage return directly */
	    *dp++ = 'M';
	    ccp++;
#endif /* CK_SPEED */
	    rt = LF;			/* Now make next char be linefeed. */
	}
/*
  Now handle the 8th bit of the file character.  If we have an 8-bit
  connection, we preserve the 8th bit.  If we have a 7-bit connection,
  we employ either single or locking shifts (if they are enabled).
*/
	a7 = rt & 0177;			/* Get low 7 bits of character */
	if (rt & 0200) {		/* 8-bit character? */
	    if (lscapu) {		/* Locking shifts enabled? */
		if (!lsstate) {		/* Not currently shifted? */
		    x = lslook(0200);	/* Look ahead */
		    if (x != 0 || ebqflg == 0) { /* Locking shift decision */
			xxls = 'N';	   /* Need locking shift-out */
			lsstate = 1;	   /* and change to shifted state */
		    } else if (ebqflg) {   /* Not worth it */
			xxss = (CHAR) ebq; /* Use single shift */
		    }
		}
		rt = (CHAR) a7;		/* Replace character by 7-bit value */
	    } else if (ebqflg) {	/* 8th bit prefixing is on? */
		xxss = (CHAR) ebq;	/* Insert single shift */
		rt = (CHAR) a7;		/* Replace character by 7-bit value */
	    }
/*
  In case we have a 7-bit connection and this is an 8-bit character, AND
  neither locking shifts nor single shifts are enabled, then the character's
  8th bit will be destroyed in transmission, and a block check error will
  occur.
*/
	} else if (lscapu) {		/* 7-bit character */

	    if (lsstate) {		/* Comes while shifted out? */
		x = lslook(0);		/* Yes, look ahead */
		if (x || ebqflg == 0) {	/* Time to shift in. */
		    xxls = 'O';		/* Set shift-in code */
		    lsstate = 0;	/* Exit shifted state */
		} else if (ebqflg) {	/* Not worth it, stay shifted out */
		    xxss = (CHAR) ebq;	/* Insert single shift */
		}
	    }
	}
	/* If data character is significant to locking shift protocol... */
	if (lscapu && (a7 == SO || a7 == SI || a7 == DLE))
	  xxdl = 'P';			/* Insert datalink escape */

	if (
#ifdef CK_SPEED
	    /*
	      Thwart YET ANOTHER unwanted, unneeded, and unloved sign
	      extension.  This one was particularly nasty because it prevented
	      255 (Telnet IAC) from being prefixed on some platforms -- e.g.
	      VMS with VAX C -- but not others, thus causing file transfers to
	      fail on Telnet connections by sending bare IACs.  Not to mention
	      the stray memory reference.  Whoever thought it was a good idea
	      for characters to be signed (by default or at all) should have
	      thunk again.  (Sep 96)
	    */
	    ctlp[(rt & 0xff)]		/* Lop off any "sign" extension */
#else
	    (a7 < SP) || (a7 == DEL)
#endif /* CK_SPEED */
	    ) {				/* Do control prefixing if necessary */
	    xxcq = myctlq;		/* The prefix */
	    ccp++;			/* Count it */
	    rt = (CHAR) ctl(rt);	/* Uncontrollify the character */
	}
#ifdef CK_SPEED
	else if ((a7 < SP) || (a7 == DEL)) /* Count an unprefixed one */
	  ccu++;
#endif /* CK_SPEED */

	if (a7 == myctlq)		/* Always prefix the control prefix */
	  xxcq = myctlq;

	if ((rptflg) && (a7 == rptq))	/* If it's the repeat prefix, */
	  xxcq = myctlq;		/* prefix it if doing repeat counts */

	if ((ebqflg) && (a7 == ebq))	/* Prefix the 8th-bit prefix */
	  xxcq = myctlq;		/* if doing 8th-bit prefixes */

/* Now construct the entire sequence */

	if (xxls) { *dp++ = myctlq; *dp++ = xxls; } /* Locking shift */
	odp2 = dp;				    /* (Save this place) */
	if (xxdl) { *dp++ = myctlq; *dp++ = xxdl; } /* Datalink escape */
	if (xxrc) { *dp++ = (CHAR) rptq; *dp++ = xxrc; } /* Repeat count */
	if (xxss) { *dp++ = (CHAR) ebq; }           /* Single shift */
	if (xxcq) { *dp++ = myctlq; }	    	    /* Control prefix */
	*dp++ = rt;			/* Finally, the character itself */

	if (rpt == 1) {			/* Exactly two copies? */
	    rpt = 0;
	    p2 = dp;			/* Save place temporarily */
	    for (p1 = odp2; p1 < p2; p1++) /* Copy the old chars over again */
	      *dp++ = *p1;
	    if ((p2-data) <= bufmax) odp = p2; /* Check packet bounds */
	}
	rt = rnext;			/* Next character is now current. */

/* Done encoding the character.  Now take care of packet buffer overflow. */

	if ((dp-data) >= bufmax) {	/* If too big, save some for next. */
	    size = (dp-data);		/* Calculate the size. */
	    *dp = '\0';			/* Mark the end. */
	    if ((dp-data) > bufmax) {	/* if packet is overfull */
		/* copy the part that doesn't fit into the leftover buffer, */
		/* taking care not to split a prefixed sequence. */
		int i;
		nleft = dp - odp;
#ifdef COMMENT
		for (p1 = leftover, p2 = odp; (*p1 = *p2) != '\0'; p1++, p2++)
		  ;
#else
		for (i = 0, p1 = leftover, p2 = odp; i < nleft; i++)
		  *p1++ = *p2++;
#endif /* COMMENT */
		debug(F111,"getpkt leftover",leftover,size);
		debug(F101,"getpkt osize","",(odp-data));
		size = (odp-data);	/* Return truncated packet. */
		*odp = '\0';		/* Mark the new end */
	    }
	    t = rt;			/* Save for next time */
	    return(size);
	}
    }					/* Otherwise, keep filling. */
    size = (dp-data);			/* End of file */
    *dp = '\0';				/* Mark the end of the data. */
    debug(F111,"getpkt eof/eot",data,size); /* Fell thru before packet full, */
    return(size);		     /* return partially filled last packet. */
}

/*  T I N I T  --  Initialize a transaction  */

int epktrcvd = 0, epktsent = 0;

int
tinit() {
    int x;
#ifdef CK_TIMERS
    extern int rttflg;
#else
    extern int rcvtimo;
#endif /* CK_TIMERS */
    extern char whoareu[];

    *epktmsg = NUL;
    epktrcvd = 0;
    epktsent = 0;
    ofperms = "";

    if (server) {
	moving  = 0;
#ifdef PIPESEND
	pipesend = 0; /* This takes care of multiple GETs sent to a server. */
#endif /* PIPESEND */
    }
    bestlen = 0;
    maxsend = 0;
    debug(F101,"tinit bestlen","",bestlen);
    whoareu[0] = NUL;
    debug(F100,"tinit setting justone=0","",0);
    justone = 0;
#ifdef STREAMING
    streamok = 0;			/* Streaming negotiated */
    streaming = 0;			/* Streaming being done now */
#endif /* STREAMING */

#ifndef NOCSETS
    if (tcharset == TC_TRANSP) {	/* Character set translation */
	rx = sx = NULL;			/* Transparent, no translation */
#ifdef KANJI
    } else if (tcharset == TC_JEUC) {
	rx = sx = NULL;			/* Transparent, no translation */      
#endif /* KANJI */
    } else {				/* otherwise */
	rx = xlr[tcharset][fcharset];	/* Input translation function */
	sx = xls[tcharset][fcharset];	/* Output translation function */
    }
    debug(F101,"tinit tcharset","",tcharset);
    debug(F101,"tinit fcharset","",fcharset);
#ifdef COMMENT
    debug(F101,"tinit sx   ","",sx);
    debug(F101,"tinit rx   ","",rx);
#endif /* COMMENT */
#endif /* NOCSETS */
    myinit[0] = '\0';			/* Haven't sent init string yet */
    retrans = 0;			/* Packet retransmission count */
    sndtyp = 0;				/* No previous packet */
    xflg = 0;				/* Reset x-packet flag */
    memstr = 0;				/* Reset memory-string flag */
    memptr = NULL;			/*  and buffer pointer */
    funcstr = 0;			/* Reset "read from function" flag */
    funcptr = NULL;			/*  and function pointer */
    bctu = bctl = 1;			/* Reset block check type to 1 */
    autopar = 0;			/* Automatic parity detection flag */
    rqf = -1;				/* Reset 8th-bit-quote request flag */
    ebq = MYEBQ;			/* Reset 8th-bit quoting stuff */
    ebqflg = 0;				/* 8th bit quoting not enabled */
    ebqsent = 0;			/* No 8th-bit prefix bid sent yet */
    sq = 'Y';				/* 8th-bit prefix bid I usually send */
    fncnv = f_save;			/* Back to what user last said */
    binary = b_save;			/* ... */
    gnf_binary = binary;		/* Per-file transfer mode */
    pktnum = 0;				/* Initial packet number to send */
    cxseen = czseen = discard = 0;	/* Reset interrupt flags */
    *filnam = '\0';			/* Clear file name */
    spktl = 0;				/* And its length */
    nakstate = 0;			/* Assume not in a NAK'ing state */
    numerrs = 0;			/* Transmission error counter */
    if (server) 			/* If acting as server, */
      timint = srvtim;			/* Use server timeout interval. */
    else				/* Otherwise */
      timint = chktimo(rtimo,timef);	/* Begin by using local value */
    debug(F101,"tinit timint","",timint);

#ifdef CK_TIMERS
    if (rttflg && timint > 0)		/* Using round-trip timers? */
      rttinit();
#else
    rcvtimo = timint;
#endif /* CK_TIMERS */

    spsiz = spsizr;			/* Initial send-packet size */
    debug(F101,"tinit spsiz","",spsiz);
    wslots = 1;				/* One window slot */
    wslotn = 1;				/* No window negotiated yet */
    winlo = 0;				/* Packet 0 is at window-low */
    x = mksbuf(1);			/* Make a 1-slot send-packet buffer */
    if (x < 0) return(x);
    x = getsbuf(0);			/* Allocate first send-buffer. */
    debug(F101,"tinit getsbuf","",x);
    if (x < 0) return(x);
    dumpsbuf();
    x = mkrbuf(wslots);			/* & a 1-slot receive-packet buffer. */
    if (x < 0) return(x);
    what = W_INIT;			/* Doing nothing so far... */
    lsstate = 0;			/* Initialize locking shift state */
    whatru = 0;
    return(0);
}

VOID
pktinit() {				/* Initialize packet sequence */
    pktnum = 0;				/* number & window low. */
    winlo = 0;
}

/*  R I N I T  --  Respond to S or I packet  */

VOID
rinit(d) CHAR *d; {
    char *tp;
    ztime(&tp);
    tlog(F110,"Transaction begins",tp,0L); /* Make transaction log entry */
    tlog(F110,"Global file mode:", binary ? "binary" : "text", 0L);
    tlog(F110,"Collision action:", fncnam[fncact],0);
    tlog(F100,"","",0);
    filcnt = filrej = 0;		/* Init file counters */
    spar(d);
    ack1(rpar());
#ifdef datageneral
    if ((local) && (!quiet))            /* Only do this if local & not quiet */
        consta_mt();                    /* Start the asynch read task */
#endif /* datageneral */
}


/*  R E S E T C  --  Reset per-transaction character counters */

VOID
resetc() {
    rptn = 0;				/* Repeat counts */
    fsecs = flci = flco = 0L;		/* File chars in and out */
#ifdef GFTIMER
    fpfsecs = 0.0;
#endif /* GFTIMER */
    tfc = tlci = tlco = 0L;		/* Total file, line chars in & out */
    ccu = ccp = 0L;			/* Control-char statistics */
#ifdef COMMENT
    fsize = -1L;			/* File size */
#else
    if (what != W_SEND)
      fsize = -1L;
    debug(F101,"resetc fsize","",fsize);
#endif /* COMMENT */
    timeouts = retrans = 0;		/* Timeouts, retransmissions */
    spackets = rpackets = 0;		/* Packet counts out & in */
    crunched = 0;			/* Crunched packets */
    wcur = 0;				/* Current window size */
    wmax = 0;				/* Maximum window size used */
    peakcps = 0;                        /* Peak chars per second */
}

/*  S I N I T  --  Get & verify first file name, then send Send-Init packet */
/*
 Returns:
   1 if send operation begins successfully
   0 if send operation fails
*/
#ifdef DYNAMIC
char *cmargbuf = NULL;
#else
char cmargbuf[CKMAXPATH+1];
#endif /* DYNAMIC */
char *cmargp[2];

VOID
fnlist() {
    if (!calibrate)
      sndsrc = (nfils < 0) ? -1 : nfils; /* Source for filenames */
#ifdef DYNAMIC
    if (!cmargbuf && !(cmargbuf = malloc(CKMAXPATH+1)))
      fatal("fnlist: no memory for cmargbuf");
#endif /* DYNAMIC */
    cmargbuf[0] = NUL;			/* Initialize name buffer */

    debug(F101,"fnlist nfils","",nfils);
    debug(F110,"fnlist cmarg",cmarg,0);
    debug(F110,"fnlist cmarg2",cmarg2,0);
    if (!cmarg2) cmarg2 = "";
    if (nfils == 0) {			/* Sending from stdin or memory. */
	if ((cmarg2 != NULL) && (*cmarg2)) {
	    cmarg = cmarg2;		/* If F packet, "as-name" is used */
	    cmarg2 = "";		/* if provided */
	} else
	  cmarg = "stdin";		/* otherwise just use "stdin" */
	strcpy(cmargbuf,cmarg);
	cmargp[0] = cmargbuf;
	cmargp[1] = "";
	cmlist = cmargp;
	nfils = 1;
    }
}

int
sinit() {
    int x;				/* Worker int */
    char *tp, *xp, *m;			/* Worker string pointers */
/*
  The DECC prototype for sleep() moved from <signal.h> to <unistd.h>
  in DECC 5.2, but rather than pull in an entire header file (that might
  not even be there), potentially involving unwanted conflicts, just do this:
*/  
#ifdef __DECC
#ifndef __DECC_VER			/* This exists only in 5.0 and above */
    _PROTOTYP( int sleep, (unsigned) );
#else
#if __DECC_VER < 50200000
    _PROTOTYP( int sleep, (unsigned) );
#endif /* __DECC_VER */
#endif /* __DECC_VER */
#endif /* __DECC */

    filcnt = filrej = 0;		/* Initialize file counters */

    fnlist();

    xp = "";
    if (nfils < 0) {
#ifdef PIPESEND
	if (usepipes && protocol == PROTO_K && *cmarg == '!') {
	    pipesend = 1;
	    cmarg++;
	}
#endif /* PIPESEND */
	xp = cmarg;
    } else {
#ifndef NOMSEND
	if (addlist)
	  xp = filehead->fl_name;
	else
#endif /* NOMSEND */
	  if (filefile)
	    xp = filefile;
	  else if (calibrate)
	    xp = "Calibration";
	  else
	    xp = *cmlist;
    }
    debug(F110,"sinit xp",xp,0);
    x = gnfile();			/* Get first filename. */
    m = NULL;				/* Error message pointer */
    debug(F101,"sinit gnfil","",x);
    switch (x) {
      case -5: m = "Too many files match wildcard"; break;
      case -4: m = "Cancelled"; break;
      case -3: m = "Read access denied"; break;
      case -2: m = "File is not readable"; break;
      case -1: m = iswild(filnam) ? "No files match" : "File not found";
	break;
      case  0: m = "No filespec given!"; break;
      default:
	break;
    }
    debug(F101,"sinit nfils","",nfils);
    debug(F110,"sinit filnam",filnam,0);
    if (x < 1) {			/* Didn't get a file. */
	if (server)			/* Doing GET command */
	  errpkt((CHAR *)m);		/* so send Error packet. */
	else				/* Doing SEND command */
	  xxscreen(SCR_EM,0,0l,m);	/* so print message. */
	tlog(F110,xp,m,0L);		/* Make transaction log entry. */
	freerbuf(rseqtbl[0]);		/* Free the buffer the GET came in. */
	return(0);			/* Return failure code */
    }
    if (!local && !server && ckdelay > 0) /* OS-9 sleep(0) == infinite */
      sleep(ckdelay);			/* Delay if requested */
#ifdef datageneral
    if ((local) && (!quiet))            /* Only do this if local & not quiet */
        consta_mt();                    /* Start the asynch read task */
#endif /* datageneral */
    freerbuf(rseqtbl[0]);		/* Free the buffer the GET came in. */
    sipkt('S');				/* Send the Send-Init packet. */
    ztime(&tp);				/* Get current date/time */
    tlog(F110,"Transaction begins",tp,0L); /* Make transaction log entry */
    tlog(F110,"Global file mode:", binary ? "binary" : "text", 0L);
    tlog(F100,"","",0);
    debug(F111,"sinit ok",filnam,0);
    return(1);
}

int
#ifdef CK_ANSIC
sipkt(char c)				/* Send S or I packet. */
#else
sipkt(c) char c;
#endif
/* sipkt */ {
    CHAR *rp; int k, x;
    debug(F101,"sipkt pktnum","",pktnum);
    k = sseqtbl[pktnum];		/* Find slot for this packet */
    debug(F101,"sipkt k","",k);
    if (k < 0) {			/* No slot? */
	k = getsbuf(winlo = pktnum);	/* Make one. */
	debug(F101,"sipkt getsbuf","",k);    
    }
    ttflui();				/* Flush pending input. */
    rp = rpar();			/* Get protocol parameters. */
    x = spack(c,pktnum,(int)strlen((char *)rp),rp); /* Send them. */
    return(x);
}

/*  X S I N I T  --  Retransmit S-packet  */
/*
  For use in the GET-SEND sequence, when we start to send, but receive another
  copy of the GET command because the receiver didn't get our S packet.
  This retransmits the S packet and frees the receive buffer for the ACK.
  This special case is necessary because packet number zero is being re-used.
*/
VOID
xsinit() {
    int k;
    k = rseqtbl[0];
    debug(F101,"xsinit k","",k);
    if (k > -1)
    freerbuf(k);
    resend(0);
}

/*  R C V F I L -- Receive a file  */

/*
  Incoming filename is in data field of F packet.
  This function decodes it into the srvcmd buffer, substituting an
  alternate "as-name", if one was given.
  Then it does any requested transformations (like converting to
  lowercase), and finally if a file of the same name already exists, 
  takes the desired collision action.
  Returns 0 on failure, 1 on success.
*/
char ofn1[CKMAXPATH+1];			/* Buffer for output file name */
char * ofn2;				/* Pointer to backup file name */
int ofn1x;				/* Flag output file already exists */
int opnerr;				/* Flag for open error */

int					/* Returns success ? 1 : 0 */
rcvfil(n) char *n; {
    extern int en_cwd;
    char * n2;
#ifdef OS2ONLY
    char *zs, *longname, *newlongname, *pn; /* OS/2 long name items */
#endif /* OS2ONLY */
#ifdef DTILDE
    char *dirp;
#endif /* DTILDE */
    int dirflg, x, y;
#ifdef PIPESEND
    extern char * rcvfilter;
#endif /* PIPESEND */
    extern char * rrfspec;
#ifdef CALIBRATE
    extern int dest;
    int csave;
    csave = calibrate;			/* So we can decode filename */
    calibrate = 0;
#endif /* CALIBRATE */

    ofperms = "";			/* Reset old-file permissions */
    opnerr = 0;				/* No open error (yet) */
    ofn2 = NULL;			/* No new name (yet) */
    lsstate = 0;			/* Cancel locking-shift state */
    srvptr = srvcmd;			/* Decode file name from packet. */

    debug(F110,"rcvfil rdatap",rdatap,0);
    decode(rdatap,putsrv,0);		/* Don't xlate charsets. */
#ifdef CALIBRATE
    calibrate = csave;
    if (dest == DEST_N) {
	calibrate = 1;
	cmarg2 = "CALIBRATE";
    }
#endif /* CALIBRATE */
    if (*srvcmd == '\0')		/* Watch out for null F packet. */
      strcpy((char *)srvcmd,"NONAME");
    makestr(&rrfspec,(char *)srvcmd);
#ifdef DTILDE
    if (*srvcmd == '~') {
	dirp = tilde_expand((char *)srvcmd); /* Expand tilde, if any. */
	if (*dirp != '\0') strcpy((char *)srvcmd,dirp);
    }
#else
#ifdef OS2
    if (isalpha(*srvcmd) && srvcmd[1] == ':' && srvcmd[2] == '\0')
      strcat((char *)srvcmd,"NONAME");
#endif /* OS2 */
#endif /* DTILDE */

    if (!ENABLED(en_cwd)) {		/* CD is disabled */
	zstrip((char *)(srvcmd+2),&n2); /* and they included a pathname, */
	if (strcmp((char *)(srvcmd+2),n2)) { /* so refuse. */
	    rf_err = "Access denied";
	    return(0);
	}
    }	
    xxscreen(SCR_FN,0,0l,(char *)srvcmd); /* Put it on screen if local */
    debug(F110,"rcvfil",(char *)srvcmd,0); /* Debug log entry */
    tlog(F110,"Receiving",(char *)srvcmd,0L); /* Transaction log entry */

    if (!cmarg2)			/* No core dumps please */
      cmarg2 = "";
    debug(F110,"rcvfil cmarg2",cmarg2,0);
    if (*cmarg2) {			/* Check for alternate name */
#ifndef NOSPL
	int y; char *s;			/* Pass it thru the evaluator */
	extern int cmd_quoting;
	if (cmd_quoting) {
	    y = MAXRP;
	    strncpy(ofn1,(char *)srvcmd,CKMAXPATH); /* cuz of \v(filename) */
	    s = (char *)srvcmd;
	    zzstring(cmarg2,&s,&y);
	} else
	  *srvcmd = NUL;
	if (!*srvcmd)			/* If we got something */
#endif /* NOSPL */
	  strcpy((char *)srvcmd,cmarg2);
    }
    debug(F110,"rcvfil srvcmd",srvcmd,0);

#ifdef PIPESEND
    /* If it starts with "bang", it's a pipe, not a file. */
    if (usepipes && protocol == PROTO_K && *srvcmd == '!' && !rcvfilter) {
	CHAR *s;
	s = srvcmd+1;			/* srvcmd[] is not a pointer. */
	while (*s) {			/* So we have to slide the contents */
	    *(s-1) = *s;		/* over 1 space to the left. */
	    s++;
	}
	*(s-1) = NUL;
	pipesend = 1;
    }
#endif /* PIPESEND */

#ifdef COMMENT
/*
  This is commented out because we need to know whether the name we are
  using was specified by the local user as an override, or came from the
  incoming packet.  In the former case, we don't do stuff to it (like
  strip the pathname) that we might do to it in the latter.
*/
    cmarg2 = "";			/* Done with alternate name */
#endif /* COMMENT */

    if ((int)strlen((char *)srvcmd) > CKMAXPATH) /* Watch out for overflow */
      *(srvcmd + CKMAXPATH - 1) = NUL;

    /* At this point, srvcmd[] contains the incoming filename or as-name */

#ifdef CK_LABELED
#ifdef VMS
/*
  If we have an as-name, this overrides the internal name if we are doing
  a labeled-mode transfer.
*/
    if (*cmarg2) {
	extern int lf_opts;
	lf_opts &= ~LBL_NAM;
    }
#endif /* VMS */
#endif /* CK_LABELED */

#ifdef PIPESEND
    /* Skip all the filename manipulation and collision actions */
    if (pipesend) {
	dirflg = 0;
	sprintf(ofn1,"!%s",(char *)srvcmd);
	strcpy(n,ofn1);
	strncpy(fspec,ofn1,CKMAXPATH);
	makestr(&rfspec,fspec);
	debug(F110,"rcvfil pipesend",ofn1,0);
	goto rcvfilx;
    }
#endif /* PIPESEND */

#ifdef NZLTOR
    if (*cmarg2)
      strncpy((char *)ofn1,(char *)srvcmd,CKMAXPATH);
    else
      nzrtol((char *)srvcmd,		/* Filename from packet */
	     (char *)ofn1,		/* Where to put result */
	     fncnv,			/* Filename conversion */
	     fnrpath,			/* Pathname handling */
	     CKMAXPATH			/* Size of result buffer */
	     );
#else
    debug(F101,"rcvfil fnrpath","",fnrpath); /* Handle pathnames */
    if (fnrpath == PATH_OFF && !*cmarg2) { /* RECEIVE PATHNAMES OFF? */
	char *t;			/* Yes. */
	zstrip((char *)srvcmd,&t);	/* If there is a pathname, strip it */
	debug(F110,"rcvfil PATH_OFF zstrip",t,0);
	if (!t)				/* Be sure we didn't strip too much */
	  sprintf(ofn1,"FILE%02ld",filcnt);
	else if (*t == '\0')
	  sprintf(ofn1,"FILE%02ld",filcnt);
	else
	  strcpy(ofn1,t);
	strcpy((char *)srvcmd,ofn1);	/* Now copy it back. */
    }
/*
  SET RECEIVE PATHNAMES RELATIVE...
  The following doesn't belong here but doing it right would require
  defining and implementing a new file routine for all ck?fio.c modules.
  So for now...
*/
#ifdef UNIXOROSK
    else if (fnrpath == PATH_REL && !*cmarg2) {
	if (isabsolute((char *)srvcmd)) {
	    sprintf(ofn1,".%s",(char *)srvcmd);
	    strcpy((char *)srvcmd,ofn1);
	    debug(F110,"rcvfil PATH_REL",ofn1,0);
	}
    }
#else
#ifdef OS2
    else if (fnrpath == PATH_REL && !*cmarg2) {
	if (isabsolute((char *)srvcmd)) {
	    char *p = (char *)srvcmd;
	    if (isalpha(*p) && *(p+1) == ':')
	      p += 2;
	    if (*p == '\\' || *p == '/')
	      p++;
	    strncpy(ofn1,p,CKMAXPATH);
	    strcpy((char *)srvcmd,ofn1);
	    debug(F110,"rcvfil OS2 PATH_REL",ofn1,0);
	}
    }
#endif /* OS2 */
#endif /* UNIXOROSK */

    /* Now srvcmd contains incoming filename with path possibly stripped */

    if (fncnv)				/* FILE NAMES CONVERTED? */
      zrtol((char *)srvcmd,(char *)ofn1); /* Yes, convert to local form */
    else
      strcpy(ofn1,(char *)srvcmd);	/* No, copy literally. */
#endif /* NZLTOR */

#ifdef PIPESEND
    if (rcvfilter) {
	char * s, * p = NULL, * q;
	int n = MAXRP;
	pipesend = 1;
	debug(F110,"rcvfil rcvfilter ",rcvfilter,0);
#ifndef NOSPL
	if (p = (char *) malloc(n + 1)) {
	    q = p;
#ifdef COMMENT
            /* We have already processed srvcmd and placed it into ofn1 */
            strcpy(ofn1,(char *)srvcmd); /* For \v(filename) */
#endif /* COMMENT */
	    debug(F110,"rcvfile pipesend filter",rcvfilter,0);
	    zzstring(rcvfilter,&p,&n);
	    debug(F111,"rcvfil pipename",q,n);
	    if (n <= 0) {
		printf(
		       "?Sorry, receive filter + filename too long, %d max.\n",
		       CKMAXPATH
		       );
		rf_err = "Name too long";
		free(q);
		return(0);
	    }
	    strcpy((char *)srvcmd,q);
	    free(q);
	}
#endif /* NOSPL */
    }
#endif /* PIPESEND */


    /* Now the incoming filename, possibly converted, is in ofn1[]. */

#ifdef OS2
    /* Don't refuse the file just because the name is illegal. */
    if (!IsFileNameValid(ofn1)) {	/* Name is OK for OS/2? */
#ifdef OS2ONLY
	char *zs = NULL;
	zstrip(ofn1, &zs);		/* Not valid, strip unconditionally */
	if (zs) {
	    if (iattr.longname.len &&	/* Free previous longname, if any */
		iattr.longname.val)
	      free(iattr.longname.val);
	    iattr.longname.len = strlen(zs); /* Store in attribute structure */
	    iattr.longname.val = (char *) malloc(iattr.longname.len + 1);
	    if (iattr.longname.val)	/* Remember this (illegal) name */
	      strcpy(iattr.longname.val, zs);
	}
#endif /* OS2ONLY */
	debug(F110,"rcvfil: invalid file name",ofn1,0);
	ChangeNameForFAT(ofn1);	/* Change to an acceptable name */
	debug(F110,"rcvfil: FAT file name",ofn1,0);

    } else {				/* Name is OK. */

	debug(F110,"rcvfil: valid file name",ofn1,0);
#ifdef OS2ONLY
	if (iattr.longname.len &&
	     iattr.longname.val)	/* Free previous longname, if any */
	  free(iattr.longname.val);
	iattr.longname.len = 0;
	iattr.longname.val = NULL;	/* This file doesn't need a longname */
#endif /* OS2ONLY */
    }
#endif /* OS2 */
    debug(F110,"rcvfil as",ofn1,0);

/* Filename collision action section. */

    dirflg =				/* Is it a directory name? */
#ifdef CK_TMPDIR
        isdir(ofn1)
#else
	0
#endif /* CK_TMPDIR */
	  ;
    debug(F101,"rcvfil dirflg","",dirflg);
    ofn1x = (zchki(ofn1) != -1);	/* File already exists? */
    debug(F101,"rcvfil ofn1x",ofn1,ofn1x);

    if ( (
#ifdef UNIX
	strcmp(ofn1,"/dev/null") &&	/* It's not the null device? */
#else
#ifdef OSK
	strcmp(ofn1,"/nil") &&
#endif /* OSK */
#endif /* UNIX */
	!stdouf ) &&			/* Not copying to standard output? */
	ofn1x ||			/* File of same name exists? */
	dirflg ) {			/* Or file is a directory? */
        debug(F111,"rcvfil exists",ofn1,fncact);
#ifdef CK_PERMS
	ofperms = zgperm((char *)ofn1);	/* Get old file's permissions */
	debug(F110,"rcvfil perms",ofperms,0);
#endif /* CK_PERMS */

	switch (fncact) {		/* Yes, do what user said. */
	  case XYFX_A:			/* Append */
	    ofperms = "";
	    debug(F100,"rcvfil append","",0);
	    if (dirflg) {
		rf_err = "Can't append to a directory";
		tlog(F100," error - can't append to directory","",0);
		discard = opnerr = 1;
		return(0);
	    }
	    tlog(F110," appending to",ofn1,0);
	    break;
#ifdef COMMENT
	  case XYFX_Q:			/* Query (Ask) */
	    break;			/* not implemented */
#endif /* COMMENT */
	  case XYFX_B:			/* Backup (rename old file) */
	    if (dirflg) {
		rf_err = "Can't rename existing directory";
		tlog(F100," error - can't rename directory","",0);
		discard = opnerr = 1;
		return(0);
	    }
	    znewn(ofn1,&ofn2);		/* Get new unique name */
	    tlog(F110," backup:",ofn2,0);
	    debug(F110,"rcvfil backup ofn1",ofn1,0);
	    debug(F110,"rcvfil backup ofn2",ofn2,0);
#ifdef CK_LABELED
#ifdef OS2ONLY
/*
  In case this is a FAT file system, we can't change only the FAT name, we
  also have to change the longname from the extended attributes block.
  Otherwise, we'll have many files with the same longname and if we copy them
  to an HPFS volume, only one will survive.
*/
	    if (os2getlongname(ofn1, &longname) > -1) {
		if (strlen(longname)) {
		    char tmp[10];
		    extern int ck_znewn;
		    sprintf(tmp,".~%d~",ck_znewn);
		    newlongname =
		      (char *) malloc(strlen(longname) + strlen(tmp) + 1);
		    if (newlongname) {
			strcpy(newlongname, longname);
			strcat(newlongname, tmp);
			os2setlongname(ofn1, newlongname);
			free(newlongname);
			newlongname = NULL;
		    }
		}
	    } else debug(F100,"rcvfil os2getlongname failed","",0);
#endif /* OS2ONLY */
#endif /* CK_LABELED */

#ifdef COMMENT
	    /* Do this later, in opena()... */
	    if (zrename(ofn1,ofn2) < 0) {
		rf_err = "Can't transform filename";
		debug(F110,"rcvfil rename fails",ofn1,0);
		discard = opnerr = 1;
		return(0);
	    }
#endif /* COMMENT */
	    break;

	  case XYFX_D:			/* Discard (refuse new file) */
	    ofperms = "";
	    discard = 1;
	    rejection = 1;		/* Horrible hack: reason = name */
	    debug(F101,"rcvfil discard","",discard);
	    tlog(F100," refused: name","",0);
	    break;

	  case XYFX_R:			/* Rename incoming file */
	    znewn(ofn1,&ofn2);		/* Make new name for it */
#ifdef OS2ONLY
	    if (iattr.longname.len) {
		char tmp[10];
		extern int ck_znewn;
		sprintf(tmp,".~%d~",ck_znewn);
		newlongname =
		  (char *) malloc(iattr.longname.len + strlen(tmp) + 1);
		if (newlongname) {
		    strcpy(newlongname, iattr.longname.val);
		    strcat(newlongname, tmp);
		    debug(F110,
			  "Rename Incoming: newlongname",newlongname,0);

		    if (iattr.longname.len &&
			iattr.longname.val)
		      free(iattr.longname.val);
		    iattr.longname.len = strlen(newlongname);
		    iattr.longname.val = newlongname;
		}
	    }
#endif /* OS2ONLY */
	    break;
	  case XYFX_X:			/* Replace old file */
	    debug(F100,"rcvfil overwrite","",0);
	    if (dirflg) {
		rf_err = "Can't overwrite existing directory";
		tlog(F100," error - can't overwrite directory","",0);
		discard = opnerr = 1;
#ifdef COMMENT
		return(0);
#else
		break;
#endif /* COMMENT */
	    }
	    tlog(F110,"overwriting",ofn1,0);
	    break;
	  case XYFX_U:			/* Refuse if older */
	    debug(F110,"rcvfil update",ofn1,0);
	    if (dirflg) {
		rf_err = "File has same name as existing directory";
		tlog(F110," error - directory exists:",ofn1,0);
		discard = opnerr = 1;
#ifdef COMMENT
		/* Don't send an error packet, just refuse the file */
		return(0);
#endif /* COMMENT */
	    }
	    break;			/* Not here, we don't have */
					/* the attribute packet yet. */
	  default:
	    ofperms = "";
	    debug(F101,"rcvfil bad collision action","",fncact);
	    break;
	}
    }
    debug(F110,"rcvfil ofn1",ofn1,0);
    debug(F110,"rcvfil ofn2",ofn2,0);
    debug(F110,"rcvfil ofperms",ofperms,0);
    if (fncact == XYFX_R && ofn1x && ofn2) { /* Renaming incoming file? */
	xxscreen(SCR_AN,0,0l,ofn2);	/* Display renamed name */
	strcpy(n, ofn2);		/* Return it */
    } else {				/* No */
	xxscreen(SCR_AN,0,0l,ofn1);	/* Display regular name */
	strcpy(n, ofn1);		/* and return it. */
    }

#ifdef CK_MKDIR
/*  Create directory(s) if necessary.  */
    if (!discard && fnrpath != PATH_OFF) { /* RECEIVE PATHAMES ON? */
	int x;
	debug(F110,"rcvfil calling zmkdir",ofn1,0); /* Yes */
	if ((x = zmkdir(ofn1)) < 0) {
	    debug(F100,"zmkdir fails","",0);
	    tlog(F110," error - directory creation failure:",ofn1,0);
	    rf_err = "Directory creation failure.";
	    discard = 1;
	    return(0);
	}
#ifdef TLOG
	else if (x > 0)
	  tlog(F110," path created:",ofn1,0);
#endif /* TLOG */
    }
#else
    debug(F110,"rcvfil CK_MKDIR not defined",ofn1,0);
#endif /* CK_MKDIR */

/* #ifndef MAC */
/* Why not Mac? */
#ifdef ZFNQFP
    if (calibrate)
      strcpy(fspec,ofn1);		/* Here too for \v(filespec) */
    else
      zfnqfp(ofn1,CKMAXPATH,fspec);
#else
    strncpy(fspec,ofn1,CKMAXPATH);	/* Here too for \v(filespec) */
#endif /* ZFNQFP */
/* #endif */ 
    fspec[CKMAXPATH] = NUL;
    makestr(&rfspec,fspec);

  rcvfilx:
    debug(F110,"rcvfilx: n",n,0);
    debug(F110,"rcvfilx: ofn1",ofn1,0);
    ffc = 0L;				/* Init per-file counters */
    cps = oldcps = 0L;
    rs_len = 0L;
    rejection = -1;
    fsecs = gtimer();			/* Time this file started */
#ifdef GFTIMER
    fpfsecs = gftimer();
    debug(F101,"rcvfil fpfsecs","",fpfsecs);
#endif /* GFTIMER */
    filcnt++;
    intmsg(filcnt);
    return(1);				/* Successful return */
}


/*  R E O F  --  Receive End Of File packet for incoming file */

/*
  Closes the received file.
  Returns:
    0 on success.
   -1 if file could not be closed.
    2 if disposition was mail, mail was sent, but temp file not deleted.
    3 if disposition was print, file was printed, but not deleted.
   -2 if disposition was mail and mail could not be sent
   -3 if disposition was print and file could not be printed
*/
int
reof(f,yy) char *f; struct zattr *yy; {
    int x;
    char *p;
    char c;

    debug(F111,"reof fncact",f,fncact);
    debug(F101,"reof discard","",discard);
    success = 1;			/* Assume status is OK */
    lsstate = 0;			/* Cancel locking-shift state */
    if (discard) {			/* Handle attribute refusals, etc. */
	debug(F101,"reof discarding","",0);
	success = 0;			/* Status = failed. */
	if (rejection == '#' ||		/* Unless rejection reason is */
	    rejection ==  1  ||		/* date or name (SET FILE COLLISION */
	    rejection == '?')		/* UPDATE or DISCARD) */
	  success = 1;			    
	debug(F101,"reof success","",success);
	filrej++;			/* Count this rejection. */
	discard = 0;			/* We never opened the file, */
	return(0);			/* so we don't close it. */
    }
#ifdef DEBUG
    if (deblog) {
	debug(F101,"reof cxseen","",cxseen);
	debug(F101,"reof czseen","",czseen);
	debug(F110,"reof rdatap",rdatap,0);
    }
#endif /* DEBUG */

    if (cxseen == 0) cxseen = (*rdatap == 'D');	/* Got cancel directive? */
    success = (cxseen || czseen) ? 0 : 1; /* Set SUCCESS flag appropriately */
    if (!success) filrej++;		/* "Uncount" this file */
    debug(F101,"reof success","",success);

#ifdef CK_CTRLZ
    if (success) {
	debug(F101,"reof lastchar","",lastchar);
	if (!binary && eofmethod == XYEOF_Z && lastchar != 26 &&
	    (!xflg || (xflg && remfile)))
	  zmchout((char)26);
    }
#endif /* CK_CTRLZ */

    x = clsof(cxseen || czseen);	/* Close the file (resets cxseen) */
    debug(F101,"reof closf","",x);
    if (x < 0) {			/* If failure to close, FAIL */
	if (success) filrej++;
	success = 0;
    }
    if (!calibrate) {
    /* Set file modification date and/or permissions */
	if (success)
	  zstime(f,yy,0);
#ifdef OS2ONLY
#ifdef CK_LABELED
	if (success && yy->longname.len)
	  os2setlongname(f, yy->longname.val);
#endif /* CK_LABELED */
#endif /* OS2ONLY */
    }
    if (success == 0) xitsta |= W_RECV;	/* And program return code */

/* Handle dispositions from attribute packet... */

#ifndef NOFRILLS
    if (!calibrate && yy->disp.len != 0) {
	p = yy->disp.val;
	c = *p++;
#ifndef UNIX
/*
  See ckcpro.w.  In UNIX we don't use temp files any more -- we pipe the
  stuff right into mail or lpr.
*/
	if (c == 'M') {			/* Mail to user. */
	    x = zmail(p,filnam);	/* Do the system's mail command */
	    if (x < 0) success = 0;	/* Remember status */
	    tlog(F110,"mailed",filnam,0L);
	    tlog(F110," to",p,0L);
	    zdelet(filnam);		/* Delete the file */
	} else if (c == 'P') {		/* Print the file. */
	    x = zprint(p,filnam);	/* Do the system's print command */
	    if (x < 0) success = 0;	/* Remember status */
	    tlog(F110,"printed",filnam,0L);
	    tlog(F110," with options",p,0L);
#ifndef VMS
#ifndef STRATUS
	    /* spooler will delete file after print complete in VOS & VMS */
	    if (zdelet(filnam) && x == 0) x = 3; /* Delete the file */
#endif /* STRATUS */
#endif /* VMS */
	}
#endif /* UNIX */
    }
#endif /* NOFRILLS */
    debug(F101,"reof returns","",x);
    *filnam = '\0';
    return(x);
}

/*  R E O T  --  Receive End Of Transaction  */

VOID
reot() {
    cxseen = czseen = discard = 0;	/* Reset interruption flags */
    tstats();
}

/*  S F I L E -- Send File header or teXt header packet  */

/*
  Call with x nonzero for X packet, zero for F packet.
  If X == 0, filename to send is in filnam[], and if cmarg2 is not null
  or empty, the file should be sent under this name rather than filnam[].
  If sndfilter not NULL, it is the name of a send filter.
  Returns 1 on success, 0 on failure.
*/
int
sfile(x) int x; {
#ifdef pdp11
#define PKTNL 64
#else
#define PKTNL 256
#endif /* pdp11 */
    char pktnam[PKTNL+1];		/* Local copy of name */
    char *s;
    int rc;
#ifdef PIPESEND
    extern char * sndfilter;

    debug(F101,"sfile x","",x);
    if (sndfilter) {
	pipesend = 1;
	debug(F110,"sfile send filter ",sndfilter,0);
    }
#else
    debug(F101,"sfile x","",x);
#endif /* PIPESEND */

    /* cmarg2 or filnam (with that precedence) have the file's name */

    lsstate = 0;			/* Cancel locking-shift state */
    if (nxtpkt() < 0) return(0);	/* Bump packet number, get buffer */
    pktnam[0] = NUL;			/* Buffer for name we will send */
    if (x == 0) {			/* F-Packet setup */
	if (!cmarg2) cmarg2 = "";
	debug(F111,"sfile cmarg2",cmarg2,cmarg2);
#ifdef PATTERNS
	debug(F101,"sfile binary 1","",binary);

/* Select transfer mode based on filename pattern match */

	if (patterns && xfermode == XMODE_A /* TRANSFER MODE AUTOMATIC */
#ifdef PIPESEND
	    && !pipesend		/* But not if sending from pipe */
#endif /* PIPESEND */
#ifdef CK_LABELED
	    && binary != XYFT_L		/* And not if FILE TYPE LABELED */
#endif /* CK_LABELED */
#ifdef VMS
	    && binary != XYFT_I		/* or FILE TYPE IMAGE */
#endif /* VMS */
	    ) {
	    if (binary != XYFT_T &&
		sendmode != SM_RESEND && /* Also not if resending */
		txtpatterns[0]) {
		int i;
		for (i = 0; i < FTPATTERNS && txtpatterns[i]; i++) {
		    if (ckmatch(txtpatterns[i],filnam,alphacase)) {
			binary = XYFT_T;
			break;
		    }
		}
	    }
	    if (binary != XYFT_B && binpatterns[0]) {
		int i;
		for (i = 0; i < FTPATTERNS && binpatterns[i]; i++) {
		    if (ckmatch(binpatterns[i],filnam,alphacase)) {
			binary = XYFT_B;
			break;
		    }
		}
	    }
	}
	debug(F101,"sfile binary 2","",binary);
#endif /* PATTERNS */

    	if (*cmarg2) {			/* If we have a send-as name... */
	    int y; char *s;
#ifndef NOSPL				/* and a script programming language */
	    extern int cmd_quoting;
	    if (cmd_quoting) {		/* and it's not turned off */
		y = PKTNL;		/* pass as-name thru the evaluator */
		s = pktnam;
		zzstring(cmarg2,&s,&y);
#ifdef COMMENT
/* This ruins macros like BSEND */
		if (!pktnam[0])		/* and make sure result is not empty */
		  sprintf(pktnam,"FILE%02ld",filcnt);
#endif /* COMMENT */
	    } else
#endif /* NOSPL */
	      strncpy(pktnam,cmarg2,PKTNL); /* copy it literally, */

	    debug(F110,"sfile pktnam",pktnam,0);
#ifdef COMMENT
/* We don't do this any more because now we have filename templates */
	    cmarg2 = "";		/* and blank it out for next time. */
#endif /* COMMENT */
    	} 
	if (!*pktnam) {			/* No as-name... */
#ifdef NZLTOR
	    int xfncnv, xpath;
	    debug(F101,"sfile fnspath","",fnspath);
	    debug(F101,"sfile fncnv","",fncnv);
	    debug(F101,"sfile calibrate","",calibrate);
	    xfncnv = fncnv;
	    xpath = fnspath;
#ifdef CALIBRATE
	    if (calibrate) {
		xfncnv = 0;		/* Don't convert */
		xpath = PATH_OFF;	/* Remove path */
	    }
#endif /* CALIBRATE */
#ifdef PIPESEND
	    debug(F101,"sfile pipesend","",pipesend);
	    debug(F101,"sfile sndfilter","",sndfilter);
	    if (pipesend || sndfilter) {
		xfncnv = 0;		/* Don't convert */
		xpath = PATH_ABS;	/* Leave or put path on */
	    }
#endif /* PIPESEND */
	    debug(F101,"sfile xpath","",xpath);
	    debug(F101,"sfile xfncnv","",xfncnv);
	    nzltor(filnam,pktnam,xfncnv,xpath,PKTNL);

#else  /* Not NZLTOR */

	    debug(F101,"sfile fnspath","",fnspath);
	    if (fnspath == PATH_OFF	/* Stripping path names? */
#ifdef PIPESEND
		&& (!pipesend || !sndfilter) /* does this make sense? */
#endif /* PIPESEND */
		) {
		char *t;		/* Yes. */
		zstrip(filnam,&t);	/* Strip off the path. */
		debug(F110,"sfile zstrip",t,0);
		if (!t) t = "UNKNOWN";	/* Be cautious... */
		else if (*t == '\0')
		  t = "UNKNOWN";
		strncpy(pktnam,t,PKTNL); /* Copy stripped name literally. */
	    } else if (fnspath == PATH_ABS && !calibrate) {
		/* Converting to absolute form */
#ifdef ZFNQFP
		zfnqfp(filnam,PKTNL,pktnam);
#else
		strncpy(pktnam,filnam,PKTNL);
#endif /* ZFNQFP */
	    } else
		strncpy(pktnam,filnam,PKTNL);

	    /* pktnam[] has the packet name, filnam[] has the original name. */
	    /* But we still need to convert pktnam if FILE NAMES CONVERTED.  */

	    debug(F101,"sfile fncnv","",fncnv);
	    if (fncnv
#ifdef PIPESEND
		&& (!pipesend || !sndfilter)
#endif /* PIPESEND */
#ifdef CALIBRATE
		&& !calibrate
#endif /* CALIBRATE */
		) {			/* If converting names, */
		zltor(pktnam,(char *)srvcmd); /* convert it to common form, */
		strcpy(pktnam,(char *)srvcmd); /* with srvcmd as temp buffer */
		*srvcmd = NUL;
	    }
#endif /* NZLTOR */
    	}
	if (!*pktnam)			/* Failsafe... */
	  sprintf(pktnam,"FILE%02ld",filcnt);
	debug(F110,"sfile filnam 1",filnam,0);
	debug(F110,"sfile pktnam 1",pktnam,0);
#ifdef PIPESEND
/* If we have a send filter, substitute the current filename into it */

	if (sndfilter) {
	    char * p = NULL, * q;
	    int n = CKMAXPATH;
#ifndef NOSPL
	    if (p = (char *) malloc(n + 1)) {
		q = p;
		debug(F110,"sfile pipesend filter",sndfilter,0);
		zzstring(sndfilter,&p,&n);
		debug(F111,"sfile pipename",q,n);
		if (n <= 0) {
		    printf(
			  "?Sorry, send filter + filename too long, %d max.\n",
			   CKMAXPATH
			   );
		    free(q);
		    return(0);
		}
		strcpy(filnam,q);
		free(q);
	    }
#endif /* NOSPL */
	}
#endif /* PIPESEND */
    	debug(F110,"sfile filnam 2",filnam,0); /* Log debugging info */
    	debug(F110,"sfile pktnam 2",pktnam,0);
    	if (openi(filnam) == 0) 	/* Try to open the input file */
	  return(0); 		
#ifdef CK_RESEND
	if (sendmode == SM_PSEND)	/* PSENDing? */
	  if (sendstart > 0L)		/* Starting position */
	    if (zfseek(sendstart) < 0)	/* seek to it... */
	      return(0);
#endif /* CK_RESEND */
    	s = pktnam;			/* Name for packet data field */
#ifdef OS2
	/* Never send a disk letter. */
	if (isalpha(*s) && (*(s+1) == ':'))
	  s += 2;
#endif /* OS2 */

    } else {				/* X-packet setup, not F-packet. */
	binary = XYFT_T;		/* Text always */
    	debug(F110,"sfile X packet",cmdstr,0); /* Log debugging info */
    	s = cmdstr;			/* Name for data field */
    }

    /* Now s points to the string that goes in the packet data field. */

    debug(F101,"sfile binary","",binary); /* Log debugging info */    
    encstr((CHAR *)s);			/* Encode the name. */
					/* Send the F or X packet */
    /* If the encoded string did not fit into the packet, it was truncated. */
   
    rc = spack((char) (x ? 'X' : 'F'), pktnum, size, data);
    if (rc < 0)
      return(rc);
    if (x == 0) {			/* Display for F packet */
    	if (displa) {			/* Screen */
	    xxscreen(SCR_FN,'F',(long)pktnum,filnam);
	    xxscreen(SCR_AN,0,0L,pktnam);
	    xxscreen(SCR_FS,0,calibrate ? calibrate : fsize,"");
    	}
#ifdef pdp11
    	tlog(F110,"Sending",filnam,0L);	/* Transaction log entry */
	makestr(&sfspec,filnam);
#else
#ifndef ZFNQFP
    	tlog(F110,"Sending",filnam,0L);
	makestr(&sfspec,filnam);
#else
	{				/* Log fully qualified filename */
	    char *p = NULL, *q = filnam;
	    if ((p = malloc(CKMAXPATH+1)))
	      if (zfnqfp(filnam, CKMAXPATH, p))
		q = p;
	    tlog(F110,"Sending",q,0L);
	    makestr(&sfspec,p);
	    if (p) free(p);
	}
#endif /* ZFNQFP */
#endif /* pdp11 */
    	tlog(F110," as",pktnam,0L);
	if (binary) {			/* Log file mode in transaction log */
	    tlog(F101," mode: binary","",(long) binary);
	} else {			/* If text mode, check character set */
	    tlog(F100," mode: text","",0L);
#ifndef NOCSETS
	    tlog(F110," file character set",fcsinfo[fcharset].name,0L);
	    if (tcharset == TC_TRANSP)
	      tlog(F110," xfer character set","transparent",0L);
	    else
	      tlog(F110," xfer character set",tcsinfo[tcharset].name,0L);
#endif /* NOCSETS */
	}
    } else {				/* Display for X-packet */

    	xxscreen(SCR_XD,'X',(long)pktnum,cmdstr); /* Screen */
    	tlog(F110,"Sending from:",cmdstr,0L);	/* Transaction log */
    }
    intmsg(++filcnt);			/* Count file, give interrupt msg */
    first = 1;				/* Init file character lookahead. */
    ffc = 0L;				/* Init file character counter. */
    cps = oldcps = 0L;			/* Init cps statistics */
    rejection = -1;
    fsecs = gtimer();			/* Time this file started */
#ifdef GFTIMER
    fpfsecs = gftimer();
    debug(F101,"SFILE fpfsecs","",fpfsecs);
#endif /* GFTIMER */
    debug(F101,"SFILE fsecs","",fsecs);
    return(1);
}

/*  S D A T A -- Send a data packet */

/*
  Returns -1 if no data to send (end of file), -2 if connection is broken.
  If there is data, a data packet is sent, and sdata() returns 1.

  In the streaming case, the window is regarded as infinite and we keep
  sending data packets until EOF or there appears to be a Kermit packet on the
  reverse channel.  When not streaming and the window size is greater than 1,
  we keep sending data packets until window is full or characters start to
  appear from the other Kermit.

  In the windowing or streaming case, when there is no more data left to send
  (or when sending has been interrupted), sdata() does nothing and returns 0
  each time it is called until the acknowledgement sequence number catches up
  to the last data packet that was sent.
*/
int
sdata() {
    int i, x, len;
    
    debug(F101,"sdata entry, first","",first);
    debug(F101,"sdata drain","",drain);
/*
  The "drain" flag is used with window size > 1.  It means we have sent
  our last data packet.  If called and drain is not zero, then we return
  0 as if we had sent an empty data packet, until all data packets have
  been ACK'd, then then we can finally return -1 indicating EOF, so that
  the protocol can switch to seof state.  This is a kludge, but at least
  it's localized...
*/
    if (first == 1) drain = 0;		/* Start of file, init drain flag. */

    if (drain) {			/* If draining... */
	debug(F101,"sdata draining, winlo","",winlo);
	if (winlo == pktnum)		/* If all data packets are ACK'd */
	  return(-1);			/* return EOF indication */
	else				/* otherwise */
	  return(0);			/* pretend we sent a data packet. */
    }
    debug(F101,"sdata sbufnum","",sbufnum);
    for (i = sbufnum;
	 i > 0
#ifdef STREAMING
	 || streaming
#endif /* STREAMING */
	 ;
	 i--) {
        debug(F101,"sdata countdown","",i);
	if (streaming) {
	    pktnum = (pktnum + 1) % 64;
	} else {
	    x = nxtpkt();		/* Get next pkt number and buffer */
	    debug(F101,"sdata nxtpkt","",x);
	    if (x < 0) return(0);
	}
	debug(F101,"sdata packet","",pktnum);
	if (chkint() < 0)		/* Especially important if streaming */
	  return(-9);
	if (cxseen || czseen) {		/* If interrupted, done. */
	    if (wslots > 1) {
		drain = 1;
		debug(F101,"sdata cx/zseen, drain","",cxseen);
		return(0);
	    } else {
		return(-1);
	    }
	}
#ifdef DEBUG
	if (deblog) {
	    debug(F101,"sdata spsiz","",spsiz);
	    debug(F101,"sdata binary","",binary);
	    debug(F101,"sdata parity","",parity);
	}
#endif /* DEBUG */
#ifdef CKTUNING
	len = (binary && !parity && !memstr && !funcstr) ?
	  bgetpkt(spsiz) : getpkt(spsiz,1);
#else
	len = getpkt(spsiz,1);
#endif /* CKTUNING */
	if (len == 0) {			/* Done if no data. */
	    if (pktnum == winlo)
	      return(-1);
	    drain = 1;			/* But can't return -1 until all */
	    debug(F101,"sdata eof, drain","",drain);
	    return(0);			/* ACKs are drained. */
	}
	x = spack('D',pktnum,len,data);	/* Send the data packet. */
	if (x < 0)
	  return(x);
	if (streaming)			/* What an ACK would do. */
	  winlo = pktnum;
	x = ttchk();			/* Peek at input buffer. */
	debug(F101,"sdata ttchk","",x);	/* ACKs waiting, maybe?  */
	if (x < 0)			/* Or connection broken? */
	  return(x);
/*
  Here we check to see if any ACKs or NAKs have arrived, in which case we
  break out of the D-packet-sending loop and return to the state switcher
  to process them.  This is what makes our windows slide instead of lurch.
*/
	if (
#ifdef GEMDOS
/*
  In the Atari ST version, ttchk() can only return 0 or 1.  But note: x will
  probably always be > 0, since the as-yet-unread packet terminator from the
  last packet is probably still in the buffer, so sliding windows will
  probably never happen when the Atari ST is the file sender.  The alternative
  is to say "if (0)", in which case the ST will always send a window full of
  packets before reading any ACKs or NAKs.
*/
	    x > 0
  
#else /* !GEMDOS */
/*
  In most other versions, ttchk() returns the actual count.
  It can't be a Kermit packet if it's less than five bytes long.
*/
	    x > 4 + bctu
  
#endif /* GEMDOS */
	    )
	  return(1);			/* Yes, stop sending data packets */
    }					/* and go try to read the ACKs. */
    return(1);
}


/*  S E O F -- Send an End-Of-File packet */

/*  Call with a string pointer to character to put in the data field, */
/*  or else a null pointer or "" for no data.  */

/*
  There are two "send-eof" functions.  seof() is used to send the normal eof
  packet at the end of a file's data (even if the file has no data), or when
  a file transfer is interrupted.  sxeof() is used to send an EOF packet that
  occurs because of attribute refusal.  The difference is purely a matter of
  buffer allocation and packet sequence number management.  Both functions
  act as "front ends" to the common send-eof function, szeof().
*/

/* Code common to both seof() and sxeof() */

int
szeof(s) CHAR *s; {
    int x;
    lsstate = 0;			/* Cancel locking-shift state */
    if (!s) s = (CHAR *)"";
    if (*s) {
	x = spack('Z',pktnum,1,s);
	xitsta |= W_SEND;
	tlog(F100," *** interrupted, sending discard request","",0L);
	filrej++;
    } else {
	x = spack('Z',pktnum,0,(CHAR *)"");
    }
    if (x < 0)
      return(x);
    discard = 0;			/* Turn off per-file discard flag */
#ifdef PIPESEND
/* If we were sending from a pipe, we're not any more... */
    pipesend = 0;
#endif /* PIPESEND */
    return(0);
}

int
seof(s) CHAR *s; {
/*
  ckcpro.w, before calling seof(), sets window size back to 1 and then calls
  window(), which clears out the old buffers.  This is OK because the final
  data packet for the file has been ACK'd.  However, sdata() has already
  called nxtpkt(), which set the new value of pktnum which seof() will use.
  So all we need to do here is is allocate a new send-buffer.
*/
    if (getsbuf(pktnum) < 0) {	/* Get a buffer for packet n */
	debug(F101,"seof can't get s-buffer","",pktnum);
	return(-1);
    }
    return(szeof(s));
}

/*
  Version of seof() to be called when sdata() has not been called before.  The
  difference is that this version calls nxtpkt() to allocate a send-buffer and
  get the next packet number.
*/
int
sxeof(s) CHAR *s; {
    int x;
    x = nxtpkt();			/* Get next pkt number and buffer */
    if (x < 0)
      debug(F101,"sxeof nxtpkt fails","",pktnum);
    else
      debug(F101,"sxeof packet","",pktnum);
    return(szeof(s));
}

/*  S E O T -- Send an End-Of-Transaction packet */

int
seot() {
    int x;
    x = nxtpkt();
    debug(F101,"seot nxtpkt","",x);    
    if (x < 0) return(-1);		/* Bump packet number, get buffer */
    x = spack('B',pktnum,0,(CHAR *)"");	/* Send the EOT packet */
    if (x < 0)
      return(x);
    cxseen = czseen = discard = 0;	/* Reset interruption flags */
    tstats();				/* Log timing info */
    return(0);
}


/*   R P A R -- Fill the data array with my send-init parameters  */

CHAR dada[32];				/* Use this instead of data[]. */
					/* To avoid some kind of wierd */
					/* addressing foulup in spack()... */
					/* (which might be fixed now...) */

CHAR *
rpar() {
    char *p;
    int i, x, max;
    extern int xfermode;
    extern int sprmlen;

    max = maxdata();			/* Biggest data field I can send */
    debug(F101, "rpar max 1","",max);
    debug(F101, "rpar sprmlen","",sprmlen);
    if (sprmlen > 1 && sprmlen < max)	/* User override */
      max = sprmlen;
    debug(F101, "rpar max 2","",max);

    if (rpsiz > MAXPACK)		/* Biggest normal packet I want. */
      dada[0] = (char) tochar(MAXPACK);	/* If > 94, use 94, but specify */
    else				/* extended packet length below... */
      dada[0] = (char) tochar(rpsiz);	/* else use what the user said. */
    dada[1] = (char) tochar(chktimo(pkttim,0)); /* When to time me out */
    dada[2] = (char) tochar(mypadn);	/* How much padding I need (none) */
    dada[3] = (char) ctl(mypadc);	/* Padding character I want */
    dada[4] = (char) tochar(eol);	/* End-Of-Line character I want */
    dada[5] = myctlq;			/* Control-Quote character I send */

    if (max < 6) { dada[6] = NUL; rqf = 0; ebq = sq = NUL; return(dada); }

    switch (rqf) {			/* 8th-bit prefix (single-shift) */
      case -1:				/* I'm opening the bids */
      case  1:				/* Other Kermit already bid 'Y' */
	if (parity) ebq = sq = MYEBQ;	/* So I reply with '&' if parity */
	break;				/*  otherwise with 'Y'. */
      case  0:				/* Other Kermit bid nothing */
      case  2:				/* Other Kermit sent a valid prefix */
	break;				/* So I reply with 'Y'. */
    }
    debug(F000,"rpar 8bq sq","",sq);
    debug(F000,"rpar 8bq ebq","",ebq);
    if (lscapu == 2)			/* LOCKING-SHIFT FORCED */
      dada[6] = 'N';			/* requires no single-shift */
    else				/* otherwise send prefix or 'Y' */
      dada[6] = (char) sq;
    ebqsent = dada[6];			/* And remember what I really sent */

    if (max < 7) { dada[7] = NUL; bctr = 1; return(dada); }

    dada[7] = (char) (bctr == 4) ? 'B' : bctr + '0'; /* Block check type */

    if (max < 8) { dada[8] = NUL; rptflg = 0; return(dada); }

    if (rptena) {
	if (rptflg)			/* Run length encoding */
	  dada[8] = (char) rptq;	/* If receiving, agree */
	else				/* by replying with same character. */
	  dada[8] = (char) (rptq = myrptq); /* When sending use this. */
    } else dada[8] = SP;		/* Not enabled, put a space here. */

    /* CAPAS mask */

    if (max < 9) {
	dada[9] = NUL;
	atcapr = 0;
	lpcapr = 0;
	swcapr = 0;
	rscapr = 0;
	return(dada);
    }
    dada[9] = (char) tochar((lscapr ? lscapb : 0) | /* Locking shifts */
		     (atcapr ? atcapb : 0) | /* Attribute packets */
		     (lpcapr ? lpcapb : 0) | /* Long packets */
		     (swcapr ? swcapb : 0) | /* Sliding windows */
		     (rscapr ? rscapb : 0)); /* RESEND */
    if (max < 10) { wslotr = 1; return(dada); }
    dada[10] = (char) tochar(swcapr ? wslotr : 1); /* CAPAS+1 = Window size */

    if (max < 12) { rpsiz = 80; return(dada); }
    if (urpsiz > 94)
      rpsiz = urpsiz - 1;		/* Long packets ... */
    dada[11] = (char) tochar(rpsiz / 95); /* Long packet size, big part */
    dada[12] = (char) tochar(rpsiz % 95); /* Long packet size, little part */

    if (max < 16) return(dada);
    dada[13] = '0';			/* CAPAS+4 = WONT CHKPNT */
    dada[14] = '_';			/* CAPAS+5 = CHKINT (reserved) */
    dada[15] = '_';			/* CAPAS+6 = CHKINT (reserved) */
    dada[16] = '_';			/* CAPAS+7 = CHKINT (reserved) */
    if (max < 17) return(dada);
#ifndef WHATAMI
    dada[17] = ' ';
#else
    x = 0;
    if (server) x |= WMI_SERVE;		/* Whether I am a server */
    if (binary) x |= WMI_FMODE;		/* My file transfer mode is ... */
    if (fncnv)  x |= WMI_FNAME;		/* My filename conversion is ... */
#ifdef STREAMING
    if (streamrq == SET_ON)		/* Whether I can stream */
      x |= WMI_STREAM;
    else if (streamrq == SET_AUTO && reliable > 0)
      x |= WMI_STREAM;
#endif /* STREAMING */
#ifdef TCPSOCKET    
    if (clearrq == SET_ON)
      x |= WMI_CLEAR;
    else if (clearrq == SET_AUTO &&	/* SET CLEAR-CHANNEL AUTO */
	     ((network && nettype == 1)	/* We have a NET_TCPB connection */
	     || (reliable > 0)))	/* or RELIABLE is ON */
      x |= WMI_CLEAR;
#endif /* TCPSOCKET */
    x |= WMI_FLAG;
    dada[17] = (char) tochar(x);
#endif /* WHATAMI */
    i = 18;				/* Position of next field */
    if (xfermode == XMODE_A) {		/* If TRANSFER MODE AUTOMATIC */
	p = cksysid;			/* WHOAMI (my system ID) */
	x = strlen(p);
	if (max - i < x + 1) return(dada);
	if (x > 0) {
	    dada[i++] = (char) tochar(x);
	    while (*p)
	      dada[i++] = *p++;
	}
    } else {				/* TRANSFER MODE MANUAL */
	if (max - i < 2) return(dada);
	dada[i++] = (char) tochar(1);	/* Length */
	dada[i++] = '0';		/* "0" means anonymous - don't do it */
    }
    dada[i] = '\0';			/* Terminate the init string */

#ifdef DEBUG
    if (deblog) {
	debug(F110,"rpar",dada,0);
	rdebu(dada,(int)strlen((char *)dada));
    }
#endif /* DEBUG */
    strcpy((char *)myinit,(char *)dada);
    return(dada);			/* Return pointer to string. */
}

int
spar(s) CHAR *s; {			/* Set parameters */
    int x, y, lpsiz, biggest;
    extern int rprmlen, lastspmax;

    biggest = rln;
    debug(F101, "rpar biggest 1","",biggest);
    debug(F101, "rpar rprmlen","",rprmlen);
    if (rprmlen > 1 && rprmlen < biggest)
      biggest = rprmlen;
    debug(F101, "rpar biggest 2","",biggest);
    debug(F110,"spar packet",s,0);

    s--;				/* Line up with field numbers. */

/* Limit on size of outbound packets */
    x = (biggest >= 1) ? xunchar(s[1]) : 80;
    lpsiz = spsizr;			/* Remember what they SET. */
    if (spsizf) {			/* SET-command override? */
	if (x < spsizr) spsiz = x;	/* Ignore LEN unless smaller */
    } else {				/* otherwise */
	spsiz = (x < 10) ? 80 : x;	/* believe them if reasonable */
    }
    spmax = spsiz;			/* Remember maximum size */

/* Timeout on inbound packets */
    if (timef) {
	timint = rtimo;			/* SET SEND TIMEOUT value overrides */
    } else {				/* Otherwise use requested value, */
	x = (biggest >= 2) ? xunchar(s[2]) : rtimo; /* if it is legal. */
	timint = (x < 0) ? rtimo : x;
    }
    timint = chktimo(timint,timef);	/* Adjust if necessary */

/* Outbound Padding */
    npad = 0; padch = '\0';
    if (biggest >= 3) {
	npad = xunchar(s[3]);
	if (biggest >= 4) padch = (CHAR) ctl(s[4]); else padch = 0;
    }
    if (npad) {
	int i;
	for (i = 0; i < npad; i++) padbuf[i] = dopar(padch);
    }

/* Outbound Packet Terminator */
    seol = (CHAR) (biggest >= 5) ? xunchar(s[5]) : CR;
    if ((seol < 1) || (seol > 31)) seol = CR;

/* Control prefix that the other Kermit is sending */
    x = (biggest >= 6) ? s[6] : '#';
    ctlq = (CHAR) (((x > 32 && x < 63) || (x > 95 && x < 127)) ? x : '#');

/* 8th-bit prefix */
/*
  NOTE: Maybe this could be simplified using rcvtyp.
  If rcvtyp == 'Y' then we're reading the ACK,
  otherwise we're reading the other Kermit's initial bid.
  But his horrendous code has been working OK for years, so...
*/
    rq = (biggest >= 7) ? s[7] : 0;
    if (rq == 'Y') rqf = 1;
      else if ((rq > 32 && rq < 63) || (rq > 95 && rq < 127)) rqf = 2;
        else rqf = 0;
    debug(F000,"spar 8bq rq","",rq);
    debug(F000,"spar 8bq sq","",sq);
    debug(F000,"spar 8bq ebq","",ebq);
    debug(F101,"spar 8bq rqf","",rqf);
    switch (rqf) {
      case 0:				/* Field is missing from packet. */
	ebqflg = 0;			/* So no 8th-bit prefixing. */
	break;
      case 1:				/* Other Kermit sent 'Y' = Will Do. */
	/*
          When I am the file receiver, ebqsent is 0 because I didn't send a
          negotiation yet.  If my parity is set to anything other than NONE,
          either because my user SET PARITY or because I detected parity bits
          on this packet, I reply with '&', otherwise 'Y'.

	  When I am the file sender, ebqsent is what I just sent in rpar(),
          which can be 'Y', 'N', or '&'.  If I sent '&', then this 'Y' means
          the other Kermit agrees to do 8th-bit prefixing.

          If I sent 'Y' or 'N', but then detected parity on the ACK packet
          that came back, then it's too late: there is no longer any way for
          me to tell the other Kermit that I want to do 8th-bit prefixing, so
          I must not do it, and in that case, if there is any 8-bit data in 
          the file to be transferred, the transfer will fail because of block
          check errors.

          The following clause covers all of these situations:
	*/
	if (parity && (ebqsent == 0 || ebqsent == '&')) {
	    ebqflg = 1;
	    ebq = MYEBQ;
	}
	break;
      case 2:				/* Other Kermit sent a valid prefix */
	if (ebqflg = (ebq == sq || sq == 'Y')) {
	    ebq = rq;
	    debug(F101,"spar setting parity to space","",ebq);
	    if (!parity) parity = ttprty = 's';
	}
    }
    if (lscapu == 2) {     /* But no single-shifts if LOCKING-SHIFT FORCED */
	ebqflg = 0;
	ebq = 'N';
    }

/* Block check */
    x = 1;
    if (biggest >= 8) {
	if (s[8] == 'B') x = 4;
	else x = s[8] - '0';
	if ((x < 1) || (x > 4)) x = 1;
    }
    bctr = x;

/* Repeat prefix */

    rptflg = 0;				/* Assume no repeat-counts */
    if (biggest >= 9) {			/* Is there a repeat-count field? */
	char t;				/* Yes. */
	t = s[9];			/* Get its contents. */
/*
  If I'm sending files, then I'm reading these parameters from an ACK, and so
  this character must agree with what I sent.
*/
	if (rptena) {			/* If enabled ... */
	    if ((char) rcvtyp == 'Y') {	/* Sending files, reading ACK. */
		if (t == myrptq) rptflg = 1;
	    } else {			/* I'm receiving files */
		if ((t > 32 && t < 63) || (t > 95 && t < 127)) {
		    rptflg = 1;
		    rptq = t;
		}
	    }
	} else rptflg = 0;
    }

/* Capabilities */

    atcapu = lpcapu = swcapu = rscapu = 0; /* Assume none of these. */
    if (lscapu != 2) lscapu = 0;	/* Assume no LS unless forced. */
    y = 11;				/* Position of next field, if any */
    if (biggest >= 10) {
        x = xunchar(s[10]);
	debug(F101,"spar capas","",x);
        atcapu = (x & atcapb) && atcapr; /* Attributes */
	lpcapu = (x & lpcapb) && lpcapr; /* Long packets */
	swcapu = (x & swcapb) && swcapr; /* Sliding windows */
	rscapu = (x & rscapb) && rscapr; /* RESEND */
	debug(F101,"spar lscapu","",lscapu);
	debug(F101,"spar lscapr","",lscapr);
	debug(F101,"spar ebqflg","",ebqflg);
	if (lscapu != 2) lscapu = ((x & lscapb) && lscapr && ebqflg) ? 1 : 0;
	debug(F101,"spar swcapr","",swcapr);
	debug(F101,"spar swcapu","",swcapu);
	debug(F101,"spar lscapu","",lscapu);
	for (y = 10; (xunchar(s[y]) & 1) && (biggest >= y); y++);
	debug(F101,"spar y","",y);
    }

/* Long Packets */
    debug(F101,"spar lpcapu","",lpcapu);
    if (lpcapu) {
        if (biggest > y+1) {
	    x = xunchar(s[y+2]) * 95 + xunchar(s[y+3]);
	    debug(F101,"spar lp len","",x);
	    if (spsizf) {		/* If overriding negotiations */
		spsiz = (x < lpsiz) ? x : lpsiz; /* do this, */
	    } else {			         /* otherwise */
		spsiz = (x > MAXSP) ? MAXSP : x; /* do this. */
	    }
	    if (spsiz < 10) spsiz = 80;	/* Be defensive... */
	}
    }
    /* (PWP) save current send packet size for optimal packet size calcs */
    spmax = spsiz;			/* Maximum negotiated length */
    lastspmax = spsiz;			/* For stats */
    if (slostart && spsiz > 499)	/* Slow start length */
      spsiz = 244;
    debug(F101,"spar slow-start spsiz","",spsiz);
    debug(F101,"spar lp spmax","",spmax);
    timint = chktimo(timint,timef);	/* Recalculate the packet timeout */
    
/* Sliding Windows... */

    if (swcapr) {			/* Only if requested... */
        if (biggest > y) {		/* See what other Kermit says */
	    x = xunchar(s[y+1]);
	    debug(F101,"spar window","",x);
	    wslotn = (x > MAXWS) ? MAXWS : x;
/*
  wslotn = negotiated size (from other Kermit's S or I packet).
  wslotr = requested window size (from this Kermit's SET WINDOW command).
*/
	    if (wslotn > wslotr)	/* Use the smaller of the two */
	      wslotn = wslotr;
	    if (wslotn < 1)		/* Watch out for bad negotiation */
	      wslotn = 1;
	    if (wslotn > 1) {
		swcapu = 1;		/* We do windows... */
		if (wslotn > maxtry)	/* Retry limit must be greater */
		  maxtry = wslotn + 1;	/* than window size. */
	    }
	    debug(F101,"spar window after adjustment","",x);
	} else {			/* No window size specified. */
	    wslotn = 1;			/* We don't do windows... */
	    debug(F101,"spar window","",x);
	    swcapu = 0;
	    debug(F101,"spar no windows","",wslotn);
	}
    }

/* Now recalculate packet length based on number of windows.   */
/* The nogotiated number of window slots will be allocated,    */
/* and the maximum packet length will be reduced if necessary, */
/* so that a windowful of packets can fit in the big buffer.   */

    if (wslotn > 1) {			/* Shrink to fit... */
	x = adjpkl(spmax,wslotn,bigsbsiz);
	if (x < spmax) {
	    spmax = x;
	    lastspmax = spsiz; 
	    if (slostart && spsiz > 499) spsiz = 244; /* Slow start again */
	    debug(F101,"spar sending, redefine spmax","",spmax);
	}
    }
#ifdef WHATAMI
    if (biggest > y+7)			/* Get WHATAMI info if any */
      whatru = xunchar(s[y+8]);
    if (whatru & WMI_FLAG) {		/* Only valid if this bit is set */
#ifdef STREAMING
	streamok = 0;
	streaming = 0;
	if (whatru & WMI_STREAM) {
	    if (streamrq == SET_ON ||
		(streamrq == SET_AUTO && reliable)) {
		streamok = 1;		/* Streaming negotiated */
		slostart = 0;		/* Undo slow-start machinations */
		spsiz = lastspmax;
	    }
	}
	streamed = streamok;
	debug(F101,"spar streamok","",streamok);
	urclear = (whatru & WMI_CLEAR);
	debug(F101,"spar urclear","",urclear);
#ifdef CK_SPEED
	if (urclear)
	  setprefix(PX_NON);
#endif /* CK_SPEED */
	cleared = urclear;
#endif /* STREAMING */
    }
#endif /* WHATAMI */

    if (biggest > y+8) {		/* Get WHOAREYOU info if any */
	int x;
	x = xunchar(s[y+9]);		/* Length of it */
	if (x > 0 && x < 16 && biggest >= y + 9 + x) {
	    strncpy(whoareu,(char *)s+y+10,x); /* Other Kermit's system ID */
	    if (whoareu[0]) {		/* Got one? */
		char *p;
#ifdef COMMENT
		char buf[64];
#endif /* COMMENT */
		extern int sysindex;
		extern struct sysdata sysidlist[];

		sysindex = getsysix((char *)whoareu);
		if (sysindex > -1) {
		    p = sysidlist[sysindex].sid_name;
#ifdef COMMENT
		    sprintf(buf,"Remote system type is %s", p);
		    xxscreen(SCR_ST,ST_MSG,0L,buf);
#endif /* COMMENT */
		    tlog(F110,"Remote system type: ",p,0L);
		    if (sysindex > 0) {	/* Skip this for "anonymous" */
			whoarewe();
#ifdef CK_SPEED
/* Never unprefix XON and XOFF when sending to VMS */
			debug(F111,"proto whoareu",whoareu,sysindex);
			if (!strcmp((char *)whoareu,"D7")) {
			    debug(F111,
				  "proto special VMS prefixing","",0);
			    ctlp[XON] = ctlp[XOFF] = 1;
			    ctlp[XON+128] = ctlp[XOFF+128] = 1;
			    ctlp[3] = 1; /* Ctrl-C might be dangerous too */
			    ctlp[14] = ctlp[15] = 1; /* And SO/SI */
			    ctlp[24] = ctlp[25] = 1; /* And ^X/^Y */
			    ctlp[141] = 1; /* And CR+128 */
			}
#endif /* CK_SPEED */
		    }
		}
	    }
	}
    }
/* Record parameters in debug log */
#ifdef DEBUG
    if (deblog) sdebu(biggest);
#endif /* DEBUG */
    numerrs = 0;			/* Start counting errors here. */
    return(0);
}

/* Criteria used by gnfile()... */

char sndafter[19]   = { NUL, NUL };
char sndbefore[19]  = { NUL, NUL };
char sndnafter[19]  = { NUL, NUL };
char sndnbefore[19] = { NUL, NUL };
char *sndexcept[8]  = { NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL };
long sndsmaller = -1L;
long sndlarger  = -1L;

/*  F I L E S E L E C T  --  Select this file for sending  */

static int
fileselect(f) char * f; {
    char *fdate, *s, *p;
    int n, q, bc; long z;

    debug(F110,"fileselect",f,0);
    if (sndafter[0] || sndbefore[0] || sndnafter[0] || sndnbefore[0]) {
	fdate = zfcdat(f);		/* Mod date/time of this file */
	if (!fdate) fdate = "";
	n = strlen(fdate);
	debug(F111,"fileselect fdate",fdate,n);
	if (n != 17)			/* Failed to get it */
	  return(1);
	/* /AFTER: */
	if (sndafter[0] && (strcmp(fdate,(char *)sndafter) <= 0)) {
	    debug(F110,"fileselect sndafter",sndafter,0);
	    /* tlog(F110,"Skipping (too old)",f,0); */
	    return(0);
	}
	/* /BEFORE: */
	if (sndbefore[0] && (strcmp(fdate,(char *)sndbefore) >= 0)) {
	    debug(F110,"fileselect sndbefore",sndbefore,0);
	    /* tlog(F110,"Skipping (too new)",f,0); */
	    return(0);
	}
	/* /NOT-AFTER: */
	if (sndnafter[0] && (strcmp(fdate,(char *)sndnafter) > 0)) {
	    debug(F110,"fileselect sndnafter",sndnafter,0);
	    /* tlog(F110,"Skipping (too new)",f,0); */
	    return(0);
	}
	/* /NOT-BEFORE: */
	if (sndnbefore[0] && (strcmp(fdate,(char *)sndnbefore) < 0)) {
	    debug(F110,"fileselect sndnbefore",sndnbefore,0);
	    /* tlog(F110,"Skipping (too old)",f,0); */
	    return(0);
	}
    }
    if (sndsmaller > -1L || sndlarger > -1L) { /* Smaller or larger */
	z = zchki(f);			       /* Get size */
	debug(F101,"fileselect filesize","",z);	
	if (z < 0)
	  return(1);
	if ((sndsmaller > -1L) && (z >= sndsmaller)) {
	    debug(F111,"fileselect sndsmaller skipping",f,sndsmaller);
	    /* tlog(F111,"Skipping (too big)",f,z); */
	    return(0);
	}
	if ((sndlarger > -1L) && (z <= sndlarger)) {
	    debug(F111,"fileselect sndlarger skipping",f,sndlarger);
	    /* tlog(F110,"Skipping (too small)",f,0); */
	    return(0);
	}
    }
    for (n = 0; n < 8; n++) {
	if (!sndexcept[n]) {
	    debug(F101,"fileselect sndexcept empty",0,n);
	    break;
	}
	if (ckmatch(sndexcept[n],f,alphacase)) {
	    debug(F111,"fileselect sndexcept",sndexcept[n],n);
	    debug(F110,"fileselect skipping",f,0);
	    return(0);
	}
    }
    debug(F110,"fileselect selecting",f,0);
    return(1);
}

/*  G N F I L E  --  Get name of next file to send  */
/*
  Expects global sndsrc to be:
   -9: if we are generating a file internally for calibration.
   -1: next filename to be obtained by calling znext().
    0: no next file name
    1: (or greater) next filename to be obtained from **cmlist,
       or if addlist != 0, from the "filehead" linked list,
       or if filefile pointer not null from that file (which is already open).
  Returns:
    1, with name of next file in filnam.
    0, no more files, with filnam set to empty string.
   -1, file not found
   -2, file is not readable (but then we just skip to the next one if any)
   -3, read access denied
   -4, cancelled
   -5, too many files match wildcard
*/
int
gnfile() {
    int i, x; long y;
    int retcode = 0;
    char fullname[CKMAXPATH+1];

    debug(F101,"gnfile sndsrc","",sndsrc);
    debug(F101,"gnfile filcnt","",filcnt);
    debug(F101,"gnfile what","",what);
    fsize = -1L;			/* Initialize file size */
    fullname[0] = NUL;
    if (what != W_REMO && xfermode == XMODE_A) {
#ifdef WHATAMI
	/* We don't do this in server mode because it undoes WHATAMI */
	if (!server || (server && (whatru & WMI_FLAG == 0)))
#endif /* WHATAMI */
	  binary = gnf_binary;		/* Restore prevailing transfer mode */
	debug(F101,"gnfile binary = gnf_binary","",gnf_binary);
    }
#ifdef PIPESEND
    debug(F101,"gnfile pipesend","",pipesend);
    if (pipesend) {			/* First one */
	if (filcnt == 0) {
	    strcpy(filnam,cmarg);
	    return(1);
	} else {			/* There's only one... */
	    *filnam = NUL;
	    pipesend = 0;
	    return(0);
	}
    }
#endif /* PIPESEND */

#ifdef CALIBRATE
    if (sndsrc == -9) {
	strcpy(filnam,"CALIBRATION");
	nfils = 0;
	cal_j = 0;
	fsize = calibrate;
	sndsrc = 0;			/* For next time */
	nfils = 0;
	return(1);
    }
#endif /* CALIBRATE */

    if (sndsrc == 0) {			/* It's not really a file */
	if (nfils > 0) {		/* It's a pipe, or stdin */
	    strcpy(filnam, *cmlist);	/* Copy its "name" */
	    nfils = 0;			/* There is no next file */
	    return(1);			/* OK this time */
	} else return(0);		/* but not next time */
    }

/* If file group interruption (C-Z) occurred, fail.  */

    if (czseen) {
	tlog(F100,"Transaction cancelled","",0L);
        debug(F100,"gnfile czseen","",0);
	return(-4);
    }

/* Loop through file list till we find a readable, sendable file */

    y = -1L;				/* Loop exit (file size) variable */
    while (y < 0L) {			/* Keep trying till we get one... */
	if (sndsrc > 0) {		/* File list in cmlist or file */
	    if (filefile) {		/* Reading list from file... */
		if (zsinl(ZMFILE,filnam,CKMAXPATH) < 0) { /* Read a line */
		    zclose(ZMFILE);	                  /* Failed */
		    debug(F110,"gnfile filefile EOF",filefile,0);
		    makestr(&filefile,NULL);
		    return(0);
		}
		debug(F110,"gnfile filefile filnam",filnam,0);
	    }
	    debug(F101,"gnfile nfils","",nfils);
	    if (nfils-- > 0 || filefile) { /* Still some left? */
#ifndef NOMSEND
		if (addlist) {
		    if (filenext && filenext->fl_name) {
			strncpy(filnam,filenext->fl_name,CKMAXPATH);
			cmarg2 =
			  filenext->fl_alias ?
			    filenext->fl_alias :
			      "";
			binary = filenext->fl_mode;
		    } else {
			printf("?Internal error expanding ADD list\n");
			return(-5);
		    }
		    filenext = filenext->fl_next;
		    debug(F111,"gnfile addlist filnam",filnam,nfils);
		} else if (sndsrc > 0 && !filefile) {
#endif /* NOMSEND */
		    strncpy(filnam,*cmlist++,CKMAXPATH);
		    debug(F111,"gnfile cmlist filnam",filnam,nfils);
#ifndef NOMSEND
		}
#endif /* NOMSEND */
		i = 0;
#ifndef NOSERVER
		debug(F101,"gnfile ngetpath","",ngetpath);
#endif /* NOSERVER */
nextinpath:
#ifndef NOSERVER
		fromgetpath = 0;
		if (server && !isabsolute(filnam) && (ngetpath > i)) {
		    strncpy(fullname,getpath[i],CKMAXPATH);
		    strncat(fullname,filnam,CKMAXPATH);
		    debug(F111,"gnfile getpath",fullname,i);
		    fromgetpath = 1;
		    i++;
		} else {
		    i = ngetpath + 1;
#else
		    i = 1;		/* ? */
#endif /* NOSERVER */
		    strncpy(fullname,filnam,CKMAXPATH);
		    debug(F110,"gnfile absolute",fullname,0);
#ifndef NOSERVER
		}
#endif /* NOSERVER */
		if (iswild(fullname)
#ifdef RECURSIVE
		    || recursive > 0 || !strcmp(fullname,".")
#endif /* RECURSIVE */
		    ) {	/* It looks wild... */
		    /* First check if a file with this name exists */
		    debug(F110,"gnfile wild",fullname,0);
		    if (zchki(fullname) > -1) {
			/*
			   Here we have a file whose name actually
			   contains wildcard characters.
			*/
			goto gotnam;
		    }
		    x = zxpand(fullname); /* Now try to expand wildcards */
		    debug(F101,"gnfile zxpand","",x);
		    if (x == 1) {
			znext(fullname);
			debug(F110,"gnfile znext",fullname,0);
			goto gotnam;
		    }
		    if (x == 0) {	/* None match */
#ifndef NOSERVER
			if (server && ngetpath > i)
			  goto nextinpath;
#endif /* NOSERVER */
			retcode = -1;
			continue;
		    }
		    if (x < 0)		/* Too many to expand */
		      return(-5);
		    sndsrc = -1;	/* Change send-source to znext() */
		}
	    } else {			/* We're out of files. */
		debug(F101,"gnfile done","",nfils);
		*filnam = '\0';
		return(retcode);
	    }
	}

/* Otherwise, step to next element of internal wildcard expansion list. */

	if (sndsrc == -1) {
	    x = znext(filnam);
	    debug(F111,"gnfile znext",filnam,x);
	    if (!filnam[0]) {		/* If no more, */
		sndsrc = 1;		/* go back to previous list */
		debug(F101,"gnfile setting sndsrc back","",sndsrc);
		continue;
	    } else strncpy(fullname,filnam,CKMAXPATH);
	}

/* Get here with a filename. */

gotnam:
	if (fullname[0]) {
#ifdef DTILDE
	    char * dirp = "";
	    if (fullname[0] == '~') {
		dirp = tilde_expand((char *)fullname);
		if (*dirp) strncpy(fullname,dirp,CKMAXPATH);
	    }
#endif /* DTILDE */
	    y = zchki(fullname);	/* Check if file readable */
	    retcode = (int) y;		/* Possible return code */
	    if (y == -1L) {		/* If not found */
#ifndef NOSERVER
		if (server && ngetpath > i)
		  goto nextinpath;
#endif /* NOSERVER */
		debug(F110,"gnfile skipping:",fullname,0);
		tlog(F111,fullname,"not sent",0);
		xxscreen(SCR_ST,ST_SKIP,0l,fullname);
#ifdef TLOG
		if (tralog && !tlogfmt)
		  doxlog(what,fullname,fsize,binary,1,"Skipped");
#endif /* TLOG */
		continue;
	    } else if (y < 0) {
		if (y == -3) {		/* Exists but not readable */
		    filrej++;		/* Count this one as not sent */
		    tlog(F110,"Read access denied",fullname,0); /* Log this */
		    xxscreen(SCR_ST,ST_SKIP,0l,fullname); /* Display message */
#ifdef TLOG
		    if (tralog && !tlogfmt)
		      doxlog(what,fullname,fsize,binary,1,"Skipped");
#endif /* TLOG */
		}
		continue;
	    } else {
		fsize = y;
		if (!fileselect(fullname)) {
		    y = -1L;
		    continue;
		}
		strncpy(filnam,fullname,CKMAXPATH);
		return(1);
	    }
	} else {			/* sndsrc is 0... */
	    if (!fileselect(fullname)) {
		y = -1L;
		continue;
	    }
	    strncpy(filnam,fullname,CKMAXPATH);
	    return(1);
	}
    }
    *filnam = '\0';			/* Should never get here */
    return(0);
}

/*
  The following bunch of routines feed internally generated data to the server
  to send to the client in response to REMOTE commands like DIRECTORY, DELETE,
  and so on.  We have to write these lines in the format appropriate to our
  platform, so they can be converted to generic (CRLF) text format by the
  packetizer.
*/
#ifdef UNIX
char * endline = "\12";
#else
#ifdef datageneral
char * endline = "\12";
#else
#ifdef MAC
char * endline = "\15";
#else
#ifdef OSK
char * endline = "\15";
#else
char * endline = "\15\12";
#endif /* OSK */
#endif /* MAC */
#endif /* datageneral */
#endif /* UNIX */

static
char funcbuf[512];
static int
  funcnxt = 0,
  funclen = 0,
  nxpnd = - 1;
static long
  ndirs =   0,
  nfiles =  0,
  nbytes =  0;


sndstring(p) char * p; {
#ifndef NOSERVER
    nfils = 0;				/* No files, no lists. */
    xflg = 1;				/* Flag we must send X packet. */
    strcpy(cmdstr,versio);		/* Data for X packet. */
    first = 1;				/* Init getchx lookahead */
    memstr = 1;				/* Just set the flag. */
    memptr = p;				/* And the pointer. */
    binary = XYFT_T;			/* Text mode for this. */
    return(sinit());
#else
    return(0);
#endif /* NOSERVER */
}

/*  S N D H L P  --  Routine to send builtin help  */

static int srvhlpnum = 0;
char *nm[] =  { "disabled", "local only", "remote only", "enabled" };

int
nxthlp() {
    extern int
      en_cpy, en_cwd, en_del, en_dir, en_fin, en_get, en_bye, en_mai, 
      en_pri, en_hos, en_ren, en_sen, en_spa, en_set, en_typ, en_who, 
      en_ret, en_mkd, en_rmd, en_asg, en_que;
    extern char * ckxsys;
    if (funcnxt < funclen)
      return (funcbuf[funcnxt++]);
    switch (srvhlpnum++) {
      case 0:
	sprintf(funcbuf," GET                %s\n",nm[en_get]); break;
      case 1:
	sprintf(funcbuf," SEND               %s\n",nm[en_sen]); break;
      case 2:
	sprintf(funcbuf," MAIL               %s\n",nm[en_mai]); break;
      case 3:
	sprintf(funcbuf," PRINT              %s\n",nm[en_pri]); break;
      case 4:
#ifndef NOSPL
	sprintf(funcbuf," REMOTE ASSIGN      %s\n",nm[en_asg]);
#else
	sprintf(funcbuf," REMOTE ASSIGN      not configured\n");
#endif /* NOSPL */
	break;
      case 5:
	sprintf(funcbuf," REMOTE CD/CWD      %s\n",nm[en_cwd]); break;
      case 6:
#ifdef ZCOPY
	sprintf(funcbuf," REMOTE COPY        %s\n",nm[en_cpy]); 
#else
	sprintf(funcbuf," REMOTE COPY        not configured\n"); 
#endif /* ZCOPY */
	break;
      case 7:
	sprintf(funcbuf," REMOTE DELETE      %s\n",nm[en_del]); break;
      case 8:
	sprintf(funcbuf," REMOTE DIRECTORY   %s\n",nm[en_dir]); break;
      case 9:
	sprintf(funcbuf," REMOTE HOST        %s\n",nm[en_hos]); break;
      case 10:
#ifndef NOSPL
	sprintf(funcbuf," REMOTE QUERY       %s\n",nm[en_que]); 
#else
	sprintf(funcbuf," REMOTE QUERY       not configured\n"); 
#endif /* NOSPL */
	break;
      case 11:
	sprintf(funcbuf," REMOTE MKDIR       %s\n",nm[en_mkd]); break;
      case 12:
	sprintf(funcbuf," REMOTE RMDIR       %s\n",nm[en_rmd]); break;
      case 13:
	sprintf(funcbuf," REMOTE RENAME      %s\n",nm[en_ren]); break;
      case 14:
	sprintf(funcbuf," REMOTE SET         %s\n",nm[en_set]); break;
      case 15:
	sprintf(funcbuf," REMOTE SPACE       %s\n",nm[en_spa]); break;
      case 16:
	sprintf(funcbuf," REMOTE TYPE        %s\n",nm[en_typ]); break;
      case 17:
	sprintf(funcbuf," REMOTE WHO         %s\n",nm[en_who]); break;
      case 18:
	sprintf(funcbuf," BYE                %s\n",nm[en_bye]); break;
      case 19:
	sprintf(funcbuf," FINISH             %s\n",nm[en_fin]); break;
      case 20:
	return(-1);
    }
    funcnxt = 0;
    funclen = strlen(funcbuf);
    return(0);
}

int
sndhlp() {
#ifndef NOSERVER
    extern char * ckxsys;
    first = 1;				/* Init getchx lookahead */
    nfils = 0;				/* No files, no lists. */
    xflg = 1;				/* Flag we must send X packet. */
    strcpy(cmdstr,"REMOTE HELP");	/* Data for X packet. */
    sprintf(funcbuf,"C-Kermit %s,%s\n SERVER FUNCTIONS:\n", versio, ckxsys);
    funcnxt = 0;
    funclen = strlen(funcbuf);
    funcptr = nxthlp;
    funcstr = 1;
    srvhlpnum = 0;
    binary = XYFT_T;			/* Text mode for this. */
    return(sinit());
#else
    return(0);
#endif /* NOSERVER */
}

/*
   Returns the next available character, 
  -1 if no more data.
*/
int 
nxttype() {
    int c;
    if (zchin(ZIFILE,&c) < 0) {
        zclose(ZIFILE);
        return(-1);
    } else {
	return((unsigned)c);
    }
}

/*  S N D T Y P -- TYPE a file to remote client */

int
#ifdef CK_ANSI
sndtype(char * file) 
#else
sndtype(file) char * file; 
#endif /* CK_ANSI */
/* sndtype */ {

#ifndef NOSERVER
    char * p = NULL, name[CKMAXPATH+1];

#ifdef OS2
    if (*file) {
        strcpy(name, file);
        /* change / to \. */
        p = name;
        while (*p) {			/* Change them back to \ */
            if (*p == '/') *p = '\\';
            p++;
        }
    } else
      return(0);
#else
    strcpy(name, file);
#endif /* OS2 */

    funcnxt = 0;
    funclen = strlen(funcbuf);
    if (zchki(name) == -2) {
        /* Found a directory */
        return(0);
    }
    if (!zopeni(ZIFILE,name))
      return(0);

    nfils = 0;				/* No files, no lists. */
    xflg = 1;				/* Flag we must send X packet. */
    strcpy(cmdstr,"type");		/* Data for X packet. */
    first = 1;				/* Init getchx lookahead */
    funcstr = 1;			/* Just set the flag. */
    funcptr = nxttype;			/* And the pointer. */
    binary = XYFT_T;			/* Text mode for this */
    return(sinit());
#else
    return(0);
#endif /* NOSERVER */
}

/*
   N X T D I R  --  Provide data for senddir()

   Returns the next available character or -1 if no more data.
*/
static int
nxtdir() {
    char name[257], *p = NULL;
    char * dstr = NULL;
    long len = 0;
    short month = 0, date = 0, year = 0, hour = 0, minute = 0, seconds = 0;

    debug(F111,"nxtdir","funcnxt",funcnxt);
    debug(F110,"nxtdir funcbuf",funcbuf+funcnxt,0);
    if (funcnxt < funclen)
      return (funcbuf[funcnxt++]);

    if (nxpnd > 0) {
        nxpnd--; 
	znext(name);
        if (!name[0]) {
            nxpnd = 0;
            return(nxtdir());
        }
        dstr = zfcdat(name);
	if (!dstr) dstr = "";
        if (*dstr) {
	    month = (dstr[4]-48)*10 + (dstr[5]-48);
#ifdef COMMENT
	    switch(month) {		/* For anglophones only... */
	      case 1:  mstr = "Jan"; break;
	      case 2:  mstr = "Feb"; break;
	      case 3:  mstr = "Mar"; break;
	      case 4:  mstr = "Apr"; break;
	      case 5:  mstr = "May"; break;
	      case 6:  mstr = "Jun"; break;
	      case 7:  mstr = "Jul"; break;
	      case 8:  mstr = "Aug"; break;
	      case 9:  mstr = "Sep"; break;
	      case 10: mstr = "Oct"; break;
	      case 11: mstr = "Nov"; break;
	      case 12: mstr = "Dec"; break;
	      default: mstr = "   ";
	    }
#endif /* COMMENT */
	    date = (dstr[6]-48)*10 + (dstr[7]-48);
	    year = (((dstr[0]-48)*10 + (dstr[1]-48))*10
		    + (dstr[2]-48))*10 + (dstr[3]-48);
	    hour = (dstr[9]-48)*10 + (dstr[10]-48);
	    minute = (dstr[12]-48)*10 + (dstr[13]-48);
	    seconds = (dstr[15]-48)*10 + (dstr[16]-48);
        } else {
            month = 0;
	    date = 0;
	    year = 0;
	    hour = 0;
	    minute = 0;
	    seconds = 0;
        }
        len = zchki(name);
        /* Find just the name of the file */
#ifdef VMS
	p = name + ckindex("]",name,0,0,1);
#else
        for (p = name + (int) strlen(name); 
	     p != name && *p != '/' 
#ifdef OS2
	     && *p != '\\' && *p != ':' 
#endif /* OS2 */              
              ; p--
	     );
        if (*p == '/' 
#ifdef OS2
             || *p == '\\' || *p == ':'
#endif /* OS2 */
            )
          p++;
#endif /* VMS */
        if (len > -1L) {
            nfiles++;
            nbytes += len;
            sprintf(funcbuf," %04d-%02d-%02d %02d:%02d %11ld %s%s",
                    year, month, date, hour, minute, len, p, endline);
        } else {
            ndirs++;
            sprintf(funcbuf,
		    " %04d-%02d-%02d %02d:%02d %11s %s%s", 
                    year, month, date, hour, minute,
		    "(directory)", p, endline);
        }
        funcnxt = 0;
        funclen = strlen(funcbuf);
    } else if (nxpnd == 0) {		/* Done, send summary */
	char *blankline = "";		/* At beginning of summary */
/*
  The idea is to prevent (a) unnecessary multiple blanklines, and (b)
  prompt-stomping.  Preventing (b) is practically impossible, because it
  depends on the client so for now always include that final CRLF.
*/
	if (!ndirs || !nbytes || !nfiles)
	  blankline = endline;
        sprintf(funcbuf,
		"%sSummary: %ld director%s, %ld file%s, %ld byte%s%s",
		blankline,
		ndirs,
		(ndirs == 1) ? "y" : "ies",
		nfiles,
		(nfiles == 1) ? "" : "s",
		nbytes,
		(nbytes == 1) ? "" : "s",
		endline
		);
        nxpnd--;
        funcnxt = 0;
        funclen = strlen(funcbuf);
    } else {
        funcbuf[0] = '\0';
        funcnxt = 0;
        funclen = 0;
    }
    /* If we have data to send... */

    if (funcnxt < funclen)
      return(funcbuf[funcnxt++]);	/* Return a character */
    else
      return(-1);			/* Nothing left, done. */
}

/*  S N D D I R -- send directory listing  */

int
#ifdef CK_ANSI
snddir(char * spec) 
#else
snddir(spec) char * spec; 
#endif /* CK_ANSI */
/* snddir */ {
#ifndef NOSERVER
    char * p = NULL, name[257];  
    int rc = 0;
    extern int dironly, fileonly;
    int fsave, dsave;

    fsave = fileonly;
    dsave = dironly;

    if (!spec) spec = "";
    debug(F110,"snddir",spec,0);
    if (*spec) {
        strcpy(name, spec);
#ifdef OS2
        /* change / to \. */
        p = name;
        while (*p) {			/* Change them back to \ */
            if (*p == '/') *p = '\\';
            p++;
        }
#endif /* OS2 */
    } else {
#ifdef OS2
	strcpy(name, ".");
#else
#ifdef UNIX
	strcpy(name, "./*");
#else
#ifdef VMS
	strcpy(name, "*.*");
#else
#ifdef datageneral
	strcpy(name, "+");
#else
	return(0);
#endif /* datageneral */
#endif /* VMS */
#endif /* UNIX */
#endif /* OS2 */
	p = name + strlen(name); 	/* Move it to end of list */
    }
    ndirs = 0L;
    nfiles = 0L;
    nbytes = 0L;
    sprintf(funcbuf,"Listing files: \"%s\"%s%s",name,endline,endline);
    funcnxt = 0;
    funclen = strlen(funcbuf);

    dironly = 0;
    fileonly = 0;

#ifdef OS2
    debug(F110,"snddir about to zchki",name,0);
    if (zchki(name) == -2) {		/* Found a directory */
        p--;
        if (*p == '\\' || *p == '/')
          strcat(name, "*");
        else if (*p == ':')
          strcat(name, ".");
        else
          strcat(name, "\\*");
    }
#endif /* OS2 */

    nxpnd = zxpand(name);
    debug(F111,"snddir zxpand","nxpnd",nxpnd);

    nfils = 0;				/* No files, no lists. */
    xflg = 1;				/* Flag we must send X packet. */
    strcpy(cmdstr,"REMOTE DIRECTORY");	/* Data for X packet. */
    first = 1;				/* Init getchx lookahead */
    funcstr = 1;			/* Just set the flag. */
    funcptr = nxtdir;			/* And the pointer. */
    rc = sinit();
    debug(F111,"snddir","sinit()",rc);
    dironly = dsave;
    fileonly = fsave;
    return(rc);
#else
    return(0);
#endif /* NOSERVER */
}

/*  N X T D E L -- provide data for delete   */

/*  Returns the next available character or -1 if no more data  */

static int
nxtdel() {
    char name[257], *p = NULL;
    int len = 0;

    if (funcnxt < funclen)
      return (funcbuf[funcnxt++]);

    if (nxpnd > 0) {
        nxpnd--; 
	znext(name);
        if (!name[0]) {
            nxpnd = 0;
            return(nxtdel());
        }
        len = zchki(name);

        /* Find just the name of the file */

        for (p = name + strlen(name); p != name && *p != '/' ; p--) ;
        if (*p == '/') p++;

        if (len > -1L) {
	    if (zdelet(name)) {
		sprintf(funcbuf," %10s: %s%s","skipping",p,endline);
	    } else {
		nfiles++;
		nbytes += len;
		sprintf(funcbuf," %10s: %s%s","deleted",p,endline);
	    }
        } else
	  sprintf(funcbuf," directory: %s%s", p, endline);
        funcnxt = 0;
        funclen = strlen(funcbuf);
    } else 
    
    /* If done processing the expanded entries send a summary statement */

      if (nxpnd == 0) {
	  sprintf(funcbuf,
		  "%s%ld file%s deleted, %ld byte%s freed%s",
		  endline,
		  nfiles,
		  (nfiles == 1) ? "" : "s",
		  nbytes,
		  (nbytes == 1) ? "" : "s",
		  endline
		  );
	  nxpnd--;
	  funcnxt = 0;
	  funclen = strlen(funcbuf);
      } else {
	  funcbuf[0] = '\0';
	  funcnxt = 0;
	  funclen = 0;
      }

    /* If we have data to send */

    if (funcnxt < funclen)
      return (funcbuf[funcnxt++]);	/* Return a character */
    else
      return(-1);			/* No more input */
}

/*  S N D D E L  --  Send delete message  */

int
#ifdef CK_ANSI
snddel(char * spec) 
#else
snddel(spec) char * spec; 
#endif /* CK_ANSI */
/* snddel */ {
#ifndef NOSERVER
    char * p=NULL, name[257]; 

    if (!*spec)
      return(0);

    strcpy(name, spec);

#ifdef OS2
    /* change / to \. */
    p = name;
    while (*p) {			/* Change them back to \ */
        if (*p == '/') *p = '\\';
        p++;
    }
#endif /* OS2 */

    nfiles = nbytes = 0L;
    sprintf(funcbuf,"Deleting \"%s\"%s",name,endline);
    funcnxt = 0;
    funclen = strlen(funcbuf);

    nxpnd = zxpand(name);
    nfils = 0;				/* No files, no lists. */
    xflg = 1;				/* Flag we must send X packet. */
    strcpy(cmdstr,"REMOTE DELETE");	/* Data for X packet. */
    first = 1;				/* Init getchx lookahead */
    funcstr = 1;			/* Just set the flag. */
    funcptr = nxtdel;			/* And the pointer. */
    binary = XYFT_T;			/* Use text mode for this, */
    return(sinit());
#else
    return(0);
#endif /* NOSERVER */
}

#ifdef OS2
/*  S N D S P A C E -- send disk space message  */
int
sndspace(int drive) {
#ifndef NOSERVER
    static char spctext[64];
    if (drive)
      sprintf(spctext,
	      " Drive %c: %ldK free%s",
	      drive, 
	      zdskspace(drive - 'A' + 1) / 1024L,
	      endline
	      );
    else
      sprintf(spctext, " Free space: %ldK%s", zdskspace(0) / 1024L, endline);
    nfils = 0;				/* No files, no lists. */
    xflg = 1;				/* Flag we must send X packet. */
    strcpy(cmdstr,"free space");	/* Data for X packet. */
    first = 1;				/* Init getchx lookahead */
    memstr = 1;				/* Just set the flag. */
    memptr = spctext;			/* And the pointer. */
    binary = XYFT_T;			/* Text mode for this. */
    return(sinit());
#else
    return(0);
#endif /* NOSERVER */
}

/*  S N D W H O -- send who message  */
int
sndwho(char * who) {
#ifndef NOSERVER
    nfils = 0;				/* No files, no lists. */
    xflg = 1;				/* Flag we must send X packet. */
    strcpy(cmdstr,"who");		/* Data for X packet. */
    first = 1;				/* Init getchx lookahead */
    memstr = 1;				/* Just set the flag. */
#ifdef NT
    memptr = "\15\12K95 SERVER\15\12";	/* And the pointer. */
#else
    memptr = "\15\12K/2 SERVER\15\12";
#endif /* NT */
    binary = XYFT_T;			/* Use text mode */
    return(sinit());
#else
    return(0);
#endif /* NOSERVER */
}
#endif /* OS2 */

/*  C W D  --  Change current working directory  */

/*
 String passed has first byte as length of directory name, rest of string
 is name.  Fails if can't connect, else ACKs (with name) and succeeds. 
*/

int
cwd(vdir) char *vdir; {
    char *cdd, *zgtdir(), *dirp;

    vdir[xunchar(*vdir) + 1] = '\0';	/* Terminate string with a null */
    dirp = vdir+1;
    tlog(F110,"Directory requested: ",dirp,0L);
    if (zchdir(dirp)) {			/* Try to change */
	cdd = zgtdir();			/* Get new working directory. */
	debug(F110,"cwd",cdd,0);
	encstr((CHAR *)cdd);
#ifdef COMMENT
	ack1((CHAR *)(encbuf+7));
#else
	ack1(data);
#endif /* COMMENT */
	xxscreen(SCR_CD,0,0l,cdd);
	tlog(F110,"Changed directory to",cdd,0L);
	return(1); 
    } else {
	debug(F110,"cwd failed",dirp,0);
	tlog(F110,"Failed to change directory to",dirp,0L);
	return(0);
    }
}


/*  S Y S C M D  --  Do a system command  */

/*  Command string is formed by concatenating the two arguments.  */

int
syscmd(prefix,suffix) char *prefix, *suffix; {
    extern int i_isopen;
#ifndef NOPUSH
    char *cp;

    i_isopen = 0;
    if (!prefix)
      return(0);
    if (!*prefix)
      return(0);
    for (cp = cmdstr; *prefix != '\0'; *cp++ = *prefix++);
    while (*cp++ = *suffix++)
#ifdef OS2
        /* This takes away more than we gain in convenience 
        if (*(cp-1) == '/') *(cp-1) = '\\' */
#endif /* OS2 */
      ;					/* Copy suffix */

    debug(F110,"syscmd",cmdstr,0);

    if (zxcmd(ZIFILE,cmdstr) > 0) {
    	debug(F110,"syscmd zxcmd ok",cmdstr,0);
	nfils = sndsrc = 0;		/* Flag that input is from stdin */
    	xflg = hcflg = 1;		/* And special flags for pipe */
	binary = XYFT_T;		/* Go to text mode */
	i_isopen = 1;
    	return (sinit());		/* Send S packet */
    } else {
    	debug(F100,"syscmd zxcmd failed",cmdstr,0);
	i_isopen = 0;
    	return(0);
    }
#else
    debug(F100,"syscmd zxcmd NOPUSH",cmdstr,0);
    i_isopen = 0;
    return(0);
#endif /* NOPUSH */
}

/*  R E M S E T  --  Remote Set  */
/*  Called by server to set variables as commanded in REMOTE SET packets.  */
/*  Returns 1 on success, 0 on failure.  */

int
remset(s) char *s; {
    extern int c_save;
    int len, i, x, y;
    char *p;

    len = xunchar(*s++);		/* Length of first field */
    p = s + len;			/* Pointer to second length field */
    *p++ = '\0';			/* Zero out second length field */
    x = atoi(s);			/* Value of first field */
    debug(F111,"remset",s,x);
    debug(F110,"remset",p,0);
    switch (x) {			/* Do the right thing */
      case 132:				/* Attributes (all, in) */
	atcapr = atoi(p);
	return(1);
      case 133:				/* File length attributes */
      case 233:				/* IN/OUT combined */
      case 148:				/* Both kinds of lengths */
      case 248:
	atleni = atleno = atoi(p);
	return(1);
      case 134:				/* File Type (text/binary) */
      case 234:
	attypi = attypo = atoi(p);
	return(1);
      case 135:				/* File creation date */
      case 235:
	atdati = atdato = atoi(p);
	return(1);
      case 139:				/* File Blocksize */
      case 239:
	atblki = atblko = atoi(p);
	return(1);
      case 141:				/* Encoding / Character Set */
      case 241:
	atenci = atenco = atoi(p);
	return(1);
      case 142:				/* Disposition */
      case 242:
	atdisi = atdiso = atoi(p);
	return(1);
      case 145:				/* System ID */
      case 245:
	atsidi = atsido = atoi(p);
	return(1);
      case 147:				/* System-Dependent Info */
      case 247:
	atsysi = atsyso = atoi(p);
	return(1);
      case 232:				/* Attributes (all, out) */
	atcapr = atoi(p);
	return(1);
      case 300:				/* File type (text, binary) */
	binary = atoi(p);
	b_save = binary;
#ifndef NOICP
	g_binary = -1;
#endif /* NOICP */
	return(1);
      case 301:				/* File name conversion */
	fncnv = 1 - atoi(p);		/* (oops) */
	f_save = fncnv;
#ifndef NOICP
	g_fncnv = -1;
#endif /* NOICP */
	return(1);
      case 302:				/* File name collision */
	x = atoi(p);
	if (x == XYFX_R) ckwarn = 1;	/* Rename */
	if (x == XYFX_X) ckwarn = 0;	/* Replace */
	fncact = x;
	return(1);
      case 310:				/* Incomplete File Disposition */
	keep = atoi(p);			/* Keep, Discard */
	return(1);
      case 311:				/* Blocksize */
	fblksiz = atoi(p);
	return(1);
      case 312:				/* Record Length */
	frecl = atoi(p);
	return(1);
      case 313:				/* Record format */
	frecfm = atoi(p);
	return(1);
      case 314:				/* File organization */
	forg = atoi(p);
	return(1);
      case 315:				/* File carriage control */
	fcctrl = atoi(p);
	return(1);
      case 400:				/* Block check */
	y = atoi(p);
	if (y < 5 && y > 0) {
	    bctr = y;
	    c_save = -1;
	    return(1);
	} else if (*p == 'B') {
	    bctr = 4;
	    c_save = -1;
	    return(1);
	}
	return(0);
      case 401:				/* Receive packet-length */
	rpsiz = urpsiz = atoi(p);
	if (urpsiz > MAXRP) urpsiz = MAXRP; /* Max long-packet length */
	if (rpsiz > 94) rpsiz = 94;	    /* Max short-packet length */
	urpsiz = adjpkl(urpsiz,wslots,bigrbsiz);
	return(1);
      case 402:				/* Receive timeout */
	y = atoi(p);			/* Client is telling us */
	if (y > -1 && y < 999) {	/* the timeout that it wants */
	    pkttim = chktimo(y,timef);	/* us to tell it to use. */
	    return(1);
	} else return(0);
      case 403:				/* Retry limit */
	y = atoi(p);
	if (y > -1 && y < 95) {
	    maxtry = y;
	    return(1);
	} else return(0);
      case 404:				/* Server timeout */
	y = atoi(p);
	if (y < 0) return(0);
	srvtim = y;
	return(1);

#ifndef NOCSETS
      case 405:				/* Transfer character set */
	for (i = 0; i < ntcsets; i++) { 
	    if (!strcmp(tcsinfo[i].designator,p)) break;
	}
	debug(F101,"remset xfer charset lookup","",i);
	if (i == ntcsets) return(0);
	tcharset = tcsinfo[i].code;	/* if known, use it */
	if (tcharset == TC_TRANSP)
	  rx = NULL;
	else
	  rx = xlr[tcharset][fcharset];	/* translation function */
	return(1);
#endif /* NOCSETS */

      case 406:				/* Window slots */
	y = atoi(p);
	if (y == 0) y = 1;
	if (y < 1 || y > MAXWS) return(0);
	wslotr = y;
	swcapr = 1;
	urpsiz = adjpkl(urpsiz,wslotr,bigrbsiz);
	return(1);

      case 410:				/* Transfer mode */
	y = atoi(p);			/* 0 = automatic, nonzero = manual */
	if (y != 0) y = 1;
	xfermode = y;
	return(1);

      default:				/* Anything else... */
	return(0);
    }
}

/* Adjust packet length based on number of window slots and buffer size */

int
adjpkl(pktlen,slots,bufsiz) int pktlen, slots, bufsiz; {
    if (protocol != PROTO_K) return(pktlen);
    debug(F101,"adjpkl len","",pktlen);
    debug(F101,"adjpkl slots","",slots);
    debug(F101,"adjpkl bufsiz","",bufsiz);
    if (((pktlen + 6) * slots) > bufsiz)
      pktlen = (bufsiz / slots) - 6;
    debug(F101,"adjpkl new len","",pktlen);
    return(pktlen);
}

/* Set transfer mode and file naming based on comparison of system types */

VOID
whoarewe() {
    extern int xfermode;
    extern char whoareu[];
    int flag = 0;
    debug(F101,"whoarewe xfermode","",xfermode);
    if (xfermode == XMODE_A) {		/* If TRANSFER MODE AUTOMATIC */
	if (whoareu[0]) {		/* and we know partner's system type */
	    char * p = (char *)whoareu;
	    debug(F110,"whoarewe remote sysid",whoareu,0);
	    if (!strcmp(p,cksysid))	/* Other system same as us */
	      flag = 1;
#ifdef UNIX
	    if (!strcmp(p,"L3"))	/* UNIX is sort of like AmigaDOS */
	      flag = 1;			/* (same directory separator) */
	    else if (!strcmp(p,"N3"))	/* UNIX like Aegis */
	      flag = 1;
#else
#ifdef AMIGA
/* Like UNIX, but case distinctions are ignored and can begin with device:. */
	    else if (!strcmp(p,"U1"))	/* Amiga is sort of like UNIX */
	      flag = 1;
	    else if (!strcmp(p,"N3"))	/* Amiga is sort of like Aegis */
	      flag = 1;
#else
#ifdef OS2				/* (Includes Windows 95/NT) */

	    /* DOS, GEMDOS, Windows 3.x, Windows 95, Windows NT */
	    /* All "the same" for FAT partitions but all bets off otherwise */
	    /* so this part needs some refinement ...  */

	    else if (!strcmp(p,"U8"))	/* MS-DOS */
	      flag = 1;
	    else if (!strcmp(p,"UO"))	/* OS/2 */
	      flag = 1;
	    else if (!strcmp(p,"UN"))	/* Windows NT or 95 */
	      flag = 1;
	    else if (!strcmp(p,"K2"))	/* GEMDOS */
	      flag = 1;
#else
#ifdef GEMDOS
	    else if (!strcmp(p,"U8"))
	      flag = 1;
	    else if (!strcmp(p,"UO"))
	      flag = 1;
	    else if (!strcmp(p,"UN"))
	      flag = 1;
	    else if (!strcmp(p,"K2"))
	      flag = 1;
#endif /* GEMDOS */
#endif /* OS2 */
#endif /* AMIGA */
#endif /* UNIX */

	    debug(F101,"whoarewe flag","",flag);
	    if (flag) {			/* We have a match */
		fncnv = XYFN_L;		/* so literal filenames */
#ifdef VMS
		binary = XYFT_L;	/* For VMS-to-VMS, use labeled */
#else
#ifdef OS2
		if (!strcmp(cksysid,"UO") && !strcmp((char *)whoareu,"UO"))
		  binary = XYFT_L;	/* For OS/2-to-OS/2, use labeled */
#else
		binary = XYFT_B;	/* For all others use binary */
#endif /* OS2 */
#endif /* VMS */
		gnf_binary = binary;	/* Prevailing type for gnfile() */
	    }
	}
    }
}
