/*
 * IceWM
 *
 * Copyright (C) 1997,1998 Marko Macek
 */

#include "icewm.h"

#ifdef IMLIB
#include <Imlib.h>

static ImlibData *hImlib = 0;
#endif

YPixmap::YPixmap(const char *fileName) {
#ifdef IMLIB
    if(!hImlib) hImlib=Imlib_init(app->display());

    fOwned = true;
    
    ImlibImage *im = Imlib_load_image(hImlib, (char *)REDIR_ROOT(fileName));
    if(im) {
        fWidth = im->rgb_width;
	fHeight = im->rgb_height;
	Imlib_render(hImlib, im, fWidth, fHeight);
	fPixmap = (Pixmap)Imlib_move_image(hImlib, im);
	fMask = (Pixmap)Imlib_move_mask(hImlib, im);
	Imlib_destroy_image(hImlib, im);
    } else {
        fprintf(stderr, "Warning: loading image %s failed\n", fileName);
	fPixmap = fMask = None;
        fWidth = fHeight = 16;
    }
#else
    XpmAttributes xpmAttributes;
    int rc;

    xpmAttributes.colormap  = defaultColormap;
    xpmAttributes.closeness = 65535;
    xpmAttributes.valuemask = XpmSize|XpmReturnPixels|XpmColormap|XpmCloseness;

    rc = XpmReadFileToPixmap(app->display(),
                             desktop->handle(),
                             (char *)REDIR_ROOT(fileName),
                             &fPixmap, &fMask,
                             &xpmAttributes);

    fOwned = true;
    if (rc == 0) {
       fWidth = xpmAttributes.width;
       fHeight = xpmAttributes.height;
    } else { 
       fWidth = fHeight = 16; /// should be 0, fix
       fPixmap = fMask = None;
    }

    if (rc != 0)
        fprintf(stderr, "warning: load pixmap %s failed with rc=%d\n", fileName, rc);
#endif
}

#ifdef IMLIB
/* Load pixmap at specified size */
YPixmap::YPixmap(const char *fileName, int w, int h) {

    if(!hImlib) hImlib = Imlib_init(app->display());

    fOwned = true;
    fWidth = w;
    fHeight = h;
    
    ImlibImage *im = Imlib_load_image(hImlib, (char *)REDIR_ROOT(fileName));
    if(im) {
	Imlib_render(hImlib, im, fWidth, fHeight);
	fPixmap = (Pixmap) Imlib_move_image(hImlib, im);
	fMask = (Pixmap) Imlib_move_mask(hImlib, im);
	Imlib_destroy_image(hImlib, im);
    } else {
        fprintf(stderr, "Warning: loading image %s failed\n", fileName);
	fPixmap = fMask = None;
    }
}
#endif

YPixmap::YPixmap(Pixmap pixmap, Pixmap mask, int w, int h) {
    fOwned = false;
    fWidth = w;
    fHeight = h;
    fPixmap = pixmap;
    fMask = mask;
}

YPixmap::~YPixmap() {
    if (fOwned) {
        if (fPixmap != None)
            XFreePixmap(app->display(), fPixmap);
        if (fMask != None)
            XFreePixmap(app->display(), fMask);
    }
}

YIcon *firstIcon = 0;

YIcon::YIcon(const char *fileName) {
    fNext = firstIcon;
    firstIcon = this;
    loadedS = loadedL = 0;
    
    fLarge = fSmall = 0;
    fPath = new char [strlen(fileName) + 1];
    if (fPath)
        strcpy(fPath, fileName);
}

YIcon::YIcon(int count, int reclen, CARD32 *elem) {
    int i;

    fLarge = fSmall = 0;
    loadedS = 1;
    loadedL = 1;
    fPath = 0;
    fNext = 0;

    if (reclen < 2)
        return ;

    for (i = 0; i < count; i++, elem += reclen) {
        Pixmap pixmap, mask;
        
        pixmap = elem[0];
        mask = elem[1];

        if (pixmap == None)
            continue;

        int x, y;
        Window root;
        unsigned int w = 0, h = 0, border, depth = 0;

        if (reclen >= 6) {
            w = elem[2];
            h = elem[3];
            depth = elem[4];
            root = elem[5];
        } else {
            if (XGetGeometry(app->display(), pixmap, 
                             &root, &x, &y, &w, &h, &border, &depth) == BadDrawable)
                continue;
        }
        
        if (w != h || w == 0 || h == 0)
            continue;
        
        if (depth == (unsigned)DefaultDepth(app->display(),
                                            DefaultScreen(app->display())))
        {
	    //printf("%d %d\n", w, h);
            if (w == ICON_SMALL)
                fSmall = new YPixmap(pixmap, mask, w, h);
            else if (w == ICON_LARGE)
                fLarge = new YPixmap(pixmap, mask, w, h);
        }
    }
}

YIcon::~YIcon() {
    delete fLarge;
    delete fSmall;
    delete fPath;
}

int YIcon::findIcon(char **fullPath, int size) {
    char icons_size[1024];
    char path[1024];

    sprintf(icons_size, "%s_%dx%d.xpm", REDIR_ROOT(fPath), size, size);

    /// !!! fix: do this at startup (merge w/ iconPath)
    pathelem *pe = icon_paths;
    for (; pe->root; pe++) {
        joinPath(path, pe, "/icons/", "");

        int l = strlen(path);

        strcpy(path + l, icons_size);
        if (is_reg(path)) {

            path[l] = 0;
            if (!findPath(path, R_OK, icons_size, fullPath)) {
                fprintf(stderr, "icon '%s' for '%s' not found.\n", icons_size, fPath);
                return 0;
            }
            return 1;
        }
    }

    if (!findPath(iconPath, R_OK, icons_size, fullPath)) {
        fprintf(stderr, "icon '%s' for '%s' not found.\n", icons_size, fPath);
        return 0;
    }
    return 1;
}

YPixmap *YIcon::loadIcon(int size) {
    YPixmap *icon = 0;
 
    if (icon == 0) {
#ifdef IMLIB
        if(is_reg(fPath)) {
            icon = new YPixmap(fPath, size, size);
            if (icon == 0)
                fprintf(stderr, "out of memory for pixmap %s", fPath);
        } else
#endif
        {
            char *fullPath;

            if (findIcon(&fullPath, size)) {
                icon = new YPixmap(fullPath);
                if (icon == 0)
                    fprintf(stderr, "out of memory for pixmap %s", fullPath);
                FREE(fullPath);
            }
        }
    }
    return icon;
}

YPixmap *YIcon::large() {
    if (fLarge == 0 && !loadedL)
        fLarge = loadIcon(ICON_LARGE);
    loadedL = 1;
    return fLarge;
}

YPixmap *YIcon::small() {
    if (fSmall == 0 && !loadedS)
        fSmall = loadIcon(ICON_SMALL);
    loadedS = 1;
    //return large(); // for testing menus...
    return fSmall;
}

YIcon *getIcon(const char *name) {
    YIcon *icn = firstIcon;

    while (icn) {
        if (strcmp(name, icn->iconName()) == 0)
            return icn;
        icn = icn->next();
    }
    return new YIcon(name);
}

void freeIcons() {
    YIcon *icn, *next;

    for (icn = firstIcon; icn; icn = next) {
        next = icn->next();
        delete icn;
    }
}
