/*
   forms.c - forms handling for html.

   ParseHTML() creates a linked list of forms. Each form has a linked
   list of fields. This file includes routines for drawing fields,
   handling mouse clicks/drags and key presses on fields, plus sending
   the contents of forms to HTTP servers.

   When resizing windows, it is important to keep the existing form
   data structures as otherwise the current values of fields will be
   lost and overridden by the original starting values. This requires
   the global new_form to be set to false. It should be set to true
   when reading new documents. In addition, The browser history list
   needs to preserve the form data structures so that they can be
   reused upon backtracking to restore previous field values.

   The desire to evolve to a wysiwyg editor means that its a bad idea
   to have pointers into the paint stream as this would be expensive
   to update when users edit the html document source.
   Consequently, one has to search the form structure to find fields
   that need to be redrawn under programatic control,
   e.g. unsetting of radio buttons. Note that anyway, the y-position of
   a field is unresolved when it is added to the paint stream,
   and only becomes fixed when the end of the line is reached.
   A compromise is to cache the baseline position when painting each
   field and trash the cache each time the user alters the document.

   Another complication is the need to save current field values
   when following a link to another document, as otherwise there
   is now way of restoring them when the user backtracks to the
   document containing the form. A simple approach is to save the
   linked list of forms in memory as part of a memory resident
   history list. Note for local links, the same form data should
   be used, rather than recreating the list from new. The corollary
   is that the list should only be freed when backtracking the
   document containing the form or reloading the same.

   Note that in many cases a sequence of form interactions with
   a server causes a state change in the server (e.g. updating a
   database) and hence is not simply reversed by backtracking in
   the normal way. In this case it makes sense to avoid pushing
   intermediate steps onto the history stack, e.g. when submitting
   the form. This probably requires the document or server to
   disable the history nechanism via suitable attributes.
*/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include "arena.h"
#include "bridge.h"
#include "colour.h"
#ifdef ARENA_DEBUG	/* QingLong.02-02-97 */
#  include "debug.h"
#endif
#include "display.h"
#include "forms.h"
#include "forms_i.h"
#include "html.h"
#include "mailto.h"
#include "main.h"
#include "neweditor.h"
#include "parsehtml.h"
#include "status.h"
#include "types.h"
#include "util.h"
#include "x11.h"


Form* forms = NULL;
Field* focus;   /* which field has focus (NULL if none do) */
int cursorpos = 5; /* cursor position in characters */
Bool TextFieldFocus = False;

/*
   The next 3 pointers indicate records to reuse for next form/field/option
   when resizing a document or during 2nd pass thru table data. If NULL the
   corresponding record is created from scratch
*/

Form* next_form = NULL;
Field* next_field = NULL;
Option* next_option = NULL;


/*
 * version of strdup that copes with null string pointers and dummy strings
 */
#define StrDup(s) ((s) ? ((*s != '\0') ? strdup(s) : NULL) : NULL)


/* Searches the given list for a form of the given access method and
 * which has action string located at the given point in the memory.
 */
Form* FindFormInTheList(Form* theForms,
			FormSubmitMethod theMethod, char* theAction)
{
 Form* form_ix;

 if (theForms)
   {
    for (form_ix = theForms; form_ix != NULL; form_ix = form_ix->next)
      {
       if (form_ix->method == theMethod)
	 if (form_ix->action == theAction) return form_ix;
      }
   }

 return NULL; /* failed */
}


/* Searches the current list for a form of the given access method and
 * which has action string located at the given point in the memory.
 */
Form* FindForm(FormSubmitMethod theMethod, char* theAction)
{
 extern Form* forms;

 return FindFormInTheList(forms, theMethod, theAction);
}


#if 0
/* Searches the curent list for a form of the given access method and
 * which has an action string equal (as a string) to the given one.
 */
Form* FindFormFuzzy(FormSubmitMethod theMethod, char* theAction)
{
 extern Form* forms;
 Form* form_ix;
 int theActionL = Arena_StrLen(theAction);

 if (forms)
   {
    for (form_ix = forms; form_ix != NULL; form_ix = form_ix->next)
      {
       if (form_ix->method == theMethod)
	 if (strncmp(form_ix->action, theAction, theActionL) == 0)
	   return form_ix;
      }
   }

 return NULL; /* failed */
}
#endif


/* Searches the current list for a form which has
 * action string located at the given point in the memory.
 */
Form* FindFormByAction(char* theAction)
{
 extern Form* forms;
 Form* form_ix;

 if (forms)
   for (form_ix = forms; form_ix != NULL; form_ix = form_ix->next)
     if (form_ix->action == theAction) return form_ix;

 return NULL; /* failed */
}


/*
 * Find a field of given type and name in the given form,
 * name comparison being made as strings for fields of all types,
 * but RADIOBUTTON and CHECKBOX for which comparison is strict
 * (cause they are allowed to be many with the same name).
 * Strict coincidence means coincidence of pointers, rather than strings.
 */
Field* FindField_in_List(Field* fieldList,
			 int type, char *name, int namelen)
{
 Field* field_link;

 for (field_link = fieldList;
      field_link != NULL ;
      field_link = field_link->next)
   {    /* check type then name... assume it's enough */
    if (field_link->type == type)
      {
       if ((type == RADIOBUTTON) || (type == CHECKBOX))
	 {
	  if (name == field_link->name) return(field_link);
	 }
        else
	 if (strncasecmp(name, field_link->name, namelen) == 0)
	   return(field_link);
      }
   }

 return NULL;
}


/*
 * Find a field of given type and name in the given form,
 * name comparison being made as strings for fields of all types,
 * but RADIOBUTTON and CHECKBOX for which comparison is strict
 * (cause they are allowed to be many with the same name).
 * Strict matching means coincidence of pointers, rather than strings.
 */
Field* FindField(Form* form,
		 int type, char *name, int namelen)
{
 if (form)
   if (form->fields)
     return FindField_in_List(form->fields, type, name, namelen);

 return NULL;
}


/* Find a field of given type and name in the given form,
 * name comparison being made as strings for fields of ALL types.
 * Be aware, for CHECKBOXes and RADIOBUTTONs this means,
 * that only the first field with such name is found.
 * Search 
 */
Field* FindFieldFuzzy_in_List(Field* fieldList,
			      int type, char *name, int namelen)
{
 Field* field_link;

 for (field_link = fieldList;
      field_link != NULL ;
      field_link = field_link->next)
   {    /* check type then name... assume it's enough */
    if (field_link->type == type)
      if (strncasecmp(name, field_link->name, namelen) == 0)
	return(field_link);
   }

 return NULL;
}


/* Find a field of given type and name in the given form,
 * name comparison being made as strings for fields of ALL types.
 * Be aware, for CHECKBOXes and RADIOBUTTONs this means,
 * that only the first field with such name is found.
 * Search 
 */
Field* FindFieldFuzzy(Form* form,
		      int type, char *name, int namelen)
{
 if (form)
   if (form->fields)
     return FindFieldFuzzy_in_List(form->fields, type, name, namelen);

 return NULL;
}


int FieldCount(Form* form)
{
 int n;
 Field* field;

 n = 0;
 field = form->fields;

 while (field)
   {
    ++n;
    field = field->next;
   }

 return n;
}


void FreeForms(void)
{
 Form* form;
 Field *fields, *field;
 Option* option;

 next_form = NULL;
 next_field = NULL;
 next_option = NULL;

 while (forms != NULL)
   {
    form = forms;
    forms = forms->next;
    fields = form->fields;

    while (fields)
      {
       field = fields;
       /* Free(field->name); --Spif name is only a copy of pointer 13-Oct-95 */
       Free(field->value);
       /* --Spif we must clear image */
       if (field->image != NULL)
	 {
	  Free( field->image->pixels );
	  Free( field->image->url );
	  Free( field->image );
	 };

       fields = fields->next;

       while ((option = field->options))
	 {
	  Free(option->label);
	  Free(option->value);

	  field->options = option->next;
	  Free(option);
	 }

       Free(field);
      }

    if (form->action) Free(form->action);

    Free(form);
   }
}


/* scroll field up/down by delta - this is called
 * to adjust y offset of field - not to paint fields
 */
void ScrollForms(long delta)
{
 Form* form;
 Field* field;

 for (form = forms; form; form = form->next)
   {
    for (field = form->fields; field; field = field->next)
      if (field->baseline >= 0 && field->baseline - delta >= 0)
	field->baseline -= delta;
   }
}


/*
 * Preserves form contents across window resizing/backtracking.
 * It is assumed that FreeForms() is called before
 * loading new documents or reloading existing ones.
 */
void ResizeForm(void)
{
 if (forms != NULL)
   {
    next_form = forms;
    next_field = next_form->fields;
    next_option = next_field->options;
   }
}


/*
 * ParseTable() makes 2 passes thru table data,
 * so we need to avoid recreating elements on 2nd pass.
 * This procedure should be called before pass == 1 and pass == 2
 */
void ProcessTableForm(int pass)
{
 static Form* form;
 static Field* field;
 static Option* option;

 if (pass == 1)
   {
    form = next_form;
    field = next_field;
    option = next_option;
   }
  else  /* pass == 2 */
   {
    next_form = form;
    next_field = field;
    next_option = option;
   }
}


Form* GetForm(FormSubmitMethod method, char *url, int len)
{
 Form *form, *last;

 /* is form already defined */

 if (next_form)
   {
    form = next_form;
    next_form = form->next;
    next_field = form->fields;
    next_option = next_field->options;
    return form;
   }

 form = (Form *)Arena_MAlloc(sizeof(Form), False);

 form->next = NULL;
 form->fields = NULL;
 form->action = url; /* strndup(url, len); --Spif */
 form->alen = len;
 form->method = method;
 next_field = NULL;
 next_option = NULL;

 /* insert at end of list */

 if (forms)
   {
    last = forms;
    while (last->next) last = last->next;
    last->next = form;
   }
  else
   forms = form;

 return form;
}


Form* DefaultForm(void)
{
 return GetForm(GET, CurrentDoc->url, Arena_StrLen(CurrentDoc->url));
}


/*
 * Reset the form.
 */
Bool ResetForm(GC theGC, ViewWindow theViewWindow, Form* theForm)
{
 Bool TotalSuccess = True;


 if (theForm == NULL) return False;

 PaintForm(theGC, theViewWindow, theForm);

 return TotalSuccess;
}


/*
 * QingLong.05-02-97: convert all `entities' (e.g. "&quot;") in the string.
 */
char* UnEntitify(char* theString)
{
 char* s = NULL;
 int i, j, k, len, ok;

 len = Arena_StrLen(theString);
 s = (char *)Arena_CAlloc(len + 1, sizeof(char), False);

 for (i = k = 0; i < len; i++, k++)
   if (theString[i] == '&')
     {
      for (ok = False, j = i; !ok && j < len; j++)
	ok = ((theString[j] == ';') || (theString[j] == ' '));

      s[k] = entity(theString + i + 1, &j);
      i += j - 1;
     }
    else
     s[k] = theString[i];

 return(s);
}


/*
 * Give up focus from the given field. Don't bother on where `focus' is. Paint.
 */
Bool _Arena_GiveUpFocus(GC theGC, ViewWindow theViewWindow,
			Field** theField_pp, EditorBuffer** theEditorBuffer_pp)
{
 Bool TotalSuccess = True;
#ifdef ARENA_DEBUG
 char Iam[] = "_Arena_GiveUpFocus";
#endif


 if (theField_pp == NULL) return False;
 if ((*theField_pp) == NULL) return False;

 TextFieldFocus = False;

 switch ((*theField_pp)->type)
   {
    case TEXTAREA:
    case TEXTFIELD:
    case PASSWD:
      (*theField_pp)->flags &= ~CHECKED;
      if (theEditorBuffer_pp)
	{
	 if ((*theEditorBuffer_pp))
	   {
	    Free((*theField_pp)->value);
	    (*theField_pp)->value = Buffer2Str((*theEditorBuffer_pp));
	   }

	 FreeBuffer((*theEditorBuffer_pp));
	 (*theEditorBuffer_pp) = NULL;
	}
      break;

    case OPTIONLIST:
      if ((*theField_pp)->flags & CHECKED)
	{
	 (*theField_pp)->flags &= ~CHECKED;
	 HideDropDown(theGC, theViewWindow, CurrentDoc, -1, (*theField_pp));
	}
      break;

    case CHECKBOX:
      break;

    case RADIOBUTTON:
      break;

    case RESETBUTTON:
      (*theField_pp)->flags &= ~CHECKED;
      break;

    case SUBMITBUTTON:
      (*theField_pp)->flags &= ~CHECKED;
      break;

    case HIDDEN:
#ifdef ARENA_DEBUG
      if (FORM_TRACE)
	Arena_TracePrint(Iam,
			 " ERROR!\n"
			 "\tTrying to remove focus from the HIDDEN field.\n",
			 (*theField_pp)->type);
#endif
      TotalSuccess = False;
      break;

    default:
#ifdef ARENA_DEBUG
      if (FORM_TRACE)
	Arena_TracePrint(Iam,
			 " ERROR! Unexpected the field type %d.\n",
			 (*theField_pp)->type);
#endif
      TotalSuccess = False;
      break;
   }
 /* End ``switch ((*theField_pp)->type)'' */

 if (TotalSuccess)
   {
    Field* theField = (*theField_pp);

    (*theField_pp) = NULL; /* Remove focus! */
    PaintField(theGC, theViewWindow, -1, theField->baseline, theField);
   }
  else
   (*theField_pp) = NULL; /* Remove focus! */

 return TotalSuccess;
}


/*
 * Give up focus from the given field. Do care about `focus' variable.
 */
Bool GiveUpFocus(GC theGC, ViewWindow theViewWindow,
		 Field** theField_pp, EditorBuffer** theEditorBuffer_pp)
{
 extern Field* focus;
 Bool TotalSuccess = True;
#ifdef ARENA_DEBUG
 char Iam[] = "GiveUpFocus";
#endif


 if (focus)
   {
    Bool defocus;
    EditorBuffer** focusEditorBuffer_pp = NULL;

    if (theField_pp)
      {
       if (*theField_pp)
	 {
	  if (focus == (*theField_pp))
	    {
	     defocus = False;
	     focus = NULL;
	    }
	   else
	    {
	     TotalSuccess = False;
#ifdef ARENA_DEBUG
	     if (FORM_TRACE)
	       Arena_TracePrint(Iam,
				" ERROR!\n"
				"\tRemoving focus from unfocused field!\n");
#endif
	    defocus = True;
	    focusEditorBuffer_pp = NULL;
	    }
	 }
        else
	 {
	  defocus = True;
	  focusEditorBuffer_pp = theEditorBuffer_pp;
	 }
      }
     else
      {
       defocus = True;
       focusEditorBuffer_pp = theEditorBuffer_pp;
      }

    if (defocus)
      {
       Bool defocus_succes;

       defocus_succes = _Arena_GiveUpFocus(theGC, theViewWindow,
					   &focus, focusEditorBuffer_pp);
       TotalSuccess = (TotalSuccess && defocus_succes);
      }
   }
 /* End ``if (focus)'' */

 {
  Bool defocus_succes;

  defocus_succes = _Arena_GiveUpFocus(theGC, theViewWindow,
				      theField_pp, theEditorBuffer_pp);
  TotalSuccess = (TotalSuccess && defocus_succes);
 }

 return TotalSuccess;
}


/*
 * Finish editing current form field --- remove focus from it.
 */
Bool FinishCurrentFieldEditing(GC theGC, ViewWindow theViewWindow)
{
 extern Doc* CurrentDoc;

 if (CurrentDoc)
   if (CurrentDoc->edited_field)
     if (GiveUpFocus(theGC, theViewWindow,
		     &(CurrentDoc->edited_field), &(CurrentDoc->field_editor)))
       return True;

 return False;
}


/*
 * Focus on the given field (start editing).
 */
Bool FocusOn(GC theGC, ViewWindow theViewWindow,
	     int indent, int baseline, Field* theField)
{
 extern Field* focus;
 Bool TotalSuccess = True;
#ifdef ARENA_DEBUG
 char Iam[] = "FocusOn";
#endif


 if (theField == NULL) return False;
 if (CurrentDoc == NULL) return False;

#ifdef ARENA_DEBUG
 if ((theField == focus) && (theField == CurrentDoc->edited_field))
   if (FORM_TRACE)
     Arena_TracePrint(Iam, " Hmmm, ERROR? Focus already there.\n");
#endif

 switch (theField->type)
   {
    case TEXTAREA:
    case TEXTFIELD:
    case PASSWD:
      CurrentDoc->edited_field = focus = theField;
      CurrentDoc->field_editor = Str2Buffer(theField->value);
      CurrentDoc->field_editor->pos = theField->pos;
      PaintTextField(theGC, theViewWindow, indent, baseline, True, theField);
      break;

    case OPTIONLIST:
      CurrentDoc->edited_field = focus = theField;
      PaintSelectField(theGC, theViewWindow, indent, baseline, True, theField);
      break;

    case CHECKBOX:
      CurrentDoc->edited_field = focus = theField;
      PaintCheckboxField(theGC, theViewWindow, indent, baseline, True, theField);
      break;

    case RADIOBUTTON:
      CurrentDoc->edited_field = focus = theField;
      PaintRadioButtonGroup(theGC, theViewWindow, theField);
      break;

    case RESETBUTTON:
    case SUBMITBUTTON:
      CurrentDoc->edited_field = focus = theField;
      PaintButtonField(theGC, theViewWindow, indent, baseline, True, theField);
      break;

    case HIDDEN:
#ifdef ARENA_DEBUG
      if (FORM_TRACE)
	Arena_TracePrint(Iam,
			 " ERROR! Trying to focus on HIDDEN field.\n",
			 theField->type);
#endif
      TotalSuccess = False;
      break;

    default:
#ifdef ARENA_DEBUG
      if (FORM_TRACE)
	Arena_TracePrint(Iam,
			 " ERROR! Unexpected the field type %d.\n",
			 theField->type);
#endif
      TotalSuccess = False;
      break;
   }
 /* End ``switch (theField->type)'' */

 return TotalSuccess;
}


/*
 * Move focus from the current field to the given one.
 */
Bool MoveFocusTo(GC theGC, ViewWindow theViewWindow,
		 Doc* theDoc, int indent, int baseline, Field *theField)
{
#ifdef ARENA_DEBUG
 char Iam[] = "MoveFocusTo";
#endif

 if (theDoc)
   {
    if (theField)
      {
       if (theField == theDoc->edited_field)	/* Already there! */
	 {
#ifdef ARENA_DEBUG
	  if (FORM_TRACE)
	    Arena_TracePrint(Iam, " Hmmm, ERROR? Focus already there.\n" );
#endif
	  return True;
	 }

       if (theDoc->edited_field)	/* remove focus from current field */
	 GiveUpFocus(theGC, theViewWindow,
		     &(theDoc->edited_field), &(theDoc->field_editor));

       /* Shift page vertical position properly. */
       MoveToYpage(theDoc, baseline - ASCENT(font) - 2);

       /* Focus on the given field */
       return FocusOn(theGC, theViewWindow, indent, baseline, theField);
      }
     else
      {
#ifdef ARENA_DEBUG
       if (FORM_TRACE)
	 Arena_TracePrint(Iam, " ERROR! The field to focus is NULL.\n");
#endif
       return False;
      }
   }
  else
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The doc given is NULL.\n");
#endif
    return False;
   }
}


/*
 * font should be passed in lower 4 bits of flags.
 */
Field* GetField(Form* form, int type, int x, char *name, int nlen,
		char *value, int vlen, int pos, int rows, int cols,
		int maxlength, int flags)
{
 int em, font;
 char *s;
 Field *field, *last;

 if (form == NULL) return NULL;

 /* if we are resizing the form or backtracking to it then
    we need to reuse the existing data structures to preserve
    any changes that have been made to the default values */

 if ((field = FindField(form, type, name, nlen)) == NULL)
   {
    if (next_field)
      {
       field = next_field;
       next_field = field->next;
       next_option = field->options;
       goto set_geometry;
      }

    field = (Field *)Arena_MAlloc(sizeof(Field), False);
    next_option = NULL;

    field->next = NULL;
    field->form = form;
    field->type = type;
    field->name = name;
    field->nlen = nlen;

    if ((value != NULL) && (vlen > 0))
      {	/* QingLong.29-03-97 */
       char* s = strndup(value, vlen);

       field->value = UnEntitify(s);  /* UnEntitify calloc's it's own string */
                                      /* Possibly the length has changed !!! */
       field->buflen = vlen = Arena_StrLen(field->value);
       field->bufsize = vlen + 1;
       Free(s);
      }
    else
      {
       field->value = NULL;
       field->bufsize = field->buflen = vlen = 0;
      }

    field->pos = pos;
    field->maxlength = maxlength;
    field->flags = flags;
    field->x_indent = 0;
    field->y_indent = 0;
    field->baseline = -1;
    field->options = 0;

    /* insert at end of list */

    if (form->fields)
      {
       last = form->fields;

       while (last->next) last = last->next;

       last->next = field;
      }
     else
      form->fields = field;

    if (type == TEXTFIELD || type == TEXTAREA)
      {
       field->text_field = (TextField *)Arena_MAlloc(sizeof(TextField), False);
       field->text_field->buffer = NULL;
       field->text_field->length = 0;
       field->text_field->used = 0;
       field->text_field->scratch[0] = '\0';
      }

   set_geometry:

    field->x = x;
    font = flags & 0x0F;

    em = WIDTH(font, "mite", 4)/4;

    if (type == RADIOBUTTON || type == CHECKBOX)
      {
       field->width = ASCENT(font) + DESCENT(font) - 2;
       field->height = field->width;
       field->above = ASCENT(font) - 1;
      }
     else  /* TEXTFIELD and OPTIONLIST */
      {
       if (type == SUBMITBUTTON)
	 {
	  if (vlen == 0)
	    {
	     s = " Submit Query ";
	     vlen = Arena_StrLen(s);
	     field->value = strndup(s, vlen);
	     field->buflen = vlen;
	     field->bufsize = vlen + 1;
	    } 

	  field->width = 8 + WIDTH(font, field->value, field->buflen);
	 }
        else
	 if (type == RESETBUTTON)
	   {
	    if (vlen == 0)
	      {
	       s = " Reset ";
	       vlen = Arena_StrLen(s);
	       field->value = strndup(s, vlen);
	       field->buflen = vlen;
	       field->bufsize = vlen + 1;
	      } 

	    field->width = 8 + WIDTH(font, field->value, field->buflen);
	   }
	  else
	   field->width = 9 + cols * em;

       field->height = 4 + rows * SPACING(Fonts[font]);
       field->above = ASCENT(font) + 3;
      }
   }
  else
   {
   /*
    * As FindField() is used only in order to recall and reuse
    * the previous textual values and settings, and selections,
    * the old geometry (x) is out of date and should be trashed!
    * The abscissa should _always_ be set to the provided value! (QL.11-04-97)
    */
    field->x = x;

    return(field);
   }

 return field;
}


/*
 * howcome 27/7/95: should AddOption return Option?
 */
void AddOption(Field* field, int flags, char *label, int len, char* value)
{
 int width, font;
 Option *option, *i;


 if (field == NULL) return;

 /*
  * if we are resizing the form or backtracking to it
  * then we need to reuse the existing data structures to preserve
  * any changes that have been made to the default values.
  */

 if (next_option)
   {
    option = next_option;
    next_option = option->next;
    return /*option*/;
   }

 if ((option = (Option*)Arena_MAlloc(sizeof(Option), False)))
   {
    option->next = NULL;
    option->flags = flags;
    option->label = StrDup(label);
    option->value = StrDup(value);
    font = flags & 0xF;
    width = 6 + WIDTH(font, label, len) + field->height;
    if (width > field->width) field->width = width;
   }
  else
   return;

 /* insert at end of list */

 if (field->options)
   {
    Option* last;
    Bool AlreadyHaveSuchOption, theLabelisDummy, theOptionLabelisDummy;


    theLabelisDummy = ((len == 0) || ((label ? (*label) : '\0') == '\0'));

    for (last = i = field->options; i; i = (last = i)->next)
      {
       theOptionLabelisDummy = ((i->label ? (*(i->label)) : '\0') == '\0');

       if (theLabelisDummy)
	 {
	  AlreadyHaveSuchOption = theOptionLabelisDummy;
	 }
        else
	 {
	  if (theOptionLabelisDummy)
	    AlreadyHaveSuchOption = False;
	   else
	    {
	     AlreadyHaveSuchOption = False;

	     if (Arena_StrLen(i->label) == len)
	       if (strncmp(i->label, label, len) == 0)
		 AlreadyHaveSuchOption = True;
	    }
	 }

       if (AlreadyHaveSuchOption) /* don't add 2 times the same option */
	 {
	  if (option) /* avoid any possibility of memory leak
			 --SPif 18/Oct/95 */
	    {
	     Free(option->label);
	     Free(option->value);
	     Free(option);
	    };
	  return /*NULL*/;
	 }
      }

    last->next = option;
   }
  else
   {
    field->options = option;
   }
 /* End ``if (field->options)'' */

 if ((!(flags & OPTION_DISABLED)) && (flags & OPTION_SELECTED))
   {
    int vlen;
           /* UnEntitify() calloc's it's own string */
    char* s = UnEntitify(option->label);

    Free(field->value);

    field->value = s;
    field->buflen = vlen = Arena_StrLen(s); /* Possibly the length has changed */
    field->bufsize = vlen + 1;

    if (field->flags & MULTIPLE)
      option->flags |= OPTION_CHECKED;
     else
      {
      /* If there already exists a checked option in the list,
       * don't check this one. The first selected option wins the race.
       */
       for (i = field->options; i; i = i->next)
	 if (i->flags & OPTION_CHECKED) break;

       if (i == NULL) option->flags |= OPTION_CHECKED;
      }
   }

 field->y_indent++;	/* What the heck is this? QingLong.29-03-97 */
}



Field* CurrentRadioButton(Form* form, char *name, int nlen)
{
 Field* field;

 if (form == NULL) return NULL;

 field = form->fields;

 while (field)
   {
    if (strncasecmp(field->name, name, nlen) == 0 && (field->flags & CHECKED))
      return field;

    field = field->next;
   }

 return field;
}


/*
 * Set option list cursor to default position.
 */
int InitiateSelectFieldCursorPosition(Field* theField)
{
 Option* opt;
 int i, n, first_nondisabled, cur_pos = 0;
#ifdef ARENA_DEBUG
 char Iam[] = "InitiateSelectFieldCursorPosition";
#endif

 if (theField == NULL) return False;

 if (theField->type != OPTIONLIST)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not <select>.\n");
#endif
    return False;
   }

 if (theField->options == NULL)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field has empty options list.\n");
#endif
    return False;
   }

 first_nondisabled = 0;
 for (opt = theField->options, i = n = 1; opt; opt = opt->next, i++)
   {
    if (opt->flags & OPTION_DISABLED)
      {
       if (first_nondisabled == 0) n++;
      }
     else
      {
       if (first_nondisabled == 0) first_nondisabled = n;
       if (opt->flags & OPTION_CHECKED)
	 {
	  cur_pos = i;
	  break;
	 }
#ifdef ARENA_DEBUG
        else
	 if (opt->flags & OPTION_SELECTED)
	   if (FORM_TRACE)
	     Arena_TracePrint(Iam, " ERROR! Selected option not checked.\n");
#endif
      }
   }

 return theField->pos = (cur_pos ?
			 cur_pos : (first_nondisabled ?
				    first_nondisabled : 1));
}


/* Called when testing mouse down/up (event = BUTTONDOWN or BUTTONUP).
 * Returns True if really clicked in the given field.
 */
Bool ClickedInField(GC theGC, ViewWindow theViewWindow,
		    int indent, int baseline, Field* theField,
		    int x, int y, int event)
{
 int y1;
 unsigned char theFieldType;
#ifdef ARENA_DEBUG
 char Iam[] = "ClickedInField";
#endif


 if (CurrentDoc == NULL) return False;

 if (theField == NULL)
   {
   /* Finish editing of the current field (the previous one) */
    if (event == BUTTONDOWN)
      GiveUpFocus(theGC, theViewWindow,
		  &(CurrentDoc->edited_field), &(CurrentDoc->field_editor));

    return False;
   }

 theFieldType = theField->type;

 switch (theFieldType)
   {
    case TEXTFIELD:
    case TEXTAREA:
    case PASSWD:
      y1 = -2;
      break;

    case OPTIONLIST:
      y1 = -2;
      break;

    case RADIOBUTTON:
    case CHECKBOX:
      y1 = 2;
      break;

    case SUBMITBUTTON:
    case RESETBUTTON:
      y1 = -2;
      break;

    case HIDDEN:
#ifdef ARENA_DEBUG
      if (FORM_TRACE)
	Arena_TracePrint(Iam, " ERROR! Clicked in HIDDEN field.\n");
#endif
      return False;
      break;

    default:
#ifdef ARENA_DEBUG
      if (FORM_TRACE)
	Arena_TracePrint(Iam, " ERROR! Clicked in unknown type %d field.\n");
#endif
      return False;
      break;
   }

#if 1	/* QingLong.14-10-07 with a hint fron Stephen Zahn. */
 font = theField->flags & 0x0f;
#endif

 y1 += baseline - ASCENT(font);

 /* fix event bug with form fields *//* howcome: applied 18/11/94 */
 x -= indent;

 if ((theField->x <= x) && (x < theField->x + theField->width) &&
              (y1 <= y) && (y < y1 + theField->height))
   {
    if (event == BUTTONDOWN)
      {
      /* Finish editing of the current field (the previous one) */
       if (theField != CurrentDoc->edited_field)
	 GiveUpFocus(theGC, theViewWindow,
		     &(CurrentDoc->edited_field), &(CurrentDoc->field_editor));

       return True;
      }
     else
      if (event == BUTTONUP)
	{
	/* Take appropriate logical action */
	 switch (theFieldType)
	   {
	    case TEXTFIELD:
	    case TEXTAREA:
	    case PASSWD:
	      theField->flags |= CHECKED;
	      break;

	    case OPTIONLIST:
	      theField->flags ^= CHECKED;
	      if ((theField->flags & CHECKED) && (theField->pos <= 0))
		InitiateSelectFieldCursorPosition(theField);
	      break;

	    case RADIOBUTTON:
	      SetRadioButton(theField);
	      break;

	    case CHECKBOX:
	      theField->flags ^= CHECKED;
	      break;

	    case SUBMITBUTTON:
	    case RESETBUTTON:
	      break;

#ifdef ARENA_DEBUG	/* These cases have already been filtered out. But? */
	    case HIDDEN:
	      if (FORM_TRACE)
		Arena_TracePrint(Iam, " ERROR! Pressed in HIDDEN field?\n");
	      return False;
	      break;

	    default:
	      if (FORM_TRACE)
		Arena_TracePrint(Iam,
				" ERROR! Pressed in unknown type %d field.\n");
	      return False;
	      break;
#endif
	   }
	 /* End ``switch (theFieldType)'' */

	 if (theField == CurrentDoc->edited_field)
	   PaintField(theGC, theViewWindow, indent, baseline, theField);
	  else
	   MoveFocusTo(theGC, theViewWindow,
		       CurrentDoc, indent, baseline, theField);

	 return True;
	}
   }

 return False;
}


/*
 * Standard action in text (TEXTFIELD and TEXTAREA) fields repair sequence.
 */
void PaintField_and_FieldCursorBuffer(GC theGC, ViewWindow theViewWindow,
				      Field* theField,
				      EditorBuffer* theEditorBuffer)
{
 if (theField && theEditorBuffer)
   {
    /* First synchronize position field->pos to the editor one */
    theField->pos = theEditorBuffer->pos;

    /* Now paint */
    PaintField(theGC, theViewWindow, -1, theField->baseline, theField);
   }

 return;
}


/*
 * Edit text field (TEXTFIELD, TEXTAREA, PASSWD).
 */
static KeyProcessFeedback EditFormTextField(GC theGC, ViewWindow theViewWindow,
					    Field* theField,
					    EditorBuffer* theEditorBuffer,
					    KeySym theKey,
					    char* theString)
{
 KeyProcessFeedback theKeyProcessed = keyPROCESS_NOACTION;
 unsigned char theFieldType;
 char LineFeed[] = "\n";
#ifdef ARENA_DEBUG
 extern Field* focus;
 char Iam[] = "EditFormTextField";
#endif


 if (theField == NULL) return keyPROCESS_FAILED;
 if (theEditorBuffer == NULL) return keyPROCESS_FAILED;

#ifdef ARENA_DEBUG
 if (focus != theField)
   {
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not in focus.\n");
   }
#endif

 theFieldType = theField->type;

 if (!((theFieldType == TEXTFIELD) ||
       (theFieldType == TEXTAREA) ||
       (theFieldType == PASSWD)))
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field isn't of text type.\n");
#endif
    return keyPROCESS_FAILED;
   }

 switch (theKey)
   {
#if 0
    case XK_Home:
      break;

    case XK_Begin:
      break;

    case XK_End:
      break;

    case XK_Next:
#if (XK_Page_Down != XK_Next)
    case XK_Page_Down:
#endif
      break;

    case XK_Prior:
#if (XK_Page_Up != XK_Prior)
    case: XK_Page_Up:
#endif
    break;

    case XK_Escape:
      break;

    case XK_Pause:
      break;
#endif

    case XK_Left:
      if (theEditorBuffer->pos)
	{
	 theEditorBuffer->pos--;
	 PaintField_and_FieldCursorBuffer(theGC, theViewWindow,
					  theField, theEditorBuffer);
	 theKeyProcessed = keyPROCESS_SUCCESS;
	}
       else
	theKeyProcessed = keyPROCESS_SWALLOW;

      break;

    case XK_Right:
      if (theEditorBuffer->pos < theEditorBuffer->size)
	{
	 theEditorBuffer->pos++;
	 PaintField_and_FieldCursorBuffer(theGC, theViewWindow,
					  theField, theEditorBuffer);
	 theKeyProcessed = keyPROCESS_SUCCESS;
	}
       else
	theKeyProcessed = keyPROCESS_SWALLOW;

      break;

    case XK_Up:
      if (theFieldType == TEXTAREA)
	{
	 PrevLine(theEditorBuffer);
	 PaintField_and_FieldCursorBuffer(theGC, theViewWindow,
					  theField, theEditorBuffer);
	 theKeyProcessed = keyPROCESS_SUCCESS;
	}
       else
	theKeyProcessed = keyPROCESS_SWALLOW;

      break;

    case XK_Down:
      if (theFieldType == TEXTAREA)
	{
	 NextLine(theEditorBuffer);
	 PaintField_and_FieldCursorBuffer(theGC, theViewWindow,
					  theField, theEditorBuffer);
	 theKeyProcessed = keyPROCESS_SUCCESS;
	}
       else
	theKeyProcessed = keyPROCESS_SWALLOW;

      break;

    case XK_BackSpace:
      if (theEditorBuffer->pos)
	{
	 theEditorBuffer->pos--;
	 /* _NO_ `break'! */
	}
       else
	{
	 theKeyProcessed = keyPROCESS_SWALLOW;
	 break;
	}
      /* _NO_ `break'! */

    case XK_Delete:
#if (XK_DeleteChar != XK_Delete)
    case XK_DeleteChar:
#endif
      if (theEditorBuffer->pos < theEditorBuffer->size)
	{
	 DeleteChar(theEditorBuffer, theEditorBuffer->pos);

	 Free(theField->value);
	 theField->value   = Buffer2Str(theEditorBuffer);
	 theField->buflen  = theEditorBuffer->size;
	 theField->bufsize = theEditorBuffer->size + 1;

	 PaintField_and_FieldCursorBuffer(theGC, theViewWindow,
					  theField, theEditorBuffer);
	 theKeyProcessed = keyPROCESS_SUCCESS;
	}
       else
	theKeyProcessed = keyPROCESS_SWALLOW;

      break;

    case XK_Return:
      if ((theFieldType == TEXTFIELD) || (theFieldType == PASSWD))
	{
	 theKeyProcessed = keySUBMIT_FORM;
	 break;
	}
       else
	{
	 if (theString == NULL) theString = LineFeed;
	 theKeyProcessed = keyPROCESS_SUCCESS;
	 /* NO `break'! */
	}
      /* NO `break'! */

      /* normal keypress... adding */
    default:
      {
       char c;
       char* cp;

       if (theString)
	 {
	  for (cp = theString; (c = *cp); cp++)
	    if (isprint(c) || (c == '\n') || (c == '\r'))
	      {
	       if (c == '\r') c = '\n';
	       if ((theField->maxlength <= 0) ||
		   (theField->buflen < theField->maxlength))
		 {
		  InsertChar(theEditorBuffer, theEditorBuffer->pos, c);
		  theKeyProcessed = keyPROCESS_SUCCESS;
		 }

	       theEditorBuffer->pos++;
	       Free(theField->value);
	       theField->value   = Buffer2Str(theEditorBuffer);
	       theField->buflen  = theEditorBuffer->size;
	       theField->bufsize = theEditorBuffer->size + 1;

	       PaintField_and_FieldCursorBuffer(theGC, theViewWindow,
						theField, theEditorBuffer);
	      }
	     else
	      {
	       Beep();
	       theKeyProcessed = keyPROCESS_SWALLOW;
	       break;
	      }
	 }
        else
	 {
	  Beep();
	  theKeyProcessed = keyPROCESS_SWALLOW;
	 }
      }
      break;
   }
 /* End ``switch (theKey)'' */

 return theKeyProcessed;
}


/*
 * Get <select> menu item (<option>) by it's number (index) in the list.
 */
static Option* GetSelectOption_by_N(Field* theField, int ix)
{
 Option* opt_ix;


 if ((ix <= 0) || (theField == NULL)) return NULL;
 if (theField->options == NULL) return NULL;

 for (opt_ix = theField->options, ix--; ix && opt_ix; ix--)
   opt_ix = opt_ix->next;

 return opt_ix;
}


#ifdef ARENA_DEBUG
/*
 * Get <select> menu item index (<option> number in the list).
 */
static int GetSelectOptionIndex(Field* theField, Option* theOption)
{
 int ix;
 Option* opt_ix;
#ifdef ARENA_DEBUG
 char Iam[] = "GetSelectOptionIndex";
#endif


 if (theField == NULL) return 0;
 if (theField->options == NULL) return 0;

 if (theOption == NULL) return 0;
 if (theOption->next == NULL) return 0;

 if (theField->type != OPTIONLIST)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not <select>.\n");
#endif
    return 0;
   }

 for (opt_ix = theField->options, ix = 1; opt_ix; opt_ix = opt_ix->next, ix++)
   if (opt_ix == theOption) break;

 return (opt_ix ? ix : 0);
}
#endif


/*
 * Find previous nondisabled <option> (item in the menu (<select> field)).
 */
static NumberedOption GetPrevNondisabledSelectOption(Field* theField,
						     int theNumber)
{
 int ix;
 Option* opt_ix;
 NumberedOption nOpt = { 0, NULL };
#ifdef ARENA_DEBUG
 char Iam[] = "GetPrevNondisabledSelectOption";
#endif


 if (theNumber <= 0) return nOpt;
 if (theField == NULL) return nOpt;
 if (theField->options == NULL) return nOpt;
 if (theField->options->next == NULL) return nOpt;

 if (theField->type != OPTIONLIST)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not <select>.\n");
#endif
    return nOpt;
   }

 /*
  * Check if the given option is in the given list.
  */
 if ((opt_ix = GetSelectOption_by_N(theField, theNumber)) == NULL)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! No such option in the list.\n");
#endif
    return nOpt;
   }

 for (ix = theNumber; opt_ix; )
   {
    if (opt_ix->next)
      {
       opt_ix = opt_ix->next;
       ix++;
      }
     else
      {
       opt_ix = theField->options;
       ix = 1;
      }

    if (ix == theNumber)
      break;
     else
      if (!(opt_ix->flags & OPTION_DISABLED))
	{
	 nOpt.number = ix;
	 nOpt.option = opt_ix;
	}
   }

 return nOpt;
}


/*
 * Find next nondisabled <option> (item in the menu (<select> field)).
 */
static NumberedOption GetNextNondisabledSelectOption(Field* theField,
						     int theNumber)
{
 int ix;
 Option* opt_ix;
 NumberedOption nOpt = { 0, NULL };
#ifdef ARENA_DEBUG
 char Iam[] = "GetNextNondisabledSelectOption";
#endif


 if (theNumber <= 0) return nOpt;
 if (theField == NULL) return nOpt;
 if (theField->options == NULL) return nOpt;
 if (theField->options->next == NULL) return nOpt;

 if (theField->type != OPTIONLIST)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not <select>.\n");
#endif
    return nOpt;
   }

 /*
  * Check if the given option is in the given list.
  */
 if ((opt_ix = GetSelectOption_by_N(theField, theNumber)) == NULL)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! No such option in the list.\n");
#endif
    return nOpt;
   }

 for (ix = theNumber; opt_ix; )
   {
    if (opt_ix->next)
      {
       opt_ix = opt_ix->next;
       ix++;
      }
     else
      {
       opt_ix = theField->options;
       ix = 1;
      }

    if (ix == theNumber)
      {
       opt_ix = NULL;
       break;
      }
     else
      if (!(opt_ix->flags & OPTION_DISABLED)) break;
   }

 if (opt_ix)
   {
    nOpt.number = ix;
    nOpt.option = opt_ix;
   }

 return nOpt;
}


/* Take the proper editing action for button/key click/press in
 * the given (by it's number) <option> item of the <select> menu.
 */
Bool ClickedInSelectFieldMenuItem_N(Field* theField, int theItemIndex)
{
 Option* option;

 if ((option = GetSelectOption_by_N(theField, theItemIndex)))
   {
    if (option->flags & OPTION_DISABLED)
      return False;	/* The key/button press/click ignored */
     else
      {
       if (theField->flags & MULTIPLE)
	 {
	  if (option->flags & OPTION_CHECKED)
	    {
	     option->flags &= ~OPTION_CHECKED;

	     for (option = theField->options; option; option = option->next)
	       {
		if (option->flags & OPTION_CHECKED) break;
	       }
	    }
	   else
	    option->flags |= OPTION_CHECKED;
	 }
        else
	 option->flags |= OPTION_CHECKED;

       theField->pos = theItemIndex;
      }
   }

 if (!(theField->flags & MULTIPLE)) theField->flags &= ~CHECKED;

 Free(theField->value);
 theField->value = (option ? StrDup(option->label) : NULL);

 return True;
}


/*
 * Edit option list (<select>) field with mouse.
 */
Bool ClickedInDropDown(GC theGC, ViewWindow theViewWindow,
		       Field* field, int indent, int event, int x, int y)
{
 int font, baseline, bw, bh, dh, x1, y1;
 Option *option;
#ifdef ARENA_DEBUG
 char Iam[] = "ClickedInDropDown";
#endif


 if (field == NULL) return False;
 if (field->type != OPTIONLIST)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam , " ERROR? The field is not <select>.\n");
#endif
    return False;
   }

      /* If the dropdown menu is empty, nobody can click in it. */
 if (field->options == NULL)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! <select> options list is empty.\n");
#endif
    return False;
   }

      /* If the dropdown menu is hidden (rolled up), nobody can click in it. */
 if (!(field->flags & CHECKED)) return False;

 font = field->flags & 0x0F;

 baseline = field->baseline;
 bw = field->width - field->height + 2;
 bh = 6 + field->y_indent * (field->height - 4);
 dh = field->height;
 x1 = field->frame_indent + field->x;
 y1 = field->height + baseline - ASCENT(font) - 2;

 if (x1 + 2 < x && x < x1 + bw - 2 && y1 + 2 < y && y < y1 + bh)
   {
    if (event == BUTTONUP)
      {
       int cur_pos;

       if (!(field->flags & MULTIPLE))
	 {
	  for (option = field->options; option; option = option->next)
	    option->flags &= ~OPTION_CHECKED;
	 }

       cur_pos = ((y - y1) * field->y_indent)/bh + 1;

       if (ClickedInSelectFieldMenuItem_N(field, cur_pos))
	 PaintSelectField(theGC, theViewWindow,
			  -1, focus->baseline, True, field);
        else
	 return True;	/* Key/Button press/click produced no action */
      }
    /* End ``if mouse event is button up'' */

    return True;
   }

 return False;
}


/*
 * Edit <select> field (OPTIONLIST).
 */
static KeyProcessFeedback EditFormSelectField(GC theGC, ViewWindow theViewWindow,
					      Field* theField, KeySym theKey)
{
 KeyProcessFeedback theKeyProcessed = keyPROCESS_NOACTION;
#ifdef ARENA_DEBUG
 extern Field* focus;
 char Iam[] = "EditFormSelectField";
#endif


 if (theField == NULL) return keyPROCESS_FAILED;

#ifdef ARENA_DEBUG
 if (focus != theField)
   {
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not in focus.\n");
   }
#endif

 if (theField->type != OPTIONLIST)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not <select>.\n");
#endif
    return keyPROCESS_FAILED;
   }

 if (theField->options == NULL)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! <select> options list is empty.\n");
#endif
    return keyPROCESS_FAILED;
   }

 if (theField->flags & CHECKED)
   {
    switch (theKey)
      {
       case XK_Up:
       case XK_Down:
       if (theField->options->next)	/* If the option is not alone */
	 {
	  Option* theOption = GetSelectOption_by_N(theField, theField->pos);
	  NumberedOption nOpt;

#ifdef ARENA_DEBUG
	  if ((!(theField->flags & MULTIPLE)) &&
	      (!(theOption->flags & OPTION_CHECKED)) &&
	      (FORM_TRACE))
	    Arena_TracePrint(Iam, " ERROR! Current option not checked.\n");
#endif

	  /* Find previous/next nondisabled option */
	  if (theKey == XK_Up)
	    nOpt = GetPrevNondisabledSelectOption(theField, theField->pos);
	   else
	    nOpt = GetNextNondisabledSelectOption(theField, theField->pos);

	  if (nOpt.number)
	    {
	     theField->pos = nOpt.number;
	     if (!(theField->flags & MULTIPLE))
	       {
		theOption->flags ^= OPTION_CHECKED;
#ifdef ARENA_DEBUG
		if (((nOpt.option)->flags & OPTION_CHECKED) && (FORM_TRACE))
		  Arena_TracePrint(Iam, " ERROR! Multiple checked options.\n");
#endif
		(nOpt.option)->flags |= OPTION_CHECKED;
	       }

	     theKeyProcessed = keyPROCESS_SUCCESS;
	    }
	   else
	    theKeyProcessed = keyPROCESS_SWALLOW;
	 }
        else
	 theKeyProcessed = keyPROCESS_SWALLOW;

      break;

       case XK_space:
	 if (ClickedInSelectFieldMenuItem_N(theField, theField->pos))
	   theKeyProcessed = keyPROCESS_SUCCESS;
	  else
	   theKeyProcessed = keyPROCESS_SWALLOW;

	 break;

       case XK_Return:
	 theField->flags ^= CHECKED;
	 if (theField->flags & MULTIPLE)
	   theKeyProcessed = keyPROCESS_SUCCESS;
	  else
	   if (ClickedInSelectFieldMenuItem_N(theField, theField->pos))
	     theKeyProcessed = keyPROCESS_SUCCESS;
	    else
	     theKeyProcessed = keyPROCESS_SWALLOW;

	 break;

       default:
	 theKeyProcessed = keyPROCESS_NOACTION;
	 break;
      }
    /* End ``switch (theKey)'' */
   }
  else
   {
    switch (theKey)
      {
       case XK_Up:
       case XK_Down:
       case XK_Return:
       case XK_space:
	 theField->flags |= CHECKED;
	 if (theField->pos <= 0) InitiateSelectFieldCursorPosition(theField);
	 theKeyProcessed = keyPROCESS_SUCCESS;
	 break;

       default:
	 theKeyProcessed = keyPROCESS_NOACTION;
	 break;
      }
    /* End ``switch (theKey)'' */
   }
 /* End ``if (theField->flags & CHECKED)'' */

 if (theKeyProcessed == keyPROCESS_SUCCESS)
   {
#ifdef ARENA_DEBUG
    PaintSelectField(theGC, theViewWindow,
		     -1, theField->baseline, (focus == theField), theField);
# else
    PaintSelectField(theGC, theViewWindow,
		     -1, theField->baseline, True, theField);
#endif
   }

 return theKeyProcessed;
}


/*
 * Edit radio button field (and sync the appropriate radio buttons group).
 */
static KeyProcessFeedback EditFormRadioButton(GC theGC,
					      ViewWindow theViewWindow,
					      Field* theField, KeySym theKey)
{
 KeyProcessFeedback theKeyProcessed = keyPROCESS_NOACTION;
#ifdef ARENA_DEBUG
 extern Field* focus;
 char Iam[] = "EditFormRadioButton";
#endif


 if (theField == NULL) return keyPROCESS_FAILED;

#ifdef ARENA_DEBUG
 if (focus != theField)
   {
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not in focus.\n");
   }
#endif

 if (theField->type != RADIOBUTTON)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not a radio button.\n");
#endif
    return keyPROCESS_FAILED;
   }
                                 /* the button is already set */
 if (theField->flags & CHECKED) return keyPROCESS_SWALLOW;

 switch (theKey)
   {
    case XK_Return:
    case XK_space:
      if (SetRadioButton(theField))
	theKeyProcessed = keyPROCESS_SUCCESS;
       else
	theKeyProcessed = keyPROCESS_FAILED;
      break;

    default:
      theKeyProcessed = keyPROCESS_NOACTION;
      break;
   }
 /* End ``switch (theKey)'' */

 if (theKeyProcessed == keyPROCESS_SUCCESS)
   PaintRadioButtonGroup(theGC, theViewWindow, theField);

 return theKeyProcessed;
}


/*
 * Edit checkbox field.
 */
static KeyProcessFeedback EditFormCheckboxField(GC theGC, ViewWindow theViewWindow,
						Field* theField, KeySym theKey)
{
 KeyProcessFeedback theKeyProcessed = keyPROCESS_NOACTION;
#ifdef ARENA_DEBUG
 extern Field* focus;
 char Iam[] = "EditFormCheckboxField";
#endif


 if (theField == NULL) return keyPROCESS_FAILED;

#ifdef ARENA_DEBUG
 if (focus != theField)
   {
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not in focus.\n");
   }
#endif

 if (theField->type != CHECKBOX)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not a checkbox.\n");
#endif
    return keyPROCESS_FAILED;
   }

 switch (theKey)
   {
    case XK_Return:
    case XK_space:
      theField->flags ^= CHECKED;
      theKeyProcessed = keyPROCESS_SUCCESS;
      break;

    default:
      theKeyProcessed = keyPROCESS_NOACTION;
      break;
   }
 /* End ``switch (theKey)'' */

#ifdef ARENA_DEBUG
 if (theKeyProcessed == keyPROCESS_SUCCESS)
   PaintCheckboxField(theGC, theViewWindow,
		      -1, theField->baseline, (focus == theField), theField);
# else
 if (theKeyProcessed == keyPROCESS_SUCCESS)
   PaintCheckboxField(theGC, theViewWindow,
		      -1, theField->baseline, True, theField);
#endif

 return theKeyProcessed;
}


/*
 * ``Edit'' submit button --- just nanlyze the key pressed.
 */
static KeyProcessFeedback EditFormSubmitButton(GC theGC, ViewWindow theViewWindow,
					       Field* theField, KeySym theKey)
{
 KeyProcessFeedback theKeyProcessed = keyPROCESS_NOACTION;
#ifdef ARENA_DEBUG
 extern Field* focus;
 char Iam[] = "EditFormSubmitButton";
#endif


 if (theField == NULL) return keyPROCESS_FAILED;

#ifdef ARENA_DEBUG
 if (focus != theField)
   {
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not in focus.\n");
   }
#endif

 if (theField->type != SUBMITBUTTON)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not a submit button.\n");
#endif
    return keyPROCESS_FAILED;
   }

 switch (theKey)
   {
    case XK_Return:
    case XK_space:
      theKeyProcessed = keySUBMIT_FORM;
      break;

    default:
      theKeyProcessed = keyPROCESS_NOACTION;
      break;
   }
 /* End ``switch (theKey)'' */

 return theKeyProcessed;
}


/*
 * ``Edit'' reset button --- just nanlyze the key pressed.
 */
static KeyProcessFeedback EditFormResetButton(GC theGC, ViewWindow theViewWindow,
					      Field* theField, KeySym theKey)
{
 KeyProcessFeedback theKeyProcessed = keyPROCESS_NOACTION;
#ifdef ARENA_DEBUG
 extern Field* focus;
 char Iam[] = "EditFormResetButton";
#endif


 if (theField == NULL) return keyPROCESS_FAILED;

#ifdef ARENA_DEBUG
 if (focus != theField)
   {
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not in focus.\n");
   }
#endif

 if (theField->type != RESETBUTTON)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The field is not a reset button.\n");
#endif
    return keyPROCESS_FAILED;
   }

 switch (theKey)
   {
    case XK_Return:
    case XK_space:
      theKeyProcessed = keyPROCESS_SUCCESS;
      break;

    default:
      theKeyProcessed = keyPROCESS_NOACTION;
      break;
   }
 /* End ``switch (theKey)'' */

 return theKeyProcessed;
}


/*
 * Take form editing action in response to keyboard event.
 */
Bool EditDocFormField(GC theGC, ViewWindow theViewWindow,
		      Doc* theDoc, KeySym theKey, char* theString)
{
 Bool theKeyProcessed = False;
 KeyProcessFeedback theKeyProcessedFeedback = keyPROCESS_NOACTION;
#ifdef ARENA_DEBUG
 char Iam[] = "EditDocFormField";
#endif


 if (theDoc == NULL) return False;
 if (theDoc->edited_field == NULL) return False;
 if (theDoc->edited_field->type == HIDDEN)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! Trying to edit HIDDEN field.\n");
#endif
    return False;
   }

 switch (theKey)
   {
#if 0
    case XK_Home:
      break;

    case XK_Begin:
      break;

    case XK_End:
      break;

    case XK_Next:
#if (XK_Page_Down != XK_Next)
    case XK_Page_Down:
#endif
      break;

    case XK_Prior:
#if (XK_Page_Up != XK_Prior)
    case: XK_Page_Up:
#endif
    break;

    case XK_Escape:
      break;

    case XK_Pause:
      break;
#endif
    case XK_Tab:
      {
      /* Auxilliary pointer required as GiveUpFocus()
       * frees and/or NULL's it's arguments (dereferenced).
       */
       Field* theField = theDoc->edited_field;
       Form*  theForm  = theField->form;

       if (theForm->fields->next)
	 {
	  Field* field_ix;

	  for (field_ix = theField; field_ix; )
	    {
	     field_ix = (field_ix->next ? field_ix->next : theForm->fields);
	     if (field_ix == theField)
	       {
	        field_ix = NULL;
		break;
	       }
	      else
	       if (field_ix->type != HIDDEN) break;
	    }

	  theField = field_ix;
	 }

       if (theField)
	 MoveFocusTo(theGC, theViewWindow,
		     theDoc, -1, theField->baseline, theField);
        else
	 GiveUpFocus(theGC, theViewWindow,
		     &(theDoc->edited_field), &(theDoc->field_editor));
      }
      theKeyProcessed = True;
      break;

      /* normal keypress... adding */
    default:
      switch (theDoc->edited_field->type)
	{
	 case TEXTFIELD:
	 case TEXTAREA:
	 case PASSWD:
	   if (theDoc->field_editor)
	     {
	      theKeyProcessedFeedback = EditFormTextField(theGC, theViewWindow,
							  theDoc->edited_field,
							  theDoc->field_editor,
							  theKey,
							  theString);
	      switch (theKeyProcessedFeedback)
		{
		 case keyPROCESS_SUCCESS:
		 case keyPROCESS_SWALLOW:
		 case keySUBMIT_FORM:
		   theKeyProcessed = True;
		   break;

		 case keyPROCESS_NOACTION:
		 case keyPROCESS_FAILED:
		   theKeyProcessed = False;
		   break;
		}
	     }
	   break;

	 case OPTIONLIST:
	   theKeyProcessedFeedback = EditFormSelectField(theGC, theViewWindow,
							 theDoc->edited_field,
							 theKey);
	   switch (theKeyProcessedFeedback)
	     {
	      case keyPROCESS_SUCCESS:
	      case keyPROCESS_SWALLOW:
		theKeyProcessed = True;
		break;

	      case keySUBMIT_FORM:
#ifdef ARENA_DEBUG
		if (FORM_TRACE)
		  Arena_TracePrint(Iam, " Select field handler ERROR.\n");
		/* NO `break'! */
#endif
	      case keyPROCESS_NOACTION:
	      case keyPROCESS_FAILED:
		theKeyProcessed = False;
		break;
	     }
	   /* End ``switch (theKeyProcessedFeedback)'' */
	   break;

	 case RADIOBUTTON:
	   theKeyProcessedFeedback = EditFormRadioButton(theGC, theViewWindow,
							 theDoc->edited_field,
							 theKey);
	   switch (theKeyProcessedFeedback)
	     {
	      case keyPROCESS_SUCCESS:
	      case keyPROCESS_SWALLOW:
		theKeyProcessed = True;
		break;

	      case keySUBMIT_FORM:
#ifdef ARENA_DEBUG
		if (FORM_TRACE)
		  Arena_TracePrint(Iam, " Radio button handler ERROR.\n");
		/* NO `break'! */
#endif
	      case keyPROCESS_NOACTION:
	      case keyPROCESS_FAILED:
		theKeyProcessed = False;
		break;
	     }
	   /* End ``switch (theKeyProcessedFeedback)'' */
	   break;

	 case CHECKBOX:
	   theKeyProcessedFeedback = EditFormCheckboxField(theGC, theViewWindow,
							  theDoc->edited_field,
							   theKey);
	   switch (theKeyProcessedFeedback)
	     {
	      case keyPROCESS_SUCCESS:
	      case keyPROCESS_SWALLOW:
		theKeyProcessed = True;
		break;

	      case keySUBMIT_FORM:
#ifdef ARENA_DEBUG
		if (FORM_TRACE)
		  Arena_TracePrint(Iam, " Checkbox field handler ERROR.\n");
		/* NO `break'! */
#endif
	      case keyPROCESS_NOACTION:
	      case keyPROCESS_FAILED:
		theKeyProcessed = False;
		break;
	     }
	   /* End ``switch (theKeyProcessedFeedback)'' */
	   break;

	 case SUBMITBUTTON:
	   theKeyProcessedFeedback = EditFormSubmitButton(theGC, theViewWindow,
							  theDoc->edited_field,
							  theKey);
	   switch (theKeyProcessedFeedback)
	     {
	      case keySUBMIT_FORM:
		theKeyProcessed = True;
		break;

	      case keyPROCESS_SUCCESS:
	      case keyPROCESS_SWALLOW:
#ifdef ARENA_DEBUG
		if (FORM_TRACE)
		  Arena_TracePrint(Iam, " Submit button handler ERROR.\n");
		/* NO `break'! */
#endif
	      case keyPROCESS_NOACTION:
	      case keyPROCESS_FAILED:
		theKeyProcessed = False;
		break;
	     }
	   /* End ``switch (theKeyProcessedFeedback)'' */
	   break;

	 case RESETBUTTON:
	   theKeyProcessedFeedback = EditFormResetButton(theGC, theViewWindow,
							 theDoc->edited_field,
							 theKey);
	   switch (theKeyProcessedFeedback)
	     {
	      case keyPROCESS_SUCCESS:
		theKeyProcessed = ResetForm(theGC, theViewWindow,
					    theDoc->edited_field->form);
		break;

	      case keySUBMIT_FORM:
	      case keyPROCESS_SWALLOW:
#ifdef ARENA_DEBUG
		if (FORM_TRACE)
		  Arena_TracePrint(Iam, " Reset button handler ERROR.\n");
		/* NO `break'! */
#endif
	      case keyPROCESS_NOACTION:
	      case keyPROCESS_FAILED:
		theKeyProcessed = False;
		break;
	     }
	   /* End ``switch (theKeyProcessedFeedback)'' */
	   break;

#ifdef ARENA_DEBUG
	 case HIDDEN:
	   if (FORM_TRACE)
	     Arena_TracePrint(Iam, " ERROR! Editing HIDDEN field.\n");
	   theKeyProcessed = False;
	   break;
#endif

	 default:
#ifdef ARENA_DEBUG
	   if (FORM_TRACE)
	     Arena_TracePrint(Iam, " ERROR! Editing weird type %d field.\n");
#endif
	   theKeyProcessed = False;
	   break;
	}
      /* End if ``switch (theDoc->field->type)'' */

      if (theKeyProcessed && (theKeyProcessedFeedback == keySUBMIT_FORM))
	{
	/* Auxilliary pointer required as GiveUpFocus()
	 * frees and NULL's it's arguments (dereferenced).
	 */
	 Field* theField = theDoc->edited_field;
	 Form*  theForm  = theField->form;

	 GiveUpFocus(theGC, theViewWindow,
		     &(theDoc->edited_field), &(theDoc->field_editor));

	 SubmitForm(theForm,
		    theForm->method, theForm->action, theForm->alen,
		    theField->type,
		    theField->name, theField->nlen,
		    theField->value, Arena_StrLen(theField->value),
		    NULL, 0, 0,
		    NULL);
	}
      /* End ``if the form submission required'' */

      break;
   }
 /* End ``switch (theKey)'' */

 return theKeyProcessed;
}


/*
 * Set clip rectangle to clip text strings in text fields.
 */
Bool ClipIntersection(GC theGC,
		     int x, int y, unsigned int width, unsigned int height)
{
 XRectangle rect = {x, y, width, height};

 XSetClipRectangles(display, theGC, 0, 0, &rect, 1, Unsorted);
 return True;
}


ArenaXPoint FieldCursorRelativeXPoint(Field* thefield, int thefont)
{
 ArenaXPoint theCursorXPoint;
 int r = 0;
 int lineheight = ASCENT(thefont) + 4;
 int numcol, numline;

 numcol  = ColumnNumber_in_String(thefield->value, thefield->buflen,
				  thefield->pos);
 numline = LineNumber_in_String(thefield->value, thefield->buflen,
				thefield->pos);

 if ((thefield->bufsize > 0) && (thefield->buflen > 0))
   r = WIDTH(thefont, thefield->value + thefield->pos - numcol, numcol);
  else
   r = 0;

 theCursorXPoint.x = r;
 theCursorXPoint.y = numline * lineheight;

 return theCursorXPoint;
}


/*
 * Paint text-type field, i.e. TEXTFIELD, TEXTAREA, or PASSWD type field.
 */
Bool PaintTextField(GC theGC, ViewWindow theViewWindow,
		    int indent, int baseline, Bool theFieldIsActive,
		    Field* theField)
{
 Bool TotalSuccess = True, PaintFieldCursor = False;
 int font, width, height, x1, y2, lineheight;
#ifdef ARENA_DEBUG
 char Iam[] = "PaintTextField";
#endif


 if (theField == NULL) return False;

 {
  unsigned char theFieldType = theField->type;

  if (!((theFieldType == TEXTFIELD) ||
	(theFieldType == TEXTAREA) ||
	(theFieldType == PASSWD)))
    {
#ifdef ARENA_DEBUG
     if (FORM_TRACE)
       Arena_TracePrint(Iam,
			" ERROR! The field type %d isn't proccessable.\n",
			theFieldType);
#endif
     return False;
    }
 }

 PaintFieldCursor = (theFieldIsActive ? (theField->type != PASSWD) : False);

 font = theField->flags & 0x0F;

 width  = theField->width;
 height = theField->height;

 lineheight = ASCENT(font) + 4;

 /* cache absolute position of baseline */
 theField->baseline = baseline;

 /* kludge for fields in different frames */
 if (indent >= 0)
   theField->frame_indent = indent;
  else
   indent = theField->frame_indent;

 /***/

 x1 = indent + theField->x - theViewWindow.rect.x;
 y2 = baseline - lineheight + 2 - theViewWindow.rect.y;

 ClipIntersection(theGC, x1, y2, width, height);
 XSetForeground(display, theGC,
		(theFieldIsActive ? statusColour : windowColour));
 XFillRectangle(display, theViewWindow.win, theGC, x1, y2, width, height);
 DrawInSet(theViewWindow.win, theGC, x1, y2, width, height);
 ClipIntersection(theGC, x1 + 2, y2 + 2, width - 4, height - 4);

 if ((theField->buflen > 0) || PaintFieldCursor)
   {
    int dx = x1 + 2, dy = y2 + 2, cursorWidth = 1;
    char *pos, *dump;
    ArenaXPoint theCursorXPoint = FieldCursorRelativeXPoint(theField, font);

    {
     int d;

     if ((d = theCursorXPoint.x - theField->x_indent) < 0)
       theField->x_indent = theCursorXPoint.x;
      else
       if ((d +=  ((cursorWidth + 4) - width)) > 0) theField->x_indent += d;

     if ((d = theCursorXPoint.y - theField->y_indent) < 0)
       theField->y_indent = theCursorXPoint.y;
      else
       if ((d += ((lineheight + 4) - height)) > 0) theField->y_indent += d;
    }

    dx -= theField->x_indent;
    dy -= theField->y_indent;

    if ((theField->buflen > 0) && theField->value)
      {
       int dyi;

       dump = (char *)Arena_MAlloc(Arena_StrLen(theField->value) + 1, False);
       strncpy(dump, theField->value, theField->buflen);
       *(dump + theField->buflen) = 0;

       pos = dump;
       if (theField->type == PASSWD)
	 {
	  while (*pos)
	    {
	     if (*pos == '\n')
	       {
		*pos++ = '\0';
		break;
	       }
	      else
	       *pos++ = '*';
	    }
	 }
        else
	 {
	  while (*pos)
	    {
	     if (*pos == '\n')
	       *pos++ = '\0';
	      else
	       pos++;
	    }
	 }
       /* End ``if (theField->type == PASSWD)'' */

       dyi = dy + (lineheight - 4);
       {
	int i, slen = 0;

	pos = dump;
	for (i = theField->buflen; i > 0; i -= slen)
	  {
	   slen = Arena_StrLen(pos);
	   SetFont(theGC, font);
	   XSetForeground(display, theGC, textColour);
	   XDrawString(display, theViewWindow.win, theGC, dx, dyi, pos, slen);
	   dyi += lineheight;
	   pos += (++slen);	/* increment counts trailing zero */
	  }
       }

       Free(dump);
      }

    if (PaintFieldCursor)
      {
       XSetForeground(display, theGC, strikeColour);
       XFillRectangle(display, theViewWindow.win, theGC,
		      dx + theCursorXPoint.x, dy + theCursorXPoint.y,
		      cursorWidth, lineheight);
      }
   }

 XSetForeground(display, theGC, textColour);
 XSetClipMask(display, theGC, None);

 return TotalSuccess;
}


Bool PaintDropDown(GC theGC, ViewWindow theViewWindow,
		   int indent, Field* theField)
{
 Bool TotalSuccess = True;
 int font, baseline, width, height, h, x1, y1, y2, i;
 Option* option_i;
#ifdef ARENA_DEBUG
 Bool CursorDrawn = False;
 char Iam[] = "PaintDropDown";
#endif


 if (theField == NULL) return False;

 if (theField->type != OPTIONLIST)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, "ERROR! The form field is not <select>.\n");
#endif
    return False;
   }

 if (theField->options == NULL)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! <select> options list is empty.\n");
#endif
    return False;
   }

 /* nasty kludge !!!! */
 if (indent < 0) indent = theField->frame_indent;

 baseline = theField->baseline;
 font = theField->flags & 0x0F;

 SetFont(theGC, font);
 height = theField->height;
 width  = theField->width - height + 2;
 x1 = indent + theField->x - theViewWindow.rect.x;
 y1 = height + baseline - ASCENT(font) - 2 - theViewWindow.rect.y;
 h = 6 + theField->y_indent * (height - 4);
 y2 = baseline + height;

 XSetForeground(display, theGC, windowColour);
 XFillRectangle(display, theViewWindow.win, theGC, x1, y1, width, h);
 DrawOutSet(theViewWindow.win, theGC, x1, y1, width, h);

 y1 += 2;

 for (option_i = theField->options, i = 0; option_i; option_i = option_i->next)
   {
    if (option_i->flags & OPTION_DISABLED)
      {
#ifdef ARENA_DEBUG
       if (option_i->flags & OPTION_CHECKED)
	 {
	  if (FORM_TRACE)
	    Arena_TracePrint(Iam,
			     " ERROR! A field item is disabled && checked.\n");

	  option_i->flags &= ~OPTION_CHECKED;
	 }
#endif
       XSetForeground(display, theGC, statusColour);
      }
     else
      {
       if (option_i->flags & OPTION_CHECKED)
	 {
	  XSetForeground(display, theGC, statusColour);
	  XFillRectangle(display, theViewWindow.win, theGC,
			 x1 + 2, y1, width - 4, height - 3);
#ifdef ARENA_DEBUG
	  if (!(theField->flags & MULTIPLE))
	    {
	     if (CursorDrawn)
	       Arena_TracePrint(Iam, " ERROR! Multiple selected fields.\n");
	      else
	       CursorDrawn = True;
	    }
#endif
	 }

       XSetForeground(display, theGC, textColour);
      }
    /* End ``if the item is disabled'' */

    if (option_i->label)
      XDrawString(display, theViewWindow.win, theGC, x1 + 4, y2,
		  option_i->label, Arena_StrLen(option_i->label));

    if (++i == theField->pos)
      {
       if (theField->flags & MULTIPLE)
	 {
	  XSetForeground(display, theGC, strikeColour);
	  XDrawRectangle(display, theViewWindow.win, theGC,
			 x1 + 3, y1, width - 6, height - 5);
#ifdef ARENA_DEBUG
	  CursorDrawn = True;
#endif
	 }
#ifdef ARENA_DEBUG
        else
	 {
	  if (!(option_i->flags & OPTION_CHECKED))
	    if (FORM_TRACE)
	      Arena_TracePrint(Iam, " ERROR! Cursor item isn't checked.\n");
	  CursorDrawn = True;
	 }
#endif
      }

    y2 += height - 4;
    y1 += height - 4;
   }

#ifdef ARENA_DEBUG
 if ((theField->pos != 0) && !CursorDrawn)
   if (FORM_TRACE)
     Arena_TracePrint(Iam, " ERROR! Corrupted cursor position!\n");     
#endif

 /* Set the field auxilliary flag which in the context of OPTIONLIST type
  * fields indicates the dropdown menu has been drawn (is rolled out (down?)).
  */
 theField->flags |= FIELD_AUX_FLAG;

 return TotalSuccess;
}


Bool HideDropDown(GC theGC, ViewWindow theViewWindow,
		  Doc* theDoc, int indent, Field* theField)
{
#ifdef ARENA_DEBUG
 char Iam[] = "HideDropDown";
#endif

 if (theDoc)
   {
    if (theField)
      {
       if (theField->type == OPTIONLIST)
	 {
	  if (theField->options)
	    {
	     Bool TotalSuccess = True;
	     int width, height;

	     /* The field auxilliary flag in the context of OPTIONLIST type
	      * fields indicates the dropdown menu has been drawn
	      * (is rolled out (down?)).
	      * Hence it's unreasonable to hide (roll up) the dropdown menu,
	      * if it is not rolled out.
	      */
	     if (theField->flags & FIELD_AUX_FLAG)
	       {
	       /* Reset the field auxilliary flag,
		* i.e. flag the dropdown menu as hidden (rolled up).
		*/
		theField->flags &= ~FIELD_AUX_FLAG;

		/* nasty kludge !!!! */
		if (indent < 0) indent = theField->frame_indent;

		height = theField->height;
		width  = theField->width - height + 2;
		{
		 ViewRectangle theArea = {(theField->frame_indent +
					   theField->x - theViewWindow.rect.x),
					  (height + theField->baseline -
					   ASCENT(font) - 2 -
					   theViewWindow.rect.y),
					  (width),
					  (6 +
					   theField->y_indent * (height - 4))};

		 DisplayHTML(theDoc, theGC, theViewWindow, theArea);
		}
		return TotalSuccess;
	       }
	      else
	       return False;
	    }
	   else
	    {
#ifdef ARENA_DEBUG
	     if (FORM_TRACE)
	       Arena_TracePrint(Iam,
				" ERROR! <select> options list is empty.\n");
#endif
	     return False;
	    }
	 }
        else
	 {
#ifdef ARENA_DEBUG
	  if (FORM_TRACE)
	    Arena_TracePrint(Iam, "ERROR! The form field is not <select>.\n");
#endif
	  return False;
	 }
      }
     else
      {
#ifdef ARENA_DEBUG
       if (FORM_TRACE)
	 Arena_TracePrint(Iam, " ERROR! The field to focus is NULL.\n");
#endif
       return False;
      }
   }
  else
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! The doc given is NULL.\n");
#endif
    return False;
   }
}


/*
 * Paint select (option list) field, i.e. OPTIONLIST type field.
 */
Bool PaintSelectField(GC theGC, ViewWindow theViewWindow,
		      int indent, int baseline,
		      Bool theFieldIsActive, Field* theField)
{
 Bool TotalSuccess = True;
 int font, width, height, x1, y2, lineheight;
 Option* option;
#ifdef ARENA_DEBUG
 char Iam[] = "PaintSelectField";
#endif


 if (theField == NULL) return False;

 {
  unsigned char theFieldType = theField->type;

  if (theFieldType != OPTIONLIST)
    {
#ifdef ARENA_DEBUG
     if (FORM_TRACE)
       Arena_TracePrint(Iam,
			" ERROR! The field type %d isn't proccessable.\n",
			theFieldType);
#endif
     return False;
    }
 }

 font = theField->flags & 0x0F;

 width  = theField->width;
 height = theField->height;

 lineheight = ASCENT(font) + 4;

 /* cache absolute position of baseline */
 theField->baseline = baseline;

 /* kludge for fields in different frames */
 if (indent >= 0)
   theField->frame_indent = indent;
  else
   indent = theField->frame_indent;

 /***/

 x1 = indent + theField->x - theViewWindow.rect.x;
 y2 = baseline - lineheight + 2 - theViewWindow.rect.y;

 XSetForeground(display, theGC,
		(theFieldIsActive ? statusColour : windowColour));
 XFillRectangle(display, theViewWindow.win, theGC, x1, y2, width, height);
 DrawOutSet(theViewWindow.win, theGC, x1, y2, width, height);

 if (theField->flags & MULTIPLE)
   {
    DrawOutSet(theViewWindow.win, theGC,
	       x1 + 3 + width - height, y2 - 2 + height/3,
	       height - 7, 6);
    DrawOutSet(theViewWindow.win, theGC,
	       x1 + 3 + width - height, y2 - 3 + 2 * height/3,
	       height - 7, 6);
   }
  else /* single choice menu drawn with one bar */
   {
    Bool initialize_field_value = True;

    DrawOutSet(theViewWindow.win, theGC,
	       x1 + 3 + width - height, y2 - 3 + height/2,
	       height - 7, 6);

    if (theField->value)
      {
       if (*(theField->value) == '\0' && theField->options != NULL)
	 initialize_field_value = True;
        else
	 initialize_field_value = False;
      }
     else
      initialize_field_value = True;

    if (initialize_field_value)
      {
       theField->pos = 1;
       Free(theField->value);
       if ((option = theField->options))
	 {
	  option->flags |= OPTION_CHECKED;
	  theField->value = StrDup(option->label);
	 }
      }
   }

 if (theField->y_indent > 0 && theField->value)
   {
    SetFont(theGC, font);
    XSetForeground(display, theGC, textColour);
    XDrawString(display, theViewWindow.win, theGC, x1 + 4, baseline,
		theField->value, Arena_StrLen(theField->value));
   }
  else
   XSetForeground(display, theGC, textColour);

 if (theFieldIsActive)
   {
    if (theField->options)
      {
       if (theField->flags & CHECKED)
	 PaintDropDown(theGC, theViewWindow, indent, theField);
        else
	 {
	 /*
	  * The field auxilliary flag in the context of OPTIONLIST type fields
	  * indicates the dropdown menu has been drawn (is rolled out (down?)).
	  * Hence it's unreasonable to hide (roll up) the dropdown menu,
	  * if it is not rolled out.
	  */
	  if (theField->flags & FIELD_AUX_FLAG)
	    HideDropDown(theGC, theViewWindow, CurrentDoc, indent, theField);
	 }
      }
#ifdef ARENA_DEBUG
     else
      {
       if (FORM_TRACE)
	 Arena_TracePrint(Iam, " ERROR! <select> options list is empty.\n");
      }
#endif
   }

 return TotalSuccess;
}


/*
 * Paint radio button field, i.e. RADIOBUTTON field type.
 */
Bool PaintRadioButton(GC theGC, ViewWindow theViewWindow,
		      int indent, int baseline,
		      Bool theFieldIsActive, Field* theField)
{
 Bool TotalSuccess = True, checked;
 int font, width, height, x1, y2, r, lineheight;
#ifdef ARENA_DEBUG
 char Iam[] = "PaintRadioButton";
#endif


 if (theField == NULL) return False;

 if (baseline < 0)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " baseline has a negative value.\n");
#endif
    return False;
   }

 {
  unsigned char theFieldType = theField->type;

  if (theFieldType != RADIOBUTTON)
    {
#ifdef ARENA_DEBUG
     if (FORM_TRACE)
       Arena_TracePrint(Iam,
			" ERROR! The field type %d isn't proccessable.\n",
			theFieldType);
#endif
     return False;
    }
 }

 checked = theField->flags & CHECKED;
 font = theField->flags & 0x0F;

 width  = theField->width;
 height = theField->height;

 lineheight = ASCENT(font) + 4;

 /* cache absolute position of baseline */
 theField->baseline = baseline;

 /* kludge for fields in different frames */
 if (indent >= 0)
   theField->frame_indent = indent;
  else
   indent = theField->frame_indent;

 /***/

 x1 = indent + theField->x - theViewWindow.rect.x;
 y2 = baseline - lineheight + 6 - theViewWindow.rect.y;

 XSetForeground(display, theGC,
		(theFieldIsActive ? statusColour : windowColour));
 XFillArc(display, theViewWindow.win, theGC,
	  x1, y2, width, width, 0, (360 << 6));

 if (checked)
   {
    r = width/4;
    DrawInSetCircle(theViewWindow.win, theGC, x1, y2, width, width);
    XSetForeground(display, theGC, windowBottomShadow);
    width -= (r + r);
    XFillArc(display, theViewWindow.win, theGC,
	     x1 + r, y2 + r, width, width, 0, (360 << 6));
   }
  else
   DrawOutSetCircle(theViewWindow.win, theGC, x1, y2, width, width);

 XSetForeground(display, theGC, textColour);

 return TotalSuccess;
}


/*
 * Repaint all the buttons with the same name.
 */
Bool PaintRadioButtonGroup(GC theGC, ViewWindow theViewWindow, Field* theField)
{
 extern Field* focus;
 Field* field_i;


 if (theField == NULL) return False;

 for (field_i = theField->form->fields;
      field_i != NULL;
      field_i = field_i->next)
   {
    if (field_i->type == RADIOBUTTON)
      if (strncasecmp(field_i->name, theField->name, theField->nlen) == 0)
	if (field_i->baseline >= 0)
	  PaintRadioButton(theGC, theViewWindow,
			   -1, field_i->baseline,
			   (Bool)(focus == field_i), field_i);
   }

 return True;
}


/*
 * Set the given radio button and reset all teh others in it's group.
 */
Bool SetRadioButton(Field* theField)
{
 Field* field_i;


 if (theField == NULL) return False;

 /*
  * Reset all the buttons with the same name.
  */
 for (field_i = theField->form->fields;
      field_i != NULL;
      field_i = field_i->next)
   {
    if (field_i->type == RADIOBUTTON)
      if (strncasecmp(field_i->name, theField->name, theField->nlen) == 0)
	field_i->flags &= ~CHECKED;
   }
 
 /* Set the clicked button */
 theField->flags |= CHECKED;

 return True;
}


void PaintTickMark(GC theGC, Window theWindow,
		   int x, int y, unsigned int w, unsigned int h)
{
 int x1, y1, x2, y2, x3, y3;

 x1 = x;
 x2 = x + w/3;
 x3 = x + w - 1;
 y1 = y + h - h/3 - 1;
 y2 = y + h - 1;
 y3 = y;

 XSetForeground(display, theGC, textColour);
 XDrawLine(display, theWindow, theGC, x1, y1, x2, y2);
 XDrawLine(display, theWindow, theGC, x2, y2, x3, y3);
}


void PaintCross(GC theGC, Window theWindow,
		int x, int y, unsigned int w, unsigned int h)
{
 XSetForeground(display, theGC, strikeColour);
 XDrawLine(display, theWindow, theGC, x, y,     x + w, y + w);
 XDrawLine(display, theWindow, theGC, x, y + w, x + w, y);
 XSetForeground(display, theGC, textColour);
}


/*
 * Paint checkbox field, i.e. CHECKBOX type field.
 */
Bool PaintCheckboxField(GC theGC, ViewWindow theViewWindow,
			int indent, int baseline,
			Bool theFieldIsActive, Field* theField)
{
 Bool TotalSuccess = True, checked;
 int font, width, height, x1, y2, lineheight;
#ifdef ARENA_DEBUG
 char Iam[] = "PaintCheckboxField";
#endif


 if (theField == NULL) return False;

 {
  unsigned char theFieldType = theField->type;

  if (theFieldType != CHECKBOX)
    {
#ifdef ARENA_DEBUG
     if (FORM_TRACE)
       Arena_TracePrint(Iam,
			" ERROR! The field type %d isn't proccessable.\n",
			theFieldType);
#endif
     return False;
    }
 }

 checked = theField->flags & CHECKED;
 font = theField->flags & 0x0F;

 width  = theField->width;
 height = theField->height;

 lineheight = ASCENT(font) + 4;

 /* cache absolute position of baseline */
 theField->baseline = baseline;

 /* kludge for fields in different frames */
 if (indent >= 0)
   theField->frame_indent = indent;
  else
   indent = theField->frame_indent;

 /***/

 x1 = indent + theField->x - theViewWindow.rect.x;
 y2 = baseline - lineheight + 6 - theViewWindow.rect.y;

 XSetForeground(display, theGC,
		(theFieldIsActive ? statusColour : windowColour));
 XFillRectangle(display, theViewWindow.win, theGC, x1, y2, width, width);

 if (checked)
   {
    PaintTickMark(theGC, theViewWindow.win, x1 + 3, y2 + 3, width - 6, width - 7);
#if 0
    XSetForeground(display, theGC, windowBottomShadow);
    XFillRectangle(display, theViewWindow.win, theGC, x1 + 3, y2 + 3, width - 6, width - 6);
#endif
    DrawInSet(theViewWindow.win, theGC, x1, y2, width, width);
   }
  else
   DrawOutSet(theViewWindow.win, theGC, x1, y2, width, width);

 XSetForeground(display, theGC, textColour);

 return TotalSuccess;
}


/*
 * Paint button field, i.e. RESETBUTTON or SUBMITBUTTON type field.
 */
Bool PaintButtonField(GC theGC, ViewWindow theViewWindow,
		      int indent, int baseline,
		      Bool theFieldIsActive, Field* theField)
{
 Bool TotalSuccess = True;
 int font, width, height, x1, y2, lineheight;
#ifdef ARENA_DEBUG
 char Iam[] = "PaintButtonField";
#endif


 if (theField == NULL) return False;

 {
  unsigned char theFieldType = theField->type;

  if (!((theFieldType == SUBMITBUTTON) ||
	(theFieldType == RESETBUTTON)))
    {
#ifdef ARENA_DEBUG
     if (FORM_TRACE)
       Arena_TracePrint(Iam,
			" ERROR! The field type %d isn't proccessable.\n",
			theFieldType);
#endif
     return False;
    }
 }

 font = theField->flags & 0x0F;

 width  = theField->width;
 height = theField->height;

 lineheight = ASCENT(font) + 4;

 /* cache absolute position of baseline */
 theField->baseline = baseline;

 /* kludge for fields in different frames */
 if (indent >= 0)
   theField->frame_indent = indent;
  else
   indent = theField->frame_indent;

 /***/

 if (theField->buflen > 0)
   width = theField->width =
                     WIDTH(font, theField->value, Arena_StrLen(theField->value)) + 8;

 x1 = indent + theField->x - theField->x_indent - theViewWindow.rect.x;;
 y2 = baseline - lineheight + 2 - theViewWindow.rect.y;

 ClipIntersection(theGC, x1, y2, width, height);
 XSetForeground(display,
		theGC, (theFieldIsActive ? statusColour : windowColour));
 XFillRectangle(display, theViewWindow.win, theGC, x1, y2, width, height);
 DrawOutSet(theViewWindow.win, theGC, x1, y2, width, height);
 ClipIntersection(theGC, x1 + 2, y2, width - 4, height);

 if (theField->buflen > 0)
   {
    SetFont(theGC, font);
    XSetForeground(display, theGC, textColour);
    XDrawString(display, theViewWindow.win, theGC, x1 + 4, baseline,
		theField->value, Arena_StrLen(theField->value));
   }

 XSetForeground(display, theGC, textColour);
 XSetClipMask(display, theGC, None);

 return TotalSuccess;
}


/*
 * Paint form field.
 */
Bool PaintField(GC theGC, ViewWindow theViewWindow,
		int indent, int baseline, Field* field)
{
 extern Field* focus;

 Bool active, TotalSuccess = True;
#ifdef ARENA_DEBUG
 char Iam[] = "PaintField";
#endif


 if (field == NULL) return False;

 if (baseline < 0)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " baseline has a negative value.\n");
#endif
    return False;
   }

 active = False;
 if (focus)
   if (focus == field)
     active = True;

 switch (field->type)
   {
    case PASSWD:
    case TEXTAREA:
    case TEXTFIELD:
      TotalSuccess = PaintTextField(theGC, theViewWindow,
				    indent, baseline, active, field);
      break;

    case OPTIONLIST:
      TotalSuccess = PaintSelectField(theGC, theViewWindow,
				      indent, baseline, active, field);
      break;

    case RADIOBUTTON:
      TotalSuccess = PaintRadioButton(theGC, theViewWindow,
				      indent, baseline, active, field);
      break;

    case CHECKBOX:
      TotalSuccess = PaintCheckboxField(theGC, theViewWindow,
					indent, baseline, active, field);
      break;

    case SUBMITBUTTON:
    case RESETBUTTON:
      TotalSuccess = PaintButtonField(theGC, theViewWindow, indent,
				      baseline, active, field);
      break;

    case HIDDEN:
#ifdef ARENA_DEBUG	/* QingLong.05-02-97 */
      if (FORM_TRACE) 
	Arena_TracePrint(Iam, " ERROR! Painting HIDDEN field.\n");
      TotalSuccess = False;
      break;
#endif
    default:
#ifdef ARENA_DEBUG	/* QingLong.05-02-97 */
      if (FORM_TRACE)
	Arena_TracePrint(Iam,
			 " unexpected form field type %i.\n", field->type);
#endif
      TotalSuccess = False;
      break;
   }

 return TotalSuccess;
}


/*
 * Paint the form --- paint all it's (visible) fields.
 */
Bool PaintForm(GC theGC, ViewWindow theViewWindow,
	       Form* theForm)
{
 Bool success, TotalSuccess = True;
 Field* field_i;

 if (theForm == NULL) return False;

 for (field_i = theForm->fields; field_i; field_i = field_i->next)
   {
    if (field_i->type != HIDDEN)
      if (field_i->baseline >= 0)
	{
         success = PaintField(theGC, theViewWindow,
			      -1, field_i->baseline, field_i);
	 if (TotalSuccess) TotalSuccess = success;
	}
   }

 return TotalSuccess;
}


/*
 * QingLong.29-01-97: hacked strongly (totally rewritten).
 */
char *WWWEncode(char *raw)
{
 char *raw_i = raw, *encoded, *j;
 int i, rlen, encoded_length;

 if (raw == NULL) return NULL;
 if ((*raw) == '\0') return NULL;

 rlen = Arena_StrLen(raw);
 encoded_length = 3 * rlen;	/* max size = size * 3 */
 encoded = (char *)Arena_MAlloc(encoded_length + 1, False);	/* Consider trailing zero */

 j = encoded;

 for(i = rlen; i--; raw_i++)
   {
    if (*raw_i == ' ')
      *j++ = '+';	/* QingLong.28-01-97: May be better "%20" ? */
     else
      if ((*raw_i == '-') ||
	  (*raw_i == '_') ||
	  ((*raw_i >= '0') && (*raw_i <= '9')) ||
	  ((*raw_i >= 'A') && (*raw_i <= 'Z')) ||
	  ((*raw_i >= 'a') && (*raw_i <= 'z')))
        *j++ = *raw_i;
       else
	 {
#ifdef HAVE_SNPRINTF
	  snprintf(j, 3+1, "%%%02x", (unsigned char)*raw_i);
# else
          sprintf(j, "%%%02x", (unsigned char)*raw_i);
#endif
	  j += 3;
	 }
   };
 *j = 0;

 /* Find out the real encoded length (not max one) with trailing zero */
 encoded_length = Arena_StrLen(encoded) + 1;

 j = (char *)Arena_MAlloc(encoded_length, False);

 /* copy the string again to save the extra space allocated */
 strncpy(j, encoded, encoded_length);
 Free(encoded);

#ifdef ARENA_DEBUG	/* QingLong.29-01-97 */
 if (VERBOSE_TRACE) Arena_TracePrint("WWWEncode", " encoded:\n\t\"%s\"\n", j);
#endif

 return (j);
}


/*
 * Encode the given string and append to the buffer.
 */
Bool AppendStringEncoded(EditorBuffer* theBuffer, char* theString)
{
 Bool TotalSuccess = False;
 char* encoded = NULL;

 if ((theBuffer == NULL) || (theString == NULL)) return False;
 if ((*theString) == '\0') return True;

 if ((encoded = WWWEncode(theString)))
   {
    if (AppendString(theBuffer, encoded)) TotalSuccess = True;
    Free(encoded);
   }

 return TotalSuccess;
}


/*
 * Encode the given string (of the given length) and append to the buffer.
 */
Bool AppendNStringEncoded(EditorBuffer* theBuffer,
			  char* theString, int theStringLength)
{
 Bool TotalSuccess = False;
 char* encode_buffer = NULL;

 if ((theBuffer == NULL) || (theString == NULL) || (theStringLength < 0))
   return False;

 if ((theStringLength == 0) || ((*theString) == '\0'))
   return True;


 if ((encode_buffer = strndup(theString, theStringLength)))
   {
#if 1	/* QL.08-05-97: Do we need to ensure zero ending? */
    encode_buffer[theStringLength] = '\0';
#endif
    if (AppendStringEncoded(theBuffer, encode_buffer)) TotalSuccess = True;
    Free(encode_buffer);
   }

 return TotalSuccess;
}


void SubmitForm(Form* form,
		FormSubmitMethod method, char *action, int alen, int type,
		char *name, int nlen, char *value, int vlen,
		Image* image, int dx, int dy, char *bufpos)
{
#define SUBMITFORM_ITOABUFF_LENGTH 32
 EditorBuffer* buffer;
 char* query;
 char itoabuff[SUBMITFORM_ITOABUFF_LENGTH];
 Field *field, *field_i;
 Option* option;
 int first,size;
#ifdef ARENA_DEBUG
 char Iam[] = "SubmitForm";
#endif


 if (form == NULL)
   {
#ifdef ARENA_DEBUG
    if (FORM_TRACE)
      Arena_TracePrint(Iam, " ERROR! Form to submit is NULL.\n");
#endif
    return;
   }

#ifdef ARENA_DEBUG
#  define ARENA_SUBMITFORM_BUFPOS_CUTLENGTH 66
 if (FORM_TRACE && VERBOSE_TRACE)
   {
    char* arenaSubmitFormBufpos = NULL;
    char* arenaSubmitFormAction = NULL;
    char* arenaSubmitFormName   = NULL;
    char* arenaSubmitFormValue  = NULL;

    if (bufpos)
      arenaSubmitFormBufpos = strndup(bufpos,
				      ARENA_SUBMITFORM_BUFPOS_CUTLENGTH);

    if (alen < 0) alen = 0;
    if (nlen < 0) nlen = 0;
    if (vlen < 0) vlen = 0;

    arenaSubmitFormAction = strndup(action, alen);
    arenaSubmitFormName   = strndup(name,   nlen);
    arenaSubmitFormValue  = strndup(value,  vlen);

    Arena_TracePrint(Iam,
		     " (the parameters given)\n"
		     "\tform   = "POINTER_FORMAT",\n"
		     "\tmethod = \"%s\",\n"
		     "\taction = \"%s\", alen = %i,\n"
		     "\ttype   = 0x%x,\n"
		     "\tname   = \"%s\", nlen = %i,\n"
		     "\tvalue  = \"%s\", vlen = %i,\n"
		     "\timage  = "POINTER_FORMAT",\n"
		     "\tdx = %i, dy = %i,\n"
		     "\tbufpos = \"%s\".\n",
		     form,
		     ((method == GET)     ?       "GET" :
		      ((method == POST)    ?     "POST" :
		       ((method == ISINDEX) ? "ISINDEX" :
			((method == MAILTO) ?  "MAILTO" : "???")))),
		     arenaSubmitFormAction, alen,
		     type,
		     arenaSubmitFormName, nlen,
		     arenaSubmitFormValue, vlen,
		     image,
		     dx, dy,
		     arenaSubmitFormBufpos);
    Free(arenaSubmitFormBufpos);
    Free(arenaSubmitFormAction);
    Free(arenaSubmitFormName);
    Free(arenaSubmitFormValue);
   }
#  undef ARENA_SUBMITFORM_BUFPOS_CUTLENGTH
#endif

 /*
  * If the method is MAILTO, that means that the submission is internal,
  * the control should be passed to MTSubmitForm,
  * which would get the proper internal action.
  */
 if (method == MAILTO)
   {
    MTSubmitForm(form, method, action, alen, type,
		 name, nlen, value, vlen,
		 image, dx, dy, bufpos);
    return;
   }

 buffer = CreateBuffer();

 first = ((method == GET) || (method == ISINDEX));

 if (image != NULL)
   {
    if (first) AppendChar(buffer, '?');
    AppendnChar(buffer, name, nlen);
    AppendString(buffer, ".x=");
#ifdef HAVE_SNPRINTF
    snprintf(itoabuff, SUBMITFORM_ITOABUFF_LENGTH, "%d&", dx);
# else
    sprintf(itoabuff, "%d&", dx);
#endif
    AppendString(buffer, itoabuff);
    AppendnChar(buffer, name, nlen);
    AppendString(buffer,".y=");
#ifdef HAVE_SNPRINTF
    snprintf(itoabuff, SUBMITFORM_ITOABUFF_LENGTH, "%d", dy);
# else
    sprintf(itoabuff, "%d", dy);
#endif
    AppendString(buffer,itoabuff);
    field = FindField(form, type, name, nlen);
    first = 0;
   }
  else
   {
    if (nlen)
      {
       if (vlen)
	 {
	  if (first) AppendChar(buffer, '?');

	  if (form->method != ISINDEX)     
	    {
	     AppendnChar(buffer, name, nlen); /* don't name= with isindex   */
	     AppendChar(buffer, '=');         /* just ?value+value etc...   */
	    }

	  AppendNStringEncoded(buffer, value, vlen);
	  field = FindField(form, type, name, nlen);
	  first = 0;
	 }
        else
	 {
	  if(first) AppendChar(buffer, '?');
	  AppendnChar(buffer, name, nlen);
	  AppendString(buffer, "=Submit+Query");
	  field = FindField(form, type, name, nlen);
	  first = 0;
	 }
      }
     else
      {
       first = ((method == GET) || (method == ISINDEX)) ? 1 : 2;
       field = FindField(form, type, bufpos, 7);
      }
   }

 for (field_i = form->fields; field_i != NULL; field_i = field_i->next)
   {
    if (field_i != field)
      {
       switch (field_i->type)
	 {
	  case SUBMITBUTTON:
	  case RESETBUTTON:
	    break;

	  case RADIOBUTTON:
	    if (!(field_i->flags & CHECKED)) break;
	  case HIDDEN:
	  case TEXTAREA:
	  case TEXTFIELD:
	    {
	     char* field_value = field_i->value;

	     if (field_value)
	       {
		if (*field_value != '\0')
		  {
		   if (first)
		     {
		      if (first == 1) AppendChar(buffer, '?');
		      first = 0;
		     }
		   else
		     AppendChar(buffer, '&');

		   AppendNStringEncoded(buffer,
					field_i->name, field_i->nlen);
		   AppendChar(buffer, '=');
		   AppendNStringEncoded(buffer,
					field_i->value, field_i->buflen);
		  }
	       }
	    }
	    break;

	  case CHECKBOX:
	    if (field_i->flags & CHECKED)
	      {
	       if (first)
		 {
		  if (first == 1) AppendChar(buffer, '?');
		  first = 0;
		 }
	        else
		 AppendChar(buffer, '&');

	       AppendNStringEncoded(buffer, field_i->name, field_i->nlen);
	       AppendChar(buffer, '=');
	       if (field_i->value)
		 if ((*(field_i->value)) != '\0')
		   {
		    AppendStringEncoded(buffer, field_i->value);
		    break;
		   }

	       AppendString(buffer, "on");
	      };
	    break; 

	  case OPTIONLIST:
	    for (option = field_i->options; option; option = option->next)
	      {
	       if (option->flags & OPTION_CHECKED)
		 {
		  char* option_value = option->value ?
		                       option->value : option->label;
		  Bool PrintThisField = False, theOptionValue = False;

		  if (option_value)
		    if (*option_value != '\0')
		      theOptionValue = True;

		  if (field_i->flags & MULTIPLE)
		    PrintThisField = True;
		   else
		    {
		     if (theOptionValue) PrintThisField = True;
		    }

		  if (PrintThisField)
		    {
		     if (first)
		       {
			if (first == 1) AppendChar(buffer, '?');
			first = 0;
		       }
		     else
		       AppendChar(buffer, '&');
		    }

		  if (theOptionValue)
		    {
		     AppendNStringEncoded(buffer,
					  field_i->name, field_i->nlen);
		     AppendChar(buffer, '=');

		     if ((size = Arena_StrLen(option_value)))
		       if (option_value[size - 1] == ' ')
			 /* Eat the last blank */
			 option_value[size - 1] = 0;

		     AppendStringEncoded(buffer, option_value);
		    }
		  if (!(field_i->flags & MULTIPLE)) break;
		 }
	      }
	    break;
	 };
      };
   }

 switch(method)
   {
    case ISINDEX:
    case GET:
      if (action)
	InsertnChar(buffer, 0, action, alen);
       else
	 {
	  char* exp_href = ExpandHref(CurrentDoc->href);

	  InsertString(buffer, 0, exp_href);
	  Free(exp_href);
	 }

      query = Buffer2Str(buffer);
#ifdef ARENA_DEBUG	/* QingLong.29-01-97 */
      if (FORM_TRACE && VERBOSE_TRACE)
	Arena_TracePrint(Iam, " fetching (GET)\n\t\"%s\"\n", query);
#endif
      OpenDoc(query);
      break;

    case POST:
      {
       char* encoded = Buffer2Str(buffer);

       query = (char *) Arena_CAlloc(alen + 1, sizeof(char), False);
       strncpy(query, action, alen);
       *(query + alen) = 0;
       PostDoc(query, encoded); 
#ifdef ARENA_DEBUG	/* QingLong.29-01-97 */
       if (FORM_TRACE && VERBOSE_TRACE)
	 Arena_TracePrint(Iam, " posting\n\t[%s][%s]\n", query, encoded);
#endif
      }
      break;

    case MAILTO:
#ifdef ARENA_DEBUG
      if (FORM_TRACE)
	Arena_TracePrint(Iam, " ERROR! Called for MAILTO access method.\n");
#endif
      break;
   };
#undef SUBMITFORM_ITOABUFF_LENGTH
}
