;; -*- Emacs-Lisp -*-

(require 'cl)
(require 'pp)

(eval-and-compile (setq load-path (cons ".." (cons "." load-path))))
(provide 'reposer)
(eval-and-compile (setq load-path (cdr (cdr load-path))))

;; Turn of assertions in compiled code.
(eval-and-compile
  (setq cl-optimize-speed 3)
  (setq cl-optimize-safety 1)
  )

;; Begin

(defconst *definition-state* nil
  "List of all the names of variables containing state from the
definition file so that operations may be performed on everything in
the definition file.")
(defconst *definition-attrs* nil
  "List of attributes for sharing indices.")
(defconst *all-objects* nil)
(defconst *output-files* nil
  "List of lists (NAME BUFFER) used during output of headers.")
(defconst *output-prefix* nil
  "Prefix used for outputing header files.")
(defconst *library-id* nil
  "Identifier of this library.")
(defconst *cpp-extension* "c")

;; This defines several functions and one macro.  The indirection makes
;; it a little bit confusing to read.  It defines the macro DEFDNAME,
;; a function DEFDNAME*, MAKE-DNAME, and a setter and getter for each arg.
(eval-and-compile
(defmacro attr-index (attr)
  `(- (length *definition-attrs*) (length (memq ,attr *definition-attrs*))))

(defmacro defastmacro(dtype args attrs)
  "Defines a macro named DEFDTYPE for defining various AST properties."
  (let ((def-macr  (intern (format "def%s" dtype)))
	(def-func  (intern (format "def%s*" dtype)))
	(make-func (intern (format "make-%s" dtype)))
	(state     (intern (format "*%s-defs*" dtype)))
	(exprs nil))
    (if (not *definition-attrs*)
	(setq *definition-attrs* '(menunode menuline)))
    (let ((fields (append args attrs)))
      (while fields
	(if (not (memq (car fields) *definition-attrs*))
	    (setq *definition-attrs* (append *definition-attrs* (list (car fields)))))
	(setq fields (cdr fields))
	)
      )
    ;; Add it to *definition-state*
    (setq *definition-state* (cons state *definition-state*))
    ;; DEFCONST it
    (setq exprs (cons (list 'defconst state (quote nil)) exprs))
    ;; DEFMACRO DEFDTYPE
    (setq exprs (cons (list 'defmacro
			    def-macr
			    args
			    (append (list 'list (list 'quote def-func))
				    (mapcar (function (lambda (x)
							(list 'list (list 'quote 'quote) x)
							)
						      )
					    args)
				    )
			    )
		      exprs
		      )
	  )
    ;; DEFUN DEFDTYPE*
    (setq exprs (cons (list 'defun
			    def-func
			    args
			    (list 'setq
				  state
				  (list 'cons
					(cons make-func args)
					state
					)
				  )
			    )
		      exprs
		      )
	  )
    ;; MAKE-DTYPE
    (setq exprs (cons (list 'defun
			    make-func
			    args
			    (list 'let (list (list 'it (list 'make-vector (length *definition-attrs*) nil)))
				  (if args
				      (cons 'progn (mapcar
						    (function
						     (lambda (x)
						       (list 'aset 'it (attr-index x) x)
						       )
						     )
						    args
						    )
					    )
				    )
				  (if attrs
				      (cons 'progn (mapcar
						    (function
						     (lambda (x)
						       (list 'aset 'it (attr-index x) nil)
						       )
						     )
						    attrs
						    )
					    )
				    )
				  (if (memq 'menu args)
				      (list 'progn
					    (list 'aset 'it (attr-index 'menunode) (list 'function (intern (format "%s-menunode" dtype))))
					    (list 'aset 'it (attr-index 'menuline) (list 'function (intern (format "%s-menuline" dtype))))
					    )
				    )
				  (list 'cons (list 'quote dtype) 'it)
				  )
			    )
		      exprs
		      )
	  )
    ;; Add the fake arguments:
    (if (memq 'menu args)
	(setq attrs (append (list 'menunode 'menuline) attrs)))
    (setq args (append args attrs))
    (while args
      (let* ((thearg (car args))
	     (arg-set (intern (format "%s-%s-set" dtype thearg)))
	     (arg-get (intern (format "%s-%s-get" dtype thearg))))
	;; DTYPE-ARG-GET
	(setq exprs (cons (list 'defmacro
				(intern (format "%s-%s-get" dtype thearg))
				'(obj)
				(list 'list
				      (list 'quote 'aref)
				      (list 'list (list 'quote 'cdr) 'obj)
				      (attr-index thearg))
				)
			  exprs
			  )
	      )
	;; DTYPE-ARG-SET
	(setq exprs (cons (list 'defmacro
				(intern (format "%s-%s-set" dtype thearg))
				'(obj val)
				(list 'list
				      (list 'quote 'aset)
				      (list 'list (list 'quote 'cdr) 'obj)
				      (attr-index thearg)
				      'val)
				)
			  exprs
			  )
	      )
	)
      (setq args (cdr args))
      )
    ;; To see what it's generating uncomment the next 2 lines.
    ;;(setq message-log-max t)
    ;;(mapcar (function pp) exprs)
    (cons 'progn exprs)
    )
  )


;; This is, as the name suggests, really bogus.  Basically, each DEFASTMACRO
;; call adds to the list *definition-state*.  To compile it, however, it has
;; to be done at compile time, so this macro gets evaluated when being compiled
;; and clears the list.  Then the DEFASTMACRO calls are made, and then DEFCDS
;; is called to define CLEAR-DEFINITION-STATE which resets the list to the
;; compile-time computed value of *definition-state*, it would otherwise be
;; empty when running compiled code.
(defmacro bogus ()
  (setq *definition-state* nil)
  (setq *definition-attrs* nil)
  )

  (bogus)

;; Each DEFASTMACRO statement defines a directive for the definition
;; file along with it's argument names.
(defastmacro sertype      (name fields transients)  ())

(defmacro defcds ()
  (let ((exprs nil))
    (setq exprs (list (list 'defun 'clear-definition-state nil
			    '(setq *all-objects* nil)
			    (list 'setq '*definition-state* (list 'quote *definition-state*))
			    (list 'setq '*definition-attrs* (list 'quote *definition-attrs*))
			    '(mapcar (function (lambda (x) (set x nil))) *definition-state*)
			    )

		      )
	  )
    (mapcar
     (function
      (lambda (x)
	(setq exprs (cons (list 'defmacro
				(intern (format "obj-%s-get" x))
				'(obj)
				(list 'list
				      (list 'quote 'aref)
				      (list 'list (list 'quote 'cdr) 'obj)
				      (attr-index x))
				)
			  exprs
			  )
	      )
	(setq exprs (cons (list 'defmacro
				(intern (format "obj-%s-set" x))
				'(obj val)
				(list 'list
				      (list 'quote 'aset)
				      (list 'list (list 'quote 'cdr) 'obj)
				      (attr-index x)
				      'val)
				)
			  exprs
			  )
	      )
	(let ((get (intern (format "obj-%s-get" x))))
	  (setq exprs (cons (list 'defun
				  (intern (format "obj-%s-eq" x))
				  '(val olist)
				  `(let ((ret nil))
				     (while (and (not ret) olist)
				       (if (eq val (,get (car olist)))
					   (setq ret (car olist))
					 )
				       (setq olist (cdr olist))
				       )
				     ret
				     )
				  )
			    exprs
			    )
		)
	  )
	)
      )
     *definition-attrs*
     )
    ;;(setq message-log-max t)
    ;;(mapcar (function pp) exprs)
    (cons 'progn exprs)
    )
  )

(defcds)
)
;; Entry Points

(defun generate-ser-noargs ()
  (interactive)
  (generate-ser "repo.ser" "repo" 1))

(defun generate-ser (input-file output-prefix unique-id)
  ;(interactive "finput: \nsoutput: \nsid: ")
  (let ((make-backup-files nil)
	(executing-kbd-macro t))
    (clear-definition-state)

    (load-file input-file)

    (do-it output-prefix unique-id)
    )
  )

(defun do-it(output-prefix unique-id)
  (setq *output-files* nil)
  (setq *library-id* unique-id)
  (setq *output-prefix* output-prefix)

  (if (or (<= *library-id* 0)
	  (>= *library-id* 256))
      (error "Library-id is out of range"))

  (if (or (not *sertype-defs*)
	  (> (length *sertype-defs*) 24))
      (error "0 < count <= 24"))

  (unwind-protect
      (progn

	(generate-db)

	(mapcar (function (lambda (x) (output-finish-file x))) *output-files*)
	)
    (mapcar (function (lambda (x) (kill-buffer (cadr x)))) *output-files*)
    )
  )

(defun output-header-file (name)
  (output-file (format "%s.h" name) 'c-header *output-prefix*))

(defun output-source-file (name)
  (output-file (format "%s.%s" name *cpp-extension*) 'c *output-prefix*))

(defun output-source-include-file (name)
  (output-file (format "%s.%si" name *cpp-extension*) 'c *output-prefix*))

(defun output-plain-file (name)
  (output-file (format "%s" name) 'plain ""))

(defun output-file (name type prefix)
  (let* ((name (format "%s%s" prefix name))
	 (it (assoc name *output-files*)))
    (if it
	(set-buffer (cadr it))
      (let ((nbuf (get-buffer-create (generate-new-buffer-name name))))
	(setq *output-files* (cons (list name nbuf type) *output-files*))
	(set-buffer nbuf)
	)
      )
    )
  )

(defun output-finish-file (file)
  (let ((name (car file))
	(buf (cadr file))
	(type (caddr file)))
    (set-buffer buf)
    (cond ((eq type 'c)
	   (output-to-c name nil))
	  ((eq type 'c-header)
	   (output-to-c name t))
	  )
    (write-file-if-different buf name)
    )
  )

(defun output-to-c (name is-header)
  (goto-char (point-min))
  (insert "/* -*-Mode: C;-*-
 * Xdelta - A binary delta library
 * Copyright (C) 1997, 1998  Josh MacDonald
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Josh MacDonald <jmacd@CS.Berkeley.EDU>
 *
 * This file was AUTOMATICALLY GENERATED using:
 *
 * $Id: reposer.el 1.22 Thu, 01 Oct 1998 04:56:35 -0700 jmacd $
 */

")

; (insert "/* -*- Mode: C; -*-
; * " name ":
; *
; * Copyright (C) 1998, U.C. Regents.
; * All Rights Reserved.
; *
; * This file is AUTOMATICALLY GENERATED.
; * Author: Josh MacDonald
; */

;")
  (if is-header
      (let ((cppname (string-replace-regexp (upcase name) "[-./]" "_")))
	(insert "#ifndef _" cppname "_\n")
	(insert "#define _" cppname "_\n\n")
	(insert "#include \"serializeio.h\"\n")
	(goto-char (point-max))
	(insert "\n#endif /* _" cppname "_ */\n\n")
	)
    (insert "#include \"" *output-prefix* "ser.h\"\n")
    )
  )

(defun string-replace-regexp (str regexp to-string)
  "Result of replacing all occurrences in STR of REGEXP by TO-STRING.  The
replacement is as for replace-regexp."
  (let ((work (get-buffer-create "*string-tmp*")))
    (save-excursion
      (set-buffer work)
      (erase-buffer)
      (insert str)
      (beginning-of-buffer)
      (while (re-search-forward regexp nil t)
	(replace-match to-string nil nil))
      (buffer-string))))

(defun write-file-if-different (buf filename)
  (save-excursion
    (if (not (file-exists-p filename))
	(write-file filename)
      (set-buffer buf)
      (let ((old (get-buffer-create (generate-new-buffer-name filename)))
	    (bmin (point-min))
	    (bmax (point-max)))
	(unwind-protect
	    (progn
	      (set-buffer old)
	      (insert-file filename)
	      (let ((omin (point-min))
		    (omax (point-max))
		    (case-fold-search nil))
		(if (= 0 (compare-buffer-substrings old omin omax buf bmin bmax))
		    (message "Output file %s is unchanged." filename)
		  (set-buffer buf)
		  (write-file filename)
		  )
		)
	      )
	  (kill-buffer old)
	  )
	)
      )
    )
  )


(defun format-comlist (func l)
  (let ((x ""))
    (while l
      (setq x (concat x (funcall func (car l))))
      (if (cdr l)
	  (setq x (concat x ", ")))
      (setq l (cdr l))
      )
    x
    )
  )

(defun format-numbered-comlist (func l)
  (let ((x "")
	(n 0))
    (while l
      (setq x (concat x (funcall func (car l) n)))
      (setq n (+ n 1))
      (if (cdr l)
	  (setq x (concat x ", ")))
      (setq l (cdr l))
      )
    x
    )
  )

(defun capitalize1(s)
  (let ((work (get-buffer-create "*string-tmp*")))
    (save-excursion
      (set-buffer work)
      (erase-buffer)
      (insert (format "%s" s))
      (upcase-region (point-min) (+ (point-min) 1))
      (buffer-substring-no-properties (point-min) (point-max))
      )
    )
  )

(defun upcase-string (s)
  (let ((work (get-buffer-create "*string-tmp*")))
    (save-excursion
      (set-buffer work)
      (erase-buffer)
      (insert (format "%s" s))
      (upcase-region (point-min) (point-max))
      (buffer-substring-no-properties (point-min) (point-max))
      )
    )
  )

(defun downcase-string (s)
  (let ((work (get-buffer-create "*string-tmp*")))
    (save-excursion
      (set-buffer work)
      (erase-buffer)
      (insert (format "%s" s))
      (downcase-region (point-min) (point-max))
      (buffer-substring-no-properties (point-min) (point-max))
      )
    )
  )

;; HERE IT IS

(defconst *available-dt-id* 0)
(defun next-dt-id ()
  (let ((next *available-dt-id*))
    (setq *available-dt-id* (+ next 1))
    next
    )
  )

(defvar decl-point nil)
(defvar statdecl-point nil)
(defvar init-point nil)

(defun generate-db ()

  (output-header-file "ser")

  (insert "/* Init */\n")
  (insert "gboolean " *output-prefix* "ser_init (void);\n\n")

  (insert "/* Serial Types */\n\n")
  (insert (format "enum _Serial%sType {\n" (capitalize1 *output-prefix*)))
  (insert (format-comlist
	   (function
	    (lambda (x)
	      (format "\n  ST_%s = (1<<(%d+SER_LIBRARY_OFFSET_BITS))+%d" (sertype-name-get x) (next-dt-id) *library-id*))) *sertype-defs*))
  (insert "\n};\n\n")

  (setq decl-point (point-marker))

  (insert "\n\n")

  (output-source-file "ser")

  (insert "/* Static decls */\n\n")
  (setq statdecl-point (point-marker))
  (insert "\n\n")

  (insert "/* Init */\n")
  (insert "gboolean " *output-prefix* "ser_init (void) {\n")
  (setq init-point (point-marker))
  (insert "  return TRUE;\n")
  (insert "};\n\n")

  (mapcar (function generate-db-entry) *sertype-defs*)

  )

(defun generate-db-entry (entry)
  (let ((ent-upcase   (sertype-name-get entry))
	(ent-downcase (downcase-string (sertype-name-get entry))))

    (output-header-file "ser")

    ;; The typedef, structure, and declarations.

    (save-excursion
      (goto-char decl-point)
      (insert (format "typedef struct _Serial%s Serial%s;\n" ent-upcase ent-upcase))
      )

    (insert (format "struct _Serial%s {\n" ent-upcase))

    (apply (function insert)
	   (mapcar (function
		    (lambda (x)
		      (format "  %s;\n" x)))
		   (entry-typename-pairs entry nil)))

    (apply (function insert)
	   (mapcar (function
		    (lambda (x)
		      (format "  %s;\n" x)))
		   (sertype-transients-get entry)))

    (insert "};\n\n")

    (insert (format "SerialStatus unserialize_%s (SerialSource *source, Serial%s**);\n" ent-downcase ent-upcase))
    (insert (format "SerialStatus serialize_%s (SerialSink *sink%s);\n\n" ent-downcase (entry-arglist t entry)))
    (insert (format "SerialStatus serialize_%s_obj (SerialSink *sink, Serial%s* obj);\n\n" ent-downcase ent-upcase))

    (output-source-file "ser")

    ;; The init entry

    (save-excursion
      (goto-char init-point)
      (insert (format "  serializeio_initialize_type (ST_%s, &unserialize_%s_internal);\n" ent-upcase ent-downcase))
      )

    (save-excursion
      (goto-char statdecl-point)
      (insert (format "static SerialStatus\nunserialize_%s_internal_noalloc (SerialSource *source, Serial%s* );\n" ent-downcase ent-upcase))
      (insert (format "static SerialStatus\nunserialize_%s_internal (SerialSource *source, Serial%s** );\n" ent-downcase ent-upcase))
      (insert (format "static SerialStatus\nserialize_%s_internal (SerialSink *sink%s);\n" ent-downcase (entry-arglist t entry)))
      (insert (format "static guint\ncount_%s (%s);\n" ent-downcase (entry-arglist nil entry)))
      )

    ;; Count code

    (insert (format "guint\ncount_%s (%s) {\n" ent-downcase (entry-arglist nil entry)))
    (insert (format "  guint size = sizeof (Serial%s);\n" ent-upcase))
    (apply (function insert)
	   (mapcar (function (lambda (x) (concat
					  (format "  ALIGN_8 (size);\n")
					  (entry-count-field entry x (format "%s" (car x)) "  "))))
		   (sertype-fields-get entry)))
    (insert (format "  ALIGN_8 (size);\n"))
    (insert (format "  return size;\n"))
    (insert (format "}\n\n"))

    ;; Serialize code

    (insert (format "static SerialStatus\nserialize_%s_internal (SerialSink *sink%s)\n" ent-downcase (entry-arglist t entry)))
    (insert (format "{\n"))
    (insert (format "  sink->status = SerialSuccess;\n"))

    (apply (function insert)
	   (mapcar (function (lambda (x) (entry-serialize-field entry x (format "%s" (car x)) "  ")))
		   (sertype-fields-get entry)))

    (if (sertype-fields-get entry)
	(insert (format "bail:\n")))
    (insert (format "  return sink->status;\n"))
    (insert (format "}\n"))

    ;; External Serialize code

    (insert (format "SerialStatus\nserialize_%s (SerialSink *sink%s)\n" ent-downcase (entry-arglist t entry)))
    (insert (format "{\n"))

    (insert (format "  sink->status = SerialSuccess;\n"))

    (insert (format "  if (! (* sink->sink_type) (sink, ST_%s, count_%s (%s))) { sink->status = SerialIncorrectType; goto bail; }\n" ent-upcase ent-downcase (entry-plist nil nil "" entry)))

    (insert (format "  if (serialize_%s_internal (sink%s) != SerialSuccess) goto bail;\n" ent-downcase (entry-plist nil t "" entry)))

    (insert (format "  return (* sink->sink_quantum) (sink);\n"))
    (insert (format "bail:\n"))
    (insert (format "  return sink->status;\n"))
    (insert (format "}\n\n"))

    ;; External serialize_obj

    (insert (format "SerialStatus\nserialize_%s_obj (SerialSink *sink, Serial%s* obj) {\n\n" ent-downcase ent-upcase))
    (insert (format "  return serialize_%s (sink%s);\n" ent-downcase (entry-plist t t "obj->" entry)))
    (insert (format "}\n"))

    ;; Unserialize code

    (insert (format "SerialStatus\nunserialize_%s_internal_noalloc (SerialSource *source, Serial%s* result)\n" ent-downcase ent-upcase))
    (insert (format "{\n"))

    (insert (format "  source->status = SerialSuccess;\n"))

    (apply (function insert)
	   (mapcar (function (lambda (x) (entry-unserialize-field entry x (format "result->%s" (car x)) "  ")))
		   (sertype-fields-get entry)))

    (if (sertype-fields-get entry)
	(insert (format "bail:\n")))
    (insert (format "  return source->status;\n"))
    (insert (format "}\n\n"))


    (insert (format "SerialStatus\nunserialize_%s_internal (SerialSource *source, Serial%s** result)\n" ent-downcase ent-upcase))
    (insert (format "{\n"))

    (insert (format "  Serial%s* unser;\n" ent-upcase))
    (insert (format "  (*result) = NULL;\n"))
    (insert (format "  source->status = SerialSuccess;\n"))
    (insert (format "  unser = (* source->source_alloc) (source, sizeof (Serial%s));\n" ent-upcase))
    (insert (format "  if (! unser) goto bail;\n"))

    (insert (format "  if (unserialize_%s_internal_noalloc (source, unser) != SerialSuccess) goto bail;\n" ent-downcase))

    (insert (format "  (*result) = unser;\n"))
    (insert (format "bail:\n"))
    (insert (format "  return source->status;\n"))
    (insert (format "}\n\n"))

    ;; External unserialize

    (insert (format "SerialStatus\nunserialize_%s (SerialSource *source, Serial%s** result)\n"  ent-downcase ent-upcase))
    (insert (format "{\n"))

    (insert (format "  source->status = SerialSuccess;\n"))

    (insert (format "  if ( (* source->source_type) (source) != ST_%s) { source->status = SerialIncorrectType; goto bail; }\n" ent-upcase))

    (insert (format "  if (unserialize_%s_internal (source, result) != SerialSuccess) goto bail;\n" ent-downcase))
    (insert (format "  source->alloc = NULL;\n"))
    (insert (format "  (* source->source_reset) (source);\n"))

    (insert (format "bail:\n"))
    (insert (format "  return source->status;\n"))

    (insert (format "}\n\n"))

    )
  )

(defun entry-typename-pairs (entry pass-by-ref)
  (let ((pairs nil)
	(fields (sertype-fields-get entry)))
    (while fields
      (let ((field (car fields)))
	(when (or (equal (cadr field) 'bytes)
		  (and (consp (cadr field)) (equal (caadr field) 'array)))
	  (setq pairs (cons (format "guint32 %s_len" (car field)) pairs))
	  )
	(setq pairs (cons (field-decl field pass-by-ref) pairs))
	)
      (setq fields (cdr fields))
      )
    (nreverse pairs)
    )
  )

(defun entry-param-names (prefix entry need_pbr)
  (let ((pairs nil)
	(fields (sertype-fields-get entry)))
    (while fields
      (let ((field (car fields)))
	(when (or (equal (cadr field) 'bytes)
		  (and (consp (cadr field)) (equal (caadr field) 'array)))
	  (setq pairs (cons (format "%s%s_len" prefix (car field)) pairs))
	  )
	(setq pairs (cons (format "%s%s%s" (if (and need_pbr (needs-ref field)) "&" "") prefix (car field)) pairs))
	)
      (setq fields (cdr fields))
      )
    (nreverse pairs)
    )
  )

(defun field-ctype (field)
  (cond ((equal (cadr field) 'string)
	 "const gchar*")
	((equal (cadr field) 'uint)
	 "guint32")
	((equal (cadr field) 'uint32)
	 "guint32")
	((equal (cadr field) 'uint16)
	 "guint16")
	((equal (cadr field) 'uint8)
	 "guint8")
	((equal (cadr field) 'boolean)
	 "gboolean")
	((equal (cadr field) 'bytes)
	 "const guint8*")
	((member (cadr field) (mapcar (lambda (x) (sertype-name-get x)) *sertype-defs*))
	 (format "Serial%s" (cadr field)))
	((equal (car (cadr field)) 'bytes)
	 "const guint8*")
	((member (car (cadr field)) '(array ptr))
	 (concat (field-ctype (cadr field)) "*"))
	(t (error "unrecognized field type: %s" (cadr field))))
  )

(defun field-decl (field pass-by-ref)
  (if (and (consp (cadr field))
	   (equal (car (cadr field)) 'bytes))
      (format "guint8 %s[%d]" (car field) (cadr (cadr field)))
    (format "%s %s"
	    (cond ((member (cadr field) (mapcar (lambda (x) (sertype-name-get x)) *sertype-defs*))
		   (format "Serial%s%s" (cadr field) (if pass-by-ref "*" "")))
		  ((equal (cadr field) 'string)
		   "const gchar*")
		  ((equal (cadr field) 'uint)
		   "guint32")
		  ((equal (cadr field) 'uint32)
		   "guint32")
		  ((equal (cadr field) 'uint16)
		   "guint16")
		  ((equal (cadr field) 'uint8)
		   "guint8")
		  ((equal (cadr field) 'boolean)
		   "gboolean")
		  ((equal (cadr field) 'bytes)
		   "const guint8*")
		  ((member (car (cadr field)) '(array ptr))
		   (concat (field-ctype (cadr field)) "*"))
		  (t (error "unrecognized field type: %s" (cadr field))))
	    (car field)))
  )

(defun entry-arglist (need_first entry)
  (concat
   (if (and need_first (sertype-fields-get entry)) ", " "")
   (format-comlist (function (lambda (x) x)) (entry-typename-pairs entry t))))

(defun needs-ref (field)
  (member (cadr field) (mapcar (lambda (x) (sertype-name-get x)) *sertype-defs*))
  )

(defun entry-plist (need_pbr need_first prefix entry)
  (concat
   (if (and need_first (sertype-fields-get entry)) ", " "")
   (format-comlist (function (lambda (x) (format "%s" x)))
		   (entry-param-names prefix entry need_pbr))))

(defun entry-unserialize-field (entry field name prefix)
  (cond ((equal (cadr field) 'uint)
	 (format "%sif (! (* source->next_uint) (source, &%s)) goto bail;\n" prefix name))
	((equal (cadr field) 'uint32)
	 (format "%sif (! (* source->next_uint32) (source, &%s)) goto bail;\n" prefix name))
	((equal (cadr field) 'uint16)
	 (format "%sif (! (* source->next_uint16) (source, &%s)) goto bail;\n" prefix name))
	((equal (cadr field) 'uint8)
	 (format "%sif (! (* source->next_uint8) (source, &%s)) goto bail;\n" prefix name))
	((equal (cadr field) 'boolean)
	 (format "%sif (! (* source->next_bool) (source, &%s)) goto bail;\n" prefix name))
	((equal (cadr field) 'string)
	 (format "%sif (! (* source->next_string) (source, &%s)) goto bail;\n" prefix name))
	((equal (cadr field) 'bytes)
	 (format "%sif (! (* source->next_bytes) (source, &%s, &%s_len)) goto bail;\n" prefix name name))
	((member (cadr field) (mapcar (lambda (x) (sertype-name-get x)) *sertype-defs*))
	 (format "%sif (unserialize_%s_internal_noalloc (source, &%s) != SerialSuccess) goto bail;\n" prefix (downcase-string (cadr field)) name))
	((and (equal (car (cadr field)) 'ptr)
	      (member (cadr (cadr field)) (mapcar (lambda (x) (sertype-name-get x)) *sertype-defs*)))
	 (format "%sif (unserialize_%s_internal (source, &%s) != SerialSuccess) goto bail;\n" prefix (downcase-string (cadr (cadr field))) name))
	((equal (car (cadr field)) 'bytes)
	 (format "%sif (! (* source->next_bytes_known) (source, %s, %d)) goto bail;\n" prefix name (cadr (cadr field))))
	((equal (car (cadr field)) 'array)
	 (format "%s{
%s  gint i;
%s  if (! (* source->next_uint) (source, &%s_len)) goto bail;
%s  if (! (%s = (* source->source_alloc) (source, sizeof (%s) * %s_len))) goto bail;
%s  for (i = 0; i < %s_len; i += 1)
%s    {
%s%s      }
%s}
"
		 prefix
		 prefix prefix
		 name
		 prefix
		 name
		 (field-ctype (cadr field))
		 name
		 prefix
		 name
		 prefix
		 prefix
		 (entry-unserialize-field entry (cadr field) (concat "(" name "[i])") (concat prefix "    "))
		 prefix
		 ))
	(t (error "unrecognized field type: %s" (cadr field)))))


(defun entry-serialize-field (entry field name prefix)
  (cond ((equal (cadr field) 'uint)
	 (format "%sif (! (* sink->next_uint) (sink, %s)) goto bail;\n" prefix name))
	((equal (cadr field) 'uint16)
	 (format "%sif (! (* sink->next_uint16) (sink, %s)) goto bail;\n" prefix name))
	((equal (cadr field) 'uint8)
	 (format "%sif (! (* sink->next_uint8) (sink, %s)) goto bail;\n" prefix name))
	((equal (cadr field) 'uint32)
	 (format "%sif (! (* sink->next_uint32) (sink, %s)) goto bail;\n" prefix name))
	((equal (cadr field) 'boolean)
	 (format "%sif (! (* sink->next_bool) (sink, %s)) goto bail;\n" prefix name))
	((equal (cadr field) 'string)
	 (format "%sif (! (* sink->next_string) (sink, %s)) goto bail;\n" prefix name))
	((equal (cadr field) 'bytes)
	 (format "%sif (! (* sink->next_bytes) (sink, %s, %s_len)) goto bail;\n" prefix name name))
	((member (cadr field) (mapcar (lambda (x) (sertype-name-get x)) *sertype-defs*))
	 (format "%sif (serialize_%s_internal (sink%s) != SerialSuccess) goto bail;\n" prefix (downcase-string (cadr field))
		 (entry-plist t t (concat name "->") (obj-name-eq (cadr field) *sertype-defs*))))
	((and (equal (car (cadr field)) 'ptr)
	      (member (cadr (cadr field)) (mapcar (lambda (x) (sertype-name-get x)) *sertype-defs*)))
	 (format "%sif (serialize_%s_internal (sink%s) != SerialSuccess) goto bail;\n" prefix (downcase-string (cadr (cadr field)))
		 (entry-plist t t (concat name "->") (obj-name-eq (cadr (cadr field)) *sertype-defs*))))
	((equal (car (cadr field)) 'bytes)
	 (format "%sif (! (* sink->next_bytes_known) (sink, %s, %d)) goto bail;\n" prefix name (cadr (cadr field))))
	((equal (car (cadr field)) 'array)
	 (format "%s{
%s  gint i;
%s  if (! (* sink->next_uint) (sink, %s_len)) goto bail;
%s  for (i = 0; i < %s_len; i += 1)
%s    {
%s%s      }
%s}
"
		 prefix prefix prefix
		 name
		 prefix
		 name
		 prefix
		 prefix
		 (entry-serialize-field entry (cadr field) (array-index name (cadr field)) (concat prefix "    "))
		 prefix
		 ))
	(t (error "unrecognized field type: %s" (cadr field)))))

(defun array-index (name field)
  (concat "(" (if (needs-ref field) "&" "") name "[i])")
  )

(defun entry-count-field (entry field name prefix)
  (cond ((equal (cadr field) 'uint)
	 (format "%ssize += sizeof (guint32);\n" prefix)) ;; this is possibly not enough
	((equal (cadr field) 'uint32)
	 (format "%ssize += sizeof (guint32);\n" prefix))
	((equal (cadr field) 'uint16)
	 (format "%ssize += sizeof (guint16);\n" prefix))
	((equal (cadr field) 'uint8)
	 (format "%ssize += sizeof (guint8);\n" prefix))
	((equal (cadr field) 'boolean)
	 (format "%ssize += 1;\n" prefix))
	((equal (cadr field) 'string)
	 (format "%ssize += strlen (%s) + 1;\n" prefix name))
	((equal (cadr field) 'bytes)
	 (format "%ssize += %s_len;\n" prefix name))
	((member (cadr field) (mapcar (lambda (x) (sertype-name-get x)) *sertype-defs*))
 	 (format "%ssize += count_%s (%s) - sizeof (Serial%s);\n"
		 prefix
		 (downcase-string (cadr field))
		 (entry-plist t nil (concat name "->") (obj-name-eq (cadr field) *sertype-defs*))
		 (cadr field)
 		 ))
	((and (equal (car (cadr field)) 'ptr)
	      (member (cadr (cadr field)) (mapcar (lambda (x) (sertype-name-get x)) *sertype-defs*)))
 	 (format "%ssize += count_%s (%s);\n" prefix (downcase-string (cadr (cadr field)))
 		 (entry-plist t nil (concat name "->") (obj-name-eq (cadr (cadr field)) *sertype-defs*))))
	((equal (car (cadr field)) 'bytes)
	 (format "%ssize += 0;\n" prefix (cadr (cadr field))))
	((equal (car (cadr field)) 'array)
	 (format "%s{
%s  gint i;
%s  for (i = 0; i < %s_len; i += 1)
%s    {
%s%s      }
%s}
"
		 prefix prefix prefix
		 name
		 prefix
		 prefix
		 (entry-count-field entry (cadr field) (array-index name (cadr field)) (concat prefix "    "))
		 prefix
		 ))
	(t (error "unrecognized field type: %s" (cadr field)))))
