/***************************************
  $Header: /home/amb/procmeter/RCS/log.c 1.8 1998/01/06 19:13:36 amb Exp $

  ProcMeter - A performance metering/logging program for Linux - Version 2.4.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997,98 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <syslog.h>
#include <signal.h>
#include <sys/time.h>

#include "procmeter.h"


#ifndef SYSLOG_PRIORITY
/*+ The logging priority to use for syslog. +*/
#define SYSLOG_PRIORITY LOG_INFO
#endif

#ifndef SYSLOG_FACILITY
/*+ The logging facility to use for syslog. +*/
#define SYSLOG_FACILITY LOG_LOCAL7
#endif


static void PrintMeterTitles(void);


/*+ The log file. +*/
static FILE *file=NULL;

/*+ Have we changed the stats that are in the file. +*/
static int changed=-1;

/*+ The number of types of statistics to log. +*/
static int n_logged=0;

/*+ The name of the host that we are running on. +*/
extern char HostName[40];

/*+ The current time. +*/
extern double StatsTime;

/*+ The name of the logfile if any. +*/
char *logfile=NULL;

/*+ A buffer to store the string to log. +*/
static char buffer[1024];


/*++++++++++++++++++++++++++++++++++++++
  Start the logging to a file.
  ++++++++++++++++++++++++++++++++++++++*/

void StartLog(void)
{
 if(!strcmp(logfile,"syslog"))
   {
    openlog("procmeter",0,SYSLOG_FACILITY);
    file=NULL;
   }
 else if(strcmp(logfile,"-"))
   {
    file=fopen(logfile,"a");
 
    if(!file)
      {fprintf(stderr,"ProcMeter: The specified log file '%s' cannot be opened for writing.\n",logfile);exit(1);}
   }
 else
    file=stdout;
}


/*++++++++++++++++++++++++++++++++++++++
  Stop the logging to a file.
  ++++++++++++++++++++++++++++++++++++++*/

void StopLog(void)
{
 if(!file)
    closelog();
 else if(file!=stdout)
    fclose(file);
}


/*++++++++++++++++++++++++++++++++++++++
  Add a new line to the log file.

  time_t now The current time.
  ++++++++++++++++++++++++++++++++++++++*/

void UpdateMetersLog(time_t now)
{
 int i;
 char buf[16];

 if(changed==-1)
    if(file)
       fprintf(file,"# Starting log on %s at %s",*HostName?HostName:"(unknown host)",ctime(&now));
    else
       syslog(SYSLOG_PRIORITY,"# Starting log on %s at %s",*HostName?HostName:"(unknown host)",ctime(&now));

 if(changed>0)
    PrintMeterTitles();

 changed=0;

 if(!n_logged)
    return;

 if(file)
    sprintf(buffer,"%ld",(long)now);
 else
    sprintf(buffer,"%ld",(long)now);

 for(i=0;i<NProcStats;i++)
    if(ProcStats[i].used&2)
      {
       if(i==LOAD)
          sprintf(buf," %.2f",ProcStats[i].value);
       else
          if(i==CPU || i==CPU_USER || i==CPU_SYS || i==CPU_IDLE)
             sprintf(buf," %.1f",ProcStats[i].value);
          else
             sprintf(buf," %d",(int)ProcStats[i].value);
       strcat(buffer,buf);
      }

 for(i=1;i<=NOtherStats;i++)
    if(OtherStats[i].used&2)
      {
       sprintf(buf," %d",(int)OtherStats[i].value);
       strcat(buffer,buf);
      }

 if(file)
   {
    strcat(buffer,"\n");
    fprintf(file,buffer);
    fflush(file);
   }
 else
   {
    /* When syslogd is killed, we receive a SIGPIPE, so we ignore it */
    signal(SIGPIPE, SIG_IGN);
    syslog(SYSLOG_PRIORITY, buffer);
    signal(SIGPIPE, SIG_DFL);
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Add or remove a meter from the list of those that are logged.

  int type The type of meter to add or remove, +ve for a /proc one, -ve otherwise.
  ++++++++++++++++++++++++++++++++++++++*/

void AddRemoveMeterLog(int type)
{
 if(type<0)
   {
    if(OtherStats[-type].used)
       n_logged++;
    else
       n_logged--;
   }
 else
   {
    if(ProcStats[type].used)
       n_logged++;
    else
       n_logged--;
   }

 if(!changed)
    changed=1;
}


/*++++++++++++++++++++++++++++++++++++++
  Print the title line that indicates the stats that are in the file.
  ++++++++++++++++++++++++++++++++++++++*/

static void PrintMeterTitles(void)
{
 int i;
 char buf[16];

 strcpy(buffer,"# time");

 for(i=0;i<NProcStats;i++)
    if(ProcStats[i].used)
      {
       sprintf(buf," %s",ProcStats[i].name);
       strcat(buffer,buf);
      }

 for(i=1;i<=NOtherStats;i++)
    if(OtherStats[i].used)
      {
       sprintf(buf," %s",OtherStats[i].name);
       strcat(buffer,buf);
      }

 if(file)
   {
    strcat(buffer,"\n");
    fprintf(file,buffer);
    fflush(file);
   }
 else
   {
    signal(SIGPIPE, SIG_IGN);
    syslog(SYSLOG_PRIORITY, buffer);
    signal(SIGPIPE, SIG_DFL);
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Sleep for the specified interval in seconds.

  time_t until The time to sleep until.
  ++++++++++++++++++++++++++++++++++++++*/

void SleepNotX(time_t until)
{
 struct timeval now;
 int delay;

 gettimeofday(&now,NULL);

 delay=1000*(until-now.tv_sec)-now.tv_usec/1000;

 if(delay>0)
    usleep(1000*delay);
}
