/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Memory allocation routines.
 */

#include "driver.h"

/*
 *  page allocation
 */

MUTEX_DEFINE_STATIC( memory );
MUTEX_DEFINE_STATIC( dma );
static long gus_pages;

char *gus_malloc_pages( unsigned long size, int *res_pg, int dma )
{
  int pg;
  char *buf;
  
  for ( pg = 0; PAGE_SIZE * ( 1 << pg ) < size; pg++ );
#if LinuxVersionCode( 2, 1, 85 ) <= LINUX_VERSION_CODE
  buf = (char *)__get_free_pages( GFP_KERNEL | (dma ? GFP_DMA : 0), pg );
#else
  buf = (char *)__get_free_pages( GFP_KERNEL, pg, dma ? MAX_DMA_ADDRESS : 0 );
#endif
#if 0 
  printk( "pg = %i, buf = 0x%x\n", pg, (int)buf );
#endif
  if ( buf )
    {
      int page, last_page;
      
      page = MAP_NR( buf );
      last_page = page + ( 1 << pg );
      while ( page < last_page )
        set_bit( PG_reserved, &mem_map[ page++ ].flags );
      gus_pages += 1 << pg;
    }
  *res_pg = pg;
  return buf;
}

void gus_free_pages( char *ptr, unsigned long size )
{
  int pg, page, last_page;

  for ( pg = 0; PAGE_SIZE * ( 1 << pg ) < size; pg++ );
  page = MAP_NR( ptr );
  last_page = page + ( 1 << pg );
  while ( page < last_page )
    clear_bit( PG_reserved, &mem_map[ page++ ].flags );
  free_pages( (unsigned long)ptr, pg );
  gus_pages -= 1 << pg; 
}

/*
 *  ok.. new allocation routines..
 */

#define BLOCKS_IN_PAGE		(PAGE_SIZE / 32)
#define HEADER_SIZE		sizeof( struct alloc_header )
#define FREE_SPACE		(PAGE_SIZE - HEADER_SIZE)
#define ALIGN_VALUE( x )	( ( (x) + ( sizeof( void * ) - 1 ) ) & ~( sizeof( void * ) - 1 ) )

struct alloc_header {
  struct alloc_header *next;			/* next page */
  unsigned short sizes[ BLOCKS_IN_PAGE ];	/* size of block */
};

static struct alloc_header *gus_alloc_first_page;

#ifdef GUSCFG_DEBUG_MEMORY
static void *gusmem_alloc_ptr[ 1024 ];
static unsigned long gusmem_alloc_size[ 1024 ];
#endif

void gus_malloc_init( void )
{
  gus_pages = 0;
  gus_alloc_first_page = NULL;
#ifdef GUSCFG_DEBUG_MEMORY
  memset( gusmem_alloc_ptr, 0, sizeof( gusmem_alloc_ptr ) );
#endif
}

void gus_malloc_done( void )
{
  struct alloc_header *hdr, *next;

  if ( ( hdr = gus_alloc_first_page ) != NULL )
    {
#ifdef GUSCFG_DEBUG
      PRINTK( "gus: malloc done - trouble\n" );
#endif
      while ( hdr )
        {
          next = hdr -> next;
          gus_free_pages( (char *)hdr, PAGE_SIZE );
          hdr = next;
        }
    }    
#ifdef GUSCFG_DEBUG
  if ( gus_pages > 0 )
    PRINTK( "gus: malloc done - gus_pages = %li\n", gus_pages );
#endif
}

void *gus_malloc( unsigned long size )
{
  int pg;

  if ( !size ) return NULL;
#if 0
  PRINTK( "mallocccccccccccccccc %li\n", size );
#endif
  if ( size > FREE_SPACE )
    {
      void *res;
      
      res = gus_malloc_pages( size, &pg, 0 );
#ifdef GUSCFG_DEBUG_MEMORY
      {
        int idx;

        for ( idx = 0; idx < 1024; idx++ )
          if ( !gusmem_alloc_ptr[ idx ] )
            {
              gusmem_alloc_ptr[ idx ] = res;
              gusmem_alloc_size[ idx ] = size;
              break;
            }
      }
#endif
      return res;
    }
   else
    {
      struct alloc_header *hdr;
      struct alloc_header *phdr = NULL;

      MUTEX_DOWN_STATIC( memory );      
      hdr = gus_alloc_first_page;
      size = ALIGN_VALUE( size );
      while ( hdr )
        {
          int idx;
          unsigned short used;
          
          for ( idx = 0, used = 0; idx < BLOCKS_IN_PAGE; idx++ )
            {
              unsigned short bused;
              unsigned short bsize;
              
              bsize = hdr -> sizes[ idx ];
              bused = bsize & 0x8000;
              bsize &= 0x7fff;
              if ( ( !bsize && ( used + HEADER_SIZE + size <= PAGE_SIZE ) ) || 
                   ( !bused && bsize >= size ) )  /* free & passed */
                {
                  if ( bsize > size )
                    {
                      int i;
                      if ( hdr -> sizes[ BLOCKS_IN_PAGE - 1 ] & 0x8000 ) continue;	/* next block */
                      for ( i = BLOCKS_IN_PAGE - 1; i > idx + 1; i-- )
                        hdr -> sizes[ i ] = hdr -> sizes[ i - 1 ];
                      hdr -> sizes[ idx + 1 ] = bsize - size;
                    }
                  hdr -> sizes[ idx ] = 0x8000 | size;
                  MUTEX_UP_STATIC( memory );
                  return (void *)((char *)hdr + HEADER_SIZE + used);
                }
              used += bsize;
            }
          hdr = ( phdr = hdr ) -> next;
        }
      hdr = (struct alloc_header *)gus_malloc_pages( PAGE_SIZE, &pg, 0 );
      if ( !hdr ) {
        MUTEX_UP_STATIC( memory );
        return NULL;
      }
      memset( hdr, 0, HEADER_SIZE );
      if ( !phdr )
        gus_alloc_first_page = hdr; 
       else
        phdr -> next = hdr;
      hdr -> sizes[ 0 ] = 0x8000 | size;
      MUTEX_UP_STATIC( memory );
      return (void *)((char *)hdr + HEADER_SIZE);
    }
}

void gus_free( void *obj, unsigned long size )
{
  if ( !size ) return;
  if ( !obj )
    {
#ifdef GUSCFG_DEBUG_MEMORY
      PRINTK( "gus: gus_free - NULL?\n" ); 
#endif
      return;
    }
#if 0
  PRINTK( "freeeeeeeeeeee 0x%x, %li\n", obj, size );
#endif
  if ( size > FREE_SPACE )
    {
      gus_free_pages( obj, size );
#ifdef GUSCFG_DEBUG_MEMORY
      {
        int idx;
     
        for ( idx = 0; idx < 1024; idx++ )
          if ( gusmem_alloc_ptr[ idx ] == obj )
            {
              gusmem_alloc_ptr[ idx ] = NULL;
              return;
            }
        PRINTK( "gus: gus_free (1) - free of unallocated block at 0x%x size %li\n", (int)obj, size );
      }
#endif
    }
   else
    {
      struct alloc_header *hdr;
      struct alloc_header *phdr = NULL;	/* previous header */

      MUTEX_DOWN_STATIC( memory );
      hdr = gus_alloc_first_page;
      while ( hdr )
        {
          int idx;
          unsigned short used;
          
          for ( idx = 0, used = 0; hdr -> sizes[ idx ] != 0 && idx < BLOCKS_IN_PAGE; idx++ )
            {
              if ( (char *)hdr + HEADER_SIZE + used == obj )
                {
                  hdr -> sizes[ idx ] &= ~0x8000;
                  if ( idx + 1 < BLOCKS_IN_PAGE && hdr -> sizes[ idx + 1 ] == 0 )
                    hdr -> sizes[ idx ] = 0;
                  for ( idx = 0, used = 0; hdr -> sizes[ idx ] != 0 && idx < BLOCKS_IN_PAGE; idx++ )
                    {
                      if ( idx + 1 < BLOCKS_IN_PAGE &&
                           !( hdr -> sizes[ idx ] & 0x8000 ) &&		/* join blocks */
                           !( hdr -> sizes[ idx + 1 ] & 0x8000 ) )
                        {
                          int i;
                        
                          hdr -> sizes[ idx ] += hdr -> sizes[ idx + 1 ];
                          for ( i = idx + 1; i + 1 < BLOCKS_IN_PAGE; i++ )
                            hdr -> sizes[ i ] = hdr -> sizes[ i + 1 ];
			  hdr -> sizes[ BLOCKS_IN_PAGE - 1 ] = 0;
                        }
                      if ( hdr -> sizes[ idx ] & 0x8000 ) used++;
                    }
                  if ( idx > 0 && !(hdr -> sizes[ idx - 1 ] & 0x8000) )
                    hdr -> sizes[ idx - 1 ] = 0;
                  if ( !used )
                    {
                      if ( phdr )
                        phdr -> next = hdr -> next;
                       else
                        gus_alloc_first_page = hdr -> next;
                      gus_free_pages( (char *)hdr, PAGE_SIZE );
                    }
                  MUTEX_UP_STATIC( memory );
                  return;		/* ok.. successfull */
                }
              used += hdr -> sizes[ idx ] & 0x7fff;
            }
          phdr = hdr;
          hdr = hdr -> next;
        }
#ifdef GUSCFG_DEBUG_MEMORY
      PRINTK( "gus: gus free (2) - free of unallocated block at 0x%x size %li\n", (int)obj, size );
#endif
      MUTEX_UP_STATIC( memory ); 
    }
}

char *gus_malloc_strdup( char *string, int space )
{
  int len;
  char *res;

  if ( !string ) return NULL;
  if ( space == SP_USER )
    {
      for ( res = string, len = 0; get_fs_byte( res ) != 0; res++, len++ );
      if ( ( res = gus_malloc( ++len ) ) == NULL ) return NULL;
      MEMCPY_FROMFS( res, string, len );
    }
   else
    {
      len = strlen( string ) + 1;
      if ( ( res = gus_malloc( len ) ) == NULL ) return NULL;
      MEMCPY( res, string, len );
    }
  return res;
}

void gus_free_str( char *string )
{
  if ( string )
    gus_free( string, strlen( string ) + 1 );
}

void gus_memory_info( gus_info_buffer_t *buffer )
{
  gus_iprintf( buffer, "Driver using %i page%s (%li bytes) of kernel memory for data.\n",
          			gus_pages,
          			gus_pages > 1 ? "s" : "",
		          	gus_pages * PAGE_SIZE );
}

#ifdef GUSCFG_DEBUG_MEMORY
void gus_memory_debug( gus_info_buffer_t *buffer )
{
  int i, used, free, total_blocks, total_used;
  struct alloc_header *hdr;

  MUTEX_DOWN_STATIC( memory );
  gus_iprintf( buffer, "\n-----\n\n"
  		         "Memory allocation table:\n" );
  for ( i = used = 0; i < 1024; i++ )
    if ( gusmem_alloc_ptr[ i ] )
      {
        used++;
        gus_iprintf( buffer, "  %04i: 0x%08lx [%08i] = %8li\n",
      					i,
        				gusmem_alloc_ptr[ i ],
        				MAP_NR( gusmem_alloc_ptr[ i ] ),
        				gusmem_alloc_size[ i ] );
      }
  for ( hdr = gus_alloc_first_page; hdr; hdr = hdr -> next, i++ )
    gus_iprintf( buffer, "  %04i: 0x%08lx [%08i] = %8li (internal)\n", i, (long)hdr, MAP_NR( hdr ), PAGE_SIZE );
  if ( !used && !gus_alloc_first_page )
    gus_iprintf( buffer, "  -none-\n" );
  gus_iprintf( buffer, "\nMemory allocation table (internal):\n" );
  hdr = gus_alloc_first_page;
  if ( !hdr )
    gus_iprintf( buffer, "  -none-\n" );
   else
    {
      total_blocks = total_used = 0;
      while ( hdr )
        {
          gus_iprintf( buffer, "  page at 0x%x:\n", (int)hdr );
          for ( i = used = free = 0; i < BLOCKS_IN_PAGE; i++ )
            {
              unsigned short bsize, bused;
              
              bsize = hdr -> sizes[ i ];
              if ( !bsize ) continue;
              bused = bsize & 0x8000;
              bsize &= 0x7fff;
              total_blocks++;
              gus_iprintf( buffer, "    %04i: 0x%08x = %4li (%s)\n", i, (int)((char *)hdr + HEADER_SIZE + used), bsize, bused ? "allocated" : "free" );
              if ( !bused ) free += bsize; else total_used += bsize;
              used += bsize;
            }
          gus_iprintf( buffer, "    sum : size = %li, used = %li, free = %li, unused = %li\n", PAGE_SIZE - HEADER_SIZE, used - free, free, PAGE_SIZE - HEADER_SIZE - used );
          hdr = hdr -> next;
        }
      gus_iprintf( buffer, "    total: blocks = %i, used = %i, recomended block allocation = %i\n",
      			total_blocks, total_used,
      			PAGE_SIZE / ( total_used / total_blocks ) );
    }
  MUTEX_UP_STATIC( memory );
}
#endif

/*
 *  DMA allocation routines
 */

int gus_dma_malloc( gus_card_t *card, int dma_number, char *owner, int lock )
{
  struct GUS_STRU_DMA *dma;
  unsigned char *buf;
  long size;
  int pg = 0;

  MUTEX_DOWN_STATIC( dma );
  dma = card -> dmas[ dma_number ];
  if ( dma -> used > 0 )
    {
      if ( ( dma -> lock & WK_LOCK ) && lock )
        {
#ifdef GUSCFG_DEBUG
          PRINTK( "gus: gus_dma_malloc - trying allocate locked DMA channel\n" );
#endif
          MUTEX_UP_STATIC( dma );
          return -1;
        }
      if ( lock )
        {
          dma -> old_owner = dma -> owner;
          dma -> owner = owner;
          dma -> lock = WK_LOCK;
        }
       else
        dma -> old_owner = owner;
      dma -> used++;
      MUTEX_UP_STATIC( dma );
      return 0;		/* ok... dma is allocated */
    }
  if ( dma -> buf )
    {
      if ( !dma -> static_alloc )
        {
#ifdef GUSCFG_DEBUG        
          PRINTK( "gus: gus_dma_malloc - already?\n" );
#endif
        }
       else
        {
          if ( !dma -> used )
            {
              dma -> usize = dma -> ursize;
              if ( dma -> usize > dma -> size )
                dma -> usize = dma -> size;
            }
        }
      dma -> used++;
      dma -> lock = lock ? WK_LOCK : 0;
      dma -> owner = owner;
      MUTEX_UP_STATIC( dma );
      return 0;
    }
  if ( dma -> rsize < PAGE_SIZE ) dma -> rsize = PAGE_SIZE;
  buf = NULL;
  size = 64 * 1024L;
  if ( dma -> dma > 3 ) size <<= 1;
  if ( size > dma -> rsize ) size = dma -> rsize;
  while ( !buf && size >= PAGE_SIZE )
    {
      buf = gus_malloc_pages( size, &pg, 1 );
      if ( !buf ) size >>= 1;
    }
  if ( buf )
    {
      dma -> size = ( 1 << pg ) * PAGE_SIZE;
      dma -> usize = dma -> ursize;
      if ( dma -> usize > dma -> size )
        dma -> usize = dma -> size;
      dma -> buf = buf;
      if ( lock < 0 )
        {
          dma -> used = 0;
          dma -> static_alloc = 1;
          dma -> owner = NULL;
          dma -> lock = 0;
        }
       else
        {
          dma -> used++;
          dma -> static_alloc = 0;
          dma -> owner = owner;
          dma -> lock = lock ? WK_LOCK : 0;
        }
      dma -> mmaped = 0;
    }
  MUTEX_UP_STATIC( dma );
  return buf == NULL ? -1 : 0;
}

void gus_dma_free( gus_card_t *card, int dma_number, int lock )
{
  struct GUS_STRU_DMA *dma;

  MUTEX_DOWN_STATIC( dma );
  dma = card -> dmas[ dma_number ];
  if ( lock >= 0 )
    {
      if ( dma -> used ) dma -> used--;
      if ( dma -> used )
        {
          if ( lock )
            {
              dma -> owner = dma -> old_owner;
              dma -> old_owner = NULL;
              dma -> lock = 0;
            }
           else
            dma -> old_owner = NULL;
          MUTEX_UP_STATIC( dma );
          return;
        }
    }
#ifdef GUSCFG_DEBUG
   else
    if ( dma -> used > 0 )
      PRINTK( "gus: gus_dma_free - static error?\n" );
#endif
  if ( !dma -> static_alloc )
    {
      if ( dma -> buf )
        {
          gus_free_pages( dma -> buf, dma -> size );
          dma -> buf = NULL;
        }
      dma -> size = dma -> usize = 0;
    }
   else
    {
      if ( lock < 0 )
        {
          gus_free_pages( dma -> buf, dma -> size );
          dma -> buf = NULL;
          dma -> static_alloc = 0;
          dma -> size = dma -> usize = 0;
        }
    }  
  dma -> lock = 0;
  dma -> owner = dma -> old_owner = NULL;
  MUTEX_UP_STATIC( dma ); 
}
