/*
 *  Copyright (c) by Jaroslav Kysela (Perex soft)
 *  Routines for control of LFO generators (tremolo & vibrato)
 */

#include "driver.h"

/*
 *  called by engine routines
 */

static signed char gus_lfo_compute_value( gus_card_t *card, unsigned char *ptr )
{
  unsigned int twaveinc, depth_delta;
  signed int result;
  unsigned short control, twave, depth, depth_final;
  unsigned char *ptr1;

  control = *(unsigned short *)(ptr + 0x00);
  ptr1 = ptr + ( ( control & 0x4000 ) >> 12 );
  /* 1. add TWAVEINC to TWAVE and write the result back */
  /* LFO update rate is 689Hz, effect timer is in ms */
  if ( card -> gf1.timer_slave )
    twaveinc = ( 689 * card -> gf1.timer_master_card -> gf1.effect_timer ) / 1000;
   else
    twaveinc = ( 689 * card -> gf1.effect_timer ) / 1000;
  if ( !twaveinc ) twaveinc++;
#if 0
  printk( "twaveinc = 0x%x, effect_timer = %i\n", twaveinc, card -> gf1.effect_timer );
#endif

  depth = *(unsigned short *)(ptr1 + 0x0a);
  depth_final = *(unsigned char *)(ptr + 0x02) << 5;
  if ( depth != depth_final )
    {
      depth_delta = ( ( twaveinc * *(ptr + 0x03) ) + *(unsigned short *)(ptr + 0x04) );
      *(unsigned short *)(ptr + 0x04) = depth_delta % 8000;
      depth_delta /= 8000;
      if ( depth < depth_final )
        {
          if ( depth + depth_delta > depth_final )
            depth = depth_final;
           else
            depth += depth_delta;  
        }
      if ( depth > depth_final )
        {
          if ( depth - depth_delta < depth_final )
            depth = depth_final;
           else
            depth -= depth_delta;
        }
      *(unsigned short *)(ptr1 + 0x0a) = depth;
    }
  
  twaveinc *= (unsigned int)control & 0x7ff;
  twaveinc += *(unsigned short *)( ptr + 0x06 );
  *(unsigned short *)(ptr + 0x06) = twaveinc % 1000;

  twave = *(unsigned short *)(ptr1 + 0x08);
  twave += (unsigned short)(twaveinc / (unsigned int)1000);
  *(unsigned short *)(ptr1 + 0x08) = twave;

  if ( !(control & 0x2000) )
    {
      /* 2. if shift is low */
      if ( twave & 0x4000 )	/* bit 14 high -> invert TWAVE 13-0 */
        {
          twave ^= 0x3fff;
          twave &= ~0x4000;
        }
      /* TWAVE bit 15 is exclusive or'd with the invert bit (12) */
      twave ^= ( control & 0x1000 ) << 3;
    }
   else
    {
      /* 2. if shift is high */
      if ( twave & 0x8000 )	/* bit 15 high -> invert TWAVE 14-0 */
        twave ^= 0x7fff;
      /* the invert bit (12) is used as sign bit */
      if ( control & 0x1000 )
        twave |= 0x8000;
       else
        twave &= ~0x8000;
    }
  /* 3. multiply the 14-bit LFO waveform magnitude by 13-bit DEPTH */
#if 0
  printk( "c=0x%x,tw=0x%x,to=0x%x,d=0x%x,df=0x%x,di=0x%x,r=0x%x,r1=%i\n",
  		control, twave, 
  		*(unsigned short *)(ptr1 + 0x08),
  		depth, depth_final, *(ptr + 0x03),
  		(twave & 0x7fff) * depth, ( (twave & 0x7fff) * depth ) >> 21 );
#endif
  result = (twave & 0x7fff) * depth;
  if ( result )
    {
      /* shift */
      result >>= 21;
      result &= 0x3f;
    }
  /* add sign */      
  if ( twave & 0x8000 ) result = -result;
#if 0
  printk( "lfo final value = %i\n", result );
#endif
  return result;
}

static void gus_lfo_register_setup( gus_card_t *card, gus_voice_t *voice, int lfo_type )
{
  unsigned long flags;

  if ( card -> gf1.enh_mode )
    {
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      if ( lfo_type & 1 )
        {
          gus_write8( card, GF1_VB_FREQUENCY_LFO, voice -> lfo_fc );
          voice -> lfo_fc = 0;
        }
      if ( lfo_type & 2 )
        {
          gus_write8( card, GF1_VB_VOLUME_LFO, voice -> lfo_volume );
          voice -> lfo_volume = 0;
        }
      STI( &flags );
    }
   else
    {
      /*
       * ok.. with old GF1 chip can be only vibrato emulated...
       * volume register can be in volume ramp state, so tremolo isn't simple..
       */
      if ( !(lfo_type & 1) ) return;
#if 0
      if ( voice -> lfo_fc )
        printk( "setup - %i = %i\n", voice -> number, voice -> lfo_fc );
#endif
      CLI( &flags );
      gf1_select_voice( card, voice -> number );
      gus_write16( card, GF1_VW_FREQUENCY, voice -> fc_register + voice -> lfo_fc );
      STI( &flags );
    }
}

void gus_lfo_effect_interrupt( gus_card_t *card, gus_voice_t *voice )
{
  unsigned char *ptr;
  
#if 0
  if ( voice -> number != 0 ) return;
#endif
  ptr = card -> gf1.lfos + ( ( voice -> number ) << 5 );
  /* 1. vibrato */
  if ( *(unsigned short *)(ptr + 0x00) & 0x8000 )
    voice -> lfo_fc = gus_lfo_compute_value( card, ptr );
  /* 2. tremolo */
#if 1
  ptr += 16;
  if ( *(unsigned short *)(ptr + 0x00) & 0x8000 )
    voice -> lfo_volume = gus_lfo_compute_value( card, ptr );
#endif
  /* 3. register setup */
  gus_lfo_register_setup( card, voice, 3 );
}

/*
 *
 */

void gus_lfo_init( gus_card_t *card )
{
#ifdef GUSCFG_INTERWAVE
  if ( card -> gf1.hw_lfo )
    {
      gus_i_write16( card, GF1_GW_LFO_BASE, 0x0000 );
      gus_dram_setmem( card, 0, 0x0000, 1024 );
      /* now enable LFO */
      gus_i_write8( card, GF1_GB_GLOBAL_MODE, gus_i_look8( card, GF1_GB_GLOBAL_MODE ) | 0x02 );
    }
#endif
  if ( card -> gf1.sw_lfo )
    {
#if 1
      card -> gf1.lfos = gus_malloc( 1024 );
      if ( card -> gf1.lfos )
        memset( card -> gf1.lfos, 0, 1024 );
       else
#endif
        card -> gf1.sw_lfo = 0;
    }
}

void gus_lfo_done( gus_card_t *card )
{
  if ( card -> gf1.sw_lfo )
    {
      if ( card -> gf1.lfos )
        {
          gus_free( card -> gf1.lfos, 1024 );
          card -> gf1.lfos = NULL;
        }
    }
}

void gus_lfo_program( gus_card_t *card, int voice, int lfo_type, struct GUS_STRU_IW_LFO_PROGRAM *program )
{
  unsigned int lfo_addr, wave_select;
  
  wave_select = ( program -> freq_and_control & 0x4000 ) >> 12;
  lfo_addr = ( voice << 5 ) | ( lfo_type << 4 );
#ifdef GUSCFG_INTERWAVE
  if ( card -> gf1.hw_lfo )
    {
#if 0
      printk( "LMCI = 0x%x\n", gus_i_look8( card, 0x53 ) );
      printk( "lfo_program: lfo_addr=0x%x,wave_sel=0x%x,fac=0x%x,df=0x%x,di=0x%x,twave=0x%x,depth=0x%x\n",
      		lfo_addr, wave_select,
      		program -> freq_and_control,
      		program -> depth_final,
      		program -> depth_inc,
      		program -> twave,
      		program -> depth );
#endif
      gus_poke( card, lfo_addr + 0x02, program -> depth_final );
      gus_poke( card, lfo_addr + 0x03, program -> depth_inc );
      gus_pokew( card, lfo_addr + 0x08 + wave_select, program -> twave );
      gus_pokew( card, lfo_addr + 0x0a + wave_select, program -> depth );
      gus_pokew( card, lfo_addr + 0x00, program -> freq_and_control );
#if 0
      {
        int i = 0;
        for ( i = 0; i < 16; i++ )
          printk( "%02x:", gus_peek( card, lfo_addr + i ) );
        printk( "\n" );
      }
#endif
    }
#endif
  if ( card -> gf1.sw_lfo )
    {
      unsigned char *ptr = card -> gf1.lfos + lfo_addr;
    
      *(ptr + 0x02) = program -> depth_final;
      *(ptr + 0x03) = program -> depth_inc;
      *(unsigned short *)(ptr + 0x08 + wave_select) = program -> twave;
      *(unsigned short *)(ptr + 0x0a + wave_select) = program -> depth;
      *(unsigned short *)(ptr + 0x00) = program -> freq_and_control;
    }
}

void gus_lfo_enable( gus_card_t *card, int voice, int lfo_type )
{
  unsigned int lfo_addr;
  
  lfo_addr = ( voice << 5 ) | ( lfo_type << 4 );
#ifdef GUSCFG_INTERWAVE
  if ( card -> gf1.hw_lfo )
    gus_pokew( card, lfo_addr + 0x00, gus_peekw( card, lfo_addr + 0x00 ) | 0x8000 );
#endif
  if ( card -> gf1.sw_lfo )
    {
      unsigned char *ptr = card -> gf1.lfos + lfo_addr;
      
      *(unsigned short *)(ptr + 0x00) |= 0x8000;
    }
}

void gus_lfo_disable( gus_card_t *card, int voice, int lfo_type )
{
  unsigned int lfo_addr;

  lfo_addr = ( voice << 5 ) | ( lfo_type << 4 );  
#ifdef GUSCFG_INTERWAVE
  if ( card -> gf1.hw_lfo )
    gus_pokew( card, lfo_addr + 0x00, gus_peekw( card, lfo_addr + 0x00 ) & ~0x8000 );
#endif
  if ( card -> gf1.sw_lfo )
    {
      unsigned char *ptr = card -> gf1.lfos + lfo_addr;
      
      *(unsigned short *)(ptr + 0x00) &= ~0x8000;
    }
}

void gus_lfo_change_freq( gus_card_t *card, int voice, int lfo_type, int freq )
{
  unsigned int lfo_addr;

  lfo_addr = ( voice << 5 ) | ( lfo_type << 4 );
#ifdef GUSCFG_INTERWAVE
  if ( card -> gf1.hw_lfo )
    gus_pokew( card, lfo_addr + 0x00, ( gus_peekw( card, lfo_addr + 0x00 ) & ~0x7ff ) | ( freq & 0x7ff ) );
#endif
  if ( card -> gf1.sw_lfo )
    {
      unsigned long flags;
      unsigned char *ptr = card -> gf1.lfos + lfo_addr;
      
      CLI( &flags );
      *(unsigned short *)(ptr + 0x00) &= ~0x7ff;
      *(unsigned short *)(ptr + 0x00) |= freq & 0x7ff;
      STI( &flags );
    }
}

void gus_lfo_change_depth( gus_card_t *card, int voice, int lfo_type, int depth )
{
  unsigned long flags;
  unsigned int lfo_addr;
  unsigned short control = 0;
  unsigned char *ptr;

  lfo_addr = ( voice << 5 ) | ( lfo_type << 4 );
  ptr = card -> gf1.lfos + lfo_addr;
#ifdef GUSCFG_INTERWAVE
  if ( card -> gf1.hw_lfo )
    control = gus_peekw( card, lfo_addr + 0x00 );
#endif
  if ( card -> gf1.sw_lfo )
    control = *(unsigned short *)(ptr + 0x00);
  if ( depth < 0 )
    {
      control |= 0x1000;
      depth = -depth;
    }
   else
    control &= ~0x1000;
#ifdef GUSCFG_INTERWAVE
  if ( card -> gf1.hw_lfo )
    {
      CLI( &flags );
      gus_poke( card, lfo_addr + 0x02, (unsigned char)depth );
      gus_pokew( card, lfo_addr + 0x0a + ( ( control & 0x4000 ) >> 12 ), depth << 5 );
      gus_pokew( card, lfo_addr + 0x00, control );
      STI( &flags );
    }
#endif
  if ( card -> gf1.sw_lfo )
    {
      unsigned char *ptr = card -> gf1.lfos + lfo_addr;
      
      CLI( &flags );
      *(ptr + 0x02) = (unsigned char)depth;
      *(unsigned short *)(ptr + 0x0a + ( (control & 0x4000) >> 12 )) = depth << 5;
      *(unsigned short *)(ptr + 0x00) = control;
      STI( &flags );      
    }
}

void gus_lfo_setup( gus_card_t *card, int voice, int lfo_type, int freq, int current_depth, int depth, int sweep, int shape )
{
  struct GUS_STRU_IW_LFO_PROGRAM program;

  program.freq_and_control = 0x8000 | ( freq & 0x7ff );
  if ( shape & GUS_STRU_IW_LFO_SHAPE_POSTRIANGLE )
    program.freq_and_control |= 0x2000;
  if ( depth < 0 )
    {
      program.freq_and_control |= 0x1000;
      depth = -depth;
    }
  program.twave = 0;
  program.depth = current_depth;
  program.depth_final = depth;
  if ( sweep )
    {
      program.depth_inc = (unsigned char)(((int)((depth<<5)-current_depth)<<9)/(sweep*4410L));
      if ( !program.depth_inc ) program.depth_inc++;
    }
   else
    program.depth = (unsigned short)(depth << 5);
  gus_lfo_program( card, voice, lfo_type, &program );
}

void gus_lfo_shutdown( gus_card_t *card, int voice, int lfo_type )
{
  unsigned long flags;
  unsigned int lfo_addr;

  lfo_addr = ( voice << 5 ) | ( lfo_type << 4 );
#ifdef GUSCFG_INTERWAVE
  if ( card -> gf1.hw_lfo )
    {
      gus_pokew( card, lfo_addr + 0x00, 0x0000 );
      CLI( &flags );
      gf1_select_voice( card, voice );
      gus_write8( card, lfo_type == GUS_LFO_VIBRATO ? GF1_VB_FREQUENCY_LFO : GF1_VB_VOLUME_LFO, 0 );
      STI( &flags );
    }
#endif
  if ( card -> gf1.sw_lfo )
    {
      unsigned char *ptr = card -> gf1.lfos + lfo_addr;
      gus_voice_t *pvoice;
    
      *(unsigned short *)(ptr + 0x00) = 0;
      *(unsigned short *)(ptr + 0x04) = 0;
      *(unsigned short *)(ptr + 0x06) = 0;
      if ( card -> gf1.syn_voices )
        {
          pvoice = card -> gf1.syn_voices + voice;
          if ( lfo_type == GUS_LFO_VIBRATO )
            pvoice -> lfo_fc = 0;
           else
            pvoice -> lfo_volume = 0;
          gus_lfo_register_setup( card, pvoice, lfo_type == GUS_LFO_VIBRATO ? 1 : 2 );
        }
       else
      if ( card -> gf1.enh_mode )
        {
          CLI( &flags );
          gf1_select_voice( card, voice );
          gus_write8( card, lfo_type == GUS_LFO_VIBRATO ? GF1_VB_FREQUENCY_LFO : GF1_VB_VOLUME_LFO, 0 );
          STI( &flags );
        }
    }
}

void gus_lfo_command( gus_card_t *card, int voice, unsigned char *data )
{
  int lfo_type;
  int lfo_command;
  int temp1, temp2;

  lfo_type = *data >> 7;
  lfo_command = *data & 0x7f;
  switch ( lfo_command ) {
    case GUS_LFO_SETUP:		/* setup */
      temp1 = gus_get_word( data, 2 );
      temp2 = gus_get_word( data, 4 );
      gus_lfo_setup( card, voice, lfo_type, temp1 & 0x7ff, 0, temp2 > 255 ? 255 : temp2, gus_get_byte( data, 1 ), ( temp1 & 0x2000 ) >> 13 );
      break;
    case GUS_LFO_FREQ:		/* freq */
      gus_lfo_change_depth( card, voice, lfo_type, gus_get_word( data, 2 ) );
      break;
    case GUS_LFO_DEPTH:		/* depth */
      gus_lfo_change_freq( card, voice, lfo_type, gus_get_word( data, 2 ) );
      break;
    case GUS_LFO_ENABLE:	/* enable */
      gus_lfo_enable( card, voice, lfo_type );
      break;
    case GUS_LFO_DISABLE:	/* disable */
      gus_lfo_disable( card, voice, lfo_type );
      break;
    case GUS_LFO_SHUTDOWN:	/* shutdown */
      gus_lfo_shutdown( card, voice, lfo_type );
      break;
  }
}
