//
//  Mixer.app
// 
//  Copyright (c) 1998 Per Liden
// 
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, 
//  USA.
//

#include "Mixer.h"

#include "pixmaps/mixer_bg.xpm"
#include "pixmaps/mixer_button.xpm"
#include "pixmaps/mixer_redled.xpm"


Display* Dpy;
Window Root;
Mixer* App;
const int ButtonX[] = {6, 24, 42};


void VolumeUpdate(int dummy)
{   
    App->GetVolume();
    signal(SIGALRM, VolumeUpdate);
}


Mixer::Mixer(int argc, char** argv)
{
    struct 
    {
	char name[16];
	int dev;
    } mixernames[SOUND_MIXER_NRDEVICES] = { 
	{ "VOLUME", SOUND_MIXER_VOLUME },
	{ "BASS", SOUND_MIXER_BASS },
	{ "TREBLE", SOUND_MIXER_TREBLE },
	{ "SYNTH", SOUND_MIXER_SYNTH },
	{ "PCM", SOUND_MIXER_PCM },
	{ "SPEAKER", SOUND_MIXER_SPEAKER },
	{ "LINE", SOUND_MIXER_LINE },
	{ "MIC", SOUND_MIXER_MIC },
	{ "CD", SOUND_MIXER_CD },
	{ "IMIX", SOUND_MIXER_IMIX },
	{ "ALTPCM", SOUND_MIXER_ALTPCM },
	{ "RECLEV", SOUND_MIXER_RECLEV },
	{ "IGAIN", SOUND_MIXER_IGAIN },
	{ "OGAIN", SOUND_MIXER_OGAIN },
	{ "LINE1", SOUND_MIXER_LINE1 },
	{ "LINE2", SOUND_MIXER_LINE2 },
	{ "LINE3", SOUND_MIXER_LINE3 }
    };

    int opt;
    char* dispname = NULL;

    // Init.
    App = this;
    Error = 0;
    VolumeDevice[0] = -1;
    VolumeDevice[1] = -1;
    VolumeDevice[2] = -1;

    // Parse command line
    while ((opt = getopt(argc, argv, "d:1:2:3:vh?")) != EOF) 
    {
        switch (opt) 
	{
        case 'd':       // Display
            dispname = optarg;      
            break;          
	
	case '1':	// Mixer device
	case '2':
	case '3':
	    for(int i=0; i<SOUND_MIXER_NRDEVICES; i++)	    
		if(!strcasecmp(mixernames[i].name, optarg))		
		    VolumeDevice[opt-'1'] = mixernames[i].dev;		   	    
	    if(VolumeDevice[opt-'1'] == -1)
	    {
		cerr << APPNAME << ": unknown mixer device " << optarg << endl;
		Error = 1;
	    }
	    break;

        case 'v':       // Version
            cerr << APPNAME << " version " << VERSION << endl;
	    exit(0);
	    break;
        case '?':       // Ilegal
            cerr << endl;
        case 'h':   // Help
	    cerr << "usage:  " << APPNAME << " [-options]" << endl
                 << "options:" << endl
		 << " -1 <mixer device>      set mixer device for control 1 (default VOLUME)" << endl
		 << " -2 <mixer device>      set mixer device for control 2 (default CD)" << endl
		 << " -3 <mixer device>      set mixer device for control 3 (default PCM)" << endl
                 << " -d <disp>              set display" << endl
                 << " -v                     print version and exit" << endl << endl
		 << "mixer devices:" << endl
		 << " VOLUME, BASS, TREBLE, SYNTH, PCM, SPEAKER, LINE, MIC, CD," << endl
		 << " IMIX, ALTPCM, RECLEV, IGAIN, OGAIN, LINE1, LINE2, LINE3." << endl
		 << endl;
            exit(0);
            break;
        }
    }
     
    // Set default if not specified in cmd-line
    if(VolumeDevice[0] == -1) VolumeDevice[0] = SOUND_MIXER_VOLUME;
    if(VolumeDevice[1] == -1) VolumeDevice[1] = SOUND_MIXER_CD;
    if(VolumeDevice[2] == -1) VolumeDevice[2] = SOUND_MIXER_PCM;

    // Open display
    if((Dpy = XOpenDisplay(dispname)) == NULL)
    {
        cerr << APPNAME << ": could not open display " << dispname << endl;
        exit(0);
    }
 
    // Get root window    
    Root = RootWindow(Dpy, DefaultScreen(Dpy));

    // Create windows
    AppWin = XCreateSimpleWindow(Dpy, Root, 1, 1, 10, 10, 0, 0, 0);
    IconWin = XCreateSimpleWindow(Dpy, AppWin, 1, 1, 10, 10, 0, 0, 0);
        
    // Set classhint
    XClassHint classhint;   
    classhint.res_name =  "mixer_app";
    classhint.res_class = "Mixer_app";
    XSetClassHint(Dpy, AppWin, &classhint);        

    // Create delete atom
    Atom deletewin = XInternAtom(Dpy, "WM_DELETE_WINDOW", False);
    XSetWMProtocols(Dpy, AppWin, &deletewin, 1);
    XSetWMProtocols(Dpy, IconWin, &deletewin, 1);

    // Set windowname
    XStoreName(Dpy, AppWin, APPNAME);
    XSetIconName(Dpy, AppWin, APPNAME);

    // Set sizehints
    XSizeHints sizehints;
    sizehints.flags= USPosition;
    sizehints.x = 0;
    sizehints.y = 0;
    XSetWMNormalHints(Dpy, AppWin, &sizehints);

    // Set wmhints
    XWMHints wmhints;
    wmhints.initial_state = WithdrawnState;
    wmhints.icon_window = IconWin;
    wmhints.icon_x = 0;
    wmhints.icon_y = 0;
    wmhints.window_group = AppWin;
    wmhints.flags = StateHint | IconWindowHint | IconPositionHint | WindowGroupHint;
    XSetWMHints(Dpy, AppWin, &wmhints);

    // Set command
    XSetCommand(Dpy, AppWin, argv, argc); 

    // Set background image
    Xpm* image = new Xpm(mixer_bg);
    image->SetWindowPixmapShaped(IconWin);
    delete image;

    // Create buttons
    Button[0] = XCreateSimpleWindow(Dpy, IconWin, ButtonX[0], BUTTON_MAX, 5, 5, 0, 0, 0);
    Button[1] = XCreateSimpleWindow(Dpy, IconWin, ButtonX[1], BUTTON_MAX, 5, 5, 0, 0, 0);
    Button[2] = XCreateSimpleWindow(Dpy, IconWin, ButtonX[2], BUTTON_MAX, 5, 5, 0, 0, 0);

    Xpm* buttonimage = new Xpm(mixer_button);
    buttonimage->SetWindowPixmap(Button[0]);
    buttonimage->SetWindowPixmap(Button[1]);
    buttonimage->SetWindowPixmap(Button[2]);
    delete buttonimage;

    XSelectInput(Dpy, Button[0], ButtonPressMask | ButtonReleaseMask | PointerMotionMask);
    XSelectInput(Dpy, Button[1], ButtonPressMask | ButtonReleaseMask | PointerMotionMask);
    XSelectInput(Dpy, Button[2], ButtonPressMask | ButtonReleaseMask | PointerMotionMask);

    XMapWindow(Dpy, Button[0]);
    XMapWindow(Dpy, Button[1]);
    XMapWindow(Dpy, Button[2]);

    // Show window
    XMapWindow(Dpy, AppWin);
    
    // Check if error
    if(Error)
	ShowErrorLed();
    else
    {
	// Get volume status
	GetVolume();

	// Set update timer
	SetTimer(TIMERINTERVAL);
    }
}


Mixer::~Mixer()
{
    // Never called
}


void Mixer::ShowErrorLed()
{
    Window ledwin;
    ledwin = XCreateSimpleWindow(Dpy, IconWin, LED_X, LED_Y, 8, 5, 0, 0, 0);

    // Set background image
    Xpm* image = new Xpm(mixer_redled);
    image->SetWindowPixmap(ledwin);
    delete image;

    // Show window
    XMapWindow(Dpy, ledwin);
    Error = 1;
}


void Mixer::SetTimer(int interval)
{
    itimerval i;

    i.it_value.tv_usec = interval;
    i.it_value.tv_sec = 0;
    i.it_interval.tv_usec = interval;
    i.it_interval.tv_sec = 0;

    signal(SIGALRM, VolumeUpdate);
    setitimer(ITIMER_REAL, &i, 0);
}


void Mixer::GetVolume()
{    
    static prev_vol[3] = {0, 0, 0};

    if(Error)
  	return;

    // Open device
    int fd = open(DEVICE, 0);    
    if(fd < 0)
    {
	cerr << APPNAME << ": unable to open " << DEVICE << endl;
	ShowErrorLed();
	return;
    }

    // Read from device
    for(int i=0; i<3; i++)
    {
	if(ioctl(fd, MIXER_READ(VolumeDevice[i]), &Volume[i]) < 0)
	    Error = 1;

	Volume[i] = Volume[i] >> 8;

	if(prev_vol[i] != Volume[i])
	{
	    // Set button position
	    if(Error)
		XMoveWindow(Dpy, Button[i], ButtonX[i], BUTTON_MAX);
	    else
		XMoveWindow(Dpy, Button[i], ButtonX[i], 
			    BUTTON_MAX - (Volume[i] * (BUTTON_MAX - BUTTON_MIN)) / 100);
	    XFlush(Dpy);
	    prev_vol[i] = Volume[i];
	}	
    }

    // Close device
    close(fd);                 
    
    if(Error)
    {
	cerr << APPNAME << ": unable to read from " << DEVICE << endl;
	ShowErrorLed();
	return;
    }
}


void Mixer::SetVolume(int button, int volume)
{
    if(Error)
  	return;
    
    // Open device
    int fd = open(DEVICE, 0);

    // Calculate volume
    Volume[button] = 100 - (((volume - BUTTON_MIN) * 100) / (BUTTON_MAX - BUTTON_MIN));
    Volume[button] |= Volume[button] << 8;

    // Write to device
    ioctl(fd, MIXER_WRITE(VolumeDevice[button]), &Volume[button]);

    // Close devicw
    close(fd);
}


void Mixer::Run()
{
    XEvent event;
    int buttondown = 0;
    int buttondownpos = 0;

    // Start looping
    while(1)
    {   
	XNextEvent(Dpy, &event);

	switch(event.type)
	{   
	// On button press
	case ButtonPress:
	    buttondown = 1;
	    buttondownpos = event.xbutton.y;
	    SetTimer(0);
	    break;
	
	// On button release
	case ButtonRelease:
	    buttondown = 0;
	    SetTimer(TIMERINTERVAL);
	    break;
	
	// On mouse motion
	case MotionNotify:
	    if(buttondown)
	    {
		// Find button
		for(int i=0; i<3; i++)		
		    if(Button[i] == event.xmotion.window)
		    {
			// Get current button position
			int x, y;
			Window dummy_win;
			unsigned int dummy_int;
			XGetGeometry(Dpy, Button[i], &dummy_win, &x, &y, 
				     &dummy_int, &dummy_int, &dummy_int, &dummy_int); 

			// Calc new button position
			y = y + event.xmotion.y - buttondownpos;
			if(y > BUTTON_MAX) y = BUTTON_MAX;
			if(y < BUTTON_MIN) y = BUTTON_MIN;
			
			// Set button position and volume
			XMoveWindow(Dpy, Button[i], ButtonX[i], y);
			SetVolume(i, y);
			break;		       
		    }
	    }
	    break;
	}
    }
}

