/*==============================================================================

  $Id: mikmod.c,v 1.27 1998/09/20 23:22:13 miod Exp $

  Module player example of MikMod

==============================================================================*/

/*
	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
 
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.
 
	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <unistd.h>
#include <sys/signal.h>
#ifdef HAVE_GNU_GETOPT
#include <getopt.h>
#else
#include "../extra/getopt.h"
#endif
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifdef NEED_USLEEP
void usleep(unsigned long);
#endif

#ifdef HAVE_NCURSES
#include <ncurses.h>
#else
#if defined(__FreeBSD__)||defined(__NetBSD__)
#include <ncurses.h>
#else
#include <curses.h>
#endif
#endif

#include <mikmod.h>

#include "player.h"

#ifdef CPUTIME_SNAGGER
#if defined(__FreeBSD__)||defined(__NetBSD__)||defined(__OpenBSD__)
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/types.h>
#ifdef __FreeBSD__
#include <sys/rtprio.h>
#endif
#else /* __linux */
#ifdef BROKEN_SCHED
#include <linux/sched.h>
#else
#include <sched.h>
#endif
#endif
#endif

PLAYLIST playlist;
#ifdef CPUTIME_SNAGGER
#ifdef __FreeBSD__
struct rtprio rtp;
#endif
#endif

char helptext[]=
"Available switches:\n"
"-d[river] n             Use nth driver for output. Default is 0 (autodetect).\n"
"-8, -ei[ghtbits]        Force 8 bit output.\n"
"-F, -fa[deout]          Force volume fade at the end of the module.\n"
"-f[requency] nnnnn      Sets mixing frequency. Default is 44100.\n"
#ifndef VIRTCH2
"-i[nterpolate]          Use interpolate mixing.\n"
#endif
"-k, -o[nce]             Play files only once.\n"
"-m[ono]                 Force mono output.\n"
"-n[oloops]              Disables in-module loops.\n"
"-p, -pa[nning] nn       Sets default panning (0=left,255=right,128=center).\n"
"-pl[aylist] file        Use playlist.\n"
"-x, -pr[otracker]       Disable extended protracker effects.\n"
"-q[uiet]                Quiet mode, no interface, displays only errors.\n"
"-ra[ndom]               Random file play.\n"
#ifdef CPUTIME_SNAGGER
"-s, -ren[ice]           Renice to -20 (more scheduling priority).\n"
#ifndef __OpenBSD__
"-S, -rea[ltime]         Get realtime priority (will hog CPU power).\n"
#endif
#endif
"-r[epeat]               Repeat module after it's finished.\n"
"-e, -rev[erb] nn        Set reverb amount. Default is 0 (none), max is 15.\n"
"-te[rse]                Terse output : status line only.\n"
"-t[olerant]             Tolerant mode : don't halt on file access errors.\n"
"-v[olume] nn            Sets volume from 0% (silence) to 100%. Default is 100%.\n"
"\n"
"-l, -inf[ormation]      List all available drivers and module loaders.\n"
"\n"
"-ve[rsion]              Display MikMod version.\n"
"-h[elp]                 Display this help screen.\n";

/* Long options definition */
struct option options[]={
	{"driver",		required_argument,	NULL,'d'},
	{"eightbits",	no_argument,		NULL,'8'},
    {"fadeout",		no_argument,		NULL,'F'},
	{"frequency",	required_argument,	NULL,'f'},
	{"interpolate",	no_argument,		NULL,'i'},
	{"mono",		no_argument,		NULL,'m'},
	{"noloops",		no_argument,		NULL,'n'},
	{"once",		no_argument,		NULL,'k'},
	{"panning",		optional_argument,	NULL,'p'},
	{"playlist",	required_argument,	NULL,  2},	/* pl */
	{"protracker",	no_argument,		NULL,'x'},
	{"quiet",		no_argument,		NULL,'q'},
	{"random",		no_argument,		NULL,  1},	/* rp */
	{"realtime",	no_argument,		NULL,'S'},
	{"renice",		no_argument,		NULL,'s'},
	{"repeat",		no_argument,		NULL,'r'},
	{"reverb",		required_argument,	NULL,'e'},
	{"terse",		no_argument,		NULL,  3},  /* so */
	{"tolerant",	no_argument,		NULL,'t'},
	{"volume",		required_argument,	NULL,'v'},
	{"information",	no_argument,		NULL,'l'},
	{"version",		no_argument,		NULL,'V'},
	{"help",		no_argument,		NULL,'h'},
/* compatibility aliases */
	{"rp",			no_argument,		NULL,  1},
	{"so",			no_argument,		NULL,  3},
	{NULL,			0,					NULL,  0}
};

/* current module */
UNIMOD *mf=NULL;

/* module parameters */
BOOL cfg_extspd=1;	/* Extended Speed enable */
BOOL cfg_panning=1;	/* DMP panning enable (8xx effects) */
BOOL cfg_wrap=0;	/* auto song-wrapping disable */
BOOL cfg_loop=1;	/* allow module to loop */
BOOL cfg_fade=0;	/* allow volume fade during last pattern */

/* set if quiet mode is enabled */
int quiet=0;
int semiquiet=0;

int player_on=0;

/* playlist handling */
int dorandom=0,playonce=0;
BOOL next;

/* signal handlers */
void GotoNext(int signum)
{
	next=1;

	signal(SIGUSR1, GotoNext);
}

void GotoPrev(int signum)
{
	PL_GetPrev(&playlist,NULL,NULL);
	PL_GetPrev(&playlist,NULL,NULL);
	next=1; 

	signal(SIGUSR2, GotoPrev);
}

void ExitGracefully(int signum)
{
	if (player_on) MikMod_Exit();
	if (curses_on) exit_display();

	if(!quiet)
		fputs((signum==SIGTERM)?"Halted by SIGTERM\n":"Halted by SIGINT\n",stderr);

	signal(SIGINT,SIG_DFL);
	signal(SIGTERM,SIG_DFL);
	exit(0);
}

int main(int argc,char *argv[])
{
	int t,quit=0,finished=0;
	char filename[PATH_MAX];
	char archive[PATH_MAX];
	char *playfile;
	int cfg_maxchn=64,c;
	int tolerant=0;

	/* Initialize soundcard parameters.. you _have_ to do this before calling
	   MD_Init(), and it's illegal to change them after you've called MD_Init()
	*/
	md_mixfreq      = 44100;	/* standard mixing freq */
	md_dmabufsize   = 32768;	/* standard dma buf size (max 32768) */
	md_device       = 0;		/* standard device: autodetect */
	md_volume       = 96;		/* driver volume (max 128) */
	md_musicvolume  = 128;		/* music volume (max 128) */
	md_sndfxvolume  = 128;		/* sound effects volume (max 128) */
	md_pansep       = 128;		/* panning separation (0=mono 128=full stereo)*/
/*	md_stereodelay  = 10;		*//* Stereo Delay (max 15) */
	md_reverb       = 0;		/* Reverb (max 15) */
	/* default mixing mode */
	md_mode			= DMODE_16BITS | DMODE_STEREO | DMODE_SOFT_MUSIC;

	/* Register the loaders we want to use..  */
	MikMod_RegisterAllLoaders();

	/* Register the drivers we want to use: */
	MikMod_RegisterAllDrivers();

	PL_InitList(&playlist);

	/* Getopt comes back ! */
	opterr=0;
	while((t=getopt_long_only(argc,argv,"8Fiklmnqrtxd:e:f:p::v:",
	       options,NULL))!=ERR) {
		switch(t) {
			case   1: /* --random */
				dorandom=1;
				break;
			case   2: /* --playlist xxx */
				PL_Load(&playlist,optarg);
				break;
			case   3: /* --terse */
				semiquiet=1;
				break;
			case '8': /* -8 --eightbits */
				md_mode&=~DMODE_16BITS;
				break;
			case 'F': /* -F --fadeout */
				cfg_fade=1;
				break;
			case 'i': /* -i --interpolate */
#ifndef VIRTCH2
				md_mode|=DMODE_INTERP;
#endif
				break;
			case 'k': /* -k --once */
				playonce=1;
				break;
			case 'm': /* -m --mono */
				md_mode&=~DMODE_STEREO;
				break;
			case 'n': /* -n --noloops */
				cfg_loop=0;
				break;
			case 'q': /* -q --quiet */
				quiet=1;
				break;
			case 'r': /* -r --repeat */
				cfg_wrap=1;
				break;
			case 's': /* -s --renice */
#ifdef CPUTIME_SNAGGER
#if defined(__FreeBSD__)||defined(__NetBSD__)||defined(__OpenBSD__)
				if(setpriority(PRIO_PROCESS,0,-20)==-1)
					perror("renice to -20");
#else
				if (nice(-20)==-1)
					perror("renice to -20");
#endif
#endif
				break;
			case 'S': /* -S --realtime */
#ifdef CPUTIME_SNAGGER
#ifdef __FreeBSD__
				rtp.type=RTP_PRIO_REALTIME;
				rtp.prio=0;
				if (rtprio(RTP_SET,0,&rtp)==-1)
					perror("realtime priority");
#else
#ifdef __linux
			{
				struct sched_param sp;

				memset(&sp,0,sizeof(struct sched_param));
				sp.sched_priority=1;
				if (sched_setscheduler(0,SCHED_RR,&sp)==-1)
					perror("realtime priority");
			}
#endif
#endif
#endif
				break;
			case 't': /* -t --tolerant */
				tolerant=1;
				break;
			case 'x': /* -x --protracker */
				cfg_extspd=0;
				break;
			case 'd': /* -d --driver */
				md_device=atoi(optarg);
				break;
			case 'e': /* -e --reverb */
				md_reverb=atoi(optarg);
				break;
			case 'f': /* -f --frequency */
				md_mixfreq=atol(optarg);
				break;
			case 'v': /* -v --volume */
				md_volume=(atoi(optarg)*128)/100;
				break;
			case 'p': /* -p --panning */
				cfg_panning=optarg?atoi(optarg):0;
				break;
			case 'l': /* -l --information */
				puts(mikversion);
				printf("Available drivers are :\n%s\n\nAvailable module loaders are :\n%s\n",
				       MD_InfoDriver(),ML_InfoLoader());
				return 0;
			case 'h': /* -h --help */
				puts(mikversion);
				puts(helptext);
				return 0;
			case 'V': /* --version */
				puts(mikversion);
				return 0;
			default:
				/* ignore errors */
				break;
		}

	}

	/* if neither a module nor a playlist was specified, display help and exit */
	if((optind>=argc)&&(!playlist.numused)) {
		puts(mikversion);
		puts(helptext);
		return 0;
	}

	/* Add remaining parameters to the playlist */
	for(t=optind;t<argc;t++)
		MA_FindFiles(&playlist,argv[t]);
	if (dorandom)
		PL_Randomize(&playlist);

	if (!quiet)
		puts(mikbanner);

	if (!playlist.numused) {
		puts("No file in playlist...");
		return 0;
	}

	/*  initialize volume and tempo multipliers */
	if (MikMod_Init()) {
		fprintf(stderr,"MikMod initialisation error : %s\n",_mm_errmsg[_mm_errno]);
		return 1;
	}
	player_on=1;

	/*  initialize interface */
	init_display();

	signal(SIGTERM,ExitGracefully);
	signal(SIGINT,ExitGracefully);

	while(!quit) {
		memset(filename,0,sizeof(filename));	
		memset(archive,0,sizeof(archive));	
		if((PL_GetNext(&playlist,filename,archive))) {
			finished=1;
			break;
		}
		if(!filename[0]) {
			quit=1;
			break;
		}

		/* load the module */
		playfile=MA_dearchive(archive,filename);
		display_loadbanner();
		if(!(mf=MikMod_LoadSong(playfile,cfg_maxchn))) {
			/* didn't work -> exit with errormsg. */
			if(!tolerant) {
				exit_display();
				if(archive[0]) {
					unlink(playfile);
					fprintf(stderr,"MikMod error : can't load %s from archive "
					    "%s\n(reason: %s)\n",
				        filename,archive,_mm_errmsg[_mm_errno]);
				} else
					fprintf(stderr,"MikMod error : can't load %s\n"
					    "(reason: %s)\n",
				        playfile,_mm_errmsg[_mm_errno]);
				free(playfile);
				MikMod_Exit();
				player_on=0;
				exit(1);
			} else {
				if(archive[0]) unlink(playfile);
				free(playfile);
				PL_DelCurrent(&playlist);
				continue;
			}
		}
		if(archive[0]) unlink(playfile);
		free(playfile);

		mf->extspd  = cfg_extspd;
		mf->panflag = cfg_panning;
		mf->wrap    = cfg_wrap;
		mf->loop    = cfg_loop;
		mf->fadeout = cfg_fade;

		Player_Start(mf);

		firstinst=0; 
		display_all();

		next=0;
		signal(SIGUSR1, GotoNext);
		signal(SIGUSR2, GotoPrev);

		/* if we have a quit signal, exit loop */
		while(Player_Active() && !quit && !next) {
			MikMod_Update();
#ifdef VIRTCH2
			usleep(200);
#else
			usleep(500);
#endif

			/* update the status display... */
			if(!Player_Paused())
				display_status();

			if(!quiet && (c=getch())!=ERR) {
				/* toggle pause */
				if (c==' ') {
					if(Player_Paused())
						display_all();
					else
						display_pausebanner();
					Player_TogglePause();
				}
				if (!Player_Paused())
					switch(c) {
						case '+': 
						case KEY_RIGHT:
							Player_NextPosition();
							break;
						case '-': 
						case KEY_LEFT:
							Player_PrevPosition();
							break;
						case '(':
							Player_SetSpeed(mf->sngspd-1);
							break;
						case ')':
							Player_SetSpeed(mf->sngspd+1);
							break;
						case '{':
							Player_SetTempo(mf->bpm-1);
							break;
						case '}':
							Player_SetTempo(mf->bpm+1);
							break;
#ifndef VIRTCH2
						case ':':
							md_mode^=DMODE_INTERP;
							display_driver();
							break;
#endif
						case 'N':
						case 'n':
							next=1;
							break;
						case 'p':
						case 'P':
							PL_GetPrev(&playlist,filename,archive);
							PL_GetPrev(&playlist,filename,archive);
							next=1;
							break;
						case 'q':
						case 'Q':
							quit=1;
							break;
						case KEY_DOWN:
							firstinst++;
							display_information();
							break;
						case KEY_UP:
							if (firstinst) {
								firstinst--;
								display_information();
							}
							break;
						case KEY_NPAGE:
							firstinst+=(winy-6);
							display_information();
							break;
						case KEY_PPAGE:
							firstinst-=(winy-6);
							display_information();
							break;
						case KEY_F(1):
						case KEY_F(2):
						case KEY_F(3):
						case KEY_F(4):
						case KEY_F(5):
						case KEY_F(6):
						case KEY_F(7):
						case KEY_F(8):
						case KEY_F(9):
						case KEY_F(10):
							Player_SetVolume((((c-KEY_F(0))<<7)+5)/10);
							break;
						case 'w':
						case 'W':
							PL_Save(&playlist, "./mikmodplaylist");
							break;
						case 'i':
						case 'I':
							firstinst=0;
							dispsamp=(dispsamp&1)^1;
							display_all();
							break;
						case 'm':
						case 'M':
						case KEY_F(19): /* shift-F9 */
							if(mf->comment) {
								firstinst=0;
								if(dispsamp&4)
									dispsamp=(dispsamp&3)|2;
								else
									dispsamp=(dispsamp&3)^2;
								display_information();
							}
							break;
						case 'h':
						case 'H':
							if (dispsamp&4)
								dispsamp=(dispsamp&~4)|8;
							else
								dispsamp=(dispsamp&~8)|4;
							display_information();
							break;
						case 12: /* ^L */
						case KEY_CLEAR:
							display_all();
							break;
					}
				if(!quiet) flushinp();
			}
		}
		Player_Stop();			/* stop playing */
		MikMod_FreeSong(mf);	/* and free the module */
		if (playonce) PL_DelCurrent(&playlist);
	}

	MikMod_Exit();
	player_on=0;
	exit_display();
	if ((!quit)&&(!quiet)) {
		if(!finished) {
			fprintf(stderr,"MikMod error : %s\n",_mm_errmsg[_mm_errno]);
			return 1;
		} else
			puts("Finished playlist...");
	}
	return 0;
}
