/*==============================================================================

  $Id: load_ult.c,v 1.4 1998/09/20 21:45:21 miod Exp $

  Ultratracker (ULT) module loader

==============================================================================*/

/*
	This library is free software; you can redistribute it and/or modify
	it under the terms of the GNU Library General Public License as
	published by the Free Software Foundation; either version 2 of
	the License, or (at your option) any later version.
 
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU Library General Public License for more details.
 
	You should have received a copy of the GNU Library General Public
	License along with this library; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <string.h>

#include <mikmod.h>

/*========== Module structure */

/* header */
typedef struct ULTHEADER {
	CHAR  id[16];
	CHAR  songtitle[32];
	UBYTE reserved;
} ULTHEADER;

/* sample information */
typedef struct ULTSAMPLE {
	CHAR  samplename[32];
	CHAR  dosname[12];
	SLONG loopstart;
	SLONG loopend;
	SLONG sizestart;
	SLONG sizeend;
	UBYTE volume;
	UBYTE flags;
	SWORD finetune;
} ULTSAMPLE;

typedef struct ULTEVENT {
	UBYTE note,sample,eff,dat1,dat2;
} ULTEVENT;

/*========== Loader variables */

#define ULTS_16BITS     4
#define ULTS_LOOP       8
#define ULTS_REVERSE    16

CHAR *ULT_Version[]={
	"Ultra Tracker V1.3",
	"Ultra Tracker V1.4",
	"Ultra Tracker V1.5",
	"Ultra Tracker V1.6"
};

static ULTEVENT ev;

/*========== Loader code */

BOOL ULT_Test(void)
{
	CHAR id[16];

	if(!_mm_read_string(id,15,modfp)) return 0;
	return(!strncmp(id,"MAS_UTrack_V00",14));
}

BOOL ULT_Init(void)
{
	return 1;
}

void ULT_Cleanup(void)
{
}

UBYTE ReadUltEvent(ULTEVENT* event)
{
	UBYTE flag,rep=1;

	flag = _mm_read_UBYTE(modfp);
	if(flag==0xfc) {
		rep = _mm_read_UBYTE(modfp);
		event->note =_mm_read_UBYTE(modfp);
	} else
		event->note = flag;

	event->sample   =_mm_read_UBYTE(modfp);
	event->eff      =_mm_read_UBYTE(modfp);
	event->dat1     =_mm_read_UBYTE(modfp);
	event->dat2     =_mm_read_UBYTE(modfp);

	return rep;
}

BOOL ULT_Load(void)
{
	int t,u,tracks=0;
	SAMPLE *q;
	ULTSAMPLE s;
	ULTHEADER mh;
	UBYTE nos,noc,nop;

	/* try to read module header */
	_mm_read_string(mh.id,15,modfp);
	_mm_read_string(mh.songtitle,32,modfp);
	mh.reserved=_mm_read_UBYTE(modfp);

	if(feof(modfp)) {
		_mm_errno = MMERR_LOADING_HEADER;
		return 0;
	}

	if((mh.id[14]<'1')||(mh.id[14]>'4')) {
		_mm_errno = MMERR_NOT_A_MODULE;
		return 0;
	}

	of.modtype   = strdup(ULT_Version[mh.id[14]-'1']);
	of.initspeed = 6;
	of.inittempo = 125;

	/* read songtext */
	if (mh.id[14]>'1')
		if(!ReadLinedComment((UWORD)mh.reserved,32)) return 0;

	nos=_mm_read_UBYTE(modfp);
	if(feof(modfp)) {
		_mm_errno = MMERR_LOADING_HEADER;
		return 0;
	}

	of.songname=DupStr(mh.songtitle,32);
	of.numins=of.numsmp=nos;

	if(!AllocSamples()) return 0;
	q = of.samples;
	for(t=0;t<nos;t++) {
		/* try to read sample info */
		_mm_read_string(s.samplename,32,modfp);
		_mm_read_string(s.dosname,12,modfp);
		s.loopstart     =_mm_read_I_ULONG(modfp);
		s.loopend       =_mm_read_I_ULONG(modfp);
		s.sizestart     =_mm_read_I_ULONG(modfp);
		s.sizeend       =_mm_read_I_ULONG(modfp);
		s.volume        =_mm_read_UBYTE(modfp);
		s.flags         =_mm_read_UBYTE(modfp);
		s.finetune      =_mm_read_I_SWORD(modfp);

		if(feof(modfp)) {
			_mm_errno = MMERR_LOADING_SAMPLEINFO;
			return 0;
		}

		q->samplename=DupStr(s.samplename,32);
		q->speed=8363;
		if(mh.id[14]>='4') {
			_mm_read_I_UWORD(modfp);	/* read 1.6 extra info(??) word */
			q->speed=s.finetune;
		}
		q->length    = s.sizeend-s.sizestart;
		q->volume    = s.volume>>2;
		q->loopstart = s.loopstart;
		q->loopend   = s.loopend;
		q->flags = SF_SIGNED;
		if(s.flags&ULTS_LOOP) q->flags|=SF_LOOP;
		if(s.flags&ULTS_16BITS) {
			s.sizeend+=(s.sizeend-s.sizestart);
			s.sizestart<<=1;
			q->flags|=SF_16BITS;
			q->loopstart>>=1;
			q->loopend>>=1;
		}
		q++;
	}

	if(!AllocPositions(256)) return 0;
	for(t=0;t<256;t++)
		of.positions[t]=_mm_read_UBYTE(modfp);
	for(t=0;t<256;t++)
		if(of.positions[t]==255) break;
	of.numpos=t;

	noc=_mm_read_UBYTE(modfp);
	nop=_mm_read_UBYTE(modfp);

	of.numchn=++noc;
	of.numpat=++nop;
	of.numtrk=of.numchn*of.numpat;
	if(!AllocTracks()) return 0;
	if(!AllocPatterns()) return 0;
	for(u=0;u<of.numchn;u++)
		for(t=0;t<of.numpat;t++)
			of.patterns[(t*of.numchn)+u]=tracks++;

	/* read pan position table for v1.5 and higher */
	if(mh.id[14]>='3')
		for(t=0;t<of.numchn;t++) of.panning[t]=_mm_read_UBYTE(modfp)<<4;

	for(t=0;t<of.numtrk;t++) {
		int rep,row=0;

		UniReset();
		while(row<64) {
			rep=ReadUltEvent(&ev);

			if(feof(modfp)) {
				_mm_errno = MMERR_LOADING_TRACK;
				return 0;
			}

			while(rep--) {
				UBYTE eff;

				if(ev.sample) UniInstrument(ev.sample-1);
				if(ev.note)   UniNote(ev.note+23);
				eff = ev.eff>>4;

				/*  ULT panning effect fixed by Alexander Kerkhove : */
				if(eff==0xc) UniPTEffect(eff,ev.dat2>>2);
				else if(eff==0xb) UniPTEffect(8,ev.dat2*0xf);
				else UniPTEffect(eff,ev.dat2);

				eff=ev.eff&0xf;
				if(eff==0xc) UniPTEffect(eff,ev.dat1>>2);
				else if(eff==0xb) UniPTEffect(8,ev.dat1*0xf);
				else UniPTEffect(eff,ev.dat1);

				UniNewline();
				row++;
			}
		}
		if(!(of.tracks[t]=UniDup())) return 0;
	}
	return 1;
}

CHAR *ULT_LoadTitle(void)
{
	CHAR s[32];

	_mm_fseek(modfp,15,SEEK_SET);
	if(!fread(s,32,1,modfp)) return NULL;

	return(DupStr(s,32));
}

/*========== Loader information */

MLOADER load_ult={
	NULL,
	"ULT",
	"ULT loader v0.1",
	ULT_Init,
	ULT_Test,
	ULT_Load,
	ULT_Cleanup,
	ULT_LoadTitle
};

