/*
 *   ALSA sequencer Timing queue handling
 *   Copyright (c) 1998 by Frank van de Pol <F.K.W.van.de.Pol@inter.nl.net>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "driver.h"

#include "seq_memory.h"
#include "seq_queue.h"
#include "seq_clientmgr.h"
#include "seq_fifo.h"
#include "seq_timer.h"
#include "seq_info.h"


/* root for the queues */
static queue_t *queuetab[SND_SEQ_MAX_QUEUES];

/* number of queues allocated */
static int num_queues = 0;


/* FIXME: should add list of active queues (for optimisation of timer), sort of 'run queue' */



/* setup queues */
void snd_seq_queues_init(int num)
{
	int c;

	if (num > SND_SEQ_MAX_QUEUES) {
		snd_printk("Limiting number of timing queues to %d (SND_SEQ_MAX_QUEUES)\n", SND_SEQ_MAX_QUEUES);
		num = SND_SEQ_MAX_QUEUES;
	}
	num_queues = num;

	for (c = 0; c < num; c++) {
		queuetab[c] = snd_seq_queue_new();
	}
}

/* delete queues */
void snd_seq_queues_delete(void)
{
	int c;

	for (c = 0; c < num_queues; c++)
		snd_seq_queue_delete(&queuetab[c]);

	num_queues = 0;
}


/* create new queue (constructor) */
queue_t *snd_seq_queue_new(void)
{
	queue_t *q;

	q = snd_malloc(sizeof(queue_t));
	if (q == NULL) {
		snd_printk("malloc failed for snd_seq_queue_new()\n");
		return NULL;
	}
	/* clear unused variables */
	memset(q, 0, sizeof(queue_t));

	q->tickq = snd_seq_prioq_new();
	q->timeq = snd_seq_prioq_new();
	q->timer = snd_seq_timer_new();

	q->owner = SEQ_QUEUE_NO_OWNER;
	q->locked = 0;

	return q;
}

/* delete queue (destructor) */
void snd_seq_queue_delete(queue_t ** queue)
{
	queue_t *q = *queue;

	*queue = NULL;

	if (q == NULL) {
		snd_printk("oops: snd_seq_queue_delete() called with NULL queue\n");
		return;
	}
	if (q->owner != SEQ_QUEUE_NO_OWNER) {
		snd_printk("warning: snd_seq_queue_delete(), timer is still owned by client #%d\n", q->owner);
	}
	q->locked = 0;

	/* release resources... */
	snd_seq_prioq_delete(&q->tickq);
	snd_seq_prioq_delete(&q->timeq);
	snd_seq_timer_delete(&q->timer);

	snd_free(q, sizeof(queue_t));
}



/* return pointer to queue structure for specified id */
queue_t *queueptr(int queueid)
{
	if (queueid < 0 || queueid >= num_queues) {
		snd_printk("Seq: oops. Trying to get pointer to queue %d\n", queueid);
		return NULL;
	}
	return queuetab[queueid];
}




/* -------------------------------------------------------- */

snd_spin_define_static(chkqueues_lock);


/* check queues and dispatch events */
void snd_seq_check_queues(void)
{
	static int blocked;
	unsigned long flags;
	int c;
	int dequeue = SND_SEQ_MAX_DEQUEUE;
	snd_seq_event_cell_t *cell;

	/* make this function non-reentrant */
	snd_spin_lock_static(chkqueues_lock, &flags);
	if (blocked) {
		snd_spin_unlock_static(chkqueues_lock, &flags);
		//snd_printk("Seq: check_queues: other thread already active\n");
		return;		/* other thread is already checking queues */
	}
	blocked = 1;
	snd_spin_unlock_static(chkqueues_lock, &flags);


	/* NOTE: this is not efficient for large number of queues! */
	for (c = 0; c < SND_SEQ_MAX_QUEUES; c++) {
		queue_t *q = queueptr(c);

		if (q == NULL)
			continue;

		/* Process tick queue... */

		/* limit the number of elements dequeued per pass to save the machine from lockups */
		while (dequeue > 0) {
			cell = snd_seq_prioq_cell_peek(q->tickq);
			if (cell == NULL)
				break;
			if (snd_seq_compare_tick_time(&q->timer->cur_tick, &cell->event.time.tick)) {
				cell = snd_seq_prioq_cell_out(q->tickq);
				if (cell != NULL)
					snd_seq_dispatch_event(cell);
				dequeue--;
			} else {
				/* event remains in the queue */
				break;
			}
		}


		/* Process time queue... */

		/* limit the number of elements dequeued per pass to save the machine from lockups */
		while (dequeue > 0) {
			cell = snd_seq_prioq_cell_peek(q->timeq);
			if (cell == NULL)
				break;
			if (snd_seq_compare_real_time(&q->timer->cur_time, &cell->event.time.real)) {
				cell = snd_seq_prioq_cell_out(q->timeq);
				if (cell != NULL)
					snd_seq_dispatch_event(cell);
				dequeue--;
			} else {
				/* event remains in the queue */
				break;
			}
		}
	}

	/* free lock */
	blocked = 0;
}



/* enqueue a event to singe queue */
static void snd_seq_enqueue_single_event(snd_seq_event_cell_t * cell)
{
	int dest;
	queue_t *q;

	if (cell == NULL) {
		snd_printk("oops: snd_seq_enqueue_single_event() called with NULL cell\n");
		return;
	}
	dest = cell->event.dest.queue;	/* dest queue */
	q = queueptr(dest);
	if (q == NULL) {
		snd_printk("oops: snd_seq_enqueue_single_event() called with NULL queue\n");
		return;
	}
	/* handle relative time stamps, convert them into absolute */
	if ((cell->event.flags & SND_SEQ_TIME_MODE_MASK) == SND_SEQ_TIME_MODE_REL) {
		switch (cell->event.flags & SND_SEQ_TIME_STAMP_MASK) {
			case SND_SEQ_TIME_STAMP_TICK:
				cell->event.time.tick += q->timer->cur_tick;
				cell->event.flags &= ~SND_SEQ_TIME_STAMP_MASK;
				cell->event.flags |= SND_SEQ_TIME_STAMP_TICK;
				break;

			case SND_SEQ_TIME_STAMP_REAL:
				snd_seq_inc_real_time(&cell->event.time.real, &q->timer->cur_time);
				cell->event.flags &= ~SND_SEQ_TIME_STAMP_MASK;
				cell->event.flags |= SND_SEQ_TIME_STAMP_REAL;
				break;
		}
	}
	/* enqueue event in the real-time or midi queue */
	switch (cell->event.flags & SND_SEQ_TIME_STAMP_MASK) {
		case SND_SEQ_TIME_STAMP_TICK:
			snd_seq_prioq_cell_in(q->tickq, cell);
			break;

		case SND_SEQ_TIME_STAMP_REAL:
			snd_seq_prioq_cell_in(q->timeq, cell);
			break;
	}

	/* trigger dispatching */
	snd_seq_check_queues();

	return;
}



/* enqueue a event received from one the clients */
void snd_seq_enqueue_event(snd_seq_event_cell_t * cell)
{
	int dest;

	if (cell == NULL) {
		snd_printk("oops: snd_seq_enqueue_event() called with NULL cell\n");
		return;
	}
	dest = cell->event.dest.queue;	/* dest queue */

	/* handle special addressing types */
	switch (dest) {
		case SND_SEQ_ADDRESS_BROADCAST:
			snd_printk("Seq: broadcasts to all queues are not supported yet (dest = %d)\n", dest);

			/* release the original cell */
			snd_seq_cell_free(cell);
			break;

		case SND_SEQ_ADDRESS_SUBSCRIBERS:
			{
				client_port_t *p;
				subscribers_t *subs;

				/* get pointer to subscription list */
				p = snd_seq_get_client_port_ptr(cell->event.source.client, cell->event.source.port);

				if (p == NULL) {
					snd_printk("seq: unable to get pointer for client %d, port %d\n", cell->event.source.client, cell->event.source.port);

					/* release the original cell */
					snd_seq_cell_free(cell);
					break;
				}
				if (!(p->capability & SND_SEQ_PORT_CAP_SUBSCRIPTION)) {
					snd_printk("seq: client %d, port %d doesn't support subscription\n", cell->event.source.client, cell->event.source.port);

					/* release the original cell */
					snd_seq_cell_free(cell);
					break;
				}
				/* go through list of subscribers */
				for (subs = p->subscribers; subs != NULL; subs = subs->next) {

					snd_seq_event_cell_t *new_cell;

					/* snd_printk("seq: message to subscriber %d:%d:%d\n",
					   subs->dest.queue,
					   subs->dest.client,
					   subs->dest.port); */

					/* send copy of the cell to the specified destination */
					new_cell = snd_seq_cell_dup(cell);
					if (new_cell) {
						new_cell->event.dest.queue = subs->data->dest.queue;
						new_cell->event.dest.client = subs->data->dest.client;
						new_cell->event.dest.port = subs->data->dest.port;
						snd_seq_dispatch_single_event(new_cell);
					} else {
						snd_printk("seq: failed sending to subscriber %d:%d:%d (no mem)\n",
						     subs->data->dest.queue,
						    subs->data->dest.client,
						     subs->data->dest.port);
					}
				}

				/* release the original cell */
				snd_seq_cell_free(cell);
				break;

			}
			break;

		default:
			snd_seq_enqueue_single_event(cell);
	}
}


/* check if client has access to queue's parameters, return 1 if client is
   allowed to make changes to the queue */

static int check_access(queue_t * q, int client)
{
	return ((q->locked == 0) || (q->owner == client));
}

int snd_seq_queue_check_access(int queueid, int client)
{
	queue_t *q = queueptr(queueid);

	if (q) {
		return check_access(q, client);
	} else {
		return 0;
	}
}

static void update_owner(queue_t * q, int client)
{
	if (q->owner == SEQ_QUEUE_NO_OWNER)
		q->owner = client;
}


/* access to queue's timer */
void snd_seq_queue_timer_stop(int queueid, int client)
{
	queue_t *q = queueptr(queueid);

	if (!q)
		return;

	if (check_access(q, client)) {
		snd_seq_timer_stop(q->timer);
		update_owner(q, client);
	}
}


void snd_seq_queue_timer_start(int queueid, int client)
{
	queue_t *q = queueptr(queueid);

	if (!q)
		return;

	if (check_access(q, client)) {
		snd_seq_timer_start(q->timer);
		update_owner(q, client);
	}
}


void snd_seq_queue_timer_continue(int queueid, int client)
{
	queue_t *q = queueptr(queueid);

	if (!q)
		return;

	if (check_access(q, client)) {
		snd_seq_timer_continue(q->timer);
		update_owner(q, client);
	}
}


void snd_seq_queue_timer_set_tempo(int queueid, int client, int tempo)
{
	queue_t *q = queueptr(queueid);

	if (!q)
		return;

	if (check_access(q, client)) {
		snd_seq_timer_set_tempo(q->timer, tempo);
		update_owner(q, client);
	}
}

void snd_seq_queue_timer_set_ppq(int queueid, int client, int ppq)
{
	queue_t *q = queueptr(queueid);

	if (!q)
		return;

	if (check_access(q, client)) {
		snd_seq_timer_set_ppq(q->timer, ppq);
		update_owner(q, client);
	}
}


void snd_seq_queue_set_owner(int queueid, int client)
{
	queue_t *q = queueptr(queueid);

	if (!q)
		return;

	if (check_access(q, client))
		q->owner = client;
}



void snd_seq_queue_set_locked(int queueid, int client, int locked)
{
	queue_t *q = queueptr(queueid);

	if (!q)
		return;

	if (check_access(q, client)) {
		q->locked = locked;
		update_owner(q, client);
	}
}


/* exported to seq_info.c */
void snd_seq_info_queues_read(snd_info_buffer_t * buffer, void *private_data)
{
	int c;
	int bpm;
	queue_t *q;
	timer_t *tmr;

	for (c = 0; c < SND_SEQ_MAX_QUEUES; c++) {
		q = queueptr(c);
		if (q == NULL)
			continue;
		tmr = q->timer;

		if (tmr->tempo)
			bpm = 60E6 / tmr->tempo;
		else
			bpm = 0;

		snd_iprintf(buffer, "queue %d:\n", c);
		snd_iprintf(buffer, "owned by client    : %d\n", q->owner);
		snd_iprintf(buffer, "lock status        : %s\n", q->locked ? "Locked" : "Free");
		snd_iprintf(buffer, "queued time events : %d\n", snd_seq_prioq_avail(q->timeq));
		snd_iprintf(buffer, "queued tick events : %d\n", snd_seq_prioq_avail(q->tickq));
		snd_iprintf(buffer, "timer state        : %s\n", tmr->running ? "Running" : "Stopped");
		snd_iprintf(buffer, "timer PPQ          : %d\n", tmr->ppq);
		snd_iprintf(buffer, "current tempo      : %d\n", tmr->tempo);
		snd_iprintf(buffer, "current BPM        : %d\n", bpm);
		snd_iprintf(buffer, "current time       : %d.%09d s\n", tmr->cur_time.tv_sec, tmr->cur_time.tv_nsec);
		snd_iprintf(buffer, "current tick       : %d\n", tmr->cur_tick);
		snd_iprintf(buffer, "\n");
	}
}
