
/*  Protocol compatible masqdialer server written in C
    Copyright (C) 1998 Charles P. Wright 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
*/
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <netdb.h>
#include <netinet/in.h>
#include <stdio.h>
#include <syslog.h>
#include <pwd.h>
#include <shadow.h>

#include "mserver.h"

extern bool inetd;
extern int csfd;

char authuser[100];
char authpass[100];

bool verified = false;

bool auth_check(const char *cname)
{
	if (!auth_checkip(cname))
	{
		syslog(LOG_ERR, "Invalid IP address for: %s", cname);
		return false;
	}
	if (!auth_checkuser(cname))
	{
		syslog(LOG_ERR, "Invalid user for: %s", cname);
		return false;
	}
}

bool auth_checkip(const char *cname)
{
	char addr_str[1024];
	char temp[1024];

	if (inetd)
	{
		char ipallow[1024];

		snprintf (temp, 1024, "%s_ipallow", cname);
		config_getvalue(temp, ipallow, 1024);
					
		if (!strncmp("0.0.0.0", temp, 1024))
		{
			return false;
		}

		return true;
	}

	if (sock_host(csfd, addr_str, 1024) != 0)
	{
		syslog (LOG_NOTICE, "IP:UNKNOWN\n");
	}
	else
	{	
		char ipallow[1024];

		snprintf (temp, 1024, "%s_ipallow", cname);
		config_getvalue(temp, ipallow, 1024);
					
		syslog (LOG_INFO, "connection name: %s", cname);
		syslog (LOG_INFO, "allowed ips: %s", ipallow);

		if (auth_checkip_backend(addr_str, ipallow))
		{	
			return true;
		}
		else
		{
			return false;
		}
	}
}

bool auth_checkuser(const char *cname)
{
	char addr_str[1024];
	char temp[1024];

	char userallow[1024];
	char user[1024];

	bool found;
	int i, n;

	snprintf (temp, 1024, "%s_userallow", cname);
	config_getvalue(temp, userallow, 1024);

	syslog (LOG_INFO, "connection name: %s", cname);
	syslog (LOG_INFO, "allowed users: %s", userallow);
				
	if (!strncmp("", userallow, 1024))
	{
		syslog (LOG_DEBUG, "All users allowed to dial out connection.");
		return true;
	}
	
	if (!strncmp("*", userallow, 1024))
	{
		syslog (LOG_DEBUG, "All users allowed to dial out connection.");
		return true;
	}

	if (!verified)
	{
		syslog (LOG_INFO, "User not verified!", userallow);
		return false;
	}

	found = false;

	for (i = 0, n = 0; i < strlen(userallow); i++, n++)
	{
		if (userallow[i] == ',')
		{
			user[n] = '\0';
			if (!strncmp(user, authuser, 1024))
			{
				found = true;	
				break;
			}
			n = -1;
		}
		else
		{
			user[n] = userallow[i];
		}
	}
	user[n] = '\0';
	if (!strncmp(user, authuser, 1024))
	{
		found = true;	
	}

	return found;
}

void auth_setuser(const char *username)
{
	verified = false;
	strncpy(authuser, username, 100);
}

void auth_setpass(const char *password)
{
	struct passwd *pw_entry;

	char *cryptpass;
	char salt[3];

	verified = false;
	strncpy(authpass, password, 100);

	pw_entry = auth_getpwent(authuser);

	if (pw_entry == NULL)
	{
#ifdef DEBUG
		syslog(LOG_DEBUG, "Null password entry for %s!", authuser);
#endif
		return;
	}

	strncpy(salt, pw_entry->pw_passwd, 2);
#ifdef DEBUG
		syslog(LOG_DEBUG, "Salt for %s!", salt);
#endif
	salt[3] = '\0';

	if (!strncmp(crypt(authpass, salt), pw_entry->pw_passwd, 1024))
	{
		verified = true;
	}
}

struct passwd *auth_getpwent(const char *authuser)
{
	FILE *afile;
	
	struct spwd *sp_local;
	struct passwd *pw_ent;

	char authfile[1024];
	char shadowfile[1024];
	char temp[1024];

	bool shadow;
	bool found;

	config_getvalue("authfile", authfile, 1024);
	shadow = config_getvalue_bool("shadow", false);

	if (shadow)
	{
#ifdef DEBUG
		syslog (LOG_DEBUG, "Using shadow passwords!");	
#endif

		pw_ent = getpwnam(authuser);		
		sp_local = getspnam(authuser);		
		
		if (sp_local == NULL || pw_ent == NULL)
		{
#ifdef DEBUG
			if (sp_local == NULL)
			{
				syslog (LOG_DEBUG, "sp_local null");
			}

			if (pw_ent == NULL)
			{
				syslog (LOG_DEBUG, "pw_ent null");
			}
#endif
			return NULL;
		}

		pw_ent->pw_passwd = sp_local->sp_pwdp;
	}
	else
	{
		afile = fopen(authfile, "r");
		
		found = false;

		while (found == false)
		{
			pw_ent = fgetpwent(afile);
		
			if (!strncmp(pw_ent->pw_name, authuser, 1024))
			{
				found = true;
			}
		}

		fclose(afile);
	}

	return pw_ent;
}

bool auth_checkip_backend(const char *ip, const char *allowed)
{
	char allowed_ips[10][13]; // Allow up to 10 different IPs to try and match
	char checkquads[4][4];
	char quads[4][4];

	bool match, badquad;
	int masks;
	int iplen;

	int i, n, j, k;

	n = masks = 0;

	for (i = 0; i < (strlen(allowed) + 1); i++)
	{
		if (allowed[i] == ':' || allowed[i] == '\0')
		{
			allowed_ips[masks][n] = '\0';
			n = 0;
#ifdef DEBUG
			syslog (LOG_DEBUG, "ipmask[%d]: %s", masks, allowed_ips[masks]);
#endif
			masks++;
			allowed_ips[masks][0] = '\0';
		}
		else if (!isdigit(allowed[i]) && (allowed[i] != '.') && (allowed[i] != '*'))
		{
			syslog(LOG_INFO, "Invalid characters in an ipallow variable: '%c'!", allowed[i]);
		}
		else
		{
			if (n <= 13)
			{
				allowed_ips[masks][n++] = allowed[i];
			}
			else
			{
				syslog(LOG_ERR, "Invalid IP address, exceeds 13 characters: %s", allowed);
			}
		}
	}

	n = j = k = 0;

	iplen = strlen(ip);

	for (i = 0; i <= iplen; i++)
	{
		if (ip[i] == '.' || ip[i] == '\0')
		{
			checkquads[n][j] = '\0';
			n++;
			j = 0;
		}
		else
		{
			if (j < 4)
			{
				checkquads[n][j] = ip[i];
			}
			j++;
		}
	}
#ifdef DEBUG
	syslog (LOG_DEBUG, "Quadded Check IP: %s.%s.%s.%s", checkquads[0], checkquads[1], checkquads[2], checkquads[3]);	
#endif

	i = 0;
	
	while ((match == false) && (i < masks))
	{
		badquad = false;

		n = j = k = 0;

		for (k = 0; k <= strlen(allowed_ips[i]); k++)
		{
			if (allowed_ips[i][k] == '.' || allowed_ips[i][k] == '\0')
			{
				quads[n][j] = '\0';
				n++;
				j = 0;
			}
			else
			{
				if (j < 4)
				{
					quads[n][j] = allowed_ips[i][k];
				}
				j++;
			}
		}
	
#ifdef 	DEBUG
		syslog (LOG_DEBUG, "Quadded IP: %s.%s.%s.%s", quads[0], quads[1], quads[2], quads[3]);	
#endif

		badquad = false;

		for (n = 0; n < 4; n++)
		{
			if (!strncmp(quads[n], "*", 4))
			{
#ifdef DEBUG
				syslog (LOG_DEBUG, "Good quad [%d]: %s:%s", n, quads[n], checkquads[n]);
#endif
			}
			else if (atoi(quads[n]) == atoi(checkquads[n], 4))
			{
#ifdef DEBUG
				syslog (LOG_DEBUG, "Good quad [%d]: %s:%s", n, quads[n], checkquads[i]);
#endif
			}
			else
			{	
#ifdef DEBUG
				syslog (LOG_DEBUG, "Bad quad [%d]: %s:%s", n, quads[n], checkquads[n]);
#endif
				badquad = true;
			}
		}

		if (badquad == false)
		{
#ifdef DEBUG
			syslog(LOG_DEBUG, "Match!");
#endif
			match = true;
		}

		i++;
#ifdef DEBUG
		syslog (LOG_DEBUG, "i: %d masks: %d", i, masks);
#endif
	}

	return match;	
}
