/*
 * ipneigh.c		"ip neigh".
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 * Authors:	Alexey Kuznetsov, <kuznet@ms2.inr.ac.ru>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <fcntl.h>
#include <net/if.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/ip.h>
#include <arpa/inet.h>
#include <string.h>
#include <sys/time.h>
#include <resolv.h>

#include "utils.h"

#if 1
/* Ough... They are in private linux header file at the moment. */

#define NTF_ROUTER	0x80

#define NUD_INCOMPLETE	0x01
#define NUD_REACHABLE	0x02
#define NUD_STALE	0x04
#define NUD_DELAY	0x08
#define NUD_PROBE	0x10
#define NUD_FAILED	0x20

/* Dummy states */
#define NUD_NOARP	0x40
#define NUD_PERMANENT	0x80
#define NUD_NONE	0x00

#define NUD_VALID	(NUD_PERMANENT|NUD_NOARP|NUD_REACHABLE|NUD_PROBE|NUD_STALE|NUD_DELAY)
#endif


static void usage(void)
{
	fprintf(stderr, "Usage: ip neigh { add | del } { ADDR [ lladdr LLADDR ]\n"
		        "          [ nud { permanent | noarp | stale | reachable } ]\n"
		        "          | proxy ADDR } [ dev DEVICE ]\n");
	fprintf(stderr, "       ip neigh show [ ipv4 | ipv6 | all ]\n");
	exit(-1);
}

static int ipneigh_modify(int cmd, int argc, char **argv)
{
	struct rtnl_handle rth;
	struct {
		struct nlmsghdr 	n;
		struct ndmsg 		ndm;
		char   			buf[256];
	} req;
	char  d[16];
	int dst_ok = 0;
	int lladdr_ok = 0;
	char * lla = NULL;
	inet_prefix dst;

	memset(&req, 0, sizeof(req));
	d[0] = 0;

	req.n.nlmsg_len = NLMSG_LENGTH(sizeof(struct ndmsg));
	req.n.nlmsg_flags = NLM_F_REQUEST;
	req.n.nlmsg_type = cmd;
	req.ndm.ndm_family = preferred_family;
	req.ndm.ndm_state = NUD_PERMANENT;

	while (argc > 0) {
		if (matches(*argv, "lladdr") == 0) {
			NEXT_ARG();
			if (lladdr_ok)
				usage();
			lla = *argv;
			lladdr_ok = 1;
		} else if (strcmp(*argv, "nud") == 0) {
			unsigned state;
			NEXT_ARG();
			if (matches(*argv, "permanent") == 0)
				state = NUD_PERMANENT;
			else if (matches(*argv, "noarp") == 0)
				state = NUD_NOARP;
			else if (matches(*argv, "stale") == 0)
				state = NUD_STALE;
			else if (matches(*argv, "reachable") == 0)
				state = NUD_REACHABLE;
			else if (scan_number(*argv, &state))
				usage();
			if (state>=0x100 || (state&(state-1)))
				usage();
			req.ndm.ndm_state = state;
		} else if (matches(*argv, "proxy") == 0) {
			NEXT_ARG();
			if (dst_ok)
				usage();
			get_addr(&dst, *argv, preferred_family);
			dst_ok = 1;
			req.ndm.ndm_flags |= NTF_PROXY;
		} else if (strcmp(*argv, "dev") == 0) {
			NEXT_ARG();
			strcpy(d, *argv);
		} else {
			if (dst_ok)
				usage();
			get_addr(&dst, *argv, preferred_family);
			dst_ok = 1;
		}
		argc--; argv++;
	}
	if (d[0] == 0 || !dst_ok || dst.family == AF_UNSPEC)
		usage();
	req.ndm.ndm_family = dst.family;
	addattr_l(&req.n, sizeof(req), NDA_DST, &dst.data, dst.bytelen);

	if (lla && strcmp(lla, "null")) {
		__u8 llabuf[16];
		int i;

		for (i=0; i<sizeof(llabuf); i++) {
			int temp;
			char *cp = strchr(lla, ':');
			if (cp) {
				*cp = 0;
				cp++;
			}
			if (sscanf(lla, "%x", &temp) != 1)
				usage();
			if (temp < 0 || temp > 255)
				usage();
			llabuf[i] = temp;
			if (!cp)
				break;
			lla = cp;
		}
		addattr_l(&req.n, sizeof(req), NDA_LLADDR, llabuf, i+1);
	}

	if (rtnl_open(&rth, 0) < 0) {
		fprintf(stderr, "cannot open rtnetlink\n");
		exit(1);
	}

	ll_init_map(&rth);

	if ((req.ndm.ndm_ifindex = ll_name_to_index(d)) == 0)
		usage();

	if (rtnl_talk(&rth, &req.n, 0, 0, NULL, NULL, NULL) < 0)
		exit(2);

	exit(0);
}


int print_neigh(struct sockaddr_nl *who, struct nlmsghdr *n, void *arg)
{
	FILE *fp = (FILE*)arg;
	struct ndmsg *r = NLMSG_DATA(n);
	int len = n->nlmsg_len;
	struct rtattr * tb[NDA_MAX+1];
	char abuf[256];

	if (n->nlmsg_type != RTM_NEWNEIGH) {
		fprintf(stderr, "Not RTM_NEWNEIGH: %08x %08x %08x\n",
			n->nlmsg_len, n->nlmsg_type, n->nlmsg_flags);
		
		return 0;
	}
	len -= NLMSG_LENGTH(sizeof(*r));
	if (len < 0) {
		fprintf(stderr, "Wrong len %d\n", len);
		return -1;
	}

	memset(tb, 0, sizeof(tb));
	parse_rtattr(tb, NDA_MAX, NDA_RTA(r), n->nlmsg_len - NLMSG_LENGTH(sizeof(*r)));

	if (tb[NDA_DST]) {
		if (inet_ntop(r->ndm_family, RTA_DATA(tb[NDA_DST]), abuf, sizeof(abuf)))
			fprintf(fp, "%s ", abuf);
		else
			fprintf(fp, "? ");
	}
	if (r->ndm_ifindex)
		fprintf(fp, "dev %s ", ll_index_to_name(r->ndm_ifindex));
	if (tb[NDA_LLADDR]) {
		int i;
		__u8 *lladdr = RTA_DATA(tb[NDA_LLADDR]);
		fprintf(fp, "lladdr ");
		for (i=0; RTA_LENGTH(i) < tb[NDA_LLADDR]->rta_len; i++) {
			if (i)
				printf(":");
			printf("%02x", lladdr[i]);
		}
	}
	if (r->ndm_flags & NTF_ROUTER) {
		printf(" router");
	}
	if (tb[NDA_CACHEINFO] && show_stats) {
		struct nda_cacheinfo *ci = RTA_DATA(tb[NDA_CACHEINFO]);
		if (ci->ndm_refcnt)
			printf(" ref %d", ci->ndm_refcnt);
		printf(" used %d%d/%d", ci->ndm_used/HZ,
		       ci->ndm_confirmed/HZ, ci->ndm_updated/HZ);
	}

	switch (r->ndm_state) {
	case NUD_NONE:	
		break;
	case NUD_INCOMPLETE:	
		printf(" nud incomplete");
		break;
	case NUD_REACHABLE:	
		printf(" nud reachable");
		break;
	case NUD_STALE:	
		printf(" nud stale");
		break;
	case NUD_DELAY:	
		printf(" nud delay");
		break;
	case NUD_PROBE:	
		printf(" nud probe");
		break;
	case NUD_FAILED:	
		printf(" nud failed");
		break;
	case NUD_NOARP:	
		printf(" nud noarp");
		break;
	case NUD_PERMANENT:	
		printf(" nud permanent");
		break;
	default:	
		printf(" nud ?%x", r->ndm_state);
			break;
	}
	printf("\n");

	fflush(fp);
	return 0;
}

int do_show(int argc, char **argv)
{
	int family = preferred_family;
	struct rtnl_handle rth;

	while (argc > 0) {
		if (matches(*argv, "ipv4") == 0) {
			family = AF_INET;
		} else if (matches(*argv, "ipv6") == 0) {
			family = AF_INET6;
		} else if (matches(*argv, "all") == 0) {
			family = AF_UNSPEC;
		} else
			usage();
		argc--; argv++;
	}
	
	if (rtnl_open(&rth, 0) < 0) {
		fprintf(stderr, "cannot open rtnetlink\n");
		exit(1);
	}

	ll_init_map(&rth);

	if (rtnl_wilddump_request(&rth, family, RTM_GETNEIGH) < 0) {
		perror("cannot send dump request");
		exit(1);
	}

	if (rtnl_dump_filter(&rth, print_neigh, stdout, NULL, NULL) < 0) {
		fprintf(stderr, "dump terminated\n");
		exit(1);
	}

	return 0;
}

int do_ipneigh(int argc, char **argv)
{
	if (argc > 0) {
		if (matches(*argv, "add") == 0)
			return ipneigh_modify(RTM_NEWNEIGH, argc-1, argv+1);
		if (matches(*argv, "del") == 0)
			return ipneigh_modify(RTM_DELNEIGH, argc-1, argv+1);
		if (matches(*argv, "get") == 0) {
			fprintf(stderr, "Sorry, \"neigh get\" is not implemented :-(\n");
			return -1;
		}
		if (matches(*argv, "show") == 0 ||
		    matches(*argv, "lst") == 0 ||
		    matches(*argv, "list") == 0)
			return do_show(argc-1, argv+1);
	} else
		return do_show(0, NULL);

	usage();
}
