binom.test <- function(x, n, p = 0.5, alternative = "two.sided")
{
  if ((length(n) > 1) || is.na(n) || (n < 1) || (n != round(n)))
    stop("n must be a positive integer")
  if ((length(x) > 1) || is.na(x) || (x < 0) || (x > n) || (x != round(x)))
    stop("x must be an integer between 0 and n")
  if (!missing(p) && (length(p) > 1 || is.na(p) || p < 0 || p > 1))
    stop ("p must be a single number between 0 and 1")

  CHOICES <- c("two.sided", "less", "greater")
  alternative <- CHOICES[pmatch(alternative, CHOICES)]
  if (length(alternative) > 1 || is.na(alternative))
    stop ("alternative must be \"two.sided\", \"less\" or \"greater\"")

  DNAME <- paste(deparse(substitute(x)), "and", deparse(substitute(n)))
  if (alternative == "less")
    PVAL <- pbinom(x, n, p)
  else if (alternative == "greater")
    PVAL <- 1 - pbinom(x - 1, n, p)
  else {
    eps <- 10^(-6)
    if (x/n < p) {
      PVAL <- pbinom(x, n, p)
      v <- 1 - pbinom(0:n, n, p)
    }
    else {
      PVAL <- 1 - pbinom(x - 1, n, p)
      v <- pbinom(0:n, n, p)
    }
    PVAL <- min(1, PVAL + max(v[v <= (1 + eps) * PVAL]))
  }
  
  names(x) <- "number of successes"	# or simply "x" ??
  names(n) <- "number of trials"	# or simply "n" ??
  names(p) <- "probability of success"	# or simply "p" ??

  structure(list(statistic = x,
		 parameter = n,
		 p.value = PVAL,
		 null.value = p,
		 alternative = alternative,
		 method = "Exact binomial test",
		 data.name = DNAME),
	    class = "htest")
}
