#include <stdio.h>
#include <stdlib.h>
#include <strings.h>

#include <unistd.h>
#include <sys/time.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/param.h>
#include <errno.h>

#include <GL/gl.h>
#include <GL/glu.h>
#include <GL/glut.h>

#include "draw.h"
#include "glbiff.h"
#include "mail.h"

///////////////////////// constants
const int CHECK_INTERVAL=60;

///////////////////////// globals

// if you don't like these defaults, change them, or use command lines 
// switches to choose different actions
char* cmd_mail_reader ="xterm -e elm";	// by default, bring up elm
char* cmd_new_mail = "echo -n \a";	// by default, ring bell

int mail_count=0;		// how much mail to draw in mailbox
bool unreadmail=false;		// is there any unread mail?

bool fMailReaderUp = false;	// is the mail reader up?

pid_t child_pid;		// pid of mail reading program when
				// it's up

// bezier patch control points of the rounded top of mailbox
GLfloat mbox_top_ctrlpts[4][4][3] = 
	{
	  {{0,0,0}, {0,0,1}, {1,0,1}, {1,0,0}},
	  {{0,0.33,0}, {0,0.33,1}, {1,0.33,1}, {1,0.33,0}},
	  {{0,0.66,0}, {0,0.66,1}, {1,0.66,1}, {1,0.66,0}},
	  {{0,1,0}, {0,1,1}, {1,1,1}, {1,1,0}},
	};

// bezier curve control points for the mailbox door
GLfloat mbox_door_ctrlpts[4][3] =
	{
	  {0,0,0}, {0,0,1}, {1,0,1}, {1,0,0},
	};
	  
////////////////////////// subroutines


/*
 * new_mail_arrived(): 	gets called whenever the arrival of new mail has
 * 			been detected.
 */
void new_mail_arrived(void) {
  if( cmd_new_mail )
    system(cmd_new_mail);
}


/*
 * check_mail(): this is the function to call when you need to reassess
 * 		the status of all mailboxes
 */
void check_mail(void) {
  bool anymail, newmail;
  mail_count = check_all_mailboxes( anymail, unreadmail, newmail );
  if( newmail )
    new_mail_arrived();
}


/*
 * alarm_handler(): handles SIGALRM
 */
void alarm_handler( int ) {
 
#ifdef DEBUG
  printf("Handling SIGALRM.\n");
#endif

  check_mail();
  display();
}

/*
 * child_sig_handler(): handles SIGCHLD
 * 		It's main purpose is to watch for the mail reader
 * 		shutting down, and then taking appropriate action.
 */
void child_sig_handler( int ) {

#ifdef DEBUG
  printf("Handling SIGCHLD.\n");
#endif

  if( !child_pid ) {

#ifdef DEBUG
    printf("Child trying to handle a SIGCHILD. Ignoring...\n");
#endif

    return;
  }

  if( fMailReaderUp ) {
    int status;
    pid_t res = waitpid( child_pid, &status, WNOHANG );
#ifdef DEBUG
    printf("waitpid() returned %d (child_pid==%d).\n",res,child_pid);
#endif

    if( res==child_pid ) {
#ifdef DEBUG
      printf("Detected mail program shutting down.\n");
#endif
      fMailReaderUp = false;
      fDoorOpen = false;
      fXformOn = false;
      check_mail();		// to reflect changes made by user in mboxes
      display();
    }
  }
}

/*
 * mouse_handler()
 *
 * handles any mouse events handed down from GLUT, in particular it
 * watches for mouse button presses.
 */
void mouse_handler( int button, int state, int x, int y ) {
  if( state==GLUT_UP && button==GLUT_LEFT_BUTTON ) {
    // update mailbox status
    check_mail();
      
    if( !fMailReaderUp ) {
      // first do the display so it is not jittery due to mailer startup
      fDoorOpen = true;
      fXformOn = true;
      display();

      // launch mail reader ...
      fMailReaderUp = true;
      child_pid = fork();
      if( child_pid ) {	// parent thread
	// do nothing (for now)
#ifdef DEBUG
	printf("Parent doing nothing in fork.\n");
#endif
      } else {
	system( cmd_mail_reader ); 
	exit(0);
      }
    }
  }
  if( state==GLUT_UP && button==GLUT_RIGHT_BUTTON ) {
    if( !fMailReaderUp ) {	// if mail reader is up, leave it alone
      // update mailbox status
      check_mail();
      
      if( !fDoorOpen ) {
	fDoorOpen = true;
	fXformOn = true;
	display();
      } else {
	fDoorOpen = false;
	fXformOn = false;
	display();
      }
    }
  }
}


/*
 * general_init()
 *
 * some basic initialization of Mesa/OpenGL; also installs the signal 
 * handlers.
 */
void general_init(void)
{
  ///////// Mesa stuff

  glClearColor(0.4,0.8,0.8, 1.0);	// sky colour

  GLfloat ambient[] = { 0.3, 0.3, 0.3, 1.0 };
  GLfloat diffuse[] = { 1.0, 1.0, 1.0, 1.0 };
  GLfloat specular[] = { 1.0, 1.0, 1.0, 1.0 };
  GLfloat position[] = { 0.5, 1, 3.0, 0.0 };

  GLfloat lmodel_ambient[] = { 0.5, 0.5, 0.5, 1.0 };
  GLfloat local_view[] = { 0.0 };

  GLfloat mat[4];

  /**** set lighting parameters ****/
  glLightfv(GL_LIGHT0, GL_AMBIENT, ambient);
  glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse);
  glLightfv(GL_LIGHT0, GL_POSITION, position);
  glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
  glLightModelfv(GL_LIGHT_MODEL_LOCAL_VIEWER, local_view);
//  glLightModeli(GL_LIGHT_MODEL_TWO_SIDE, GL_TRUE);

  glFrontFace (GL_CW);
  glEnable(GL_LIGHTING);
  glEnable(GL_LIGHT0);
  glEnable(GL_AUTO_NORMAL);
  glEnable(GL_NORMALIZE);
  glEnable(GL_DEPTH_TEST);
  glDepthFunc(GL_LESS);

  glMap2f(GL_MAP2_VERTEX_3, 0, 1, 3, 4, 0, 1, 12, 4, (float*)mbox_top_ctrlpts);
  glMapGrid2f(DOME_SEGS, 0.0, 1.0, DOME_SEGS, 0.0, 1.0);

  glMap1f(GL_MAP1_VERTEX_3, 0, 1, 3, 4, (float*)mbox_door_ctrlpts);


  ////////// non-Mesa stuff

  // collect initial info on mailboxes
  check_mail();

  // signal handlers
  struct sigaction sig;
  
  sig.sa_handler = alarm_handler;
  sig.sa_flags = 0;
  sigemptyset( &sig.sa_mask );
#ifndef BSD4_4
  sig.sa_restorer = NULL;
#endif  
  if( sigaction(SIGALRM, &sig, NULL) )
    printf("Error setting signal handler for SIGALRM (%d).\n", errno);
  else
    printf("SIGALRM handler installed ok.\n");

  sig.sa_handler = child_sig_handler;
  sig.sa_flags = 0;
  sigemptyset( &sig.sa_mask );
#ifndef BSD4_4
  sig.sa_restorer = NULL;		// not used, clear it just in case
#endif
  if( sigaction(SIGCHLD, &sig, NULL) )
    printf("Error setting signal handler for SIGCHLD (%d).\n", errno);
  else
    printf("SIGCHLD handler installed ok.\n");

  struct itimerval itim;
  itim.it_interval.tv_sec = CHECK_INTERVAL;
  itim.it_interval.tv_usec = 0;
  itim.it_value.tv_sec = CHECK_INTERVAL;
  itim.it_value.tv_usec = 0;
  if( setitimer(ITIMER_REAL, &itim, NULL) )
    printf("Error setting alarm timer (%d).\n", errno );
  else
    printf("Timer initialized ok.\n");
}

/*
 * myReshape()
 *
 * This function gets called when the window is resized; (w,h) are the
 * new width and height.
 */
void myReshape(int w, int h)
{
  glViewport(0, 0, w, h);
  glMatrixMode(GL_PROJECTION);
  glLoadIdentity();

  glFrustum(-1,1,-1,1,5,100);
  glMatrixMode(GL_MODELVIEW);
  glLoadIdentity();

  gluLookAt(1,-9,1, 1,2,1.2, 0,0,1);
}


/*
 * main()
 */
int main(int argc, char** argv)
{
  // let GLUT parse the default X switches (such as -geometry, -iconic, etc.)
  glutInit( &argc, argv );

  // do our own parsing of the remaining command line
  for( int arg=1; arg<argc; arg++ ) {
    if( !strcmp( argv[arg], "-newmail" ) ) {
      arg++;
      cmd_new_mail = strdup( argv[arg] );
    } else if( !strcmp( argv[arg], "-mailprog" ) ) {
      arg++;
      cmd_mail_reader = strdup( argv[arg] );
      cmd_mail_reader = strcat( "xterm -e" , cmd_mail_reader );
    } else
      fprintf( stderr, "Ignoring unkown switch %s\n", argv[arg] );
  }
  
  // do the rest of setup
  glutInitDisplayMode (GLUT_DOUBLE | GLUT_RGB | GLUT_DEPTH);
  glutCreateWindow (argv[0]);
  general_init();
  glutReshapeFunc( myReshape );
  glutMouseFunc( mouse_handler );
  glutDisplayFunc( display );

  // enter the main loop
  glutMainLoop();
}

  
