/* 
 * tkScrollbar.c --
 *
 *	This module implements a scrollbar widgets for the Tk
 *	toolkit.  A scrollbar displays a slider and two arrows;
 *	mouse clicks on features within the scrollbar cause
 *	scrolling commands to be invoked.
 *
 * Copyright (c) 1990-1994 The Regents of the University of California.
 * Copyright (c) 1994-1995 Sun Microsystems, Inc.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * SCCS: @(#) tkScrollbar.c 1.79 96/02/15 18:52:40
 */
/*
 * TkSTEP modifications Copyright by Alfredo K. Kojima 
 */ 

#include "tkPort.h"
#include "default.h"
#include "tkInt.h"

/*
 * A data structure of the following type is kept for each scrollbar
 * widget managed by this file:
 */

typedef struct {
    Tk_Window tkwin;		/* Window that embodies the scrollbar.  NULL
				 * means that the window has been destroyed
				 * but the data structures haven't yet been
				 * cleaned up.*/
    Display *display;		/* Display containing widget.  Used, among
				 * other things, so that resources can be
				 * freed even after tkwin has gone away. */
    Tcl_Interp *interp;		/* Interpreter associated with scrollbar. */
    Tcl_Command widgetCmd;	/* Token for scrollbar's widget command. */
    Tk_Uid orientUid;		/* Orientation for window ("vertical" or
				 * "horizontal"). */
    int vertical;		/* Non-zero means vertical orientation
				 * requested, zero means horizontal. */
    int width;			/* Desired narrow dimension of scrollbar,
				 * in pixels. */
    char *command;		/* Command prefix to use when invoking
				 * scrolling commands.  NULL means don't
				 * invoke commands.  Malloc'ed. */
    int commandSize;		/* Number of non-NULL bytes in command. */
    int repeatDelay;		/* How long to wait before auto-repeating
				 * on scrolling actions (in ms). */
    int repeatInterval;		/* Interval between autorepeats (in ms). */
    int jump;			/* Value of -jump option. */

    /*
     * Information used when displaying widget:
     */

    int borderWidth;		/* Width of 3-D borders. */
    Tk_3DBorder bgBorder;	/* Used for drawing background (all flat
				 * surfaces except for trough). */
    Tk_3DBorder activeBorder;	/* For drawing backgrounds when active (i.e.
				 * when mouse is positioned over element). */
    XColor *troughColorPtr;	/* Color for drawing trough. */
    XColor *darkPtr;		/* Color for background stipple */    
    GC troughGC;		/* For drawing trough's stippled part. */
    GC troughBackGC;		/* For drawing trough */
    GC copyGC;			/* Used for copying from pixmap onto screen. */
    int relief;			/* Indicates whether window as a whole is
				 * raised, sunken, or flat. */
    int highlightWidth;		/* Width in pixels of highlight to draw
				 * around widget when it has the focus.
				 * <= 0 means don't draw a highlight. */
    XColor *highlightBgColorPtr;
				/* Color for drawing traversal highlight
				 * area when highlight is off. */
    XColor *highlightColorPtr;	/* Color for drawing traversal highlight. */
    Pixmap stipple;		/* trough base pattern */
    Pixmap bump;		/* bump in the slider */
    int inset;			/* Total width of all borders, including
				 * traversal highlight and 3-D border.
				 * Indicates how much interior stuff must
				 * be offset from outside edges to leave
				 * room for borders. */
    int elementBorderWidth;	/* Width of border to draw around elements
				 * inside scrollbar (arrows and slider).
				 * -1 means use borderWidth. */
    int arrowLength;		/* Length of arrows along long dimension of
				 * scrollbar, including space for a small gap
				 * between the arrow and the slider.
				 * Recomputed on window size changes. */
    int sliderFirst;		/* Pixel coordinate of top or left edge
				 * of slider area, including border. */
    int sliderLast;		/* Coordinate of pixel just after bottom
				 * or right edge of slider area, including
				 * border. */
    int activeField;		/* Names field to be displayed in active
				 * colors, such as TOP_ARROW, or 0 for
				 * no field. */
    int activeRelief;		/* Value of -activeRelief option: relief
				 * to use for active element. */

    /*
     * Information describing the application related to the scrollbar.
     * This information is provided by the application by invoking the
     * "set" widget command.  This information can now be provided in
     * two ways:  the "old" form (totalUnits, windowUnits, firstUnit,
     * and lastUnit), or the "new" form (firstFraction and lastFraction).
     * FirstFraction and lastFraction will always be valid, but
     * the old-style information is only valid if the NEW_STYLE_COMMANDS
     * flag is 0.
     */

    int totalUnits;		/* Total dimension of application, in
				 * units.  Valid only if the NEW_STYLE_COMMANDS
				 * flag isn't set. */
    int windowUnits;		/* Maximum number of units that can be
				 * displayed in the window at once.  Valid
				 * only if the NEW_STYLE_COMMANDS flag isn't
				 * set. */
    int firstUnit;		/* Number of last unit visible in
				 * application's window.  Valid only if the
				 * NEW_STYLE_COMMANDS flag isn't set. */
    int lastUnit;		/* Index of last unit visible in window.
				 * Valid only if the NEW_STYLE_COMMANDS
				 * flag isn't set. */
    double firstFraction;	/* Position of first visible thing in window,
				 * specified as a fraction between 0 and
				 * 1.0. */
    double lastFraction;	/* Position of last visible thing in window,
				 * specified as a fraction between 0 and
				 * 1.0. */

    /*
     * Miscellaneous information:
     */

    Tk_Cursor cursor;		/* Current cursor for window, or None. */
    char *takeFocus;		/* Value of -takefocus option;  not used in
				 * the C code, but used by keyboard traversal
				 * scripts.  Malloc'ed, but may be NULL. */
    int flags;			/* Various flags;  see below for
				 * definitions. */
#ifdef FULL_STEP
    int noArrows;		/* if arrow buttons should not be drawn */
#endif    
} Scrollbar;

/*
 * Legal values for "activeField" field of Scrollbar structures.  These
 * are also the return values from the ScrollbarPosition procedure.
 */

#define OUTSIDE		0
#define TOP_ARROW	1
#define TOP_GAP		2
#define SLIDER		3
#define BOTTOM_GAP	4
#define BOTTOM_ARROW	5

/*
 * Flag bits for scrollbars:
 * 
 * REDRAW_PENDING:		Non-zero means a DoWhenIdle handler
 *				has already been queued to redraw
 *				this window.
 * NEW_STYLE_COMMANDS:		Non-zero means the new style of commands
 *				should be used to communicate with the
 *				widget:  ".t yview scroll 2 lines", instead
 *				of ".t yview 40", for example.
 * GOT_FOCUS:			Non-zero means this window has the input
 *				focus.
 */

#define REDRAW_PENDING		1
#define NEW_STYLE_COMMANDS	2
#define GOT_FOCUS		4

#define RECONFIGURE		256
/*
 * Minimum slider length, in pixels (designed to make sure that the slider
 * is always easy to grab with the mouse).
 */

#define MIN_SLIDER_LENGTH	6

/*
 * If you want to make Tk ignore attribute settings that may be violating
 * NEXTSTEP(tm) styles leave DISABLE_FORCE_STEP undef'ed, otherwise define it
 * somewhere
 */
#ifndef DISABLE_FORCE_STEP
#  define DISABLE_CHANGE	TK_CONFIG_DONT_CHANGE_DEFAULT
#else
#  define DISABLE_CHANGE	0
#endif

/*
 * Information used for argv parsing.
 */

static Tk_ConfigSpec configSpecs[] = {
    {TK_CONFIG_BORDER, "-activebackground", "activeBackground", "Foreground",
	DEF_SCROLLBAR_ACTIVE_BG_COLOR, Tk_Offset(Scrollbar, activeBorder),
	TK_CONFIG_COLOR_ONLY|DISABLE_CHANGE},
    {TK_CONFIG_BORDER, "-activebackground", "activeBackground", "Foreground",
	DEF_SCROLLBAR_ACTIVE_BG_MONO, Tk_Offset(Scrollbar, activeBorder),
	TK_CONFIG_MONO_ONLY|DISABLE_CHANGE},
    {TK_CONFIG_RELIEF, "-activerelief", "activeRelief", "Relief",
	DEF_SCROLLBAR_ACTIVE_RELIEF, Tk_Offset(Scrollbar, activeRelief), 
        0},
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	DEF_SCROLLBAR_BG_COLOR, Tk_Offset(Scrollbar, bgBorder),
	TK_CONFIG_COLOR_ONLY|DISABLE_CHANGE},
    {TK_CONFIG_BORDER, "-background", "background", "Background",
	DEF_SCROLLBAR_BG_MONO, Tk_Offset(Scrollbar, bgBorder),
	TK_CONFIG_MONO_ONLY|DISABLE_CHANGE},
    {TK_CONFIG_SYNONYM, "-bd", "borderWidth", (char *) NULL,
	(char *) NULL, 0, 0},
    {TK_CONFIG_SYNONYM, "-bg", "background", (char *) NULL,
	(char *) NULL, 0, 0},
    {TK_CONFIG_PIXELS, "-borderwidth", "borderWidth", "BorderWidth",
	DEF_SCROLLBAR_BORDER_WIDTH, Tk_Offset(Scrollbar, borderWidth), 
        DISABLE_CHANGE},
    {TK_CONFIG_STRING, "-command", "command", "Command",
	DEF_SCROLLBAR_COMMAND, Tk_Offset(Scrollbar, command),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_ACTIVE_CURSOR, "-cursor", "cursor", "Cursor",
	DEF_SCROLLBAR_CURSOR, Tk_Offset(Scrollbar, cursor), TK_CONFIG_NULL_OK},
    {TK_CONFIG_PIXELS, "-elementborderwidth", "elementBorderWidth",
	"BorderWidth", DEF_SCROLLBAR_EL_BORDER_WIDTH,
	Tk_Offset(Scrollbar, elementBorderWidth), 0},
    {TK_CONFIG_COLOR, "-highlightbackground", "highlightBackground",
	"HighlightBackground", DEF_SCROLLBAR_HIGHLIGHT_BG,
	Tk_Offset(Scrollbar, highlightBgColorPtr), 
        DISABLE_CHANGE},
    {TK_CONFIG_COLOR, "-highlightcolor", "highlightColor", "HighlightColor",
	DEF_SCROLLBAR_HIGHLIGHT,
	Tk_Offset(Scrollbar, highlightColorPtr), 
        DISABLE_CHANGE},
    {TK_CONFIG_PIXELS, "-highlightthickness", "highlightThickness",
	"HighlightThickness",
	DEF_SCROLLBAR_HIGHLIGHT_WIDTH, Tk_Offset(Scrollbar, highlightWidth), 
        DISABLE_CHANGE},
    {TK_CONFIG_BOOLEAN, "-jump", "jump", "Jump",
	DEF_SCROLLBAR_JUMP, Tk_Offset(Scrollbar, jump), 0},
#ifdef FULL_STEP
    {TK_CONFIG_BOOLEAN, "-noarrows", "noArrows", "NoArrows",
	DEF_SCROLLBAR_NOARROWS, Tk_Offset(Scrollbar, noArrows), 0},  
#endif      
    {TK_CONFIG_UID, "-orient", "orient", "Orient",
	DEF_SCROLLBAR_ORIENT, Tk_Offset(Scrollbar, orientUid), 0},
    {TK_CONFIG_RELIEF, "-relief", "relief", "Relief",
	DEF_SCROLLBAR_RELIEF, Tk_Offset(Scrollbar, relief), 
        DISABLE_CHANGE},
    {TK_CONFIG_INT, "-repeatdelay", "repeatDelay", "RepeatDelay",
	DEF_SCROLLBAR_REPEAT_DELAY, Tk_Offset(Scrollbar, repeatDelay), 0},
    {TK_CONFIG_INT, "-repeatinterval", "repeatInterval", "RepeatInterval",
	DEF_SCROLLBAR_REPEAT_INTERVAL, Tk_Offset(Scrollbar, repeatInterval), 0},
    {TK_CONFIG_STRING, "-takefocus", "takeFocus", "TakeFocus",
	DEF_SCROLLBAR_TAKE_FOCUS, Tk_Offset(Scrollbar, takeFocus),
	TK_CONFIG_NULL_OK},
    {TK_CONFIG_COLOR, "-troughcolor", "troughColor", "Background",
	DEF_SCROLLBAR_TROUGH_COLOR, Tk_Offset(Scrollbar, troughColorPtr),
	TK_CONFIG_COLOR_ONLY|DISABLE_CHANGE},
    {TK_CONFIG_COLOR, "-troughcolor", "troughColor", "Background",
	DEF_SCROLLBAR_TROUGH_MONO, Tk_Offset(Scrollbar, troughColorPtr),
	TK_CONFIG_MONO_ONLY|DISABLE_CHANGE},
    {TK_CONFIG_PIXELS, "-width", "width", "Width",
	DEF_SCROLLBAR_WIDTH, Tk_Offset(Scrollbar, width), 
        DISABLE_CHANGE},
    {TK_CONFIG_END, (char *) NULL, (char *) NULL, (char *) NULL,
	(char *) NULL, 0, 0}
};


 
#define SB_BUMP_WIDTH  6
#define SB_BUMP_HEIGHT 6
static char scrollbar_bump[] = {
" dbbb "\
"dbdddd"\
"bdd   "\
"bd  ww"\
"bd www"\
" d ww "
};


static unsigned char scrollbar_bump_bmp[] = {
   0x1e, 0x21, 0x39, 0x35, 0x3d, 0x1e};


/*
 * Forward declarations for procedures defined later in this file:
 */

static void		ComputeScrollbarGeometry _ANSI_ARGS_((
			    Scrollbar *scrollPtr));
static int		ConfigureScrollbar _ANSI_ARGS_((Tcl_Interp *interp,
			    Scrollbar *scrollPtr, int argc, char **argv,
			    int flags));
static void		DestroyScrollbar _ANSI_ARGS_((char *memPtr));
static void		DisplayScrollbar _ANSI_ARGS_((ClientData clientData));
static void		EventuallyRedraw _ANSI_ARGS_((Scrollbar *scrollPtr));
static void		ScrollbarCmdDeletedProc _ANSI_ARGS_((
			    ClientData clientData));
static void		ScrollbarEventProc _ANSI_ARGS_((ClientData clientData,
			    XEvent *eventPtr));
static int		ScrollbarPosition _ANSI_ARGS_((Scrollbar *scrollPtr,
			    int x, int y));
static int		ScrollbarWidgetCmd _ANSI_ARGS_((ClientData clientData,
			    Tcl_Interp *, int argc, char **argv));


/*
 *--------------------------------------------------------
 * PaintPixmap -
 *    Dirty routine to draw pixmaps from char data
 *    
 * gotta find a better way to do this...
 *--------------------------------------------------------
 */
static void PaintPixmap(clientData, d, data, width, height)
    ClientData clientData;
    Drawable d;
    char *data;
    int width, height;
{
    int x, y, ofs=0;
    Scrollbar *scrollPtr = (Scrollbar *)clientData;
    Display *display = scrollPtr->display;
    GC lightGC, backGC, darkGC, dark2GC;


    backGC = Tk_3DBorderGC( scrollPtr->tkwin, scrollPtr->bgBorder, 
			   TK_3D_FLAT_GC);
    lightGC = Tk_3DBorderGC( scrollPtr->tkwin, scrollPtr->bgBorder, 
			  TK_3D_LIGHT_GC);
    darkGC = Tk_3DBorderGC( scrollPtr->tkwin, scrollPtr->bgBorder, 
			   TK_3D_DARK_GC);
    dark2GC = Tk_3DBorderGC( scrollPtr->tkwin, scrollPtr->bgBorder, 
			   TK_3D_DARK2_GC);
    for(y = 0; y < height; y++) {
	for(x = 0; x < width; x++) {
	    switch (data[ofs++]) {
	     case 'b': /* black */
		XDrawPoint( scrollPtr->display, d, dark2GC, x, y);
		break;
	     case 'd': /* dark */
		XDrawPoint( scrollPtr->display, d, darkGC, x, y);
		break;
	     case 'w': /* white */
		XDrawPoint( scrollPtr->display, d, lightGC, x, y);
		break;
	     default:
		XDrawPoint( scrollPtr->display, d, backGC, x, y);
	    }
	}
    }
}


/*
 *---------------------------------------------
 *  
 * MakePixmaps
 *   Make pixmaps needed by scrollbar with 
 * PaintPixmap
 *
 *---------------------------------------------
 */
static void MakePixmaps(clientData)
    ClientData clientData;
{
    register Scrollbar *scrollPtr = (Scrollbar *) clientData;

    if (Tk_Depth(scrollPtr->tkwin) > 1) {
	scrollPtr->bump = Tk_GetPixmap( scrollPtr->display, 
	    Tk_WindowId(scrollPtr->tkwin),
	    SB_BUMP_WIDTH, SB_BUMP_HEIGHT, Tk_Depth(scrollPtr->tkwin) );
	PaintPixmap(clientData, scrollPtr->bump, scrollbar_bump,
		    SB_BUMP_WIDTH, SB_BUMP_HEIGHT);
	
    } else { /* monochrome display */
	scrollPtr->bump = XCreateBitmapFromData( scrollPtr->display,
		Tk_WindowId(scrollPtr->tkwin), scrollbar_bump_bmp,
		SB_BUMP_WIDTH, SB_BUMP_HEIGHT);
    }
}

/*
 *--------------------------------------------------------------
 *
 * Tk_ScrollbarCmd --
 *
 *	This procedure is invoked to process the "scrollbar" Tcl
 *	command.  See the user documentation for details on what
 *	it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

int
Tk_ScrollbarCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Main window associated with
				 * interpreter. */
    Tcl_Interp *interp;		/* Current interpreter. */
    int argc;			/* Number of arguments. */
    char **argv;		/* Argument strings. */
{
    Tk_Window tkwin = (Tk_Window) clientData;
    register Scrollbar *scrollPtr;
    Tk_Window new;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " pathName ?options?\"", (char *) NULL);
	return TCL_ERROR;
    }

    new = Tk_CreateWindowFromPath(interp, tkwin, argv[1], (char *) NULL);
    if (new == NULL) {
	return TCL_ERROR;
    }

    /*
     * Initialize fields that won't be initialized by ConfigureScrollbar,
     * or which ConfigureScrollbar expects to have reasonable values
     * (e.g. resource pointers).
     */

    scrollPtr = (Scrollbar *) ckalloc(sizeof(Scrollbar));
    scrollPtr->tkwin = new;
    scrollPtr->display = Tk_Display(new);
    scrollPtr->interp = interp;
    scrollPtr->widgetCmd = Tcl_CreateCommand(interp,
	    Tk_PathName(scrollPtr->tkwin), ScrollbarWidgetCmd,
	    (ClientData) scrollPtr, ScrollbarCmdDeletedProc);
    scrollPtr->orientUid = NULL;
    scrollPtr->vertical = 0;
    scrollPtr->width = 0;
    scrollPtr->command = NULL;
    scrollPtr->commandSize = 0;
    scrollPtr->repeatDelay = 0;
    scrollPtr->repeatInterval = 0;
    scrollPtr->borderWidth = 0;
    scrollPtr->bgBorder = NULL;
    scrollPtr->activeBorder = NULL;
    scrollPtr->troughColorPtr = NULL;
    scrollPtr->darkPtr = NULL;
    scrollPtr->troughBackGC = None;
    scrollPtr->bump = None;
    scrollPtr->stipple = None;	
    scrollPtr->troughGC = None;
    scrollPtr->copyGC = None;
    scrollPtr->relief = TK_RELIEF_FLAT;
    scrollPtr->highlightWidth = 0;
    scrollPtr->highlightBgColorPtr = NULL;
    scrollPtr->highlightColorPtr = NULL;
    scrollPtr->inset = 0;
    scrollPtr->elementBorderWidth = -1;
    scrollPtr->arrowLength = 0;
    scrollPtr->sliderFirst = 0;
    scrollPtr->sliderLast = 0;
    scrollPtr->activeField = 0;
    scrollPtr->activeRelief = TK_RELIEF_RAISED;
    scrollPtr->totalUnits = 0;
    scrollPtr->windowUnits = 0;
    scrollPtr->firstUnit = 0;
    scrollPtr->lastUnit = 0;
    scrollPtr->firstFraction = 0.0;
    scrollPtr->lastFraction = 0.0;
    scrollPtr->cursor = None;
    scrollPtr->takeFocus = NULL;
    scrollPtr->flags = 0;
#ifdef FULL_STEP
    scrollPtr->noArrows = 0;
#endif

    Tk_SetClass(scrollPtr->tkwin, "Scrollbar");
    Tk_CreateEventHandler(scrollPtr->tkwin,
	    ExposureMask|StructureNotifyMask|FocusChangeMask,
	    ScrollbarEventProc, (ClientData) scrollPtr);
    if (ConfigureScrollbar(interp, scrollPtr, argc-2, argv+2, 0) != TCL_OK) {
	goto error;
    }
    interp->result = Tk_PathName(scrollPtr->tkwin);
    return TCL_OK;

    error:
    Tk_DestroyWindow(scrollPtr->tkwin);
    return TCL_ERROR;
}

/*
 *--------------------------------------------------------------
 *
 * ScrollbarWidgetCmd --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */

static int
ScrollbarWidgetCmd(clientData, interp, argc, argv)
    ClientData clientData;	/* Information about scrollbar
					 * widget. */
    Tcl_Interp *interp;			/* Current interpreter. */
    int argc;				/* Number of arguments. */
    char **argv;			/* Argument strings. */
{
    register Scrollbar *scrollPtr = (Scrollbar *) clientData;
    int result = TCL_OK;
    size_t length;
    int c;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
		argv[0], " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }
    Tcl_Preserve((ClientData) scrollPtr);
    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'a') && (strncmp(argv[1], "activate", length) == 0)) {
	if (argc == 2) {
	    switch (scrollPtr->activeField) {
		case TOP_ARROW:		interp->result = "arrow1";	break;
		case SLIDER:		interp->result = "slider";	break;
		case BOTTOM_ARROW:	interp->result = "arrow2";	break;
	    }
	    goto done;
	}
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " activate element\"", (char *) NULL);
	    goto error;
	}
	c = argv[2][0];
	length = strlen(argv[2]);
	if ((c == 'a') && (strcmp(argv[2], "arrow1") == 0)) {
#ifdef FULL_STEP
	    scrollPtr->activeField = scrollPtr->noArrows ? OUTSIDE : TOP_ARROW;
#else
	    scrollPtr->activeField = TOP_ARROW;
#endif
	} else if ((c == 'a') && (strcmp(argv[2], "arrow2") == 0)) {
#ifdef FULL_STEP
	    scrollPtr->activeField = scrollPtr->noArrows ? OUTSIDE :
	    							BOTTOM_ARROW;
#else
	    scrollPtr->activeField = BOTTOM_ARROW;
#endif
	} else if ((c == 's') && (strncmp(argv[2], "slider", length) == 0)) {
	    scrollPtr->activeField = SLIDER;
	} else {
	    scrollPtr->activeField = OUTSIDE;
	}
	EventuallyRedraw(scrollPtr);
    } else if ((c == 'c') && (strncmp(argv[1], "cget", length) == 0)
	    && (length >= 2)) {
	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " cget option\"",
		    (char *) NULL);
	    goto error;
	}
	result = Tk_ConfigureValue(interp, scrollPtr->tkwin, configSpecs,
		(char *) scrollPtr, argv[2], 0);
    } else if ((c == 'c') && (strncmp(argv[1], "configure", length) == 0)
	    && (length >= 2)) {
	/* make sure that fg colors will be changed */	
	scrollPtr->flags|=RECONFIGURE;
	if (argc == 2) {
	    result = Tk_ConfigureInfo(interp, scrollPtr->tkwin, configSpecs,
		    (char *) scrollPtr, (char *) NULL, 0);
	} else if (argc == 3) {
	    result = Tk_ConfigureInfo(interp, scrollPtr->tkwin, configSpecs,
		    (char *) scrollPtr, argv[2], 0);
	} else {
	    result = ConfigureScrollbar(interp, scrollPtr, argc-2, argv+2,
		    TK_CONFIG_ARGV_ONLY);
	}
    } else if ((c == 'd') && (strncmp(argv[1], "delta", length) == 0)) {
	int xDelta, yDelta, pixels, length;
	double fraction;

	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " delta xDelta yDelta\"", (char *) NULL);
	    goto error;
	}
	if ((Tcl_GetInt(interp, argv[2], &xDelta) != TCL_OK)
		|| (Tcl_GetInt(interp, argv[3], &yDelta) != TCL_OK)) {
	    goto error;
	}
	if (scrollPtr->vertical) {
	    pixels = yDelta;
	    length = Tk_Height(scrollPtr->tkwin)
		    - 2*(scrollPtr->arrowLength + 1 + scrollPtr->inset);
	} else {
	    pixels = xDelta;
	    length = Tk_Width(scrollPtr->tkwin)
		    - 2*(scrollPtr->arrowLength + 1 + scrollPtr->inset);
	}
	if (length == 0) {
	    fraction = 0.0;
	} else {
	    fraction = ((double) pixels / (double) length);
	}
	sprintf(interp->result, "%g", fraction);
    } else if ((c == 'f') && (strncmp(argv[1], "fraction", length) == 0)) {
	int x, y, pos, length;
	double fraction;

	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " fraction x y\"", (char *) NULL);
	    goto error;
	}
	if ((Tcl_GetInt(interp, argv[2], &x) != TCL_OK)
		|| (Tcl_GetInt(interp, argv[3], &y) != TCL_OK)) {
	    goto error;
	}
	if (scrollPtr->vertical) {
	    pos = y - scrollPtr->inset;
	    length = Tk_Height(scrollPtr->tkwin) 
		    - 2*(scrollPtr->arrowLength + scrollPtr->inset + 1);
	} else {
	    pos = x - 2*(scrollPtr->arrowLength + 1) - scrollPtr->inset;
	    length = Tk_Width(scrollPtr->tkwin) 
		    - 2*(scrollPtr->arrowLength + scrollPtr->inset + 1);
	}
	if (length == 0) {
	    fraction = 0.0;
	} else {
	    fraction = ((double) pos / (double) length);
	}
	if (fraction < 0) {
	    fraction = 0;
	} else if (fraction > 1.0) {
	    fraction = 1.0;
	}
	sprintf(interp->result, "%g", fraction);
    } else if ((c == 'g') && (strncmp(argv[1], "get", length) == 0)) {
	if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " get\"", (char *) NULL);
	    goto error;
	}
	if (scrollPtr->flags & NEW_STYLE_COMMANDS) {
	    char first[TCL_DOUBLE_SPACE], last[TCL_DOUBLE_SPACE];

	    Tcl_PrintDouble(interp, scrollPtr->firstFraction, first);
	    Tcl_PrintDouble(interp, scrollPtr->lastFraction, last);
	    Tcl_AppendResult(interp, first, " ", last, (char *) NULL);
	} else {
	    sprintf(interp->result, "%d %d %d %d", scrollPtr->totalUnits,
		    scrollPtr->windowUnits, scrollPtr->firstUnit,
		    scrollPtr->lastUnit);
	}
    } else if ((c == 'i') && (strncmp(argv[1], "identify", length) == 0)) {
	int x, y, thing;

	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " identify x y\"", (char *) NULL);
	    goto error;
	}
	if ((Tcl_GetInt(interp, argv[2], &x) != TCL_OK)
		|| (Tcl_GetInt(interp, argv[3], &y) != TCL_OK)) {
	    goto error;
	}
	thing = ScrollbarPosition(scrollPtr, x,y);
	switch (thing) {
	    case TOP_ARROW:	interp->result = "arrow1";	break;
	    case TOP_GAP:	interp->result = "trough1";	break;
	    case SLIDER:	interp->result = "slider";	break;
	    case BOTTOM_GAP:	interp->result = "trough2";	break;
	    case BOTTOM_ARROW:	interp->result = "arrow2";	break;
	}
    } else if ((c == 's') && (strncmp(argv[1], "set", length) == 0)) {
	int totalUnits, windowUnits, firstUnit, lastUnit;

	if (argc == 4) {
	    double first, last;

	    if (Tcl_GetDouble(interp, argv[2], &first) != TCL_OK) {
		goto error;
	    }
	    if (Tcl_GetDouble(interp, argv[3], &last) != TCL_OK) {
		goto error;
	    }
	    if (first < 0) {
		scrollPtr->firstFraction = 0;
	    } else if (first > 1.0) {
		scrollPtr->firstFraction = 1.0;
	    } else {
		scrollPtr->firstFraction = first;
	    }
	    if (last < scrollPtr->firstFraction) {
		scrollPtr->lastFraction = scrollPtr->firstFraction;
	    } else if (last > 1.0) {
		scrollPtr->lastFraction = 1.0;
	    } else {
		scrollPtr->lastFraction = last;
	    }
	    scrollPtr->flags |= NEW_STYLE_COMMANDS;
	} else if (argc == 6) {
	    if (Tcl_GetInt(interp, argv[2], &totalUnits) != TCL_OK) {
		goto error;
	    }
	    if (totalUnits < 0) {
		totalUnits = 0;
	    }
	    if (Tcl_GetInt(interp, argv[3], &windowUnits) != TCL_OK) {
		goto error;
	    }
	    if (windowUnits < 0) {
		windowUnits = 0;
	    }
	    if (Tcl_GetInt(interp, argv[4], &firstUnit) != TCL_OK) {
		goto error;
	    }
	    if (Tcl_GetInt(interp, argv[5], &lastUnit) != TCL_OK) {
		goto error;
	    }
	    if (totalUnits > 0) {
		if (lastUnit < firstUnit) {
		    lastUnit = firstUnit;
		}
	    } else {
		firstUnit = lastUnit = 0;
	    }
	    scrollPtr->totalUnits = totalUnits;
	    scrollPtr->windowUnits = windowUnits;
	    scrollPtr->firstUnit = firstUnit;
	    scrollPtr->lastUnit = lastUnit;
	    if (scrollPtr->totalUnits == 0) {
		scrollPtr->firstFraction = 0.0;
		scrollPtr->lastFraction = 1.0;
	    } else {
		scrollPtr->firstFraction = ((double) firstUnit)/totalUnits;
		scrollPtr->lastFraction = ((double) (lastUnit+1))/totalUnits;
	    }
	    scrollPtr->flags &= ~NEW_STYLE_COMMANDS;
	} else {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
		    argv[0], " set firstFraction lastFraction\" or \"",
		    argv[0],
		    " set totalUnits windowUnits firstUnit lastUnit\"",
		    (char *) NULL);
	    goto error;
	}
	ComputeScrollbarGeometry(scrollPtr);
	EventuallyRedraw(scrollPtr);
    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be activate, cget, configure, delta, fraction, ",
		"get, identify, or set", (char *) NULL);
	goto error;
    }
    done:
    Tcl_Release((ClientData) scrollPtr);
    return result;

    error:
    Tcl_Release((ClientData) scrollPtr);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroyScrollbar --
 *
 *	This procedure is invoked by Tcl_EventuallyFree or Tcl_Release
 *	to clean up the internal structure of a scrollbar at a safe time
 *	(when no-one is using it anymore).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Everything associated with the scrollbar is freed up.
 *
 *----------------------------------------------------------------------
 */

static void
DestroyScrollbar(memPtr)
    char *memPtr;	/* Info about scrollbar widget. */
{
    register Scrollbar *scrollPtr = (Scrollbar *) memPtr;

    /*
     * Free up all the stuff that requires special handling, then
     * let Tk_FreeOptions handle all the standard option-related
     * stuff.
     */

    if (scrollPtr->troughGC != None) {
	Tk_FreeGC(scrollPtr->display, scrollPtr->troughGC);
    }
    if (scrollPtr->copyGC != None) {
	Tk_FreeGC(scrollPtr->display, scrollPtr->copyGC);
    }
    if (scrollPtr->troughBackGC != None) {
	Tk_FreeGC(scrollPtr->display, scrollPtr->troughBackGC);
    }
    if (scrollPtr->bump != None) {
	Tk_FreePixmap(scrollPtr->display, scrollPtr->bump);
    }
    if (scrollPtr->stipple != None) {
	Tk_FreeBitmap(scrollPtr->display, scrollPtr->stipple);
    }    
    if (scrollPtr->darkPtr != NULL) {
	Tk_FreeColor(scrollPtr->darkPtr);
    }    
    Tk_FreeOptions(configSpecs, (char *) scrollPtr, scrollPtr->display, 0);
    ckfree((char *) scrollPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * ConfigureScrollbar --
 *
 *	This procedure is called to process an argv/argc list, plus
 *	the Tk option database, in order to configure (or
 *	reconfigure) a scrollbar widget.
 *
 * Results:
 *	The return value is a standard Tcl result.  If TCL_ERROR is
 *	returned, then interp->result contains an error message.
 *
 * Side effects:
 *	Configuration information, such as colors, border width,
 *	etc. get set for scrollPtr;  old resources get freed,
 *	if there were any.
 *
 *----------------------------------------------------------------------
 */
static int
ConfigureScrollbar(interp, scrollPtr, argc, argv, flags)
    Tcl_Interp *interp;			/* Used for error reporting. */
    register Scrollbar *scrollPtr;	/* Information about widget;  may or
					 * may not already have values for
					 * some fields. */
    int argc;				/* Number of valid entries in argv. */
    char **argv;			/* Arguments. */
    int flags;				/* Flags to pass to
					 * Tk_ConfigureWidget. */
{
    size_t length;
    XGCValues gcValues;
    XColor newcolor;
    GC new;

    if (Tk_StrictMotif(scrollPtr->tkwin)) {
	flags|=TK_CONFIG_CHECK_MY_FLAG;
    } 
    if (Tk_ConfigureWidget(interp, scrollPtr->tkwin, configSpecs,
	    argc, argv, (char *) scrollPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }
    /*
     * A few options need special processing, such as parsing the
     * orientation or setting the background from a 3-D border.
     */

    length = strlen(scrollPtr->orientUid);
    if (strncmp(scrollPtr->orientUid, "vertical", length) == 0) {
	scrollPtr->vertical = 1;
    } else if (strncmp(scrollPtr->orientUid, "horizontal", length) == 0) {
	scrollPtr->vertical = 0;
    } else {
	Tcl_AppendResult(interp, "bad orientation \"", scrollPtr->orientUid,
		"\": must be vertical or horizontal", (char *) NULL);
	return TCL_ERROR;
    }

    if (scrollPtr->command != NULL) {
	scrollPtr->commandSize = strlen(scrollPtr->command);
    } else {
	scrollPtr->commandSize = 0;
    }

    if (scrollPtr->stipple==None) {
	scrollPtr->stipple = Tk_GetBitmap((Tcl_Interp *)NULL, scrollPtr->tkwin,
		Tk_GetUid("gray50"));
	if (scrollPtr->stipple == None) {
	    panic("ScrollBar couldn't allocate bitmap for trough");
	}
    }
    
    Tk_SetBackgroundFromBorder(scrollPtr->tkwin, scrollPtr->bgBorder);    
    /* alloc color for trough base stipple */
    newcolor.red = (60 * (int) scrollPtr->troughColorPtr->red)/100;
    newcolor.green =(60 * (int) scrollPtr->troughColorPtr->green)/100;
    newcolor.blue = (60 * (int) scrollPtr->troughColorPtr->blue)/100;
    
    if (scrollPtr->darkPtr != NULL) {
	Tk_FreeColor(scrollPtr->darkPtr);
    }    
    scrollPtr->darkPtr = Tk_GetColorByValue(scrollPtr->tkwin,
	&newcolor);

    gcValues.background = scrollPtr->troughColorPtr->pixel;
    gcValues.foreground = scrollPtr->darkPtr->pixel;   
    gcValues.stipple = scrollPtr->stipple;
    gcValues.fill_style = FillOpaqueStippled;

    new = Tk_GetGC(scrollPtr->tkwin, GCForeground|GCBackground|GCStipple|
	GCFillStyle, &gcValues);
    if (scrollPtr->troughGC != None) {
	Tk_FreeGC(scrollPtr->display, scrollPtr->troughGC);
    }
    scrollPtr->troughGC = new;
    gcValues.foreground = scrollPtr->troughColorPtr->pixel;
    new = Tk_GetGC(scrollPtr->tkwin, GCForeground, &gcValues);
    if (scrollPtr->troughBackGC != None) {
	Tk_FreeGC(scrollPtr->display, scrollPtr->troughBackGC);
    }
    scrollPtr->troughBackGC = new;
	
    if (scrollPtr->copyGC == None) {
	gcValues.graphics_exposures = False;
	scrollPtr->copyGC = Tk_GetGC(scrollPtr->tkwin, GCGraphicsExposures,
	    &gcValues);
    }

    if (Tk_IsMapped(scrollPtr->tkwin)&&(scrollPtr->flags & RECONFIGURE)) {
	scrollPtr->flags &= ~RECONFIGURE;
	if (scrollPtr->bump != None) {
	    Tk_FreePixmap(scrollPtr->display, scrollPtr->bump);
	}
	MakePixmaps(scrollPtr);
    }
    /*
     * Register the desired geometry for the window (leave enough space
     * for the two arrows plus a minimum-size slider, plus border around
     * the whole window, if any).  Then arrange for the window to be
     * redisplayed.
     */

    ComputeScrollbarGeometry(scrollPtr);
    EventuallyRedraw(scrollPtr);

    return TCL_OK;
}

#define ARROW_UP	0
#define ARROW_DOWN	1
#define ARROW_LEFT	2
#define ARROW_RIGHT	3
/*
 * DrawArrow --
 *   	Draws an antialiased arrow with direction dir and size w x h. 
 *      It's coordinates are relative to the center of the arrow. 
 */
static void DrawArrow(scrollPtr, drawable, x, y, w, h, dir)
    Scrollbar   *scrollPtr;
    Drawable	drawable;
    int 	x, y;	/* position relative to middle of arrow */
    int 	w, h;
    char	dir;
{
    int i, hw;
    float d, s;
    GC 	core, border;

    core = Tk_3DBorderGC(scrollPtr->tkwin,scrollPtr->bgBorder,TK_3D_DARK2_GC);
    border = Tk_3DBorderGC(scrollPtr->tkwin,scrollPtr->bgBorder,TK_3D_DARK_GC);

    if (h<3) h=3;
    if (w<3) w=3;
    if (dir<2) { /* vertical */
	hw = w/2;
	d = ((float)w/2)/(float)h;
	y -= h/2;	
    } else {
	hw = h/2;
	d = ((float)h/2)/(float)w;
	x -= w/2;
    }    
    s=0;
    switch (dir) {
     case ARROW_DOWN:
	s=h*d+0.5;
	d=-d;
     case ARROW_UP:
	for(i = 0; i < h; i++) {
	    s += d;	
	    XDrawLine(scrollPtr->display, drawable, core,
		      (unsigned)(x-(int)(s-0.5)), y+i,
		      (unsigned)(x+(int)(s-0.5)), y+i);
	    /* anti-aliasing */	
	    if ((int)(s+0.5)!=(int)s) {
		XDrawPoint(scrollPtr->display, drawable, 
			   border, (unsigned)(x+(int)s), y+i);
		XDrawPoint(scrollPtr->display, drawable,
			   border, (unsigned)(x-(int)s), y+i);
	    }
	}
	break;
     case ARROW_RIGHT:
	s=h*d+0.5;
	d=-d;
     case ARROW_LEFT:
	for(i = 0; i < h; i++) {
	    s += d;	
	    XDrawLine(scrollPtr->display, drawable,
		      core, x+i,(unsigned)(y-(int)(s-0.5)),
		      x+i,(unsigned)(y+(int)(s-0.5)));
	    /* anti-aliasing */	
	    if ((int)(s+0.5)!=(int)s) {
		XDrawPoint(scrollPtr->display, drawable,
			   border, x+i,(unsigned)(y+(int)s));
		XDrawPoint(scrollPtr->display, drawable,
			   border, x+i,(unsigned)(y-(int)s));
	    }
	}
	break;
    }
}


/*
 *--------------------------------------------------------------
 *
 * DisplayScrollbar --
 *
 *	This procedure redraws the contents of a scrollbar window.
 *	It is invoked as a do-when-idle handler, so it only runs
 *	when there's nothing else for the application to do.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Information appears on the screen.
 *
 *--------------------------------------------------------------
 */

static void
DisplayScrollbar(clientData)
    ClientData clientData;	/* Information about window. */
{
    register Scrollbar *scrollPtr = (Scrollbar *) clientData;
    register Tk_Window tkwin = scrollPtr->tkwin;
    Tk_3DBorder border;
    int relief, width, elementBorderWidth, sliderSize, delta;
    int arrowl = scrollPtr->arrowLength, arrsize;
    int buttonSize;
    Pixmap pixmap;
    int empty=0;	

    if ((scrollPtr->tkwin == NULL) || !Tk_IsMapped(tkwin)) {
	goto done;
    }

    if ((scrollPtr->firstFraction<=0.0) && (scrollPtr->lastFraction>=1.0)) {
	empty=1;
    }
    
    if (scrollPtr->vertical) {
	width = Tk_Width(tkwin) - 2*scrollPtr->inset;
    } else {
	width = Tk_Height(tkwin) - 2*scrollPtr->inset;
    }
    elementBorderWidth = scrollPtr->elementBorderWidth;
    if (elementBorderWidth < 0) {
	elementBorderWidth = scrollPtr->borderWidth;
    }

    if (scrollPtr->bump == None) {
	MakePixmaps((ClientData)scrollPtr);
    }

    /*
     * In order to avoid screen flashes, this procedure redraws
     * the scrollbar in a pixmap, then copies the pixmap to the
     * screen in a single operation.  This means that there's no
     * point in time where the on-sreen image has been cleared.
     */

    pixmap = Tk_GetPixmap(scrollPtr->display, Tk_WindowId(tkwin),
	    Tk_Width(tkwin), Tk_Height(tkwin), Tk_Depth(tkwin));

    sliderSize = scrollPtr->sliderLast - scrollPtr->sliderFirst;
    if (scrollPtr->highlightWidth != 0) {
	GC gc;

	if (scrollPtr->flags & GOT_FOCUS) {
	    gc = Tk_GCForColor(scrollPtr->highlightColorPtr, pixmap);
	} else {
	    gc = Tk_GCForColor(scrollPtr->highlightBgColorPtr, pixmap);
	}
	Tk_DrawFocusHighlight(tkwin, gc, scrollPtr->highlightWidth, pixmap);
    }
    XFillRectangle(scrollPtr->display, pixmap, scrollPtr->troughBackGC,
	scrollPtr->highlightWidth, scrollPtr->highlightWidth,
 	Tk_Width(tkwin) - 2*scrollPtr->highlightWidth,
  	Tk_Height(tkwin) - 2*scrollPtr->highlightWidth);
    
    XDrawRectangle(scrollPtr->display, pixmap, 
	Tk_3DBorderGC(scrollPtr->tkwin,scrollPtr->bgBorder,TK_3D_DARK2_GC),
 	scrollPtr->highlightWidth, scrollPtr->highlightWidth,
 	Tk_Width(tkwin) - 2*scrollPtr->highlightWidth-1,
 	Tk_Height(tkwin) - 2*scrollPtr->highlightWidth-1);

    buttonSize = arrowl+1;
    if (empty) {
	XFillRectangle(scrollPtr->display, pixmap, scrollPtr->troughGC,
		   scrollPtr->inset, scrollPtr->inset,
		   (unsigned) (Tk_Width(tkwin) - 2*scrollPtr->inset),
		   (unsigned) (Tk_Height(tkwin) - 2*scrollPtr->inset));
    } else {
	if (scrollPtr->vertical) {
	    XFillRectangle(scrollPtr->display, pixmap, scrollPtr->troughGC,
			   scrollPtr->inset, scrollPtr->inset,
			   (unsigned) (Tk_Width(tkwin) - 2*scrollPtr->inset),
			   (unsigned) (Tk_Height(tkwin) - 2*buttonSize -
				       2*scrollPtr->inset));
	} else {
	    XFillRectangle(scrollPtr->display, pixmap, scrollPtr->troughGC,
			   scrollPtr->inset+(arrowl+1)*2, scrollPtr->inset,
			   (unsigned) (Tk_Width(tkwin) - 2*buttonSize -
				       2*scrollPtr->inset),
			   (unsigned) (Tk_Height(tkwin) - 2*scrollPtr->inset));
	}
    }    

    /* 
     * don't draw slider/arrows if everything is visible
     */
    if (empty) {
	goto docopy;
    }    
#ifdef FULL_STEP    
    if (scrollPtr->noArrows) {
	goto doSlider;
    }
#endif    
    if (scrollPtr->activeField == TOP_ARROW) {
	border = scrollPtr->activeBorder;
	relief = scrollPtr->activeField == TOP_ARROW ? scrollPtr->activeRelief
		: TK_RELIEF_RAISED;
	delta = (relief == TK_RELIEF_SUNKEN) ? (scrollPtr->borderWidth)/2+1 : 0;
    } else {
	border = scrollPtr->bgBorder;
	relief = TK_RELIEF_RAISED;
	delta = 0;
    }
    
    arrsize = scrollPtr->arrowLength-elementBorderWidth*2-3;
    if (scrollPtr->vertical) {
	Tk_Fill3DRectangle(tkwin, pixmap, border, scrollPtr->inset,
			   Tk_Height(tkwin) - 2*buttonSize -
			   scrollPtr->inset+1, arrowl, arrowl,
			   elementBorderWidth, relief);
	DrawArrow(scrollPtr, pixmap, Tk_Width(tkwin)/2 - 1 + delta,
		  Tk_Height(tkwin) - buttonSize*2 -
		  scrollPtr->inset + buttonSize/2 + delta,
		  arrsize, arrsize, ARROW_UP);
    } else {
	Tk_Fill3DRectangle(tkwin, pixmap, border,
			   scrollPtr->inset, scrollPtr->inset,
			   arrowl, arrowl, elementBorderWidth, relief);
	DrawArrow(scrollPtr, pixmap, scrollPtr->inset + buttonSize/2 + delta-1,
		  (Tk_Height(tkwin)+1)/2 - 1 + delta,
		  arrsize, arrsize, ARROW_LEFT);
    }
    
    /*
     * Display the bottom or right arrow.
     */

    if (scrollPtr->activeField == BOTTOM_ARROW) {
	border = scrollPtr->activeBorder;
	relief = scrollPtr->activeField == BOTTOM_ARROW
		? scrollPtr->activeRelief : TK_RELIEF_RAISED;
	delta = (relief == TK_RELIEF_SUNKEN) ? (scrollPtr->borderWidth)/2+1 : 0;
    } else {
	border = scrollPtr->bgBorder;
	relief = TK_RELIEF_RAISED;
	delta = 0;
    }
    if (scrollPtr->vertical) {
	Tk_Fill3DRectangle(tkwin, pixmap, border, scrollPtr->inset,
	 	Tk_Height(tkwin)-buttonSize-scrollPtr->inset+1,
		arrowl, arrowl,
		elementBorderWidth, relief);
	DrawArrow(scrollPtr, pixmap, Tk_Width(tkwin)/2 - 1 + delta,
		  Tk_Height(tkwin) - buttonSize + buttonSize/2 - 
		  scrollPtr->inset+ delta, arrsize, arrsize, ARROW_DOWN);
    } else {
	Tk_Fill3DRectangle(tkwin, pixmap, border,
	 	scrollPtr->inset + buttonSize, scrollPtr->inset, 
		arrowl, arrowl, elementBorderWidth, relief);
	DrawArrow(scrollPtr, pixmap, 
		  scrollPtr->inset + buttonSize + buttonSize/2 + delta-1,
		  (Tk_Height(tkwin)+1)/2 - 1 + delta,
		  arrsize, arrsize, ARROW_RIGHT);
    }
#ifdef FULL_STEP
    doSlider:
#endif    
    /*
     * Display the slider.
     */
    if (scrollPtr->activeField == SLIDER) {
	border = scrollPtr->activeBorder;
	relief = TK_RELIEF_RAISED;
    } else {
	border = scrollPtr->bgBorder;
	relief = TK_RELIEF_RAISED;
    }
    if (scrollPtr->vertical) {
	Tk_Fill3DRectangle(tkwin, pixmap, border,
		scrollPtr->inset, scrollPtr->sliderFirst,
		width, sliderSize,
		elementBorderWidth, relief);
	if ((sliderSize > SB_BUMP_HEIGHT*2) && (width > SB_BUMP_WIDTH)) {
	    XCopyArea(scrollPtr->display, scrollPtr->bump, pixmap,
		scrollPtr->copyGC, 0, 0, SB_BUMP_WIDTH, SB_BUMP_HEIGHT,
		(Tk_Width(tkwin) - SB_BUMP_WIDTH)/2,
		scrollPtr->sliderFirst + (sliderSize - 1 - SB_BUMP_HEIGHT)/2);
	} else {
	    int bump_size;
	    
	    bump_size = width - 2*(scrollPtr->inset+4);
	    if ((sliderSize > bump_size+2) && (bump_size > 1)) {
		Tk_Fill3DRectangle(tkwin, pixmap, border,
		    (Tk_Width(tkwin) - bump_size)/2,
		    scrollPtr->sliderFirst + (sliderSize - bump_size)/2,
		    bump_size, bump_size, 2, TK_RELIEF_SUNKEN);
	    }	    
	}	
    } else {
	Tk_Fill3DRectangle(tkwin, pixmap, border,
		scrollPtr->sliderFirst, scrollPtr->inset,
		sliderSize, width,
		elementBorderWidth, relief);
	if ((sliderSize > SB_BUMP_HEIGHT*2) && (width > SB_BUMP_HEIGHT)) {
	    XCopyArea(scrollPtr->display, scrollPtr->bump, pixmap,
		scrollPtr->copyGC, 0, 0, SB_BUMP_WIDTH, SB_BUMP_HEIGHT,
		scrollPtr->sliderFirst + (sliderSize - SB_BUMP_HEIGHT)/2,
		(Tk_Height(tkwin) - 1 - SB_BUMP_WIDTH)/2);
	} else {
	    int bump_size;
	    
	    bump_size = width - 2*(scrollPtr->inset+4);	    
	    if ((sliderSize > bump_size+2) && (bump_size > 1)) {
		Tk_Fill3DRectangle(tkwin, pixmap, border,
		   scrollPtr->sliderFirst + (sliderSize - bump_size)/2-1, 
		   (Tk_Height(tkwin) - bump_size)/2,
		   bump_size, bump_size, 2, TK_RELIEF_SUNKEN);
	    }
	}
    }

    /*
     * Copy the information from the off-screen pixmap onto the screen,
     * then delete the pixmap.
     */
    docopy:
    XCopyArea(scrollPtr->display, pixmap, Tk_WindowId(tkwin),
	    scrollPtr->copyGC, 0, 0, (unsigned) Tk_Width(tkwin),
	    (unsigned) Tk_Height(tkwin), 0, 0);
    Tk_FreePixmap(scrollPtr->display, pixmap);

    done:
    scrollPtr->flags &= ~REDRAW_PENDING;
}

/*
 *--------------------------------------------------------------
 *
 * ScrollbarEventProc --
 *
 *	This procedure is invoked by the Tk dispatcher for various
 *	events on scrollbars.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	When the window gets deleted, internal structures get
 *	cleaned up.  When it gets exposed, it is redisplayed.
 *
 *--------------------------------------------------------------
 */

static void
ScrollbarEventProc(clientData, eventPtr)
    ClientData clientData;	/* Information about window. */
    XEvent *eventPtr;		/* Information about event. */
{
    Scrollbar *scrollPtr = (Scrollbar *) clientData;

    if ((eventPtr->type == Expose) && (eventPtr->xexpose.count == 0)) {
	EventuallyRedraw(scrollPtr);
    } else if (eventPtr->type == DestroyNotify) {
	if (scrollPtr->tkwin != NULL) {
	    scrollPtr->tkwin = NULL;
	    Tcl_DeleteCommand(scrollPtr->interp,
		    Tcl_GetCommandName(scrollPtr->interp,
		    scrollPtr->widgetCmd));
	}
	if (scrollPtr->flags & REDRAW_PENDING) {
	    Tcl_CancelIdleCall(DisplayScrollbar, (ClientData) scrollPtr);
	}
	Tcl_EventuallyFree((ClientData) scrollPtr, DestroyScrollbar);
    } else if (eventPtr->type == ConfigureNotify) {
	ComputeScrollbarGeometry(scrollPtr);
	EventuallyRedraw(scrollPtr);
    } else if (eventPtr->type == FocusIn) {
	if (eventPtr->xfocus.detail != NotifyInferior) {
	    scrollPtr->flags |= GOT_FOCUS;
	    if (scrollPtr->highlightWidth > 0) {
		EventuallyRedraw(scrollPtr);
	    }
	}
    } else if (eventPtr->type == FocusOut) {
	if (eventPtr->xfocus.detail != NotifyInferior) {
	    scrollPtr->flags &= ~GOT_FOCUS;
	    if (scrollPtr->highlightWidth > 0) {
		EventuallyRedraw(scrollPtr);
	    }
	}
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ScrollbarCmdDeletedProc --
 *
 *	This procedure is invoked when a widget command is deleted.  If
 *	the widget isn't already in the process of being destroyed,
 *	this command destroys it.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The widget is destroyed.
 *
 *----------------------------------------------------------------------
 */

static void
ScrollbarCmdDeletedProc(clientData)
    ClientData clientData;	/* Pointer to widget record for widget. */
{
    Scrollbar *scrollPtr = (Scrollbar *) clientData;
    Tk_Window tkwin = scrollPtr->tkwin;

    /*
     * This procedure could be invoked either because the window was
     * destroyed and the command was then deleted (in which case tkwin
     * is NULL) or because the command was deleted, and then this procedure
     * destroys the widget.
     */

    if (tkwin != NULL) {
	scrollPtr->tkwin = NULL;
	Tk_DestroyWindow(tkwin);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ComputeScrollbarGeometry --
 *
 *	After changes in a scrollbar's size or configuration, this
 *	procedure recomputes various geometry information used in
 *	displaying the scrollbar.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The scrollbar will be displayed differently.
 *
 *----------------------------------------------------------------------
 */

static void
ComputeScrollbarGeometry(scrollPtr)
    register Scrollbar *scrollPtr;	/* Scrollbar whose geometry may
					 * have changed. */
{
    int width, fieldLength, minLength;

    if (scrollPtr->highlightWidth < 0) {
	scrollPtr->highlightWidth = 0;
    }
    scrollPtr->inset = scrollPtr->highlightWidth + 2;
    width = (scrollPtr->vertical) ? Tk_Width(scrollPtr->tkwin)
	    : Tk_Height(scrollPtr->tkwin);
#ifdef FULL_STEP
    if (scrollPtr->noArrows) {
	scrollPtr->arrowLength = -1;
    } else
#endif
      scrollPtr->arrowLength = width - 2*scrollPtr->inset;

    fieldLength = (scrollPtr->vertical ? Tk_Height(scrollPtr->tkwin)
	    : Tk_Width(scrollPtr->tkwin))
	    - 2*(scrollPtr->arrowLength + 1 + scrollPtr->inset);
    
    minLength = width - 2*scrollPtr->inset;
    if (minLength < MIN_SLIDER_LENGTH)
      minLength = MIN_SLIDER_LENGTH;
    
    if (fieldLength < 0) {
	fieldLength = 0;
    }    
    scrollPtr->sliderFirst = fieldLength*scrollPtr->firstFraction;
    scrollPtr->sliderLast = fieldLength*scrollPtr->lastFraction;

    /*
     * Adjust the slider so that some piece of it is always
     * displayed in the scrollbar and so that it has at least
     * a minimal width (so it can be grabbed with the mouse).
     */

    if (scrollPtr->sliderFirst > (fieldLength - 2*scrollPtr->borderWidth)) {
	scrollPtr->sliderFirst = fieldLength - 2*scrollPtr->borderWidth;
    }
    if (scrollPtr->sliderFirst < 0) {
	scrollPtr->sliderFirst = 0;
    }
    if (scrollPtr->sliderLast < scrollPtr->sliderFirst + minLength) {
	scrollPtr->sliderLast = scrollPtr->sliderFirst + minLength;
    }
    if (scrollPtr->sliderLast > fieldLength) {
	scrollPtr->sliderLast = fieldLength;
    }
    if (scrollPtr->vertical) {	
	scrollPtr->sliderFirst += scrollPtr->inset;
	scrollPtr->sliderLast += scrollPtr->inset;
    } else {
	scrollPtr->sliderFirst += (scrollPtr->arrowLength+1)*2+scrollPtr->inset;
	scrollPtr->sliderLast += (scrollPtr->arrowLength+1)*2+scrollPtr->inset;
    }        

    /*
     * Register the desired geometry for the window (leave enough space
     * for the two arrows plus a minimum-size slider, plus border around
     * the whole window, if any).  Then arrange for the window to be
     * redisplayed.
     */

    if (scrollPtr->vertical) {
	Tk_GeometryRequest(scrollPtr->tkwin,
		scrollPtr->width + 2*scrollPtr->inset,
		2*(scrollPtr->arrowLength + scrollPtr->borderWidth
		+ scrollPtr->inset + 1));
    } else {
	Tk_GeometryRequest(scrollPtr->tkwin,
		2*(scrollPtr->arrowLength + scrollPtr->borderWidth
		+ scrollPtr->inset + 1),scrollPtr->width + 2*scrollPtr->inset);
    }
    Tk_SetInternalBorder(scrollPtr->tkwin, scrollPtr->inset);
}

/*
 *--------------------------------------------------------------
 *
 * ScrollbarPosition --
 *
 *	Determine the scrollbar element corresponding to a
 *	given position.
 *
 * Results:
 *	One of TOP_ARROW, TOP_GAP, etc., indicating which element
 *	of the scrollbar covers the position given by (x, y).  If
 *	(x,y) is outside the scrollbar entirely, then OUTSIDE is
 *	returned.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static int
ScrollbarPosition(scrollPtr, x, y)
    register Scrollbar *scrollPtr;	/* Scrollbar widget record. */
    int x, y;				/* Coordinates within scrollPtr's
					 * window. */
{
    int length, width, tmp;

    if (scrollPtr->vertical) {
	length = Tk_Height(scrollPtr->tkwin);
	width = Tk_Width(scrollPtr->tkwin);
    } else {
	tmp = x;
	x = y;
	y = tmp;
	length = Tk_Width(scrollPtr->tkwin);
	width = Tk_Height(scrollPtr->tkwin);
    }

    if ((x < scrollPtr->inset) || (x >= (width - scrollPtr->inset))
	    || (y < scrollPtr->inset) || (y >= (length - scrollPtr->inset))) {
	return OUTSIDE;
    }

    /*
     * All of the calculations in this procedure mirror those in
     * DisplayScrollbar.  Be sure to keep the two consistent.
     */
    if (scrollPtr->vertical) {	
	if (y < scrollPtr->sliderFirst) {
	    return TOP_GAP;
	}
	if (y < scrollPtr->sliderLast) {	    
	    return SLIDER;
	}
#ifdef FULL_STEP
	if (scrollPtr->noArrows || 
	    y < (length - ((scrollPtr->arrowLength+1)*2 + scrollPtr->inset))) {
#else	
	if (y < (length - ((scrollPtr->arrowLength+1)*2 + scrollPtr->inset))) {
#endif	    
	    return BOTTOM_GAP;
	}
#ifdef FULL_STEP	
	if (y < (length - (scrollPtr->arrowLength +1+ scrollPtr->inset))
	    && !scrollPtr->noArrows) {
#else
	if (y < (length - (scrollPtr->arrowLength+1+ scrollPtr->inset))) {
#endif	    
	    return TOP_ARROW;
	}	
	return BOTTOM_ARROW;
    } else {
#ifdef FULL_STEP
	if (!scrollPtr->noArrows && 
	    y < (scrollPtr->arrowLength + 1+scrollPtr->inset)) {
#else	
	if (y < (scrollPtr->arrowLength + 1+scrollPtr->inset)) {
#endif	    
	    return TOP_ARROW;
	}
#ifdef FULL_STEP
	if (!scrollPtr->noArrows &&
	    y < ((scrollPtr->arrowLength+1)*2 + scrollPtr->inset)) {
#else	    
	if (y < ((scrollPtr->arrowLength+1)*2 + scrollPtr->inset)) {
#endif	    
	    return BOTTOM_ARROW;
	}
	if (y < (scrollPtr->sliderFirst)) {
	    return TOP_GAP;
	}
	if (y < (scrollPtr->sliderLast)) {
	    return SLIDER;
	}
	return BOTTOM_GAP;
    }
}

/*
 *--------------------------------------------------------------
 *
 * EventuallyRedraw --
 *
 *	Arrange for one or more of the fields of a scrollbar
 *	to be redrawn.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *--------------------------------------------------------------
 */

static void
EventuallyRedraw(scrollPtr)
    register Scrollbar *scrollPtr;	/* Information about widget. */
{
    if ((scrollPtr->tkwin == NULL) || (!Tk_IsMapped(scrollPtr->tkwin))) {
	return;
    }
    if ((scrollPtr->flags & REDRAW_PENDING) == 0) {
	Tcl_DoWhenIdle(DisplayScrollbar, (ClientData) scrollPtr);
	scrollPtr->flags |= REDRAW_PENDING;
    }
}
