/* Copyright 1997-1998 by Corporation for National Research Initiatives.
 * See the file LICENSE for details.
 */

#include <Python.h>
#include <t1lib.h>


static PyObject *T1Error = NULL;

static inited = 0;
static init_logging = NO_LOGFILE;

static int
t1lib_init(logging_p)
     int logging_p;
{
    if (logging_p == -1)
	logging_p = init_logging;
    if (inited && logging_p != init_logging) {
	PyErr_SetString(T1Error,
			"cannot re-initialize t1lib with changed logging");
	return (0);
    }
    if (!inited) {
	if (T1_InitLib(logging_p) == NULL) {
	    PyErr_SetString(T1Error, "failed call to T1_InitLib()");
	    return (0);
	}
	init_logging = logging_p;
	inited = 1;
    }
    return (1);

}   /* t1lib_init() */


static PyObject*
InitLib(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int log = -1;

    if (PyArg_ParseTuple(args, "|i:InitLib", &log)) {
	if (t1lib_init(log))
	    Py_INCREF(result = Py_None);
    }
    return (result);

}   /* InitLib() */


static PyObject*
SetBitmapPad(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int pad;

    if (PyArg_ParseTuple(args, "i:SetBitmapPad", &pad)) {
	if ((pad != 8) && (pad != 16) && (pad != 32))
	    PyErr_SetString(PyExc_ValueError, "padding must be 8, 16, or 32");
	else {
	    int rc = T1_SetBitmapPad(pad);

	    if (rc == 0)
		Py_INCREF(result = Py_None);
	    else if (rc == -1)
		PyErr_SetString(T1Error, "t1lib already initialized");
	    else {
		char buffer[80];

		sprintf(buffer,
			"Unknown error: T1_SetBitmapPad() returned %d",
			rc);
		PyErr_SetString(T1Error, buffer);
	    }
	}
    }
    return (result);

}   /* SetBitmapPad() */


static PyObject*
GetBitmapPad(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;

    if (PyArg_ParseTuple(args, ":GetBitmapPad")) {
	result = PyInt_FromLong(T1_GetBitmapPad());
    }
    return (result);

}   /* GetBitmapPad() */


static PyObject*
SetLogLevel(self, args)
     PyObject *self;
     PyObject *args;
{
    int log_level;

    if (PyArg_ParseTuple(args, "i:SetLogLevel", &log_level)) {
	/* does not require T1_InitLib() */
	T1_SetLogLevel(log_level);
	Py_INCREF(Py_None);

	return (Py_None);
    }
    return (NULL);

}   /* SetLogLevel() */


static PyObject*
PrintLog(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    unsigned char *func_ident;
    unsigned char *msg_txt;
    int level = T1LOG_WARNING;

    if (PyArg_ParseTuple(args, "ss|i:PrintLog",
			 &func_ident, &msg_txt, &level)) {
	if (!t1lib_init(-1))
	    goto finally;
	T1_PrintLog(func_ident, msg_txt, level);
	result = Py_None;
	Py_INCREF(result);
    }
 finally:
    return (result);

}   /* PrintLog() */


/* T1_LoadFont() is not documented but included in t1lib.h. */
static PyObject*
LoadFont(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int FontID;

    if (PyArg_ParseTuple(args, "i:LoadFont", &FontID)) {
	int rc;

	if (!t1lib_init(-1))
	    return (NULL);
	rc = T1_LoadFont(FontID);
	if (rc == -1)
	    PyErr_SetString(T1Error, "illegal font id or could not load font");
	else {
	    result = Py_None;
	    Py_INCREF(result);
	}
    }
    return (result);

}   /* LoadFont() */


static PyObject*
convert_glyph(glyph)
     GLYPH *glyph;
{
    int nbytes, width, height, bpp;

    if (glyph == NULL) {
	PyErr_SetString(T1Error, "received NULL glyph from t1lib");
	return (NULL);
    }
    width = glyph->metrics.rightSideBearing - glyph->metrics.leftSideBearing;
    height = glyph->metrics.ascent + glyph->metrics.descent;
    bpp = glyph->bpp;
    if (bpp == 24)
	bpp = 32;
    nbytes = (width * bpp) / 8;
    if ((width * bpp) % 8)
	++nbytes;
    nbytes *= height;

    return (Py_BuildValue("s#(iiiiii)i", glyph->bits, nbytes,
			  glyph->metrics.ascent,
			  glyph->metrics.descent,
			  glyph->metrics.leftSideBearing,
			  glyph->metrics.rightSideBearing,
			  glyph->metrics.advanceX,
			  glyph->metrics.advanceY,
			  glyph->bpp));

}   /* convert_glyph() */


static PyObject*
set_char(args, argfmt, setfunc)
     PyObject *args;
     char *argfmt;
     GLYPH *(*setfunc)(int, unsigned char, float, float);
{
    PyObject *result = NULL;
    int FontID;
    unsigned char charcode;
    float size, angle = 0.0;

    if (PyArg_ParseTuple(args, argfmt, &FontID, &charcode, &size, &angle)) {
	if (t1lib_init(-1)) {
	    GLYPH *glyph = setfunc(FontID, charcode, size, angle);

	    result = convert_glyph(glyph);
	}
    }
    return (result);

}   /* set_char() */


static PyObject*
SetChar(self, args)
     PyObject *self;
     PyObject *args;
{
    return (set_char(args, "icf|f:SetChar", T1_SetChar));

}   /* SetChar() */


static PyObject*
AASetChar(self, args)
     PyObject *self;
     PyObject *args;
{
    return (set_char(args, "icf|f:AASetChar", T1_AASetChar));

}   /* AASetChar() */


static PyObject*
set_string(args, argfmt, setfunc)
     PyObject *args;
     char *argfmt;
     GLYPH *(*setfunc)(int, unsigned char*, int, long, int, float, float);
{
    PyObject *result = NULL;
    PyObject *kerning;
    int FontID;
    unsigned char *string;
    int len;
    long spaceoff;
    float size, angle = 0.0;

    if (PyArg_ParseTuple(args, argfmt, &FontID, &string, &len, &spaceoff,
			 &kerning, &size, &angle)) {
	fflush(stdout);
	if (t1lib_init(-1)) {
	    GLYPH *glyph = setfunc(FontID, string, len, spaceoff,
				   (PyObject_IsTrue(kerning)
				    ? T1_KERNING : 0),
				   size, angle);

	    result = convert_glyph(glyph);
	}
    }
    return (result);

}   /* set_string() */


static PyObject*
SetString(self, args)
     PyObject *self;
     PyObject *args;
{
    return (set_string(args, "is#iOf|f:SetString", T1_SetString));

}   /* AASetString() */


static PyObject*
AASetString(self, args)
     PyObject *self;
     PyObject *args;
{
    return (set_string(args, "is#iOf|f:AASetString", T1_AASetString));

}   /* AASetString() */


static PyObject*
SetDeviceResolutions(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    float x_res, y_res = -1.0;

    if (PyArg_ParseTuple(args, "f|f:SetDeviceResolutions", &x_res, &y_res)) {
	if (!t1lib_init(-1))
	    goto finally;
	if (y_res == -1.0)
	    y_res = x_res;
	/*
	 * T1_SetDeviceResolutions() has an undocumented return value;
	 * if t1lib has not been initialized, it returns -1.  This should
	 * not be a factor here.
	 */
	T1_SetDeviceResolutions(x_res, y_res);
	result = Py_None;
	Py_INCREF(result);
    }
 finally:
    return (result);

}   /* SetDeviceResolutions() */


static PyObject*
LoadEncoding(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    PyObject *tuple = NULL;
    PyObject *temp;
    char *filename = NULL;
    char **encoding = NULL;

    if (PyArg_ParseTuple(args, "s:LoadEncoding", &filename)) {
	int i;

	if (!t1lib_init(-1))
	    return (NULL);
	if ((encoding = T1_LoadEncoding(filename)) == NULL) {
	    PyErr_SetString(T1Error, "error loading font encoding");
	    return (NULL);
	}
	if ((tuple = PyTuple_New(256)) == NULL)
	    goto finally;
	for (i = 0; i < 256; ++i) {
	    if ((temp = PyString_FromString(encoding[i])) == NULL)
		goto finally;
	    PyTuple_SET_ITEM(tuple, i, temp);
	}
	result = tuple;
	tuple = NULL;
    }
 finally:
    if (encoding != NULL) {
	free(encoding[0]);
	free(encoding);
	Py_XDECREF(tuple);
    }
    return (result);

}   /* LoadEncoding() */


static int num_encodings = 0;
static char ***map_encodings = NULL;

static char**
get_encoding(FontID)
     int FontID;
{
    if (FontID <= num_encodings)
	return (NULL);
    return (map_encodings[FontID]);

}   /* get_encoding() */


static int
set_encoding(FontID, encoding)
     int FontID;
     char **encoding;
{
    if (encoding == NULL) {
	if (get_encoding(FontID) != NULL) {
	    /* if true, storage is already available */
	    T1_DeleteEncoding(map_encodings[FontID]);
	    map_encodings[FontID] = NULL;
	}
    }
    else {
	if (FontID >= num_encodings) {
	    /* allocate the vector of encoding vectors */
	    int num = T1_Get_no_fonts();
	    char ***vec = (char***)realloc(map_encodings,
					   num * sizeof(char**));

	    if (vec == NULL) {
		PyErr_NoMemory();
		return (0);
	    }
	    memset(vec + num_encodings, 0,
		   ((num - num_encodings) * sizeof(char**)));
	    map_encodings = vec;
	    num_encodings = num;
	}
	if (map_encodings[FontID] != NULL)
	    T1_DeleteEncoding(map_encodings[FontID]);
	map_encodings[FontID] = encoding;
    }
    return (1);
    
}   /* set_encoding() */


static PyObject*
ReencodeFont(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    PyObject *seq = NULL;
    int FontID;

    if (PyArg_ParseTuple(args, "i|O:ReencodeFont", &FontID, &seq)) {
	char **encvector = NULL;
	char **encoding;

	if (seq == Py_None)		/* make None the same as omitting */
	    seq = NULL;
	if (seq != NULL) {
	    PyObject *charname;
	    int charbufsize = 0;
	    char *charbuf;
	    char *cp;
	    int i;

	    if ((encoding = (char**)Py_Malloc(256 * sizeof(char*))) == NULL)
		return (NULL);

	    /* the encoding vector block will be leaked */
	    if (!PySequence_Check(seq)) {
		PyErr_SetString(PyExc_TypeError,
				"encoding vector must be a sequence");
		Py_Free(encoding);
		return (NULL);
	    }
	    if (PySequence_Length(seq) != 256) {
		PyErr_SetString(PyExc_ValueError,
				"encoding vector must have 256 entries");
		Py_Free(encoding);
		return (NULL);
	    }
	    memset(encoding, 0, sizeof(encoding));
	    for (i = 0; i < 256; ++i) {
		if ((charname = PySequence_GetItem(seq, i)) == NULL) {
		    Py_Free(encoding);
		    return (NULL);
		}
		if (!PyString_Check(charname)) {
		    PyErr_SetString(PyExc_TypeError,
				    "encoding vector entries must be strings");
		    Py_Free(encoding);
		    return (NULL);
		}
		charbufsize += PyString_GET_SIZE(charname) + 1;
		Py_DECREF(charname);
	    }
	    if ((charbuf = (char*)Py_Malloc(charbufsize)) == NULL) {
		Py_Free(encoding);
		return NULL;
	    }
	    cp = charbuf;
	    for (i = 0; i < 256; ++i) {
		encoding[i] = cp;
		charname = PySequence_GetItem(seq, i);
		memcpy(cp, PyString_AS_STRING(charname),
		       PyString_GET_SIZE(charname));
		cp += PyString_GET_SIZE(charname);
		*cp++ = (char)'\0';
		Py_DECREF(charname);
	    }
	    encvector = encoding;
	}
	if (t1lib_init(-1)) {
	    if (T1_ReencodeFont(FontID, encvector)) {
		PyErr_SetString(T1Error, "could not reencode font");
		T1_DeleteEncoding(encvector);
	    }
	    else if (set_encoding(FontID, encvector))
		    Py_INCREF(result = Py_None);
	}
	/* note that the encoding vector is not reclaimed.... */
    }
    return (result);

}   /* ReencodeFont() */


static PyObject*
DeleteSize(self, args)
     PyObject*self;
     PyObject *args;
{
    PyObject *result = NULL;
    int FontID;
    float size;

    if (PyArg_ParseTuple(args, "if:DeleteSize", &FontID, &size)) {
	if (!t1lib_init(-1))
	    return (NULL);
	if (T1_DeleteSize(FontID, size))
	    PyErr_SetString(T1Error, "no matching size-dependent data");
	else {
	    result = Py_None;
	    Py_INCREF(result);
	}
    }
    return (result);
    
}   /* DeleteSize() */


static PyObject*
DeleteAllSizes(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int FontID;

    if (PyArg_ParseTuple(args, "i:DeleteAllSizes", &FontID)) {
	if (t1lib_init(-1)) {
	    int status = T1_DeleteAllSizes(FontID);

	    if (status < 0)
		PyErr_SetString(T1Error, "error deleting size-dependent data");
	    else
		result = PyInt_FromLong(status);
	}
    }
    return (result);
    
}   /* DeleteAllSizes() */


static PyObject*
DeleteFont(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int FontID;

    if (PyArg_ParseTuple(args, "i:DeleteFont", &FontID)) {
	if (t1lib_init(-1)) {
	    int status = T1_DeleteFont(FontID);

	    if (status < 0)
		PyErr_SetString(T1Error, "invalid font id");
	    else if (status > 0)
		PyErr_SetString(T1Error,
				"font still referenced by logical fonts");
	    else {
		result = Py_None;
		Py_INCREF(result);
	    }
	}
    }
    return (result);

}   /* DeleteFont() */


static PyObject*
Get_no_fonts(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;

    if (PyArg_ParseTuple(args, ":Get_no_fonts")) {
	if (t1lib_init(-1))
	    result = PyInt_FromLong(T1_Get_no_fonts());
    }
    return (result);
    
}   /* Get_no_fonts() */


static PyObject*
GetKerning(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int FontID;
    char char1, char2;

    if (PyArg_ParseTuple(args, "icc:GetKerning", &FontID, &char1, &char2)) {
	if (t1lib_init(-1)) {
	    int status = T1_GetKerning(FontID, char1, char2);

	    if (status < 0)
		PyErr_SetString(T1Error, "font is not loaded");
	    else
		result = PyInt_FromLong(status);
	}
    }
    return (result);

}   /* GetKerning() */


static PyObject*
GetCharWidth(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int FontID;
    char char1;

    if (PyArg_ParseTuple(args, "ic:GetCharWidth", &FontID, &char1)) {
	if (t1lib_init(-1)) {
	    int width = T1_GetCharWidth(FontID, char1);

	    if (width < 0)
		PyErr_SetString(T1Error, "font is not loaded");
	    else
		result = PyInt_FromLong(width);
	}
    }
    return (result);

}   /* GetCharWidth() */


static PyObject*
GetCharBBox(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int FontID;
    char char1;

    if (PyArg_ParseTuple(args, "ic:GetCharBBox", &FontID, &char1)) {
	if (t1lib_init(-1)) {
	    BBox bbox = T1_GetCharBBox(FontID, char1);

	    result = Py_BuildValue("iiii", bbox.llx, bbox.lly,
				   bbox.urx, bbox.ury);
	}
    }
    return (result);

}   /* GetCharBBox() */


static PyObject*
GetStringWidth(self, args)
     PyObject self;
     PyObject *args;
{
    PyObject *result = NULL;
    PyObject *kerning = Py_False;
    int FontID, len;
    long spaceoff;
    char *string;

    if (PyArg_ParseTuple(args, "is#|iO:GetStringWidth", &FontID,
			 &string, &len, &spaceoff, &kerning)) {
	if (t1lib_init(-1)) {
	    int width = T1_GetStringWidth(FontID, string, len, spaceoff,
					  (PyObject_IsTrue(kerning)
					   ? T1_KERNING : 0));
	    if (width < 0)
		PyErr_SetString(T1Error, "font is not loaded");
	    else
		result = PyInt_FromLong(width);
	}
    }
    return (result);

}   /* GetStringWidth() */


static PyObject*
GetStringBBox(self, args)
     PyObject self;
     PyObject *args;
{
    PyObject *result = NULL;
    PyObject *kerning = Py_False;
    int FontID, len;
    long spaceoff;
    char *string;

    if (PyArg_ParseTuple(args, "is#|iO:GetStringBBox", &FontID,
			 &string, &len, &spaceoff, &kerning)) {
	if (t1lib_init(-1)) {
	    BBox bbox = T1_GetStringBBox(FontID, string, len, spaceoff,
					 (PyObject_IsTrue(kerning)
					  ? T1_KERNING : 0));

	    result = Py_BuildValue("iiii", bbox.llx, bbox.lly,
				   bbox.urx, bbox.ury);
	}
    }
    return (result);

}   /* GetStringBBox() */


static PyObject*
GetMetricsInfo(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    PyObject *charpos = NULL;
    PyObject *kerning = Py_False;
    int FontID, len;
    long spaceoff;
    char *string;

    if (PyArg_ParseTuple(args, "is#|iO:GetMetricsInfo", &FontID,
			 &string, &len, &spaceoff, &kerning)) {
	if (t1lib_init(-1)) {
	    METRICSINFO info;
	    int i;

	    info = T1_GetMetricsInfo(FontID, string, len, spaceoff,
				     (PyObject_IsTrue(kerning)
				      ? T1_KERNING : 0));
	    if ((charpos = PyTuple_New(info.numchars)) == NULL)
		goto finally;
	    for (i = 0; i < info.numchars; ++i) {
		PyObject *temp = PyInt_FromLong(info.charpos[i]);
		if (temp == NULL)
		    goto finally;
		PyTuple_SET_ITEM(charpos, i, temp);
	    }
	    result = Py_BuildValue("i(iiii)iO", info.width,
				   info.bbox.llx, info.bbox.lly,
				   info.bbox.urx, info.bbox.ury,
				   info.numchars, charpos);
	}
    }
 finally:
    Py_XDECREF(charpos);

    return (result);

}   /* GetMetricsInfo() */


static PyObject*
get_float_nz_attr(args, argfmt, errmsg, function)
     PyObject *args;
     char *argfmt;
     char *errmsg;
     float (*function)(int);
{
    PyObject *result = NULL;
    int FontID;

    if (PyArg_ParseTuple(args, argfmt, &FontID)) {
	if (t1lib_init(-1)) {
	    float value = function(FontID);

	    if (value == 0.0)
		PyErr_SetString(T1Error, errmsg);
	    else
		result = PyFloat_FromDouble(value);
	}
    }
    return (result);

}   /* get_float_nz_attr() */


static PyObject*
GetUnderlinePosition(self, args)
     PyObject *self;
     PyObject *args;
{
    return (get_float_nz_attr(args, "i:GetUnderlinePosition",
			      "no underline position or font is not loaded",
			      T1_GetUnderlinePosition));

}   /* GetUnderlinePosition() */


static PyObject*
GetUnderlineThickness(self, args)
     PyObject *self;
     PyObject *args;
{
    return (get_float_nz_attr(args, "i:GetUnderlineThickness",
			      "no underline thickness or font is not loaded",
			      T1_GetUnderlineThickness));

}   /* GetUnderlineThickness() */


static PyObject*
get_float_attr(args, argfmt, function)
     PyObject *args;
     char *argfmt;
     float (*function)(int);
{
    PyObject *result = NULL;
    int FontID;

    if (PyArg_ParseTuple(args, argfmt, &FontID)) {
	if (t1lib_init(-1))
	    result = PyFloat_FromDouble(function(FontID));
    }
    return (result);

}   /* get_float_attr() */


static PyObject*
GetItalicAngle(self, args)
     PyObject *self;
     PyObject *args;
{
    return (get_float_attr(args, "i:GetItalicAngle",
			   T1_GetItalicAngle));

}   /* GetUnderlineThickness() */


static PyObject*
GetFontBBox(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int FontID;

    if (PyArg_ParseTuple(args, "i:GetFontBBox", &FontID)) {
	if (t1lib_init(-1)) {
	    BBox bbox = T1_GetFontBBox(FontID);

	    result = Py_BuildValue("iiii", bbox.llx, bbox.lly,
				   bbox.urx, bbox.ury);
	}
    }
    return (result);

}   /* GetFontBBox() */


static PyObject*
get_string_attr(args, argsfmt, func)
     PyObject *args;
     char *argsfmt;
     char *(*func)(int);
{
    PyObject *result = NULL;
    int FontID;

    if (PyArg_ParseTuple(args, argsfmt, &FontID)) {
	if (t1lib_init(-1)) {
	    char *value = func(FontID);

	    if (value == NULL)
		PyErr_SetString(T1Error, "font is not loaded");
	    else
		result = PyString_FromString(value);
	}
    }
    return (result);

}   /* get_string_attr() */


static PyObject*
GetFontFileName(self, args)
     PyObject *self;
     PyObject *args;
{
    return (get_string_attr(args, "i:GetFontFileName", T1_GetFontFileName));

}   /* GetFontFileName() */


static PyObject*
GetFontName(self, args)
     PyObject *self;
     PyObject *args;
{
    return (get_string_attr(args, "i:GetFontName", T1_GetFontName));

}   /* GetFontName() */


static PyObject*
GetFullName(self, args)
     PyObject *self;
     PyObject *args;
{
    return (get_string_attr(args, "i:GetFullName", T1_GetFullName));

}   /* GetFullName() */


static PyObject*
GetFamilyName(self, args)
     PyObject *self;
     PyObject *args;
{
    return (get_string_attr(args, "i:GetFamilyName", T1_GetFamilyName));

}   /* GetFamilyName() */


static PyObject*
GetWeight(self, args)
     PyObject *self;
     PyObject *args;
{
    return (get_string_attr(args, "i:GetWeight", T1_GetWeight));

}   /* GetWeight() */


static PyObject*
GetVersion(self, args)
     PyObject *self;
     PyObject *args;
{
    return (get_string_attr(args, "i:GetVersion", T1_GetVersion));

} /* GetVersion() */


static PyObject*
GetNotice(self, args)
     PyObject *self;
     PyObject *args;
{
    return (get_string_attr(args, "i:GetNotice", T1_GetNotice));

} /* GetNotice() */


static PyObject*
GetIsFixedPitch(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int FontID;

    if (PyArg_ParseTuple(args, "i:GetIsFixedPitch", &FontID)) {
	if (t1lib_init(-1))
	    result = PyInt_FromLong(T1_GetIsFixedPitch(FontID));
    }
    return (result);

}   /* GetIsFixedPitch() */


static PyObject*
GetCharName(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int FontID;
    char char1;

    if (PyArg_ParseTuple(args, "ic:GetCharName", &FontID, &char1)) {
	if (t1lib_init(-1)) {
	    char *charname = T1_GetCharName(FontID, char1);

	    if (charname == NULL)
		PyErr_SetString(T1Error, "font is not loaded");
	    else
		result = PyString_FromString((char*)charname);
	}
    }
    return (result);

}   /* GetCharName() */


static PyObject*
GetEncodingIndex(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int FontID;
    char *charname;

    if (PyArg_ParseTuple(args, "is:GetEncodingIndex", &FontID, &charname)) {
	if (t1lib_init(-1)) {
	    int index = T1_GetEncodingIndex(FontID, charname);

	    if (index < 0)
		PyErr_SetString(T1Error,
		     "character not in encoding vector or font is not loaded");
	    else
		result = PyInt_FromLong(index);
	}
    }
    return (result);

}   /* GetEncodingIndex() */


static PyObject*
GetAllCharNames(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    PyObject *list = NULL;
    int FontID;

    if (PyArg_ParseTuple(args, "i:GetAllCharNames", &FontID)) {
	if (t1lib_init(-1)) {
	    char **charnames = T1_GetAllCharNames(FontID);
	    int num = 0;

	    if (charnames == NULL)
		PyErr_SetString(T1Error, "font is not loaded");
	    else {
		int i = 0;
		PyObject *s;
		PyObject *list = NULL;

		while (charnames[num] != NULL)
		    ++num;
		list = PyList_New(num);
		while (i < num) {
		    if ((s = PyString_FromString(charnames[i])) == NULL)
			goto finally;
		    PyList_SetItem(list, i++, s);
		}
		result = list;
		list = NULL;
	    }
	}
    }
 finally:
    Py_XDECREF(list);

    return (result);

}   /* GetAllCharNames() */


static PyObject*
QueryLigs(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    PyObject *tuple = NULL;
    int FontID;
    char char1;

    if (PyArg_ParseTuple(args, "ic:QueryLigs", &FontID, &char1)) {
	if (t1lib_init(-1)) {
	    char *successors = NULL;
	    char *ligatures = NULL;
	    int i, n = T1_QueryLigs(FontID, char1, &successors, &ligatures);

	    if (n < 0) {
		PyErr_SetString(T1Error,
				"no metrics available or font is not loaded");
		goto finally;
	    }
	    if ((tuple = PyTuple_New(n)) == NULL)
		goto finally;
	    for (i = 0; i < n; ++i) {
		PyObject *temp = Py_BuildValue("cc", successors[i],
					       ligatures[i]);
		if (temp == NULL)
		    goto finally;
		PyTuple_SET_ITEM(tuple, i, temp);
	    }
	    result = tuple;
	    tuple = NULL;
	}
    }
 finally:
    Py_XDECREF(tuple);

    return (result);

}   /* QueryLigs() */


static PyObject*
transform_font(args, argfmt, transform)
     PyObject *args;
     char *argfmt;			/* so we get the function name	*/
     int (*transform)(int, double);
{
    PyObject *result = NULL;
    int FontID;
    double extend;

    if (PyArg_ParseTuple(args, argfmt, &FontID, &extend)) {
	if (!t1lib_init(-1))
	    return (NULL);
	if (T1_DeleteAllSizes(FontID) < 0)
	    PyErr_SetString(T1Error, "error deleting size-specific data");
	else if (transform(FontID, extend))
	    PyErr_SetString(T1Error, "font is not loaded");
	else {
	    result = Py_None;
	    Py_INCREF(result);
	}
    }
    return (result);

}   /* transform_font() */


static PyObject*
ExtendFont(self, args)
     PyObject *self;
     PyObject *args;
{
    return (transform_font(args, "id:ExtendFont", T1_ExtendFont));

}   /* ExtendFont() */


static PyObject*
SlantFont(self, args)
     PyObject *self;
     PyObject *args;
{
    return (transform_font(args, "id:SlantFont", T1_SlantFont));

}   /* SlantFont() */


static PyObject*
CopyFont(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int FontID;

    if (PyArg_ParseTuple(args, "i:CopyFont", &FontID)) {
	int newFontID;

	if (!t1lib_init(-1))
	    return (NULL);
	newFontID = T1_CopyFont(FontID);
	if (newFontID == -1)
	    PyErr_SetString(T1Error, "original font is not loaded");
	else if (newFontID == -2)
	    PyErr_SetString(T1Error, "original font is logical - cannot copy");
	else if (newFontID == -3)
	    PyErr_NoMemory();
	else
	    result = PyInt_FromLong(newFontID);
    }
    return (result);

}   /* CopyFont() */


static PyObject*
AASetBitsPerPixel(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int bpp;

    if (PyArg_ParseTuple(args, "i:AASetBitsPerPixel", &bpp)) {
	if (!t1lib_init(-1))
	    return (NULL);
	if (T1_AASetBitsPerPixel(bpp))
	    PyErr_SetString(PyExc_ValueError, "invalid bit depth");
	else {
	    result = Py_None;
	    Py_INCREF(result);
	}
    }
    return (result);

}   /* AASetBitsPerPixel() */


static int
conv_UnsignedLong(object, target)
     PyObject *object;
     long *target;
{
    if (PyInt_Check(object)) {
	*target = (unsigned long)PyInt_AsLong(object);
	return (1);
    }
    if (PyLong_Check(object)) {
	*target = PyLong_AsUnsignedLong(object);
	return (1);
    }
    PyErr_SetString(PyExc_TypeError,
		    "must be able to convert to unsigned long");
    return (0);

}   /* conv_UnsignedLong() */


static PyObject*
AASetGrayValues(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    unsigned long white, gray75, gray50, gray25, black;

    if (PyArg_ParseTuple(args, "O&O&O&O&O&:AASetGrayValues",
			 conv_UnsignedLong, &white,
			 conv_UnsignedLong, &gray75,
			 conv_UnsignedLong, &gray50,
			 conv_UnsignedLong, &gray25,
			 conv_UnsignedLong, &black)) {
	if (!t1lib_init(-1))
	    return (NULL);
	T1_AASetGrayValues(white, gray75, gray50, gray25, black);
	result = Py_None;
	Py_INCREF(result);
    }
    return (result);

}   /* AASetGrayValues() */


static PyObject*
AddFont(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    char *fontfilename;

    if (PyArg_ParseTuple(args, "s:AddFont", &fontfilename)) {
	if (t1lib_init(-1)) {
	    int rc = T1_AddFont(fontfilename);

	    if (rc == 0)
		Py_INCREF(result = Py_None);
	    else if (rc == -1)
		PyErr_SetObject(PyExc_IOError,
				Py_BuildValue("is", ENOENT, strerror(ENOENT)));
	    else if (rc == -2)
		PyErr_NoMemory();
	    else if (rc == -3)
		PyErr_NoMemory();
	    else {
		char buffer[80];

		sprintf(buffer,
			"Unknown error: T1_AddFont() returned %d", rc);
		PyErr_SetString(T1Error, buffer);
	    }
	}
    }
    return (result);

}   /* AddFont() */


static PyObject*
WriteAFMFallbackFile(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int FontID;

    if (PyArg_ParseTuple(args, "i:WriteAFMFallbackFile", &FontID)) {
	if (t1lib_init(-1)) {
	    int rc = T1_WriteAFMFallbackFile(FontID);

	    if (rc == 0)
		Py_INCREF(result = Py_None);
	    else if (rc == -1)
		PyErr_SetString(T1Error, "font already has an AFM file");
	    else if (rc == -2)
		PyErr_SetString(T1Error, "font is not loaded");
	    else if (rc == -3)
		PyErr_SetString(T1Error, "could not completely raster font");
	    else if (rc == -4)
		PyErr_SetFromErrno(PyExc_IOError);
	    else if (rc == -5)
		PyErr_SetFromErrno(PyExc_IOError);
	    else if (rc == -6)
		PyErr_NoMemory();
	    else {
		char buffer[80];

		sprintf(buffer,
			"Unknown error: T1_WriteAFMFallbackFile() returned %d",
			rc);
		PyErr_SetString(T1Error, buffer);
	    }
	}
    }
    return (result);

}   /* WriteAFMFallbackFile() */


static PyObject*
SetFontDataBase(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    char *filename;

    if (PyArg_ParseTuple(args, "s:SetFontDataBase", &filename)) {
	if (t1lib_init(-1)) {
	    if (T1_SetFontDataBase(filename))
		PyErr_NoMemory();
	    else
		Py_INCREF(result = Py_None);
	}
    }
    return (result);

}   /* SetFontDataBase() */


static PyObject*
SetFileSearchPath(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int type;
    char *pathname;

    if (PyArg_ParseTuple(args, "is:SetFileSearchPath", &type, &pathname)) {
	if (t1lib_init(-1)) {
	    if (T1_SetFileSearchPath(type, pathname))
		if (T1_Get_no_fonts() > 0)
		    /* just no allowed */
		    PyErr_SetString(T1Error, "cannot set search path"
				    " after font database is loaded");
		else
		    /* only possible error if no fonts are loaded */
		    PyErr_NoMemory();
	    else
		Py_INCREF(result = Py_None);
	}
    }
    return (result);

}   /* SetFileSearchPath() */


static int
num_bits(num)
     int num;
{
    int i = 0;
    int j = 0;

    for (; j < (sizeof(num) * 8); ++j) {
	i += (num & 1);
	num >>= 1;
    }
    return (i);

}   /* num_bits() */


static PyObject*
GetFileSearchPath(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int type;

    if (PyArg_ParseTuple(args, "i:GetFileSearchPath", &type)) {
	if (t1lib_init(-1)) {
	    if (num_bits(type) != 1)
		PyErr_SetString(PyExc_ValueError,
				"only one bit can be set in path type");
	    else {
		char *value = T1_GetFileSearchPath(type);

		if (value == NULL)
		    PyErr_SetString(PyExc_ValueError, "invalid path type");
		else {
		    result = PyString_FromString(value);
		    free(value);
		}
	    }
	}
    }
    return (result);

}   /* GetFileSearchPath() */


static PyObject*
AddToFileSearchPath(self, args)
     PyObject *self;
     PyObject *args;
{
    PyObject *result = NULL;
    int pathtype;
    int mode;
    char *pathname;

    if (PyArg_ParseTuple(args, "iis:AddToFileSearchPath",
			 &pathtype, &mode, &pathname)) {
	if (t1lib_init(-1)) {
	    if ((mode != T1_APPEND_PATH) && (mode != T1_PREPEND_PATH))
		PyErr_SetString(PyExc_ValueError, "illegal change mode");
	    else if (T1_AddToFileSearchPath(pathtype, mode, pathname))
		PyErr_NoMemory();
	    else
		Py_INCREF(result = Py_None);
	}
    }
    return (result);

}   /* AddToFileSearchPath() */


static PyMethodDef functions[] = {
    {"InitLib",			InitLib,		METH_VARARGS},
    {"GetBitmapPad",		GetBitmapPad,		METH_VARARGS},
    {"SetBitmapPad",		SetBitmapPad,		METH_VARARGS},
    {"SetLogLevel",		SetLogLevel,		METH_VARARGS},
    {"PrintLog",		PrintLog,		METH_VARARGS},
    {"LoadFont",		LoadFont,		METH_VARARGS},
    {"SetChar",			SetChar,		METH_VARARGS},
    {"AASetChar",		AASetChar,		METH_VARARGS},
    {"SetString",		SetString,		METH_VARARGS},
    {"AASetString",		AASetString,		METH_VARARGS},
    {"SetDeviceResolutions",	SetDeviceResolutions,	METH_VARARGS},
    {"LoadEncoding",		LoadEncoding,		METH_VARARGS},
    {"ReencodeFont",		ReencodeFont,		METH_VARARGS},
    {"DeleteSize",		DeleteSize,		METH_VARARGS},
    {"DeleteAllSizes",		DeleteAllSizes,		METH_VARARGS},
    {"DeleteFont",		DeleteFont,		METH_VARARGS},
    {"Get_no_fonts",		Get_no_fonts,		METH_VARARGS},
    {"GetKerning",		GetKerning,		METH_VARARGS},
    {"GetCharWidth",		GetCharWidth,		METH_VARARGS},
    {"GetCharBBox",		GetCharBBox,		METH_VARARGS},
    {"GetStringWidth",		GetStringWidth,		METH_VARARGS},
    {"GetStringBBox",		GetStringBBox,		METH_VARARGS},
    {"GetMetricsInfo",		GetMetricsInfo,		METH_VARARGS},
    {"GetIsFixedPitch",		GetIsFixedPitch,	METH_VARARGS},
    {"GetItalicAngle",		GetItalicAngle,		METH_VARARGS},
    {"GetUnderlinePosition",	GetUnderlinePosition,	METH_VARARGS},
    {"GetUnderlineThickness",	GetUnderlineThickness,	METH_VARARGS},
    {"GetFontBBox",		GetFontBBox,		METH_VARARGS},
    {"GetFontFileName",		GetFontFileName,	METH_VARARGS},
    {"GetFontName",		GetFontName,		METH_VARARGS},
    {"GetFullName",		GetFullName,		METH_VARARGS},
    {"GetFamilyName",		GetFamilyName,		METH_VARARGS},
    {"GetWeight",		GetWeight,		METH_VARARGS},
    {"GetVersion",		GetVersion,		METH_VARARGS},
    {"GetNotice",		GetNotice,		METH_VARARGS},
    {"GetCharName",		GetCharName,		METH_VARARGS},
    {"GetAllCharNames",		GetAllCharNames,	METH_VARARGS},
    {"GetEncodingIndex",	GetEncodingIndex,	METH_VARARGS},
    {"QueryLigs",		QueryLigs,		METH_VARARGS},
    {"ExtendFont",		ExtendFont,		METH_VARARGS},
    {"SlantFont",		SlantFont,		METH_VARARGS},
    {"CopyFont",		CopyFont,		METH_VARARGS},
    {"AASetBitsPerPixel",	AASetBitsPerPixel,	METH_VARARGS},
    {"AASetGrayValues",		AASetGrayValues,	METH_VARARGS},
    {"AddFont",			AddFont,		METH_VARARGS},
    {"WriteAFMFallbackFile",	WriteAFMFallbackFile,	METH_VARARGS},
    {"SetFontDataBase",		SetFontDataBase,	METH_VARARGS},
    {"SetFileSearchPath",	SetFileSearchPath,	METH_VARARGS},
    {"GetFileSearchPath",	GetFileSearchPath,	METH_VARARGS},
    {"AddToFileSearchPath",	AddToFileSearchPath,	METH_VARARGS},
    {NULL,			NULL}
};  /* functions */


void
init_t1lib() {
    PyObject *module = Py_InitModule("_t1lib", functions);

    if (module != NULL) {
	PyObject *dict = PyModule_GetDict(module);

	/* Lie about the module name of this exception, since the
	 * public interface will be the Python t1lib module.
	 */
	T1Error = PyErr_NewException("t1lib.T1Error", NULL, NULL);
	PyDict_SetItemString(dict, "T1Error", T1Error);
	PyDict_SetItemString(dict, "LOG_ERROR",
			     PyInt_FromLong(T1LOG_ERROR));
	PyDict_SetItemString(dict, "LOG_WARNING",
			     PyInt_FromLong(T1LOG_WARNING));
	PyDict_SetItemString(dict, "LOG_STATISTIC",
			     PyInt_FromLong(T1LOG_STATISTIC));
	PyDict_SetItemString(dict, "LOG_DEBUG",
			     PyInt_FromLong(T1LOG_DEBUG));
	PyDict_SetItemString(dict, "IGNORE_CONFIGFILE",
			     PyInt_FromLong(IGNORE_CONFIGFILE));
	PyDict_SetItemString(dict, "IGNORE_FONTDATABASE",
			     PyInt_FromLong(IGNORE_FONTDATABASE));
	PyDict_SetItemString(dict, "APPEND_PATH",
			     PyInt_FromLong(T1_APPEND_PATH));
	PyDict_SetItemString(dict, "PREPEND_PATH",
			     PyInt_FromLong(T1_PREPEND_PATH));
	PyDict_SetItemString(dict, "PFAB_PATH",
			     PyInt_FromLong(T1_PFAB_PATH));
	PyDict_SetItemString(dict, "AFM_PATH",
			     PyInt_FromLong(T1_AFM_PATH));
	PyDict_SetItemString(dict, "ENC_PATH",
			     PyInt_FromLong(T1_ENC_PATH));
	PyDict_SetItemString(dict, "__version__",
			     PyString_FromString("0.3"));
    }
}   /* init_t1lib() */
