/* Zgv v2.8 - GIF, JPEG and PBM/PGM/PPM viewer, for VGA PCs running Linux.
 * Copyright (C) 1993-1997 Russell Marks. See README for license details.
 *
 * rcfile.c - config file handling.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>	/* getopt() */
#include <vga.h>
#include "rc_config.h"
#include "zgv.h"	/* for ZGV_VER */


#ifndef GLASTMODE
/* not sure if this is possible, but just in case...
 * (this is fairly conservative, of course)
 */
#define GLASTMODE	G1280x1024x16M
#endif


/* prototypes */
void getconfig(void);
void parseconfig(FILE *in);
int getmodenumber(char *tptr, int *mp);
void getrgbval(int line,void *arg2,void *arg3);
void getbool(int line,void *arg2,void *arg3);
void getint(int line,void *arg2,void *arg3);
void getdouble(int line,void *arg2,void *arg3);
mouse_type strmousetype(char *arg);
void getmousearg(int line,void *arg2,void *arg3);
void getjis(int line,void *arg2,void *arg3);
void do_allmodesbad(int line,void *arg2,void *arg3);
void do_allmodesgood(int line,void *arg2,void *arg3);
void do_badmode(int line,void *arg1,void *arg3);
void do_goodmode(int line,void *arg1,void *arg3);
void do_startmode(int line,void *arg1,void *arg3);
void defaultcfg(void);
void findtoken(char **ptr);
int tokenlength(char *ptr);
int tokencompare(char *tptr, char *txt);
int modematch(int x, int y, int bpp);
int parsecommandline(int argc, char *argv[]);
void usage_help(void);

 

struct zgv_config cfg;

struct cfglookup_tag
  {
  char optname[32];
  void (*routine)(int,void *,void *);
  void *arg3;	/* arg1 is line no., arg2 is the after-token-pointer */
  };

struct cfglookup_tag cfglookup[]=
  {
  {"allmodesbad",	do_allmodesbad,	NULL},
  {"allmodesgood",	do_allmodesgood,NULL},
  {"badmode",		do_badmode,	NULL},
  {"betterpgm",		getbool,	&cfg.betterpgm},
  {"black",		getrgbval,	&cfg.black},
  {"blockcursor",	getbool,	&cfg.blockcursor},
  {"brightness",	getint,		&cfg.brightness},
  {"center",		getbool,	&cfg.centreflag},
  {"centre",		getbool,	&cfg.centreflag},
  {"cleartext",		getbool,	&cfg.cleartext},
  {"contrast",		getdouble,	&cfg.contrast},
  {"dark",		getrgbval,	&cfg.dark},
  {"force16fs",		getbool,	&cfg.force16fs},
  {"forgetoldpos",	getbool,	&cfg.forgetoldpos},
  {"fs16col",		getbool,	&cfg.fs16col},
  {"fullsel",		getbool,	&cfg.fullsel},
  {"gamma",		getbool,	&cfg.gamma},
  {"goodmode",		do_goodmode,	NULL},
  {"hicolmodes",	getbool,	&cfg.hicolmodes},
  {"jpeg24bit",		getbool,	&cfg.jpeg24bit},
  {"jpegindexstyle",	getjis,		&cfg.jpegindexstyle},
  {"jpegspeed",		getint,		&cfg.jpegspeed},
  {"light",		getrgbval,	&cfg.light},
  {"linetext",		getbool,	&cfg.linetext},
  {"medium",		getrgbval,	&cfg.medium},
  {"mouse",		getmousearg,	&cfg.mtype},
  {"nodelprompt",	getbool,	&cfg.nodelprompt},
  {"onefile_progress",	getbool,	&cfg.onefile_progress},
  {"perfectindex",	getbool,	&cfg.perfectindex},
  {"revert",		getbool,	&cfg.revert},
  {"showxvpicdir",	getbool,	&cfg.showxvpicdir},
  {"smallfiletext",	getbool,	&cfg.smallfiletext},
  {"startmode",		do_startmode,	NULL},
  {"tagged",		getrgbval,	&cfg.marked},
  {"tagtimeout",	getint,		&cfg.tag_timer},
  {"thicktext",		getbool,	&cfg.thicktext},
  {"viewer16col",	getbool,	&cfg.viewer16col},
  {"visual",		getbool,	&cfg.xvpic_index},
  {"vkludge",		getbool,	&cfg.vkludge},
  {"zoom",		getbool,	&cfg.zoom},
  {"",NULL,NULL}
  };



void getconfig()
{
FILE *in;
char cfgfile[1024];
int gotrcfile;

defaultcfg();

gotrcfile=0;
strcpy(cfgfile,getenv("HOME"));
strcat(cfgfile,"/.zgvrc");

if((in=fopen(cfgfile,"r"))!=NULL)
  gotrcfile=1;
else if((in=fopen("/etc/zgv.conf","r"))!=NULL)
  gotrcfile=1;
else if((in=fopen("/etc/system.zgvrc","r"))!=NULL)
  gotrcfile=1;

if(gotrcfile)
  {
  parseconfig(in);
  fclose(in);
  }

if(cfg.jpegspeed<1) cfg.jpegspeed=1;
if(cfg.jpegspeed>3) cfg.jpegspeed=3;
}


void parseconfig(in)
FILE *in;
{
static char inpline[1024];
char *ptr;
int ln,f,c,inpc,found=0;

ln=0;
while(fgets(inpline,sizeof(inpline),in)!=NULL) 
  {
  ln++;
  if(inpline[strlen(inpline)-1]=='\n') inpline[strlen(inpline)-1]=0;
  if((ptr=strchr(inpline,'#'))!=NULL)
    *ptr=0;
  if(inpline[0])
    {
    found=0;
    ptr=inpline;
    findtoken(&ptr);
    inpc=*ptr;
    for(f=0;(c=cfglookup[f].optname[0]);f++)
      if(inpc==c && tokencompare(ptr,cfglookup[f].optname))
        {
        (*cfglookup[f].routine)(ln,ptr,cfglookup[f].arg3);
        found=1;
        break;
        }
    if(!found)
      {
      fprintf(stderr,"Error in line #%d of rc file.\n",ln);
      exit(1);
      }
    }
  }
}


/* get mode number from something like '640 480 8' -
 * returns 0 if it can't find one that matches, else 1.
 * (the mode number is put into *mp)
 */
int getmodenumber(char *tptr,int *mp)
{
int x,y,bpp,rtn;

/* skip past the current token */
tptr+=tokenlength(tptr);
findtoken(&tptr);
x=atoi(tptr);
tptr+=tokenlength(tptr);

findtoken(&tptr);
y=atoi(tptr);
tptr+=tokenlength(tptr);

findtoken(&tptr);
bpp=atoi(tptr);

rtn=modematch(x,y,bpp);
if(rtn>=0)
  {
  *mp=rtn;
  return(0);
  }
else
  return(-1);
}


void getrgbval(int line,void *arg2,void *arg3)
{
char *tptr=(char *)arg2;
struct cfgrgb_tag *p=(struct cfgrgb_tag *)arg3;

/* skip past the current token */
tptr+=tokenlength(tptr);
findtoken(&tptr);
p->r=atoi(tptr);
tptr+=tokenlength(tptr);

findtoken(&tptr);
p->g=atoi(tptr);
tptr+=tokenlength(tptr);

findtoken(&tptr);
p->b=atoi(tptr);
}


void getbool(int line,void *arg2,void *arg3)
{
char *tptr=(char *)arg2;
int *bp=(int *)arg3;

/* skip past the current token */
tptr+=tokenlength(tptr);
findtoken(&tptr);
if(tokencompare(tptr,"on") || tptr[0]=='y')
  *bp=1;
else
  *bp=0;
}


void getint(int line,void *arg2,void *arg3)
{
char *tptr=(char *)arg2;
int *ip=(int *)arg3;

/* skip past the current token */
tptr+=tokenlength(tptr);
findtoken(&tptr);
*ip=atoi(tptr);
}


void getdouble(int line,void *arg2,void *arg3)
{
char *tptr=(char *)arg2;
double *dp=(double *)arg3;

/* skip past the current token */
tptr+=tokenlength(tptr);
findtoken(&tptr);
*dp=atof(tptr);
}


/* convert -M or 'mouse' arg to cfg.mtype value */
mouse_type strmousetype(char *arg)
{
if(strcmp(arg,"ms")  ==0) return(P_MS);
if(strcmp(arg,"sun") ==0) return(P_SUN);
if(strcmp(arg,"msc") ==0) return(P_MSC);
if(strcmp(arg,"mm")  ==0) return(P_MM);
if(strcmp(arg,"logi")==0) return(P_LOGI);
if(strcmp(arg,"bm")  ==0) return(P_BM);
if(strcmp(arg,"ps2") ==0) return(P_PS2);

return(NO_MOUSE);
}


void getmousearg(int line,void *arg2,void *arg3)
{
char *tptr=(char *)arg2;
mouse_type *mp=(mouse_type *)arg3;

/* skip past the current token */
tptr+=tokenlength(tptr);
findtoken(&tptr);
*mp=strmousetype(tptr);
}


void getjis(int line,void *arg2,void *arg3)
{
int *jisp=(int *)arg3;
getint(line,arg2,arg3);
if(*jisp<1) *jisp=1;
if(*jisp>3) *jisp=3;
}


void do_allmodesbad(int line,void *arg2,void *arg3)
{
int f;

for(f=0;f<256;f++) cfg.mode_allowed[f]=0;
}


void do_allmodesgood(int line,void *arg2,void *arg3)
{
int f;

/* we still disallow ones it hasn't got */
for(f=0;f<=GLASTMODE;f++)
  cfg.mode_allowed[f]=vga_hasmode(f)?1:0;
}


void do_badmode(int line,void *arg1,void *arg3)
{
char *ptr=(char *)arg1;
int f;

if(getmodenumber(ptr,&f)==-1)
  {
  fprintf(stderr,"Mode not found on line #%d of rc file.\n",line);
  exit(1);
  }
cfg.mode_allowed[f]=0;
}


void do_goodmode(int line,void *arg1,void *arg3)
{
char *ptr=(char *)arg1;
int f;

if(getmodenumber(ptr,&f)==-1)
  {
  fprintf(stderr,"Mode not found on line #%d of rc file.\n",line);
  exit(1);
  }
cfg.mode_allowed[f]=1;
}


void do_startmode(int line,void *arg1,void *arg3)
{
char *ptr=(char *)arg1;
int f;

if(getmodenumber(ptr,&f)==-1)
  {
  fprintf(stderr,"Mode not found on line #%d of rc file.\n",line);
  exit(1);
  }
cfg.videomode=f;
}


void defaultcfg()
{
int f;

/* it'll use 360x480x256 if you haven't got 640x480x256,
 * and 320x200x256 if you've locked that out.
 */
cfg.videomode=G640x480x256;
cfg.zoom=0;
cfg.fullsel=0;
cfg.vkludge=0;
cfg.brightness=0;
cfg.jpeg24bit=1;	/* only if possible, of course */
cfg.jpegspeed=2;	/* slow int by default */
cfg.jpegindexstyle=2;	/* jpeg thumbnail speed/quality */
cfg.betterpgm=0;
cfg.centreflag=1;
cfg.blockcursor=0;
cfg.thicktext=0;
cfg.hicolmodes=0;	/* don't force high-colour mode test to true */
cfg.nodelprompt=0;	/* default to prompting on delete */
cfg.perfectindex=0;	/* make selector cols look nice, not index cols */
cfg.xvpic_index=1;	/* visual index */
cfg.onefile_progress=1;	/* progress report while loading file given as arg */
cfg.cleartext=0;	/* clear text screen on startup/exit? */
cfg.repeat_timer=0;	/* don't reread after a timeout */
cfg.tag_timer=4;	/* 4 seconds per tagged file */
cfg.force16fs=0;        /* don't force 16-colour mode for file selector */
cfg.revert=1;		/* auto-reset scaling to off between pics */
cfg.gamma=1;		/* try to fake more greys/colours in 8-bit modes */
cfg.selecting=0;	/* no selection normally */
cfg.fs16col=0;		/* normally use mono in 16-col file selector */
cfg.viewer16col=0;	/* normally use mono in 16-col viewer mode */
cfg.echotagged=0;	/* echo tagged files on exit */
cfg.forgetoldpos=0;	/* don't return to old position in revisited dir */
cfg.linetext=0;		/* use old line-based text instead of bitmap fonts */
/* but bitmap fonts are far too slow in 16-colour mode, so stick
 * with the old line-based font for that by default:
 */
if(!vga_hasmode(G640x480x256)) cfg.linetext=1;
cfg.smallfiletext=0;	/* use smaller font for filenames in file selector */
cfg.contrast=(double)1;
cfg.black.r =cfg.black.g =cfg.black.b = 0;
cfg.dark.r  =cfg.dark.g  =cfg.dark.b  =20;
cfg.medium.r=cfg.medium.g=cfg.medium.b=30;
cfg.light.r =cfg.light.g =cfg.light.b =40;
cfg.marked.r=cfg.marked.g=cfg.marked.b= 0; cfg.marked.r+=30;
cfg.loop=0;
cfg.mtype=MTYPE;
for(f=0;f<256;f++) cfg.mode_allowed[f]=0;
cfg.showxvpicdir=0;
for(f=0;f<=GLASTMODE;f++)
  cfg.mode_allowed[f]=vga_hasmode(f)?1:0;
cfg.cmd[0]=0;
cfg.errignore=0;
}


void findtoken(char **ptr)
{
while((*(*ptr)!=0)&&(strchr(" \t",*(*ptr))!=NULL))
  (*ptr)++;
}


int tokenlength(char *ptr)
{
int siz;

siz=0;
while((*ptr!=0)&&(strchr(" \t",*ptr)==NULL))
  {
  ptr++;
  siz++;
  }

return(siz);
}


/* returns 1 if equal, 0 otherwise */
int tokencompare(char *tptr,char *txt)
{
int tlen;

tlen=tokenlength(tptr);
if(tlen!=strlen(txt))
  return(0);
else
  if(strncmp(tptr,txt,tlen))
    return(0);
  else
    return(1);
}


/* returns mode number which matches x,y,bpp or -1 if none did.
 * put '-1' in x,y or bpp to wildcard them.
 *
 * maybe this routine should be somewhere else?
 */
int modematch(int x,int y,int bpp)
{
int numcols,f;
vga_modeinfo *vminfo;

if((bpp>24)||(bpp==2))
  /* they must have used numcols, not bpp. hmm, let 'em get away with it. */
  numcols=bpp;
else
  numcols=(1<<bpp);

/* we check 0 - 255 at the most */
for(f=0;f<256;f++)
  {
  vminfo=vga_getmodeinfo(f);
  if(vminfo!=NULL)
    if(((x==-1)||(x==vminfo->width))&&
       ((y==-1)||(y==vminfo->height))&&
       ((numcols==-1)||(numcols==vminfo->colors)))
      break;
  }

if(f<255)
  return(f);
else
  return(-1);
}


/* all booleans are actually toggles. parsing happens after rc file reading.
 */
int parsecommandline(int argc,char *argv[])
{
int done;
char buf[1024];

done=0;
opterr=0;

do
  switch(getopt(argc,argv,"a:bcghijJ:klM:m:r:sptTz"))
    {
    case 'a':	/* alternative command */
      strcpy(cfg.cmd,optarg);
      if(strstr(cfg.cmd,"%s")==NULL)
        {
        fprintf(stderr,"Need '%%s' filename placeholder in command string.\n");
        exit(1);
        }
      break;
    case 'b':	/* blockcursor */
      cfg.blockcursor=!cfg.blockcursor; break;
    case 'c':	/* centre */
      cfg.centreflag=!cfg.centreflag; break;
    case 'g':	/* betterpgm */
      cfg.betterpgm=!cfg.betterpgm; break;
    case 'h':	/* help on usage */
      usage_help();
      exit(1);
    case 'i':	/* errignore */
      cfg.errignore=!cfg.errignore; break;
    case 'j':	/* jpeg24bit */
      cfg.jpeg24bit=!cfg.jpeg24bit; break;
    case 'k':	/* vkludge */
      cfg.vkludge=!cfg.vkludge; break;
    case 'l':	/* loop in slideshow */
      cfg.loop=1; break;
    case 'M':    /* mouse type */
      if((cfg.mtype=strmousetype(optarg))==NO_MOUSE)
        {
        fprintf(stderr, "Invalid mouse type for -M.\n");
        usage_help();
        exit(1);
        }
      break;
    case 'm':	/* startup mode (takes arg) */
      {
      int vidnum;
      
      strcpy(buf,"dummytoken ");
      strcat(buf,optarg);
      if(getmodenumber(buf,&vidnum)==-1)
        {
        fprintf(stderr,"Mode '%s' not found.\n",optarg);
        exit(1);
        }
      cfg.videomode=vidnum;
      }
      break;
    case 'r':	/* repeat timer (takes arg) */
      cfg.repeat_timer=atoi(optarg); break;
    case 'J':	/* jpeg speed (takes arg) */
      cfg.jpegspeed=atoi(optarg);
      break;
    case 's':	/* selection - useful with pnmcut */
      cfg.selecting=1;
      break;
    case 'p':   /* onefile_progress */
      cfg.onefile_progress=!cfg.onefile_progress; break;
    case 't':	/* thicktext */
      cfg.thicktext=!cfg.thicktext; break;
    case 'T':	/* echo tagged files on exit */
      cfg.echotagged=1; break;
    case 'z':	/* zoom */
      cfg.zoom=!cfg.zoom; break;
    case '?':
      switch(optopt)
        {
        case 'm':
          fprintf(stderr,"The -m (startup mode) option takes the mode as an ");
          fprintf(stderr,"argument,\n  e.g.  zgv -m \"640 480 8\"\n");
          break;
        case 'M':
          fprintf(stderr,"You must specify a mouse type with -M. Try ");
          fprintf(stderr,"'zgv -h' for details.\n");
          break;
        case 'a':
          fprintf(stderr,"The -a (alternative view) option takes the command");
          fprintf(stderr," to run as an argument,\n");
          fprintf(stderr,"  e.g.  zgv -a \"djpeg -grey %%s | pgmtopbm |");
          fprintf(stderr," pbmtoascii -2x4 > /dev/tty8\"\n");
          break;
        case 'r':
  	  fprintf(stderr,"The -r (repeat timer) option takes an alarm() ");
	  fprintf(stderr,"arg, i.e. the timeout\nin seconds, or -1 for");
          fprintf(stderr,"'continuous' update.\n");
          break;
        case 'J':
  	  fprintf(stderr,"The -J (jpeg speed) options needs a speed value. ");
	  fprintf(stderr,"Try 'zgv -h' for details.\n");
          break;
        default:
          fprintf(stderr,"Option '%c' not recognised.\n",optopt);
        }
      exit(1);
    case -1:
      done=1;
    }
while(!done);

return(argc-optind);
}


void usage_help()
{
printf("Zgv v%s - (c) 1993-1998 Russell Marks for improbabledesigns.",ZGV_VER);

printf(
"
usage: zgv [-bcghijklpstTz] [-a \"command\"] [-J jpg_decmp_type] [-M mousetype]
	[-m modespec] [-r seconds] [dirname | filename1 [... filenameN]]

	-a	alternative command (see man page for details)
	-b	blockcursor toggle, gives outline cursor not tacky 3d effect
	-c	centre toggle, whether to centre pictures on the screen
	        (defaults to on, so using `-c' will turn it off)
	-g	betterpgm toggle, whether to use 8-bit or high-colour modes
	        for PGM files (defaults to on, so using `-g' will turn it off)
	-h	give this usage help
        -i	ignore errors with GIF and PNG files and display whatever
        	part of the picture could be read anyway, if possible
	-j	jpeg24bit toggle (see documentation - defaults to on, so using
        	`-j' will turn it off)
        -J	sets JPEG speed/quality tradeoff:
        		1 = floating-point - slow but accurate
                        2 = slow integer - faster but not as accurate
                        3 = fast integer - fastest but less accurate still
                              (default is 2)
	-k	vkludge toggle, smooths a bit in 320x400 and 360x480 modes,
	        and also when 'zooming' a big picture down to screen size
	-l	loop forever in slideshow mode
	-M	sets mouse type;
		Microsoft = `ms', Mouse Systems Corp = `msc',
		MM Series = `mm', Logitech = `logi', BusMouse = `bm',
		MSC 3-bytes = `sun', PS/2 = `ps2'
	-m	startup mode; the `modespec' should be in quotes but
	        otherwise in the same format as the .zgvrc entry `startmode'
	        e.g. \"640 480 8\"
        -p      turn off (or on) progress report when loading a single file
        -r	re-read and redisplay picture every `seconds' seconds
        -s	print dimensions of section of picture being viewed to stdout
        	on exit
	-t	thicktext toggle, makes the text a little bit clearer, kind of
	-T	echo tagged files on exit
	-z	zoom toggle, i.e. whether to scale pictures to the screen size

      dirname			makes zgv start in a certain directory.
     filename   		makes zgv view that one file only.
filename1 ... filenameN		view the N files as a slideshow.
(i.e. more than one file)

All options are processed after any .zgvrc or /etc/zgv.conf file.
");	/* end of printf() */
}
