/*****************************************************************************
 *                                                                           *
 * Programm:  paul                                                           *
 *            (P)rogramm zur (A)uswertung und (U)mformung von                *
 *            (L)aserbildern                                                 *
 * Verwendet: GTK, Imlib                                                     *
 * Modul:     file.c                                                         *
 *            GTK load/save files for paul                                   *
 * Autor:     Andreas Tille                                                  *
 * Datum:     14.06.1998                                                     *
 *                                                                           *
 *****************************************************************************/

#include <assert.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/param.h>

#include "paul.h"
#include "callback.h"

static char *ChdirToLoadDir(char *filepath)
/* Change this directory where the file to load resides if neccessary
 * --- Parameter: ---
 * char *filepath        : file to load with complete path
 * --- Return: ---
 * char *ChdirToLoadDir(): plain filename of file to load
 */
{
   char *cwd, *file;
   
   if ( !(file = strrchr(filepath, '/')) ) return filepath;  /* unchanged */
   *(file++) = 0;
   cwd       = getcwd(NULL, 512);

   if ( strcmp(cwd, filepath) ) {
      if ( chdir(filepath) ) 
         g_warning("chdir(%s) wasnt successful", filepath);
   } 
   FREE(cwd);
   return file;
}


static void FileLoadOk(GtkWidget *widget, GtkFileSelection *fs)
/*
 * --- Parameter: ---
 * GtkWidget        *widget : OK-button inside GtkFileSelection widget (can be ignored)
 * GtkFileSelection *fs     : GtkFileSelection widget
 *                            connected data: PAUL *p
 */
{
   PAUL      *p;
   PICTURE   *bild;
   char      *filename;

   p        = gtk_object_get_user_data(GTK_OBJECT(fs));

   if ( DoMatchPictures(p->opt->f) ) {
      g_warning("Load files while moving images not supported yet.");
      gtk_widget_destroy(GTK_WIDGET(fs));
      return;
   }
   filename = ChdirToLoadDir(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));

   bild = ReadPic(filename, p->opt);
   gtk_widget_destroy(GTK_WIDGET(fs));
   if ( !bild ) return;
   
   AddImageToFilelist(p, bild);
}

static void FileSaveOk(GtkWidget *w, GtkFileSelection *fs)
/*
 * --- Parameter: ---
 * GtkWidget *w         : can be ignored
 * GtkFileSelection *fs : fileselection widget, PAUL *p set as user_data
 */
{
   PAUL    *p;
   PICTURE *bild;

   p        = gtk_object_get_user_data(GTK_OBJECT(fs));
   bild     = BILD(p->activ);

   if ( FilenameExtractFromSelection(fs, bild) ) {
      gtk_label_set(GTK_LABEL(bild->label), bild->file);
      gdk_window_set_title(p->show->window, bild->file);
   }
   FileSave(NULL, p);
}

void FileLoad(GtkWidget *widget, PAUL *p)
/*
 * --- Parameter: ---
 * GtkWidget *widget: menu entry for "Open" file (can be ignored)
 * PAUL      *p     : PAUL structure
 */
{
   char     filename[MAXPATHLEN] = "";
   PICTURE *bild = BILD(g_list_last(p->piclist));
   
   if ( bild ) {
      if ( bild->file ) strcpy(filename, bild->file);
      if ( bild->ext  ) strcat(strcat(filename, "."), bild->ext);
   }
   FileSelectionLoadSave(p, GTK_SIGNAL_FUNC(FileLoadOk), filename, "Load image");
}
		   
void FileSave(GtkWidget *widget, PAUL *p)
/*
 * --- Parameter: ---
 * GtkWidget *widget: menu entry for "Save" file (can be ignored)
 * PAUL      *p     : PAUL structure
 */
{
   register PICTURE *bild = BILD(p->activ);
   GList            *pl;

   if ( DoMatchPictures(p->opt->f) && !IsMatchAuto(p->opt->mov) ) {
      g_message("Please accept all images before saving!");
      return;
   }

   if ( !bild->flag ) {
      printf("No changes in %s need to be saved.\n", bild->file);
      return;
   }
   pl = g_list_append(NULL, bild);
   SavePictures(pl, SAFEFLAG);
   g_list_free(pl);
   bild->flag = 0;   /* to recognize further changes */
}
		   
void FileSaveAs(GtkWidget *widget, PAUL *p)
/*
 * --- Parameter: ---
 * GtkWidget *widget: menu entry for "Save AS" file (can be ignored)
 * PAUL      *p     : PAUL structure
 */
{
   BILD(p->activ)->flag |= SAFEFLAG;
   FileSelectionLoadSave(p, GTK_SIGNAL_FUNC(FileSaveOk), BILD(p->activ)->file, "Save image");
}

static void FirstPictureLoaded(GtkWidget *widget, GtkFileSelection *fs)
/*
 * --- Parameter: ---
 * GtkWidget        *widget : OK-button inside GtkFileSelection widget (can be ignored)
 * GtkFileSelection *fs     : GtkFileSelection widget
 *                            connected data: PAUL *p
 */
{
   PAUL      *p;
   PICTURE   *bild;
   char      *filename;

   p        = gtk_object_get_user_data(GTK_OBJECT(fs));
   filename = ChdirToLoadDir(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));
      
   bild = ReadPic(filename, p->opt);
   gtk_widget_destroy(GTK_WIDGET(fs));
   if ( !bild ) {
      LoadFirstPicture(p);
      return;
   }
   
   if ( p->piclist ) g_warning("p->piclist isnt NULL!!!");
   p->piclist = g_list_append(NULL, bild);
   CreateShowPixmaps(p->activ = p->piclist);
   GtkInterface(p);
}

void LoadFirstPicture(PAUL *p)
{
   GtkWidget *window = NULL;

   window = gtk_file_selection_new("Load image");

   gtk_window_position(GTK_WINDOW(window), GTK_WIN_POS_MOUSE);
   gtk_signal_connect(GTK_OBJECT(window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &window);

   gtk_object_set_user_data(GTK_OBJECT(window), p);

   gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(window)->ok_button), "clicked", 
                      GTK_SIGNAL_FUNC(FirstPictureLoaded), window);
   gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(window)->cancel_button), "clicked", 
                      GTK_SIGNAL_FUNC(PaulExit), p);
  
   gtk_widget_show_all(window);
}

static void SureDeleteFile(GtkWidget *button, char *file)
/*
 * --- Parameter: ---
 * GtkWidget *button: OK-button to delete file (window connected)
 * char      *file  : file to delete
 */
{
   unlink(file);
}

static void DeleteFromList(GtkWidget *button, PAUL *p)
/*
 * --- Parameter: ---
 * GtkWidget *button: Delete from list button
 * PAUL      *p     : PAUL structure
 */
{
   GList     *pl = g_list_append(NULL, BILD(p->activ));
   GtkWidget *list_item = SearchListItem(GTK_LIST(p->filelist)->children, p->activ);

   FreeBild(pl);
   g_list_free(pl);
   RemoveImageFromFilelist(p, list_item);
}

static void DeleteFromDisk(GtkWidget *button, PAUL *p)
/*
 * --- Parameter: ---
 * GtkWidget *button: Delete from disk button
 * PAUL      *p     : PAUL structure
 */
{
   char    *file, buf[MAXPATHLEN];
   struct   stat  stat_buf;
   PICTURE *bild = BILD(p->activ);

   assert ( (file = malloc(MAXPATHLEN)) );
   *file = 0;
   if ( bild->dir ) {
      char *cwd;
      
      if ( strcmp((cwd = getcwd(NULL, MAXPATHLEN)), bild->dir) ) {
         strcat(strcpy(file, bild->dir), "/");
         FREE(cwd);
      }
   }
   
   strcat(file, bild->file);
   if ( bild->ext ) strcat(strcat(file, "."), bild->ext);

   if ( stat(file, &stat_buf) ) {
      g_warning("File %s doesnt exist", file);
      return;
   }
   sprintf(buf, "Delete %s from disk?", file);
   AreYouSure(file, GTK_SIGNAL_FUNC(SureDeleteFile), buf);
   DeleteFromList(button, p);
}

void FileDelete(GtkWidget *widget, PAUL *p)
/*
 * --- Parameter: ---
 * GtkWidget *widget: menu entry for "Delete" file (can be ignored)
 * PAUL      *p     : PAUL structure
 */
{
   static    GtkWidget *window = NULL;
   GtkWidget           *main_vbox, /* what should this mean ?? :)        */
                       *button;    /* OK / Cancel                        */
   char                 buf[256];

   if ( DoMatchPictures(p->opt->f) && !IsMatchAuto(p->opt->mov) ) {
      g_message("Please accept all images before deleting!");
      return;
   }

   if (!window) {
      window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
      gtk_window_position(GTK_WINDOW(window), GTK_WIN_POS_MOUSE);
      gtk_signal_connect(GTK_OBJECT(window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed),
                         &window);
      gtk_window_set_title(GTK_WINDOW(window), "Delete image from list");
      main_vbox = gtk_vbox_new(FALSE, 5);
      gtk_container_border_width(GTK_CONTAINER(main_vbox), 10);
      gtk_container_add(GTK_CONTAINER(window), main_vbox);
      
      sprintf(buf, "Delete %s from %s's image list?", BILD(p->activ)->file, exename);
      button = gtk_button_new_with_label(buf);
      gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(DeleteFromList), p);
      gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
                                GTK_OBJECT(window));
      gtk_box_pack_start(GTK_BOX(main_vbox), button, FALSE, TRUE, 5);
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
      gtk_widget_grab_default(button);
      
      sprintf(buf, "Delete %s from disk???", BILD(p->activ)->file);
      button = gtk_button_new_with_label(buf);
      gtk_signal_connect(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(DeleteFromDisk), p);
      gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
                                GTK_OBJECT(window));
      gtk_box_pack_start(GTK_BOX(main_vbox), button, TRUE, TRUE, 5);
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);

      button = gtk_button_new_with_label("Cancel");
      gtk_signal_connect_object(GTK_OBJECT(button), "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
                                GTK_OBJECT(window));
      gtk_box_pack_start(GTK_BOX(main_vbox), button, TRUE, TRUE, 5);
      GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
   }

   if (!GTK_WIDGET_VISIBLE(window) ) gtk_widget_show_all(window);
   else                              gtk_widget_destroy(window);

}
		   
