; 
; PDBVPLT.SCM - window manager routines for PDBView
; 
; Source Version: 2.0
; Software Release #92-0043
;
; #include <pact-copyright.h>
; 

(define window-manager-table #f)
(define window-list nil)
(define must-clear-viewport #t)

(define domain-1d (list cartesian polar insel))
(define domain-2d (list contour image fill-poly wire-frame shaded mesh vector))
(define domain-3d (list shaded mesh))

(define current-window nil)
(define WIN-device #f)
(define PS-device #f)
(define CGM-device #f)
(define JPEG-device #f)

(define hard-copy-device-types '("PS" "CGM" "JPEG"))

(define WINDOW   -5)   ; clear window
(define VIEWPORT -6)   ; clear PGS viewport

(define window-index 65)

(define window-highlight-color black)
(define viewport-highlight-color black)
(define un-highlight-color dark-gray)

(define image-box     (list 0.175 0.85 0.25 0.825))
(define contour-box   (list 0.175 0.735 0.175 0.825))
(define mesh-box      (list 0.175 0.85 0.25 0.825))
(define fill-poly-box (list 0.175 0.85 0.25 0.825))
(define vector-box    (list 0.175 0.9 0.175 0.9))

(define contour-box-size   1)
(define image-box-size     2)
(define mesh-box-size      2)
(define fill-poly-box-size 2)
(define vector-box-size    3)
(define overlay-box-size   2)

(define default-vr nil)
;--------------------------------------------------------------------------

;                          HELPER ROUTINES

;--------------------------------------------------------------------------

; OK-TO-DRAW? - return TRUE iff the device is HARDCOPY or we are in
;             - graphics mode

(define (ok-to-draw? dev)
    (interactive off)
    (if dev
	(let* ((props (pg-device-properties dev)))
	  (or (= (graphics-mode) 1)
	      (memv (list-ref props 1) hard-copy-device-types)))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; CYCLE-LIST - return the next item on the list after the specified
;            - one and assume the list is logically circular

(define (cycle-list item lst)

    (define (cycle item lst rem)
        (let* ((first (car rem))
	       (rest (cdr rem)))
	  (if (eqv? item first)
	      (if (pair? rest)
		  (car rest)
		  (car lst))
	      (cycle item lst rest))))

    (cycle item lst lst))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; IDENTIFY-ONE - return a mapping one way or the other

(define (identify-one x grimms)
    (if (number? x)
	(list-ref grimms (- x 1))
	x))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; VIEW-PREP-ARGS - prepare argument list for viewport operations
;                - output should be
;                - (<vname> <wname> (mapping-specs ... ))

(define (view-prep-args args)
    (let* ((x (if (pair? args)
		  (car args)
		  args))
	   (first (if (pair? x)
		      (car x)
		      nil))
	   vname wname)

      (if (or (pm-mapping? first) (pm-set? first) (pg-image? first)
	      (pg-graph? first) (number? first) (printable? first))
	  (set! vname nil)
	  (begin (set! vname first)
		 (set! x (if (pair? x)
			     (cdr x)
			     nil))
		 (set! first (if (pair? x)
				 (car x)
				 nil))))
		   
      (if (or (pm-mapping? first) (pm-set? first) (pg-image? first)
	      (pg-graph? first) (number? first) (printable? first))
	  (set! wname nil)
	  (begin (set! wname first)
		 (set! x (if (pair? x)
			     (cdr x)
			     nil))))

      (list vname wname x)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; GET-EXTREMA - return the extrema of the two specified limits

(define (get-extrema l1 l2 lims)
    (if l1
	(let* ((l1n (list-ref l1 0))
	       (l1x (list-ref l1 1))
	       (l2n (list-ref l2 0))
	       (l2x (list-ref l2 1))
	       (ln  (if (< l1n l2n) l1n l2n))
	       (lx  (if (> l1x l2x) l1x l2x)))
	  (get-extrema (list-tail l1 2)
		       (list-tail l2 2)
		       (append (list lx ln) lims)))
	(reverse lims)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; GET-DOMAIN-EXTREMA - return the extrema of the given graphs domains

(define (get-domain-extrema lst lims)
    (if lst
	(let* ((g (car lst))
	       (rest (cdr lst))
	       (limits (pg-domain-limits g))
	       (extr (if limits limits (pg-domain-extrema g))))
	  (get-domain-extrema rest
			      (get-extrema (if lims lims extr) extr nil)))
	lims))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; GET-RANGE-EXTREMA - return the extrema of the given graphs ranges

(define (get-range-extrema lst lims)
    (if lst
	(let* ((g (car lst))
	       (rest (cdr lst))
	       (limits (pg-range-limits g))
	       (extr (if limits limits (pg-range-extrema g))))
	  (get-range-extrema rest
			     (get-extrema (if lims lims extr) extr nil)))
	lims))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; SET-DR-FROM-VR - set the drawable-rendering from the viewport-rendering

(define (set-dr-from-vr vport lst indx)
    (let* ((rendering (viewport-rendering vport))
	   (item      (list-ref lst indx)))
      (if rendering
	  (set! rendering (car rendering))
	  (set! rendering default-vr))
      (cond ((eqv? rendering cartesian)
	     (set-cdr! item (list (list 'render 'cartesian))))
	    ((eqv? rendering polar)
	     (set-cdr! item (list (list 'render 'polar))))
	    ((eqv? rendering insel)
	     (set-cdr! item (list (list 'render 'insel))))
	    ((eqv? rendering histogram-plot)
	     (set-cdr! item (list (list 'render 'histogram-plot))))
	    ((eqv? rendering scatter-plot)
	     (set-cdr! item (list (list 'render 'scatter-plot))))
	    ((eqv? rendering logical-plot)
	     (set-cdr! item (list (list 'render 'logical-plot))))
	    ((eqv? rendering contour)
	     (set-cdr! item (list (list 'render 'contour))))
	    ((eqv? rendering image)
	     (set-cdr! item (list (list 'render 'image))))
	    ((eqv? rendering wire-frame)
	     (set-cdr! item (list (list 'render 'wire-frame))))
	    ((eqv? rendering shaded)
	     (set-cdr! item (list (list 'render 'shaded))))
	    ((eqv? rendering vector)
	     (set-cdr! item (list (list 'render 'vector))))
	    ((eqv? rendering fill-poly)
	     (set-cdr! item (list (list 'render 'fill-poly))))
	    ((eqv? rendering mesh)
	     (set-cdr! item (list (list 'render 'mesh)))))))

;--------------------------------------------------------------------------

;                       WINDOW STRUCTURE

;--------------------------------------------------------------------------

; WINDOW Structure - (name device viewports current-viewport)

(define (window-name window)      (list-ref window 0))
(define (window-device window)    (list-ref window 1))
(define (window-viewports window) (list-ref window 2))
(define (current-viewport window) (list-ref window 3))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (make-window name device viewports)
    (list name device viewports nil))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (set-current-viewport! window p)
    (highlight-window-viewport window un-highlight-color)
    (set-car! (list-tail window 3) p)
    (if p
        (highlight-window-viewport window viewport-highlight-color)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (set-window-device! window dev)
    (set-car! (list-tail window 1) dev))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (set-window-viewports! window p)
    (set-car! (list-tail window 2) p))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (replace-window original window)
    (if window
	(set-cdr! original (cdr window))
	(set-cdr! original nil)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (default-window-info)
    (let* ((name (sprintf "%c" window-index)))
      (set! window-index (+ window-index 1))
      (list name "COLOR" "WINDOW" 0.1 0.1 0.4 0.4)))

;--------------------------------------------------------------------------

;                       VIEWPORT STRUCTURE

;--------------------------------------------------------------------------

; VIEWPORT Structure - (name window (x y dx dy)
;                                   (<rendering info>)
;                                   (<range info>)
;                                   (<domain info>)
;                                   (mappings <graphs>)
;                                   (meshes <sets>)
;                                   (images <images>))

(define (viewport-name viewp)        (list-ref viewp 0))
(define (viewport-shape viewp)       (list-ref viewp 2))
(define (viewport-rendering viewp)   (list-ref viewp 3))
(define (viewport-range viewp)       (list-ref viewp 4))
(define (viewport-domain viewp)      (list-ref viewp 5))
(define (viewport-graph-list viewp)  (list-ref viewp 6))
(define (viewport-mesh-list viewp)   (list-ref viewp 7))
(define (viewport-image-list viewp)  (list-ref viewp 8))

(define (viewport-drawables viewp n)
    (let* ((lst (list-ref viewp n)))
      (if lst
	  (cdr lst)
	  lst)))

(define (viewport-graphs viewp)
    (viewport-drawables viewp 6))

(define (viewport-meshes viewp)
    (viewport-drawables viewp 7))

(define (viewport-images viewp)
    (viewport-drawables viewp 8))

(define (viewport-window viewp)
    (assv (list-ref viewp 1) window-list))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (make-viewport name window x y dx dy range-limit domain-limit
		       rendering graphs meshes images)
    (list name window
	  (list x y dx dy)
	  rendering range-limit domain-limit
	  (cons 'graphs graphs)
	  (cons 'meshes meshes)
	  (cons 'images images)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; GET-VIEWPORT - get the specified viewport
;                Usage: (get-viewport [viewport] | [viewport-name [window-name]])

(define (get-viewport vp wind)
    (cond ((null? vp)
	   (current-viewport (if wind
				 wind
				 current-window)))
	  ((pair? vp)
	   vp)
	  (else
	   (let* ((window (if wind
			      (assv wind window-list)
			      current-window))
		  (vports (window-viewports window)))
	     (assv vp vports)))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; FIND-VIEWPORT - sort out the given specifications and try to come
;               - up with a viewport to match

(define (find-viewport spec)
    (let* ((cspec (car spec))
	   (arg (if (pair? cspec)
		    cspec
		    spec))
	   (vname (if arg (car arg) "a"))
	   (vp (get-viewport vname nil)))
      (if (not vp)
	  (let* ((info (if arg
			   (cdr arg)
			   (default-viewport-info)))
		 (x    (list-ref info 0))
		 (y    (list-ref info 1))
		 (dx   (list-ref info 2))
		 (dy   (list-ref info 3))
		 (rest (list-tail info 4))
		 (window (if rest
			     (assv (car rest) window-list)
			     current-window))
		 (wname (window-name window))
		 (vports (window-viewports window)))
	    (set! vp (make-viewport vname wname x y dx dy nil
				    nil nil nil nil nil))
	    (set-window-viewports! window (cons vp vports))
	    (set-current-viewport! window vp)))
      vp))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (setup-viewport device shape)
    (if (and device shape)
	(let* ((xmin (list-ref shape 0))
	       (ymin (list-ref shape 1))
	       (dx   (list-ref shape 2))
	       (dy   (list-ref shape 3))
	       xmax ymax)

; non-optimal way to do this
	  (pg-set-vector-attributes! device vector-scale vect-scale)
	  (pg-set-vector-attributes! device vector-headsize vect-headsz)
	  (pg-set-vector-attributes! device vector-color vect-color)

	  (pg-set-frame! device xmin (+ xmin dx) ymin (+ ymin dy))
	  (pg-set-viewport! device 0.2 0.9 0.2 0.9))))

;--------------------------------------------------------------------------

;                         DRAWABLE STRUCTURE 

;--------------------------------------------------------------------------

; DRAWABLE Structure - (<graph> | <image> | <set> [rendering-info]* )

(define (make-drawable d . info)
   (cons d info))

(define (drawable-data dr)
    (if (pair? dr)
	(car dr)
	dr))

(define (drawable-render dr)
    (if (pair? dr)
	(cdr dr)
	nil))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (highlight-window window color)
    (if window
       (let* ((device (window-device window))
	      (pal   (pg-current-palette device))
	      (wmin  (pg-normalized->world device 0.000 0.000))
	      (wmax  (pg-normalized->world device 1.000 1.000))
	      (wxmin (car wmin))
	      (wymin (cadr wmin))
	      (wxmax (car wmax))
	      (wymax (cadr wmax)))
	 (pg-make-device-current device)
	 (pg-set-palette! device "standard")
	 (pg-set-line-color! device color)
	 (pg-set-line-width! device 2.0)
	 (pg-draw-box device wxmin wxmax wymin wymax)
	 (pg-set-palette! device pal)
	 (pg-set-line-width! device 0.0)
	 (pg-update-view-surface device)
	 (pg-make-device-current WIN-device))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (highlight-port device xmin xmax ymin ymax title color)
    (let* ((pal   (pg-current-palette device))
	   (wmin  (pg-normalized->world device xmin ymin))
	   (wmax  (pg-normalized->world device xmax ymax))
	   (wxmin (car wmin))
	   (wymin (cadr wmin))
	   (wxmax (car wmax))
	   (wymax (cadr wmax))
	   (dx    (- wxmax wxmin))
	   (dy    (- wymax wymin))
	   (txmin (+ wxmin (* 0.01 dx)))
	   (tymin (+ wymin (* 0.01 dy))))
      (pg-make-device-current device)
      (pg-set-palette! device "standard")
      (pg-set-line-color! device color)
      (pg-set-line-width! device 0.0)
      (pg-draw-box device wxmin wxmax wymin wymax)
      (pg-set-text-color! device color)
      (pg-draw-text-abs device txmin tymin title)
      (pg-set-palette! device pal)
      (pg-update-view-surface device)
      (pg-make-device-current WIN-device)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (highlight-window-viewport window color . p)
    (let* ((port (if p
		     (car p)
		     (current-viewport window))))
      (if port
	  (let* ((device (window-device window))
		 (shape  (viewport-shape port))
		 (xmin   (list-ref shape 0))
		 (ymin   (list-ref shape 1))
		 (dx     (list-ref shape 2))
		 (dy     (list-ref shape 3))
		 (xmax   (+ xmin dx))
		 (ymax   (+ ymin dy)))
	    (highlight-port device xmin xmax ymin ymax
			    (viewport-name port) color)))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (pg-clear-viewport device port)
    (let* ((shape (viewport-shape port)))
      (if shape
	  (let* ((xmin  (list-ref shape 0))
		 (ymin  (list-ref shape 1))
		 (dx    (list-ref shape 2))
		 (dy    (list-ref shape 3))
		 (xmax  (+ xmin dx))
		 (ymax  (+ ymin dy)))
	    (pg-clear-region device xmin xmax ymin ymax 4)
	    (pg-update-view-surface device)))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (set-viewport-shape! viewp . rest)
    (set-car! (list-tail viewp 2) rest))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (set-viewport-rendering! viewp info)
    (let* ((rend (list-tail viewp 3)))
        (set-car! rend info)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (set-viewport-range! viewp info)
    (let* ((ran (list-tail viewp 4)))
        (set-car! ran info)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (set-viewport-domain! viewp info)
    (let* ((dom (list-tail viewp 5)))
        (set-car! dom info)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (set-viewport-graphs! viewp g)
    (set-cdr! (viewport-graph-list viewp) g))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (set-viewport-meshes! viewp g)
    (set-cdr! (viewport-mesh-list viewp) g))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (set-viewport-images! viewp i)
    (set-cdr! (viewport-image-list viewp) i))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

(define (default-viewport-info)
    (list 0.0 0.0 1.0 1.0))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------
(define (set-viewport-area! gr ms im)

    (if (eqv? viewport-area-save nil)
        (begin
           (set! overlay-box-size 100)
           (if im
               (if (> overlay-box-size image-box-size)
                   (begin (set! overlay-box image-box)
                          (set! overlay-box-size image-box-size))))

           (if ms
               (if (> overlay-box-size mesh-box-size)
                   (begin (set! overlay-box mesh-box)
                          (set! overlay-box-size mesh-box-size))))

           (if gr
               (for-each
                (lambda (x)
                    (let* ((rspec (cadar (drawable-render x))))
		      (cond ((eqv? rspec 'image)
			     (if (> overlay-box-size image-box-size)
				 (begin (set! overlay-box image-box)
					(set! overlay-box-size image-box-size))))
			    ((eqv? rspec 'contour)
			     (if (> overlay-box-size contour-box-size)
				 (begin (set! overlay-box contour-box)
					(set! overlay-box-size contour-box-size))))
			    ((eqv? rspec 'mesh)
			     (if (> overlay-box-size mesh-box-size)
				 (begin (set! overlay-box mesh-box)
					(set! overlay-box-size mesh-box-size))))
			    ((eqv? rspec 'fill-poly)
			     (if (> overlay-box-size fill-poly-box-size)
				 (begin (set! overlay-box fill-poly-box)
					(set! overlay-box-size fill-poly-box-size))))
			    ((eqv? rspec 'vector)
			     (if (> overlay-box-size vector-box-size)
				 (begin (set! overlay-box vector-box)
					(set! overlay-box-size vector-box-size)))))))
                 gr))
           
           (set! viewport-area-box overlay-box)))) 
            
         
;--------------------------------------------------------------------------

;                       WINDOW MANAGER ROUTINES

;--------------------------------------------------------------------------

; WM-ADD - the window manager add command

(define (wm-add name fnc)
    (hash-install name fnc window-manager-table))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-WIN-OPEN - the window manager "wopen" command

(define (wm-win-open . args)
    (highlight-window current-window un-highlight-color)
    (cond ((or (and args (car args)) (not window-list))
	   (let* ((args  (if args
			     (car args)
			     nil))
		  (info  (if (and args (< 6 (length args)))
			     args
			     (default-window-info)))
		  (wname (if args
			     (car args)
			     (list-ref info 0)))
		  (type  (list-ref info 1))
		  (name  (list-ref info 2))
		  (winx  (list-ref info 3))
		  (winy  (list-ref info 4))
		  (windx (list-ref info 5))
		  (windy (list-ref info 6))
		  (window (assv wname window-list))
		  (neverbeen (not window))
		  new)
	     (if neverbeen
		 (let* ((title (sprintf "PDBView Display %s" wname))
			(device (pg-make-device name type title)))
		   (if (or (pg-open-device device winx winy windx windy)
			   (not (ok-to-draw? device)))
		       (set! new (make-window wname device nil))
		       (set! new #f))
		   (if new
		       (begin
			 (pg-set-palette! device "spectrum")
			 (if neverbeen
			     (set! window-list (cons new window-list))
			     (replace-window window new)))
		       (printf nil "Can't open window %s\n" wname))
		   (set! window new)))
	     (set! current-window window)))
	  (else
	   (set! current-window (cycle-list current-window window-list))))
    (set! WIN-device (window-device current-window))
    (pg-make-device-current WIN-device)
    (highlight-window current-window window-highlight-color))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-WIN-CLOSE - the window manager "wclose" command
;              - Args: <window-list>

(define (wm-win-close windows)
    (if (null? windows)
        (set! windows (list current-window)))

    (define (close-one x)
        (let* ((window (if (pair? x)
			   x
			   (assv x window-list))))
	  (if window
	      (let* ((device (window-device window)))
		(if device
		    (begin
		      (pg-close-device device)
		      (replace-window window nil)
		      (if (eqv? device WIN-device)
			  (set! WIN-device #f))
		      (if (eqv? window current-window)
			  (set! current-window nil))))
		(set! window-list
		      (splice-out window window-list equal?))))))

    (if windows
	(begin
	  (map close-one windows)
	  (if (and (null? current-window) (not (null? window-list)))
	      (wm-win-open (car window-list))))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-WIN-CONNECT - the window manager "wconnect" command
;                - Args: [<window>] <device>

(define (wm-win-connect window device)
    (if window
	(set-window-device! window device)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-WIN-DISCONNECT - the window manager "wdisconnect" command
;                   - Args: <window>

(define (wm-win-disconnect window)
    (if window
	(let* ((device (window-device window)))
	    (if device
		(begin
;		    (pg-close-device device)
		    (set-window-device! window nil)
;		    (if (eqv? device WIN-device)
;			(set! WIN-device #f))
;		    (if (eqv? window current-window)
;			(set! current-window #f)))))))
)))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-WIN-HARDCOPY - the window manager "whardcopy" command

(define (wm-win-hardcopy device res window . rest)
    (let* ((orig (window-device window))
	   (finfo (cons device (pg-text-font orig)))
	   (vports (if window
		       (window-viewports window)
		       #f)))
      (if vports
	  (begin (if (> res 0)
		     (pg-set-resolution-scale-factor! device res))

		 (wm-win-disconnect window)
		 (wm-win-connect window device)

		 (apply pg-set-text-font! finfo)

		 (pg-clear-window device)
		 (pg-set-clear-mode! device off)
		 (pg-set-finish-state! device off)
		 (for-each (lambda (x)
			     (vp-hardcopy device res
					  (viewport-name x)
					  rest))
			   vports)

		 (pg-set-clear-mode! device WINDOW)
		 (pg-set-finish-state! device on)
		 (pg-finish-plot device)

		 (wm-win-disconnect window)
		 (wm-win-connect window orig))))
    window)

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-WIN-UPDATE - the window manager "wupdate" command
;               - Args: <window-list>

(define (wm-win-update windows)
    (interactive off)
    (if (= (graphics-mode) 1)
	(begin
	    (if (null? windows)
		(set! windows window-list))

	    (define (update-one x)
		(let* ((window (if (pair? x)
				   x
				   (assv x window-list)))
		       (vports (if window
				   (window-viewports window)
				   #f)))
		  (if (and window vports)
		      (for-each wm-vp-update vports))
		  (if (eqv? window current-window)
		      (highlight-window window window-highlight-color)
		      (highlight-window window un-highlight-color)))
		(window-name x))

	    (if windows
		(begin
		  (map update-one windows))))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-VP-OPEN - the window manager "vopen" command

(define (wm-vp-open . spec)
    (if (not current-window)
	(wm-win-open))
    (cond ((and spec (car spec))
	   (set-current-viewport! current-window
				  (find-viewport spec)))

	  (current-window
	   (let* ((cv (current-viewport current-window))
		  (ports (window-viewports current-window)))
	     (if (not cv)
		 (let* ((name "A"))
		   (apply wm-vp-open (cons name (default-viewport-info)))
		   (set! cv (current-viewport current-window))
		   (set! ports (window-viewports current-window))))
	     (set-current-viewport! current-window
				    (cycle-list cv ports))))
	  
	  (else
	   (wm-win-open)
	   (wm-vp-open))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-VP-CLOSE - the window manager "vclose" command
;             - Args: [vname [wname]]

(define (wm-vp-close . args)
    (let* ((x     (view-prep-args args))
	   (vname (list-ref x 0))
	   (wname (list-ref x 1))
	   (vp (get-viewport vname wname)))
      (if vp
	  (let* ((window (viewport-window vp))
		 (device (window-device window))
		 (ports  (window-viewports window)))
	    (if (eqv? vp (current-viewport window))
		(let* ((next (cycle-list vp ports)))
		  (if (eqv? vp next)
		      (set-current-viewport! window nil)
		      (set-current-viewport! window next))))
	    (pg-clear-viewport device vp)
	    (set! ports (splice-out vp ports eqv?))
	    (set-window-viewports! window ports)))
      vname))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-VP-DLIMIT - the window manager "vdlimit" command
;              - Args: [vname [wname]] <x1_min> <x1_max> ...

(define (wm-vp-dlimit . args)
    (let* ((x     (view-prep-args args))
	   (vname (list-ref x 0))
	   (wname (list-ref x 1))
	   (vp    (get-viewport vname wname)))
      (if vp
	  (set-viewport-domain! vp (car (list-tail x 2))))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-VP-RLIMIT - the window manager "vrlimit" command
;              - Args: [vname [wname]] <x1_min> <x1_max> ...

(define (wm-vp-rlimit . args)
    (let* ((x      (view-prep-args args))
	   (vname  (list-ref x 0))
	   (wname  (list-ref x 1))
	   (vp     (get-viewport vname wname)))
      (if vp
	  (set-viewport-range! vp (car (list-tail x 2))))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-MAP-DLIMIT - the window manager "mdlimit" command
;               - Args: [vname [wname]] <mapping> <x1_min> <x1_max> ...

(define (wm-map-dlimit . args)
    (let* ((x      (view-prep-args args))
	   (vname  (list-ref x 0))
	   (wname  (list-ref x 1))
	   (items  (list-ref x 2))
	   (map    (car items))
	   (limits (cdr items))
	   (vp     (get-viewport vname wname))
	   (gr     (viewport-graphs vp)))
      (if (integer? map)
	  (if gr
	      (pg-set-domain-limits! (car (list-ref gr (- map 1))) limits))
	  (pg-set-domain-limits! map limits))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-MAP-RLIMIT - the window manager "mrlimit" command
;               - Args: [vname [wname]] <mapping> <x1_min> <x1_max> ...

(define (wm-map-rlimit . args)
    (let* ((x      (view-prep-args args))
	   (vname  (list-ref x 0))
	   (wname  (list-ref x 1))
	   (items  (list-ref x 2))
	   (map    (car items))
	   (limits (cdr items))
	   (vp     (get-viewport vname wname))
	   (gr     (viewport-graphs vp)))
      (if (integer? map)
	  (if gr
	      (pg-set-range-limits! (car (list-ref gr (- map 1))) limits))
	  (pg-set-range-limits! map limits))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; SET-WORLD-COORDINATES! 

(define (set-world-coordinates! device dlimits)
    (if dlimits
	(let* ((x1 (list-ref dlimits 0))
	       (x2 (list-ref dlimits 1))
	       (y1 (list-ref dlimits 2))
	       (y2 (list-ref dlimits 3)))
	  (pg-set-world-coordinate-system! device x1 x2 y1 y2))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; VP-UPDATE-DRAWABLE - do the work of plotting for graphs, images,
;                    - and domains

(define (vp-update-drawable device func dr dlimits rlimits info)
    (let* ((data  (drawable-data dr))
	   (rspec (drawable-render dr))
	   (basic (list device data))
	   (im    (pg-image? (car dr)))
	   (dlim  (if im
		      (pg-world-coordinate-system device)
		      (pg-domain-limits dr)))
	   (rlim  (if im
		      nil
		      (pg-range-limits dr))))

      (if im
	  (set-world-coordinates! device dlimits)
	  (begin
	    (pg-set-domain-limits! data dlimits)
	    (pg-set-range-limits! data rlimits)))

; set the global rendering state from the render list
      (if rspec
	  (for-each (lambda (x)
		      (let* ((f (car x))
			     (args (cdr x)))
			(cond ((eqv? f 'render)
			       (set! info (map eval args)))
			      (else
			       (let* ((fnc (eval f)))
				 (cond ((eqv? fnc view-angle)
					(apply pg-set-view-angle! args))
				       ((procedure? fnc)
					(apply fnc args))))))))
		    rspec))
      (apply func (append basic info))
      (if im
	  (set-world-coordinates! device dlim)
	  (begin
	    (pg-set-domain-limits! data dlim)
	    (pg-set-range-limits! data rlim)))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-VP-UPDATE - the window manager "vupdate" command

(define (wm-vp-update . args)

    (let* ((x       (view-prep-args args))
	   (vname   (list-ref x 0))
	   (wname   (list-ref x 1))
	   (port    (get-viewport vname wname))
	   (window  (viewport-window port))
	   (cp      (current-viewport window))
	   (device  (window-device window))
	   (swin    current-window)
	   (sdev    WIN-device)
	   (shape   (viewport-shape port))
	   (info    (viewport-rendering port))
	   (rlimits (viewport-range port))
	   (dlimits (viewport-domain port))
	   (gr      (viewport-graphs port))
	   (ms      (viewport-meshes port))
	   (im      (viewport-images port))
	   (rextr   (if rlimits
			rlimits
			(get-range-extrema (append gr (append im ms))
					   nil)))
	   (dextr   (if dlimits
			dlimits
			(get-domain-extrema (append gr (append im ms))
					    nil))))

      (if (ok-to-draw? device)
	  (begin
	    (set! current-window window)
	    (set! WIN-device device)
	    (pg-make-device-current WIN-device)

	    (if must-clear-viewport
		(pg-clear-viewport device port))

	    (if (eqv? cp port)
		(highlight-window-viewport
		 window
		 viewport-highlight-color
		 port)
		(highlight-window-viewport
		 window
		 un-highlight-color port))
	    (if (eqv? window current-window)
		(highlight-window window window-highlight-color)
		(highlight-window window un-highlight-color))

	    (setup-viewport device shape)
	    (pg-set-finish-state! device off)
	    (pg-set-clear-mode! device off)

	    (if overlay-flag
		(begin (set-viewport-area! gr ms im)
		       (window-manager "vsattr" "VIEW-PORT" "double *" viewport-area-box nil)))

	    (if gr
		(begin
		  (set! gr (reverse gr))
		  (for-each (lambda (x)
			      (vp-update-drawable device pg-draw-graph
						  x dextr rextr info))
			    gr)
		  (set! gr (reverse gr))))

	    (if ms
		(begin
		  (set! ms (reverse ms))
		  (for-each (lambda (x)
			      (vp-update-drawable device pg-draw-domain
						  x dextr nil info))
			    ms)
		  (set! ms (reverse ms))))

	    (if im
		(begin
		  (set! im (reverse im))
		  (for-each (lambda (x)
			      (vp-update-drawable device pg-draw-image
						  x dextr nil viewport-area-box))
			    im)
		  (set! im (reverse im))))

	    (pg-set-clear-mode! device WINDOW)
	    (pg-finish-plot device)

	    (set! current-window swin)
	    (set! WIN-device sdev)
	    (pg-make-device-current WIN-device)))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; VP-HARDCOPY - the worker for hardcopy of viewports

(define (vp-hardcopy device res vname rest)
    (let* ((port    (get-viewport vname rest))
	   (window  (viewport-window port))
	   (shape   (viewport-shape port))
	   (info    (viewport-rendering port))
	   (rlimits (viewport-range port))
	   (dlimits (viewport-domain port))
	   (gr      (viewport-graphs port))
	   (ms      (viewport-meshes port))
	   (im      (viewport-images port))
	   (rextr   (if rlimits
			rlimits
			(get-range-extrema gr nil)))
	   (dextr   (if dlimits
			dlimits
			(get-domain-extrema gr nil))))

      (setup-viewport device shape)
      (if gr
	  (begin
	    (set! gr (reverse gr))
	    (for-each (lambda (x)
			(vp-update-drawable device pg-draw-graph
					    x dextr rextr info))
		      gr)
	    (set! gr (reverse gr))))

      (if ms
	  (begin
	    (set! ms (reverse ms))
	    (for-each (lambda (x)
			(vp-update-drawable device pg-draw-domain
					    x dextr rextr info))
		      ms)
	    (set! ms (reverse ms))))

      (if im
	  (begin
	    (set! im (reverse im))
	    (for-each (lambda (x)
			(vp-update-drawable device pg-draw-image
					    x nil nil viewport-area-box))
		      im)
	    (set! im (reverse im))))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-VP-HARDCOPY - the window manager "vhardcopy" command

(define (wm-vp-hardcopy device res vname . rest)
    (if (> res 0)
	(pg-set-resolution-scale-factor! device res))
    (pg-clear-window device)
    (pg-set-finish-state! device off)
    (pg-set-clear-mode! device off)

    (vp-hardcopy device res vname rest)

    (pg-set-finish-state! device on)
    (pg-set-clear-mode! device WINDOW)
    (pg-finish-plot device))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-VP-RENDER - the window manager "vrender" command
;              - Args: [vname [wname]] render-info

(define (wm-vp-render . args)
    (let* ((x     (view-prep-args args))
	   (vname (list-ref x 0))
	   (wname (list-ref x 1))
	   (info  (list-ref x 2))
	   (mode  (if info (car info) nil))
	   (vp (get-viewport vname wname))
	   (gr (viewport-graphs vp))
	   (gr (if gr (car gr)))
	   (g  (drawable-data gr))
	   (ms (viewport-meshes vp))
	   (ms (if ms (car ms)))
	   (im (viewport-images vp))
	   (im (if im (car im))))

      (if mode
          (let* ((opt-1d (memv mode domain-1d))
		 (opt-2d (memv mode domain-2d))
		 (opt-3d (memv mode domain-3d)))
	    (if (cond (g
		       (or (and opt-1d
				(= (car (pm-mapping-dimension g)) 1))
			   (and opt-2d
				(= (car (pm-mapping-dimension g)) 2))
			   (and opt-3d
				(= (car (pm-mapping-dimension g)) 3))))
		      (ms #t)
		      (im
		       (memv mode domain-2d))
		      (else #f))
		(set-viewport-rendering! vp info)
		(cond (opt-1d (render-1d-1d mode))
		      (opt-2d (render-2d-1d mode))
		      (opt-3d (render-3d-1d mode)))))
	  (set-viewport-rendering! vp nil))

      (if mode
          (set! default-vr mode))

      (viewport-rendering vp)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-VP-ATTACH - the window manager "vattach" command
;              - Args: [vname [wname]] mappings|images ...

(define (wm-vp-attach . args)
    (if (not current-window)
	(wm-win-open))
    (if (not (current-viewport current-window))
	(wm-vp-open))

    (let* ((x     (view-prep-args args))
	   (vname (list-ref x 0))
	   (wname (list-ref x 1))
	   (g     (list-ref x 2))
	   (vp (get-viewport vname wname))
	   (gr (viewport-graphs vp))
	   (ms (viewport-meshes vp))
	   (im (viewport-images vp)))

      (define-macro (map-one x)
	(cond ((integer? x)
	       (let* ((type (pg-menu-item-type current-file x)))
		 (cond ((or (eqv? type "mapping") (eqv? type "curve"))
			(set! x ((io-function current-file "map-in")
				 current-file x)))
		       ((eqv? type "image")
			(set! x ((io-function current-file "image-in")
				 current-file x))))))
	      ((pm-mapping? x)
	       (set! x (pg-make-graph (pm-mapping-domain x)
				      (pm-mapping-range x))))
	      ((symbol? x)
	       (let* ((type (strtok (sprintf "%s" (variable-type x)) " *")))
		 (cond ((or (eqv? type "PM_mapping") (eqv? type "char"))
			(set! x (pdbdata->pg-graph current-file x)))
		       ((eqv? type "PG_image")
			(set! x (pdbdata->pg-image current-file x)))
		       ((eqv? type "PM_set")
			(set! x (pdbdata->pm-set current-file x)))))))

					; Add check of domain dimension of each new x for consistency
					; with first existing entry in gr, ms, or im (any one is enough).
					; Among other things this may mean writing a pm-set-dimension.
	(let* ((y (make-drawable x)))
	  (cond ((pg-graph? x)
		 (if (or (not gr)
			 (= (car (pm-mapping-dimension x))
			    (car (pm-mapping-dimension
				  (drawable-data (car gr))))))
		     (set! gr (cons y gr))
		     (printf nil "\nDimensions don't match previous mapping\n")))
		((pg-image? x)
		 (set! im (cons y im)))
		((pm-set? x)
		 (set! ms (cons y ms))))))

      (if g
	  (begin
	    (for-each map-one g)
	    (if gr
		(begin (if overlay-flag
			   (set-dr-from-vr vp gr 0))
		       (set-viewport-graphs! vp gr)))
	    (if im
		(begin (if overlay-flag
			   (set-dr-from-vr vp im 0))
		       (set-viewport-images! vp im)))
	    (if ms
		(begin (if overlay-flag
			   (set-dr-from-vr vp ms 0))
		       (set-viewport-meshes! vp ms)))))

      vp))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-VP-LIST - the window manager "vlist" command

(define (wm-vp-list vname wname)
    (let* ((vp (get-viewport vname wname))
	   (gr (viewport-graphs vp))
	   (im (viewport-images vp))
	   (ms (viewport-meshes vp))
	   (i 1))

      (define (lst-map x)
	(if x
	    (begin (printf nil
			   "%d       m       %s\n"
			   i
			   (pg-get-label (drawable-data (car x))))
		   (set! i (+ i 1))
		   (lst-map (cdr x) i))))

      (define (lst-image x)
	(if x
	    (begin (printf nil
			   "%d       i       %s\n"
			   i
			   (pg-get-label (drawable-data (car x))))
		   (set! i (+ i 1))
		   (lst-image (cdr x) i))))

      (define (lst-mesh x)
	(if x
	    (let* ((first (drawable-data (car x))))
	      (cond ((pm-mapping? first)
		     (printf nil
			     "%d       d       %s\n"
			     i
			     (pg-get-label first)))
		    ((pm-set? first)
		     (printf nil
			     "%d       s       %s\n"
			     i
			     (pg-get-label first))))
	      (set! i (+ i 1))
	      (lst-mesh (cdr x) i))))

					; The order gr, im, ms is significant. vm-vp-delete depends on it.
      (if (or gr im ms)
	  (begin
	    (printf nil "\n")
	    (if gr
		(lst-map gr))
	    (if im
		(lst-image im))
	    (if ms
		(lst-mesh ms))))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-VP-APPLY - the window manager "vapply" command
;              - Args: oper ([vname [wname]] <lsv-mapping-numbers> ...)

(define (wm-vp-apply . args)
    (let* ((oper   (list-ref args 0))
	   (x      (view-prep-args (list-tail args 1)))
	   (vname  (list-ref x 0))
	   (wname  (list-ref x 1))
	   (specs  (list-ref x 2))
	   (vp     (get-viewport vname wname))
	   (gr     (viewport-graphs vp))
	   (im     (viewport-images vp))
	   (ms     (viewport-meshes vp))
	   (grimms (append (append gr im) ms)))

; The order gr, im, ms in grimms is required for consistency with wm-vp-list.

      (define (apply-one x)
	(let* ((y (drawable-data x)))
	  (if (pair? y)
	      (apply oper y)
	      (oper y))))

      (if (and grimms specs)
	  (for-each apply-one
		    (map (lambda (x) (identify-one x grimms))
			 specs)))

      specs))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-VP-DELETE - the window manager "vdelete" command
;              - Args: [vname [wname]] <lsv-mapping-numbers> ...

(define (wm-vp-delete . args)
    (let* ((x      (view-prep-args args))
	   (vname  (list-ref x 0))
	   (wname  (list-ref x 1))
	   (specs  (list-ref x 2))
	   (vp     (get-viewport vname wname))
	   (gr     (viewport-graphs vp))
	   (im     (viewport-images vp))
	   (ms     (viewport-meshes vp))
	   (grimms (append (append gr im) ms)))

; The order gr, im, ms in grimms is required for consistency with wm-vp-list.

      (define (delete-one x)
	(let* ((y (drawable-data x)))
	  (cond ((pg-graph? y)
		 (set! gr (splice-out x gr eqv?)))
		((pg-image? y)
		 (set! im (splice-out x im eqv?)))
		((or (pm-set? y) (pm-mapping? x))
		 (set! ms (splice-out x ms eqv?))))))

      (if (and grimms specs)
	  (begin
	    (for-each delete-one
		      (map (lambda (x) (identify-one x grimms))
			   specs))
	    (set-viewport-graphs! vp gr)
	    (set-viewport-images! vp im)
	    (set-viewport-meshes! vp ms)))

      specs))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-VP-LABEL - the window manager "vlabel" command
;              - Args: [vname [wname]] <lsv-mapping-number>

(define (wm-vp-label . args)
    (let* ((x      (view-prep-args args))
	   (vname  (list-ref x 0))
	   (wname  (list-ref x 1))
	   (specs  (list-ref x 2))
	   (spec   (car specs))
	   (label  (cadr specs))
	   (vp     (get-viewport vname wname))
	   (gr     (viewport-graphs vp))
	   (im     (viewport-images vp))
	   (ms     (viewport-meshes vp))
	   (grimms (append (append gr im) ms)))

; The order gr, im, ms in grimms is required for consistency with wm-vp-list.

      (if (and grimms spec)
	  (let* ((y (drawable-data (identify-one spec grimms))))
	    (pg-set-label! y label)))

      spec))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-VP-SET-ATTR - the window manager "vsattr" command
;                - Args: <name> <type> <val> [<mappings>]*

(define (wm-vp-set-attr name type val things)
    (let* ((x      (view-prep-args things))
	   (vname  (list-ref x 0))
	   (wname  (list-ref x 1))
	   (vp     (get-viewport vname wname))
	   (gr     (viewport-graphs vp))
	   (ms     (viewport-meshes vp)))

      (define (do-item x l fnc)
	(if x
	    (let* ((y (if (number? x)
			  (list-ref l (- x 1))
			  x)))
	      (if y
		  (fnc (drawable-data y)
		       name type val)))))

      (if (or things gr)
	  (for-each (lambda (x) (do-item x gr pg-set-graph-attribute!))
		    (if things things gr)))
      (if (or things ms)
	  (for-each (lambda (x) (do-item x ms pm-set-set-attribute!))
		    (if things things ms)))

      things))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-VP-FONT - the window manager "vfont" command
;            - Args: <window> (<type> <size> <style>)

(define (wm-vp-font window args)
    (let* ((device (window-device window)))
      (if args
	  (let* ((type (list-ref args 0))
		 (size (list-ref args 1))
		 (style (list-ref args 2)))
	    (pg-set-text-font! device type style size))
	  (printf nil "Current font: %s\n" (pg-text-font device)))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-DR-RENDER - the window manager "drender" command
;              - Args: [vname [wname]] drawable-index render-info

(define (wm-dr-render . args)
    (let* ((x      (view-prep-args args))
	   (vname  (list-ref x 0))
	   (wname  (list-ref x 1))
	   (specs  (list-ref x 2))
	   (dr     (- (list-ref specs 0) 1))
	   (info   (list-ref specs 1))
	   (vp     (get-viewport vname wname))
	   (gr     (viewport-graphs vp))
	   (im     (viewport-images vp))
	   (ms     (viewport-meshes vp))
	   (grimms (append (append gr im) ms))
	   (item   (list-ref grimms dr)))
      (set-cdr! item info)
      item))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-PAL-CHANGE - the window manager "pchange" command
;             - Args: palette

(define (wm-pal-change args)
    (if args
	(if current-window
	    (pg-set-palette! (window-device current-window) (car args))
	    nil)
	nil))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-PAL-LIST - the window manager "plist" command
;             - Args: [wname]

(define (wm-pal-list args)
    (let* ((window (if args
		       (assv (car args) window-list)
		       current-window)))
      (if window
	  (pg-palettes (window-device window))
	  nil)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WM-PAL-SHOW - the window manager "pshow" command
;             - Args: [wname]

(define (wm-pal-show args)
    (let* ((window (if args
		       (assv (car args) window-list)
		       current-window)))
      (if window
	  (begin
	    (pg-show-palettes (window-device window))
	    (wm-win-update (list window)))
	  nil)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; WINDOW-MANAGER - a message passing approach to window management
;                - legal window messages:
;                -
;                - add         - add a new function to the window manager
;                - wopen       - open a new window with the given name
;                - wclose      - close the named window
;                - wconnect    - connect a new device to the window
;                - wdisconnect - disconnect the device from the window
;                - whardcopy   - map vhardcopy over the window viewports
;                - wupdate     - redraw everything in the window
;                - vopen       - open a new viewport in the specified window
;                - vclose      - close the named viewport
;                - vupdate     - redraw the graphs in the viewport
;                - vhardcopy   - draw the graphs in the viewport to a hc device
;                - vrender     - set the rendering mode for the viewport
;                - vapply      - apply an operator to a graph in a viewport
;                - vattach     - attach a graph to a viewport
;                - vdelete     - delete a graph from a viewport
;                - vlabel      - label a drawable
;                - vlist       - list the graphs attached to a viewport

(define (window-manager msg . rest)
   (if (not window-manager-table)
       (begin (set! window-manager-table (make-hash-table 17))
              (wm-add "add" wm-add)))

   (let* ((cmd (hash-lookup msg window-manager-table))
	  (proc (if cmd (cdr cmd))))
     (if (and proc (procedure? proc))
	 (apply proc rest)
	 (printf nil "Bad window manager command %s\n"
		 msg))))

;--------------------------------------------------------------------------

(window-manager "add" "drender"     wm-dr-render)
(window-manager "add" "mdlimit"     wm-map-dlimit)
(window-manager "add" "mrlimit"     wm-map-rlimit)
(window-manager "add" "plist"       wm-pal-list)
(window-manager "add" "pshow"       wm-pal-show)
(window-manager "add" "pchange"     wm-pal-change)
(window-manager "add" "wopen"       wm-win-open)
(window-manager "add" "wclose"      wm-win-close)
(window-manager "add" "wconnect"    wm-win-connect)
(window-manager "add" "wdisconnect" wm-win-disconnect)
(window-manager "add" "whardcopy"   wm-win-hardcopy)
(window-manager "add" "wupdate"     wm-win-update)
(window-manager "add" "vapply"      wm-vp-apply)
(window-manager "add" "vopen"       wm-vp-open)
(window-manager "add" "vclose"      wm-vp-close)
(window-manager "add" "vupdate"     wm-vp-update)
(window-manager "add" "vhardcopy"   wm-vp-hardcopy)
(window-manager "add" "vrender"     wm-vp-render)
(window-manager "add" "vattach"     wm-vp-attach)
(window-manager "add" "vdelete"     wm-vp-delete)
(window-manager "add" "vdlimit"     wm-vp-dlimit)
(window-manager "add" "vrlimit"     wm-vp-rlimit)
(window-manager "add" "vlabel"      wm-vp-label)
(window-manager "add" "vlist"       wm-vp-list)
(window-manager "add" "vsattr"      wm-vp-set-attr)
(window-manager "add" "vfont"       wm-vp-font)

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------
