; 
; PDBVCMD.SCM - PDBView commands
; 
; Source Version: 2.0
; Software Release #92-0043
;
; #include <pact-copyright.h>
; 

;(autoload* 'file-manager   "pdbvio.scm")
(autoload* 'window-manager "pdbvplt.scm")
(autoload* 'data-manager   "pdbvdat.scm")

(define autoplot-flag #t)

(define follow-directories #f)
(define directory-error-on-copy #t)

(define mapping-1d-1d (cons 1 1))
(define mapping-2d-1d (cons 2 1))
(define mapping-2d-2d (cons 2 2))

(define overlay-flag #f)

(define viewport-area-box nil)
(define viewport-area-save nil)
(define overlay-box   (list 0.175 0.85 0.25 0.825))

(define vect-scale  2.0)
(define vect-headsz 0.25)
(define vect-color  red)
(define scale 1)
(define headsize 2)
(define color 3)

;--------------------------------------------------------------------------

;                             AUXILLIARY FUNCTIONS

;--------------------------------------------------------------------------

; PRINT-NAME - return the print name of the object (a string)

(define (print-name x)
    (if (symbol? x)
	(symbol->string x)
	x))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; PLOT-HAND - handler for doing plots

(define (plot-hand func ran dom)
    (if (procedure? func)
	(display-mapping* (func (make-lr-mapping-direct ran dom)))
	(display-mapping* (make-lr-mapping-direct ran dom))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; GET-SCALAR - helper for scripts which want specific values
;            - from scalar expressions

(define (get-scalar fp name)
   (let* ((rval (read-pdbdata (if fp fp current-file) name)))
     (if rval
	 (cddr (pdb->list rval)))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; SEQ-AUX - the auxiliary sequence function

(define (seq-aux strt len delta lst)
    (if (> len 0)
        (seq-aux (+ strt delta) (- len 1) delta (cons strt lst))
        (reverse lst)))

; SEQUENCE

(define (sequence start length step) 
    "Procedure: Return a list containing <length> numbers beginning with 
                <start> and incrementing by <step>.
     Usage: sequence <start> <length> <step>"

    (seq-aux start length step nil))
;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; VARIABLE-DIMENSIONS - return a list of the variable's dimensions

(define (variable-dimensions name)
    (cddr (pdb->list (read-syment current-file name))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; VARIABLE-TYPE - return the variable's type

(define (variable-type name)
    (car (pdb->list (read-syment current-file name))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; VARIABLE-ADDRESS - return the variable's address

(define (variable-address name)
    (cadr (pdb->list (read-syment current-file name))))

;--------------------------------------------------------------------------

;                         USER LEVEL COMMANDS

;--------------------------------------------------------------------------

; ANIMATE - rotate the specified mapping thru 360 degrees

(define (animate theta dphi chi)
    "ANIMATE - Rotate the specified mapping thru 360 degress"
    (define (iter phi dp)
        (if (<= phi (+ 360 dp))
            (begin (view-angle theta phi chi)
		   (iter (+ phi dp) dp))))

    (set! must-clear-viewport #f)
    (iter (* dphi 2) dphi)

    (set! must-clear-viewport #t))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; AUTOPLOT - set the autoplot-flag

(define (autoplot . flag)
    (if flag
	(let* ((f (car flag)))
	  (set! autoplot-flag (cond ((boolean? f)
				     f)
				    ((integer? f)
				     (not (= f 0)))
				    (else
				     #t)))))
    autoplot-flag)

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; BUILD-ARRAY

(define-macro (build-array x)
    (build-array* x))

(define (build-array* x)
    (if x
      (let* ((data (read-pdbdata current-file x)))
	(if data
	    (let* ((vals (cddr (pdb->list data))))
	      (cond ((number? vals)
		     vals)
		    ((pm-array? vals)
		     (pm-array->list vals))
		    ((vector? vals)
		     (vector->list vals))))))))


;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; CD - change the current file directory

(define-macro (cd . rest)
    "CD - Change the current file directory.
     Usage: cd [<directory-name>]
     Examples: cd mydir
               cd /zoo/mammals
               cd ../reptiles
               cd"
    (if rest
	((io-function current-file "cd") current-file (car rest))
	((io-function current-file "cd") current-file)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; CHANGE - change values in the named variable or structure member

(define-macro (change name . val)
    "CHANGE - Reset values in a variable or member.
              If fewer values are supplied than elements referenced,
              the last value is repeated. If more values are supplied
              than elements referenced, the extra values are ignored.
              Note that the command keyword may be omitted.
     Usage: [change] <variable> | <structure-member> <values>
     Examples: change a[10,15] 3.2
               change time 0.0
               x[1:2] 1 2
               a[5,10,3] 5.7e4
               dir1/jkl.k 2"
  (change-aux name val))

; the change procedure
(define (change* name . val)
    "Procedure version of change macro"
    (change-aux name val))

; the auxiliary change procedure
(define (change-aux name val)
    (newline)
    (if (null? name)
	(printf nil "No variable name specified\n")
	(if (not (file-variable? current-file name #f)) ; yes, #f
	    (printf nil "Variable %s does not exist\n" name)
	    (let* ((syment (pdb->list (read-syment current-file name)))
		   (var-type (car syment))
		   (var-addr (cadr syment))
		   (var-dims (cddr syment))
		   (carval (car val)))
	      (define commlist nil)
	      (set! commlist (append (list current-file) commlist))
	      (set! commlist (append (list name) commlist))
	      (set! commlist (append (list (append (list type var-type) var-dims)) commlist))
	      (for-each
	       (lambda (x)
		 (set! commlist (append (list x) commlist)))
	       val)
	      (set! commlist (reverse commlist))
	      (show-pdb (apply write-pdbdata commlist)
			display-precision))))
    #f)

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; CHANGE-DIMENSION - change the dimensions of the named variable in memory
;                     

(define-macro (change-dimension name . rest)
    "CHANGE-DIMENSION - Change the dimensions of a variable in memory.

     Usage: change-dimension name dimension_list
     Examples: change-dimension foo 20
               change-dimension bar 10 10
               change-dimension foobar (2 . 10) (3 . 5)"

    (define (change-dim rest)
        (let* ((syment  (read-syment current-file name))
	       (symlist (pdb->list syment))
	       (address (cadr symlist))
	       (type    (car symlist))
	       (symout  (list address type name current-file)))
	  (if rest
	      (begin (set! symout (append (reverse rest) symout))
		     (set! symout (reverse symout))
		     (apply write-syment symout)
		     (file-mode current-file "r")))))
    (if rest
        (change-dim rest)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; CHANGE-FILE / CF / OPEN - change the current file

; the change-file macro

(define-macro (change-file file . arg)
    "CHANGE-FILE / CF - Change the current file to be the named one.
                 If file is in the internal list of open files,
                 make it current, otherwise, open it.
                 If mode is \"r\", open file read only.
                 If mode is \"a\", open file read/write.
                 If mode is \"w\", overwrite any existing file.
                 Mode defaults to \"a\".
                 Alias defaults to \"fd\", d is an increasing decimal number.
                 Type is the file type (options are determined by output
                 spokes you have installed).
     Usage: change-file <filename> [<mode> [<alias> [<type>]]]
     Examples: change-file foo
               change-file foo.s00 r
               change-file foo.s00 w
               change-file foo.s00 w foo
               change-file foo.s00 w bar pdb"
    (let* ((name (print-name file))
	   (mode  (if (and arg (> (length arg) 0))
		      (print-name (list-ref arg 0))
		      "a"))
	   (alias (if (and arg (> (length arg) 1))
		      (print-name (list-ref arg 1))
		      nil))
	   (type  (if (and arg (> (length arg) 2))
		      (print-name (list-ref arg 2))
		      "pdb"))
	   (fp    (file-manager "fopen" name mode type alias)))
      (set! current-file fp)
      fp))

; the change-file procedure

(define (change-file* file . arg)
    "Procedure version of change-file macro"
    (apply change-file (cons file arg)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; OPEN-TEXTFILE / OT - open a text file

; the open-textfile macro

(define-macro (open-textfile file . arg)
    "OPEN-TEXTFILE / OT - Open the text file.
     Usage: open-textfile <filename>
     Example: open-textfile foo"
    (let* ((name (print-name file))
	   (mode "w")
	   (fp   (open-text-file name mode)))
      fp))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; CLOSE-TEXTFILE / CT - close a text file

; the close-textfile macro

(define-macro (close-textfile . arg)
    "CLOSE-TEXTFILE / CT - Close the opened text file.
     Usage: close-textfile
     Example: close-textfile"
    (close-text-file)
    #t)

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; CHANGE-PALETTE

(define-macro (change-palette . pal)
    "CHANGE-PALETTE - Change the palette for the current window.
     Usage: change-palette palette
     Example: change-palette \"spectrum\""
    (window-manager "pchange" pal)
    (if autoplot-flag
	(window-manager "vupdate")))

(define (change-palette* . pal)
    "Procedure version of change-palette macro"
    (apply change-palette pal))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; CHANGE-VIEWPORT

(define (change-viewport . x)
    "CHANGE-VIEWPORT - Change current viewport.
     Usage: change-viewport [name [x y dx dy]]
     Examples: change-viewport
               change-viewport \"B\" 0.01 0.01 0.98 0.48"
    (window-manager "vopen" x))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; CHANGE-WINDOW

(define (change-window . info)
    "CHANGE-WINDOW - Change current window.
     Usage: change-window [title [mode type x y dx dy]]
     Examples: change-window
               change-window \"B\" \"COLOR\" \"WINDOW\" 0.5 0.1 0.4 0.4"
    (window-manager "wopen" info))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; CLOSE - close a file

(define-macro (close . file)
    "CLOSE - Close a file.
             The default is to close the current file.
     Usage: close [file]
     Examples: close
               close foo
               close f1"

    (cond ((null? file)
	   (if current-file
	       (set! current-file (file-manager "fclose" current-file))))
	  (else
	   (let* ((fp (car file)))
	     (if (and current-file (file-manager "fmatch" fp current-file))
		 (set! current-file (file-manager "fclose" current-file))
		 (file-manager "fclose" fp)))))
    #f)

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; CLOSE-VIEWPORT

(define (close-viewport . x)
    "CLOSE-VIEWPORT - Close a viewport.
     Usage: close-viewport [vname [wname]]"
    (window-manager "vclose" x))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; CLOSE-WINDOW / CLW

(define (close-window . window)
    "CLOSE-WINDOW  / CLW - Close a graphics window.
     Usage: close-window <window>
     Examples: close-window \"ABC\"
               clw"
    (window-manager "wclose" window))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; CMC - copy mapping as curve

(define-macro (cmc file . args)
    "CMC - Copy 1d mappings from the current file to another file
           as ULTRA curves. Mappings are referenced by the numbers
           displayed by the menu command. If the mapping list is *,
           copy all 1d mappings in the current directory.
     Usage: cmc file <mapping-list>
     Examples: cmc foo *
               cmc bar 1 5"

    (let* ((fp (stream-file (file-reference file stream-list))))

      (define (cmc_seq x)
	(let* ((type (pg-menu-item-type current-file x))
	       (mapping (if (eqv? type "image")
			    ((io-function current-file "image-in")
			     current-file x)
			    ((io-function current-file "map-in")
			     current-file x)))
	       (dims (if (or (not type) (eqv? type "image"))
			 nil
			 (pm-mapping-dimension mapping))))
	  (cond ((or (eqv? type "curve") (pg-graph? mapping))
		 (if (equal? dims mapping-1d-1d)
		     ((io-function fp "curve-out") mapping fp))
		 (cmc_seq (+ x 1)))
		((eqv? type "image")
		 (cmc_seq (+ x 1)))
		(else #t))))

      (define-macro (map_conv x)
	(let* ((type (pg-menu-item-type current-file x))
	       (mapping (if (eqv? type "image")
			    ((io-function current-file "image-in")
			     current-file x)
			    ((io-function current-file "map-in")
			     current-file x)))
	       (dims (if (or (not type) (eqv? type "image"))
			 nil
			 (pm-mapping-dimension mapping))))
	  (cond ((or (eqv? type "curve") (pg-graph? mapping))
		 (if (equal? dims mapping-1d-1d)
		     ((io-function fp "curve-out") mapping fp)
		     (printf nil "\nMapping %s has wrong dimensionality\n" x)))
		(else 
		 (printf nil "\n%s is not a valid mapping\n" x)))))

      (if args
	  (if (equal? (car args) '*)
	      (cmc_seq 1)
	      (for-each map_conv args)))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; COPY - copy variables and types from the current file to another file

(define-macro (copy file . vars)
     "COPY - Copy variables from the current file to another file.
            If variable list is *, copy all variables in current directory.
     Usage: copy file <variable-list>
     Examples: copy foo *
               copy f3 bird cat dog"

     (if vars
	 (let* ((fp (stream-file (file-reference file stream-list)))
		(in-types (reverse (list-defstrs current-file #f)))
		(the-vars (if (equal? (car vars) '*)
			      (if follow-directories
				  (list-symtab current-file)
				  (list-variables current-file))
			      vars)))

	   (set! follow-directories #f)

	   (define-macro (move-type x)
	     (write-defstr* fp (read-defstr* current-file x)))

	   (define-macro (move-data x)
	     (let* ((syment (read-syment current-file x))
		    (type (if syment (car (pdb->list syment)) nil)))
	       (if (equal? type "Directory")
		   (if (not (equal? x "/"))
		       (make-directory fp x directory-error-on-copy))
		   (let* ((data (read-pdbdata current-file x)))
		     (if (null? data)
			 (printf nil "Variable %s not found\n" x)
			 (write-pdbdata fp x data))))))

	   (if in-types
	       (begin
		 (if (equal? (file-type fp) "PDBfile")
		     (pg-def-graph-file fp))
		 (for-each move-type in-types)
		 (if the-vars
		     (for-each move-data the-vars)))))))

(define (copy* . args)
    "Procedure version of copy macro"
    (apply copy args))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; COPY-DIR - copy the contents of the specified directory from the
;          - source file to the destination file

(define (copy-dir dir sf df)
    "COPY-DIR - Copy the contents of the specified directory from the
           source file to the destination file. The directory name will
           be the same in both files.
     Usage: copy-dir <dir> <src-file> <dst-file>
     Example: copy-dir \"/a/b\" f0 f1"
    (change-directory (change-file* df) dir)
    (change-directory (change-file* sf) dir)
    (copy* df '*))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; COPY-MAPPING / CM - copy mappings from the current file to another file

(define-macro (copy-mapping file . args)
    "COPY-MAPPING / CM - Copy mappings from the current file to another file.
                  Mappings are referenced by the numbers displayed
                  by the menu command. If the mapping list is *,
                  copy all mappings in the current directory.
     Usage: copy-mapping file <mapping-list>
     Examples: copy-mapping foo *
               copy-mapping bar 1 5"

    (if args
	(let* ((fp (stream-file (file-reference file stream-list))))

	  (define (cm_seq x)
	    (let* ((type (pg-menu-item-type current-file x))
		   (item (if (eqv? type "image")
			     ((io-function current-file "image-in")
			      current-file x)
			     ((io-function current-file "map-in")
			      current-file x))))
	      (cond ((eqv? type "curve")
		     ((io-function fp "curve-out") item fp)
		     (cm_seq (+ x 1)))
		    ((pg-graph? item)
		     ((io-function fp "map-out") item fp)
		     (cm_seq (+ x 1)))
		    ((pg-image? item)
		     ((io-function fp "image-out") item fp)
		     (cm_seq (+ x 1)))
		    (else #t))))

	  (define-macro (map_conv x)
	    (let* ((type (pg-menu-item-type current-file x))
		   (item (if (eqv? type "image")
			     ((io-function current-file "image-in")
			      current-file x)
			     ((io-function current-file "map-in")
			      current-file x))))
	      (cond ((eqv? type "curve")
		     ((io-function fp "curve-out") item fp))
		    ((pg-graph? item)
		     ((io-function fp "map-out") item fp))
		    ((pg-image? item)
		     ((io-function fp "image-out") item fp))
		    (else 
		     (printf nil "\n%s is not a valid mapping\n" x)))))

	  (if (equal? (file-type fp) "PDBfile")
	      (pg-def-graph-file fp))

	  (if (equal? (car args) '*)
	      (cm_seq 1)
	      (for-each map_conv args)))))
;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; CREATE-PALETTE - interactively create a new palette

(define-macro (create-palette name . rest)
    "CREATE-PALETTE - Create a new palette.  See the SAVE-PALETTE
                      command for saving to disk.
     Usage: create-palette name [<ndims> <nx> <ny>]
     Examples: create-palette new_palette
               create-palette new_palette 2 24 8"
     (if (not (defined? current-window))
         (change-window "A"))
     (if (null? rest)
         (pg-make-palette (cadr current-window) name)
         (let* ((ndims (car rest))
		(nx    (cadr rest))
		(ny    (if (not (null? (cddr rest)))
			   (caddr rest)
			   1)))
	   (if (< 194 (* nx ny))
	       (printf nil "There are only 194 colors available.\n")
	       (pg-make-palette (cadr current-window) name  1 ndims nx ny))))
     (if autoplot-flag
         (window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; DATA-ID - turn on/off display of data-id

(define (data-id . flag)
    "DATA-ID - Turn on/off the plotting of curve markers.
     Usage: data-id [<on> | <off>]"
    (if (not (defined? current-window))
        (change-window "A"))
    (if flag
        (let* ((f (car flag)))
	  (pg-set-data-id-flag! (cadr current-window) (cond ((boolean? f)
							     0)
							    ((integer? f)
							     f)
							    (else
							     1))))
        (pg-set-data-id-flag! (cadr current-window) 1))
    (if autoplot-flag
        (window-manager "vupdate"))
     #t)

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; DATA-REFERENCE - refer to the nth mapping in the specified file

(define-macro (data-reference indx . rest)
    "DATA-REFERENCE - Refer to the nth mapping in the optionally 
                      specified file. (Macro)
     Usage: data-reference <n> [<file-name>|<file-alias>|<file-type>]"
    (let* ((i (eval indx)))
      (if rest
	  (let* ((file (file-manager "fnamed" (car rest))))
	    (mapping-ref file "mapping" i))
	  (mapping-ref current-file "mapping" i))))

(define (data-reference* indx . rest)
    "Procedure version of data-reference macro"
    (if rest
	(let* ((file (file-manager "fnamed" (car rest))))
	    (mapping-ref file "mapping" indx))
	(mapping-ref current-file "mapping" indx)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; DEF - install object in the workspace

(define-macro (def name var)
    "DEF - Install an object in the workspace under the given name.
     Usage: def <name> <object>
     Example: def foo m3"

    (let* ((ev (apply define-global (list name var))))
      (cond ((pdbdata? ev)
	     name)
	    ((pm-array? ev)
	     (pm-array->pdbdata ev nil name))
	    ((pm-set? ev)
	     (pm-set->pdbdata ev nil name))
	    ((pm-mapping? ev)
	     (pm-mapping->pdbdata ev nil name))
	    ((pg-graph? ev)
	     (pg-graph->pdbdata ev nil name))
	    ((pg-image? ev)
	     (pg-image->pdbdata ev nil name))
	    ((hash-table? ev)
	     (hash->pdbdata ev nil name))
	    (else
	     (printf nil
		     "Object not suitable for workspace: %s\n"
		     ev)))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; DEFAULT-VIEWPORT-RENDERING

(define (default-viewport-rendering . x)
    "DEFAULT-VIEWPORT-RENDERING - Specify a default rendering mode for a viewport.
     Usage: default-viewport-rendering [<viewport> [<window>]] <info>
            for 1d domain - 1d range mappings
            <info> - cartesian | polar | insel
            for 2d domain - 1d range mappings
            <info> - contour [<n-levels>] |
                     image |
                     fill-poly |
                     shaded
                     wire-frame
                     mesh
            for 2d domain - 2d range mappings
            <info> - vector
                     shaded
                     mesh
            for 3d domain
            <info> - mesh
                     shaded
     Examples: default-viewport-rendering \"A\" \"A\" polar
               default-viewport-rendering contour
               default-viewport-rendering contour 15
               default-viewport-rendering image
               default-viewport-rendering fill-poly
               default-viewport-rendering wire-frame
               default-viewport-rendering shaded
               default-viewport-rendering vector"
    (window-manager "vrender" x)
    (if (and autoplot-flag (not overlay-flag))
	(window-manager "vupdate" x)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; DEFINE-FILE-VARIABLE

(define-macro (define-file-variable type name . vals)
    "DEFINE-FILE-VARIABLE - Create a new variable in a PDB file
     Usage: define-file-variable type name [values]*
     Examples: define-file-variable double foo 3.42
               define-file-variable \"char *\" bar \"Hi\" \"there\"
               define-file-variable \"int\" baz 1"
    (defv-aux type name vals))

(define (define-file-variable* type name . vals)
    "Procedure version of define-file-variable macro"
    (defv-aux type name vals))

(define (defv-aux type name vals)
    (if vals
	(let* ((carvals (car vals))
	       (values (if (pair? carvals) carvals vals))
	       (decl (list 'type type (length values)))
	       (args (append (list current-file name decl) values)))
	  (apply write-pdbdata args))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; DELETE-MAPPING

(define (delete-mapping . x)
    "DELETE-MAPPING - Delete mappings from a viewport.
     Usage: delete-mapping [vname [wname]] <lsv-mapping-numbers>
     Examples: delete-mapping 2 4 5"
    (window-manager "vdelete" x)
    (if autoplot-flag
	(window-manager "vupdate" x)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; APPLY-OPERATOR-MAPPING

(define (apply-operator-mapping oper . x)
    "APPLY-OPERATOR-MAPPING - apply the given operator to
     mappings from a viewport.
     Usage: apply-operator-mapping [vname [wname]] <lsv-mapping-numbers>
     Examples: apply-operator-mapping 2 4 5"
    (window-manager "vapply" oper x)
    (if autoplot-flag
	(window-manager "vupdate" x)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; DESC - describe variables or structure members in current file directory

(define-macro (desc . rest)
    "DESC - Describe variables or struct members in current file directory.
            To describe part of a variable or member qualify the name with
            index expressions whose parts are in one of the three forms:
               <index>
               <index-min>:<index-max>
               <index-min>:<index-max>:<increment>
            Only the first form may be used with non-terminal members or
            with variables and terminal members with embedded pointers.
            If a selection pattern is specified in place of a variable
            name, all variables matching the pattern will be described.
            An \"*\" matches any zero or more characters and \"?\" matches
            any single character. A type qualifier may also be specified.
     Usage: desc <variable> | <structure-member> | <pattern> [<type>]
     Examples: desc Mapping1
               desc a.b.c[12:14]
               desc * double
               desc var? integer"

    (let* ((fnc (io-function current-file "desc"))
	   (name (if (pair? rest) (car rest) nil))
	   (type (if (pair? rest)
		     (if (pair? (cdr rest)) (cadr rest) nil)
		     nil))
	   (tn (symbol->string name))
	   (ln (string-length tn))
	   (tn (if (string=? "/" (substring tn (- ln 1) ln))
		   tn
		   (string-append tn "/")))
	   (isdir (memv tn (list-variables current-file)))
	   (vars (if type		;12/23/97: revised - see below
		     (list-variables current-file name type)
		     (if isdir
			 (list tn)
			 (if name
			     (list-variables current-file name)
			     (list-variables current-file))))))
;
; 12/23/97: this was the original coding replaced by above coding
;
;	     (vars (if type
;		       (list-variables current-file name type)
;		       (if name
;			   (list-variables current-file name)
;			   (list-variables current-file)))))

      (define (desc-one var)
	(fnc current-file var))

      (if vars
	  (for-each desc-one vars)
	  (if (and name (file-variable? current-file name #t))
	      (if type
		  (if (eqv? (print-name type) (variable-type name))
		      (desc-one name))
		  (desc-one name))))))

(define (desc* . args)
    "Procedure version of desc macro"
    (apply desc args))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; DISPLAY-DOMAIN

(define-macro (display-domain . x)
    "DISPLAY-DOMAIN - Plot a domain mesh in a viewport.
     Usage: display-domain [<viewport> [<window>]] <domain list>
     Example: display-domain f1.18"

; bag of cruft to make three cases work: 1, Mapping0, and f1.1
; formally: <integer>, <variable-name>, and (data-reference ...)
    (define-macro (prep y)
        (pm-mapping-domain
	 (cond ((pair? y)
		(eval (cadr y)))
	       ((integer? y)
		(data-reference* y))
	       ((printable? y)
		(if (eqv? (strtok (variable-type y) " *") "PM_set")
		    (pdbdata->pm-set current-file y)
		    (pdbdata->pg-graph current-file y)))
	       (else
		y))))

    (if (window-manager "vattach" (map prep x))
	(if autoplot-flag
	    (window-manager "vupdate" x))))

; DISPLAY-DOMAIN

(define (display-domain* . x)
    "DATA-REFERENCE* - Refer to the nth mapping in the optionally 
                       specified file. (Procedure)
     Usage: data-reference* <n> [<file-name>|<file-alias>|<file-type>]"
    "DISPLAY-DOMAIN - Plot a domain mesh in a viewport.
     Usage: display-domain [<viewport> [<window>]] <domain list>
     Example: display-domain* f1.18"
    (if (window-manager "vattach"
			(map (lambda (y)
				 (pm-mapping-domain (data-reference* y)))
			     x))
	(if autoplot-flag
	    (window-manager "vupdate" x))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; DISPLAY-MAPPING / DM / PL

(define-macro (display-mapping . x)
    "DISPLAY-MAPPING  / DM / PL - Plot the specified mappings.
     Usage: display-mapping [<viewport> [<window>]] <mapping list>
     Examples: display-mapping 5
               dm dir1/Mapping8
               pl 5 8"

; bag of cruft to make three cases work: 1, Mapping0, and f1.1
; formally: <integer>, <variable-name>, and (data-reference ...)
    (define-macro (prep y)
        (if (pair? y)
            (eval (cadr y))
	    y))

    (if (window-manager "vattach" (map prep x))
	(if autoplot-flag
	    (window-manager "vupdate" x))))

(define (display-mapping* . x)
    "Procedure version of display-mapping macro"
    (if (window-manager "vattach" x)
	(if autoplot-flag
	    (window-manager "vupdate" x))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; DRAWABLE-RENDERING

(define-macro (drawable-rendering dr . info)
    "DRAWABLE-RENDERING - Specify the rendering mode for a drawable
     Usage: drawable-rendering <drawable> <info>
            <info> - [<spec>]*
            <spec> - (render <render>) |
                   - (view-angle <theta> <phi> <chi>) |
                   - (change-palette <palette>)
            for 1d domain - 1d range mappings
            <render> - cartesian | polar | insel
            for 2d domain - 1d range mappings
            <render> - contour [<n-levels>] |
                       image |
                       fill-poly |
                       shaded
                       wire-frame
                       mesh
            for 2d domain - 2d range mappings
            <render> - vector
                       shaded
            for 3d domain
            <render> - mesh
                       shaded
     Examples: drawable-rendering 3 (render polar)
               drawable-rendering 1 (render contour)
               drawable-rendering f1.1 (render contour 15)
               drawable-rendering f1.12 (render image) (change-palette spectrum)
               drawable-rendering 18 (render fill-poly)
               drawable-rendering 4 (render wire-frame) (view-angle 60 45 0)
               drawable-rendering 1 (render shaded) (change-palette cyans)
               drawable-rendering 1 (render vector)"
    (window-manager "drender" (list dr info))
    (if autoplot-flag
	(window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; DOMM

(define (domm . args)
    "DOMM - Select plotting limits for the domain of a mapping.
            If invoked with no limit values the domain
            will default to that implied by the data set.
     Usage: domm [<viewport> [<window>]] <mapping> <x1_min> <x1_max> ...
     Examples: domm 1 -5.0 10.0 10 20
               domm 1"
    (window-manager "mdlimit" args)
    (if autoplot-flag
	(window-manager "vupdate" args)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; DOMV

(define (domv . args)
    "DOMV - Select plotting limits for the domain of a viewport.
            If invoked with no limit values the domain
            will default to that implied by the data sets.
     Usage: domv [<viewport> [<window>]] <x1_min> <x1_max> ...
     Examples: domv -5.0 10.0 10 20
               domv"
    (window-manager "vdlimit" args)
    (if autoplot-flag
	(window-manager "vupdate" args)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; END / QUIT - end PDBView

(define-macro (end . args)
    "END / QUIT - End the session of PDBView.
     Usage: end"

    (if stream-list
	(for-each
	 (lambda (x)
	     (set! current-file (stream-file x))
	     ((io-function current-file "close-file") current-file))
	 stream-list))

    (if (and (defined? PS-device) PS-device)
        (pg-close-device PS-device))
    
    (if (and (defined? CGM-device) CGM-device)
        (pg-close-device CGM-device))
    
    (printf nil "\n")
    (terminate))

(define terminate quit)
(define quit end)

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; FILE - describe the current file

(define-macro (file . rest)
    "FILE - Describe the current file.
            Options: t or type displays the file type only
     Usage: file [t | type]
     Examples: file
               file type"
    (begin
        (if (null? rest)
	    (show-pdb current-file)
	    (let ((opt (car rest)))
	         (cond ((or (eqv? opt 't) (eqv? opt 'type))
			(printf nil "\nFile type: ")
			(if current-file
			    (display (file-type current-file))
			    (display "SX_vif"))
			(newline))
		       (#t (show-pdb current-file)))))
	#t))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; FONT - set/describe the current font of the current window

(define (font . args)
    "FONT - set/get the current font information.
     Usage: font [<face> <point-size> <style>]
            <face> - helvetica | times | courier
            <point-size> - 8 | 10 | 12 | 14 ...
            <style> - medium | bold | italic
     Examples: font
               font helvetica 12 medium"
    (window-manager "vfont" current-window args))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; FORMAT - set the printing format for a specified data type

(define-macro (format type format-string)
    "FORMAT - Set the printing format for a specified data type.
              If the type argument has 1 or 2 appended, then the
              specified format is applied only to arrays shorter
              than array-length or to arrays longer than or equal
              to array-length, respectively. Otherwise, the format
              applies to both. Invoking the format command with
              the single argument, default, causes the formats for
              all types to be reset to their defaults. The format
              argument must be a standard C I/O library format
              string. Double quotes are only necessary if the
              format string contains embedded blanks. See the set
              command for more about the array-length variable.
              This command overrides the settings of the decimal-
              precision and bits-precision control parameters.
     Usage: format integer[1 | 2] | long[1 | 2] | float[1 | 2] |
                   double[1 | 2] | short[1 | 2] | char[1 | 2] <format>
     Usage: format default
     Examples: format double %12.5e
               format double2 %10.2e
               format char \"%s  \"
               format default"
    (if (null? type)
        (printf nil "\nNo type specified\n")
        (if (eqv? type 'default)
            (set-format 'default)
            (if (null? format-string)
                (printf nil "\nNo format string specified\n")
                (if (not (memv type type-list))
                    (printf nil "\n%s is not a valid type specifier\n" type)
                    (set-format type format-string))))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; HARDCOPY-VIEWPORT - send the current graph to current hard copy devices

(define-macro (hardcopy-viewport . rest)
    "HARDCOPY-VIEWPORT - Send the current graph to all open hardcopy devices.
                         Color output is indicated by an optional argument on
                         the first call for a given device.
                         The optional resolution scale factor is an integer
                         factor by which the resolution will be decreased
                         below the full resolution of the device.
     Usage: hardcopy-viewport [color | monochrome] [portrait | landscape] [resolution-scale-factor]]
     Examples: hardcopy-viewport
               hardcopy-viewport color
               hardcopy-viewport color 8"
    (interactive off)
    (let* ((psflag (ps-flag))
	   (cgmflag (cgm-flag))
	   (jpegflag (if (jpeglib_flag) (jpeg-flag) 0))
	   (args (process-hc-args rest))
	   (color (list-ref args 0))
	   (mode (list-ref args 1))
	   (res (list-ref args 2))
	   (type (sprintf "%s %s" color mode)))

      (if (= jpegflag 1)
	  (begin
	    (set! JPEG-device 
		  (pg-make-device "JPEG"
				  type))
	    (pg-open-device JPEG-device 0.0 0.0 400.0 400.0)
	    (pg-set-palette! JPEG-device (pg-current-palette WIN-device))
	    (window-manager "vhardcopy" JPEG-device res)
	    (pg-close-device JPEG-device)
	    (set! JPEG-device #f)))
      (if (= psflag 1)
	  (begin
	    (if (not PS-device)
		(begin
		  (set! PS-device
			(pg-make-device "PS"
					type))
		  (pg-open-device PS-device 0.0 0.0 0.0 0.0)))
	    (pg-set-palette! PS-device (pg-current-palette WIN-device))
	    (window-manager "vhardcopy" PS-device res)))
      (if (= cgmflag 1)
	  (begin
	    (if (not CGM-device)
		(begin
		  (set! CGM-device
			(pg-make-device "CGM"
					color))
		  (pg-open-device CGM-device 0.0 0.0 0.0 0.0)))
	    (pg-set-palette! CGM-device (pg-current-palette WIN-device))
	    (window-manager "vhardcopy" CGM-device res)))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; PROCESS-HC-ARGS - process the arguments to an HC command and make
;                 - a uniform return list of values containing
;                 - (color mode res)

(define (process-hc-args args)
   (if args
       (let* ((color (memv 'color args))
	      (land (memv 'landscape args))
	      (res (list-ref args (- (length args) 1))))
	 (list (if color "COLOR" "MONOCHROME")
	       (if land  "LANDSCAPE" "PORTRAIT")
	       (if (number? res) res 0)))
       (list "MONOCHROME" "PORTRAIT" 0)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; HARDCOPY-WINDOW / HC / HCW - draw the contents of the current window to
;                 to all open hardcopy devices

(define-macro (hardcopy-window . rest)
    "HARDCOPY-WINDOW - Draw the current window to all open hardcopy devices.
                       Color output is indicated by an optional argument.
                       Landscape mode is indicated by an optional argument.
                       The optional resolution scale factor is an integer
                       factor by which the resolution will be decreased
                       below the full resolution of the device (this
                       argument must appear last in the argument list).
     Usage: hardcopy-window [color | monochrome] [portrait | landscape] [resolution-scale-factor]
     Examples: hardcopy-window
               hc color 8"
    (interactive off)
    (let* ((psflag (ps-flag))
	   (cgmflag (cgm-flag))
	   (jpegflag (if (jpeglib_flag) (jpeg-flag) 0))
	   (args (process-hc-args rest))
	   (color (list-ref args 0))
	   (mode (list-ref args 1))
	   (res (list-ref args 2))
	   (type (sprintf "%s %s" color mode)))

      (if (= jpegflag 1)
	  (begin
	    (set! JPEG-device
		  (pg-make-device "JPEG"
				  type))
	    (pg-open-device JPEG-device 0.0 0.0 400.0 400.0)
	    (pg-set-palette! JPEG-device (pg-current-palette WIN-device))  
	    (window-manager "whardcopy"
			    JPEG-device res current-window)
	    (pg-close-device JPEG-device)
	    (set! JPEG-device #f)))
      (if (= psflag 1)
	  (begin
	    (if (not PS-device)
		(begin
		  (set! PS-device
			(pg-make-device "PS"
					type))
		  (pg-open-device PS-device 0.0 0.0 0.0 0.0)))
	    (pg-set-palette! PS-device (pg-current-palette WIN-device))
	    (window-manager "whardcopy"
			    PS-device res current-window)))
      (if (= cgmflag 1)
	  (begin
	    (if (not CGM-device)
		(begin
		  (set! CGM-device
			(pg-make-device "CGM"
					color))
		  (pg-open-device CGM-device 0.0 0.0 0.0 0.0)))
	    (pg-set-palette! CGM-device (pg-current-palette WIN-device))
	    (window-manager "whardcopy"
			    CGM-device res current-window)))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; HELP - print command list or command documentation

(define-macro (help . rest)
    "HELP - Print command list or command documentation.
     Usage: help [<command>]
     Examples: help
               help menu"
    (newline)
    (if rest
	(for-each describe rest)
	(begin
	  (printf nil "Commands: type help <command> for more information\n")
	  (newline)
	  (printf nil "AC               - Make an arbitrarily connected direct mapping\n")
          (printf nil "ANIMATE          - Rotate the specified mapping thru 360 degrees\n")
	  (printf nil "AV               - Apply operator to mappings in a viewport\n")
	  (printf nil "CD               - Change the current file directory\n")
	  (printf nil "CF               - Change the current file\n")
	  (printf nil "CHANGE           - Reset a single value in a variable or member\n")
	  (printf nil "CHANGE-DIMENSION - Change the dimensionality of a variable in memory\n");
	  (printf nil "CHANGE-PALETTE   - Change the palette for the current window\n")
	  (printf nil "CLOSE            - Close a file\n")
	  (printf nil "CLOSE-TEXTFILE   - Close a text-file.  Can use CT\n")
	  (printf nil "CLV              - Close a viewport\n")
	  (printf nil "CLW              - Close the specified windows\n")
	  (printf nil "CM               - Copy mappings from current file to specified file\n")
          (printf nil "COMMAND-LOG      - Control command logging\n")
	  (printf nil "COPY             - Copy variables from current file to specified file\n")
          (printf nil "CREATE-PALETTE   - Interactively create a new palette\n")
	  (printf nil "CV               - Change the current viewport\n")
	  (printf nil "CW               - Change the current window\n")
          (printf nil "DATA-ID          - Control plotting of curve markers\n")
          (printf nil "DOMM             - Set mapping domain plot limits\n")
          (printf nil "DOMV             - Set viewport domain plot limits\n")
	  (printf nil "DREF             - Refer to the nth mapping in a file\n")
	  (printf nil "DEF              - Install an object in the workspace\n")
	  (printf nil "DEFV             - Define a new file variable\n")
	  (printf nil "DESC             - Describe the named variable or structure member\n")
	  (printf nil "DL               - Delete mappings from a viewport\n")
	  (printf nil "END              - End the session of PDBView\n")
	  (printf nil "FILE             - Describe the current file\n")
	  (printf nil "FONT             - Set/Display the font of the current window\n")
	  (printf nil "FORMAT           - Set the printing format for a specified data type\n")
          (printf nil "HCV              - Send contents of current viewport to open hard copy devices\n")
          (printf nil "HCW              - Send contents of current window to open hard copy devices\n")
	  (printf nil "HELP             - Print command list or command documentation\n")
          (printf nil "HISTOGRAM        - Display given mappings from current viewport as histograms\n")
	  (printf nil "LD               - Load Scheme forms from a file\n")
	  (printf nil "LEVELS           - Set the contour levels\n")
	  (printf nil "LIST-FILES       - List the open files\n")
          (printf nil "LIGHT-SOURCE     - Set the theta and phi angles for the light source\n")
	  (printf nil "LIST-PALETTES    - List the available palettes\n")
          (printf nil "LNCOLOR          - Set the line color\n")
          (printf nil "LNSTYLE          - Set the line style\n")
          (printf nil "LNWIDTH          - Set the line width\n")
          (printf nil "LOAD-PALETTE     - Read a palette from a file and make it the current palette\n")
          (printf nil "LOGICAL          - Display given mappings from current viewport versus index\n")
	  (printf nil "LR               - Make a logically rectangular direct mapping\n")
	  (printf nil "LRS              - Make a logically rectangular mapping by synthesis\n")
	  (printf nil "LS               - List the variables in a file directory\n")
	  (printf nil "LS-ATTR          - List the attributes in the current file\n")
	  (printf nil "LST              - List the mappings in the work space\n")
	  (printf nil "LSV              - List the mappings in a viewport\n")
          (printf nil "MARKER-SCALE     - Set the marker scale for the specified mappings\n")
	  (printf nil "MENU             - List the labels of mappings in the current directory\n")
	  (printf nil "MODE             - Set the print mode for structures\n")
	  (printf nil "N-ENTRIES        - Print the number of symbol table entries in current file\n")
	  (printf nil "OPEN-TEXTFILE    - Open a text-file.  Can use OT\n")
	  (printf nil "PL               - Display the specified mappings\n")
	  (printf nil "PLD              - Display the domains of the specified mappings\n")
	  (printf nil "PLOT             - Plot one variable against another\n")
	  (printf nil "PM               - Print the specified mappings\n")
	  (printf nil "PRINT            - Print out all or part of a variable or member\n")
	  (printf nil "PWD              - Print the current file directory\n")
          (printf nil "RANM             - Set mapping range plot limits\n")
          (printf nil "RANV             - Set viewport range plot limits\n")
          (printf nil "REFMESH          - Display a reference mesh\n")
          (printf nil "REFMESHCOLOR     - Set the line color for the reference mesh\n")
          (printf nil "SAVE-PALETTE     - Write a pgs palette to a disk file\n")
          (printf nil "SCATTER          - Display given mappings from current viewport as scatter plots\n")
	  (printf nil "SET              - Set the value of a control parameter\n")
	  (printf nil "SHOW-PALETTES    - Show available palettes and make selection current\n")
	  (printf nil "STRUCT           - Describe the named data type\n")
	  (printf nil "SYNONYM          - Define synonyms for the given command\n")
	  (printf nil "TABLE            - Write the specified variable to a textfile.  Can use TAB\n")
	  (printf nil "TYPES            - List the types in the current file\n")
          (printf nil "SET-VECTOR-ATT   - Set vector attributes\n")
          (printf nil "VIEW-ANGLE       - Set the three Euler viewing angles\n")
	  (printf nil "VR               - Set the default rendering mode for a viewport\n")
	  (printf nil "WU               - Update all viewports in the specified windows\n"))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; HISTOGRAM

(define (histogram start . things)
    "HISTOGRAM - Display given mappings from current viewport as histograms
     Usage: histogram on | off | left | right | center [<integer>]+"
    (if (> start 0)
	(begin (window-manager "vsattr" "PLOT-TYPE"
			       "int *" histogram-plot things)
	       (window-manager "vsattr" "HIST-START"
			       "int *" (- start 1) things))
	(window-manager "vsattr" "PLOT-TYPE"
			"int *" cartesian-plot things))
    (if autoplot-flag
	(window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; LABEL-DRAWABLE

(define (label-drawable . x)
    "LABEL-DRAWABLE - label a drawable.
     Usage: label-drawable [vname [wname]] <lsv-mapping-number>
     Examples: label-drawable 2"
    (window-manager "vlabel" x)
    (if autoplot-flag
	(window-manager "vupdate" x)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; VIEWPORT-AREA

(define (viewport-area . box)
    "VIEWPORT-AREA - set drawing area, in NDC, of the viewport
     Usage: viewport-area x-min x-max y-min ymax"
    (if box
        (let* ((f (car box)))
	  (if (or (boolean? f) (integer? f))
	      (set! viewport-area-box nil)
	      (begin
		(set! viewport-area-box box)
		(window-manager "vsattr" "VIEW-PORT" "double *" box nil)))))
	  
    (if autoplot-flag
	(window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; VIEWPORT-AREA

;(define (viewport-area . box)
;    "VIEWPORT-AREA - set drawing area, in NDC, of the viewport
;     Usage: viewport-area x-min x-max y-min ymax"
;    (if box
;	(begin (set! viewport-area-box box)
;               (window-manager "vsattr" "VIEW-PORT" "double *" box nil)))
;	  
;    (if autoplot-flag
;	(window-manager "vupdate")))
;

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; LEVELS

(define (levels . clevs)
    "LEVELS - set the contour levels
     Usage: levels [<clev>]*"
    (if clevs
	(begin
	  (window-manager "vsattr" "N-LEVELS" "int *" (length clevs) nil)
	  (window-manager "vsattr" "LEVELS" "double *" clevs nil))
	(begin
	  (window-manager "vsattr" "N-LEVELS")
	  (window-manager "vsattr" "LEVELS")))
	  
    (if autoplot-flag
	(window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; LIGHT-SOURCE

(define (light-source theta phi . things)
    "LIGHT-SOURCE - set the theta and phi angles for the light source
     Usage: light-source <theta> <phi> <mapping>"
    (window-manager "vsattr" "THETA-LIGHT" "double *" theta things)
    (window-manager "vsattr" "PHI-LIGHT" "double *" phi things)
    (if autoplot-flag
	(window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; LIST-FILES

(define (list-files . x)
    "LIST-FILES - List the open files.
     Usage: list-files
     Examples: list-files
               list-files"
    (file-manager "files"))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; LIST-PALETTES

(define (list-palettes . window)
    "LIST-PALETTES - List the available palettes.
     Usage: list-palettes [window]
     Examples: list-palettes \"A\"
               list-palettes"
    (let* ((pals (window-manager "plist" window)))
      (if pals
	  (begin
	    (newline)
	    (display pals)
	    (newline)))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; LNCOLOR

(define (lncolor clr . things)
    "LNCOLOR - set the line color used in drawing the specified
     Usage: lncolor <color> [<integer>]+"
    (window-manager "vsattr" "LINE-COLOR" "int *" clr things)
    (if autoplot-flag
	(window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; LNSTYLE

(define (lnstyle sty . things)
    "LNSTYLE - set the line style used in drawing the specified
     Usage: lnstyle <style> [<integer>]+"
    (window-manager "vsattr" "LINE-STYLE" "int *" sty things)
    (if autoplot-flag
	(window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; LNWIDTH

(define (lnwidth wid . things)
    "LNWIDTH - set the line width used in drawing the specified
     Usage: lnwidth <width> [<integer>]+"
    (window-manager "vsattr" "LINE-WIDTH" "double *" wid things)
    (if autoplot-flag
	(window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; LOAD-PALETTE - read a palette from a file and make it the current
;                palette

(define-macro (load-palette . name)
    "LOAD-PALETTE - Load a palette from a file.
     Usage: load-palette fname
     Example: load-palette foo.pal"
     (if (not (defined? current-window))
         (change-window "A"))
     (if (null? name)
         (printf nil "No file name specified\n")
         (pg-read-palette (cadr current-window) (car name)))
     (if autoplot-flag
         (window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; LOGICAL

(define (logical flag . things)
    "LOGICAL - Display given mappings from current viewport versus index
     Usage: logical on | off [<integer>]+"
    (if (> flag 0)
	(window-manager "vsattr" "RENDERING-TYPE"
			"int *" logical-plot things)
	(window-manager "vsattr" "RENDERING-TYPE"
			"int *" cartesian-plot things))
    (if autoplot-flag
	(window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; LS - list the variables in the current file directory

(define-macro (ls . rest)
    "LS - List variables, links, and directories in current file directory.
          A selection pattern may be specified for the terminal pathname
          node. A type qualifier may also be specified.
     Usage: ls [<pattern> [<type>]] 
     Examples: ls
               ls curve* char
               ls var?
               ls * Directory
               ls ../mydir
               ls /foo/bar double"
    (let* ((ls-fnc (io-function current-file "ls")))
      (if (pair? rest)
	  (if (pair? (cdr rest))
	      (ls-fnc current-file (car rest) (cadr rest))
	      (ls-fnc current-file (car rest)))
	  (ls-fnc current-file))
      #t))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; LS-ATTR - list the attributes in the current file

(define-macro (ls-attr . rest)
    "LS-ATTR - List the attributes in the current file.
     Usage: ls-attr"
    (if (null? rest)
	(ls-attr-pdb current-file)
	(ls-attr-pdb current-file (car rest)))
    #t)

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; LST - list the mappings in the work space

(define-macro (lst . rest)
    "LST - List the labels of mappings in the work space.
           A selection pattern may be specified.
     Usage: lst [<pattern>]
     Examples: lst
               lst ?*d
               lst *foo*"
    (apply display-menu (cons nil rest)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; LIST-MAPPINGS-IN-VIEWPORT

(define (list-mappings-in-viewport . info)
    "LIST-MAPPINGS-IN-VIEWPORT - List the mappings or images associated with a viewport.
     Usage: list-mappings-in-viewport [vname [wname]]"
    (window-manager "vlist" info))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; MAKE-AC-MAPPING-DIRECT - make a direct mapping

(define (make-ac-mapping-direct ran dom . rest)
   (if rest
       (let* ((cent (list-ref rest 0))
	      (labl (list-ref rest 1)))
	 (data-manager "acdirect" ran dom cent labl))
       (data-manager "acdirect" ran dom nil nil)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; MAKE-LR-MAPPING-DIRECT - make a direct mapping

(define (make-lr-mapping-direct ran dom . rest)
    (if rest
	(let* ((cent (list-ref rest 0))
	       (emap (list-ref rest 1))
	       (labl (list-ref rest 2)))
	  (data-manager "lrdirect" ran dom cent emap labl))
	(data-manager "lrdirect" ran dom nil nil nil)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; MAKE-LR-MAPPING-SYNTH - make a direct mapping by synthesis

(define (make-lr-mapping-synth ran dom . rest)
   (data-manager "lrproduct" ran dom (if rest (car rest) rest)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; ARRAY-ND-SET - make a set with the specified dimensions out of
;              - a linear array

(define-macro (array-nd-set var . dims)
   "ARRAY-ND-SET - Construct a set from an array which has specific
                   dimensions.
    Usage: array-nd-set var [<dim>]*
    Example: array-nd-set x 10 20"
   (data-manager "ndset" var dims))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; MAKE-CP-SET - make a cartesian product set

(define-macro (make-cp-set . rest)
   "MAKE-CP-SET - Construct a set which is the Cartesian product of
                  a list of sets or arrays.  Unary operations can
                  optionally be applied to the components.
    Usage: make-cp-set [<item>]*
           item := <var> | <oper> <var>
    Example: make-cp-set x y z
             make-cp-set log10 x y exp z"
   (data-manager "cpset" rest))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; MENU - list the images and mappings in the current file

(define-macro (menu . rest)
    "MENU - List the labels of mappings in the current file directory.
            A selection pattern may be specified.
     Usage: menu [<pattern>]
     Examples: menu
               menu ?*d
               menu *foo*"
    (apply display-menu (cons current-file rest)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; MINMAX - Print the minimum value, maximum value, and their offsets

(define-macro (minmax . rest)
  (if (printable? (car rest))
      (let* ((data (pdb-read-numeric-data current-file (car rest)))
	     (extr (pm-array-extrema data)))
	(printf nil "\nmin: %12.5e  imin: %s\nmax: %12.5e  imax: %s\n"
		(car extr) (caddr extr) (cadr extr) (cadddr extr))
	extr)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; MKDIR - create a new file directory

(define-macro (mkdir . rest)
    "MKDIR - Create a new file directory.
     Usage: mkdir [<directory-name>]
     Examples: mkdir /foo/bar
               mkdir ../baz
               mkdir mydir"
    (if (null? rest)
	(printf nil "\nNo directory name\n")
	((io-function current-file "mkdir") current-file (car rest))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; MODE - set structure display modes

(define-macro (mode type)
    "MODE - Set the print mode for structures.
     Modes : full-path - the full path name is printed at each branch
                         e.g. foo.bar[3].baz
             indent    - indent 4 spaces at each branch (default)
             tree      - display as a tree (lines connecting branches)
             no-type   - turns off the display of types
             type      - displays the type of each item and branch
             recursive - indent each level of recursive structures
             iterative - number each level of recursive structures
     Usage: mode full-path | indent | tree |
                 no-type | type | recursive | iterative
     Example: mode no-type"
    (cond ((eqv? type 'full-path)
	   (set-switch 0 0))
	  ((eqv? type 'indent)
	   (set-switch 0 1))
	  ((eqv? type 'tree)
	   (set-switch 0 2))
	  ((eqv? type 'no-type)
	   (set-switch 1 0))
	  ((eqv? type 'type)
	   (set-switch 1 1))
	  ((eqv? type 'recursive)
	   (set-switch 2 0))
	  ((eqv? type 'iterative)
	   (set-switch 2 1))
	  ((null? type)
	   (printf nil "\nNo print mode specified\n"))
	  (#t
	   (printf nil "\nPrint mode %s unknown\n", type))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; N-ENTRIES - return the number of symbol table entries in the current file

(define-macro (n-entries . x)
    "N-ENTRIES - Print the number of symbol table entries in the current file.
     Usage: n-entries"

    (printf nil
	    "\nNumber of symbol table entries in current file: %d\n"
	    ((io-function current-file "n-entries") current-file)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; SAVE-PALETTE - write a pgs palette to a disk file

(define-macro (save-palette pname fname)
    "SAVE-PALETTE - Write palette pname out to disk file fname.
     Usage: save-palette pname fname
     Example: save-palette foo foo.pal"
     (if (not (null? pname))
         (if (not (null? fname))
             (if (not (null? current-window))
                 (pg-write-palette (cadr current-window) pname fname))))
     #t)

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; SCALE-FILE-VARIABLE - write a new variable into the current file which
;                        has name = alias and values scaled by scale_factor.

(define-macro (scale-file-variable name aliasname scale_factor)
    "SCALE-FILE-VARIABLE - Scale variable name by scale_factor and
                           write it into the current file with name
                           aliasname.
     Usage: scale-file-variable name aliasname scale_factor
     Examples: scale-file-variable foo two_foo 2.0
               scale-file-variable bar -bar -1.0"
    (let* ((syment  (read-syment current-file name))
	   (symlist (pdb->list syment))
	   (dims    (cddr symlist))
	   (type    (car symlist))
	   (nameout (sprintf "@#%s#@" aliasname))
	   (vals    (build-array* name))
	   (typeout (sprintf "%s" type))
	   (args    (list nameout typeout)))
      (for-each
       (lambda (x)
	 (set! args (append (list (* scale_factor x)) args)))
       vals)
      (set! args (reverse args))
      (apply define-file-variable args)

      (let* ((syment    (read-syment current-file nameout))
	     (symlist   (pdb->list syment))
	     (address   (cadr symlist))
	     (symout    (list address typeout aliasname current-file)))
	(for-each
	 (lambda (x)
	   (set! symout (append (list x) symout)))
	 dims)
	(set! symout (reverse symout))
	(apply write-syment symout)))) 

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; OVERLAY - turn on/off overlay mode

(define (overlay . flag)
    (if flag
	(let* ((f (car flag)))
	  (set! overlay-flag (cond ((boolean? f)
				    f)
				   ((integer? f)
				    (not (= f 0)))
				   (else
				    #t)))))
    (if overlay-flag
        (begin (set! viewport-area-save viewport-area-box)
               (if (null? viewport-area-box)
                   (set! viewport-area-box overlay-box)))
        (set! viewport-area-box viewport-area-save))

    overlay-flag)

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; PDB-COPY - make a PDB file whose contents match the current file

; GOTCHA - this macro only copies variables in the current directory

(define-macro (pdb-copy . rest)
    "PDB-COPY - Make a new PDB file whose contents match the current file
     Usage: pdb-copy <filename>"
    (if rest
	(let* ((inf current-file)
	       (order (major-order inf))
	       (offset (default-offset inf))
	       (outfile (car rest)))
	  (change-file* outfile 'w)

	  (major-order current-file order)
	  (default-offset current-file offset)

	  (change-file* inf)

	  (set! follow-directories #t)

	  (copy* outfile '*))

	(printf nil "\nNo output file specified\n")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; PLOT - plot one variable against another

(define-macro (plot . args)
    "PLOT - Plot one variable against another.
     Usage: plot [<func>] <range> [<domain>]
     Examples: plot log10 y x
               plot log10 z (cps log10 x log10 y)
               plot (nds x 10 15)
               plot y x
               plot z"
    (let* ((a1 (list-ref args 0))
	   (a2 (list-ref args 1))
	   (a3 (list-ref args 2))
	   (v1 (if (printable? a1) a1 (eval a1)))
	   (v2 (if (printable? a2) a2 (eval a2)))
	   (v3 (if (printable? a2) a3 (eval a3)))
	   (func (if (procedure? v1) v1 nil))
	   (ran (if func v2 v1))
	   (dom (if func v3 v2)))
      (plot-hand (eval func) ran (if (pair? dom) (eval dom) dom))))

(define (plot* . args)
  "Procedure version of plot macro"
  (apply plot args))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; PLOTAC - plot an arbitrarily connected mapping
;          this command and connect-map need to be generalized

(define-macro (plotac . vars)
    (display-mapping* (apply connect-map vars)))


; CONNECT-MAP - make a mapping from a dump with the specified connectivity
;               this will be moved to pdbvdat.scm later

(define-macro (connect-map . vars)
    (let* ((cnnct (pdb-read-numeric-data current-file "zones"))
	   (dom (pm-connection->ac-domain current-file "x" "y"
					  "n_zones"
					  "n_nodes"
					  cnnct)))
      (ac vars dom zone)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; PRINT - print all or part of the named variable or structure member

(define-macro (print name)
    "PRINT - Print out all or part of a variable or member.
             Note that the command keyword may be omitted.
             To print part of a variable or member qualify
             the name with index expressions whose parts
             are in one of the three forms:
               <index>
               <index-min>:<index-max>
               <index-min>:<index-max>:<increment>
             Only the first form may be used to qualify
             variables or terminal structure members with
             embedded pointers and non-terminal members.
     Usage: [print] <variable> | <structure-member>
     Examples: Mapping2
               print Mapping4.domain.elements
               print Mapping2.range.elements[1]
               a[5,10:20,1:8:3]
               print a.b[3].c[5,10:20,1:8:3]"
    (print-aux name))

; the print procedure
(define (print* name)
    "Procedure version of print macro"
    (print-aux name))

; the auxiliary print procedure
(define (print-aux name)
    (newline)
    (if (null? name)
	(printf nil "No variable name specified\n")
        (let* ((data (cond ((pdbdata? name)
			    name)
			   ((pm-mapping? name)
			    (pm-mapping->pdbdata name))
			   ((pm-set? name)
			    (pm-set->pdbdata name))
			   ((pm-array? name)
			    (pm-array->pdbdata name))
			   ((pg-graph? name)
			    (pg-graph->pdbdata name))
			   ((pg-image? name)
			    (pg-image->pdbdata name))
			   ((file-variable? current-file name #f)
					; last arg is #f so index expr errors
					; can be reported as such
			    (read-pdbdata current-file name))
			   (else
			    (printf nil "Variable %s unknown\n" name)))))
	  (if data
	      (show-pdb data display-precision))))
    #f)

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; TABLE - write all or part of the named variable or
;         structure member to an opened textfile

; the table macro

(define-macro (table name)
    "TABLE - Write out all or part of a variable or
             member structure to an opened textfile
             To write part of a variable or member qualify
             the name with index expressions whose parts
             are in one of the three forms:
               <index>
               <index-min>:<index-max>
               <index-min>:<index-max>:<increment>
             Only the first form may be used to qualify
             variables or terminal structure members with
             embedded pointers and non-terminal members.
     Usage: table <variable> | <structure-member>
     Examples: table Mapping2
               table Mapping4.domain.elements
               table Mapping2.range.elements[1]
               table a[5,10:20,1:8:3]
               table a.b[3].c[5,10:20,1:8:3]"
    (table-aux name))

; the table procedure
(define (table* name)
    "Procedure version of table macro"
    (table-aux name))

; the auxiliary print procedure
(define (table-aux name)
    (if (null? name)
	(printf nil "No variable name specified\n")
        (let* ((data (cond ((pdbdata? name)
			    name)
			   ((pm-mapping? name)
			    (pm-mapping->pdbdata name))
			   ((pm-set? name)
			    (pm-set->pdbdata name))
			   ((pm-array? name)
			    (pm-array->pdbdata name))
			   ((pg-graph? name)
			    (pg-graph->pdbdata name))
			   ((pg-image? name)
			    (pg-image->pdbdata name))
			   ((file-variable? current-file name #f)
					; last arg is #f so index expr errors
					; can be reported as such
			    (read-pdbdata current-file name))
			   (else
			    (printf nil "Variable %s unknown\n" name)))))
	  (if data
	      (wr-to-textfile data display-precision))))
    #f)

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; PRINT-MAPPING / PM - print the specified mappings

(define-macro (print-mapping . x)
    "PRINT-MAPPING / PM - Print the specified mappings.
                   Mappings are referenced by the numbers displayed
                   by the menu command.
     Usage: print-mapping <mapping-list>
     Examples: print-mapping 3 4"

    (define-macro (print-item item)
	(let* ((mapping-data
		(if (pair? item)
		    (let* ((mapping (eval (cadr item))))
		      (if (pg-image? mapping)
			  (pg-image->pdbdata mapping)
			  (pm-mapping->pdbdata mapping)))
		    ((io-function current-file "map-print")
		     current-file item))))
	  (if (not (pdbdata? mapping-data))
	      (printf nil "%s is not a valid mapping\n" item)
	      (begin 
		(newline)
		(show-pdb mapping-data display-precision)
		#t))))

    (if x (for-each print-item x)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; PWD - print the current file directory

(define-macro (pwd . rest)
    "PWD - Print the current file directory.
     Usage: pwd"
    (printf nil "\n%s\n" ((io-function current-file "pwd") current-file)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; MARKER-SCALE

(define (MARKER-SCALE sca . things)
    "MARKER-SCALE - set the marker scale for the specified mappings
     Usage: marker-scale <scale> [<integer>]+"
    (window-manager "vsattr" "MARKER-SCALE" "double *" sca things)
    (if autoplot-flag
	(window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; RANM

(define (ranm . args)
    "RANM - Select plotting limits for the range of a mapping.
            If invoked with no limit values the range
            will default to that implied by the data set.
     Usage: ranm [<viewport> [<window>]] <mapping> <x1_min> <x1_max> ...
     Examples: ranm 1 -5.0 10.0 10 20
               ranm 1"
    (window-manager "mrlimit" args)
    (if autoplot-flag
	(window-manager "vupdate" args)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; RANV

(define (ranv . args)
    "RANV - Select plotting limits for the range of a viewport.
            If invoked with no limit values the range
            will default to that implied by the data sets.
     Usage: ranv [<viewport> [<window>]] <x1_min> <x1_max> ...
     Examples: ranv -5.0 10.0 10 20
               ranv"
    (window-manager "vrlimit" args)
    (if autoplot-flag
	(window-manager "vupdate" args)))
;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; REFMESH

(define (refmesh flag . things)
    "REFMESH - turn a reference mesh on or off
     Usage: refmesh on | off"
    (ref-mesh flag)
    (if autoplot-flag
	(window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; REFMESHCOLOR

(define (refmeshcolor color . things)
    "REFMESHCOLOR - set the line color for the reference mesh
     Usage: refmeshcolor color"
    (ref-mesh-color color)
    (if autoplot-flag
	(window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; SCATTER

(define (scatter mrk . things)
    "SCATTER - display given mappings from current viewport as scatter plots
     Usage: scatter on | off | plus | star | triangle [<integer>]+"
    (if (> mrk 0)
	(begin (window-manager "vsattr" "SCATTER"
			       "int *" 1 things)
	       (window-manager "vsattr" "MARKER-INDEX"
			       "int *" (- mrk 1) things))
	(window-manager "vsattr" "SCATTER"
			"int *" 0 things))
    (if autoplot-flag
	(window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; SET - set array display parameters

(define-macro (set name value)
    "SET - Set the value of a control parameter.
     Parameters:  line-length       - the number of array elements per line
                  array-length      - arrays shorter than value have each
                                      element labeled individually
                  bits-precision    - the number of bits displayed
                  decimal-precision - the number of digits displayed
     Usage: set line-length | array-length |
                bits-precision | decimal-precision <value>
     Examples: set line-length 3
               set decimal-precision 6"
    (cond ((null? name)
	   (printf nil "\nNo display parameter name specified\n"))
	  ((null? value)
	   (printf nil "\nNo display parameter value specified\n"))
	  ((eqv? name 'array-length)
	   (set-switch 3 value))
	  ((eqv? name 'line-length)
	   (set-switch 4 value))
	  ((eqv? name 'bits-precision)
	   (set! display-precision value))
	  ((eqv? name 'decimal-precision)
	   (set! display-precision
		 (truncate (abs (/ (- value 1) (log 2))))))
	  (#t (printf nil "\n%s is not a valid display parameter\n" name))))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; SHOW-PALETTES - show available palettes

(define (show-palettes . window)
    "SHOW-PALETTES - Show the available palettes and make selection current
     Usage: show-palettes [window]
     Examples: show-palettes
               show-palettes \"B\""
    (window-manager "pshow" window))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; STRUCT - display the structure of a data type

(define-macro (struct name)
    "STRUCT - Describe the named data type.
     Usage: struct <data-type>
     Examples: struct double
               struct PM_mapping"
    (newline)
    (if (null? name)
	(printf nil "No struct name specified\n")
	(begin (show-pdb (read-defstr* current-file name))
	       #t)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; SYNONYM - define synonyms for the given command

(define-macro (synonym name . syn)
    "SYNONYM - Define synonyms for the given function.
     Usage: synonym <func> [<synonym> ...]
     Examples: synonym change-file cf"
    (define-macro (defsyn x)
	(apply define-global (list x name)))
    (if syn
	(for-each defsyn syn)))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; TYPES - list the data types in the current file

(define-macro (types . args)
    "TYPES - List the types in the current file.
     Usage: types"
    (newline)
    (display (list-defstrs current-file))
    (newline)
    #t)

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; UPDATE-WINDOW

(define (update-window . windows)
    "UPDATE-WINDOW - Update all viewports in specified windows.
     Usage: update-window <window name list>"
    (window-manager "wupdate" windows))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; SET-VECTOR-ATT - set vector attributes

(define (set-vector-att attr value)
    "SET-VECTOR-ATT - Set vector plotting attributes.
     Usage: set-vector-att scale | headsize | color value"

     (cond ((eqv? attr scale)
            (set! vect-scale value))
           ((eqv? attr headsize)
            (set! vect-headsz value))
           ((eqv? attr color)
            (set! vect-color value)))

     (if autoplot-flag
         (window-manager "vupdate"))) 
     
;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

; VIEW-ANGLE

(define (view-angle theta phi chi)
    "VIEW-ANGLE - Set the three Euler viewing angles.
         The view angle is a rotation by <phi> about the z axis
         starting from the negative y axis counterclockwise
         followed by a rotation by <theta> about the x' axis
         counterclockwise and followed by a rotation by
         <chi> about the z'' axis again starting from the negative
         y'' axis counterclockwise.
     Usage: view-angle <theta> <phi> <chi>"
    (pg-set-view-angle! theta phi chi)
    (if autoplot-flag
	(window-manager "vupdate")))

;--------------------------------------------------------------------------
;--------------------------------------------------------------------------

