/*
 * PANACEA.C - main service routines provided by PANACEA
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "panace.h"

#ifdef DOS
#define MAXSIZE 275000L
#else

#ifdef CRAY
#define MAXSIZE 1000000L
#else

#ifdef MAC
#define MAXSIZE 275000L
#else
#define MAXSIZE 4000000L
#endif

#endif

#endif

/* RESTART VARIABLES */

int
 PA_name_spaces = FALSE;            /* flag controlling name space handling */

int
 _PA_halt_fl = FALSE;

PROCESS
 *PA_pp = NULL;

SC_THREAD_LOCK(PA_file_mon_lock);

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_SIMULATE - given that a numerical simulation code has been defined
 *             - with def_system, and properly initialized, perform the
 *             - simulation as defined
 *             -
 *             - Arguments:
 *             -
 *             -    tc       - current problem time
 *             -    nc       - current problem cycle
 *             -    ti       - initial problem time
 *             -    tf       - final time
 *             -    dtf_init - initial fractional dt
 *             -    dtf_min  - minimum fractional dt
 *             -    dtf_max  - maximum fractional dt
 *             -    dtf_inc  - fractional increase in dt per cycle
 */

void PA_simulate(tc, nc, nz, ti, tf, dtf_init, dtf_min, dtf_max, dtf_inc,
                 rsname, edname, ppname, gfname)
   double tc;
   int nc, nz;
   double ti, tf, dtf_init, dtf_min, dtf_max, dtf_inc;
   char *rsname, *edname, *ppname, *gfname;
   {double deltat, tconv, dtmn, dtmx, dtf, dt, t;
    int cycle;

/* clear the halt flag */
    _PA_halt_fl = FALSE;

/* initialize the time-cycle loop */
    tconv  = convrsn[SEC]/unit[SEC];
    deltat = tf - ti;
    dt     = dtf_init*deltat;
    dtmn   = dtf_min*deltat;
    dtmx   = dtf_max*deltat;
    PRINT(stdout, "Starting simulation: ti = %10.3e to tf = %10.3e\n",
           tc*tconv, tf*tconv);

/* loop over time-cycles */
    for (cycle = nc+1, t = tc; t <= tf; cycle++)

/* update the sourced variables */
        {PA_source_variables(t, dt);

/* call the packages and their post-processors */
         PA_run_packages(t, dt, cycle);
         
/* make post processor entries from the global data base */
         PA_dump_pp(t, dt, cycle);

/* monitor the auxilliary files */
         PA_file_mon(edname, ppname, gfname);

/* if a halt has been called to the simulation clean up and exit */
         if (_PA_halt_fl)
            {PRINT(stdout, "Halting computation: t = %10.3e\n\n",
                           t*tconv);
             break;};

/* advance the time */
         t  += dt;
         dt  = PA_advance_t(dtmn, dt*dtf_inc, dtmx);

/* adjust for the final cycle it this is it */
         if ((dtf = tf - t) > 0.0)
            dt = min(dtf, dt);};

/* announce the completion of the calculation */
    PRINT(stdout, "\nSimulation complete: t = %10.3e\n\n", (t-dt)*tconv);

/* finalize the packages and dump the statistics */
    PA_fin_system(nz, cycle, FALSE);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_TRAP_SIGNALS - trap errors and to do graceful things
 *                 - attach the specified functions to the specified signals
 */

#ifdef ANSI

void PA_trap_signals(int nsig, ...)

#endif

#ifdef PCC

void PA_trap_signals(nsig, va_alist)
   int nsig;
   va_dcl

#endif

   {int i, sig;
    PFVoid fnc;

    SC_VA_START(nsig);

    for (i = 0; i < nsig; i++)
        {sig = SC_VA_ARG(int);
         fnc = SC_VA_ARG(PFVoid);
         SIGNAL(sig, fnc);};

    SC_VA_END;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_SIGNAL_HANDLER - handle signals for PANACEA */

void PA_signal_handler(sig)
   int sig;
   {switch (sig)
       {
#ifdef SIGSEGV
        case SIGSEGV :
             PA_ERR(TRUE, "SEGMENTATION VIOLATION");
             break;
#endif

#ifdef SIGABRT
        case SIGABRT :
             PA_ERR(TRUE, "ABORT SIGNAL RECIEVED");
             break;
#endif

#ifdef SIGBUS
        case SIGBUS  :
             PA_ERR(TRUE, "BUS ERROR");
             break;
#endif

#ifdef SIGFPE
        case SIGFPE  :
             PA_ERR(TRUE, "FLOATING POINT EXCEPTION");
             break;
#endif
        default      :
             PA_ERR(TRUE, "UNKNOWN SIGNAL");
             break;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_INTERRUPT_HANDLER - handle interrupts for PANACEA */

void PA_interrupt_handler(sig)
   int sig;
   {char bf[MAXLINE], *cmnd, *name, *s;
    int cyc;
    double tcon, t, dt;
    PA_package *pck;

    SIGNAL(SIGINT, PA_interrupt_handler);

    PRINT(stdout, "\n\nInterrupt:\n");
    PRINT(stdout, "  c     - Enter a command loop\n");
    PRINT(stdout, "  f     - Finish after the current cycle\n");
    PRINT(stdout, "  p     - Current package\n");
    PRINT(stdout, "  q     - Quit immediately (no state dump)\n");
    PRINT(stdout, "  r     - Resume\n");
    PRINT(stdout, "  t     - Problem time\n");
    PRINT(stdout, "\nI-> ");
    GETLN(bf, MAXLINE, stdin);
    cmnd = SC_strtok(bf, " \t\n\r", s);
    SC_strtok(bf, " \t\n\r", s);

    fflush(stdout);

    pck  = PA_current_package();
    tcon = convrsn[SEC]/unit[SEC];
    name = pck->name;
    t    = pck->p_t*tcon;
    dt   = pck->p_dt*tcon;
    cyc  = pck->p_cycle;

    switch (cmnd[0])
       {case 'c' : PA_get_commands(stdin, NULL);
                   break;

        case 'f' : _PA_halt_fl = TRUE;
                   break;

        case 'p' : PRINT(stdout, "Running %s\n", name);
                   break;

        case 'q' : exit(1);

        case 'r' : 
        default  : PRINT(stdout, "\nResuming\n\n");
                   break;

        case 't' : PRINT(stdout, "Cycle = %4d, Time = %10.3e, Dt = %10.3e\n",
                                 cyc, t, dt);
                   break;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PA_INIT_FILES - initialize the auxilliary files */

void _PA_init_files(edname, ppname, gfname)
   char *edname, *ppname, *gfname;
   {

/* make the name of the first edit file */
    if (edname != NULL)
       {PA_edit_file = io_open(edname, "w");
	PA_ERR((PA_edit_file == NULL),
	       "CAN'T OPEN FILE %s - _PA_INIT_FILES", edname);
	PRINT(stdout, "Edit file %s opened\n", edname);};

/* initialize the post processor file */
    _PA_init_pp(ppname, gfname);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_FILE_MON - monitor the auxilliary files */

void PA_file_mon(edname, ppname, gfname)
   char *edname, *ppname, *gfname;
   {static long edstride = 0L, ppstride = 0L;
    long curpos;

/*    SC_LOCKON(PA_file_mon_lock); */

/* check on the edit file */
    if (PA_edit_file != NULL)
       {if (edstride == 0L)
           edstride = io_tell(PA_edit_file);
        curpos = io_tell(PA_edit_file);
        if (curpos+edstride >= MAXSIZE)
           {io_close(PA_edit_file);
            PA_advance_name(edname);
            PA_edit_file = io_open(edname, "w");
            PA_ERR((PA_edit_file == NULL),
                   "CAN'T OPEN FILE %s - PA_FILE_MON", edname);
            PRINT(stdout, "Edit file %s opened\n", edname);
            edstride = 0L;};};

/* check on the post processor */
    if (PA_pp_file != NULL)
       {if (ppstride == 0L)
           ppstride = io_tell(PA_pp_file->stream);
        curpos = io_tell(PA_pp_file->stream);
        if (curpos+ppstride >= MAXSIZE)
           {PA_close_pp();
            PA_advance_name(ppname);
            PA_pp_file = PA_open(ppname, "w", FALSE);
            PRINT(stdout, "Post processor file %s opened\n", ppname);
            ppstride = 0L;};};

/* check on the PVA file */
    if (PA_pva_file != NULL)
       {if (ppstride == 0L)
           ppstride = io_tell(PA_pva_file->stream);
        curpos = io_tell(PA_pva_file->stream);
        if (curpos+ppstride >= MAXSIZE)
           {PD_close(PA_pva_file);
            PA_advance_name(gfname);
            PA_pva_file = PA_open(gfname, "w", FALSE);
            PA_ERR(!PD_def_mapping(PA_pva_file),
                   "CAN`T DEFINE MAPPINGS - PA_FILE_MON");
            PRINT(stdout, "PVA file %s opened\n", gfname);
            ppstride = 0L;};};

/*    SC_LOCKOFF(PA_file_mon_lock); */

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_RD_RESTART - a friendly interface to the process of reading a
 *               - restart dump
 *               -
 *               - Arguments:
 *               -   rsname - the name of the restart file
 *               -   convs  - the type of conversions to do (see options)
 *               -
 *               - Options:
 *               -   NONE    - perform no conversions
 *               -   INT_CGS - convert from internal units to CGS units
 *               -   INT_EXT - convert from internal units to external units
 *               -   EXT_CGS - convert from external units to CGS units
 *               -   EXT_INT - convert from external units to internal units
 *               -   CGS_INT - convert from CGS units to internal units
 *               -   CGS_EXT - convert from CGS units to external units
 *               -
 *               - The internal system of units is defined by the "unit"
 *               - array.
 *               - The external system of units is defined by the "convrsn"
 *               - array.
 *               -
 *               - The system of units of the data in the restart dump is
 *               - under the control of the code developer, but it must
 *               - be consistent.
 */

void PA_rd_restart(rsname, convs)
   char *rsname;
   int convs;
   {long lsa, lsb;
    float space;

    SC_mem_stats(&lsb, NULL, NULL, NULL);

    _PA_rdrstrt(rsname, convs);

    SC_mem_stats(&lsa, NULL, NULL, NULL);

    space = ((float) (lsa - lsb)) / 1024.0;

    PRINT(STDOUT, "Space for state data and variable data base:");
    PRINT(STDOUT, " %g (kByte)\n", space);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_WR_RESTART - prepare and write a restart dump */

void PA_wr_restart(rsname)
   char *rsname;
   {

    if (rsname != NULL)
       {PRINT(stdout, "Restart dump %s ...", rsname);
	_PA_wrrstrt(rsname, NONE);
	PRINT(stdout, " written\n");};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_INIT_SYSTEM - initialize the code system
 *                - connect the global variables, open the post-processor
 *                - file and the edit file, and run the package initializers
 */

void PA_init_system(t, dt, nc, edname, ppname, gfname)
   double t, dt;
   int nc;
   char *edname, *ppname, *gfname;
   {PA_package *pck;
    PFInt pck_init, hook;

    SC_sizeof_hook = _PA_sizeof;

    PA_init_strings();
    PA_cpp_init();
    PA_cpp_default();
    PA_def_var_init();

/* initialize the source files and their variables */
    _PA_init_sources(t, dt);

/* initialize the pp files and structures */
    _PA_init_files(edname, ppname, gfname);

/* initialize the packages */
    for (pck = Packages; pck != NULL; pck = pck->next)
        {pck_init = pck->inizer;
         if (pck_init != NULL)
            {PA_control_set(pck->name);
             (*pck_init)(pck);};

         pck->time    = 0.0;
         pck->space   = 0.0;
         pck->dt      = HUGE;
         pck->dt_zone = -1;};

/* reset the global control arrays */
    PA_control_set("global");

    hook = PA_GET_FUNCTION(PFInt, "build_mapping");
    PA_ERR(((PA_pva_file != NULL) && (hook == NULL)),
           "CAN`T BUILD GRAPHS WITHOUT HOOK - PA_INIT_SYSTEM");

/* reprocess plots after the package initializers and the source files
 * have had their impact on both the data base and the plot requests
 */
    _PA_scan_pp();

/* do an initial dump */
    PA_dump_pp(t, dt, nc);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_INIT_STRINGS - convert the compiler allocated strings to
 *                 - dynamic memory so PDBLib is happy
 */

void PA_init_strings()
   {static int first = TRUE;

/* replace the static type strings with dynamic ones */
    if (first)
       {PA_SET_INDEX_P_S = SC_strsavef("PA_set_index *",
                            "char*:PA_INIT_STRINGS:indexp"),
        PA_SET_INDEX_S   = SC_strsavef("PA_set_index",
                            "char*:PA_INIT_STRINGS:index"),

        SC_CHAR_S      = SC_strsavef("char",
                          "char*:PA_INIT_STRINGS:char");
        SC_SHORT_S     = SC_strsavef("short",
                          "char*:PA_INIT_STRINGS:short");
        SC_INTEGER_S   = SC_strsavef("integer",
                          "char*:PA_INIT_STRINGS:integer");
        SC_LONG_S      = SC_strsavef("long",
                          "char*:PA_INIT_STRINGS:long");
        SC_FLOAT_S     = SC_strsavef("float",
                          "char*:PA_INIT_STRINGS:float");
        SC_DOUBLE_S    = SC_strsavef("double",
                          "char*:PA_INIT_STRINGS:double");
        SC_REAL_S      = SC_strsavef(SC_REAL_S,
                          "char*:PA_INIT_STRINGS:real");
        SC_STRING_S    = SC_strsavef("char *",
                          "char*:PA_INIT_STRINGS:char_s");
        SC_POINTER_S   = SC_strsavef("void *",
                          "char*:PA_INIT_STRINGS:void");

	SC_VOID_S      = SC_strsavef("void",
                          "char*:PA_INIT_STRINGS:void_s");
	SC_SHORT_P_S   = SC_strsavef("short *",
                          "char*:PA_INIT_STRINGS:short_p");
	SC_INTEGER_P_S = SC_strsavef("integer *",
                          "char*:PA_INIT_STRINGS:integer_p");
	SC_LONG_P_S    = SC_strsavef("long *",
                          "char*:PA_INIT_STRINGS:long_p");
	SC_FLOAT_P_S   = SC_strsavef("float *",
                          "char*:PA_INIT_STRINGS:float_p");
	SC_DOUBLE_P_S  = SC_strsavef("double *",
                          "char*:PA_INIT_STRINGS:double_p");
	SC_REAL_P_S    = SC_strsavef(SC_REAL_P_S,
                          "char*:PA_INIT_STRINGS:real_p");
        SC_PCONS_S     = SC_strsavef("pcons",
                          "char*:PA_INIT_STRINGS:pcons");
        SC_PCONS_P_S   = SC_strsavef("pcons *",
                          "char*:PA_INIT_STRINGS:pcons_p");
        SC_STRUCT_S    = SC_strsavef("struct",
                          "char*:PA_INIT_STRINGS:struct_s");
        SC_UNKNOWN_S   = SC_strsavef("unknown",
                          "char*:PA_INIT_STRINGS:struct_s");

/* do not delete any of these strings */
	SC_permanent(PA_SET_INDEX_P_S);
	SC_permanent(PA_SET_INDEX_S);

        SC_permanent(SC_CHAR_S);
        SC_permanent(SC_SHORT_S);
        SC_permanent(SC_INTEGER_S);
        SC_permanent(SC_LONG_S);
        SC_permanent(SC_FLOAT_S);
        SC_permanent(SC_DOUBLE_S);
	SC_permanent(SC_REAL_S);
        SC_permanent(SC_STRING_S);
        SC_permanent(SC_POINTER_S);
        SC_permanent(SC_PCONS_P_S);

	SC_permanent(SC_VOID_S);
	SC_permanent(SC_SHORT_P_S);
	SC_permanent(SC_INTEGER_P_S);
	SC_permanent(SC_LONG_P_S);
	SC_permanent(SC_FLOAT_P_S);
	SC_permanent(SC_DOUBLE_P_S);
	SC_permanent(SC_REAL_P_S);
        SC_permanent(SC_PCONS_S);
        SC_permanent(SC_STRUCT_S);
        SC_permanent(SC_UNKNOWN_S);

        first = FALSE;}

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_DEF_STR - let the packages define their defstr's
 *            - for the restart dump
 */

void PA_def_str(pdrs)
   PDBfile *pdrs;
   {PA_package *pck;
    PFInt pck_dfstrc, hook;

/* get the PM_set in for the PA_plot_request */
    PA_ERR(!PD_def_mapping(pdrs),
           "CAN`T DEFINE MAPPINGS - PA_DEF_STR");

/* define PML types */

    PD_defstr(pdrs, "complex",
              "double real",
              "double imaginary",
              LAST);

    PD_defstr(pdrs, "C_array",
              "char *type",
              "long length",
              "char *data",
              LAST);

    PD_cast(pdrs, "C_array", "data", "type");

    PD_defstr(pdrs, "PM_matrix",
	      "int nrow",
	      "int ncol",
	      "double *array",
	      LAST);

/* define PANACEA types */

    PD_defstr(pdrs, "PA_iv_specification",
              "integer type",
              "char *name",
              "char *file",
              "integer num",
              "integer index",
              "integer interpolate",
              "char *spec",           /* this is a fudge, spec will be NULL */
              "double *data",
              "PA_iv_specification *next",
              LAST);

    PD_defstr(pdrs, "PA_set_spec",
              "char *var_name",
              "char *function",
              "char *text",
	      "integer n_values",
	      "double *values",
              "PA_set_spec *next",
              LAST);

    PD_defstr(pdrs, "PA_plot_request",
              "PA_set_spec *range",
              "char *range_name",
              "PA_set_spec *domain",
              "char *base_domain_name",
              "C_array *domain_map",
              "char *text",
              "integer time_plot",
              "integer mesh_plot",
              "integer status",
              "long size",
              "long offset",
              "long stride",
              "integer str_index",
              "double conv",
              "integer centering",
              "integer allocate_data",
              "PM_set *data",
              "integer data_index",
              "integer data_type",
              "PA_plot_request *next",
              LAST);

    PD_defstr(pdrs, "PA_set_index",
              "double val",
              "integer index",
              "integer imax",
              "double conv",
              "char *name",
              LAST);

/* see if there is a special hook */
    hook = PA_GET_FUNCTION(PFInt, "define-types");
    if (hook != NULL)
       (*hook)(pdrs);

/* poll the packages */
    for (pck = Packages; pck != NULL; pck = pck->next)
        {pck_dfstrc = pck->dfstrc;
         if (pck_dfstrc != NULL)
            (*pck_dfstrc)(pdrs);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_RUN_PACKAGES - call the packages and their post-processors */

void PA_run_packages(t, dt, cycle)
   double t, dt;
   int cycle;
   {PA_package *pck;
    PFInt pck_entry;
    char *pck_name;

/* loop over all packages */
    for (pck = Packages; pck != NULL; pck = pck->next)
        {pck_entry = pck->main;
         pck_name  = pck->name;

/* execute the package */
         if (pck_entry != NULL)
            {pck->p_t     = t;
             pck->p_dt    = dt;
             pck->p_cycle = cycle;

             PA_control_set(pck_name);
             _PA_allocate_mapping_space(pck);

             (*pck_entry)(pck);

             _PA_dump_package_mappings(pck, t, dt, cycle);};};

/* reconnect the global controls */
    PA_control_set("global");

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_ADVANCE_T - compute and return the new dt
 *              - the time steps returned by the packages are checked
 *              - against the min and max allowable dt (dtmn and dtmx
 *              - respectively) and an initial dt value
 */

double PA_advance_t(dtmn, dtn, dtmx)
   double dtmn, dtn, dtmx;
   {double dt, pck_dt;
    PA_package *pck;

/* get the new time step */
    for (dt = dtn, pck = Packages; pck != NULL; pck = pck->next)
        {pck_dt = pck->dt;
         dt = min(pck_dt, dt);};

    dt = min(dt, dtmx);
    dt = max(dt, dtmn);

    return(dt);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_FIN_SYSTEM - call the finalizers for all of the packages and
 *               - dump the performance statistics for the run
 */

void PA_fin_system(nz, nc, silent)
   int nz, nc, silent;
   {int i;
    long lmx;
    PA_package *pck;
    PFInt pck_fin;
    char *pck_name, bf[MAXLINE];
    double pck_t, pck_s, prb_s, prb_t, izc, smx, szf;

/* finalize the packages and dump the statistics */
    if (!silent)
       {smx = 0.0;
	for (pck = Packages; pck != NULL; pck = pck->next)
	    {pck_s = pck->space;
	     smx = max(smx, pck_s);};

	PRINT(STDOUT, "\nProblem Statistics:\n\n");

	memset(bf, ' ', MAXLINE);
	sprintf(bf, "Package");
	sprintf(&bf[30], "Rate");
	sprintf(&bf[45], "Time");

	sprintf(&bf[60], "Space");
	sprintf(&bf[79], "\n");
	for (i = 0; i < 80; i++)
	    if (bf[i] == '\0')
	       bf[i] = ' ';
	bf[80] = '\0';
	PRINT(STDOUT, "%s", bf);

	memset(bf, ' ', MAXLINE);
	sprintf(&bf[21], "(microsec/zone-cycle)");
	sprintf(&bf[45], "(sec)");

	if (smx > 1.0e9)
	   {szf = 1.0e9;
	    sprintf(&bf[59], "(GBytes)");}

	else if (smx > 1.0e6)
	   {szf = 1.0e6;
	    sprintf(&bf[59], "(MBytes)");}

	else
	   {szf = 1.0e3;
	    sprintf(&bf[59], "(kBytes)");};

	sprintf(&bf[78], "\n\n");
	for (i = 0; i < 80; i++)
	    if (bf[i] == '\0')
	       bf[i] = ' ';
	bf[80] = '\0';
	PRINT(STDOUT, "%s", bf);};

    if (nc == 0)
       izc = 0.0;
    else
       izc = (1.0e6)/((double) nz*nc);

    for (prb_t = 0.0, pck = Packages; pck != NULL; pck = pck->next)
        {pck_fin  = pck->finzer;
         pck_name = pck->name;
         pck_t    = pck->time;
         pck_s    = pck->space;

         PA_control_set(pck_name);

         if (pck_fin != NULL)
            (*pck_fin)(pck);

/* if no time was spent in the package and it took no space don't log it */
         if ((pck_t == 0.0) && (pck_s == 0))
            continue;

/* only collect the statistics for the packages which are on */
         if (PA_PACKAGE_ON(pck) && !silent)
            {prb_t += pck_t;

             memset(bf, ' ', MAXLINE);
             sprintf(bf, "%s", pck_name);
             sprintf(&bf[27], "%10.3e", izc*pck_t);
             sprintf(&bf[42], "%10.3e", pck_t);

             pck_s /= szf;
             sprintf(&bf[59], "%g", pck_s);
             sprintf(&bf[79], "\n");
             for (i = 0; i < 80; i++)
                 if (bf[i] == '\0')
                    bf[i] = ' ';
             bf[80] = '\0';
             PRINT(STDOUT, "%s", bf);};};

    if (!silent)
       {memset(bf, '-', 79);
	PRINT(STDOUT, "%s", bf);

	memset(bf, ' ', MAXLINE);
	sprintf(bf, "Total");
	sprintf(&bf[27], "%10.3e", izc*prb_t);
	sprintf(&bf[42], "%10.3e", prb_t);

	SC_mem_stats(NULL, NULL, NULL, &lmx);

	prb_s = ((float) lmx) / 1024.0;
	sprintf(&bf[59], "%g", prb_s);
	sprintf(&bf[78], "\n\n");
	for (i = 0; i < 80; i++)
	    if (bf[i] == '\0')
	       bf[i] = ' ';
	bf[80] = '\0';
	PRINT(STDOUT, "%s", bf);};

/* reconnect the global controls */
    PA_control_set("global");

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_TERMINATE - close all the open files and generally gracefully end
 *              - the PANACEA run
 */

void PA_terminate(edname, ppname, gfname, cycle)
   char *edname, *ppname, *gfname;
   int cycle;
   {int i;

    if (PA_edit_file != NULL)
       PA_ERR(io_close(PA_edit_file),
              "TROUBLE CLOSING EDIT FILE %s - PA_TERMINATE", edname);

/* save the number of time plots and the number of points (cycles) */
    if (PA_pp_file != NULL)
       PA_close_pp();

    if (PA_pva_file != NULL)
        PA_ERR(!PD_close(PA_pva_file),
               "TROUBLE CLOSING PVA FILE %s - PA_TERMINATE", gfname);
     
    if (PA_cache_file != NULL)
        PA_ERR(!PD_close(PA_cache_file),
               "TROUBLE CLOSING CACHE FILE - PA_TERMINATE");
     
/* close open state files */
    for (i = 0; i < _PA_n_state_files; i++)
        {PA_ERR(!PD_close(_PA_state_files[i]),
                "TROUBLE CLOSING STATE FILE %d - PA_TERMINATE", i);
         _PA_state_files[i] = NULL;};

    SFREE(_PA_state_files);

    PA_pva_file   = NULL;
    PA_edit_file  = NULL;
    PA_cache_file = NULL;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PA_CLOSE_PP - close out the current pp file */

void PA_close_pp()
   {int err;

    PA_ERR(err = !PD_close(PA_pp_file),
           "TROUBLE CLOSING POST PROCESSOR FILE %s - PA_CLOSE_PP",
           (err) ? PA_pp_file->name : NULL);

    PA_current_pp_cycle = -1;

    PA_pp_file = NULL;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
