%---------------------------------------------------------------------------%
% Copyright (C) 1995-1997 The University of Melbourne.
% This file may only be copied under the terms of the GNU General
% Public License - see the file COPYING in the Mercury distribution.
%---------------------------------------------------------------------------%
%
% Original author: squirrel (Jane Anna Langley).
% Some bugs fixed by fjh.
% Extensive revision by zs.
% More revision by stayl.
%
% This module attempts to optimise out instances where a variable is
% decomposed and then soon after reconstructed from the parts. If possible
% we would like to "short-circuit" this process.
% It also optimizes deconstructions of known cells, replacing them with
% assignments to the arguments where this is guaranteed to not increase
% the number of stack slots required by the goal.
% Repeated calls to predicates with the same input arguments are replaced by
% assigments and warnings are returned.
%
% IMPORTANT: This module does a small subset of the job of compile-time
% garbage collection, but it does so without paying attention to uniqueness
% information, since the compiler does not yet have such information.
% Once we implement ctgc, the assumptions made by this module will have
% to be revisited.
%
%---------------------------------------------------------------------------%

:- module common.
:- interface.

:- import_module hlds_pred, simplify.

	% If we find a deconstruction or a construction we cannot optimize,
	% record the details of the memory cell in CommonInfo.

	% If we find a construction that constructs a cell identical to one
	% we have seen before, replace the construction with an assignment
	% from the variable unified with that cell.

:- pred common__optimise_unification(unification, var, unify_rhs,
	unify_mode, unify_context, hlds_goal_expr, hlds_goal_info,
	hlds_goal_expr, hlds_goal_info, simplify_info, simplify_info).
:- mode common__optimise_unification(in, in, in, in, in, in, in, 
	out, out, in, out) is det.

	% Check whether this call has been seen before and is replaceable, if
	% so produce assignment unification for the non-local output variables,
	% and give a warning.
	% A call is replaceable if it has no uniquely moded outputs and no
	% destructive inputs.

:- pred common__optimise_call(pred_id, proc_id, list(var), hlds_goal_expr,
	hlds_goal_info, hlds_goal_expr, simplify_info, simplify_info).
:- mode common__optimise_call(in, in, in, in, in, out, in, out) is det.

:- pred common__optimise_higher_order_call(var, list(var), list(mode),
		determinism, hlds_goal_expr, hlds_goal_info, hlds_goal_expr,
		simplify_info, simplify_info).
:- mode common__optimise_higher_order_call(in, in, in, in, in, in, out,
		in, out) is det.

	% succeeds if the two variables are equivalent
	% according to the specified equivalence class.
:- pred common__vars_are_equivalent(var, var, common_info).
:- mode common__vars_are_equivalent(in, in, in) is semidet.

	% Assorted stuff used here that simplify.m doesn't need to know about.
:- type common_info.

:- pred common_info_init(common_info).
:- mode common_info_init(out) is det.

:- pred common_info_clear_structs(common_info, common_info).
:- mode common_info_clear_structs(in, out) is det.

%---------------------------------------------------------------------------%

:- implementation.

:- import_module hlds_goal, hlds_data, quantification, mode_util, type_util.
:- import_module det_util, det_report, globals, options, inst_match, instmap.
:- import_module prog_data, hlds_module, (inst).
:- import_module bool, term, map, set, list, eqvclass, require, std_util.

:- type structure	--->	structure(var, type, cons_id, list(var)).

:- type call_args
	--->	call_args(term__context, list(var), list(var)).
			% input, output args. For higher-order calls, 
			% the closure is the first input argument.

:- type struct_map	==	map(cons_id, list(structure)).
:- type seen_calls 	==	map(seen_call_id, list(call_args)).

:- type common_info
	--->	common(
			eqvclass(var),
			struct_map,	% all structs seen.
			struct_map,	% structs seen since the last call.
			seen_calls
		).

%---------------------------------------------------------------------------%

common_info_init(CommonInfo) :-
	eqvclass__init(VarEqv0),
	map__init(StructMap0),
	map__init(SeenCalls0),
	CommonInfo = common(VarEqv0, StructMap0, StructMap0, SeenCalls0).

	% Clear structs seen since the last call. Replacing deconstructions
	% of these structs with assignments after the call would cause an
	% increase in the number of stack slots required.
common_info_clear_structs(common(VarEqv, StructMap, _, SeenCalls),
		common(VarEqv, StructMap, Empty, SeenCalls)) :-
	map__init(Empty).
		
%---------------------------------------------------------------------------%

common__optimise_unification(Unification0, Left0, _Right0, Mode, Context,
		Goal0, GoalInfo0, Goal, GoalInfo, Info0, Info) :-
	(
		Unification0 = construct(Var, ConsId, ArgVars, _),
		(
			common__find_matching_cell(Var, ConsId, ArgVars,
				construction, Info0, OldStruct)
		->
			OldStruct = structure(OldVar, _, _, _),
			Unification = assign(Var, OldVar),
			Right = var(OldVar),
			Goal = unify(Left0, Right, Mode, Unification, Context), 
			common__record_equivalence(Var, OldVar, Info0, Info1),
			simplify_info_set_requantify(Info1, Info),
			set__list_to_set([OldVar, Var], NonLocals),
			goal_info_set_nonlocals(GoalInfo0, NonLocals, GoalInfo)
		;
			Goal = Goal0,
			GoalInfo = GoalInfo0,
			common__record_cell(Var, ConsId, ArgVars, Info0, Info)
		)
	;
		Unification0 = deconstruct(Var, ConsId, ArgVars, _, _),
		(
			simplify_info_get_module_info(Info0, ModuleInfo),
			Mode = LVarMode - _,
			mode_get_insts(ModuleInfo, LVarMode, Inst0, Inst1),
				% Don't optimise away partially instantiated
				% deconstruction unifications.
			inst_matches_binding(Inst0, Inst1, ModuleInfo),
			common__find_matching_cell(Var, ConsId, ArgVars,
				deconstruction, Info0, OldStruct)
		->
			OldStruct = structure(_, _, _, OldArgVars),
			goal_info_get_nonlocals(GoalInfo0, NonLocals),
			simplify_info_get_instmap(Info0, InstMap),
			goal_info_get_instmap_delta(GoalInfo0, InstMapDelta),
			common__create_output_unifications(NonLocals, InstMap,
				InstMapDelta, ArgVars, OldArgVars, Goals,
				no, RecomputeAtomic),
			Goal = conj(Goals),
			common__record_equivalences(ArgVars,
				OldArgVars, Info0, Info1),
			simplify_info_set_requantify(Info1, Info2),
			( RecomputeAtomic = yes ->
				simplify_info_set_recompute_atomic(Info2, Info)
			;
				Info = Info2
			)
		;
			Goal = Goal0,
			common__record_cell(Var, ConsId, ArgVars, Info0, Info)
		),
		GoalInfo = GoalInfo0
	;
		Unification0 = assign(Var1, Var2),
		Goal = Goal0,
		common__record_equivalence(Var1, Var2, Info0, Info),
		GoalInfo = GoalInfo0
	;
		Unification0 = simple_test(Var1, Var2),
		Goal = Goal0,
		common__record_equivalence(Var1, Var2, Info0, Info),
		GoalInfo = GoalInfo0
	;
		Unification0 = complicated_unify(_, _),
		Goal = Goal0,
		Info = Info0,
		GoalInfo = GoalInfo0
	).

%---------------------------------------------------------------------------%

:- type unification_type
	--->	deconstruction
	;	construction.
		
:- pred common__find_matching_cell(var, cons_id,
		list(var), unification_type, simplify_info, structure).
:- mode common__find_matching_cell(in, in, in, in, in, out) is semidet.

common__find_matching_cell(Var, ConsId, ArgVars, UniType, Info, OldStruct) :-
	simplify_info_get_common_info(Info, CommonInfo),
	simplify_info_get_var_types(Info, VarTypes),
	CommonInfo = common(VarEqv, StructMapAll, StructMapSinceLastFlush, _),
	(
		UniType = construction,
		StructMapToUse = StructMapAll
	;
		% For deconstructions, using the arguments of a cell
		% created before the last stack flush would cause more
		% variables to be saved on the stack.
		UniType = deconstruction,
		StructMapToUse = StructMapSinceLastFlush
	),
	map__search(StructMapToUse, ConsId, Structs),
	common__find_matching_cell_2(Structs, Var, ConsId, ArgVars, UniType,
		VarEqv, VarTypes, OldStruct).

:- pred common__find_matching_cell_2(list(structure), var, cons_id, list(var),
	unification_type, eqvclass(var), map(var, type), structure).
:- mode common__find_matching_cell_2(in, in, in, in, in,
	in, in, out) is semidet.

common__find_matching_cell_2([Struct | Structs], Var, ConsId, ArgVars,
		UniType, VarEqv, VarTypes, OldStruct) :-
	Struct = structure(OldVar, StructType, StructConsId, StructArgVars),
	(
		% Are the arguments the same (or equivalent) variables?
		ConsId = StructConsId,
		(
			UniType = construction,
			common__var_lists_are_equiv(ArgVars,
				StructArgVars, VarEqv),

			% Two structures of the same shape may have different 
			% types and therefore different representations.
			map__lookup(VarTypes, Var, VarType),
			common__compatible_types(VarType, StructType)
		;
			UniType = deconstruction,
			common__vars_are_equiv(Var, OldVar, VarEqv)
		)
	->
		OldStruct = Struct
	;
		common__find_matching_cell_2(Structs, Var, ConsId, ArgVars,
			UniType, VarEqv, VarTypes, OldStruct)
	).

%---------------------------------------------------------------------------%

	% Two structures have compatible representations if the top
	% level of their types are unifiable.  % For example, if we have
	%
	%	:- type maybe_err(T) --> ok(T) ; err(string).
	%
	%	:- pred p(maybe_err(foo)::in, maybe_err(bar)::out) is semidet.
	%	p(err(X), err(X)).
	%
	% then we want to reuse the `err(X)' in the first arg rather than
	% constructing a new copy of it for the second arg.
	% The two occurrences of `err(X)' have types `maybe_err(int)'
	% and `maybe(float)', but we know that they have the same 
	% representation.

:- pred common__compatible_types(type, type).
:- mode common__compatible_types(in, in) is semidet.

common__compatible_types(Type1, Type2) :-
	type_to_type_id(Type1, TypeId1, _),
	type_to_type_id(Type2, TypeId2, _),
	TypeId1 = TypeId2.

%---------------------------------------------------------------------------%

	% succeeds if the two lists of variables are equivalent
	% according to the specified equivalence class.
:- pred common__var_lists_are_equiv(list(var), list(var), eqvclass(var)).
:- mode common__var_lists_are_equiv(in, in, in) is semidet.

common__var_lists_are_equiv([], [], _VarEqv).
common__var_lists_are_equiv([X | Xs], [Y | Ys], VarEqv) :-
	common__vars_are_equiv(X, Y, VarEqv),
	common__var_lists_are_equiv(Xs, Ys, VarEqv).

common__vars_are_equivalent(X, Y, CommonInfo) :-
	CommonInfo = common(EqvVars, _, _, _),
	common__vars_are_equiv(X, Y, EqvVars).

	% succeeds if the two variables are equivalent
	% according to the specified equivalence class.
:- pred common__vars_are_equiv(var, var, eqvclass(var)).
:- mode common__vars_are_equiv(in, in, in) is semidet.

common__vars_are_equiv(X, Y, VarEqv) :-
	% write('looking for equivalence of '),
	% write(X),
	% write(' and '),
	% write(Y),
	% nl,
	(
		X = Y
	;
		eqvclass__is_member(VarEqv, X),
		eqvclass__is_member(VarEqv, Y),
		eqvclass__same_eqvclass(VarEqv, X, Y)
	).
	% write('they are equivalent'),
	% nl.

%---------------------------------------------------------------------------%

:- pred common__record_cell(var, cons_id, list(var),
		simplify_info, simplify_info).
:- mode common__record_cell(in, in, in, in, out) is det.

common__record_cell(Var, ConsId, ArgVars, Info0, Info) :-
	simplify_info_get_common_info(Info0, CommonInfo0),
	simplify_info_get_var_types(Info0, VarTypes),
	( ArgVars = [] ->
		% Constants do not have memory cells to reuse,
		% at least in the memory models we are interested in.
		CommonInfo = CommonInfo0
	;
		CommonInfo0 = common(VarEqv, StructMapAll0,
			StructMapLastCall0, SeenCalls),
		map__lookup(VarTypes, Var, VarType),
		Struct = structure(Var, VarType, ConsId, ArgVars),
		common__do_record_cell(StructMapAll0, ConsId,
			Struct, StructMapAll),
		common__do_record_cell(StructMapLastCall0, ConsId, Struct,
			StructMapLastCall),
		CommonInfo = common(VarEqv, StructMapAll,
			StructMapLastCall, SeenCalls)
	),
	simplify_info_set_common_info(Info0, CommonInfo, Info).

:- pred common__do_record_cell(struct_map, cons_id, structure, struct_map).
:- mode common__do_record_cell(in, in, in, out) is det.

common__do_record_cell(StructMap0, ConsId, Struct, StructMap) :-
	( map__search(StructMap0, ConsId, StructList0Prime) ->
		StructList0 = StructList0Prime
	;
		StructList0 = []
	),

	% Insert the new cell at the front of the list. If it hides
	% an equivalent cell, at least the reuse of this cell will
	% require saving its address over fewer calls.

	StructList = [Struct | StructList0],
	map__set(StructMap0, ConsId, StructList, StructMap).

%---------------------------------------------------------------------------%

:- pred common__record_equivalences(list(var), list(var),
		simplify_info, simplify_info) is det.
:- mode common__record_equivalences(in, in, in, out) is det.

common__record_equivalences(Vars0, Vars, Info0, Info) :-
	simplify_info_get_common_info(Info0, CommonInfo0),
	CommonInfo0 = common(VarEqv0, StructMap0, StructMap1, SeenCalls),
	common__record_equivalences_2(Vars0, Vars, VarEqv0, VarEqv),
	CommonInfo = common(VarEqv, StructMap0, StructMap1, SeenCalls),
	simplify_info_set_common_info(Info0, CommonInfo, Info).

:- pred common__record_equivalences_2(list(var), list(var),
		eqvclass(var), eqvclass(var)) is det.
:- mode common__record_equivalences_2(in, in, in, out) is det.

common__record_equivalences_2([], [_ | _], _, _) :-
	error("common__record_equivalences").
common__record_equivalences_2([_ | _], [], _, _) :-
	error("common__record_equivalences").
common__record_equivalences_2([], [], VarEqv, VarEqv).
common__record_equivalences_2([Var1 | Vars1], [Var2 | Vars2],
		VarEqv0, VarEqv) :-
	eqvclass__ensure_equivalence(VarEqv0, Var1, Var2, VarEqv1),
	common__record_equivalences_2(Vars1, Vars2, VarEqv1, VarEqv).

:- pred common__record_equivalence(var, var, simplify_info, simplify_info).
:- mode common__record_equivalence(in, in, in, out) is det.

common__record_equivalence(Var1, Var2, Info0, Info) :-
	simplify_info_get_common_info(Info0, CommonInfo0),
	CommonInfo0 = common(VarEqv0, StructMap0, StructMap1, SeenCalls),
	% write('ensuring equivalence of '),
	% write(Var1),
	% write(' and '),
	% write(Var2),
	% nl,
	eqvclass__ensure_equivalence(VarEqv0, Var1, Var2, VarEqv),
	CommonInfo = common(VarEqv, StructMap0, StructMap1, SeenCalls),
	simplify_info_set_common_info(Info0, CommonInfo, Info).
	
%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%

common__optimise_call(PredId, ProcId, Args, Goal0,
		GoalInfo, Goal, Info0, Info) :-
	(
		goal_info_get_determinism(GoalInfo, Det),
		common__check_call_detism(Det),
		simplify_info_get_module_info(Info0, ModuleInfo),
		module_info_pred_proc_info(ModuleInfo, PredId,
			ProcId, _, ProcInfo),
		proc_info_argmodes(ProcInfo, ArgModes),
	    	common__partition_call_args(ModuleInfo, ArgModes, Args,
			InputArgs, OutputArgs)
	->
		common__optimise_call_2(seen_call(PredId, ProcId), InputArgs,
			OutputArgs, Goal0, GoalInfo, Goal, Info0, Info)
	;
		Goal = Goal0,
		Info = Info0
	).

common__optimise_higher_order_call(Closure, Args, Modes, Det, Goal0,
		GoalInfo, Goal, Info0, Info) :-
	(
		common__check_call_detism(Det),
		simplify_info_get_module_info(Info0, ModuleInfo),
	    	common__partition_call_args(ModuleInfo, Modes, Args,
			InputArgs, OutputArgs)
	->
		common__optimise_call_2(higher_order_call,
			[Closure | InputArgs], OutputArgs, Goal0,
			GoalInfo, Goal, Info0, Info)
	;
		Goal = Goal0,
		Info = Info0
	).	

:- pred common__check_call_detism(determinism::in) is semidet.

common__check_call_detism(Det) :-
	determinism_components(Det, _, SolnCount),
	% Replacing nondet or mulidet calls would cause
	% loss of solutions.
	(	SolnCount = at_most_one
	;	SolnCount = at_most_many_cc
	).

:- pred common__optimise_call_2(seen_call_id, list(var), list(var),
		hlds_goal_expr, hlds_goal_info, hlds_goal_expr,
		simplify_info, simplify_info).
:- mode common__optimise_call_2(in, in, in, in, in, out, in, out) is det.

common__optimise_call_2(SeenCall, InputArgs, OutputArgs, Goal0,
		GoalInfo, Goal, Info0, Info) :-
	simplify_info_get_common_info(Info0, CommonInfo0),
	CommonInfo0 = common(Eqv0, Structs0, Structs1, SeenCalls0),
	(
		map__search(SeenCalls0, SeenCall, SeenCallsList0)
	->
		( common__find_previous_call(SeenCallsList0, InputArgs,
			Eqv0, OutputArgs2, PrevContext)
		->
			simplify_info_get_instmap(Info0, InstMap0),
			goal_info_get_nonlocals(GoalInfo, NonLocals),
			goal_info_get_instmap_delta(GoalInfo, InstMapDelta),
			common__create_output_unifications(NonLocals, InstMap0,
			    InstMapDelta, OutputArgs, OutputArgs2, Goals,
			    no, RecomputeAtomic),
			Goal = conj(Goals),
			simplify_info_get_var_types(Info0, VarTypes),
			(
			    simplify_do_warn_calls(Info0),
				% Don't warn for cases such as:
				% set__init(Set1 : set(int)),
				% set__init(Set2 : set(float)).
			    map__apply_to_list(OutputArgs, VarTypes,
					OutputArgTypes),
			    map__apply_to_list(OutputArgs2, VarTypes,
					OutputArgTypes)
			->
			    goal_info_get_context(GoalInfo, Context),
			    simplify_info_add_msg(Info0,
			    	duplicate_call(SeenCall, PrevContext,
					Context),
			        Info1)
			;
			    Info1 = Info0
			),
			CommonInfo = common(Eqv0, Structs0,
				Structs1, SeenCalls0),
			simplify_info_set_requantify(Info1, Info2),
			( RecomputeAtomic = yes ->
				simplify_info_set_recompute_atomic(Info2, Info3)
			;
				Info3 = Info2
			)
		;
			goal_info_get_context(GoalInfo, Context),
			ThisCall = call_args(Context, InputArgs, OutputArgs),
			map__det_update(SeenCalls0, SeenCall,
				[ThisCall | SeenCallsList0], SeenCalls),
			CommonInfo = common(Eqv0, Structs0,
				Structs1, SeenCalls),
			Goal = Goal0,
			Info3 = Info0
		)
	;
		goal_info_get_context(GoalInfo, Context),
		ThisCall = call_args(Context, InputArgs, OutputArgs),
		map__det_insert(SeenCalls0, SeenCall, [ThisCall], SeenCalls),
		CommonInfo = common(Eqv0, Structs0, Structs1, SeenCalls),
		Goal = Goal0,
		Info3 = Info0
	),
	simplify_info_set_common_info(Info3, CommonInfo, Info).

%---------------------------------------------------------------------------%

	% Partition the arguments of a call into inputs and outputs, failing
	% if any of the outputs have a unique component.
:- pred common__partition_call_args(module_info::in, list(mode)::in,
		list(var)::in, list(var)::out, list(var)::out) is semidet.

common__partition_call_args(_, [], [_ | _], _, _) :-
	error("common__partition_call_args").
common__partition_call_args(_, [_ | _], [], _, _) :-
	error("common__partition_call_args").
common__partition_call_args(_, [], [], [], []).
common__partition_call_args(ModuleInfo, [ArgMode | ArgModes], [Arg | Args],
		InputArgs, OutputArgs) :-
	common__partition_call_args(ModuleInfo, ArgModes, Args,
		InputArgs1, OutputArgs1),
	mode_get_insts(ModuleInfo, ArgMode, InitialInst, FinalInst),
	( inst_matches_binding(InitialInst, FinalInst, ModuleInfo) ->
		InputArgs = [Arg | InputArgs1],
		OutputArgs = OutputArgs1
	; 
		% Calls with partly unique outputs cannot be replaced,
		% since a unique copy of the outputs must be produced.
		inst_is_not_partly_unique(ModuleInfo, FinalInst),
		% Don't optimize calls where a partially instantiated
		% variable is further instantiated.
		inst_is_free(ModuleInfo, InitialInst),
		InputArgs = InputArgs1,
		OutputArgs = [Arg | OutputArgs1]
	).

%---------------------------------------------------------------------------%

:- pred common__find_previous_call(list(call_args)::in, list(var)::in,
		eqvclass(var)::in, list(var)::out,
		term__context::out) is semidet.

common__find_previous_call([SeenCall | SeenCalls], InputArgs,
		Eqv, OutputArgs2, PrevContext) :-
	SeenCall = call_args(PrevContext, InputArgs1, OutputArgs1),
	( common__var_lists_are_equiv(InputArgs, InputArgs1, Eqv) ->
		OutputArgs2 = OutputArgs1
	;
		common__find_previous_call(SeenCalls, InputArgs, Eqv,
			OutputArgs2, PrevContext)
	).

%---------------------------------------------------------------------------%

:- pred common__create_output_unifications(set(var)::in, instmap::in,
		instmap_delta::in, list(var)::in, list(var)::in,
		list(hlds_goal)::out, bool::in, bool::out) is det.

	% Create unifications to assign the non-local vars in OutputArgs from 
	% the corresponding var in OutputArgs2.
common__create_output_unifications(_, _, _, [], [], [],
		RecomputeAtomic, RecomputeAtomic).
common__create_output_unifications(_, _, _, [_ | _], [], _, _, _) :-
	error("common__create_output_unifications").
common__create_output_unifications(_, _, _, [], [_ | _], _, _, _) :-
	error("common__create_output_unifications").
common__create_output_unifications(NonLocals, InstMap, InstMapDelta, 
		[OutputArg | OutputArgs], [OutputArg2 | OutputArgs2],
		Goals, RecomputeAtomic0, RecomputeAtomic) :-
	( 
		set__member(OutputArg, NonLocals),
		% This can happen if the first cell was created with a
		% partially instantiated deconstruction.
		OutputArg \= OutputArg2	
	->
		Unification = assign(OutputArg, OutputArg2),
		instmap__lookup_var(InstMap, OutputArg2, Inst0),
		% Check if OutputArg2 was local to the call or deconstruction
		% that produced it. If it was, recompute_instmap_delta must 
		% recompute the instmap_deltas for atomic goals.
		( Inst0 = free ->
			Inst = ground(shared, no),
			RecomputeAtomic1 = yes
		;	
			Inst = Inst0,
			RecomputeAtomic1 = RecomputeAtomic0
		),	
		% This will be fixed up in recompute_instmap_delta
		UniMode = (free -> Inst) - (Inst -> Inst),
		UnifyContext = unify_context(explicit, []),
		Goal = unify(OutputArg, var(OutputArg2),
			UniMode, Unification, UnifyContext),
		set__list_to_set([OutputArg, OutputArg2], GoalNonLocals),
		set__list_to_set([OutputArg], OutputSet),
		instmap_delta_restrict(InstMapDelta, OutputSet,
			GoalInstMapDelta),
		goal_info_init(GoalNonLocals, GoalInstMapDelta, det, GoalInfo),
		common__create_output_unifications(NonLocals, InstMap,
			InstMapDelta, OutputArgs, OutputArgs2, Goals1,
			RecomputeAtomic1, RecomputeAtomic),
		Goals = [Goal - GoalInfo | Goals1]
	;
		common__create_output_unifications(NonLocals, InstMap,
			InstMapDelta, OutputArgs, OutputArgs2, Goals1,
			RecomputeAtomic0, RecomputeAtomic),
		Goals = Goals1
	).

%---------------------------------------------------------------------------%
%---------------------------------------------------------------------------%
