/* 
   XRCell.m

   NSCell for GNUstep GUI X/RAW Backend

   Copyright (C) 1996 Free Software Foundation, Inc.

   A completely rewritten version of the original source by Pascal Forget.
   Author: Ovidiu Predescu <ovidiu@bx.logicnet.ro>
   Date: February 1997
   Author:  Felipe A. Rodriguez <far@ix.netcom.com>
   Date: May 1998
   
   This file is part of the GNUstep GUI X/RAW Backend.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/ 

#include <config.h>

#include <Foundation/NSException.h>

#include <AppKit/NSGraphics.h>
#include <AppKit/NSControl.h>
#include <AppKit/NSFont.h>
#include <AppKit/NSImage.h>

#include <gnustep/xraw/XRCell.h>
#include <gnustep/xraw/XRWindow.h>
#include <gnustep/xraw/XR.h>

extern void XRDrawString(const char *str, NSRect rect);



@implementation XRCell

//
// Class methods
//
+ (void)initialize
{
	if (self == [XRCell class])
		[self setVersion:1];								// Initial version
}

//
// Instance methods
//
- (void)_displayImage:(NSImage*)image inFrame:(NSRect)frame
{										
NSSize size;

	if (!image)
		return;

//  PSrectclip (frame.origin.x + _xDist, frame.origin.y + _yDist,
//	      frame.size.width - 2 * _xDist, frame.size.height - 2 * _yDist);

	size = [image size];
	frame.origin.x += (frame.size.width - size.width) / 2;
	frame.origin.y += (frame.size.height - size.height) / 2;
	NSDebugLog (@"_displayImage origin ((%f, %f))",
			  		frame.origin.x, frame.origin.y);

											// Display image centered in frame.
	[image compositeToPoint:frame.origin operation:NSCompositeCopy];
}

- (void)_displayTitle:(NSString*)title				// Displays text centered
				inFrame:(NSRect)frame				// vertically.  horizontal
				titleGray:(float)titleGray			// alignment is determined
{													// by the cell alignment.
NSPoint point;
float titleWidth;

	if (!title)
		return;

	if (!cell_font)
		[NSException raise:NSInvalidArgumentException
				format:@"Request to draw a text cell but no font specified!"];

	[cell_font set];							// set the drawing engine font
	titleWidth = [cell_font widthOfString:title];
												// calc y origin for text
	point.y = frame.origin.y - (frame.size.height / 2) - 
				[cell_font descender] - 2;

//  PSrectclip (frame.origin.x + xDist, frame.origin.y + yDist,
//	      frame.size.width - 2 * xDist, frame.size.height - 2 * yDist);

	switch (text_align) 					// Determine the x position of text
		{						 
		case NSLeftTextAlignment:	 		// ignore the justified and natural
    	case NSJustifiedTextAlignment:		// alignments
   		case NSNaturalTextAlignment:
      		point.x = frame.origin.x + xDist;
     		break;
    	case NSRightTextAlignment:
      		if (titleWidth < frame.size.width)
				point.x = frame.origin.x + frame.size.width - 
							(xDist + titleWidth);
      		else
				point.x = frame.origin.x + xDist;
      		break;
    	case NSCenterTextAlignment:
      		if (titleWidth < frame.size.width)
				point.x = frame.origin.x + (frame.size.width - titleWidth) / 2;
      		else
				point.x = frame.origin.x + xDist;
      		break;
  		}

	frame.origin = point;
	[[NSColor colorWithCalibratedWhite:titleGray alpha:1.0] set];

	XRDrawString([title cString], frame);
}

- (void)drawWithFrame:(NSRect)cellFrame inView:(NSView*)controlView
{
	NSDebugLog (@"NSCell drawWithFrame:inView:");

	if ([self isOpaque])							
		{										// Clear the cell frame
		[[NSColor lightGrayColor] set];
		NSRectFill(cellFrame);
		}

	if (cell_bordered) 							// draw the border if needed
		{
    	if (cell_bezeled) 
			{
			NSDrawWhiteBezel (cellFrame, cellFrame);
			cellFrame.origin.x += 2;
			cellFrame.origin.y += 2;
			cellFrame.size.width -= 4;
			cellFrame.size.height -= 4;
    		}
    	else 
			{
			NSFrameRect (cellFrame);
			cellFrame.origin.x += 1;
			cellFrame.origin.y += 1;
			cellFrame.size.width -= 2;
			cellFrame.size.height -= 2;
    		}
  		}

//  PSrectclip (cellFrame.origin.x, cellFrame.origin.y,
//	      cellFrame.size.width, cellFrame.size.height);
//  PStranslate (cellFrame.origin.x, cellFrame.origin.y);
//  cellFrame.origin = NSZeroPoint;

	[self drawInteriorWithFrame:cellFrame inView:controlView];
}

@end

//
// Methods the backend should implement
//
@implementation XRCell (GNUstepBackend)

+ (NSSize)sizeForBorderType:(NSBorderType)aType
{												
	switch (aType)					
		{										// Returns the size of a border
		case NSLineBorder:
			return NSMakeSize(1, 1);
		case NSGrooveBorder:
		case NSBezelBorder:
			return NSMakeSize(2, 2);
		case NSNoBorder:
		default:
			return NSZeroSize;
		}
}

@end
