/* 
   EODatabaseChannel.h

   Copyright (C) 1996 Free Software Foundation, Inc.

   Author: Mircea Oancea <mircea@jupiter.elcom.pub.ro>
   Date: 1996

   This file is part of the GNUstep Database Library.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with this library; see the file COPYING.LIB.
   If not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#ifndef __EODatabaseChannel_h__
#define __EODatabaseChannel_h__

#include <Foundation/NSObject.h>

@class NSArray;
@class NSMutableArray;
@class NSDictionary;
@class NSMutableDictionary;
@class NSString;
@class NSMutableString;

@class EOAdaptor;
@class EOAdaptorContext;
@class EOAdaptorChannel;
@class EOEntity;
@class EOQualifier;
@class EORelationship;

@class EOObjectUniquer;
@class EODatabase;
@class EODatabaseContext;

@interface EODatabaseChannel : NSObject
{
    EOAdaptorChannel*		adaptorChannel;
    EODatabaseContext*		databaseContext;
    id				delegate;
    EOEntity*			currentEntity;
    Class			currentClass;
    NSArray*			currentAttributes;
    NSArray*			currentRelations;
    BOOL			currentReady;
}

// Initializing a new instance
- initWithDatabaseContext:(EODatabaseContext*)aDatabaseContext;

// Getting the adaptor channel
- (EOAdaptorChannel*)adaptorChannel;

// Getting the database context
- (EODatabaseContext*)databaseContext;

// Setting the delegate
- (void)setDelegate:aDelegate;
- delegate;

// Opening and closing a channel
- (BOOL)isOpen;
- (BOOL)openChannel;
- (void)closeChannel;

// Modifying objects
- (BOOL)insertObject:anObj;
- (BOOL)updateObject:anObj;
- (BOOL)deleteObject:anObj;
- (BOOL)lockObject:anObj;
- (BOOL)refetchObject:anObj;
- (id)allocateObjectForRow:(NSDictionary*)row
  entity:(EOEntity*)anEntity
  zone:(NSZone*)zone;
- (id)initializedObjectForRow:(NSDictionary*)row
  entity:(EOEntity*)anEntity
  zone:(NSZone*)zone;

// Fetching objects
- (BOOL)selectObjectsDescribedByQualifier:(EOQualifier*)qualifier
  fetchOrder:(NSArray*)fetchOrder;
- fetchWithZone:(NSZone*)zone;
- (BOOL)isFetchInProgress;
- (void)cancelFetch;
- (void)setCurrentEntity:(EOEntity*)anEntity;

@end /* EODatabaseChannel */

/*
 * Delegate methods
 */

@interface NSObject(EODatabaseChannelDelegateProtocol)

- databaseChannel:aChannel
  willInsertObject:anObj;
- (void)databaseChannel:aChannel
  didInsertObject:anObj;
- databaseChannel:aChannel
  willDeleteObject:anObj;
- (void)databaseChannel:aChannel
  didDeleteObject:anObj;
- databaseChannel:aChannel
  willUpdateObject:anObj;
- (void)databaseChannel:aChannel
  didUpdateObject:anObj;
- (NSDictionary*)databaseChannel:aChannel
  willRefetchObject:anObj;
- (NSDictionary*)databaseChannel:aChannel
  didRefetchObject:anObj;
- (NSDictionary*)databaseChannel:aChannel
  willRefetchObject:anObj
  fromSnapshot:(NSDictionary*)snapshot;
- (NSDictionary*)databaseChannel:aChannel
  willRefetchConflictingObject:anObj
  withSnapshot:(NSMutableDictionary*)snapshot;
- (BOOL)databaseChannel:aChannel
  willSelectObjectsDescribedByQualifier:(EOQualifier*)qualifier
  fetchOrder:(NSArray*)fetchOrder;
- (void)databaseChannel:aChannel
  didSelectObjectsDescribedByQualifier:(EOQualifier*)qualifier
  fetchOrder:(NSArray*)fetchOrder;
- (void)databaseChannel:aChannel
  willFetchObjectOfClass:(Class)class
  withZone:(NSZone*)zone;
- (void)databaseChannel:aChannel
  didFetchObject:anObj;
- databaseChannel:aChannel
  willLockObject:anObj;
- (void)databaseChannel:aChannel
  didLockObject:anObj;
- (Class)databaseChannel:aChannel
  failedToLookupClassNamed:(const char*)name;
- (EORelationship*)databaseChannel:aChannel
  relationshipForRow:(NSDictionary*)row 
  relationship:(EORelationship*)relationship;
@end

/*
 * Object Awaking (EODatabaseChannelNotification protocol)
 */

@interface NSObject(EODatabaseChannelNotification)
- (void)awakeForDatabaseChannel:(EODatabaseChannel*)channel;
@end

#endif /* __EODatabaseChannel_h__ */
