// QWeb - An SGML Web Browser
// Copyright (C) 1997  Sean Vyain
// svyain@mail.tds.net
// smvyain@softart.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
#include <qdrawutl.h>
#include <qframe.h>
#include <qpainter.h>
#include "Canvas.h"
#include "Style.h"
#include "TableRenderer.h"

Cell::Cell( Canvas*     canvas,
            SgmlParser* parser,
            int         clipWidth,
            QObject*    parent,
            const char* name )
        : VerticalRenderer( canvas, parser, clipWidth, parent, name )
{
}

Cell::~Cell()
{
}

void Cell::repaint( QPainter& p, const Rect& r )
{
//    int x = 0;
//    int y = 0;
//    for ( QObject* o = this; o; o = o->parent() ) {
//        x += ((Renderer*)o)->x();
//        y += ((Renderer*)o)->y();
//    }  
//    qDrawShadeRect( &p, x, y, width(), height(), canvas()->colorGroup(), FALSE, 2, 1, 0 );

    VerticalRenderer::repaint( p, r );
}

TableRenderer::TableRenderer( Canvas*            canvas,
                              SgmlParser*        parser,
                              int                clipWidth,
                              QObject*           parent,
                              const char*        name )
        : SgmlRenderer( canvas, parser, clipWidth, parent, name ),
          _cell( 0 ),
          _cellDepth( 0 ),
          _cellPadding( 0 ),
          _isEmpty( TRUE )
{
}

TableRenderer::~TableRenderer()
{
    while ( _rows.first() ) {
	while ( _rows.first()->first() ) {
	    delete _rows.first()->first();
	    _rows.first()->remove();
	}
	delete _rows.first();
	_rows.remove();
    }
}

void TableRenderer::content( QString text )
{
    if ( _cell ) {
        _cell->content( text );
        _isEmpty = FALSE;
    }
}

void TableRenderer::endOfData()
{
    if ( _cell ) {
        _cell->endOfData();
//        if ( _isEmpty ) _cell->hide();
        _cell = 0;
        _isEmpty = TRUE;
    }
    needRedraw();
}

void TableRenderer::endTag()
{
    if ( _cell ) {
        if ( _cellDepth >= tagStack().count() ) {
            _cell->endOfData();
//            if ( _isEmpty ) _cell->hide();
            _cell = 0;
            _isEmpty = TRUE;
        } else {
            _cell->endTag();
            _isEmpty = FALSE;
        }
    }
}

void TableRenderer::startTag()
{
    if ( _cell ) {
        _cell->startTag();
        _isEmpty = FALSE;
        return;
    }

    if ( style()->displayType() == Style::Table ) {
        // My start tag.
        if ( !style()->numberValue( "cell-padding", _cellPadding ) ) {
            _cellPadding = 4;
        }
    } else if ( style()->displayType() == Style::TableRow ) {
        // Start a new row.
        _rows.append( new Row );
    } else if ( style()->displayType() == Style::TableData ) {
        // Start a new cell.
        _cell = new Cell( canvas(), parser(), 100, this );
        connect( _cell, SIGNAL( resized() ), this, SLOT( childSizeChanged() ) );
//        _cell->show();
        _cellDepth = tagStack().count();
        _rows.last()->append( _cell );

        QString str;
        const QString* tmp = 0;
        if ( ( style()->stringValue( "row-span-attr", str ) ) && ( tmp = tagStack().getLast()->find( str ) ) ) {
            _cell->rowspan = tmp->toInt();
        } else {
            _cell->rowspan = 1;
        }
        if ( ( style()->stringValue( "column-span-attr", str ) ) && ( tmp = tagStack().getLast()->find( str ) ) ) {
            _cell->colspan = tmp->toInt();
        } else {
            _cell->colspan = 1;
        }
    }
}

bool TableRenderer::redraw()
{
    // Give my kids a chance to redraw.
    bool b = FALSE;
    Row*  r;
    Cell* c;
    for ( r = _rows.first(); r; r = _rows.next() ) {
        for ( c = r->first(); c; c = r->next() ) {
            if ( c->redraw() ) {
                b = TRUE;
            }
        }
    }
    
    // Do I need to redraw?
    if ( !_needRedraw ) return b;
    
    uint i, j;
    int  x, y, w, h;

    // Decide where to put everything...
    QArray<uint> rowspans;
    for ( r = _rows.first(); r; r = _rows.next() ) {
        for ( i = 0; i < rowspans.size(); i++ ) {
            if ( rowspans[i] > 0 ) {
                rowspans[i]--;
            }
        }
        i = 0;
        for ( c = r->first(); c; c = r->next() ) {
            for( ; ( i < rowspans.size() ) && ( rowspans[i] > 0 ); i++ );
            c->col = i;
            c->row = _rows.at();
            if ( rowspans.size() < i+c->colspan ) {
                j = rowspans.size();
                rowspans.resize( i+c->colspan );
                for ( ; j < rowspans.size(); j++ ) {
                    rowspans[j] = 0;
                }
            }
            for ( j = 0; j < c->colspan; j++ ) {
                if ( c->rowspan > rowspans[i+j] ) {
                    rowspans[i+j] = c->rowspan;
                }
            }
        }
    }

    // Compute min and max widths for each column...
    QArray<int> colMinWidth( rowspans.size() );
    QArray<int> colMaxWidth( rowspans.size() );
    colMinWidth.fill( 0 );
    colMaxWidth.fill( 0 );
    int min, max;
    for ( r = _rows.first(); r; r = _rows.next() ) {
        for ( c = r->first(); c; c = r->next() ) {
            // Account for cell spaning multiple columns.  Do this the easy way for now...
            min = c->minimumWidth() / c->colspan;
            max = c->maximumWidth() / c->colspan;

            for ( i = 0; i < c->colspan; i++ ) {
                if ( min > colMinWidth[c->col+i] ) {
                    colMinWidth[c->col+i] = min;
                }
                if ( max > colMaxWidth[c->col+i] ) {
                    colMaxWidth[c->col+i] = max;
                }
            }
        }
    }

    // Compute total min and max widths.
    _minimumWidth = ( colMinWidth.size() + 1 ) * _cellPadding;
    _maximumWidth = ( colMaxWidth.size() + 1 ) * _cellPadding;
    for ( i = 0; i < colMinWidth.size(); i++ ) {
        _minimumWidth += colMinWidth[i];
    }
    for ( i = 0; i < colMaxWidth.size(); i++ ) {
        _maximumWidth += colMaxWidth[i];
    }

    // Assign column widths.
    QArray<int> colWidth( colMinWidth.size() );
    if ( _minimumWidth > clipWidth() ) {
        // Every column gets minimum width.
        for ( i = 0; i < colWidth.size(); i++ ) {
            colWidth[i] = colMinWidth[i];
        }
    } else if ( _maximumWidth < clipWidth() ) {
        // Every column gets maximum width.
        for ( i = 0; i < colWidth.size(); i++ ) {
            colWidth[i] = colMaxWidth[i];
        }
    } else if ( _minimumWidth == _maximumWidth ) {
        // Min and max are the same, everybody gets minimum.
        for ( i = 0; i < colWidth.size(); i++ ) {
            colWidth[i] = colMinWidth[i];
        }
    } else {
        // Compromise...
        int W = clipWidth() - _minimumWidth;
        int D = _maximumWidth - _minimumWidth;
        
        for ( i = 0; i < colWidth.size(); i++ ) {
            colWidth[i] = colMinWidth[i] + W * ( colMaxWidth[i] - colMinWidth[i] ) / D;
        }
    }

    // Move and resize everything...
    QArray<int> rowHeights( _rows.count() );
    rowHeights.fill( 0 );
    for ( r = _rows.first(); r; r = _rows.next() ) {
        for ( c = r->first(); c; c = r->next() ) {
            for ( i = 0, w = ( c->colspan - 1 ) * _cellPadding; i < c->colspan; i++ ) w += colWidth[c->col+i];
            c->blockSignals( TRUE );
            c->widthChanged( w );
            c->redraw();
            c->blockSignals( FALSE );

            // Distribute height evenly across all spanned rows.
            h = c->height() / c->rowspan;
            for ( i = 0; i < c->rowspan; i++ ) {
                if ( h > rowHeights[c->row+i] ) {
                    rowHeights[c->row+i] = h;
                }
            }
        }
    }

    y = 0;
    for ( r = _rows.first(); r; r = _rows.next() ) {
        for ( c = r->first(); c; c = r->next() ) {
            for ( i = 0, x = _cellPadding; i < c->col; i++ ) {
                if ( i < colWidth.size() ) {
                    x += colWidth[i] + _cellPadding;
                } else {
                    return TRUE;
                }
            }
            c->move( x, y );
        }
        y += rowHeights[_rows.at()] + _cellPadding;
    }
    
    for ( i = 0, w = _cellPadding; i < colWidth.size(); i++ ) w += colWidth[i] + _cellPadding;
    for ( i = 0, h = _cellPadding; i < rowHeights.size(); i++ ) h += rowHeights[i] + _cellPadding;
    resize( w, y );

    _needRedraw = FALSE;

    return TRUE;
}

bool TableRenderer::findAnchor( const QString& name, int& x, int& y )
{
    Row*  r;
    Cell* c;

    for ( r = _rows.first(); r; r = _rows.next() ) {
        for ( c = r->first(); c; c = r->next() ) {
            if ( c->findAnchor( name, x, y ) ) {
                x += c->x();
                y += c->y();
                return TRUE;
            }
        }
    }

    return FALSE;
}

void TableRenderer::repaint( QPainter& p, const Rect& r )
{
    p.translate( x(), y() );

    QListIterator<Row> row( _rows );
    for ( row.toFirst(); row.current(); ++row ) {
        QListIterator<Cell> cell( *row.current() );
        for ( cell.toFirst(); cell.current(); ++cell ) {
            if ( cell.current()->intersects( r ) ) {
                Rect r1 = cell.current()->intersect( r );
                r1.moveBy( -cell.current()->x(), -cell.current()->y() );
                cell.current()->repaint( p, r1 );
            }
        }
    }

    p.translate( -x(), -y() );
}
