/* AX.25 Utilities: Integer speed to termios speed.
 * Bruce Perens, November 1994
 *
 * Copyright 1994 Bruce Perens.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */
#include <stdlib.h>
#include <unistd.h>
#include <termios.h>

struct speed_struct
{
	int     user_speed;
	speed_t termios_speed;
};

static const struct speed_struct speed_table[] =
{
{	0,	B0	},
{	50,	B50	},
{	75,	B75	},
{	110,	B110	},
{	134,	B134	},
{	150,	B150	},
{	200,	B200	},
{	300,	B300	},
{	600,	B600	},
{	1200,	B1200	},
{	1800,	B1800	},
{	2400,	B2400	},
{	4800,	B4800	},
{	9600,	B9600	},
{	19200,	B19200	},
{	38400,	B38400	},
#ifdef	B57600
{	57600,  B57600	},
#endif
#ifdef	B115200
{	115200,	B115200	},
#endif
{	-1,	-1	}
};

int
termios_speed_encoding(int speed)
{
	const struct speed_struct *	s = speed_table;

	while ( s->user_speed >= 0 ) {
		if ( s->user_speed == speed )
			return s->termios_speed;
		s++;
	}
	return -1;
}

int
closest_termios_speed_encoding(int speed)
{
	const struct speed_struct *	s = speed_table;

	/*
	 * Any speed greater than 0 but smaller than the lowest non-zero
	 * speed matches the lowest non-zero system speed.
	 * Any speed higher than the lowest non-zero system speed matches
	 * the system speed it is closest to. If a speed is exactly
	 * between two system speeds, it matches the higher one.
	 */
	if ( speed <= 0 )
		return B0;
	else if ( speed <= speed_table[1].user_speed )
		return speed_table[1].termios_speed;

	while ( s[1].user_speed >= 0 && s->user_speed != speed ) {
		if ( s->user_speed < speed && s[1].user_speed > speed ) {
			if ( (speed - s->user_speed)
			 < (s[1].user_speed - speed) )
				break;
			else
				return s[1].termios_speed;
		}
		s++;
	}
	return s->termios_speed;
}
