/* nasmlib.c	library routines for the Netwide Assembler
 *
 * The Netwide Assembler is copyright (C) 1996 Simon Tatham and
 * Julian Hall. All rights reserved. The software is
 * redistributable under the licence given in the file "Licence"
 * distributed in the NASM archive.
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

#include "nasm.h"
#include "nasmlib.h"

static efunc nasm_malloc_error;

void nasm_set_malloc_error (efunc error) {
    nasm_malloc_error = error;
}

void *nasm_malloc (size_t size) {
    void *p = malloc(size);
    if (!p)
	nasm_malloc_error (ERR_FATAL | ERR_NOFILE, "out of memory");
    return p;
}

void *nasm_realloc (void *q, size_t size) {
    void *p = q ? realloc(q, size) : malloc(size);
    if (!p)
	nasm_malloc_error (ERR_FATAL | ERR_NOFILE, "out of memory");
    return p;
}

void nasm_free (void *q) {
    if (q)
	free (q);
}

char *nasm_strdup (char *s) {
    char *p;

    p = nasm_malloc(strlen(s)+1);
    strcpy (p, s);
    return p;
}

int nasm_stricmp (char *s1, char *s2) {
    while (*s1 && toupper(*s1) == toupper(*s2))
	s1++, s2++;
    if (!*s1 && !*s2)
	return 0;
    else if (toupper(*s1) < toupper(*s2))
	return -1;
    else
	return 1;
}

int nasm_strnicmp (char *s1, char *s2, int n) {
    while (n > 0 && *s1 && toupper(*s1) == toupper(*s2))
	s1++, s2++, n--;
    if ((!*s1 && !*s2) || n==0)
	return 0;
    else if (toupper(*s1) < toupper(*s2))
	return -1;
    else
	return 1;
}

void strip_newline (char *s) {
    if (s[strlen(s)-1] == '\n')
    	s[strlen(s)-1] = '\0';
}

#define isnumchar(c)   ( isalnum(c) || (c) == '$')
#define numvalue(c)  ((c)>='a' ? (c)-'a'+10 : (c)>='A' ? (c)-'A'+10 : (c)-'0')

long readnum (char *str, int *error) {
    char *r = str, *q;
    long radix;
    long result;

    *error = FALSE;

    while (isspace(*r)) r++;	       /* find start of number */
    q = r;

    while (isnumchar(*q)) q++;	       /* find end of number */

    /* 
     * If it begins 0x, 0X or $, or ends in H, it's in hex. if it
     * ends in Q, it's octal. if it ends in B, it's binary.
     * Otherwise, it's ordinary decimal.
     */
    if (*r=='0' && (r[1]=='x' || r[1]=='X'))
	radix = 16, r += 2;
    else if (*r=='$')
	radix = 16, r++;
    else if (q[-1]=='H' || q[-1]=='h')
	radix = 16 , q--;
    else if (q[-1]=='Q' || q[-1]=='q')
	radix = 8 , q--;
    else if (q[-1]=='B' || q[-1]=='b')
	radix = 2 , q--;
    else
	radix = 10;

    result = 0;
    while (*r && r < q) {
	if (*r<'0' || (*r>'9' && *r<'A') || numvalue(*r)>=radix) {
	    *error = TRUE;
	    return 0;
	}
	result = radix * result + numvalue(*r);
	r++;
    }
    return result;
}

static long next_seg;

void seg_init(void) {
    next_seg = 0;
}

long seg_alloc(void) {
    return (next_seg += 2) - 2;
}

void fwriteshort (int data, FILE *fp) {
    fputc (data & 255, fp);
    fputc ((data >> 8) & 255, fp);
}

void fwritelong (long data, FILE *fp) {
    fputc (data & 255, fp);
    fputc ((data >> 8) & 255, fp);
    fputc ((data >> 16) & 255, fp);
    fputc ((data >> 24) & 255, fp);
}

void standard_extension (char *inname, char *outname, char *extension,
			 efunc error) {
    char *p, *q;

    q = inname;
    p = outname;
    while (*q) *p++ = *q++;	       /* copy, and find end of string */
    *p = '\0';			       /* terminate it */
    while (p > outname && *--p != '.');/* find final period (or whatever) */
    if (*p != '.') while (*p) p++;     /* go back to end if none found */
    if (!strcmp(p, extension)) {       /* is the extension already there? */
	if (*extension)
	    error(ERR_WARNING | ERR_NOFILE,
		  "file name already ends in `%s': "
		  "output will be in `nasm.out'",
		  extension);
	else
	    error(ERR_WARNING | ERR_NOFILE,
		  "file name already has no extension: "
		  "output will be in `nasm.out'");
	strcpy(outname, "nasm.out");
    } else
	strcpy(p, extension);
}
