------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             G N A T B I N D                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.29 $                             --
--                                                                          --
--     Copyright (C) 1992,1993,1994,1995 Free Software Foundation, Inc.     --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
-- MA 02111-1307, USA.                                                      --
--                                                                          --
-- GNAT was originally developed  by the GNAT team at  New York University. --
-- It is now maintained by Ada Core Technologies Inc (http://www.gnat.com). --
--                                                                          --
------------------------------------------------------------------------------

with ALI;      use ALI;
with Bcheck;   use Bcheck;
with Binde;    use Binde;
with Binderr;  use Binderr;
with Bindgen;  use Bindgen;
with Bindusg;
with Butil;    use Butil;
with Gnatvsn;  use Gnatvsn;
with Namet;    use Namet;
with Opt;      use Opt;
with Osint;    use Osint;
with Output;   use Output;
with Types;    use Types;

with System.Standard_Library;
--  This can eventually be removed when the new binder is in place ???

procedure Gnatbind is

   Total_Errors : Nat := 0;
   --  Counts total errors in all files

   Total_Warnings : Nat := 0;
   --  Total warnings in all files

   Main_Lib_File : File_Name_Type;
   --  Current main library file

   Std_Lib_File : File_Name_Type;
   --  Standard library

   Text : Text_Buffer_Ptr;
   Id   : ALI_Id;

begin
   Osint.Initialize (Binder);

   if Verbose_Mode then
      Write_Eol;
      Write_Str ("GNAT Binder Version ");
      Write_Str (Gnat_Version_String);
      Write_Str (" Copyright 1995 Free Software Foundation, Inc.");
      Write_Eol;
   end if;

   --  Output usage information if no files

   if not More_Lib_Files then
      Bindusg;
      Exit_Program (E_Fatal);
   end if;

   --  The block here is to catch the Unrecoverable_Error exception in the
   --  case where we exceed the maximum number of permissible errors or some
   --  other unrecoverable error occurs.

   begin
      --  Carry out package initializations. These are initializations which
      --  might logically be performed at elaboration time, but Namet at
      --  least can't be done that way (because it is used in the Compiler),
      --  and we decide to be consistent. Like elaboration, the order in
      --  which these calls are made is in some cases important.

      Namet.Initialize;
      Initialize_Binderr;
      Initialize_ALI;

      if Verbose_Mode then
         Write_Eol;
      end if;

      --  Input ALI files

      while More_Lib_Files loop
         Main_Lib_File := Next_Main_Lib_File;

         if Verbose_Mode then
            if Check_Only then
               Write_Str ("Checking: ");
            else
               Write_Str ("Binding: ");
            end if;

            Write_Name (Main_Lib_File);
            Write_Eol;
         end if;

         Text := Read_Library_Info (Main_Lib_File, True);
         Id := Scan_ALI (Main_Lib_File, Text);
      end loop;

      --  Add System.Standard_Library to list to ensure that these files are
      --  included in the bind, even if not directly referenced from Ada code

      Name_Buffer (1 .. 12) := "s-stalib.ali";
      Name_Len := 12;
      Std_Lib_File := Name_Find;
      Text := Read_Library_Info (Std_Lib_File, True);
      Id := Scan_ALI (Std_Lib_File, Text);

      --  Acquire all information in ALI files that have been read in

      for Index in ALIs.First .. ALIs.Last loop
         Read_ALI (Index);
      end loop;

      --  Build source file table from the ALI files we have read in

      Set_Source_Table;

      --  Check that main library file is a suitable main program

      if Bind_Main_Program
        and then ALIs.Table (ALIs.First).Main_Program = None
      then
         Error_Msg_Name_1 := Main_Lib_File;
         Error_Msg ("% does not contain a unit that can be a main program");
      end if;

      --  Perform consistency checks

      Check_Versions;
      Check_Consistency;
      Check_Configuration_Consistency;

      --  Complete bind if no errors

      if Errors_Detected = 0 then
         Find_Elab_Order;

         if Errors_Detected = 0 then
            if Elab_Order_Output then
               Write_Eol;
               Write_Str ("ELABORATION ORDER");
               Write_Eol;

               for I in Elab_Order.First .. Elab_Order.Last loop
                  Write_Str ("   ");
                  Write_Unit_Name (Unit.Table (Elab_Order.Table (I)).Uname);
                  Write_Eol;
               end loop;

               Write_Eol;
            end if;

            if not Check_Only then
               Gen_Output_File;
            end if;
         end if;
      end if;

      Total_Errors := Total_Errors + Errors_Detected;
      Total_Warnings := Total_Warnings + Warnings_Detected;

   exception
      when Unrecoverable_Error =>
         Total_Errors := Total_Errors + Errors_Detected;
         Total_Warnings := Total_Warnings + Warnings_Detected;
   end;

   --  All done. Set proper exit status.

   Finalize_Binderr;
   Namet.Finalize;

   if Total_Errors > 0 then
      Exit_Program (E_Errors);
   elsif Total_Warnings > 0 then
      Exit_Program (E_Warnings);
   else
      Exit_Program (E_Success);
   end if;

end Gnatbind;
