
/*
 * DREADERD/MAIN.C - diablo newsreader
 *
 *	This is the startup for the dreaderd daemon.
 *
 *	The diablo newsreader requires one or remote diablo news boxes to 
 *	obtain articles from, but keeps track of article assignments to groups
 *	and overview information locally.
 *
 *	Operationally, dreaderd runs as a standalone multi-forking, 
 *	multi-threaded daemon.  It forks X resolvers and Y readers.  Each
 *	reader can support N connections (NxY total connections) and maintains
 *	its own links to one or more news servers (diablo, INN, whatever). 
 *	Overview information is utilized and articles are accessed by 
 *	message-id.
 *
 *	The diablo newsreader is also able to accept header-only feeds from
 *	one or more diablo servers via 'mode headfeed'.  Processing of these
 *	feeds is not multi-threaded, only processing of newsreader functions is
 *	multithreaded.  The diablo newsreader does not operate with a dhistory
 *	file.  If taking a feed from several sources you must also run the
 *	diablo server on the same box and use that to feed the dreaderd system,
 *	though the diablo server need not store any articles itself.
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

#if NEED_TERMIOS
#include <sys/termios.h>
#endif

Prototype char *ReportedHostName;
Prototype char *XRefHost;
Prototype void RTClearStatus(int index, int count);
Prototype void RTUpdateStatus(int slot, const char *ctl, ...);

Prototype int NumReaderForks;
Prototype int ThisReaderFork;
Prototype int FeedOnlyServer;
Prototype int CoreDebugOpt;
Prototype int FastCopyOpt;
Prototype pid_t *ReaderPids;
Prototype char *RTStatus;
Prototype char *PathHost;
Prototype char *MyGroupHome;

pid_t ForkPipe(int *pfd, void (*task)(int fd, const char *id), const char *description);
void HandleDnsThread(ForkDesc *desc);
void ValidateTcpBufferSize(int *psize);
void SetMyHostName(void);
void deleteDnsResHash(DnsRes *dres, int pid);
int getReaderForkSlot(void);
void delReaderForkSlot(pid_t pid);

char *PathHost = NULL;
char *ReportedHostName = NULL;
char *XRefHost = NULL;
char *RTStatus;
char *MyGroupHome;
char *PidFile;
const char *SysLogDesc;
int CoreDebugOpt = 0;
int FastCopyOpt = 1;
int TxBufSize = 4096;
int RxBufSize = 2048;
int STxBufSize = 4096;
int SRxBufSize = 16384;
int MaxSDnsForks = 0;		/* future				*/
int MaxConnects;
int ConnectCount;
int FailCount;
int ListenersEnabled = 1;

int NumReaderFeedForks;
int NumReaderForks;
int NumDnsForks;
int NumSDnsForks;
int NumPending;			/* pending connections	*/
int NumActive;			/* active connections	*/

int ThisReaderFork;
int FeedOnlyServer = -1;
pid_t *ReaderPids;

MemPool *DnsResPool;
DnsRes  *DnsResHash[DRHSIZE];

typedef struct BindList {
    struct BindList *bl_Next;
    int		bl_Port;
    char 	*bl_Host;
} BindList;

BindList BindDef = { NULL, 119, NULL };
BindList *BindBase = &BindDef;

int
main(int ac, char **av)
{
    int i;
    int PerformIdent = 1;

    LoadDiabloConfig(ac, av);

    for (i = 1; i < ac; ++i) {
	char *ptr = av[i];
	int v;

	if (*ptr != '-') {
	    fprintf(stderr, "Expected option: %s\n", ptr);
	    exit(1);
	}
	ptr += 2;

	v = (*ptr) ? strtol(ptr, NULL, 0) : 1;

	switch(ptr[-1]) {
	case 'C':
	    if (*ptr == 0)
		++i;
	    break;
	case 'd':
	    DebugOpt = v;
	    break;
	case 'p':
	    PathHost = (*ptr) ? ptr : av[++i];
	    if (strcmp(PathHost, "0") == 0)
		PathHost = "";
	    break;
	case 's':
	    ptr = (*ptr) ? ptr : av[++i];
	    SetStatusLine(ptr, strlen(ptr));
	    break;
	case 'x':
	    XRefHost = (*ptr) ? ptr : av[++i];
	    break;
	case 'T':
	    TxBufSize = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'R':
	    RxBufSize = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'S':
	    switch(*ptr) {
	    case 'T':
		++ptr;
		STxBufSize = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
		break;
	    case 0:
	    case '0':
	    case '1':
	    case '2':
	    case '3':
	    case '4':
	    case '5':
	    case '6':
	    case '7':
	    case '8':
	    case '9':
		STxBufSize = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    case 'R':
		++ptr;
		SRxBufSize = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
		break;
	    default:
		fatal("bad option: -S (must be R#, T#, or #)");
		break;
	    }
	    break;
	case 'X':
	    CoreDebugOpt = 1;
	    break;
	case 'F':
	    DiabloReaderFeedForks = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'D':
	    DiabloReaderDns = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'M':
	    DiabloReaderForks = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'N':
	    DiabloReaderThreads = strtol(((*ptr) ? ptr : av[++i]), NULL, 0);
	    break;
	case 'h':
	    ReportedHostName = (*ptr) ? ptr : av[++i];
	    break;
	case 'P':
	    {
		BindList *bl = calloc(sizeof(BindList), 1);

		if (BindBase == &BindDef)
		    BindBase = NULL;

		bl->bl_Port = 119;
		bl->bl_Host = NULL;

		if (*ptr == 0)
		    ptr = av[++i];
		if (strrchr(ptr, ':')) {
		    bl->bl_Port = strtol(strrchr(ptr, ':') + 1, NULL, 0);
		    bl->bl_Host = strdup(ptr);
		    *strrchr(bl->bl_Host, ':') = 0;
		} else {
		    bl->bl_Port = strtol(ptr, NULL, 0);
		}
		bl->bl_Next = BindBase;
		BindBase = bl;
	    }
	    break;
	case 'r':
	    RTStatus = (*ptr) ? ptr : av[++i];
	    break;
	case 'f':
	    FastCopyOpt = v;
	    break;
	case 'I':
	    PerformIdent = 0;
	    break;
	default:
	    fprintf(stderr, "unknown option: %s\n", ptr - 2);
	    exit(1);
	}
    }
    ValidateTcpBufferSize(&RxBufSize);
    ValidateTcpBufferSize(&TxBufSize);
    ValidateTcpBufferSize(&SRxBufSize);
    ValidateTcpBufferSize(&STxBufSize);

    MaxConnects = DiabloReaderThreads * DiabloReaderForks;

    MyGroupHome = strdup(PatExpand(GroupHomePat));

    /*
     * To stdout, not an error, no arguments == user requesting 
     * options list.  We can do this because the -p option is mandatory.
     */

    if (PathHost == NULL) {
	printf(
	    "dreaderd [-d#] -p pathhost [-s \"...\"] [-T txbufsize]"
	    " [-R rxbufsize] [-F maxfeedforks] [-M maxreaderforks]"
	    " [-N maxthreads/reader] [-h reportedhostname]"
	    " [-P [bindhost]:port]\n"
	);
	exit(0);
    }

    /*
     * Setup realtime status file name and remove file in case
     * dreaderd's are left over from a previous run.
     */

    if (RTStatus == NULL) {
	RTStatus = malloc(strlen(PatExpand(LogHomePat)) + 32);
	sprintf(RTStatus, "%s/dreaderd.status", PatExpand(LogHomePat));
    }
    remove(RTStatus);

    /*
     * Set host name
     */

    if (ReportedHostName == NULL)
	SetMyHostName();

    /*
     * Setup memory zone for localized data so the children do not
     * inherit it.
     */

    /*
     * bind to our socket prior to changing uid/gid
     */

    {
	BindList *bl;

	for (bl = BindBase; bl; bl = bl->bl_Next) {
	    int lfd = socket(AF_INET, SOCK_STREAM, 0);
	    struct sockaddr_in lsin;

	    bzero(&lsin, sizeof(lsin));
	    lsin.sin_addr.s_addr = INADDR_ANY;
	    lsin.sin_port = htons(bl->bl_Port);
	    lsin.sin_family = AF_INET;

	    if (bl->bl_Host != NULL) {
		struct hostent *he;

		if ((he = gethostbyname(bl->bl_Host)) != NULL) {
		    lsin.sin_addr = *(struct in_addr *)he->h_addr;
		} else {
		    lsin.sin_addr.s_addr = inet_addr(bl->bl_Host);
		    if (lsin.sin_addr.s_addr == INADDR_NONE) {
			perror("gethostbyname(BindHost)");
			exit(1);
		    }
		}
	    }
	    {
		int on = 1;

		setsockopt(lfd, SOL_SOCKET, SO_REUSEADDR, (void *)&on, sizeof(on));
		setsockopt(lfd, SOL_SOCKET, SO_KEEPALIVE, (void *)&on, sizeof(on));
	    }
	    setsockopt(lfd, SOL_SOCKET, SO_SNDBUF, (void *)&TxBufSize, sizeof(int));
	    setsockopt(lfd, SOL_SOCKET, SO_RCVBUF, (void *)&RxBufSize, sizeof(int));

	    if (bind(lfd, (struct sockaddr *)&lsin, sizeof(lsin)) < 0) {
		perror("bind");
		exit(1);
	    }
	    listen(lfd, 256);
	    AddThread("acceptor", lfd, -1, THREAD_LISTEN, -1, 0);
	}
    }

    /*
     * Fork if not in debug mode and detach from terminal
     */

    if (DebugOpt == 0) {
	pid_t pid;
	int fd;

	fflush(stdout);
	fflush(stderr);

	if ((pid = fork()) != 0) {
	    if (pid < 0) {
		perror("fork");
		exit(1);
	    }
	    exit(0);
	}
	/*
	 * child continues, close stdin, stdout, stderr, detach tty,
	 * detach process group.
	 */
	{
	    int fd = open("/dev/null", O_RDWR);
	    if (fd != 0)
		dup2(fd, 0);
	    if (fd != 1)
		dup2(fd, 1);
	    if (fd != 2)
		dup2(fd, 2);
	    if (fd > 2)
		close(fd);
	}
#if USE_TIOCNOTTY
	if ((fd = open("/dev/tty", O_RDWR)) >= 0) {
	    ioctl(fd, TIOCNOTTY, 0);
	    close(fd);
	}
#endif
#if USE_SYSV_SETPGRP
        setpgrp();
#else
	setpgrp(0, 0);
#endif
    }

    /*
     * Save PID
     */

    {
	int fd;
	char buf[32];

	sprintf(buf, "%d\n", (int)getpid());

	PidFile = malloc(strlen(PatExpand(LogHomePat)) + 32);
	sprintf(PidFile, "%s/dreaderd.pid", PatExpand(LogHomePat));
	remove(PidFile);
	if ((fd = open(PidFile, O_RDWR|O_CREAT|O_TRUNC, 0644)) < 0) {
	    fprintf(stderr, "unable to create %s\n", PidFile);
	    exit(1);
	}
	write(fd, buf, strlen(buf));
	close(fd);
    }

    /*
     * Ignore SIGPIPE, in case a write() fails we do not
     * want to segfault.
     */
    rsignal(SIGPIPE, SIG_IGN);

    /*
     * change my uid/gid
     */

    {
        struct passwd *pw = getpwnam("news");
        struct group *gr = getgrnam("news");
        gid_t gid;

        if (pw == NULL) {
            perror("getpwnam('news')");
            exit(1);
        }
        if (gr == NULL) {
            perror("getgrnam('news')");
            exit(1);
        }
        gid = gr->gr_gid;
        setgroups(1, &gid);
        setgid(gr->gr_gid);
        setuid(pw->pw_uid);
    }

    /*
     * RTStatus file initial open
     */

    RTStatusOpen(RTStatus, 0, 1);

    /*
     * open syslog
     */

    openlog("dreaderd", (DebugOpt ? LOG_PERROR : 0) | LOG_NDELAY | LOG_PID, LOG_NEWS);
    SysLogDesc = "dreaderd";

    /*
     * Cancel cache (inherited by children)
     */

    InitCancelCache();

    ReaderPids = malloc((DiabloReaderForks + DiabloReaderFeedForks) * sizeof(pid_t));
    bzero(ReaderPids, (DiabloReaderForks + DiabloReaderFeedForks) * sizeof(pid_t));

    /*
     * accept()/pipe-processing loop, fork child processes as required
     * to maintain requested parameters (initial fork is done as part of
     * the nominal re-fork code when a child dies).
     */

    for (;;) {
	/*
	 * handle required forks and child process cleanup
	 *
	 * (A) reader NNTP processes
	 */

	while (NumReaderForks < DiabloReaderForks) {
	    pid_t pid;
	    int fd;

	    ThisReaderFork = getReaderForkSlot();
	    FeedOnlyServer = 0;
	    pid = ForkPipe(&fd, ReaderTask, "reader");
	    FeedOnlyServer = -1;
	    if (pid < 0)
		break;
	    ReaderPids[ThisReaderFork] = pid;
	    AddThread("reader", fd, pid, THREAD_READER, -1, 0);
	    ++NumReaderForks;
	}

	/*
	 * (B) reader feeder-only processes ('mode reader' not available)
	 */

	while (NumReaderFeedForks < DiabloReaderFeedForks) {
	    pid_t pid;
	    int fd;

	    ThisReaderFork = getReaderForkSlot();
	    FeedOnlyServer = 1;
	    pid = ForkPipe(&fd, ReaderTask, "feed");
	    FeedOnlyServer = -1;
	    if (pid < 0)
		break;
	    ReaderPids[ThisReaderFork] = pid;
	    AddThread("feed", fd, pid, THREAD_FEEDER, -1, 0);
	    ++NumReaderFeedForks;
	}

	/*
	 * (C) standard DNS processes
	 */

	while (NumDnsForks < DiabloReaderDns) {
	    pid_t pid;
	    int fd;

	    if ((pid = ForkPipe(&fd, DnsTask, "reader-dns")) < 0)
		break;
	    AddThread("dns", fd, pid, THREAD_DNS, -1, 0);
	    ++NumDnsForks;
	}

#ifdef NOTDEF
	/*
	 * (D) server hostname resolver	(future)
	 */

	while (NumSDnsForks < MaxSDnsForks) {
	    pid_t pid;
	    int fd;

	    if ((pid = ForkPipe(&fd, DnsTask, "reader-sdns")) < 0)
		break;
	    AddThread("sdns", fd, pid, THREAD_SDNS, -1, 0);
	    ++NumSDnsForks;
	}
#endif

	/*
	 * (E) dead processes
	 */

	{
	    pid_t pid;

	    while ((pid = wait3(NULL, WNOHANG, NULL)) > 0) {
		ForkDesc *desc;

		if ((desc = FindThread(-1, pid)) != NULL) {
		    if (desc->d_Type == THREAD_DNS) {
			logit(LOG_CRIT, "Lost dns resolver %d", (int)pid);
			--NumDnsForks;
			if (desc->d_FdPend >= 0) {
			    logit(LOG_INFO, "pending descriptor from %s on dead DNS pid closed",
				inet_ntoa(desc->d_SaveRSin.sin_addr)
			    );
			    close(desc->d_FdPend);
			    desc->d_FdPend = -1;
			    NumPending -= desc->d_Count; /* note: d_Count may already be 0 */
			    NumActive -= desc->d_Count;
			    desc->d_Count = 0;
			}
		    } else if (desc->d_Type == THREAD_READER) {
			logit(LOG_CRIT, "Lost reader process %d", (int)pid);
			--NumReaderForks;
			delReaderForkSlot(pid);
			NumActive -= desc->d_Count;
			deleteDnsResHash(NULL, (int)pid);
		    } else if (desc->d_Type == THREAD_FEEDER) {
			logit(LOG_CRIT, "Lost feeder process %d", (int)pid);
			--NumReaderFeedForks;
			delReaderForkSlot(pid);
			NumActive -= desc->d_Count;
			deleteDnsResHash(NULL, (int)pid);
		    } else if (desc->d_Type == THREAD_SDNS) {
			logit(LOG_CRIT, "Lost sdns process %d", (int)pid);
			--NumSDnsForks;
		    } else {
			fatal("main loop, unknown thread type %d pid %d\n", desc->d_Type, (int)pid);
		    }
		    DelThread(desc);
		} else {
		    fatal("main loop, wait3() pid %d, pid is not known to me!", (int)pid);
		}
	    }
	}

	/*
	 * (F) select core
	 *
	 * select on descriptors (listen socket and pipe sockets only),
	 * process results
	 */
	{
	    struct timeval tv;
	    fd_set rfds = SFds;
	    int i;

	    NextTimeout(&tv, 10 * 1000);

	    stprintf("Connect=%d Failed=%d Dns=%d/%d Act=%d/%d", 
		ConnectCount, FailCount,
		NumPending, DiabloReaderDns, 
		NumActive, MaxConnects
	    );
	    RTStatusUpdate(0, "Connect=%d Failed=%d Dns=%d/%d Act=%d/%d", 
		ConnectCount, FailCount,
		NumPending, DiabloReaderDns, 
		NumActive, MaxConnects
	    );

	    select(MaxFds, &rfds, NULL, NULL, &tv);
	    gettimeofday(&CurTime, NULL);

	    for (i = 0; i < MaxFds; ++i) {
		if (FD_ISSET(i, &rfds)) {
		    ForkDesc *desc;

		    if ((desc = FindThread(i, -1)) != NULL) {
			switch(desc->d_Type) {
			case THREAD_LISTEN:
			    /*
			     * accept new descriptor, depend on socket
			     * buffer for write to not block if descriptor
			     * beyond what we can handle.
			     */
			    {
				struct sockaddr_in rsin;
				int rslen = sizeof(rsin);

				int fd = accept(desc->d_Fd, (struct sockaddr *)&rsin, &rslen);
				if (fd >= MAXFDS) {
				    char *s = "500 Too many open descriptors, closing\r\n";
				    logit(LOG_CRIT, "fd beyond MAXFDS: %d, closing", fd);
				    write(fd, s, strlen(s));
				    close(fd);
				    fd = -1;
				    ++ConnectCount;
				    ++FailCount;
				}
				if (fd >= 0) {
				    /*
				     * queue descriptor information to DNS
				     * resolver and put descriptor on hold.
				     */
				    ForkDesc *best;
				    struct sockaddr_in lsin;
				    int lslen = sizeof(lsin);

				    getsockname(fd, (struct sockaddr *)&lsin, &lslen);

				    ++ConnectCount;

				    best = FindLeastUsedThread(THREAD_DNS, 1, 0, NULL, -1);

				    if (best != NULL) {
					DnsReq dreq;

					++NumActive;
					++NumPending;
					++best->d_Count;

					bzero(&dreq, sizeof(dreq));
					dreq.dr_RSin = rsin;
					dreq.dr_LSin = lsin;
					dreq.performident = PerformIdent;

					best->d_FdPend = fd;
					best->d_SaveRSin = rsin;
					write(best->d_Fd, &dreq, sizeof(dreq));
					/* 
					 * ignore return code.  If DNS process died, it
					 * will be cleaned up later.
					 */
				    } else {
					char *s = "500 Software error resolving DNS\r\n";
					logit(LOG_CRIT, "software error resolving DNS");
					write(fd, s, strlen(s));
					close(fd);
					fd = -1;
					++FailCount;
				    }
				}
			    }
			    break;
			case THREAD_DNS:
			    /*
			     * read query result from DNS handler, then
			     * issue appropriate action to descriptor.
			     */
			    HandleDnsThread(desc);
			    break;
			case THREAD_READER:
			case THREAD_FEEDER:
			    /*
			     * reader process returns one byte for each
			     * closed connection.  Update our ref counts.
			     */
			    {
				DnsRes dres;
				int n;

				while ((n = read(desc->d_Fd, &dres, sizeof(dres))) > 0) {
				    if (n < sizeof(dres)) {
					fcntl(desc->d_Fd, F_SETFL, 0);
					n += read(desc->d_Fd, (char *)&dres + n, sizeof(dres) - n);
					fcntl(desc->d_Fd, F_SETFL, O_NONBLOCK);
				    }
				    if (n != sizeof(dres))
					break;
				    --NumActive;
				    --desc->d_Count;
				    deleteDnsResHash(&dres, desc->d_Pid);
				}
			    }
			    break;
			default:
			    fatal("Unknown descriptor type %d", desc->d_Type);
			    break;
			}
		    } else {
			fatal("select returned unknown descriptor %d", i);
		    }
		}
	    }
	    (void)ScanTimers(1, 0);
	}

	/*
	 * Disable listen socket if we have too many connections pending or
	 * we are full up, enable listen socket otherwise.  NOTE: NumPending
	 * is included in the NumActive number.
	 */
	if (NumPending < NumDnsForks && 
	    NumActive < NumReaderForks * DiabloReaderThreads
	) {
	    if (ListenersEnabled == 0) {
		int i;

		ListenersEnabled = 1;
		for (i = 0; i < MaxFds; ++i) {
		    ForkDesc *desc;

		    if ((desc = FindThread(i, -1)) != NULL && 
			desc->d_Fd >= 0 &&
			desc->d_Type == THREAD_LISTEN
		    ) {
			FD_SET(desc->d_Fd, &SFds);
		    }
		}
	    }
	} else {
	    if (ListenersEnabled == 1) {
		int i;

		ListenersEnabled = 0;
		for (i = 0; i < MaxFds; ++i) {
		    ForkDesc *desc;

		    if ((desc = FindThread(i, -1)) != NULL && 
			desc->d_Fd >= 0 &&
			desc->d_Type == THREAD_LISTEN
		    ) {
			FD_CLR(desc->d_Fd, &SFds);
		    }
		}
	    }
	}
    }
    return(0);
}

pid_t
ForkPipe(int *pfd, void (*task)(int fd, const char *id), const char *description)
{
    int fds[2];
    pid_t pid;

    *pfd = -1;

    /*
     * NOTE: must use socketpair in order to be able to pass descriptors.
     */

    if (socketpair(PF_UNIX, SOCK_STREAM, IPPROTO_IP, fds) < 0) {
	logit(LOG_ERR, "socketpair() call failed");
	return((pid_t)-1);
    }
    if (fds[0] > MAXFDS || fds[1] > MAXFDS) {
	logit(LOG_ERR, "pipe desc > MAXFDS: %d/%d/%d", fds[0], fds[1], MAXFDS);
	close(fds[0]);
	close(fds[1]);
	return((pid_t)-1);
    }

    *pfd = fds[0];

    fflush(stdout);
    fflush(stderr);

    /*
     * Some systems can't deal with syslog through a
     * fork.
     */

    closelog();
    pid = fork();
    if (pid == 0) {
	openlog(description, (DebugOpt ? LOG_PERROR : 0) | LOG_NDELAY | LOG_PID, LOG_NEWS);
	SysLogDesc = description;
    } else {
	openlog(SysLogDesc, (DebugOpt ? LOG_PERROR : 0) | LOG_NDELAY | LOG_PID, LOG_NEWS);
    }

    if (pid == 0) {
	freePool(&DnsResPool);
	stprintf("%s startup", description);
	/* note: DnsResHash not longer valid in children */
	/* note: fds[1] is a normal descriptor, not a non-blocking desc */
	ResetThreads();
	close(*pfd);
	task(fds[1], description);
	_exit(1);
    } else if (pid < 0) {
	logit(LOG_ERR, "fork() failed: %s", strerror(errno));
	close(*pfd);
	close(fds[1]);
	*pfd = -1;
    } else if (pid > 0) {
	close(fds[1]);
	if (DebugOpt)
	    printf("forked %s child process %d\n", description, (int)pid);
    }
    return(pid);
}


void
HandleDnsThread(ForkDesc *desc)
{
    DnsRes dres;
    ForkDesc *reader = NULL;
    static int RandCounter;
    const char *error1 = NULL;
    const char *error2 = NULL;
    int r;

    errno = 0;
    if ((r = read(desc->d_Fd, &dres, sizeof(dres))) != sizeof(dres)) {
	/*
	 * The dns subprocesses return a fixed-size structure.  If we 
	 * get part of it, we should get the whole thing but may not 
	 * since the descriptor is set for non-blocking.  Handle the
	 * case.
	 */
	if (r > 0) {
	    int n;

	    fcntl(desc->d_Fd, F_SETFL, 0);
	    n = read(desc->d_Fd, (char *)&dres + r, sizeof(dres) - r);
	    fcntl(desc->d_Fd, F_SETFL, O_NONBLOCK);
	    if (n > 0)
		r += n;
	}
	if (r != sizeof(dres)) {
	    if (errno != EINTR && 
		errno != EWOULDBLOCK && 
		errno != EINPROGRESS
	    ) {
		/*
		 * let the child process cleanup any operations in
		 * progress.
		 */
		logit(LOG_CRIT, "read %d error on DNS subprocess, killing %d (%s)", r, (int)desc->d_Pid, strerror(errno));
		kill(desc->d_Pid, 9);
		NumActive -= desc->d_Count;
		NumPending -= desc->d_Count;
		desc->d_Count = 0;
	    }
	    return;
	}
    }
    --desc->d_Count;
    --NumPending;

    if (dres.dr_Code == 0) {
	error1 = "500 Unauthorized connection rejected\r\n";
	error2 = "Unauthorized";
    }

    /*
     * Pass the descriptor to the correct thread.  A DF_FEED only connection
     * can be passed to a feed-specific thread.
     */

    if (error1 == NULL) {
	if ((dres.dr_Flags & (DF_FEED|DF_READ|DF_POST)) == DF_FEED) {
	    reader = FindLeastUsedThread(THREAD_FEEDER, DiabloReaderThreads, 0, &RandCounter, -1);
	    dres.dr_Flags |= DF_FEEDONLY;
	} else {
	    reader = FindLeastUsedThread(THREAD_READER, DiabloReaderThreads, 0, &RandCounter, -1);
	}
	if (reader == NULL) {
	    error1 = "500 Server maxed out, please try again later\r\n";
	    error2 = "ServerMaxedOut";
	}
    }

    /*
     * Add DnsRes to hash table
     */
    if (error1 == NULL) {
	int hi = shash(inet_ntoa(dres.dr_Addr)) & DRHMASK;
	int hmi;
	DnsRes *dr;
	int hcount = 0;
	int ucount = 0;
	int mcount = 0;

	for (dr = DnsResHash[hi]; dr; dr = dr->dr_HNext) {
	    if (dres.dr_Addr.s_addr == dr->dr_Addr.s_addr) {
		++hcount;
		if (strcmp(dres.dr_User, dr->dr_User) == 0)
		    ++ucount;
	    }
	}
	if (dres.dr_MaxFromMatch) {
	    for (hmi=0; hmi < DRHSIZE; hmi++) 
	    for (dr = DnsResHash[hmi]; dr; dr = dr->dr_HNext)
		if (strcmp(dres.dr_Match, dr->dr_Match) == 0)
		    ++mcount;
	}

	if (dres.dr_MaxFromMatch && mcount >= dres.dr_MaxFromMatch) {
	    error1 = "500 Too many active connections\r\n";
	    error2 = "TooManyFromMatch";
	}
	if (dres.dr_MaxFromHost && hcount >= dres.dr_MaxFromHost) {
	    error1 = "500 Too many active connections from this host\r\n";
	    error2 = "TooManyFromHost";
	}
	if (dres.dr_MaxFromUserHost && ucount >= dres.dr_MaxFromUserHost) {
	    error1 = "500 Too many active connections from this user\r\n";
	    error2 = "TooManyFromUser";
	}
	if (error1 == NULL) {
	    dr = zalloc(&DnsResPool, sizeof(DnsRes));
	    *dr = dres;
	    dr->dr_HNext = DnsResHash[hi];
	    dr->dr_ReaderPid = reader->d_Pid;
	    DnsResHash[hi] = dr;
	}
    }

    /*
     * error return
     */

    if (error1) {
	if ((dres.dr_Flags & DF_QUIET) == 0) {
	    logit(
		LOG_INFO, 
		"connection from %s (%s:%d) rejected: %s", 
		dres.dr_Host,
		inet_ntoa(desc->d_SaveRSin.sin_addr),
		ntohs(desc->d_SaveRSin.sin_port),
		error2
	    );
	}
	write(desc->d_FdPend, error1, strlen(error1));
	close(desc->d_FdPend);
	desc->d_FdPend = -1;
	++FailCount;
	--NumActive;
	return;
    }

    /*
     * Leave NumActive bumped, transfer descriptor to subprocess
     *
     * pass the descriptor to an appropriate subprocess
     */

    {
	struct msghdr msg;
	struct iovec iov;
	struct {
#if FDPASS_USES_CMSG
	    struct cmsghdr cmsg;
#endif
	    int fd;
	} cmsg;

	bzero(&msg, sizeof(msg));
	bzero(&cmsg, sizeof(cmsg));

	iov.iov_base = (void *)&dres;
	iov.iov_len = sizeof(dres);

	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;
#if FDPASS_USES_ACC
	msg.msg_accrights = (caddr_t)&cmsg;
	msg.msg_accrightslen = sizeof(cmsg);
#else
	msg.msg_control = (caddr_t)&cmsg;
	msg.msg_controllen = sizeof(cmsg);
#endif
#if FDPASS_USES_CMSG
	msg.msg_flags = 0;
	cmsg.cmsg.cmsg_len = sizeof(cmsg);
	cmsg.cmsg.cmsg_level = SOL_SOCKET;
	cmsg.cmsg.cmsg_type = SCM_RIGHTS;
#endif
	cmsg.fd = desc->d_FdPend;

	if (DebugOpt)
	    printf("Pass file descriptor fd=%d\n", (int)cmsg.fd);

	errno = 0;

	/*
	 * If we do not turn off nagle, lots of little 'group' transactions
	 * can result in major delays even if nagle only kicks in on a few
	 * of them.
	 */

#ifdef TCP_NODELAY
	/*
	 * Turn on TCP_NODELAY
	 */
	if (desc->d_FdPend >= 0) {
	    int one = 1;
	    setsockopt(desc->d_FdPend, IPPROTO_TCP, TCP_NODELAY, &one, sizeof(one));
	}
#endif

	/*
	 * Send the message.  This should never fail.  If it does, try to core
	 * the reader subprocess.  Certainly kill it.
	 */

	if (sendmsg(reader->d_Fd, &msg, 0) < 0) {
	    const char *s = "500 Server error, fd pass failed\r\n";
	    logit(
		LOG_INFO, 
		"connect: error passing file descriptor: %s, killing reader pid %d", 
		strerror(errno),
		reader->d_Pid
	    );
	    if (DebugOpt)
		printf("sendmsg() failed: %s\n", strerror(errno));
	    write(desc->d_FdPend, s, strlen(s));
	    close(desc->d_FdPend);
	    desc->d_FdPend = -1;
	    ++FailCount;
	    --NumActive;
	    deleteDnsResHash(&dres, desc->d_Pid);

	    if (reader->d_Pid > 0)
		kill(reader->d_Pid, SIGQUIT);
	    return;
	}
	close(desc->d_FdPend);
	desc->d_FdPend = -1;
	++reader->d_Count;

	if ((dres.dr_Flags & DF_QUIET) == 0) {
	    logit(
		LOG_INFO,
		"connect from %s%s%s (%s:%d)", 
		(dres.dr_User[0] ? dres.dr_User : ""),
		(dres.dr_User[0] ? "@" : ""),
		dres.dr_Host,
		inet_ntoa(desc->d_SaveRSin.sin_addr),
		ntohs(desc->d_SaveRSin.sin_port)
	    );
	}
    }
}

void
ValidateTcpBufferSize(int *psize)
{
    if (*psize < 512)
	*psize = 512;
    if (*psize > 256*1024)
	*psize = 256*1024;
}

void
SetMyHostName(void)
{
    char buf[256];

    buf[sizeof(buf)-1] = 0;
    gethostname(buf, sizeof(buf) - 1);
    ReportedHostName = malloc(strlen(buf) + 1);
    strcpy(ReportedHostName, buf);
}

void
deleteDnsResHash(DnsRes *dres, int pid)
{
    if (dres == NULL) {
	int i;

	for (i = 0; i < DRHSIZE; ++i) {
	    DnsRes **pdr = &DnsResHash[i];
	    DnsRes *dr;

	    while ((dr = *pdr) != NULL) {
		if (dr->dr_ReaderPid == pid) {
		    logit(
			LOG_INFO,
			"killed from %s%s%s (%s:%d)",
			(dr->dr_User[0] ? dr->dr_User : ""),
			(dr->dr_User[0] ? "@" : ""),
			dr->dr_Host,
			inet_ntoa(dr->dr_Addr),
			ntohs(dr->dr_Port)
		    );
		    *pdr = dr->dr_HNext;
		    zfree(&DnsResPool, dr, sizeof(DnsRes));
		} else {
		    pdr = &dr->dr_HNext;
		}
	    }
	}
    } else {
	int hi = shash(inet_ntoa(dres->dr_Addr)) & DRHMASK;
	DnsRes **pdr = &DnsResHash[hi];
	DnsRes *dr;

	while ((dr = *pdr) != NULL) {
	    if (dr->dr_ReaderPid == pid &&
		dr->dr_Addr.s_addr == dres->dr_Addr.s_addr &&
		dr->dr_Port == dres->dr_Port &&
		strcmp(dr->dr_User, dres->dr_User) == 0
	    ) {
		if ((dr->dr_Flags & DF_QUIET) == 0) {
		    logit(
			LOG_INFO,
			"closed from %s%s%s (%s:%d) groups=%d arts=%d bytes=%d",
			(dr->dr_User[0] ? dr->dr_User : ""),
			(dr->dr_User[0] ? "@" : ""),
			dr->dr_Host,
			inet_ntoa(dres->dr_Addr),
			ntohs(dr->dr_Port),
			dres->dr_GrpCount,
			dres->dr_ArtCount,
			dres->dr_ByteCount
		    );
		}
		*pdr = dr->dr_HNext;
		zfree(&DnsResPool, dr, sizeof(DnsRes));
		break;
	    } else {
		pdr = &dr->dr_HNext;
	    }
	}
    }
}

int
getReaderForkSlot(void)
{
    int i;

    for (i = 0; i < DiabloReaderForks + DiabloReaderFeedForks; ++i) {
	if (ReaderPids[i] == 0)
	    return(i);
    }
    return(0); /* XXX panic */
}

void
delReaderForkSlot(pid_t pid)
{
    int i;

    for (i = 0; i < DiabloReaderForks + DiabloReaderFeedForks; ++i) {
	if (ReaderPids[i] == pid)
	    ReaderPids[i] = 0;
    }
}

