/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTPSession.cpp

	Contains:	Implementation of RTPSession class. 
					
	Change History (most recent first):

	$Log: RTPSession.cpp,v $
	Revision 1.2  1999/02/19 23:08:26  ds
	Created
	
*/

#include "RTPSession.h"
#include "RTPStream.h"
#include "RTPModule.h"
#include "RTSPProtocol.h" 
#include "RTSPServerInterface.h"
#include "RTPServerInterface.h"
#include "RTCPPacket.h"
#include "OS.h"

RTPSession::RTPSession()
: 	Task(),
	fRTSPSessionID(fRTSPSessionIDBuf, 0),
	fMediaRef(NULL),
	fModule(NULL),
	fIsFirstPlay(true),
	fFirstPlayTime(0),
	fPlayTime(0),
	fAdjustedPlayTime(0),
	fNTPPlayTime(0),
	fSessionCreateTime(0),
	fStartTime(0),
	fStopTime(0),
	fTimeoutTask(this),
	fState(kPausedState),
	fNumQualityLevels(0),
	fBytesSent(0),
	fPacketsSent(0),
	fSessionMutex('rsmt'),
	fDebugFile(NULL)
	
{


	
	//don't actually setup the fTimeoutTask until the session has been bound!
	//(we don't want to get timeouts before the session gets bound)
	fTimeout = RTSPServerInterface::GetRTSPPrefs()->GetRTPTimeoutInSecs() * 1000;

	//Build a static RTCP sender report (this way, only the info that changes
	//on the fly will have to be written)
	char theTempCName[RTPServerInterface::kMaxCNameLen];
	UInt32 cNameLen = RTPServerInterface::GetACName(theTempCName);
	
	//write the SR & SDES headers
	UInt32* theSRWriter = (UInt32*)&fSenderReportBuffer;
	*theSRWriter = htonl(0x80c80006);
	theSRWriter += 7;
	//SDES length is the length of the CName, plus 2 32bit words, minus 1
	*theSRWriter = htonl(0x81ca0000 + (cNameLen >> 2) + 1);
	::memcpy(&fSenderReportBuffer[kSenderReportSizeInBytes], theTempCName, cNameLen);
	fSenderReportSize = kSenderReportSizeInBytes + cNameLen;
	
	//mark the session create time
	fSessionCreateTime = OS::Milliseconds();
	
}

RTPSession::~RTPSession()
{
	//send kill events to each stream. Because streams have back pointers to this object,
	//make sure that each stream actually deletes itself before moving onto the next stream
	for (OSQueueIter iter(&fStreamQueue); !iter.IsDone(); )
	{
		RTPStream* stream = (RTPStream*)iter.GetCurrent()->GetEnclosingObject();
		iter.Next();
		delete stream;
	}
#if RTPSESSION_LOGGING
	if (fDebugFile != NULL)
		::fclose(fDebugFile);
#endif
	
	//we better not be in the RTPSessionMap anymore!
#if DEBUG
	Assert(!fRTPMapElem.IsInTable());
	OSRef* theRef = RTPServerInterface::GetRTPSessionMap()->Resolve(&fRTSPSessionID);
	Assert(theRef == NULL);
#endif
}

QTSS_ErrorCode	RTPSession::Activate(const StrPtrLen& inSessionID)
{
	//Set the session ID for this session
	Assert(inSessionID.Len <= RTPModuleInterface::kMaxSessionIDLength);
	::memcpy(fRTSPSessionIDBuf, inSessionID.Ptr, inSessionID.Len);
	fRTSPSessionIDBuf[inSessionID.Len] = '\0';
	fRTSPSessionID.Len = inSessionID.Len;
	
	fRTPMapElem.Set(fRTSPSessionID, this);
	
	//Activate puts the session into the RTPSession Map
	OSRefTable* sessionTable = RTPServerInterface::GetRTPSessionMap();
	Assert(sessionTable != NULL);
	QTSS_ErrorCode err = sessionTable->Register(&fRTPMapElem);
	if (err == QTSS_DupName)
		return err;
	
#if RTPSESSION_LOGGING
	if (RTSPServerInterface::GetRTSPPrefs()->GetLogTimestamps())
	{
		char theFileName[512];
		::sprintf(theFileName, "rtplog.%s", fRTSPSessionIDBuf);
		fDebugFile = ::fopen(theFileName, "w");
	}
#endif

	Assert(err == QTSS_NoErr);
	RTPServerInterface::IncrementTotalSessions();
	return QTSS_NoErr;
}

RTSPProtocol::RTSPStatusCode RTPSession::AddStream(UInt32 inStreamID, StrPtrLen* inCodecName,
														UInt32 inCodecType,
														RTSPRequestInterface* request,
														RTPStream** outStream)
{
	Assert(outStream != NULL);
	*outStream = new ('Rstm') RTPStream(inStreamID, this, inCodecName, inCodecType);
	
	//stuff the code names into the dictionary (useful for logging modules)
	//  actually, this is cheating - the log module should iterate over all of the RTPSession streams 
	//  looking for codec names at the point it wants to log - csl
	if (inCodecType == RTPStream::kVideoCodecType)
		this->GetSessionDictionary()->SetStandardValue(SessionDictionary::kVideoCodecName, inCodecName->Ptr, inCodecName->Len);
	else if  (inCodecType == RTPStream::kAudioCodecType)
		this->GetSessionDictionary()->SetStandardValue(SessionDictionary::kAudioCodecName, inCodecName->Ptr, inCodecName->Len);
												
	return (*outStream)->Setup(request);
}

void	RTPSession::Play(RTSPRequestInterface* request, bool inSendRTCP, bool inShouldWriteHeader)
{
	//first setup the play associated session interface variables
	Assert(request != NULL);
	
	if (fModule == NULL)
	{
		Assert(0);//bad module, bad!
		return ;
	}
	
	fStartTime = request->GetStartTime();
	fStopTime = request->GetStopTime();
	
	//the client doesn't necessarily specify this information in a play,
	//if it doesn't, fall back on some defaults.
	if (fStartTime == -1)
		fStartTime = 0;
		
	//what time is this play being issued at?
	fPlayTime = OS::Milliseconds() + kPlayDelayTimeInMilSecs; //don't play until we've sent the play response
	if (fIsFirstPlay)
		fFirstPlayTime = fPlayTime - kPlayDelayTimeInMilSecs;
	fAdjustedPlayTime = fPlayTime - ((SInt64)(fStartTime * 1000));

	//for RTCP SRs, we also need to store the play time in NTP
	fNTPPlayTime = OS::ConvertToSecondsSince1900(fPlayTime);
	
	//tell all the setup streams to start playing.
	for (OSQueueIter qIter(&fStreamQueue); !qIter.IsDone(); qIter.Next())
		((RTPStream*)qIter.GetCurrent()->GetEnclosingObject())->Play(request, inSendRTCP, inShouldWriteHeader);

	//we are definitely playing now, so schedule the object!
	fState = kPlayingState;
	fIsFirstPlay = false;
	//this->SetIdleTimer(kPlayDelayTime);
	this->Signal(Task::kStartEvent);
}

void RTPSession::SendPlayResponse(RTSPRequestInterface* request)
{
	RTSPProtocol::RTSPHeader theHeader = RTSPProtocol::kRTPInfoHeader;
	
	//tell all the setup streams to start playing.
	for (OSQueueIter qIter(&fStreamQueue); !qIter.IsDone(); qIter.Next())
	{
		Assert(qIter.GetCurrent() != NULL);
		RTPStream* theStream = (RTPStream*)qIter.GetCurrent()->GetEnclosingObject();
		Assert(theStream != NULL);
		if (theStream != NULL)
		{
			theStream->AppendRTPInfo(theHeader, request);
			theHeader = RTSPProtocol::kSameAsLastHeader;
		}
	}
	request->SendHeader();
}

SInt64 RTPSession::Run()
{
	Assert(this->IsBound());//argh! if we get events before being bound, bad things occur
	if (!this->IsBound())
		return 0;
		
	EventFlags events = this->GetEvents();
	
	//if we have been instructed to go away, then let's delete ourselves
	if ((events & Task::kKillEvent) || (events & Task::kTimeoutEvent))
	{
		//deletion is a bit complicated. For one thing, it must happen from within
		//the Run function to ensure that we aren't getting events when we are deleting
		//ourselves. We also need to make sure that we aren't getting RTSP requests
		//(or, more accurately, that the stream object isn't being used by any other
		//threads). We do this by first removing the session from the session map.
		
#if RTPSESSION_DEBUGGING
		printf("RTPSession %ld: about to be killed. Eventmask = %ld\n",(SInt32)this, events);
#endif
		OSRefTable* sessionTable = RTPServerInterface::GetRTPSessionMap();
		Assert(sessionTable != NULL);
		sessionTable->UnRegister(&fRTPMapElem);
		
		//Any modules that look for timeout events should be notified now
		if (events & Task::kTimeoutEvent)
			RTPModuleInterface::ProcessTimeout(this);
	
		//at this point, we know no one is using this session. So now tell the
		//module to clean up. We don't need to grab the session mutex before
		//invoking the module here, because the session is unregistered and
		//therefore there's no way another thread could get involved anyway
		Assert(fModule != NULL);
		fModule->DestroySession(this);
		return -1;//doing this will cause the destructor to get called.
	}
	
	//if the stream is currently paused, just return without doing anything.
	//We'll get woken up again when a play is issued
	if (fState == kPausedState)
		return 0;
		
	//If we are past the stoptime, we shouldn't call send packets anymore!
	SInt64 scheduleTime = 0;	
	if ((events & Task::kStartEvent) || (events & Task::kIdleEvent))
	{
		//Make sure to grab the session mutex here, to protect the module against
		//RTSP requests coming in while it's sending packets
		//SInt64 scheduleTime = 0;
		{
			OSMutexLocker locker(&fSessionMutex);

			//just make sure we haven't been scheduled before our scheduled play
			//time. If so, reschedule ourselves for the proper time. (if client
			//sends a play while we are already playing, this may occur)
			SInt64 curTime = OS::Milliseconds();
			if (fPlayTime > curTime)
				scheduleTime = fPlayTime - curTime;
			else
			{			
#if RTPSESSION_DEBUGGING
				printf("RTPSession %ld: about to call SendPackets\n",(SInt32)this);
#endif
				Assert(fModule != NULL);
				scheduleTime = fModule->SendPackets(this);
#if RTPSESSION_DEBUGGING
				printf("RTPSession %ld: back from sendPackets, nextPacketTime = %qd\n",(SInt32)this, scheduleTime);
#endif
				//make sure not to get deleted accidently!
				if (scheduleTime < 0)
					scheduleTime = 0;
			}
		}		
		//if (scheduleTime > 0)
		//	this->SetIdleTimer(scheduleTime);
	}
	
	Assert(scheduleTime >= 0);//we'd better not get deleted accidently!
	return scheduleTime;
}

